/* **************************************************** gtkdirtreeview.c *** *
 * GtkDirTreeView
 *
 * Copyright (C) 2002-2005 Yasuyuki SUGAYA <sugaya@suri.it.okayama-u.ac.jp>
 * Okayama University
 *                                    Time-stamp: <05/04/02 02:48:03 sugaya>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 *
 * Some source code come from gimageview.
 *
 * ************************************************************************* */
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include "gtkdirtreeview.h"
#include "gnome-vfs-extension.h"
#include "gtkdirtreeview-default-icons.h"
#include "glib-fileutil-extension.h"

#ifndef GTK_DISABLE_DEPRECATED
#define	GTK_DISABLE_DEPRECATED
#endif	/* GTK_DISABLE_DEPRECATED */

#define	G_ZERO_STR	"0"
#define	G_ZERO_ZERO_STR	"0:0"

/* ************************************************************************* */
typedef enum {
  COLUMN_TERMINATOR = -1,
  COLUMN_LABEL,
  COLUMN_PATH,
  COLUMN_ICON_OPEN,
  COLUMN_ICON_CLOSE,
  COLUMN_IS_DUMMY,
  N_COLUMN
} TreeStoreColumn;

typedef struct _AdjustTreeIdle {
  GtkTreeView	*treeview;
  GtkTreeIter	*iter;
} AdjustTreeIdle;

enum {
  PROP_0,
  PROP_TOP_PATH,
  PROP_SHOW_HIDDEN
};

static GdkPixbuf	*folder;
static GdkPixbuf	*folder_open;
static GdkPixbuf	*folder_deny;
static GdkPixbuf	*folder_link;
static GdkPixbuf	*folder_link_open;

/* ************************************************************************* *
 * ؿ
 * ************************************************************************* */

/* ************************************************************************* */
static void	gtk_dir_tree_view_class_init (GtkDirTreeViewClass *class);
static void	gtk_dir_tree_view_init	(GtkDirTreeView		*treeview);

static void	gtk_dir_tree_view_set_property (GObject		*obejct,
						guint		prop_id,
						const GValue	*value,
						GParamSpec	*pspec);
static void	gtk_dir_tree_view_get_property (GObject		*obejct,
						guint		prop_id,
						GValue		*value,
						GParamSpec	*pspec);

static GtkWidget* _gtk_dir_tree_view_new(const gchar	*path,
					 gboolean	show_hidden);

static void	create_default_icon	(void);

static void	create_icon_from_inline (const guint8 *folder_pixbuf,
					 const guint8 *folder_open_pixbuf,
					 const guint8 *folder_link_pixbuf,
					 const guint8 *folder_link_open_pixbuf,
					 const guint8 *folder_dennied_pixbuf);

static void	create_icon_from_file (const gchar *folder_filename,
				       const gchar *folder_open_filename,
				       const gchar *folder_link_filename,
				       const gchar *folder_link_open_filename,
				       const gchar *folder_dennied_filename);

static void	insert_dummy_row	(GtkTreeStore	*store,
					 GtkTreeIter	*parent_iter);

static void	insert_row 		(GtkTreeStore	*store,
					 GtkTreeIter	*iter,
					 GtkTreeIter	*parent_iter,
					 const gchar	*label,
					 const gchar	*path);

static void	render_row 		(GtkTreeView	*treeview);

static void	adjust_tree 		(GtkTreeView	*treeview,
					 GtkTreeIter	*iter);

static gint	idle_adjust_tree 	(gpointer	data);

static void	add_adjust_tree_idle 	(GtkTreeView	*treeview,
					 GtkTreeIter	*iter);

/* ************************************************************************* */
static gboolean	cb_button_press 	(GtkWidget	*widget,
					 GdkEventButton	*ev,
					 gpointer	data);

static void	cb_expand 		(GtkDirTreeView	*treeview,
					 GtkTreeIter	*parent_iter,
					 GtkTreePath	*tree_path,
					 gpointer	data);

/* ************************************************************************* */
static gboolean	get_iter_from_path 	(GtkTreeView	*treeview,
					 const gchar	*dir,
					 GtkTreeIter	*iter);
static void	get_expanded_path	(GtkTreeView	*treeview,
					 GtkTreePath	*path,
					 gpointer	data);
static gboolean	is_expanded 		(GtkTreeView	*treeview,
					 const gchar	*dir);

/* ************************************************************************* */
static GtkTreeViewClass	*parent_class	= NULL;

/* ************************************************************************* *
 * åؿ
 * ************************************************************************* */

/* ************************************************************************* */
GType
gtk_dir_tree_view_get_type (void) {
  static GType	tree_view_type = 0;

  if (!tree_view_type) {
    static const GTypeInfo	tree_view_info = {
      sizeof (GtkDirTreeViewClass),
      NULL,
      NULL,
      (GClassInitFunc) gtk_dir_tree_view_class_init,
      NULL,
      NULL,
      sizeof (GtkDirTreeView),
      0,
      (GInstanceInitFunc) gtk_dir_tree_view_init,
    };
    tree_view_type = g_type_register_static (gtk_tree_view_get_type(),
					     "GtkDirTreeView", 
					     &tree_view_info, 0);
  }
  return tree_view_type;  
}

/* ************************************************************************* */
static void
gtk_dir_tree_view_class_init (GtkDirTreeViewClass	*class) {
  GObjectClass		*gobject_class;
  GtkObjectClass	*object_class;	  
  GtkWidgetClass	*widget_class;

  gobject_class = (GObjectClass *)   class;
  object_class  = (GtkObjectClass *) class;
  widget_class  = (GtkWidgetClass *) class;

  parent_class = gtk_type_class (gtk_tree_view_get_type ());

  gobject_class->set_property = gtk_dir_tree_view_set_property;
  gobject_class->get_property = gtk_dir_tree_view_get_property;

  g_object_class_install_property (gobject_class,
				   PROP_TOP_PATH,
				   g_param_spec_string 
				   ("top_path",
				    "top_path",
				    "The top path of the tree",
				    NULL,
				    G_PARAM_READABLE));

  g_object_class_install_property (gobject_class, 
				   PROP_SHOW_HIDDEN,
				   g_param_spec_boolean 
				   ("show_hidden",
				    "Show hidden folders",
				    "Whether hidden folders should be displayed",
				    FALSE,
				    G_PARAM_READABLE | G_PARAM_WRITABLE));
}

/* ************************************************************************* */
static void
gtk_dir_tree_view_init (GtkDirTreeView	*treeview) {
  
}

/* ************************************************************************* */
static void
gtk_dir_tree_view_set_property (GObject		*object,
				guint		prop_id,
				const GValue	*value,
				GParamSpec	*pspec) {
  GtkDirTreeView	*treeview;

  treeview = GTK_DIR_TREE_VIEW (object);
  
  switch (prop_id) {
  case PROP_SHOW_HIDDEN:
    gtk_dir_tree_view_set_show_hidden (treeview,
				       (gboolean) g_value_get_boolean (value));
    break;
  default:
    G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
    break;
  }
}

/* ************************************************************************* */
static void
gtk_dir_tree_view_get_property (GObject		*object,
				guint		prop_id,
				GValue		*value,
				GParamSpec	*pspec) {
  GtkDirTreeView	*treeview;

  treeview = GTK_DIR_TREE_VIEW (object);
  
  switch (prop_id) {
  case PROP_TOP_PATH:
    ;
    break;
  case PROP_SHOW_HIDDEN:
    ;
    break;
  default:
    G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
    break;
  }
}

/* åζʬ ********************************************** */
static GtkWidget*
_gtk_dir_tree_view_new (const gchar	*path,
			gboolean	show_hidden) {
  GtkWidget	*treeview;
  GtkTreeStore	*store;
  GtkTreeIter	root_iter;
  GtkTreePath	*tree_path;
  gchar		*top_path, *label;

  store = gtk_tree_store_new (N_COLUMN,
			      G_TYPE_STRING,
			      G_TYPE_STRING,
			      GDK_TYPE_PIXBUF,
			      GDK_TYPE_PIXBUF,
			      G_TYPE_BOOLEAN);

  treeview = gtk_widget_new (gtk_dir_tree_view_get_type (), NULL);
  gtk_tree_view_set_model (GTK_TREE_VIEW (treeview), GTK_TREE_MODEL (store));

  if (strcmp (path, G_DIR_SEPARATOR_S) == 0) {
    top_path = g_strdup (path);
    label    = g_strdup (path);
  } else if (*(path + strlen (path) - 1) == G_DIR_SEPARATOR) {
    top_path = g_strndup (path, strlen (path) - 1);
    label    = g_path_get_basename (top_path);      
  } else {
    top_path = g_strdup (path);
    label    = g_path_get_basename (path);
  } 
  GTK_DIR_TREE_VIEW (treeview)->show_hidden = show_hidden;
  GTK_DIR_TREE_VIEW (treeview)->top_path    = g_strdup (top_path);
  
  insert_row (store, &root_iter, NULL, label, top_path);

  g_object_unref (G_OBJECT (store));
 
  g_signal_connect (G_OBJECT (treeview), "row-expanded",
		    G_CALLBACK (cb_expand), NULL);
  g_signal_connect (G_OBJECT (treeview), "button_press_event",
		    G_CALLBACK (cb_button_press), NULL);

  render_row (GTK_TREE_VIEW (treeview));
  
  gtk_tree_view_set_headers_visible (GTK_TREE_VIEW (treeview), TRUE);

  tree_path = gtk_tree_model_get_path (GTK_TREE_MODEL (store), &root_iter);
  gtk_tree_view_expand_row (GTK_TREE_VIEW (treeview), tree_path, FALSE);
  gtk_tree_path_free (tree_path);

  add_adjust_tree_idle (GTK_TREE_VIEW (treeview), NULL);

  g_free (top_path);
  g_free (label);

  return treeview;
}

/* åȤ ****************************************************** */
GtkWidget*
gtk_dir_tree_view_new (const gchar	*path,
		       gboolean		show_hidden) {
  create_default_icon ();

  return _gtk_dir_tree_view_new (path, show_hidden);
}

/* 饤GdkPixbufǥꤷåȤ *************** */
GtkWidget*
gtk_dir_tree_view_new_from_inline (const gchar	*path,
				   gboolean	show_hidden,
				   const guint8	*folder_pixbuf,
				   const guint8	*folder_open_pixbuf,
				   const guint8 *folder_link_pixbuf,
				   const guint8 *folder_link_open_pixbuf,
				   const guint8	*folder_dennied_pixbuf) {
  create_icon_from_inline (folder_pixbuf,
			   folder_open_pixbuf,
			   folder_link_pixbuf,
			   folder_link_open_pixbuf,
			   folder_dennied_pixbuf);

  return _gtk_dir_tree_view_new (path, show_hidden);  
}

/* եǥꤷåȤ ************************** */
GtkWidget*
gtk_dir_tree_view_new_from_file (const gchar	*path,
				 gboolean	show_hidden,
				 const gchar	*folder_filename,
				 const gchar	*folder_open_filename,
				 const gchar	*folder_link_filename,
				 const gchar	*folder_link_open_filename,
				 const gchar	*folder_dennied_filename) {
  create_icon_from_file (folder_filename,
			 folder_open_filename,
			 folder_link_filename,
			 folder_link_open_filename,
			 folder_dennied_filename);

  return _gtk_dir_tree_view_new (path, show_hidden);
}

/* ǥեȥκ ************************************************ */
static void
create_default_icon (void) {
  folder
    = gdk_pixbuf_new_from_inline (-1, folder_pixbuf, FALSE, NULL);
  folder_open
    = gdk_pixbuf_new_from_inline (-1, folder_open_pixbuf, FALSE, NULL);
  folder_link
    = gdk_pixbuf_new_from_inline (-1, folder_link_pixbuf, FALSE, NULL);
  folder_link_open
    = gdk_pixbuf_new_from_inline (-1, folder_link_open_pixbuf, FALSE, NULL);  
  folder_deny
    = gdk_pixbuf_new_from_inline (-1, folder_dennied_pixbuf, FALSE, NULL);
}

/* 饤GdkPixbufΥ *********************************** */
static void
create_icon_from_inline (const guint8	*folder_pixbuf,
			 const guint8	*folder_open_pixbuf,
			 const guint8	*folder_link_pixbuf,
			 const guint8	*folder_link_open_pixbuf,
			 const guint8	*folder_dennied_pixbuf) {
  folder
    = gdk_pixbuf_new_from_inline (-1, folder_pixbuf, FALSE, NULL);
  folder_open
    = gdk_pixbuf_new_from_inline (-1, folder_open_pixbuf, FALSE, NULL);
  folder_link
    = gdk_pixbuf_new_from_inline (-1, folder_link_pixbuf, FALSE, NULL);
  folder_link_open
    = gdk_pixbuf_new_from_inline (-1, folder_link_open_pixbuf, FALSE, NULL);  
  folder_deny
    = gdk_pixbuf_new_from_inline (-1, folder_dennied_pixbuf, FALSE, NULL);
}

/* ե뤫Υ ********************************************** */
static void
create_icon_from_file (const gchar	*folder_filename,
		       const gchar	*folder_open_filename,
		       const gchar	*folder_link_filename,
		       const gchar	*folder_link_open_filename,
		       const gchar	*folder_dennied_filename) {
  folder	= gdk_pixbuf_new_from_file (folder_filename, NULL);
  folder_open	= gdk_pixbuf_new_from_file (folder_open_filename, NULL);
  folder_link	= gdk_pixbuf_new_from_file (folder_link_filename, NULL);
  folder_link_open
    = gdk_pixbuf_new_from_file (folder_link_open_filename, NULL);  
  folder_deny
    = gdk_pixbuf_new_from_file (folder_dennied_filename, NULL);
}

/* ιԤɲ ************************************************************ */
static void
insert_dummy_row (GtkTreeStore	*store,
		  GtkTreeIter	*parent_iter) {
  GtkTreeIter	iter;
  gboolean	success;

  g_return_if_fail (store);

  gtk_tree_store_prepend (store, &iter, parent_iter);
  gtk_tree_store_set (store, &iter, COLUMN_IS_DUMMY, TRUE, COLUMN_TERMINATOR);

  success = gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (store),
					   &iter, parent_iter, 1);

  if (!success) return;

  while (gtk_tree_store_is_ancestor (store, parent_iter, &iter)) {
    gtk_tree_store_remove (store, &iter);
  }
}

/* (ե)ɲ ****************************************************** */
static void
insert_row (GtkTreeStore	*store,
	    GtkTreeIter		*iter,
	    GtkTreeIter		*parent_iter,
	    const gchar		*label,
	    const gchar		*path) {
  GdkPixbuf	*icon, *oicon;
  gchar		*label_utf8 = NULL;
  
  g_return_if_fail (store);
  g_return_if_fail (iter);
  g_return_if_fail (label && *label);
  g_return_if_fail (path && *path);

  if (!g_file_can_read (path)) {
    icon  = folder_deny;
    oicon = folder_deny;
  } else if (g_file_is_link (path)) {
    icon  = folder_link;
    oicon = folder_link_open;
  } else {
    icon  = folder;
    oicon = folder_open;	
  }
  gtk_tree_store_append (store, iter, parent_iter);
  label_utf8 = g_locale_to_utf8 (label, -1, NULL, NULL, NULL);

  gtk_tree_store_set (store, iter,
		      COLUMN_LABEL,	(label_utf8) ? label_utf8 : label,
		      COLUMN_PATH,	path,
		      COLUMN_ICON_OPEN,	oicon,
		      COLUMN_ICON_CLOSE,icon,
		      COLUMN_IS_DUMMY,	FALSE,
		      COLUMN_TERMINATOR);
  
  insert_dummy_row (store, iter);

  if (label_utf8) g_free (label_utf8);
}

/*  ******************************************************************** */
static void
render_row (GtkTreeView	*treeview) {
  GtkTreeViewColumn	*col;
  GtkCellRenderer	*render;

  gtk_tree_view_set_rules_hint (treeview, FALSE);
  gtk_tree_view_set_rules_hint (treeview, TRUE);
  
  col	 = gtk_tree_view_column_new ();
  render = gtk_cell_renderer_pixbuf_new ();

  gtk_tree_view_column_set_title (col, "Directory Name");
 
  gtk_tree_view_column_pack_start (col, render, FALSE);
  gtk_tree_view_column_add_attribute(col, render, "pixbuf", COLUMN_ICON_CLOSE);
  gtk_tree_view_column_add_attribute(col, render, "pixbuf-expander-open",
				     COLUMN_ICON_OPEN);
  gtk_tree_view_column_add_attribute(col, render, "pixbuf-expander-closed",
				     COLUMN_ICON_CLOSE);

  render = gtk_cell_renderer_text_new ();
  gtk_tree_view_column_pack_start (col, render, TRUE);
  gtk_tree_view_column_add_attribute (col, render, "text", COLUMN_LABEL);

  gtk_tree_view_append_column (treeview, col);
  gtk_tree_view_set_expander_column (treeview, col);
}

/* ************************************************************************* */
static void
adjust_tree (GtkTreeView	*treeview,
	     GtkTreeIter	*iter) {
  GtkTreeModel		*model;
  GtkTreePath		*path;
  GtkTreeSelection	*selection;
  GtkTreeIter		tmp_iter;
  gboolean		success;

  model = gtk_tree_view_get_model (treeview);
  selection = gtk_tree_view_get_selection (treeview);

  if (iter) {
    path	= gtk_tree_model_get_path (model, iter);
    tmp_iter	= *iter;
  } else {
    path	= gtk_tree_path_new_from_string (G_ZERO_STR);
    success	= gtk_tree_model_get_iter (model, &tmp_iter, path);
    if (!success) {
      gtk_tree_path_free (path);
      return;
    }
  }
  gtk_tree_selection_select_path (selection, path);
  gtk_tree_view_scroll_to_cell (treeview, path, NULL, TRUE, 0.0, 0.0);
  gtk_tree_path_free (path);
}

/* ************************************************************************* */
static gint
idle_adjust_tree (gpointer	data) {
  AdjustTreeIdle	*idle = data;

  adjust_tree (idle->treeview, idle->iter);
  g_free (idle->iter);
  g_free (idle);

  return FALSE;
}

/* ************************************************************************* */
static void
add_adjust_tree_idle (GtkTreeView	*treeview,
		      GtkTreeIter	*iter) {
  AdjustTreeIdle	*idle;

  g_return_if_fail (GTK_IS_TREE_VIEW (treeview));

  idle = g_new0 (AdjustTreeIdle, 1);

  idle->treeview = treeview;
  if (iter) {
    idle->iter  = g_new0 (GtkTreeIter, 1);
    *idle->iter = *iter;
  } else {
    idle->iter	= NULL;
  }
  gtk_idle_add (idle_adjust_tree, idle);
}

/* ************************************************************************* *
 * Хåؿ
 * ************************************************************************* */

/* ޥܥ󤬲줿ν ****************************************** */
static gboolean
cb_button_press (GtkWidget	*widget,
		 GdkEventButton	*ev,
		 gpointer	data) {
  GtkTreeModel	*model = NULL;
  GtkTreePath	*path = NULL;
  GtkTreeIter	iter;
  gchar		*selected_path = NULL;
  gboolean	success;
  
  g_return_val_if_fail (ev, FALSE);

  /* ֥륯åλư */
  if (ev->type == GDK_2BUTTON_PRESS) {
    /* 򤵤Ƥѥ */
    selected_path =
      gtk_dir_tree_view_get_selected_path (GTK_DIR_TREE_VIEW (widget));
    if (!selected_path) return FALSE;

    success = get_iter_from_path(GTK_TREE_VIEW (widget), selected_path, &iter);
    if (!success) {
      g_free (selected_path);
      return FALSE;
    }
    model = gtk_tree_view_get_model (GTK_TREE_VIEW (widget));
    path  = gtk_tree_model_get_path (model, &iter);
    
    if (is_expanded (GTK_TREE_VIEW (widget), selected_path)) {
      /* ĥ꡼Ĥ */
      gtk_tree_view_collapse_row (GTK_TREE_VIEW (widget), path);
    } else {
      /* ĥ꡼򳫤 */      
      gtk_tree_view_expand_row (GTK_TREE_VIEW (widget), path, FALSE);
    }
  }
  g_free (selected_path);
  
  return FALSE;
}

/* ************************************************************************* */
static void
cb_expand (GtkDirTreeView	*treeview,
	   GtkTreeIter		*parent_iter,
	   GtkTreePath		*tree_path,
	   gpointer		data) {
  GtkTreeStore	*store;
  GtkTreeIter	child_iter, iter;
  gchar		*parent_dir, *path;
  GList		*dir_list = NULL, *list;
  gboolean	dummy;
  
  store = GTK_TREE_STORE (gtk_tree_view_get_model (GTK_TREE_VIEW (treeview)));

  gtk_tree_model_iter_children (GTK_TREE_MODEL (store),
				&child_iter, parent_iter);
  gtk_tree_model_get (GTK_TREE_MODEL (store),
		      &child_iter, COLUMN_IS_DUMMY, &dummy, COLUMN_TERMINATOR);
  if (!dummy) return;

  gtk_tree_model_get (GTK_TREE_MODEL (store),
		      parent_iter, COLUMN_PATH, &parent_dir,COLUMN_TERMINATOR);
  dir_list = gnome_vfs_ext_ls_dir (parent_dir);

  if (dir_list) {
    for (list = dir_list; list; list = g_list_next (list)) {
      if ((strcmp (((GnomeVFSFileInfo *) list->data)->name, ".")  == 0) ||
	  (strcmp (((GnomeVFSFileInfo *) list->data)->name, "..") == 0) ||
	  (((GnomeVFSFileInfo *) list->data)->type
	   != GNOME_VFS_FILE_TYPE_DIRECTORY) ||
	  (!treeview->show_hidden &&
	   ((GnomeVFSFileInfo *) list->data)->name[0] == '.')) continue;
      path = g_build_filename (parent_dir,
			       ((GnomeVFSFileInfo *) list->data)->name, NULL);
      insert_row (store, &iter, parent_iter,
		  ((GnomeVFSFileInfo *) list->data)->name, path);
      g_free (path);
    }
    gnome_vfs_file_info_list_free (dir_list);
  }
  gtk_tree_store_remove (store, &child_iter);
}

/* ************************************************************************* *
 * ¾Υؿ
 * ************************************************************************* */

/* ************************************************************************* */
static gboolean
get_iter_from_path (GtkTreeView	*treeview,
		    const gchar	*dir,
		    GtkTreeIter	*iter) {
  GtkTreeModel		*model;
  GtkTreePath		*path;
  GtkTreeSelection	*selection;
  GtkTreeIter		parent_iter, child_iter;
  gint			path_len;
  gchar			*label, *dirpath;
  gboolean		next, success = FALSE;

  g_return_val_if_fail (treeview, FALSE);
  g_return_val_if_fail (iter, FALSE);

  model = gtk_tree_view_get_model (treeview);
  selection = gtk_tree_view_get_selection (treeview);
  g_return_val_if_fail (selection, FALSE);

  next = gtk_tree_model_get_iter_from_string (model, &parent_iter, G_ZERO_STR);
  if (!next) return FALSE;

  path = gtk_tree_model_get_path (model, &parent_iter);

  dirpath =
    g_build_filename ((gchar *) gtk_object_get_data (GTK_OBJECT (treeview),
						     "top"), "/", NULL);
  if (strcmp (dir, dirpath) == 0) {
    gtk_tree_path_free (path);
    g_free (dirpath);
    return gtk_tree_model_get_iter_first (model, iter);
  }
  gtk_tree_view_expand_row (treeview, path, FALSE);
  gtk_tree_path_free (path);
  g_free (dirpath);
  dirpath = NULL;
  path    = NULL;

  next =
    gtk_tree_model_get_iter_from_string (model, &parent_iter, G_ZERO_ZERO_STR);

  while (next) {
    gtk_tree_model_get (model, &parent_iter,
			COLUMN_LABEL,	&label,
			COLUMN_PATH,	&dirpath,
			COLUMN_TERMINATOR);
    path_len = strlen (dirpath);
    
    if (strcmp (dir, dirpath) == 0) {
      *iter = parent_iter;
      next = FALSE;
      success = TRUE;
    } else if (strncmp (dir, dirpath, path_len) == 0) {
      path = gtk_tree_model_get_path (model, &parent_iter);
      next = gtk_tree_view_expand_row (treeview, path, FALSE);
      next = gtk_tree_model_iter_children (model, &child_iter, &parent_iter);
      parent_iter = child_iter;
      gtk_tree_path_free (path);
      path = NULL;
    } else {
      next = gtk_tree_model_iter_next (model, &parent_iter);
    }
    g_free (label);
    g_free (dirpath);
  }
  return success;
}


/* ************************************************************************* */
static void
get_expanded_path (GtkTreeView	*treeview,
		   GtkTreePath	*path,
		   gpointer	data) {
  GtkTreeModel	*model;
  GtkTreeIter	iter;
  GList		**list;
  gchar		*dir;

  g_return_if_fail (data);

  list  = data;
  model = gtk_tree_view_get_model (treeview);

  gtk_tree_model_get_iter (model, &iter, path);
  gtk_tree_model_get (model, &iter, COLUMN_PATH, &dir, COLUMN_TERMINATOR);

  g_return_if_fail (dir && *dir);

  *list = g_list_append (*list, dir);
}

/* ************************************************************************* */
static gboolean
is_expanded (GtkTreeView	*treeview,
	     const gchar	*dir) {
  GList		*list = NULL, *node;
  gboolean	success = FALSE;
  
  gtk_tree_view_map_expanded_rows (treeview, get_expanded_path, &list);
  for (node = list; node; node = g_list_next (node)) {
    if (strcmp ((char *) node->data, dir) == 0) {
      success = TRUE;
      break;
    }
  }
  g_list_foreach (list, (GFunc) g_free, NULL);
  g_list_free (list);
  
  return success;
}

/* ************************************************************************* *
 * Хؿ
 * ************************************************************************* */

/* ************************************************************************* */
gboolean
gtk_dir_tree_view_get_show_hidden (GtkDirTreeView	*treeview) {
  g_return_val_if_fail (treeview, FALSE);
  
  return treeview->show_hidden;
}

/* ************************************************************************* */
void
gtk_dir_tree_view_set_show_hidden (GtkDirTreeView	*treeview,
				   gboolean		show_hidden) {
  g_return_if_fail (treeview);
  
  treeview->show_hidden = show_hidden;
}

/* ************************************************************************* */
gchar*
gtk_dir_tree_view_get_selected_path (GtkDirTreeView	*treeview) {
  GtkTreeSelection	*selection;
  GtkTreeModel		*model;
  GtkTreeIter		iter;
  gchar			*path;
  gint			success;
  
  g_return_val_if_fail (treeview, NULL);

  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));
  g_return_val_if_fail (selection, NULL);

  success = gtk_tree_selection_get_selected (selection, &model, &iter);
  if (!success) return NULL;

  gtk_tree_model_get (model, &iter, COLUMN_PATH, &path, COLUMN_TERMINATOR);

  return path;
}

/* ************************************************************************* */
void
gtk_dir_tree_view_expand_dir (GtkDirTreeView	*treeview,
			      const gchar	*dir,
			      gboolean		open_all) {
  GtkTreeModel	*model;
  GtkTreePath	*path;
  GtkTreeIter	iter;
  gboolean	success;

  success = get_iter_from_path (GTK_TREE_VIEW (treeview), dir, &iter);

  if (!success) return;

  model = gtk_tree_view_get_model (GTK_TREE_VIEW (treeview));
  path  = gtk_tree_model_get_path (model, &iter);

  
  gtk_tree_view_expand_row (GTK_TREE_VIEW (treeview), path, open_all);
  gtk_tree_path_free (path);
}

/* ************************************************************************* */
void
gtk_dir_tree_view_set_dir (GtkDirTreeView	*treeview,
			   const gchar		*dir) {
  GtkTreeModel	*model;
  GtkTreeIter	iter;
  gboolean	success;
  gchar		*folder;

  g_return_if_fail (treeview);

  if (!g_dir_is (dir)) return;

  if (strcmp (dir, G_DIR_SEPARATOR_S) == 0) {
    folder = g_strdup (dir);
  } else if (*(dir + strlen (dir) - 1) == G_DIR_SEPARATOR) {
    folder = g_strndup (dir, strlen (dir) - 1);
  } else {
    folder = g_strdup (dir);
  } 
  success = get_iter_from_path (GTK_TREE_VIEW (treeview), folder, &iter);
  model = gtk_tree_view_get_model (GTK_TREE_VIEW (treeview));
  if (!success) gtk_tree_model_get_iter_first (model, &iter);
  add_adjust_tree_idle (GTK_TREE_VIEW (treeview), &iter);

  g_free (folder);
}

/* ************************************************************************* */
void
gtk_dir_tree_view_refresh_tree (GtkDirTreeView	*treeview) {
  GtkTreeStore	*store;
  GtkTreeIter	root_iter, iter;
  gchar		*selected_path, *top_path, *path;
  GList		*list = NULL, *node;
  gboolean	success;

  g_return_if_fail (treeview);
  
  selected_path = gtk_dir_tree_view_get_selected_path (treeview);
  path = g_strdup (treeview->top_path);
  if (strcmp (path, G_DIR_SEPARATOR_S) == 0) {
    top_path = g_strdup (path);
  } else if (*(path + strlen (path) - 1) == G_DIR_SEPARATOR) {
    top_path = g_strndup (path, strlen (path) - 1);
  } else {
    top_path = g_strdup (path);
  } 
  if (!selected_path) selected_path = g_strdup (top_path);

  gtk_tree_view_map_expanded_rows (GTK_TREE_VIEW (treeview),
				   get_expanded_path, &list);
  store = GTK_TREE_STORE (gtk_tree_view_get_model (GTK_TREE_VIEW (treeview)));
  gtk_tree_store_clear (store);
  insert_row (store, &root_iter, NULL, path, top_path);

  for (node = list; node; node = g_list_next (node)) {
    gtk_dir_tree_view_expand_dir (treeview, node->data, FALSE);
  }
  g_list_foreach (list, (GFunc) g_free, NULL);
  g_list_free (list);

  success = get_iter_from_path(GTK_TREE_VIEW (treeview), selected_path, &iter);
  if (success) add_adjust_tree_idle (GTK_TREE_VIEW (treeview), &iter);

  g_free (selected_path);
  g_free (path);
  g_free (top_path);
}

/* ************************************************************************* */
void
gtk_dir_tree_view_set_headers_visible (GtkDirTreeView	*treeview,
				       gboolean		headers_visible) {
  gtk_tree_view_set_headers_visible (GTK_TREE_VIEW (treeview),headers_visible);
}

/* ********************************************* End of gtkdirtreeview.c *** */
