#include <ruby.h>
#include "rbteo.h"
#include <teo.h>
#include <teoutil.h>


VALUE gImage;

static ID id_new;
static ID id_getpixel;
static ID id_putpixel;

static VALUE image_s_alloc(int,VALUE*,VALUE);
static void image_free(TEOIMAGE*);

static VALUE image_width(VALUE);
static VALUE image_height(VALUE);
static VALUE image_xoffset(VALUE);
static VALUE image_yoffset(VALUE);
static VALUE image_xstart(VALUE);
static VALUE image_xend(VALUE);
static VALUE image_ystart(VALUE);
static VALUE image_yend(VALUE);
static VALUE image_type(VALUE);
static VALUE image_bit(VALUE);
static VALUE image_plane(VALUE);
static VALUE image_fsize(VALUE);

static VALUE image_read(VALUE,VALUE);
static VALUE image_write(VALUE,VALUE);

static VALUE image_to_BIT(VALUE);
static VALUE image_to_UINT8(VALUE);
static VALUE image_to_SINT8(VALUE);
static VALUE image_to_UINT16(VALUE);
static VALUE image_to_SINT16(VALUE);
static VALUE image_to_UINT32(VALUE);
static VALUE image_to_SINT32(VALUE);
static VALUE image_to_FLOAT32(VALUE);
static VALUE image_to_FLOAT64(VALUE);
static VALUE image_change_type(VALUE,VALUE);

static VALUE image_getpixel(VALUE,VALUE,VALUE,VALUE);
static VALUE image_putpixel(VALUE,VALUE,VALUE,VALUE,VALUE);

static VALUE image_getpixel2(int,VALUE*,VALUE);
static VALUE image_putpixel2(int,VALUE*,VALUE);

static VALUE image_each_pixel(VALUE);
static VALUE image_each_value(VALUE);

static VALUE image_each_pixel_with_index(VALUE);
static VALUE image_each_value_with_index(VALUE);

static VALUE image_copy(int,VALUE*,VALUE);
static VALUE image_copy_d(int,VALUE*,VALUE);

static VALUE image_dump(VALUE);
static VALUE image_s_load(VALUE,VALUE);

static void
Init_teo_image_type();
static VALUE gImageBIT;
static VALUE gImageUINT8;
static VALUE gImageSINT8;
static VALUE gImageUINT16;
static VALUE gImageSINT16;
static VALUE gImageUINT32;
static VALUE gImageSINT32;
static VALUE gImageFLOAT32;
static VALUE gImageFLOAT64;

void
Init_teo_image(){
  gImage = rb_define_class_under(mTEO,"Image",rb_cObject);
  rb_define_singleton_method(gImage,"new",image_s_alloc,-1);
  rb_define_singleton_method(gImage,"alloc",image_s_alloc,-1);

  rb_define_method(gImage,"width",image_width,0);
  rb_define_method(gImage,"height",image_height,0);
  rb_define_method(gImage,"xoffset",image_xoffset,0);
  rb_define_method(gImage,"yoffset",image_yoffset,0);
  rb_define_method(gImage,"xstart",image_xstart,0);
  rb_define_method(gImage,"xend",image_xend,0);
  rb_define_method(gImage,"ystart",image_ystart,0);
  rb_define_method(gImage,"yend",image_yend,0);
  rb_define_method(gImage,"pixel_type",image_type,0);
  rb_define_method(gImage,"pixel_bit",image_bit,0);
  rb_define_method(gImage,"plane",image_plane,0);
  rb_define_method(gImage,"fsize",image_fsize,0);

  rb_define_method(gImage,"read_frame",image_read,1);
  rb_define_method(gImage,"write_frame",image_write,1);

  rb_define_method(gImage,"to_BIT",image_to_BIT,0);
  rb_define_method(gImage,"to_UINT8",image_to_UINT8,0);
  rb_define_method(gImage,"to_SINT8",image_to_SINT8,0);
  rb_define_method(gImage,"to_UINT16",image_to_UINT16,0);
  rb_define_method(gImage,"to_SINT16",image_to_SINT16,0);
  rb_define_method(gImage,"to_UINT32",image_to_UINT32,0);
  rb_define_method(gImage,"to_SINT32",image_to_SINT32,0);
  rb_define_method(gImage,"to_FLOAT32",image_to_FLOAT32,0);
  rb_define_method(gImage,"to_FLOAT64",image_to_FLOAT64,0);
  rb_define_method(gImage,"change_type",image_change_type,1);

  rb_define_method(gImage,"get_pixel",image_getpixel,3);
  rb_define_method(gImage,"put_pixel",image_putpixel,4);

  rb_define_method(gImage,"[]",image_getpixel2,-1);
  rb_define_method(gImage,"[]=",image_putpixel2,-1);

  rb_define_method(gImage,"each_pixel",image_each_pixel,0);
  rb_define_method(gImage,"each_value",image_each_value,0);

  rb_define_method(gImage,"each_pixel_with_index",
		   image_each_pixel_with_index,0);
  rb_define_method(gImage,"each_value_with_index",
		   image_each_value_with_index,0);

  rb_define_method(gImage,"copy",image_copy,-1);
  rb_define_method(gImage,"copy!",image_copy_d,-1);

  rb_define_method(gImage,"_dump",image_dump,0);
  rb_define_singleton_method(gImage,"_load",image_s_load,1);

  Init_teo_image_type();

  id_new = rb_intern("new");
  id_getpixel = rb_intern("get_pixel");
  id_putpixel = rb_intern("put_pixel");
}

static VALUE image_s_alloc(int argc, VALUE *argv, VALUE image_class){
  VALUE obj;

  VALUE val[7];

  int width=0;
  int height=0;
  int xoffset=0;
  int yoffset=0;
  int type=TEO_UNSIGNED;
  int bit=8;
  int plane=1;

  TEOIMAGE *image;

  rb_scan_args(argc,argv,"16",&val[0],&val[1],&val[2],&val[3],
	       &val[4],&val[5],&val[6]);

  image_class = Qnil;

  if(rb_obj_is_kind_of(val[0],gFile)==Qtrue){
    TEOFILE *orig_file;
    GetTEOFILE(val[0], orig_file);
    image = TeoAllocSimilarImage(orig_file);
  } else if(rb_obj_is_kind_of(val[0],gImage)==Qtrue){
    TEOIMAGE *orig_image;
    Data_Get_Struct(val[0], TEOIMAGE, orig_image);
    image = TeoAllocSimilarImage(orig_image);
 } else  {
    width = NUM2INT(val[0]);
    height = NUM2INT(val[1]);

    if(val[2]!=Qnil) xoffset = NUM2INT(val[2]);
    if(val[3]!=Qnil) yoffset = NUM2INT(val[3]);
    if(val[4]!=Qnil) type = FIX2INT(val[4]);
    if(val[5]!=Qnil) bit = FIX2INT(val[5]);
    if(val[6]!=Qnil) plane = NUM2INT(val[6]);
    image = TeoAllocImage(width, height, xoffset, yoffset, type, bit,plane);
  }
  if(image==NULL) rb_raise(rb_eRuntimeError,TEO_ERROR_MESSAGE);
  switch(TeoBit(image)){
  case 1:
    image_class = gImageBIT;
    break;
  case 8:
    if(TeoType(image)==TEO_UNSIGNED) image_class = gImageUINT8;
    else if(TeoType(image)==TEO_SIGNED) image_class = gImageSINT8;
    break;
  case 16:
    if(TeoType(image)==TEO_UNSIGNED) image_class = gImageUINT16;
    else if(TeoType(image)==TEO_SIGNED) image_class = gImageSINT16;
    break;
  case 32:
    if(TeoType(image)==TEO_UNSIGNED) image_class = gImageUINT32;
    else if(TeoType(image)==TEO_SIGNED) image_class = gImageSINT32;
    else if(TeoType(image)==TEO_FLOAT) image_class = gImageFLOAT32;
    break;
  case 64:
    if(TeoType(image)==TEO_FLOAT) image_class = gImageFLOAT64;
    break;
  }
  obj = Data_Wrap_Struct(image_class, 0, image_free, image);

  return obj;
}

void
image_free(TEOIMAGE *image){
  TeoFreeImage(image);
}


static VALUE
image_read(VALUE self, VALUE fileobj){
  TEOFILE *file;
  TEOIMAGE *image;

  Data_Get_Struct(self, TEOIMAGE, image);
  if(rb_obj_is_kind_of(fileobj,gFile)==Qnil) rb_raise(rb_eTypeError,"Type miss match");
  GetTEOFILE(fileobj, file);
  if(TeoReadFrame(file,image)==0) rb_raise(rb_eIOError,TEO_ERROR_MESSAGE);

  return Qtrue;
}

static VALUE
image_write(VALUE self, VALUE fileobj){
  TEOFILE *file;
  TEOIMAGE *image;
  rb_secure(4);
  Data_Get_Struct(self, TEOIMAGE, image);
  if(TYPE(fileobj)==T_STRING){
    Check_SafeStr(fileobj);
    file = TeoCreateFile(RSTRING(fileobj)->ptr,
			 TeoWidth(image),  TeoHeight(image),
			 TeoXoffset(image),TeoYoffset(image),
			 TeoType(image),   TeoBit(image),
			 TeoPlane(image),  1);
    if(file==NULL) rb_raise(rb_eIOError,TEO_ERROR_MESSAGE);
    if(TeoWriteFrame(file,image)==0)
      rb_raise(rb_eIOError,TEO_ERROR_MESSAGE);;
    TeoCloseFile(file);
    return Qtrue;
  }
  if(rb_obj_is_kind_of(fileobj,gFile)==Qnil) rb_raise(rb_eTypeError,"Type miss match");
  GetTEOFILE(fileobj, file);
  if(TeoWriteFrame(file,image)==0)rb_raise(rb_eIOError,TEO_ERROR_MESSAGE);
  return Qtrue;
}


static VALUE image_width(VALUE self){
  TEOIMAGE *image;
  Data_Get_Struct(self, TEOIMAGE, image);
  return INT2NUM(TeoWidth(image));
}
static VALUE image_height(VALUE self){
  TEOIMAGE *image;
  Data_Get_Struct(self, TEOIMAGE, image);
  return INT2NUM(TeoHeight(image));
}
static VALUE image_xoffset(VALUE self){
  TEOIMAGE *image;
  Data_Get_Struct(self, TEOIMAGE, image);
  return INT2NUM(TeoXoffset(image));
}
static VALUE image_yoffset(VALUE self){
  TEOIMAGE *image;
  Data_Get_Struct(self, TEOIMAGE, image);
  return INT2NUM(TeoYoffset(image));
}
static VALUE image_xstart(VALUE self){
  TEOIMAGE *image;
  Data_Get_Struct(self, TEOIMAGE, image);
  return INT2NUM(TeoXstart(image));
}
static VALUE image_xend(VALUE self){
  TEOIMAGE *image;
  Data_Get_Struct(self, TEOIMAGE, image);
  return INT2NUM(TeoXend(image));
}
static VALUE image_ystart(VALUE self){
  TEOIMAGE *image;
  Data_Get_Struct(self, TEOIMAGE, image);
  return INT2NUM(TeoYstart(image));
}
static VALUE image_yend(VALUE self){
  TEOIMAGE *image;
  Data_Get_Struct(self, TEOIMAGE, image);
  return INT2NUM(TeoYend(image));
}
static VALUE image_type(VALUE self){
  TEOIMAGE *image;
  Data_Get_Struct(self, TEOIMAGE, image);
  return INT2NUM(TeoType(image));
}
static VALUE image_typestring(VALUE self){
  TEOIMAGE *image;
  Data_Get_Struct(self, TEOIMAGE, image);
  if(TeoBit(image)==1){
    return rb_str_new2("BIT");
  } else if(TeoBit(image)==8) {
    if(TeoType(image)==TEO_UNSIGNED){
      return rb_str_new2("UINT8");
    } else if(TeoType(image)==TEO_SIGNED){
    return rb_str_new2("SINT8");
    }
  } else if(TeoBit(image)==16) {
    if(TeoType(image)==TEO_UNSIGNED){
      return rb_str_new2("UINT16");
    } else if(TeoType(image)==TEO_SIGNED){
      return rb_str_new2("SINT16");
    }
  } else if(TeoBit(image)==32) {
    if(TeoType(image)==TEO_UNSIGNED){
      return rb_str_new2("UINT32");
    } else if(TeoType(image)==TEO_SIGNED){
      return rb_str_new2("SINT32");
    } else if(TeoType(image)==TEO_FLOAT){
      return rb_str_new2("FLOAT32");
    }
  } else if(TeoBit(image)==64) {
    if(TeoType(image)==TEO_FLOAT){
      return rb_str_new2("FLOAT64");
    }
  }
  return Qnil;
}
static VALUE image_bit(VALUE self){
  TEOIMAGE *image;
  Data_Get_Struct(self, TEOIMAGE, image);
return INT2NUM(TeoBit(image));
}
static VALUE image_plane(VALUE self){
  TEOIMAGE *image;
  Data_Get_Struct(self, TEOIMAGE, image);
  return INT2NUM(TeoPlane(image));
}
static VALUE image_fsize(VALUE self){
  TEOIMAGE *image;
  Data_Get_Struct(self, TEOIMAGE, image);
  return INT2NUM(TeoFsize(image));
}

static VALUE image_to_BIT(VALUE self){
  TEOIMAGE *self_image;
  TEOIMAGE *image;

  Data_Get_Struct(self, TEOIMAGE, self_image);
  image = TeoAllocImage(TeoWidth(self_image),TeoHeight(self_image),
			TeoXoffset(self_image),TeoYoffset(self_image),
			TEO_UNSIGNED, 1,
			TeoPlane(self_image));
  if(image==NULL) rb_raise(rb_eRuntimeError,TEO_ERROR_MESSAGE);
  TeoUtilCopyImageWithCast(self_image,image);
  return Data_Wrap_Struct(gImageBIT, 0, image_free, image);
}
static VALUE image_to_UINT8(VALUE self){
  TEOIMAGE *self_image;
  TEOIMAGE *image;

  Data_Get_Struct(self, TEOIMAGE, self_image);
  image = TeoAllocImage(TeoWidth(self_image),TeoHeight(self_image),
			TeoXoffset(self_image),TeoYoffset(self_image),
			TEO_UNSIGNED, 8,
			TeoPlane(self_image));
  if(image==NULL) rb_raise(rb_eRuntimeError,TEO_ERROR_MESSAGE);
  TeoUtilCopyImageWithCast(self_image,image);
  return Data_Wrap_Struct(gImageUINT8, 0, image_free, image);
}
static VALUE image_to_SINT8(VALUE self){
  TEOIMAGE *self_image;
  TEOIMAGE *image;

  Data_Get_Struct(self, TEOIMAGE, self_image);
  image = TeoAllocImage(TeoWidth(self_image),TeoHeight(self_image),
			TeoXoffset(self_image),TeoYoffset(self_image),
			TEO_SIGNED, 8,
			TeoPlane(self_image));
  if(image==NULL) rb_raise(rb_eRuntimeError,TEO_ERROR_MESSAGE);
  TeoUtilCopyImageWithCast(self_image,image);
  return Data_Wrap_Struct(gImageSINT8, 0, image_free, image);
}
static VALUE image_to_UINT16(VALUE self){
  TEOIMAGE *self_image;
  TEOIMAGE *image;

  Data_Get_Struct(self, TEOIMAGE, self_image);
  image = TeoAllocImage(TeoWidth(self_image),TeoHeight(self_image),
			TeoXoffset(self_image),TeoYoffset(self_image),
			TEO_UNSIGNED, 16,
			TeoPlane(self_image));
  if(image==NULL) rb_raise(rb_eRuntimeError,TEO_ERROR_MESSAGE);
  TeoUtilCopyImageWithCast(self_image,image);
  return Data_Wrap_Struct(gImageUINT16, 0, image_free, image);
}
static VALUE image_to_SINT16(VALUE self){
  TEOIMAGE *self_image;
  TEOIMAGE *image;

  Data_Get_Struct(self, TEOIMAGE, self_image);
  image = TeoAllocImage(TeoWidth(self_image),TeoHeight(self_image),
			TeoXoffset(self_image),TeoYoffset(self_image),
			TEO_SIGNED, 16,
			TeoPlane(self_image));
  if(image==NULL) rb_raise(rb_eRuntimeError,TEO_ERROR_MESSAGE);
  TeoUtilCopyImageWithCast(self_image,image);
  return Data_Wrap_Struct(gImageSINT16, 0, image_free, image);
}
static VALUE image_to_UINT32(VALUE self){
  TEOIMAGE *self_image;
  TEOIMAGE *image;

  Data_Get_Struct(self, TEOIMAGE, self_image);
  image = TeoAllocImage(TeoWidth(self_image),TeoHeight(self_image),
			TeoXoffset(self_image),TeoYoffset(self_image),
			TEO_UNSIGNED, 32,
			TeoPlane(self_image));
  if(image==NULL) rb_raise(rb_eRuntimeError,TEO_ERROR_MESSAGE);
  TeoUtilCopyImageWithCast(self_image,image);
  return Data_Wrap_Struct(gImageUINT32, 0, image_free, image);
}
static VALUE image_to_SINT32(VALUE self){
  TEOIMAGE *self_image;
  TEOIMAGE *image;

  Data_Get_Struct(self, TEOIMAGE, self_image);
  image = TeoAllocImage(TeoWidth(self_image),TeoHeight(self_image),
			TeoXoffset(self_image),TeoYoffset(self_image),
			TEO_UNSIGNED, 32,
			TeoPlane(self_image));
  if(image==NULL) rb_raise(rb_eRuntimeError,TEO_ERROR_MESSAGE);
  TeoUtilCopyImageWithCast(self_image,image);
  return Data_Wrap_Struct(gImageSINT32, 0, image_free, image);
}
static VALUE image_to_FLOAT32(VALUE self){
  TEOIMAGE *self_image;
  TEOIMAGE *image;

  Data_Get_Struct(self, TEOIMAGE, self_image);
  image = TeoAllocImage(TeoWidth(self_image),TeoHeight(self_image),
			TeoXoffset(self_image),TeoYoffset(self_image),
			TEO_FLOAT, 32,
			TeoPlane(self_image));
  if(image==NULL) rb_raise(rb_eRuntimeError,TEO_ERROR_MESSAGE);
  TeoUtilCopyImageWithCast(self_image,image);
  return Data_Wrap_Struct(gImageFLOAT32, 0, image_free, image);
}
static VALUE image_to_FLOAT64(VALUE self){
  TEOIMAGE *self_image;
  TEOIMAGE *image;

  Data_Get_Struct(self, TEOIMAGE, self_image);
  image = TeoAllocImage(TeoWidth(self_image),TeoHeight(self_image),
			TeoXoffset(self_image),TeoYoffset(self_image),
			TEO_FLOAT, 64,
			TeoPlane(self_image));
  if(image==NULL) rb_raise(rb_eRuntimeError,TEO_ERROR_MESSAGE);
  TeoUtilCopyImageWithCast(self_image,image);
  return Data_Wrap_Struct(gImageFLOAT64, 0, image_free, image);
}

static VALUE
image_change_type(VALUE self, VALUE type){
  char *typestring;
  
  Check_Type(type,T_STRING);
  typestring = RSTRING(type)->ptr;
  if(strcmp(typestring,"BIT")){
    return to_BIT(self);
  } else if(strcmp(typestring,"UINT8")){
    return to_UINT8(self);
  } else if(strcmp(typestring,"SINT8")){
    return to_SINT8(self);
  } else if(strcmp(typestring,"UINT16")){
    return to_UINT16(self);
  } else if(strcmp(typestring,"SINT16")){
    return to_SINT16(self);
  } else if(strcmp(typestring,"UINT32")){
    return to_UINT32(self);
  } else if(strcmp(typestring,"SINT32")){
    return to_SINT32(self);
  } else if(strcmp(typestring,"FLOAT32")){
    return to_FLOAT32(self);
  } else if(strcmp(typestring,"FLOAT64")){
    return to_FLOAT64(self);
  }
}

static VALUE image_getpixel(VALUE self,VALUE in_x,VALUE in_y,VALUE in_p){
  return Qfalse;
}

static VALUE image_putpixel(VALUE self,VALUE in_x,VALUE in_y,VALUE in_p,VALUE val){
  return Qfalse;
}
static VALUE image_getpixel2(int argc,VALUE *argv, VALUE self){
  return Qfalse;
}

static VALUE image_putpixel2(int argc,VALUE *argv, VALUE self){
  return Qfalse;
}

static VALUE image_each_pixel(VALUE self){
  return Qfalse;
}

static VALUE image_each_value(VALUE self){
  return Qfalse;
}

static VALUE image_each_pixel_with_index(VALUE self){
  return Qfalse;
}

static VALUE image_each_value_with_index(VALUE self){
  return Qfalse;
}

static VALUE image_copy(int argc, VALUE *argv, VALUE self){
  VALUE obj;
  TEOIMAGE *src,*dst;
  VALUE val[6];

  int src_x,src_y,width,height,dst_x,dst_y;

  Data_Get_Struct(self,TEOIMAGE,src);

  rb_scan_args(argc,argv,"06",
	       &val[0],&val[1],&val[2],&val[3],&val[4],&val[5]);

  if(val[0] == Qnil){
    obj = rb_funcall(gImage,id_new,1,self);

    Data_Get_Struct(obj,TEOIMAGE,dst);

    memcpy(TeoData(dst),TeoData(src),TeoFsize(src));
    return obj;
  }
  src_x = NUM2INT(val[0]);
  if(src_x < TeoXstart(src)) src_x = TeoXstart(src);
  if(src_x > TeoXend(src))   src_x = TeoXend(src);

  src_y = (val[1]==Qnil)? TeoXstart(src) : NUM2INT(val[1]);
  if(src_y < TeoXstart(src)) src_y = TeoXstart(src);
  if(src_y > TeoXend(src))   src_y = TeoXend(src);

  width = (val[2]==Qnil)? TeoXend(src)-src_x+1 : NUM2INT(val[2]);
  if(width>TeoXend(src)-src_x+1) TeoXend(src)-src_x+1;

  height = (val[3]==Qnil)? TeoYend(src)-src_y+1 : NUM2INT(val[3]);
  if(height>TeoYend(src)-src_y+1) TeoYend(src)-src_y+1;

  dst_x  = (val[4]==Qnil)? src_x : NUM2INT(val[4]);
  dst_y  = (val[5]==Qnil)? src_y : NUM2INT(val[5]);

  obj = rb_funcall(gImage,id_new,7,
		   INT2NUM(width),INT2NUM(height),
		   INT2NUM(dst_x),INT2NUM(dst_y),
		   INT2FIX(TeoType(src)),
		   INT2FIX(TeoBit(src)),
		   INT2NUM(TeoPlane(src)));
  Data_Get_Struct(obj,TEOIMAGE,dst);
  TeoUtilCopyArea(src,dst,src_x,src_y,width,height,dst_x,dst_y);
  return obj;
}

static VALUE image_copy_d(int argc,VALUE *argv,VALUE self){
  TEOIMAGE *src,*dst;
  VALUE val[7];

  int src_x,src_y,width,height,dst_x,dst_y,plane;
  int x,y,p;

  rb_scan_args(argc,argv,"16",
	       &val[0],&val[1],&val[2],&val[3],&val[4],&val[5],&val[6]);

  Data_Get_Struct(self,TEOIMAGE,dst);
  Data_Get_Struct(val[0],TEOIMAGE,src);
  if(val[1]==Qnil){
    if(TeoUtilCopyImageWithCast(src,dst)==1) return self;
  }

  src_x = NUM2INT(val[1]);
  if(src_x < TeoXstart(src)) src_x = TeoXstart(src);
  if(src_x > TeoXend(src))   src_x = TeoXend(src);

  src_y = (val[2]==Qnil)? TeoXstart(src) : NUM2INT(val[2]);
  if(src_y < TeoXstart(src)) src_y = TeoXstart(src);
  if(src_y > TeoXend(src))   src_y = TeoXend(src);

  width = (val[3]==Qnil)? TeoXend(src)-src_x+1 : NUM2INT(val[3]);
  if(width>TeoXend(src)-src_x+1) TeoXend(src)-src_x+1;

  height = (val[4]==Qnil)? TeoYend(src)-src_y+1 : NUM2INT(val[4]);
  if(height>TeoYend(src)-src_y+1) TeoYend(src)-src_y+1;

  dst_x  = (val[5]==Qnil)? src_x : NUM2INT(val[5]);
  dst_y  = (val[6]==Qnil)? src_y : NUM2INT(val[6]);

  plane = (TeoPlane(src) < TeoPlane(dst))? TeoPlane(src):TeoPlane(dst);

  for(y=0;y<width;y++)
    for(x=0;x<height;x++)
      for(p=0;p<plane;p++)
	rb_funcall(self,id_putpixel,4,
		   INT2NUM(dst_x+x),INT2NUM(dst_y+y),INT2NUM(p),
		   rb_funcall(val[0],id_getpixel,3,
			      INT2NUM(src_x+x),INT2NUM(src_y+y),INT2NUM(p)));
  return self;
}

static VALUE image_dump(VALUE self){
  TEOIMAGE *image;
  char *ptr;
  Data_Get_Struct(self,TEOIMAGE,image);
  ptr = (char*)malloc(sizeof(TEOIMAGE)+TeoFsize(image));
  memcpy(ptr,image,sizeof(TEOIMAGE));
  memcpy(ptr+sizeof(TEOIMAGE),TeoData(image),TeoFsize(image));
  return rb_str_new(ptr,sizeof(TEOIMAGE)+TeoFsize(image));
}
static VALUE image_s_load(VALUE image_class, VALUE str){
  TEOIMAGE *image;
  image = (TEOIMAGE*)malloc(sizeof(TEOIMAGE));
  memcpy(image,RSTRING(str)->ptr,sizeof(TEOIMAGE));
  TeoData(image) = malloc(TeoFsize(image));
  memcpy(TeoData(image),RSTRING(str)->ptr+sizeof(TEOIMAGE),TeoFsize(image));
  return Data_Wrap_Struct(image_class, 0, image_free, image);
}

static VALUE BIT_s_alloc(int,VALUE*,VALUE);
static VALUE BIT_to_BIT(VALUE);

static VALUE BIT_getpixel(VALUE,VALUE,VALUE,VALUE);
static VALUE BIT_putpixel(VALUE,VALUE,VALUE,VALUE,VALUE);

static VALUE BIT_getpixel2(int,VALUE*,VALUE);
static VALUE BIT_putpixel2(int,VALUE*,VALUE);

static VALUE BIT_each_pixel(VALUE);
static VALUE BIT_each_value(VALUE);

static VALUE BIT_each_pixel_with_index(VALUE);
static VALUE BIT_each_value_with_index(VALUE);

void
Init_teo_image_BIT(){
  gImageBIT = rb_define_class_under(gImage,"BIT",gImage);
  rb_define_singleton_method(gImageBIT,"new",BIT_s_alloc,-1);
  rb_define_singleton_method(gImageBIT,"alloc",BIT_s_alloc,-1);

  rb_define_method(gImageBIT,"to_BIT",BIT_to_BIT,0);

  rb_define_method(gImageBIT,"get_pixel",BIT_getpixel,3);
  rb_define_method(gImageBIT,"put_pixel",BIT_putpixel,4);

  rb_define_method(gImageBIT,"[]",BIT_getpixel2,-1);
  rb_define_method(gImageBIT,"[]=",BIT_putpixel2,-1);

  rb_define_method(gImageBIT,"each_pixel",BIT_each_pixel,0);
  rb_define_method(gImageBIT,"each_value",BIT_each_value,0);

  rb_define_method(gImageBIT,"each_pixel_with_index",BIT_each_pixel_with_index,0);
  rb_define_method(gImageBIT,"each_value_with_index",BIT_each_value_with_index,0);
}

static VALUE
BIT_s_alloc(int argc, VALUE *argv, VALUE image_class){
  VALUE obj;

  VALUE val[5];

  int width=0;
  int height=0;
  int xoffset=0;
  int yoffset=0;
  int plane=1;

  TEOIMAGE *image;

  rb_scan_args(argc,argv,"14",&val[0],&val[1],&val[2],&val[3],&val[4]);

  if(rb_obj_is_kind_of(val[0],gFile)==Qtrue){
    TEOFILE *orig_file;
    Data_Get_Struct(val[0], TEOFILE, orig_file);
    image = TeoAllocImage(TeoWidth(orig_file),TeoHeight(orig_file),
			  TeoXoffset(orig_file),TeoYoffset(orig_file),
			  TEO_UNSIGNED,1,TeoPlane(orig_file));
    if(image==NULL) rb_raise(rb_eRuntimeError,TEO_ERROR_MESSAGE);
  } else if(rb_obj_is_kind_of(val[0],gImage)==Qtrue){
    TEOIMAGE *orig_image;
    Data_Get_Struct(val[0], TEOIMAGE, orig_image);
    image = TeoAllocImage(TeoWidth(orig_image),TeoHeight(orig_image),
			  TeoXoffset(orig_image),TeoYoffset(orig_image),
			  TEO_UNSIGNED,1,TeoPlane(orig_image));
    if(image==NULL) rb_raise(rb_eRuntimeError,TEO_ERROR_MESSAGE);
  } else  {
    width = NUM2INT(val[0]);
    height = NUM2INT(val[1]);

    if(val[2]!=Qnil) xoffset = NUM2INT(val[2]);
    if(val[3]!=Qnil) yoffset = NUM2INT(val[3]);
    if(val[4]!=Qnil) plane = NUM2INT(val[4]);
    image = TeoAllocImage(width, height, xoffset, yoffset,
			  TEO_UNSIGNED, 1, plane);
    if(image==NULL) rb_raise(rb_eRuntimeError,TEO_ERROR_MESSAGE);
  }
  obj = Data_Wrap_Struct(image_class, 0, image_free, image);

  return obj;
}

static VALUE
BIT_to_BIT(VALUE self){
  return self;
}

static VALUE BIT_getpixel(VALUE self,VALUE in_x,VALUE in_y,VALUE in_p){
  TEOIMAGE *image;
  Data_Get_Struct(self, TEOIMAGE, image);
  return INT2FIX(TeoGetBit(image,NUM2INT(in_x),NUM2INT(in_y),NUM2INT(in_p)));
}
static VALUE BIT_putpixel(VALUE self,VALUE in_x,VALUE in_y,VALUE in_p,VALUE val){
  TEOIMAGE *image;
  Data_Get_Struct(self, TEOIMAGE, image);
  TeoPutBit(image,NUM2INT(in_x),NUM2INT(in_y),NUM2INT(in_p),NUM2INT(val));
  return val;
}
static VALUE BIT_getpixel2(int argc,VALUE *argv, VALUE self){
  TEOIMAGE *image;
  int x=0,y=0,p=0;
  VALUE val[3];
  VALUE ary;

  Data_Get_Struct(self, TEOIMAGE, image);

  rb_scan_args(argc,argv,"21",&val[0],&val[1],&val[2]);

  x = NUM2INT(val[0]);
  y = NUM2INT(val[1]);

  if(x<TeoXstart(image)||x>TeoXend(image)||
     y<TeoYstart(image)||y>TeoYend(image))
    rb_raise(rb_eIndexError,"Out of range x=%d[%d:%d],y=%d[%d:%d]",
	     x,TeoXstart(image),TeoXend(image),
	     y,TeoYstart(image),TeoYend(image));

  if(val[2]!=Qnil){
    p = NUM2INT(val[2]);
    if(p<0||p>=TeoPlane(image))
      rb_raise(rb_eIndexError,"Out of range p=%d[%d:%d]",
	       p,0,TeoPlane(image));
    return INT2FIX(TeoGetBit(image,x,y,p));
  } else {
    ary = rb_ary_new2(TeoPlane(image));
    for(p=0;p<TeoPlane(image);p++)
      rb_ary_store(ary,p,INT2FIX(TeoGetBit(image,x,y,p)));
    return ary;
  }
}

static VALUE BIT_putpixel2(int argc,VALUE *argv, VALUE self){
  TEOIMAGE *image;
  int x=0,y=0,p=0;
  VALUE val[4];
  VALUE pixel;

  Data_Get_Struct(self, TEOIMAGE, image);

  rb_scan_args(argc,argv,"31",&val[0],&val[1],&val[2],&val[3]);

  x = NUM2INT(val[0]);
  y = NUM2INT(val[1]);

  if(x<TeoXstart(image)||x>TeoXend(image)||
     y<TeoYstart(image)||y>TeoYend(image))
    rb_raise(rb_eIndexError,"Out of range x=%d[%d:%d],y=%d[%d:%d]",
	     x,TeoXstart(image),TeoXend(image),
	     y,TeoYstart(image),TeoYend(image));
  if(val[3]!=Qnil){
    p = NUM2INT(val[2]);
    if(p<0||p>=TeoPlane(image))
      rb_raise(rb_eIndexError,"Out of range p=%d[%d:%d]",
	       p,0,TeoPlane(image));
    TeoPutBit(image,x,y,p,NUM2INT(val[3]));
  } else if(TYPE(val[2])==T_ARRAY){
    pixel = val[2];
    while(RARRAY(pixel)->len<TeoPlane(image)) rb_ary_push(pixel,INT2FIX(0));
    for(p=0;p<TeoPlane(image);p++)
      TeoPutBit(image,x,y,p,NUM2INT(rb_ary_entry(pixel,p)));
    return pixel;
  } else {
    int value;
    value = NUM2INT(val[2]);
    for(p=0;p<TeoPlane(image);p++)
      TeoPutBit(image,x,y,p,value);
    return val[2];
  }
}

static VALUE BIT_each_pixel(VALUE self){
  TEOIMAGE *image;
  int xstart,xend,ystart,yend,plane;
  int x=0,y=0,p=0;
  VALUE ary;

  Data_Get_Struct(self, TEOIMAGE, image);

  xstart = TeoXstart(image);
  xend   = TeoXend(image);
  ystart = TeoYstart(image);
  yend   = TeoYend(image);
  plane  = TeoPlane(image);

  ary = rb_ary_new2(plane);

  for(y=ystart;y<=yend;y++)
    for(x=xstart;x<=xend;x++){
      for(p=0;p<plane;p++)
	rb_ary_store(ary,p,INT2FIX(TeoGetBit(image,x,y,p)));
      rb_yield(ary);
    }
  return Qtrue;
}

static VALUE BIT_each_value(VALUE self){
  TEOIMAGE *image;
  int xstart,xend,ystart,yend,plane;
  int x=0,y=0,p=0;
  VALUE value;

  Data_Get_Struct(self, TEOIMAGE, image);

  xstart = TeoXstart(image);
  xend   = TeoXend(image);
  ystart = TeoYstart(image);
  yend   = TeoYend(image);
  plane  = TeoPlane(image);

  for(y=ystart;y<=yend;y++)
    for(x=xstart;x<=xend;x++)
      for(p=0;p<plane;p++){
	value = INT2FIX(TeoGetBit(image,x,y,p));
	rb_yield(value);
      }
  return Qtrue;
}

static VALUE BIT_each_pixel_with_index(VALUE self){
  TEOIMAGE *image;
  int xstart,xend,ystart,yend,plane;
  int x=0,y=0,p=0;
  VALUE ary;

  Data_Get_Struct(self, TEOIMAGE, image);

  xstart = TeoXstart(image);
  xend   = TeoXend(image);
  ystart = TeoYstart(image);
  yend   = TeoYend(image);
  plane  = TeoPlane(image);

  ary = rb_ary_new2(plane);

  for(y=ystart;y<=yend;y++)
    for(x=xstart;x<=xend;x++){
      for(p=0;p<plane;p++)
	rb_ary_store(ary,p,INT2FIX(TeoGetBit(image,x,y,p)));
      rb_yield(rb_ary_new3(3,ary,INT2NUM(x),INT2NUM(y)));
    }
  return Qtrue;
}

static VALUE BIT_each_value_with_index(VALUE self){
  TEOIMAGE *image;
  int xstart,xend,ystart,yend,plane;
  int x=0,y=0,p=0;
  VALUE value;

  Data_Get_Struct(self, TEOIMAGE, image);

  xstart = TeoXstart(image);
  xend   = TeoXend(image);
  ystart = TeoYstart(image);
  yend   = TeoYend(image);
  plane  = TeoPlane(image);

  for(y=ystart;y<=yend;y++)
    for(x=xstart;x<=xend;x++)
      for(p=0;p<plane;p++){
	value = INT2FIX(TeoGetBit(image,x,y,p));
	rb_yield(rb_ary_new3(4,value,INT2NUM(x),INT2NUM(y),INT2NUM(p)));
      }
  return Qtrue;
}

static VALUE UINT8_s_alloc(int,VALUE*,VALUE);
static VALUE UINT8_to_UINT8(VALUE);

static VALUE UINT8_getpixel(VALUE,VALUE,VALUE,VALUE);
static VALUE UINT8_putpixel(VALUE,VALUE,VALUE,VALUE,VALUE);

static VALUE UINT8_getpixel2(int,VALUE*,VALUE);
static VALUE UINT8_putpixel2(int,VALUE*,VALUE);

static VALUE UINT8_each_pixel(VALUE);
static VALUE UINT8_each_value(VALUE);

static VALUE UINT8_each_pixel_with_index(VALUE);
static VALUE UINT8_each_value_with_index(VALUE);

void
Init_teo_image_UINT8(){
  gImageUINT8 = rb_define_class_under(gImage,"UINT8",gImage);
  rb_define_singleton_method(gImageUINT8,"new",UINT8_s_alloc,-1);
  rb_define_singleton_method(gImageUINT8,"alloc",UINT8_s_alloc,-1);

  rb_define_method(gImageUINT8,"to_UINT8",UINT8_to_UINT8,0);

  rb_define_method(gImageUINT8,"get_pixel",UINT8_getpixel,3);
  rb_define_method(gImageUINT8,"put_pixel",UINT8_putpixel,4);

  rb_define_method(gImageUINT8,"[]",UINT8_getpixel2,-1);
  rb_define_method(gImageUINT8,"[]=",UINT8_putpixel2,-1);

  rb_define_method(gImageUINT8,"each_pixel",UINT8_each_pixel,0);
  rb_define_method(gImageUINT8,"each_value",UINT8_each_value,0);

  rb_define_method(gImageUINT8,"each_pixel_with_index",UINT8_each_pixel_with_index,0);
  rb_define_method(gImageUINT8,"each_value_with_index",UINT8_each_value_with_index,0);
}

static VALUE
UINT8_s_alloc(int argc, VALUE *argv, VALUE image_class){
  VALUE obj;

  VALUE val[5];

  int width=0;
  int height=0;
  int xoffset=0;
  int yoffset=0;
  int plane=1;

  TEOIMAGE *image;

  rb_scan_args(argc,argv,"14",&val[0],&val[1],&val[2],&val[3],&val[4]);

  if(rb_obj_is_kind_of(val[0],gFile)==Qtrue){
    TEOFILE *orig_file;
    Data_Get_Struct(val[0], TEOFILE, orig_file);
    image = TeoAllocImage(TeoWidth(orig_file),TeoHeight(orig_file),
			  TeoXoffset(orig_file),TeoYoffset(orig_file),
			  TEO_UNSIGNED,8,TeoPlane(orig_file));
    if(image==NULL) rb_raise(rb_eRuntimeError,TEO_ERROR_MESSAGE);
  } else if(rb_obj_is_kind_of(val[0],gImage)==Qtrue){
    TEOIMAGE *orig_image;
    Data_Get_Struct(val[0], TEOIMAGE, orig_image);
    image = TeoAllocImage(TeoWidth(orig_image),TeoHeight(orig_image),
			  TeoXoffset(orig_image),TeoYoffset(orig_image),
			  TEO_UNSIGNED,8,TeoPlane(orig_image));
    if(image==NULL) rb_raise(rb_eRuntimeError,TEO_ERROR_MESSAGE);
  } else  {
    width = NUM2INT(val[0]);
    height = NUM2INT(val[1]);

    if(val[2]!=Qnil) xoffset = NUM2INT(val[2]);
    if(val[3]!=Qnil) yoffset = NUM2INT(val[3]);
    if(val[4]!=Qnil) plane = NUM2INT(val[4]);
    image = TeoAllocImage(width, height, xoffset, yoffset,
			  TEO_UNSIGNED, 8, plane);
    if(image==NULL) rb_raise(rb_eRuntimeError,TEO_ERROR_MESSAGE);
  }
  obj = Data_Wrap_Struct(image_class, 0, image_free, image);

  return obj;
}

static VALUE
UINT8_to_UINT8(VALUE self){
  return self;
}

static VALUE UINT8_getpixel(VALUE self,VALUE in_x,VALUE in_y,VALUE in_p){
  TEOIMAGE *image;
  Data_Get_Struct(self, TEOIMAGE, image);
  return INT2FIX(TeoGetPixel(image,NUM2INT(in_x),NUM2INT(in_y),NUM2INT(in_p),
		 TEO_UINT8));
}
static VALUE UINT8_putpixel(VALUE self,VALUE in_x,VALUE in_y,VALUE in_p,VALUE val){
  TEOIMAGE *image;
  Data_Get_Struct(self, TEOIMAGE, image);
  TeoPutPixel(image,NUM2INT(in_x),NUM2INT(in_y),NUM2INT(in_p),
	      TEO_UINT8,NUM2INT(val));
  return val;
}
static VALUE UINT8_getpixel2(int argc,VALUE *argv, VALUE self){
  TEOIMAGE *image;
  int x=0,y=0,p=0;
  VALUE val[3];
  VALUE ary;

  Data_Get_Struct(self, TEOIMAGE, image);

  rb_scan_args(argc,argv,"21",&val[0],&val[1],&val[2]);

  x = NUM2INT(val[0]);
  y = NUM2INT(val[1]);

  if(x<TeoXstart(image)||x>TeoXend(image)||
     y<TeoYstart(image)||y>TeoYend(image))
    rb_raise(rb_eIndexError,"Out of range x=%d[%d:%d],y=%d[%d:%d]",
	     x,TeoXstart(image),TeoXend(image),
	     y,TeoYstart(image),TeoYend(image));
  if(val[2]!=Qnil){
    p = NUM2INT(val[2]);
    if(p<0||p>=TeoPlane(image))
      rb_raise(rb_eIndexError,"Out of range p=%d[%d:%d]",
	       p,0,TeoPlane(image));
    return INT2FIX(TeoGetPixel(image,x,y,p,TEO_UINT8));
  } else {
    ary = rb_ary_new2(TeoPlane(image));
    for(p=0;p<TeoPlane(image);p++)
      rb_ary_store(ary,p,INT2FIX(TeoGetPixel(image,x,y,p,TEO_UINT8)));
    return ary;
  }
}

static VALUE UINT8_putpixel2(int argc,VALUE *argv, VALUE self){
  TEOIMAGE *image;
  int x=0,y=0,p=0;
  VALUE val[4];
  VALUE pixel;

  Data_Get_Struct(self, TEOIMAGE, image);

  rb_scan_args(argc,argv,"31",&val[0],&val[1],&val[2],&val[3]);

  x = NUM2INT(val[0]);
  y = NUM2INT(val[1]);

  if(x<TeoXstart(image)||x>TeoXend(image)||
     y<TeoYstart(image)||y>TeoYend(image))
    rb_raise(rb_eIndexError,"Out of range x=%d[%d:%d],y=%d[%d:%d]",
	     x,TeoXstart(image),TeoXend(image),
	     y,TeoYstart(image),TeoYend(image));
  if(val[3]!=Qnil){
    p = NUM2INT(val[2]);
    if(p<0||p>=TeoPlane(image))
      rb_raise(rb_eIndexError,"Out of range p=%d[%d:%d]",
	       p,0,TeoPlane(image));
    TeoPutPixel(image,x,y,p,TEO_UINT8,NUM2INT(val[3]));
  } else if(TYPE(val[2])==T_ARRAY){
    pixel = val[2];
    while(RARRAY(pixel)->len<TeoPlane(image)) rb_ary_push(pixel,INT2FIX(0));
    for(p=0;p<TeoPlane(image);p++)
      TeoPutPixel(image,x,y,p,TEO_UINT8,NUM2INT(rb_ary_entry(pixel,p)));
    return pixel;
  } else {
    int value;
    value = NUM2INT(val[2]);
    for(p=0;p<TeoPlane(image);p++)
      TeoPutPixel(image,x,y,p,TEO_UINT8,value);
    return val[2];
  }
}

static VALUE UINT8_each_pixel(VALUE self){
  TEOIMAGE *image;
  int xstart,xend,ystart,yend,plane;
  int x=0,y=0,p=0;
  VALUE ary;

  Data_Get_Struct(self, TEOIMAGE, image);

  xstart = TeoXstart(image);
  xend   = TeoXend(image);
  ystart = TeoYstart(image);
  yend   = TeoYend(image);
  plane  = TeoPlane(image);

  ary = rb_ary_new2(plane);

  for(y=ystart;y<=yend;y++)
    for(x=xstart;x<=xend;x++){
      for(p=0;p<plane;p++)
	rb_ary_store(ary,p,INT2FIX(TeoGetPixel(image,x,y,p,TEO_UINT8)));
      rb_yield(ary);
    }
  return Qtrue;
}

static VALUE UINT8_each_value(VALUE self){
  TEOIMAGE *image;
  int xstart,xend,ystart,yend,plane;
  int x=0,y=0,p=0;
  VALUE value;

  Data_Get_Struct(self, TEOIMAGE, image);

  xstart = TeoXstart(image);
  xend   = TeoXend(image);
  ystart = TeoYstart(image);
  yend   = TeoYend(image);
  plane  = TeoPlane(image);

  for(y=ystart;y<=yend;y++)
    for(x=xstart;x<=xend;x++)
      for(p=0;p<plane;p++){
	value = INT2FIX(TeoGetPixel(image,x,y,p,TEO_UINT8));
	rb_yield(value);
      }
  return Qtrue;
}

static VALUE UINT8_each_pixel_with_index(VALUE self){
  TEOIMAGE *image;
  int xstart,xend,ystart,yend,plane;
  int x=0,y=0,p=0;
  VALUE ary;

  Data_Get_Struct(self, TEOIMAGE, image);

  xstart = TeoXstart(image);
  xend   = TeoXend(image);
  ystart = TeoYstart(image);
  yend   = TeoYend(image);
  plane  = TeoPlane(image);

  ary = rb_ary_new2(plane);

  for(y=ystart;y<=yend;y++)
    for(x=xstart;x<=xend;x++){
      for(p=0;p<plane;p++)
	rb_ary_store(ary,p,INT2FIX(TeoGetPixel(image,x,y,p,TEO_UINT8)));
      rb_yield(rb_ary_new3(3,ary,INT2NUM(x),INT2NUM(y)));
    }
  return Qtrue;
}

static VALUE UINT8_each_value_with_index(VALUE self){
  TEOIMAGE *image;
  int xstart,xend,ystart,yend,plane;
  int x=0,y=0,p=0;
  VALUE value;

  Data_Get_Struct(self, TEOIMAGE, image);

  xstart = TeoXstart(image);
  xend   = TeoXend(image);
  ystart = TeoYstart(image);
  yend   = TeoYend(image);
  plane  = TeoPlane(image);

  for(y=ystart;y<=yend;y++)
    for(x=xstart;x<=xend;x++)
      for(p=0;p<plane;p++){
	value = INT2FIX(TeoGetPixel(image,x,y,p,TEO_UINT8));
	rb_yield(rb_ary_new3(4,value,INT2NUM(x),INT2NUM(y),INT2NUM(p)));
      }
  return Qtrue;
}


static VALUE SINT8_s_alloc(int,VALUE*,VALUE);
static VALUE SINT8_to_SINT8(VALUE);

static VALUE SINT8_getpixel(VALUE,VALUE,VALUE,VALUE);
static VALUE SINT8_putpixel(VALUE,VALUE,VALUE,VALUE,VALUE);

static VALUE SINT8_getpixel2(int,VALUE*,VALUE);
static VALUE SINT8_putpixel2(int,VALUE*,VALUE);

static VALUE SINT8_each_pixel(VALUE);
static VALUE SINT8_each_value(VALUE);

static VALUE SINT8_each_pixel_with_index(VALUE);
static VALUE SINT8_each_value_with_index(VALUE);

void
Init_teo_image_SINT8(){
  gImageSINT8 = rb_define_class_under(gImage,"SINT8",gImage);
  rb_define_singleton_method(gImageSINT8,"new",SINT8_s_alloc,-1);
  rb_define_singleton_method(gImageSINT8,"alloc",SINT8_s_alloc,-1);

  rb_define_method(gImageSINT8,"to_SINT8",SINT8_to_SINT8,0);

  rb_define_method(gImageSINT8,"get_pixel",SINT8_getpixel,3);
  rb_define_method(gImageSINT8,"put_pixel",SINT8_putpixel,4);

  rb_define_method(gImageSINT8,"[]",SINT8_getpixel2,-1);
  rb_define_method(gImageSINT8,"[]=",SINT8_putpixel2,-1);

  rb_define_method(gImageSINT8,"each_pixel",SINT8_each_pixel,0);
  rb_define_method(gImageSINT8,"each_value",SINT8_each_value,0);

  rb_define_method(gImageSINT8,"each_pixel_with_index",SINT8_each_pixel_with_index,0);
  rb_define_method(gImageSINT8,"each_value_with_index",SINT8_each_value_with_index,0);
}

static VALUE
SINT8_s_alloc(int argc, VALUE *argv, VALUE image_class){
  VALUE obj;

  VALUE val[5];

  int width=0;
  int height=0;
  int xoffset=0;
  int yoffset=0;
  int plane=1;

  TEOIMAGE *image;

  rb_scan_args(argc,argv,"14",&val[0],&val[1],&val[2],&val[3],&val[4]);

  if(rb_obj_is_kind_of(val[0],gFile)==Qtrue){
    TEOFILE *orig_file;
    Data_Get_Struct(val[0], TEOFILE, orig_file);
    image = TeoAllocImage(TeoWidth(orig_file),TeoHeight(orig_file),
			  TeoXoffset(orig_file),TeoYoffset(orig_file),
			  TEO_SIGNED,8,TeoPlane(orig_file));
    if(image==NULL) rb_raise(rb_eRuntimeError,TEO_ERROR_MESSAGE);
  } else if(rb_obj_is_kind_of(val[0],gImage)==Qtrue){
    TEOIMAGE *orig_image;
    Data_Get_Struct(val[0], TEOIMAGE, orig_image);
    image = TeoAllocImage(TeoWidth(orig_image),TeoHeight(orig_image),
			  TeoXoffset(orig_image),TeoYoffset(orig_image),
			  TEO_SIGNED,8,TeoPlane(orig_image));
    if(image==NULL) rb_raise(rb_eRuntimeError,TEO_ERROR_MESSAGE);
  } else  {
    width = NUM2INT(val[0]);
    height = NUM2INT(val[1]);

    if(val[2]!=Qnil) xoffset = NUM2INT(val[2]);
    if(val[3]!=Qnil) yoffset = NUM2INT(val[3]);
    if(val[4]!=Qnil) plane = NUM2INT(val[4]);
    image = TeoAllocImage(width, height, xoffset, yoffset,
			  TEO_SIGNED, 8, plane);
    if(image==NULL) rb_raise(rb_eRuntimeError,TEO_ERROR_MESSAGE);
  }
  obj = Data_Wrap_Struct(image_class, 0, image_free, image);

  return obj;
}

static VALUE
SINT8_to_SINT8(VALUE self){
  return self;
}

static VALUE SINT8_getpixel(VALUE self,VALUE in_x,VALUE in_y,VALUE in_p){
  TEOIMAGE *image;
  Data_Get_Struct(self, TEOIMAGE, image);
  return INT2FIX(TeoGetPixel(image,NUM2INT(in_x),NUM2INT(in_y),NUM2INT(in_p),
		 TEO_SINT8));
}
static VALUE SINT8_putpixel(VALUE self,VALUE in_x,VALUE in_y,VALUE in_p,VALUE val){
  TEOIMAGE *image;
  Data_Get_Struct(self, TEOIMAGE, image);
  TeoPutPixel(image,NUM2INT(in_x),NUM2INT(in_y),NUM2INT(in_p),
	      TEO_SINT8,NUM2INT(val));
  return val;
}
static VALUE SINT8_getpixel2(int argc,VALUE *argv, VALUE self){
  TEOIMAGE *image;
  int x=0,y=0,p=0;
  VALUE val[3];
  VALUE ary;

  Data_Get_Struct(self, TEOIMAGE, image);

  rb_scan_args(argc,argv,"21",&val[0],&val[1],&val[2]);

  x = NUM2INT(val[0]);
  y = NUM2INT(val[1]);

  if(x<TeoXstart(image)||x>TeoXend(image)||
     y<TeoYstart(image)||y>TeoYend(image))
    rb_raise(rb_eIndexError,"Out of range x=%d[%d:%d],y=%d[%d:%d]",
	     x,TeoXstart(image),TeoXend(image),
	     y,TeoYstart(image),TeoYend(image));
  if(val[2]!=Qnil){
    p = NUM2INT(val[2]);
    if(p<0||p>=TeoPlane(image))
      rb_raise(rb_eIndexError,"Out of range p=%d[%d:%d]",
	       p,0,TeoPlane(image));
    return INT2FIX(TeoGetPixel(image,x,y,p,TEO_SINT8));
  } else {
    ary = rb_ary_new2(TeoPlane(image));
    for(p=0;p<TeoPlane(image);p++)
      rb_ary_store(ary,p,INT2FIX(TeoGetPixel(image,x,y,p,TEO_SINT8)));
    return ary;
  }
}

static VALUE SINT8_putpixel2(int argc,VALUE *argv, VALUE self){
  TEOIMAGE *image;
  int x=0,y=0,p=0;
  VALUE val[4];
  VALUE pixel;

  Data_Get_Struct(self, TEOIMAGE, image);

  rb_scan_args(argc,argv,"31",&val[0],&val[1],&val[2],&val[3]);

  x = NUM2INT(val[0]);
  y = NUM2INT(val[1]);

  if(x<TeoXstart(image)||x>TeoXend(image)||
     y<TeoYstart(image)||y>TeoYend(image))
    rb_raise(rb_eIndexError,"Out of range x=%d[%d:%d],y=%d[%d:%d]",
	     x,TeoXstart(image),TeoXend(image),
	     y,TeoYstart(image),TeoYend(image));
  if(val[3]!=Qnil){
    p = NUM2INT(val[2]);
    if(p<0||p>=TeoPlane(image))
      rb_raise(rb_eIndexError,"Out of range p=%d[%d:%d]",
	       p,0,TeoPlane(image));
    TeoPutPixel(image,x,y,p,TEO_SINT8,NUM2INT(val[3]));
  } else if(TYPE(val[2])==T_ARRAY){
    pixel = val[2];
    while(RARRAY(pixel)->len<TeoPlane(image)) rb_ary_push(pixel,INT2FIX(0));
    for(p=0;p<TeoPlane(image);p++)
      TeoPutPixel(image,x,y,p,TEO_SINT8,NUM2INT(rb_ary_entry(pixel,p)));
    return pixel;
  } else {
    int value;
    value = NUM2INT(val[2]);
    for(p=0;p<TeoPlane(image);p++)
      TeoPutPixel(image,x,y,p,TEO_SINT8,value);
    return val[2];
  }
}

static VALUE SINT8_each_pixel(VALUE self){
  TEOIMAGE *image;
  int xstart,xend,ystart,yend,plane;
  int x=0,y=0,p=0;
  VALUE ary;

  Data_Get_Struct(self, TEOIMAGE, image);

  xstart = TeoXstart(image);
  xend   = TeoXend(image);
  ystart = TeoYstart(image);
  yend   = TeoYend(image);
  plane  = TeoPlane(image);

  ary = rb_ary_new2(plane);

  for(y=ystart;y<=yend;y++)
    for(x=xstart;x<=xend;x++){
      for(p=0;p<plane;p++)
	rb_ary_store(ary,p,INT2FIX(TeoGetPixel(image,x,y,p,TEO_SINT8)));
      rb_yield(ary);
    }
  return Qtrue;
}

static VALUE SINT8_each_value(VALUE self){
  TEOIMAGE *image;
  int xstart,xend,ystart,yend,plane;
  int x=0,y=0,p=0;
  VALUE value;

  Data_Get_Struct(self, TEOIMAGE, image);

  xstart = TeoXstart(image);
  xend   = TeoXend(image);
  ystart = TeoYstart(image);
  yend   = TeoYend(image);
  plane  = TeoPlane(image);

  for(y=ystart;y<=yend;y++)
    for(x=xstart;x<=xend;x++)
      for(p=0;p<plane;p++){
	value = INT2FIX(TeoGetPixel(image,x,y,p,TEO_SINT8));
	rb_yield(value);
      }
  return Qtrue;
}

static VALUE SINT8_each_pixel_with_index(VALUE self){
  TEOIMAGE *image;
  int xstart,xend,ystart,yend,plane;
  int x=0,y=0,p=0;
  VALUE ary;

  Data_Get_Struct(self, TEOIMAGE, image);

  xstart = TeoXstart(image);
  xend   = TeoXend(image);
  ystart = TeoYstart(image);
  yend   = TeoYend(image);
  plane  = TeoPlane(image);

  ary = rb_ary_new2(plane);

  for(y=ystart;y<=yend;y++)
    for(x=xstart;x<=xend;x++){
      for(p=0;p<plane;p++)
	rb_ary_store(ary,p,INT2FIX(TeoGetPixel(image,x,y,p,TEO_SINT8)));
      rb_yield(rb_ary_new3(3,ary,INT2NUM(x),INT2NUM(y)));
    }
  return Qtrue;
}

static VALUE SINT8_each_value_with_index(VALUE self){
  TEOIMAGE *image;
  int xstart,xend,ystart,yend,plane;
  int x=0,y=0,p=0;
  VALUE value;

  Data_Get_Struct(self, TEOIMAGE, image);

  xstart = TeoXstart(image);
  xend   = TeoXend(image);
  ystart = TeoYstart(image);
  yend   = TeoYend(image);
  plane  = TeoPlane(image);

  for(y=ystart;y<=yend;y++)
    for(x=xstart;x<=xend;x++)
      for(p=0;p<plane;p++){
	value = INT2FIX(TeoGetPixel(image,x,y,p,TEO_SINT8));
	rb_yield(rb_ary_new3(4,value,INT2NUM(x),INT2NUM(y),INT2NUM(p)));
      }
  return Qtrue;
}


static VALUE UINT16_s_alloc(int,VALUE*,VALUE);
static VALUE UINT16_to_UINT16(VALUE);

static VALUE UINT16_getpixel(VALUE,VALUE,VALUE,VALUE);
static VALUE UINT16_putpixel(VALUE,VALUE,VALUE,VALUE,VALUE);

static VALUE UINT16_getpixel2(int,VALUE*,VALUE);
static VALUE UINT16_putpixel2(int,VALUE*,VALUE);

static VALUE UINT16_each_pixel(VALUE);
static VALUE UINT16_each_value(VALUE);

static VALUE UINT16_each_pixel_with_index(VALUE);
static VALUE UINT16_each_value_with_index(VALUE);

void
Init_teo_image_UINT16(){
  gImageUINT16 = rb_define_class_under(gImage,"UINT16",gImage);
  rb_define_singleton_method(gImageUINT16,"new",UINT16_s_alloc,-1);
  rb_define_singleton_method(gImageUINT16,"alloc",UINT16_s_alloc,-1);

  rb_define_method(gImageUINT16,"to_UINT16",UINT16_to_UINT16,0);

  rb_define_method(gImageUINT16,"get_pixel",UINT16_getpixel,3);
  rb_define_method(gImageUINT16,"put_pixel",UINT16_putpixel,4);

  rb_define_method(gImageUINT16,"[]",UINT16_getpixel2,-1);
  rb_define_method(gImageUINT16,"[]=",UINT16_putpixel2,-1);

  rb_define_method(gImageUINT16,"each_pixel",UINT16_each_pixel,0);
  rb_define_method(gImageUINT16,"each_value",UINT16_each_value,0);

  rb_define_method(gImageUINT16,"each_pixel_with_index",UINT16_each_pixel_with_index,0);
  rb_define_method(gImageUINT16,"each_value_with_index",UINT16_each_value_with_index,0);
}

static VALUE
UINT16_s_alloc(int argc, VALUE *argv, VALUE image_class){
  VALUE obj;

  VALUE val[5];

  int width=0;
  int height=0;
  int xoffset=0;
  int yoffset=0;
  int plane=1;

  TEOIMAGE *image;

  rb_scan_args(argc,argv,"14",&val[0],&val[1],&val[2],&val[3],&val[4]);

  if(rb_obj_is_kind_of(val[0],gFile)==Qtrue){
    TEOFILE *orig_file;
    Data_Get_Struct(val[0], TEOFILE, orig_file);
    image = TeoAllocImage(TeoWidth(orig_file),TeoHeight(orig_file),
			  TeoXoffset(orig_file),TeoYoffset(orig_file),
			  TEO_UNSIGNED,16,TeoPlane(orig_file));
    if(image==NULL) rb_raise(rb_eRuntimeError,TEO_ERROR_MESSAGE);
  } else if(rb_obj_is_kind_of(val[0],gImage)==Qtrue){
    TEOIMAGE *orig_image;
    Data_Get_Struct(val[0], TEOIMAGE, orig_image);
    image = TeoAllocImage(TeoWidth(orig_image),TeoHeight(orig_image),
			  TeoXoffset(orig_image),TeoYoffset(orig_image),
			  TEO_UNSIGNED,16,TeoPlane(orig_image));
    if(image==NULL) rb_raise(rb_eRuntimeError,TEO_ERROR_MESSAGE);
  } else  {
    width = NUM2INT(val[0]);
    height = NUM2INT(val[1]);

    if(val[2]!=Qnil) xoffset = NUM2INT(val[2]);
    if(val[3]!=Qnil) yoffset = NUM2INT(val[3]);
    if(val[4]!=Qnil) plane = NUM2INT(val[4]);
    image = TeoAllocImage(width, height, xoffset, yoffset,
			  TEO_UNSIGNED, 16, plane);
    if(image==NULL) rb_raise(rb_eRuntimeError,TEO_ERROR_MESSAGE);
  }
  obj = Data_Wrap_Struct(image_class, 0, image_free, image);

  return obj;
}

static VALUE
UINT16_to_UINT16(VALUE self){
  return self;
}

static VALUE UINT16_getpixel(VALUE self,VALUE in_x,VALUE in_y,VALUE in_p){
  TEOIMAGE *image;
  Data_Get_Struct(self, TEOIMAGE, image);
  return INT2FIX(TeoGetPixel(image,NUM2INT(in_x),NUM2INT(in_y),NUM2INT(in_p),
		 TEO_UINT16));
}
static VALUE UINT16_putpixel(VALUE self,VALUE in_x,VALUE in_y,VALUE in_p,VALUE val){
  TEOIMAGE *image;
  Data_Get_Struct(self, TEOIMAGE, image);
  TeoPutPixel(image,NUM2INT(in_x),NUM2INT(in_y),NUM2INT(in_p),
	      TEO_UINT16,NUM2INT(val));
  return val;
}
static VALUE UINT16_getpixel2(int argc,VALUE *argv, VALUE self){
  TEOIMAGE *image;
  int x=0,y=0,p=0;
  VALUE val[3];
  VALUE ary;

  Data_Get_Struct(self, TEOIMAGE, image);

  rb_scan_args(argc,argv,"21",&val[0],&val[1],&val[2]);

  x = NUM2INT(val[0]);
  y = NUM2INT(val[1]);

  if(x<TeoXstart(image)||x>TeoXend(image)||
     y<TeoYstart(image)||y>TeoYend(image))
    rb_raise(rb_eIndexError,"Out of range x=%d[%d:%d],y=%d[%d:%d]",
	     x,TeoXstart(image),TeoXend(image),
	     y,TeoYstart(image),TeoYend(image));
  if(val[2]!=Qnil){
    p = NUM2INT(val[2]);
    if(p<0||p>=TeoPlane(image))
      rb_raise(rb_eIndexError,"Out of range p=%d[%d:%d]",
	       p,0,TeoPlane(image));
    return INT2FIX(TeoGetPixel(image,x,y,p,TEO_UINT16));
  } else {
    ary = rb_ary_new2(TeoPlane(image));
    for(p=0;p<TeoPlane(image);p++)
      rb_ary_store(ary,p,INT2FIX(TeoGetPixel(image,x,y,p,TEO_UINT16)));
    return ary;
  }
}

static VALUE UINT16_putpixel2(int argc,VALUE *argv, VALUE self){
  TEOIMAGE *image;
  int x=0,y=0,p=0;
  VALUE val[4];
  VALUE pixel;

  Data_Get_Struct(self, TEOIMAGE, image);

  rb_scan_args(argc,argv,"31",&val[0],&val[1],&val[2],&val[3]);

  x = NUM2INT(val[0]);
  y = NUM2INT(val[1]);

  if(x<TeoXstart(image)||x>TeoXend(image)||
     y<TeoYstart(image)||y>TeoYend(image))
    rb_raise(rb_eIndexError,"Out of range x=%d[%d:%d],y=%d[%d:%d]",
	     x,TeoXstart(image),TeoXend(image),
	     y,TeoYstart(image),TeoYend(image));
  if(val[3]!=Qnil){
    p = NUM2INT(val[2]);
    if(p<0||p>=TeoPlane(image))
      rb_raise(rb_eIndexError,"Out of range p=%d[%d:%d]",
	       p,0,TeoPlane(image));
    TeoPutPixel(image,x,y,p,TEO_UINT16,NUM2INT(val[3]));
  } else if(TYPE(val[2])==T_ARRAY){
    pixel = val[2];
    while(RARRAY(pixel)->len<TeoPlane(image)) rb_ary_push(pixel,INT2FIX(0));
    for(p=0;p<TeoPlane(image);p++)
      TeoPutPixel(image,x,y,p,TEO_UINT16,NUM2INT(rb_ary_entry(pixel,p)));
    return pixel;
  } else {
    int value;
    value = NUM2INT(val[2]);
    for(p=0;p<TeoPlane(image);p++)
      TeoPutPixel(image,x,y,p,TEO_UINT16,value);
    return val[2];
  }
}

static VALUE UINT16_each_pixel(VALUE self){
  TEOIMAGE *image;
  int xstart,xend,ystart,yend,plane;
  int x=0,y=0,p=0;
  VALUE ary;

  Data_Get_Struct(self, TEOIMAGE, image);

  xstart = TeoXstart(image);
  xend   = TeoXend(image);
  ystart = TeoYstart(image);
  yend   = TeoYend(image);
  plane  = TeoPlane(image);

  ary = rb_ary_new2(plane);

  for(y=ystart;y<=yend;y++)
    for(x=xstart;x<=xend;x++){
      for(p=0;p<plane;p++)
	rb_ary_store(ary,p,INT2FIX(TeoGetPixel(image,x,y,p,TEO_UINT16)));
      rb_yield(ary);
    }
  return Qtrue;
}

static VALUE UINT16_each_value(VALUE self){
  TEOIMAGE *image;
  int xstart,xend,ystart,yend,plane;
  int x=0,y=0,p=0;
  VALUE value;

  Data_Get_Struct(self, TEOIMAGE, image);

  xstart = TeoXstart(image);
  xend   = TeoXend(image);
  ystart = TeoYstart(image);
  yend   = TeoYend(image);
  plane  = TeoPlane(image);

  for(y=ystart;y<=yend;y++)
    for(x=xstart;x<=xend;x++)
      for(p=0;p<plane;p++){
	value = INT2FIX(TeoGetPixel(image,x,y,p,TEO_UINT16));
	rb_yield(value);
      }
  return Qtrue;
}

static VALUE UINT16_each_pixel_with_index(VALUE self){
  TEOIMAGE *image;
  int xstart,xend,ystart,yend,plane;
  int x=0,y=0,p=0;
  VALUE ary;

  Data_Get_Struct(self, TEOIMAGE, image);

  xstart = TeoXstart(image);
  xend   = TeoXend(image);
  ystart = TeoYstart(image);
  yend   = TeoYend(image);
  plane  = TeoPlane(image);

  ary = rb_ary_new2(plane);

  for(y=ystart;y<=yend;y++)
    for(x=xstart;x<=xend;x++){
      for(p=0;p<plane;p++)
	rb_ary_store(ary,p,INT2FIX(TeoGetPixel(image,x,y,p,TEO_UINT16)));
      rb_yield(rb_ary_new3(3,ary,INT2NUM(x),INT2NUM(y)));
    }
  return Qtrue;
}

static VALUE UINT16_each_value_with_index(VALUE self){
  TEOIMAGE *image;
  int xstart,xend,ystart,yend,plane;
  int x=0,y=0,p=0;
  VALUE value;

  Data_Get_Struct(self, TEOIMAGE, image);

  xstart = TeoXstart(image);
  xend   = TeoXend(image);
  ystart = TeoYstart(image);
  yend   = TeoYend(image);
  plane  = TeoPlane(image);

  for(y=ystart;y<=yend;y++)
    for(x=xstart;x<=xend;x++)
      for(p=0;p<plane;p++){
	value = INT2FIX(TeoGetPixel(image,x,y,p,TEO_UINT16));
	rb_yield(rb_ary_new3(4,value,INT2NUM(x),INT2NUM(y),INT2NUM(p)));
      }
  return Qtrue;
}


static VALUE SINT16_s_alloc(int,VALUE*,VALUE);
static VALUE SINT16_to_SINT16(VALUE);

static VALUE SINT16_getpixel(VALUE,VALUE,VALUE,VALUE);
static VALUE SINT16_putpixel(VALUE,VALUE,VALUE,VALUE,VALUE);

static VALUE SINT16_getpixel2(int,VALUE*,VALUE);
static VALUE SINT16_putpixel2(int,VALUE*,VALUE);

static VALUE SINT16_each_pixel(VALUE);
static VALUE SINT16_each_value(VALUE);

static VALUE SINT16_each_pixel_with_index(VALUE);
static VALUE SINT16_each_value_with_index(VALUE);

void
Init_teo_image_SINT16(){
  gImageSINT16 = rb_define_class_under(gImage,"SINT16",gImage);
  rb_define_singleton_method(gImageSINT16,"new",SINT16_s_alloc,-1);
  rb_define_singleton_method(gImageSINT16,"alloc",SINT16_s_alloc,-1);

  rb_define_method(gImageSINT16,"to_SINT16",SINT16_to_SINT16,0);

  rb_define_method(gImageSINT16,"get_pixel",SINT16_getpixel,3);
  rb_define_method(gImageSINT16,"put_pixel",SINT16_putpixel,4);

  rb_define_method(gImageSINT16,"[]",SINT16_getpixel2,-1);
  rb_define_method(gImageSINT16,"[]=",SINT16_putpixel2,-1);

  rb_define_method(gImageSINT16,"each_pixel",SINT16_each_pixel,0);
  rb_define_method(gImageSINT16,"each_value",SINT16_each_value,0);

  rb_define_method(gImageSINT16,"each_pixel_with_index",SINT16_each_pixel_with_index,0);
  rb_define_method(gImageSINT16,"each_value_with_index",SINT16_each_value_with_index,0);
}

static VALUE
SINT16_s_alloc(int argc, VALUE *argv, VALUE image_class){
  VALUE obj;

  VALUE val[5];

  int width=0;
  int height=0;
  int xoffset=0;
  int yoffset=0;
  int plane=1;

  TEOIMAGE *image;

  rb_scan_args(argc,argv,"14",&val[0],&val[1],&val[2],&val[3],&val[4]);

  if(rb_obj_is_kind_of(val[0],gFile)==Qtrue){
    TEOFILE *orig_file;
    Data_Get_Struct(val[0], TEOFILE, orig_file);
    image = TeoAllocImage(TeoWidth(orig_file),TeoHeight(orig_file),
			  TeoXoffset(orig_file),TeoYoffset(orig_file),
			  TEO_SIGNED,16,TeoPlane(orig_file));
    if(image==NULL) rb_raise(rb_eRuntimeError,TEO_ERROR_MESSAGE);
  } else if(rb_obj_is_kind_of(val[0],gImage)==Qtrue){
    TEOIMAGE *orig_image;
    Data_Get_Struct(val[0], TEOIMAGE, orig_image);
    image = TeoAllocImage(TeoWidth(orig_image),TeoHeight(orig_image),
			  TeoXoffset(orig_image),TeoYoffset(orig_image),
			  TEO_SIGNED,16,TeoPlane(orig_image));
    if(image==NULL) rb_raise(rb_eRuntimeError,TEO_ERROR_MESSAGE);
  } else  {
    width = NUM2INT(val[0]);
    height = NUM2INT(val[1]);

    if(val[2]!=Qnil) xoffset = NUM2INT(val[2]);
    if(val[3]!=Qnil) yoffset = NUM2INT(val[3]);
    if(val[4]!=Qnil) plane = NUM2INT(val[4]);
    image = TeoAllocImage(width, height, xoffset, yoffset,
			  TEO_SIGNED, 16, plane);
    if(image==NULL) rb_raise(rb_eRuntimeError,TEO_ERROR_MESSAGE);
  }
  obj = Data_Wrap_Struct(image_class, 0, image_free, image);

  return obj;
}

static VALUE
SINT16_to_SINT16(VALUE self){
  return self;
}

static VALUE SINT16_getpixel(VALUE self,VALUE in_x,VALUE in_y,VALUE in_p){
  TEOIMAGE *image;
  Data_Get_Struct(self, TEOIMAGE, image);
  return INT2FIX(TeoGetPixel(image,NUM2INT(in_x),NUM2INT(in_y),NUM2INT(in_p),
		 TEO_SINT16));
}
static VALUE SINT16_putpixel(VALUE self,VALUE in_x,VALUE in_y,VALUE in_p,VALUE val){
  TEOIMAGE *image;
  Data_Get_Struct(self, TEOIMAGE, image);
  TeoPutPixel(image,NUM2INT(in_x),NUM2INT(in_y),NUM2INT(in_p),
	      TEO_SINT16,NUM2INT(val));
  return val;
}
static VALUE SINT16_getpixel2(int argc,VALUE *argv, VALUE self){
  TEOIMAGE *image;
  int x=0,y=0,p=0;
  VALUE val[3];
  VALUE ary;

  Data_Get_Struct(self, TEOIMAGE, image);

  rb_scan_args(argc,argv,"21",&val[0],&val[1],&val[2]);

  x = NUM2INT(val[0]);
  y = NUM2INT(val[1]);

  if(x<TeoXstart(image)||x>TeoXend(image)||
     y<TeoYstart(image)||y>TeoYend(image))
    rb_raise(rb_eIndexError,"Out of range x=%d[%d:%d],y=%d[%d:%d]",
	     x,TeoXstart(image),TeoXend(image),
	     y,TeoYstart(image),TeoYend(image));
  if(val[2]!=Qnil){
    p = NUM2INT(val[2]);
    if(p<0||p>=TeoPlane(image))
      rb_raise(rb_eIndexError,"Out of range p=%d[%d:%d]",
	       p,0,TeoPlane(image));
    return INT2FIX(TeoGetPixel(image,x,y,p,TEO_SINT16));
  } else {
    ary = rb_ary_new2(TeoPlane(image));
    for(p=0;p<TeoPlane(image);p++)
      rb_ary_store(ary,p,INT2FIX(TeoGetPixel(image,x,y,p,TEO_SINT16)));
    return ary;
  }
}

static VALUE SINT16_putpixel2(int argc,VALUE *argv, VALUE self){
  TEOIMAGE *image;
  int x=0,y=0,p=0;
  VALUE val[4];
  VALUE pixel;

  Data_Get_Struct(self, TEOIMAGE, image);

  rb_scan_args(argc,argv,"31",&val[0],&val[1],&val[2],&val[3]);

  x = NUM2INT(val[0]);
  y = NUM2INT(val[1]);

  if(x<TeoXstart(image)||x>TeoXend(image)||
     y<TeoYstart(image)||y>TeoYend(image))
    rb_raise(rb_eIndexError,"Out of range x=%d[%d:%d],y=%d[%d:%d]",
	     x,TeoXstart(image),TeoXend(image),
	     y,TeoYstart(image),TeoYend(image));
  if(val[3]!=Qnil){
    p = NUM2INT(val[2]);
    if(p<0||p>=TeoPlane(image))
      rb_raise(rb_eIndexError,"Out of range p=%d[%d:%d]",
	       p,0,TeoPlane(image));
    TeoPutPixel(image,x,y,p,TEO_SINT16,NUM2INT(val[3]));
  } else if(TYPE(val[2])==T_ARRAY){
    pixel = val[2];
    while(RARRAY(pixel)->len<TeoPlane(image)) rb_ary_push(pixel,INT2FIX(0));
    for(p=0;p<TeoPlane(image);p++)
      TeoPutPixel(image,x,y,p,TEO_SINT16,NUM2INT(rb_ary_entry(pixel,p)));
    return pixel;
  } else {
    int value;
    value = NUM2INT(val[2]);
    for(p=0;p<TeoPlane(image);p++)
      TeoPutPixel(image,x,y,p,TEO_SINT16,value);
    return val[2];
  }
}

static VALUE SINT16_each_pixel(VALUE self){
  TEOIMAGE *image;
  int xstart,xend,ystart,yend,plane;
  int x=0,y=0,p=0;
  VALUE ary;

  Data_Get_Struct(self, TEOIMAGE, image);

  xstart = TeoXstart(image);
  xend   = TeoXend(image);
  ystart = TeoYstart(image);
  yend   = TeoYend(image);
  plane  = TeoPlane(image);

  ary = rb_ary_new2(plane);

  for(y=ystart;y<=yend;y++)
    for(x=xstart;x<=xend;x++){
      for(p=0;p<plane;p++)
	rb_ary_store(ary,p,INT2FIX(TeoGetPixel(image,x,y,p,TEO_SINT16)));
      rb_yield(ary);
    }
  return Qtrue;
}

static VALUE SINT16_each_value(VALUE self){
  TEOIMAGE *image;
  int xstart,xend,ystart,yend,plane;
  int x=0,y=0,p=0;
  VALUE value;

  Data_Get_Struct(self, TEOIMAGE, image);

  xstart = TeoXstart(image);
  xend   = TeoXend(image);
  ystart = TeoYstart(image);
  yend   = TeoYend(image);
  plane  = TeoPlane(image);

  for(y=ystart;y<=yend;y++)
    for(x=xstart;x<=xend;x++)
      for(p=0;p<plane;p++){
	value = INT2FIX(TeoGetPixel(image,x,y,p,TEO_SINT16));
	rb_yield(value);
      }
  return Qtrue;
}

static VALUE SINT16_each_pixel_with_index(VALUE self){
  TEOIMAGE *image;
  int xstart,xend,ystart,yend,plane;
  int x=0,y=0,p=0;
  VALUE ary;

  Data_Get_Struct(self, TEOIMAGE, image);

  xstart = TeoXstart(image);
  xend   = TeoXend(image);
  ystart = TeoYstart(image);
  yend   = TeoYend(image);
  plane  = TeoPlane(image);

  ary = rb_ary_new2(plane);

  for(y=ystart;y<=yend;y++)
    for(x=xstart;x<=xend;x++){
      for(p=0;p<plane;p++)
	rb_ary_store(ary,p,INT2FIX(TeoGetPixel(image,x,y,p,TEO_SINT16)));
      rb_yield(rb_ary_new3(3,ary,INT2NUM(x),INT2NUM(y)));
    }
  return Qtrue;
}

static VALUE SINT16_each_value_with_index(VALUE self){
  TEOIMAGE *image;
  int xstart,xend,ystart,yend,plane;
  int x=0,y=0,p=0;
  VALUE value;

  Data_Get_Struct(self, TEOIMAGE, image);

  xstart = TeoXstart(image);
  xend   = TeoXend(image);
  ystart = TeoYstart(image);
  yend   = TeoYend(image);
  plane  = TeoPlane(image);

  for(y=ystart;y<=yend;y++)
    for(x=xstart;x<=xend;x++)
      for(p=0;p<plane;p++){
	value = INT2FIX(TeoGetPixel(image,x,y,p,TEO_SINT16));
	rb_yield(rb_ary_new3(4,value,INT2NUM(x),INT2NUM(y),INT2NUM(p)));
      }
  return Qtrue;
}


static VALUE UINT32_s_alloc(int,VALUE*,VALUE);
static VALUE UINT32_to_UINT32(VALUE);

static VALUE UINT32_getpixel(VALUE,VALUE,VALUE,VALUE);
static VALUE UINT32_putpixel(VALUE,VALUE,VALUE,VALUE,VALUE);

static VALUE UINT32_getpixel2(int,VALUE*,VALUE);
static VALUE UINT32_putpixel2(int,VALUE*,VALUE);

static VALUE UINT32_each_pixel(VALUE);
static VALUE UINT32_each_value(VALUE);

static VALUE UINT32_each_pixel_with_index(VALUE);
static VALUE UINT32_each_value_with_index(VALUE);

void
Init_teo_image_UINT32(){
  gImageUINT32 = rb_define_class_under(gImage,"UINT32",gImage);
  rb_define_singleton_method(gImageUINT32,"new",UINT32_s_alloc,-1);
  rb_define_singleton_method(gImageUINT32,"alloc",UINT32_s_alloc,-1);

  rb_define_method(gImageUINT32,"to_UINT32",UINT32_to_UINT32,0);

  rb_define_method(gImageUINT32,"get_pixel",UINT32_getpixel,3);
  rb_define_method(gImageUINT32,"put_pixel",UINT32_putpixel,4);

  rb_define_method(gImageUINT32,"[]",UINT32_getpixel2,-1);
  rb_define_method(gImageUINT32,"[]=",UINT32_putpixel2,-1);

  rb_define_method(gImageUINT32,"each_pixel_with_index",UINT32_each_pixel_with_index,0);
  rb_define_method(gImageUINT32,"each_value_with_index",UINT32_each_value_with_index,0);
}

static VALUE
UINT32_s_alloc(int argc, VALUE *argv, VALUE image_class){
  VALUE obj;

  VALUE val[5];

  int width=0;
  int height=0;
  int xoffset=0;
  int yoffset=0;
  int plane=1;

  TEOIMAGE *image;

  rb_scan_args(argc,argv,"14",&val[0],&val[1],&val[2],&val[3],&val[4]);

  if(rb_obj_is_kind_of(val[0],gFile)==Qtrue){
    TEOFILE *orig_file;
    Data_Get_Struct(val[0], TEOFILE, orig_file);
    image = TeoAllocImage(TeoWidth(orig_file),TeoHeight(orig_file),
			  TeoXoffset(orig_file),TeoYoffset(orig_file),
			  TEO_UNSIGNED,32,TeoPlane(orig_file));
    if(image==NULL) rb_raise(rb_eRuntimeError,TEO_ERROR_MESSAGE);
  } else if(rb_obj_is_kind_of(val[0],gImage)==Qtrue){
    TEOIMAGE *orig_image;
    Data_Get_Struct(val[0], TEOIMAGE, orig_image);
    image = TeoAllocImage(TeoWidth(orig_image),TeoHeight(orig_image),
			  TeoXoffset(orig_image),TeoYoffset(orig_image),
			  TEO_UNSIGNED,32,TeoPlane(orig_image));
    if(image==NULL) rb_raise(rb_eRuntimeError,TEO_ERROR_MESSAGE);
  } else  {
    width = NUM2INT(val[0]);
    height = NUM2INT(val[1]);

    if(val[2]!=Qnil) xoffset = NUM2INT(val[2]);
    if(val[3]!=Qnil) yoffset = NUM2INT(val[3]);
    if(val[4]!=Qnil) plane = NUM2INT(val[4]);
    image = TeoAllocImage(width, height, xoffset, yoffset,
			  TEO_UNSIGNED, 32, plane);
    if(image==NULL) rb_raise(rb_eRuntimeError,TEO_ERROR_MESSAGE);
  }
  obj = Data_Wrap_Struct(image_class, 0, image_free, image);

  return obj;
}

static VALUE
UINT32_to_UINT32(VALUE self){
  return self;
}

static VALUE UINT32_getpixel(VALUE self,VALUE in_x,VALUE in_y,VALUE in_p){
  TEOIMAGE *image;
  Data_Get_Struct(self, TEOIMAGE, image);
  return INT2NUM(TeoGetPixel(image,NUM2INT(in_x),NUM2INT(in_y),NUM2INT(in_p),
		 TEO_UINT32));
}
static VALUE UINT32_putpixel(VALUE self,VALUE in_x,VALUE in_y,VALUE in_p,VALUE val){
  TEOIMAGE *image;
  Data_Get_Struct(self, TEOIMAGE, image);
  TeoPutPixel(image,NUM2INT(in_x),NUM2INT(in_y),NUM2INT(in_p),
	      TEO_UINT32,NUM2INT(val));
  return val;
}
static VALUE UINT32_getpixel2(int argc,VALUE *argv, VALUE self){
  TEOIMAGE *image;
  int x=0,y=0,p=0;
  VALUE val[3];
  VALUE ary;

  Data_Get_Struct(self, TEOIMAGE, image);

  rb_scan_args(argc,argv,"21",&val[0],&val[1],&val[2]);

  x = NUM2INT(val[0]);
  y = NUM2INT(val[1]);

  if(x<TeoXstart(image)||x>TeoXend(image)||
     y<TeoYstart(image)||y>TeoYend(image))
    rb_raise(rb_eIndexError,"Out of range x=%d[%d:%d],y=%d[%d:%d]",
	     x,TeoXstart(image),TeoXend(image),
	     y,TeoYstart(image),TeoYend(image));
  if(val[2]!=Qnil){
    p = NUM2INT(val[2]);
    if(p<0||p>=TeoPlane(image))
      rb_raise(rb_eIndexError,"Out of range p=%d[%d:%d]",
	       p,0,TeoPlane(image));
    return INT2NUM(TeoGetPixel(image,x,y,p,TEO_UINT32));
  } else {
    ary = rb_ary_new2(TeoPlane(image));
    for(p=0;p<TeoPlane(image);p++)
      rb_ary_store(ary,p,INT2NUM(TeoGetPixel(image,x,y,p,TEO_UINT32)));
    return ary;
  }
}

static VALUE UINT32_putpixel2(int argc,VALUE *argv, VALUE self){
  TEOIMAGE *image;
  int x=0,y=0,p=0;
  VALUE val[4];
  VALUE pixel;

  Data_Get_Struct(self, TEOIMAGE, image);

  rb_scan_args(argc,argv,"31",&val[0],&val[1],&val[2],&val[3]);

  x = NUM2INT(val[0]);
  y = NUM2INT(val[1]);

  if(x<TeoXstart(image)||x>TeoXend(image)||
     y<TeoYstart(image)||y>TeoYend(image))
    rb_raise(rb_eIndexError,"Out of range x=%d[%d:%d],y=%d[%d:%d]",
	     x,TeoXstart(image),TeoXend(image),
	     y,TeoYstart(image),TeoYend(image));
  if(val[3]!=Qnil){
    p = NUM2INT(val[2]);
    if(p<0||p>=TeoPlane(image))
      rb_raise(rb_eIndexError,"Out of range p=%d[%d:%d]",
	       p,0,TeoPlane(image));
    TeoPutPixel(image,x,y,p,TEO_UINT32,NUM2INT(val[3]));
  } else if(TYPE(val[2])==T_ARRAY){
    pixel = val[2];
    while(RARRAY(pixel)->len<TeoPlane(image)) rb_ary_push(pixel,INT2NUM(0));
    for(p=0;p<TeoPlane(image);p++)
      TeoPutPixel(image,x,y,p,TEO_UINT32,NUM2INT(rb_ary_entry(pixel,p)));
    return pixel;
  } else {
    int value;
    value = NUM2INT(val[2]);
    for(p=0;p<TeoPlane(image);p++)
      TeoPutPixel(image,x,y,p,TEO_UINT32,value);
    return val[2];
  }
}

static VALUE UINT32_each_pixel(VALUE self){
  TEOIMAGE *image;
  int xstart,xend,ystart,yend,plane;
  int x=0,y=0,p=0;
  VALUE ary;

  Data_Get_Struct(self, TEOIMAGE, image);

  xstart = TeoXstart(image);
  xend   = TeoXend(image);
  ystart = TeoYstart(image);
  yend   = TeoYend(image);
  plane  = TeoPlane(image);

  ary = rb_ary_new2(plane);

  for(y=ystart;y<=yend;y++)
    for(x=xstart;x<=xend;x++){
      for(p=0;p<plane;p++)
	rb_ary_store(ary,p,INT2NUM(TeoGetPixel(image,x,y,p,TEO_UINT32)));
      rb_yield(ary);
    }
  return Qtrue;
}

static VALUE UINT32_each_value(VALUE self){
  TEOIMAGE *image;
  int xstart,xend,ystart,yend,plane;
  int x=0,y=0,p=0;
  VALUE value;

  Data_Get_Struct(self, TEOIMAGE, image);

  xstart = TeoXstart(image);
  xend   = TeoXend(image);
  ystart = TeoYstart(image);
  yend   = TeoYend(image);
  plane  = TeoPlane(image);

  for(y=ystart;y<=yend;y++)
    for(x=xstart;x<=xend;x++)
      for(p=0;p<plane;p++){
	value = INT2NUM(TeoGetPixel(image,x,y,p,TEO_UINT32));
	rb_yield(value);
      }
  return Qtrue;
}

static VALUE UINT32_each_pixel_with_index(VALUE self){
  TEOIMAGE *image;
  int xstart,xend,ystart,yend,plane;
  int x=0,y=0,p=0;
  VALUE ary;

  Data_Get_Struct(self, TEOIMAGE, image);

  xstart = TeoXstart(image);
  xend   = TeoXend(image);
  ystart = TeoYstart(image);
  yend   = TeoYend(image);
  plane  = TeoPlane(image);

  ary = rb_ary_new2(plane);

  for(y=ystart;y<=yend;y++)
    for(x=xstart;x<=xend;x++){
      for(p=0;p<plane;p++)
	rb_ary_store(ary,p,INT2NUM(TeoGetPixel(image,x,y,p,TEO_UINT32)));
      rb_yield(rb_ary_new3(3,ary,INT2NUM(x),INT2NUM(y)));
    }
  return Qtrue;
}

static VALUE UINT32_each_value_with_index(VALUE self){
  TEOIMAGE *image;
  int xstart,xend,ystart,yend,plane;
  int x=0,y=0,p=0;
  VALUE value;

  Data_Get_Struct(self, TEOIMAGE, image);

  xstart = TeoXstart(image);
  xend   = TeoXend(image);
  ystart = TeoYstart(image);
  yend   = TeoYend(image);
  plane  = TeoPlane(image);

  for(y=ystart;y<=yend;y++)
    for(x=xstart;x<=xend;x++)
      for(p=0;p<plane;p++){
	value = INT2NUM(TeoGetPixel(image,x,y,p,TEO_UINT32));
	rb_yield(rb_ary_new3(4,value,INT2NUM(x),INT2NUM(y),INT2NUM(p)));
      }
  return Qtrue;
}


static VALUE SINT32_s_alloc(int,VALUE*,VALUE);
static VALUE SINT32_to_SINT32(VALUE);

static VALUE SINT32_getpixel(VALUE,VALUE,VALUE,VALUE);
static VALUE SINT32_putpixel(VALUE,VALUE,VALUE,VALUE,VALUE);

static VALUE SINT32_getpixel2(int,VALUE*,VALUE);
static VALUE SINT32_putpixel2(int,VALUE*,VALUE);

static VALUE SINT32_each_pixel(VALUE);
static VALUE SINT32_each_value(VALUE);

static VALUE SINT32_each_pixel_with_index(VALUE);
static VALUE SINT32_each_value_with_index(VALUE);

void
Init_teo_image_SINT32(){
  gImageSINT32 = rb_define_class_under(gImage,"SINT32",gImage);
  rb_define_singleton_method(gImageSINT32,"new",SINT32_s_alloc,-1);
  rb_define_singleton_method(gImageSINT32,"alloc",SINT32_s_alloc,-1);

  rb_define_method(gImageSINT32,"to_SINT32",SINT32_to_SINT32,0);

  rb_define_method(gImageSINT32,"get_pixel",SINT32_getpixel,3);
  rb_define_method(gImageSINT32,"put_pixel",SINT32_putpixel,4);

  rb_define_method(gImageSINT32,"[]",SINT32_getpixel2,-1);
  rb_define_method(gImageSINT32,"[]=",SINT32_putpixel2,-1);

  rb_define_method(gImageSINT32,"each_pixel",SINT32_each_pixel,0);
  rb_define_method(gImageSINT32,"each_value",SINT32_each_value,0);

  rb_define_method(gImageSINT32,"each_pixel_with_index",SINT32_each_pixel_with_index,0);
  rb_define_method(gImageSINT32,"each_value_with_index",SINT32_each_value_with_index,0);
}

static VALUE
SINT32_s_alloc(int argc, VALUE *argv, VALUE image_class){
  VALUE obj;

  VALUE val[5];

  int width=0;
  int height=0;
  int xoffset=0;
  int yoffset=0;
  int plane=1;

  TEOIMAGE *image;

  rb_scan_args(argc,argv,"14",&val[0],&val[1],&val[2],&val[3],&val[4]);

  if(rb_obj_is_kind_of(val[0],gFile)==Qtrue){
    TEOFILE *orig_file;
    Data_Get_Struct(val[0], TEOFILE, orig_file);
    image = TeoAllocImage(TeoWidth(orig_file),TeoHeight(orig_file),
			  TeoXoffset(orig_file),TeoYoffset(orig_file),
			  TEO_SIGNED,32,TeoPlane(orig_file));
    if(image==NULL) rb_raise(rb_eRuntimeError,TEO_ERROR_MESSAGE);
  } else if(rb_obj_is_kind_of(val[0],gImage)==Qtrue){
    TEOIMAGE *orig_image;
    Data_Get_Struct(val[0], TEOIMAGE, orig_image);
    image = TeoAllocImage(TeoWidth(orig_image),TeoHeight(orig_image),
			  TeoXoffset(orig_image),TeoYoffset(orig_image),
			  TEO_SIGNED,32,TeoPlane(orig_image));
    if(image==NULL) rb_raise(rb_eRuntimeError,TEO_ERROR_MESSAGE);
  } else  {
    width = NUM2INT(val[0]);
    height = NUM2INT(val[1]);

    if(val[2]!=Qnil) xoffset = NUM2INT(val[2]);
    if(val[3]!=Qnil) yoffset = NUM2INT(val[3]);
    if(val[4]!=Qnil) plane = NUM2INT(val[4]);
    image = TeoAllocImage(width, height, xoffset, yoffset,
			  TEO_SIGNED, 32, plane);
    if(image==NULL) rb_raise(rb_eRuntimeError,TEO_ERROR_MESSAGE);
  }
  obj = Data_Wrap_Struct(image_class, 0, image_free, image);

  return obj;
}

static VALUE
SINT32_to_SINT32(VALUE self){
  return self;
}

static VALUE SINT32_getpixel(VALUE self,VALUE in_x,VALUE in_y,VALUE in_p){
  TEOIMAGE *image;
  Data_Get_Struct(self, TEOIMAGE, image);
  return INT2NUM(TeoGetPixel(image,NUM2INT(in_x),NUM2INT(in_y),NUM2INT(in_p),
		 TEO_SINT32));
}
static VALUE SINT32_putpixel(VALUE self,VALUE in_x,VALUE in_y,VALUE in_p,VALUE val){
  TEOIMAGE *image;
  Data_Get_Struct(self, TEOIMAGE, image);
  TeoPutPixel(image,NUM2INT(in_x),NUM2INT(in_y),NUM2INT(in_p),
	      TEO_SINT32,NUM2INT(val));
  return val;
}
static VALUE SINT32_getpixel2(int argc,VALUE *argv, VALUE self){
  TEOIMAGE *image;
  int x=0,y=0,p=0;
  VALUE val[3];
  VALUE ary;

  Data_Get_Struct(self, TEOIMAGE, image);

  rb_scan_args(argc,argv,"21",&val[0],&val[1],&val[2]);

  x = NUM2INT(val[0]);
  y = NUM2INT(val[1]);

  if(x<TeoXstart(image)||x>TeoXend(image)||
     y<TeoYstart(image)||y>TeoYend(image))
    rb_raise(rb_eIndexError,"Out of range x=%d[%d:%d],y=%d[%d:%d]",
	     x,TeoXstart(image),TeoXend(image),
	     y,TeoYstart(image),TeoYend(image));
  if(val[2]!=Qnil){
    p = NUM2INT(val[2]);
    if(p<0||p>=TeoPlane(image))
      rb_raise(rb_eIndexError,"Out of range p=%d[%d:%d]",
	       p,0,TeoPlane(image));
    return INT2NUM(TeoGetPixel(image,x,y,p,TEO_SINT32));
  } else {
    ary = rb_ary_new2(TeoPlane(image));
    for(p=0;p<TeoPlane(image);p++)
      rb_ary_store(ary,p,INT2NUM(TeoGetPixel(image,x,y,p,TEO_SINT32)));
    return ary;
  }
}

static VALUE SINT32_putpixel2(int argc,VALUE *argv, VALUE self){
  TEOIMAGE *image;
  int x=0,y=0,p=0;
  VALUE val[4];
  VALUE pixel;

  Data_Get_Struct(self, TEOIMAGE, image);

  rb_scan_args(argc,argv,"31",&val[0],&val[1],&val[2],&val[3]);

  x = NUM2INT(val[0]);
  y = NUM2INT(val[1]);

  if(x<TeoXstart(image)||x>TeoXend(image)||
     y<TeoYstart(image)||y>TeoYend(image))
    rb_raise(rb_eIndexError,"Out of range x=%d[%d:%d],y=%d[%d:%d]",
	     x,TeoXstart(image),TeoXend(image),
	     y,TeoYstart(image),TeoYend(image));
  if(val[3]!=Qnil){
    p = NUM2INT(val[2]);
    if(p<0||p>=TeoPlane(image))
      rb_raise(rb_eIndexError,"Out of range p=%d[%d:%d]",
	       p,0,TeoPlane(image));
    TeoPutPixel(image,x,y,p,TEO_SINT32,NUM2INT(val[3]));
  } else if(TYPE(val[2])==T_ARRAY){
    pixel = val[2];
    while(RARRAY(pixel)->len<TeoPlane(image)) rb_ary_push(pixel,INT2NUM(0));
    for(p=0;p<TeoPlane(image);p++)
      TeoPutPixel(image,x,y,p,TEO_SINT32,NUM2INT(rb_ary_entry(pixel,p)));
    return pixel;
  } else {
    int value;
    value = NUM2INT(val[2]);
    for(p=0;p<TeoPlane(image);p++)
      TeoPutPixel(image,x,y,p,TEO_SINT32,value);
    return val[2];
  }
}

static VALUE SINT32_each_pixel(VALUE self){
  TEOIMAGE *image;
  int xstart,xend,ystart,yend,plane;
  int x=0,y=0,p=0;
  VALUE ary;

  Data_Get_Struct(self, TEOIMAGE, image);

  xstart = TeoXstart(image);
  xend   = TeoXend(image);
  ystart = TeoYstart(image);
  yend   = TeoYend(image);
  plane  = TeoPlane(image);

  ary = rb_ary_new2(plane);

  for(y=ystart;y<=yend;y++)
    for(x=xstart;x<=xend;x++){
      for(p=0;p<plane;p++)
	rb_ary_store(ary,p,INT2NUM(TeoGetPixel(image,x,y,p,TEO_SINT32)));
      rb_yield(ary);
    }
  return Qtrue;
}

static VALUE SINT32_each_value(VALUE self){
  TEOIMAGE *image;
  int xstart,xend,ystart,yend,plane;
  int x=0,y=0,p=0;
  VALUE value;

  Data_Get_Struct(self, TEOIMAGE, image);

  xstart = TeoXstart(image);
  xend   = TeoXend(image);
  ystart = TeoYstart(image);
  yend   = TeoYend(image);
  plane  = TeoPlane(image);

  for(y=ystart;y<=yend;y++)
    for(x=xstart;x<=xend;x++)
      for(p=0;p<plane;p++){
	value = INT2NUM(TeoGetPixel(image,x,y,p,TEO_SINT32));
	rb_yield(value);
      }
  return Qtrue;
}

static VALUE SINT32_each_pixel_with_index(VALUE self){
  TEOIMAGE *image;
  int xstart,xend,ystart,yend,plane;
  int x=0,y=0,p=0;
  VALUE ary;

  Data_Get_Struct(self, TEOIMAGE, image);

  xstart = TeoXstart(image);
  xend   = TeoXend(image);
  ystart = TeoYstart(image);
  yend   = TeoYend(image);
  plane  = TeoPlane(image);

  ary = rb_ary_new2(plane);

  for(y=ystart;y<=yend;y++)
    for(x=xstart;x<=xend;x++){
      for(p=0;p<plane;p++)
	rb_ary_store(ary,p,INT2NUM(TeoGetPixel(image,x,y,p,TEO_SINT32)));
      rb_yield(rb_ary_new3(3,ary,INT2NUM(x),INT2NUM(y)));
    }
  return Qtrue;
}

static VALUE SINT32_each_value_with_index(VALUE self){
  TEOIMAGE *image;
  int xstart,xend,ystart,yend,plane;
  int x=0,y=0,p=0;
  VALUE value;

  Data_Get_Struct(self, TEOIMAGE, image);

  xstart = TeoXstart(image);
  xend   = TeoXend(image);
  ystart = TeoYstart(image);
  yend   = TeoYend(image);
  plane  = TeoPlane(image);

  for(y=ystart;y<=yend;y++)
    for(x=xstart;x<=xend;x++)
      for(p=0;p<plane;p++){
	value = INT2NUM(TeoGetPixel(image,x,y,p,TEO_SINT32));
	rb_yield(rb_ary_new3(4,value,INT2NUM(x),INT2NUM(y),INT2NUM(p)));
      }
  return Qtrue;
}


static VALUE FLOAT32_s_alloc(int,VALUE*,VALUE);
static VALUE FLOAT32_to_FLOAT32(VALUE);

static VALUE FLOAT32_getpixel(VALUE,VALUE,VALUE,VALUE);
static VALUE FLOAT32_putpixel(VALUE,VALUE,VALUE,VALUE,VALUE);

static VALUE FLOAT32_getpixel2(int,VALUE*,VALUE);
static VALUE FLOAT32_putpixel2(int,VALUE*,VALUE);

static VALUE FLOAT32_each_pixel(VALUE);
static VALUE FLOAT32_each_value(VALUE);

static VALUE FLOAT32_each_pixel_with_index(VALUE);
static VALUE FLOAT32_each_value_with_index(VALUE);

void
Init_teo_image_FLOAT32(){
  gImageFLOAT32 = rb_define_class_under(gImage,"FLOAT32",gImage);
  rb_define_singleton_method(gImageFLOAT32,"new",FLOAT32_s_alloc,-1);
  rb_define_singleton_method(gImageFLOAT32,"alloc",FLOAT32_s_alloc,-1);

  rb_define_method(gImageFLOAT32,"to_FLOAT32",FLOAT32_to_FLOAT32,0);

  rb_define_method(gImageFLOAT32,"get_pixel",FLOAT32_getpixel,3);
  rb_define_method(gImageFLOAT32,"put_pixel",FLOAT32_putpixel,4);

  rb_define_method(gImageFLOAT32,"[]",FLOAT32_getpixel2,-1);
  rb_define_method(gImageFLOAT32,"[]=",FLOAT32_putpixel2,-1);

  rb_define_method(gImageFLOAT32,"each_pixel",FLOAT32_each_pixel,0);
  rb_define_method(gImageFLOAT32,"each_value",FLOAT32_each_value,0);

  rb_define_method(gImageFLOAT32,"each_pixel_with_index",FLOAT32_each_pixel_with_index,0);
  rb_define_method(gImageFLOAT32,"each_value_with_index",FLOAT32_each_value_with_index,0);
}

static VALUE
FLOAT32_s_alloc(int argc, VALUE *argv, VALUE image_class){
  VALUE obj;

  VALUE val[5];

  int width=0;
  int height=0;
  int xoffset=0;
  int yoffset=0;
  int plane=1;

  TEOIMAGE *image;

  rb_scan_args(argc,argv,"14",&val[0],&val[1],&val[2],&val[3],&val[4]);

  if(rb_obj_is_kind_of(val[0],gFile)==Qtrue){
    TEOFILE *orig_file;
    Data_Get_Struct(val[0], TEOFILE, orig_file);
    image = TeoAllocImage(TeoWidth(orig_file),TeoHeight(orig_file),
			  TeoXoffset(orig_file),TeoYoffset(orig_file),
			  TEO_FLOAT,32,TeoPlane(orig_file));
    if(image==NULL) rb_raise(rb_eRuntimeError,TEO_ERROR_MESSAGE);
  } else if(rb_obj_is_kind_of(val[0],gImage)==Qtrue){
    TEOIMAGE *orig_image;
    Data_Get_Struct(val[0], TEOIMAGE, orig_image);
    image = TeoAllocImage(TeoWidth(orig_image),TeoHeight(orig_image),
			  TeoXoffset(orig_image),TeoYoffset(orig_image),
			  TEO_FLOAT,32,TeoPlane(orig_image));
    if(image==NULL) rb_raise(rb_eRuntimeError,TEO_ERROR_MESSAGE);
  } else  {
    width = NUM2INT(val[0]);
    height = NUM2INT(val[1]);

    if(val[2]!=Qnil) xoffset = NUM2INT(val[2]);
    if(val[3]!=Qnil) yoffset = NUM2INT(val[3]);
    if(val[4]!=Qnil) plane = NUM2INT(val[4]);
    image = TeoAllocImage(width, height, xoffset, yoffset,
			  TEO_FLOAT,32, plane);
    if(image==NULL) rb_raise(rb_eRuntimeError,TEO_ERROR_MESSAGE);
  }
  obj = Data_Wrap_Struct(image_class, 0, image_free, image);

  return obj;
}

static VALUE
FLOAT32_to_FLOAT32(VALUE self){
  return self;
}

static VALUE FLOAT32_getpixel(VALUE self,VALUE in_x,VALUE in_y,VALUE in_p){
  TEOIMAGE *image;
  Data_Get_Struct(self, TEOIMAGE, image);
  return rb_float_new(TeoGetPixel(image,NUM2INT(in_x),NUM2INT(in_y),NUM2INT(in_p),
		 TEO_FLOAT32));
}
static VALUE FLOAT32_putpixel(VALUE self,VALUE in_x,VALUE in_y,VALUE in_p,VALUE val){
  TEOIMAGE *image;
  Data_Get_Struct(self, TEOIMAGE, image);
  TeoPutPixel(image,NUM2INT(in_x),NUM2INT(in_y),NUM2INT(in_p),
	      TEO_FLOAT32,NUM2DBL(val));
  return val;
}
static VALUE FLOAT32_getpixel2(int argc,VALUE *argv, VALUE self){
  TEOIMAGE *image;
  int x=0,y=0,p=0;
  VALUE val[3];
  VALUE ary;

  Data_Get_Struct(self, TEOIMAGE, image);

  rb_scan_args(argc,argv,"21",&val[0],&val[1],&val[2]);

  x = NUM2INT(val[0]);
  y = NUM2INT(val[1]);

  if(x<TeoXstart(image)||x>TeoXend(image)||
     y<TeoYstart(image)||y>TeoYend(image))
    rb_raise(rb_eIndexError,"Out of range x=%d[%d:%d],y=%d[%d:%d]",
	     x,TeoXstart(image),TeoXend(image),
	     y,TeoYstart(image),TeoYend(image));
  if(val[2]!=Qnil){
    p = NUM2INT(val[2]);
    if(p<0||p>=TeoPlane(image))
      rb_raise(rb_eIndexError,"Out of range p=%d[%d:%d]",
	       p,0,TeoPlane(image));
    return rb_float_new(TeoGetPixel(image,x,y,p,TEO_FLOAT32));
  } else {
    ary = rb_ary_new2(TeoPlane(image));
    for(p=0;p<TeoPlane(image);p++)
      rb_ary_store(ary,p,rb_float_new(TeoGetPixel(image,x,y,p,TEO_FLOAT32)));
    return ary;
  }
}

static VALUE FLOAT32_putpixel2(int argc,VALUE *argv, VALUE self){
  TEOIMAGE *image;
  int x=0,y=0,p=0;
  VALUE val[4];
  VALUE pixel;

  Data_Get_Struct(self, TEOIMAGE, image);

  rb_scan_args(argc,argv,"31",&val[0],&val[1],&val[2],&val[3]);

  x = NUM2INT(val[0]);
  y = NUM2INT(val[1]);

  if(x<TeoXstart(image)||x>TeoXend(image)||
     y<TeoYstart(image)||y>TeoYend(image))
    rb_raise(rb_eIndexError,"Out of range x=%d[%d:%d],y=%d[%d:%d]",
	     x,TeoXstart(image),TeoXend(image),
	     y,TeoYstart(image),TeoYend(image));
  if(val[3]!=Qnil){
    p = NUM2INT(val[2]);
    if(p<0||p>=TeoPlane(image))
      rb_raise(rb_eIndexError,"Out of range p=%d[%d:%d]",
	       p,0,TeoPlane(image));
    TeoPutPixel(image,x,y,p,TEO_FLOAT32,NUM2DBL(val[3]));
  } else if(TYPE(val[2])==T_ARRAY){
    pixel = val[2];
    while(RARRAY(pixel)->len<TeoPlane(image)) rb_ary_push(pixel,rb_float_new(0));
    for(p=0;p<TeoPlane(image);p++)
      TeoPutPixel(image,x,y,p,TEO_FLOAT32,NUM2DBL(rb_ary_entry(pixel,p)));
    return pixel;
  } else {
    double value;
    value = NUM2DBL(val[2]);
    for(p=0;p<TeoPlane(image);p++)
      TeoPutPixel(image,x,y,p,TEO_FLOAT32,value);
    return val[2];
  }
}

static VALUE FLOAT32_each_pixel(VALUE self){
  TEOIMAGE *image;
  int xstart,xend,ystart,yend,plane;
  int x=0,y=0,p=0;
  VALUE ary;

  Data_Get_Struct(self, TEOIMAGE, image);

  xstart = TeoXstart(image);
  xend   = TeoXend(image);
  ystart = TeoYstart(image);
  yend   = TeoYend(image);
  plane  = TeoPlane(image);

  ary = rb_ary_new2(plane);

  for(y=ystart;y<=yend;y++)
    for(x=xstart;x<=xend;x++){
      for(p=0;p<plane;p++)
	rb_ary_store(ary,p,rb_float_new(TeoGetPixel(image,x,y,p,TEO_FLOAT32)));
      rb_yield(ary);
    }
  return Qtrue;
}

static VALUE FLOAT32_each_value(VALUE self){
  TEOIMAGE *image;
  int xstart,xend,ystart,yend,plane;
  int x=0,y=0,p=0;
  VALUE value;

  Data_Get_Struct(self, TEOIMAGE, image);

  xstart = TeoXstart(image);
  xend   = TeoXend(image);
  ystart = TeoYstart(image);
  yend   = TeoYend(image);
  plane  = TeoPlane(image);

  for(y=ystart;y<=yend;y++)
    for(x=xstart;x<=xend;x++)
      for(p=0;p<plane;p++){
	value = rb_float_new(TeoGetPixel(image,x,y,p,TEO_FLOAT32));
	rb_yield(value);
      }
  return Qtrue;
}

static VALUE FLOAT32_each_pixel_with_index(VALUE self){
  TEOIMAGE *image;
  int xstart,xend,ystart,yend,plane;
  int x=0,y=0,p=0;
  VALUE ary;

  Data_Get_Struct(self, TEOIMAGE, image);

  xstart = TeoXstart(image);
  xend   = TeoXend(image);
  ystart = TeoYstart(image);
  yend   = TeoYend(image);
  plane  = TeoPlane(image);

  ary = rb_ary_new2(plane);

  for(y=ystart;y<=yend;y++)
    for(x=xstart;x<=xend;x++){
      for(p=0;p<plane;p++)
	rb_ary_store(ary,p,rb_float_new(TeoGetPixel(image,x,y,p,TEO_FLOAT32)));
      rb_yield(rb_ary_new3(3,ary,INT2NUM(x),INT2NUM(y)));
    }
  return Qtrue;
}

static VALUE FLOAT32_each_value_with_index(VALUE self){
  TEOIMAGE *image;
  int xstart,xend,ystart,yend,plane;
  int x=0,y=0,p=0;
  VALUE value;

  Data_Get_Struct(self, TEOIMAGE, image);

  xstart = TeoXstart(image);
  xend   = TeoXend(image);
  ystart = TeoYstart(image);
  yend   = TeoYend(image);
  plane  = TeoPlane(image);

  for(y=ystart;y<=yend;y++)
    for(x=xstart;x<=xend;x++)
      for(p=0;p<plane;p++){
	value = rb_float_new(TeoGetPixel(image,x,y,p,TEO_FLOAT32));
	rb_yield(rb_ary_new3(4,value,INT2NUM(x),INT2NUM(y),INT2NUM(p)));
      }
  return Qtrue;
}


static VALUE FLOAT64_s_alloc(int,VALUE*,VALUE);
static VALUE FLOAT64_to_FLOAT64(VALUE);

static VALUE FLOAT64_getpixel(VALUE,VALUE,VALUE,VALUE);
static VALUE FLOAT64_putpixel(VALUE,VALUE,VALUE,VALUE,VALUE);

static VALUE FLOAT64_getpixel2(int,VALUE*,VALUE);
static VALUE FLOAT64_putpixel2(int,VALUE*,VALUE);

static VALUE FLOAT64_each_pixel(VALUE);
static VALUE FLOAT64_each_value(VALUE);

static VALUE FLOAT64_each_pixel_with_index(VALUE);
static VALUE FLOAT64_each_value_with_index(VALUE);

void
Init_teo_image_FLOAT64(){
  gImageFLOAT64 = rb_define_class_under(gImage,"FLOAT64",gImage);
  rb_define_singleton_method(gImageFLOAT64,"new",FLOAT64_s_alloc,-1);
  rb_define_singleton_method(gImageFLOAT64,"alloc",FLOAT64_s_alloc,-1);

  rb_define_method(gImageFLOAT64,"to_FLOAT64",FLOAT64_to_FLOAT64,0);

  rb_define_method(gImageFLOAT64,"get_pixel",FLOAT64_getpixel,3);
  rb_define_method(gImageFLOAT64,"put_pixel",FLOAT64_putpixel,4);

  rb_define_method(gImageFLOAT64,"[]",FLOAT64_getpixel2,-1);
  rb_define_method(gImageFLOAT64,"[]=",FLOAT64_putpixel2,-1);

  rb_define_method(gImageFLOAT64,"each_pixel",FLOAT64_each_pixel,0);
  rb_define_method(gImageFLOAT64,"each_value",FLOAT64_each_value,0);

  rb_define_method(gImageFLOAT64,"each_pixel_with_index",FLOAT64_each_pixel_with_index,0);
  rb_define_method(gImageFLOAT64,"each_value_with_index",FLOAT64_each_value_with_index,0);
}

static VALUE
FLOAT64_s_alloc(int argc, VALUE *argv, VALUE image_class){
  VALUE obj;

  VALUE val[5];

  int width=0;
  int height=0;
  int xoffset=0;
  int yoffset=0;
  int plane=1;

  TEOIMAGE *image;

  rb_scan_args(argc,argv,"14",&val[0],&val[1],&val[2],&val[3],&val[4]);

  if(rb_obj_is_kind_of(val[0],gFile)==Qtrue){
    TEOFILE *orig_file;
    Data_Get_Struct(val[0], TEOFILE, orig_file);
    image = TeoAllocImage(TeoWidth(orig_file),TeoHeight(orig_file),
			  TeoXoffset(orig_file),TeoYoffset(orig_file),
			  TEO_FLOAT,64,TeoPlane(orig_file));
    if(image==NULL) rb_raise(rb_eRuntimeError,TEO_ERROR_MESSAGE);
  } else if(rb_obj_is_kind_of(val[0],gImage)==Qtrue){
    TEOIMAGE *orig_image;
    Data_Get_Struct(val[0], TEOIMAGE, orig_image);
    image = TeoAllocImage(TeoWidth(orig_image),TeoHeight(orig_image),
			  TeoXoffset(orig_image),TeoYoffset(orig_image),
			  TEO_FLOAT,64,TeoPlane(orig_image));
    if(image==NULL) rb_raise(rb_eRuntimeError,TEO_ERROR_MESSAGE);
  } else  {
    width = NUM2INT(val[0]);
    height = NUM2INT(val[1]);

    if(val[2]!=Qnil) xoffset = NUM2INT(val[2]);
    if(val[3]!=Qnil) yoffset = NUM2INT(val[3]);
    if(val[4]!=Qnil) plane = NUM2INT(val[4]);
    image = TeoAllocImage(width, height, xoffset, yoffset,
			  TEO_FLOAT,64, plane);
    if(image==NULL) rb_raise(rb_eRuntimeError,TEO_ERROR_MESSAGE);
  }
  obj = Data_Wrap_Struct(image_class, 0, image_free, image);

  return obj;
}

static VALUE
FLOAT64_to_FLOAT64(VALUE self){
  return self;
}

static VALUE FLOAT64_getpixel(VALUE self,VALUE in_x,VALUE in_y,VALUE in_p){
  TEOIMAGE *image;
  Data_Get_Struct(self, TEOIMAGE, image);
  return rb_float_new(TeoGetPixel(image,NUM2INT(in_x),NUM2INT(in_y),NUM2INT(in_p),
		 TEO_FLOAT64));
}
static VALUE FLOAT64_putpixel(VALUE self,VALUE in_x,VALUE in_y,VALUE in_p,VALUE val){
  TEOIMAGE *image;
  Data_Get_Struct(self, TEOIMAGE, image);
  TeoPutPixel(image,NUM2INT(in_x),NUM2INT(in_y),NUM2INT(in_p),
	      TEO_FLOAT64,NUM2DBL(val));
  return val;
}
static VALUE FLOAT64_getpixel2(int argc,VALUE *argv, VALUE self){
  TEOIMAGE *image;
  int x=0,y=0,p=0;
  VALUE val[3];
  VALUE ary;

  Data_Get_Struct(self, TEOIMAGE, image);

  rb_scan_args(argc,argv,"21",&val[0],&val[1],&val[2]);

  x = NUM2INT(val[0]);
  y = NUM2INT(val[1]);

  if(x<TeoXstart(image)||x>TeoXend(image)||
     y<TeoYstart(image)||y>TeoYend(image))
    rb_raise(rb_eIndexError,"Out of range x=%d[%d:%d],y=%d[%d:%d]",
	     x,TeoXstart(image),TeoXend(image),
	     y,TeoYstart(image),TeoYend(image));
  if(val[2]!=Qnil){
    p = NUM2INT(val[2]);
    if(p<0||p>=TeoPlane(image))
      rb_raise(rb_eIndexError,"Out of range p=%d[%d:%d]",
	       p,0,TeoPlane(image));
    return rb_float_new(TeoGetPixel(image,x,y,p,TEO_FLOAT64));
  } else {
    ary = rb_ary_new2(TeoPlane(image));
    for(p=0;p<TeoPlane(image);p++)
      rb_ary_store(ary,p,rb_float_new(TeoGetPixel(image,x,y,p,TEO_FLOAT64)));
    return ary;
  }
}

static VALUE FLOAT64_putpixel2(int argc,VALUE *argv, VALUE self){
  TEOIMAGE *image;
  int x=0,y=0,p=0;
  VALUE val[4];
  VALUE pixel;

  Data_Get_Struct(self, TEOIMAGE, image);

  rb_scan_args(argc,argv,"31",&val[0],&val[1],&val[2],&val[3]);

  x = NUM2INT(val[0]);
  y = NUM2INT(val[1]);

  if(x<TeoXstart(image)||x>TeoXend(image)||
     y<TeoYstart(image)||y>TeoYend(image))
    rb_raise(rb_eIndexError,"Out of range x=%d[%d:%d],y=%d[%d:%d]",
	     x,TeoXstart(image),TeoXend(image),
	     y,TeoYstart(image),TeoYend(image));
  if(val[3]!=Qnil){
    p = NUM2INT(val[2]);
    if(p<0||p>=TeoPlane(image))
      rb_raise(rb_eIndexError,"Out of range p=%d[%d:%d]",
	       p,0,TeoPlane(image));
    TeoPutPixel(image,x,y,p,TEO_FLOAT64,NUM2DBL(val[3]));
  } else if(TYPE(val[2])==T_ARRAY){
    pixel = val[2];
    while(RARRAY(pixel)->len<TeoPlane(image)) rb_ary_push(pixel,rb_float_new(0));
    for(p=0;p<TeoPlane(image);p++)
      TeoPutPixel(image,x,y,p,TEO_FLOAT64,NUM2DBL(rb_ary_entry(pixel,p)));
    return pixel;
  } else {
    double value;
    value = NUM2DBL(val[2]);
    for(p=0;p<TeoPlane(image);p++)
      TeoPutPixel(image,x,y,p,TEO_FLOAT64,value);
    return val[2];
  }
}

static VALUE FLOAT64_each_pixel(VALUE self){
  TEOIMAGE *image;
  int xstart,xend,ystart,yend,plane;
  int x=0,y=0,p=0;
  VALUE ary;

  Data_Get_Struct(self, TEOIMAGE, image);

  xstart = TeoXstart(image);
  xend   = TeoXend(image);
  ystart = TeoYstart(image);
  yend   = TeoYend(image);
  plane  = TeoPlane(image);

  ary = rb_ary_new2(plane);

  for(y=ystart;y<=yend;y++)
    for(x=xstart;x<=xend;x++){
      for(p=0;p<plane;p++)
	rb_ary_store(ary,p,rb_float_new(TeoGetPixel(image,x,y,p,TEO_FLOAT64)));
      rb_yield(ary);
    }
  return Qtrue;
}

static VALUE FLOAT64_each_value(VALUE self){
  TEOIMAGE *image;
  int xstart,xend,ystart,yend,plane;
  int x=0,y=0,p=0;
  VALUE value;

  Data_Get_Struct(self, TEOIMAGE, image);

  xstart = TeoXstart(image);
  xend   = TeoXend(image);
  ystart = TeoYstart(image);
  yend   = TeoYend(image);
  plane  = TeoPlane(image);

  for(y=ystart;y<=yend;y++)
    for(x=xstart;x<=xend;x++)
      for(p=0;p<plane;p++){
	value = rb_float_new(TeoGetPixel(image,x,y,p,TEO_FLOAT64));
	rb_yield(value);
      }
  return Qtrue;
}

static VALUE FLOAT64_each_pixel_with_index(VALUE self){
  TEOIMAGE *image;
  int xstart,xend,ystart,yend,plane;
  int x=0,y=0,p=0;
  VALUE ary;

  Data_Get_Struct(self, TEOIMAGE, image);

  xstart = TeoXstart(image);
  xend   = TeoXend(image);
  ystart = TeoYstart(image);
  yend   = TeoYend(image);
  plane  = TeoPlane(image);

  ary = rb_ary_new2(plane);

  for(y=ystart;y<=yend;y++)
    for(x=xstart;x<=xend;x++){
      for(p=0;p<plane;p++)
	rb_ary_store(ary,p,rb_float_new(TeoGetPixel(image,x,y,p,TEO_FLOAT64)));
      rb_yield(rb_ary_new3(3,ary,INT2NUM(x),INT2NUM(y)));
    }
  return Qtrue;
}

static VALUE FLOAT64_each_value_with_index(VALUE self){
  TEOIMAGE *image;
  int xstart,xend,ystart,yend,plane;
  int x=0,y=0,p=0;
  VALUE value;

  Data_Get_Struct(self, TEOIMAGE, image);

  xstart = TeoXstart(image);
  xend   = TeoXend(image);
  ystart = TeoYstart(image);
  yend   = TeoYend(image);
  plane  = TeoPlane(image);

  for(y=ystart;y<=yend;y++)
    for(x=xstart;x<=xend;x++)
      for(p=0;p<plane;p++){
	value = rb_float_new(TeoGetPixel(image,x,y,p,TEO_FLOAT64));
	rb_yield(rb_ary_new3(4,value,INT2NUM(x),INT2NUM(y),INT2NUM(p)));
      }
  return Qtrue;
}


void
Init_teo_image_type(){
  Init_teo_image_BIT();
  Init_teo_image_UINT8();
  Init_teo_image_SINT8();
  Init_teo_image_UINT16();
  Init_teo_image_SINT16();
  Init_teo_image_UINT32();
  Init_teo_image_SINT32();
  Init_teo_image_FLOAT32();
  Init_teo_image_FLOAT64();
}

