/**************************************************************************

 teo2avi  converter from teo to avi.
	
**************************************************************************/


#include <cstdio>
#include <string>

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif


#ifndef HAVE_LIBAVIPLAY
#include "avi.h"
#else

#include <videoencoder.h>
#include <default.h>
#include <version.h>
#include <avifile.h>
#include <aviplay.h>
#include <fourcc.h>
#endif

#include <teopp.h>
#include <unistd.h>
#include "teo_s.h"


void usage(char *pname){
  cerr << "usage: " << pname
       << "[-c codec] [-q quality] [-s start end step] [-r frame_rate] input.teo output.avi" << endl;
  exit(1);
}

int main(int argc, char** argv) {

  string codec("CVID");
  float quality=100.0;
  bool step_flag=false;
  int start,end,step;
  int frame_rate=33333;
  int frames_per_second=30;
  bool verbose=false;

  extern int optind;
  extern char *optarg;
  int ch;
  while((ch = getopt(argc,argv,"vc:q:s:r:"))!= -1)
    switch(ch){
    case 'v':
      verbose=true;
      break;
    case 'c':
      codec = string(optarg);
      break;
    case 'q':
      quality = atof(optarg);
      break;
    case 's':
      if(argc>optind+2){
	step_flag=true;
	start = atoi(optarg);
	end = atoi(argv[optind++]);
	step = atoi(argv[optind++]);
      } else {
	usage(argv[0]);
      }
      break;
    case 'r':
      frames_per_second = atoi(optarg);
      frame_rate = (int)(1000000.0/frames_per_second);
      break;
    default:
      usage(argv[0]);
      break;
    }

  int fccHandler=0;

#ifdef HAVE_LIBAVIPLAY

  if(GetAvifileVersion()!=AVIFILE_VERSION) {
    cout<<"This binary was compiled for Avifile ver. "<<AVIFILE_VERSION<<", but the library is ver. "<<GetAvifileVersion()<<". Aborting."<<endl;
    return 0;
  }

  if(codec == "DIV3"){
    fccHandler=fccDIV3;
  } else if(codec == "DIV4"){
    fccHandler=fccDIV4;
  } else if(codec == "IV31"){
    fccHandler=mmioFOURCC('I','V','3','1');
  } else if(codec == "IV32"){
    fccHandler=fccIV32;
  } else if(codec == "IV41"){
    fccHandler=fccIV41;
  } else if(codec == "IV50"){
    fccHandler=fccIV50;
  } else if(codec == "MPG4"){
    fccHandler=mmioFOURCC('M','P','G','4');
  } else if(codec == "I263"){
    fccHandler=fccI263;
  } else if(codec == "CVID"){
    fccHandler=fcccvid;
  } else if(codec == "VCR2"){
    fccHandler=fccVCR2;
  } else if(codec == "MJPG"){
    fccHandler=fccMJPG;
  } else if(codec == "YUV"){
    fccHandler=fccYUV;
  } else if(codec == "NO"){
    fccHandler=0;
  } else {
    fccHandler = mmioFOURCC(codec[0],codec[1],codec[2],codec[3]);
  }
#endif
  if(argc<optind+2)
    usage(argv[0]);

  string input(argv[optind++]);
  string output(argv[optind++]);

  TeoSequence *teofile;

  if(step_flag)
    teofile = new teo_s(input.c_str(),start,end,step);
  else
    teofile = new TeoFile(input.c_str());
  if(!(*teofile)) usage(argv[0]);

#ifndef HAVE_LIBAVIPLAY
  // 󰵽AVI κ
  avi_stream *avifile;
  avifile = new avi_stream(output.c_str(),
			   teofile->Width(), 
			   teofile->Height(),
			   teofile->Frame(),
			   frames_per_second);

  for(int count=0;teofile->CheckFrame();count++){
    avifile->write_frame(teofile->GetImage(),count);
    if(verbose || ((count+1)%10 == 0))
      cout << count+1 << " frames written" << endl;
  }

#else

  IAviWriteFile* avifile;
  BITMAPINFOHEADER bh;

  try {


    TeoImage<TeoUint8> outimg(teofile->Width(),teofile->Height(),
			      teofile->Xoffset(),teofile->Yoffset(),
			      3);

    avifile = CreateIAviWriteFile(output.c_str());

    memset(&bh,0,sizeof(bh));
    bh.biSize=sizeof(bh);
    bh.biWidth = outimg.Width();
    bh.biHeight = outimg.Height();
    bh.biSizeImage = outimg.Fsize();
    bh.biPlanes = 1;
    bh.biBitCount = outimg.Plane()*outimg.Bit();

    IAviVideoWriteStream* stream=avifile->AddVideoStream(fccHandler, 
							 &bh,
							 frame_rate);
    stream->SetQuality((int)(quality*100.0));

    CImage im((BitmapInfo*)&bh, (unsigned char*)outimg.Data(), false);
      
    stream->Start();
    for(int count=0;teofile->CheckFrame();count++) {
      TeoImage<TeoUint8> inimg = teofile->GetImage();
      if(inimg.Plane()<3){
	for(int y=inimg.Ystart();y<=inimg.Yend();y++)
	  for(int x=inimg.Xstart();x<=inimg.Xend();x++){
	    outimg(x,outimg.Yend()-y,0) = inimg(x,y,0);
	    outimg(x,outimg.Yend()-y,1) = inimg(x,y,0);
	    outimg(x,outimg.Yend()-y,2) = inimg(x,y,0);
	  }
      } else {
	for(int y=inimg.Ystart();y<=inimg.Yend();y++)
	  for(int x=inimg.Xstart();x<=inimg.Xend();x++){
	    outimg(x,outimg.Yend()-y,0) = inimg(x,y,2);
	    outimg(x,outimg.Yend()-y,1) = inimg(x,y,1);
	    outimg(x,outimg.Yend()-y,2) = inimg(x,y,0);
	  }
      }
      stream->AddFrame(&im);
      if((count+1)%10 == 0)
	cout << count+1 << " frames written" << endl;
    }

    stream->Stop();
  } catch(FatalError& error) {
    error.Print();
  } catch(...) {
    cout<<"ERROR: Caught unknown exception!"<<endl;
  }
#endif
  delete avifile;
  delete teofile;
  return 0;
}
