/* ******************************************************* gtkfilelist.c *** *
 * gtkfilelist
 *
 * Copyright (C) 2000-2003 Yasuyuki SUGAYA <sugaya@suri.it.okayama-u.ac.jp>
 * Okayama University
 *                                    Time-stamp: <03/03/11 15:09:55 sugaya>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License
 * as published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 *
 * gtkiconfileselection - gtkiconfileselection dialog widget for gtk+
 * Copyright 1999  Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 * Rewitten and customized for GTK+2 by Y.Sugaya
 * ************************************************************************* */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <dirent.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <gtk/gtk.h>
#include <gnome.h>
#include "gnome-file-icon-list.h"
#include "gtkdirtreeview.h"
#include "gtkfilelist.h"
#include "gtkfileicons.h"
#include "gfileutils.h"

#define	DEFAULT_ICON_WIDTH	16
#define	DEFAULT_TEXT_WIDTH	200

#ifdef HAVE_FNMATCH_H
#include <fnmatch.h>
#else
/* ************************************************************************* */
static int
fnmatch (const char	*pattern,
	 const char	*s,
	 int		m) {
  int slen = strlen (s);
  int plen = strlen (pattern);

  if (strstr (pattern, "*.*")) {
    return 0;
  } else if (*pattern == '*') {
    return ((strstr (s, pattern+1) == s + slen - (plen - 1)) ? 0 : 1);
  } else if ('*' == pattern[plen]) {
    return ((0 == strncmp (s, pattern, plen - 1)) ? 0 : 1);
  } else if (NULL == strstr (pattern, "*") && NULL == strstr (pattern, "?")) {
    return ((0 == strcmp (s, pattern)) ? 0 : 1);
  } else {
    g_print ("fnmatch(%s,%s,%d) ???\n", pattern, s, m);
    return 0;
  }
}
#endif

/* ************************************************************************* */
static void gtk_file_list_class_init	(GtkFileListClass	*class);
static void gtk_file_list_init 		(GtkFileList		*filelist);
static void gtk_file_list_realize 	(GtkWidget		*widget);
static void gtk_file_list_destroy 	(GtkObject		*object);
static gint sort_list 			(gpointer		a,
					 gpointer		b);

static gboolean check_dir_extra 	(gchar			*dir_name,
					 struct stat	 	*result,
					 gboolean 		*stat_subdirs);
static void gtk_file_list_parse_filter 	(GtkFileList		*filelist,
					 gchar			*filter_text);

/* ************************************************************************* */
static const GdkPixbuf	*slink = NULL;


/* ************************************************************************* */
static GnomeFileIconListClass	*parent_class = NULL;

static GtkFileListType	default_types[] = {
  {"*.cpp",	GTK_FILE_LIST_CPP},
  {"*.c",	GTK_FILE_LIST_C},
  {"*.o",	GTK_FILE_LIST_O},  
  {"*.h",	GTK_FILE_LIST_H},
  {"*.for",	GTK_FILE_LIST_F},
  {"*.f",	GTK_FILE_LIST_F},
  {"*.f77",	GTK_FILE_LIST_F},
  {"*.f90",	GTK_FILE_LIST_F},
  {"*.jar",	GTK_FILE_LIST_JAVA},
  {"*.java",	GTK_FILE_LIST_JAVA},
  {"*.j",	GTK_FILE_LIST_JAVA},
  {"*.mp3",	GTK_FILE_LIST_SOUND},
  {"*.wav",	GTK_FILE_LIST_SOUND},
  {"*.ps",	GTK_FILE_LIST_PS},
  {"*.pdf",	GTK_FILE_LIST_PDF},
  {"*.doc",	GTK_FILE_LIST_DOC},
  {"Makefile",	GTK_FILE_LIST_MAKE},  
  {"*.txt",	GTK_FILE_LIST_TEXT},
  {"README",	GTK_FILE_LIST_TEXT},
  {"TODO",	GTK_FILE_LIST_TEXT},
  {"NEWS",	GTK_FILE_LIST_TEXT},
  {"AUTHORS",	GTK_FILE_LIST_TEXT},
  {"COPYING",	GTK_FILE_LIST_TEXT},
  {"INSTALL",	GTK_FILE_LIST_TEXT},
  {"BUGS",	GTK_FILE_LIST_TEXT},
  {"*.xpm",	GTK_FILE_LIST_IMG},
  {"*.jpg",	GTK_FILE_LIST_IMG},
  {"*.jpeg",	GTK_FILE_LIST_IMG},
  {"*.gif",	GTK_FILE_LIST_IMG},
  {"*.teo",	GTK_FILE_LIST_IMG},
  {"*.tif",	GTK_FILE_LIST_IMG},
  {"*.tiff",	GTK_FILE_LIST_IMG},
  {"*.png",	GTK_FILE_LIST_IMG},
  {"*.tar",	GTK_FILE_LIST_ARCH},
  {"*.tar.bz2",	GTK_FILE_LIST_ARCH},
  {"*.bz2",	GTK_FILE_LIST_ARCH},
  {"*.zip",	GTK_FILE_LIST_ARCH},
  {"*.tar.gz",	GTK_FILE_LIST_PKG},
  {"*.tgz",	GTK_FILE_LIST_PKG},
  {"*.rpm",	GTK_FILE_LIST_RPM},
  {"*.deb",	GTK_FILE_LIST_DEB},
  {"*.htm",	GTK_FILE_LIST_HTML},
  {"*.html",	GTK_FILE_LIST_HTML},
  {"*.mpeg",	GTK_FILE_LIST_MOVIE},
  {"*.mpeg1",	GTK_FILE_LIST_MOVIE},
  {"*.mpeg2",	GTK_FILE_LIST_MOVIE},
  {"*.mpg",	GTK_FILE_LIST_MOVIE},
  {"*.mpg1",	GTK_FILE_LIST_MOVIE},
  {"*.mpg2",	GTK_FILE_LIST_MOVIE},
  {"*.avi",	GTK_FILE_LIST_MOVIE},
  {"*.mov",	GTK_FILE_LIST_MOVIE},
  {"core",	GTK_FILE_LIST_CORE}, 
  {NULL, 	0}, 
};

/* ************************************************************************* */
GtkType
gtk_file_list_get_type (void) {
  static GtkType	file_list_type = 0;

  if (!file_list_type) {
    GtkTypeInfo file_list_info = {
      "GtkFileList",
      sizeof (GtkFileList),
      sizeof (GtkFileListClass),
      (GtkClassInitFunc)  gtk_file_list_class_init,
      (GtkObjectInitFunc) gtk_file_list_init,
      NULL,
      NULL,
      (GtkClassInitFunc) NULL,
    };
    file_list_type
      = gtk_type_unique (GNOME_TYPE_FILE_ICON_LIST, &file_list_info);
  }
  return file_list_type;
}

/* ************************************************************************* */
static void
gtk_file_list_class_init (GtkFileListClass	*class) {
  GtkWidgetClass	*widget_class;
  GtkObjectClass 	*object_class;

  widget_class = (GtkWidgetClass *) class;
  object_class = (GtkObjectClass *) class;
  parent_class = gtk_type_class (GNOME_TYPE_FILE_ICON_LIST);

  widget_class->realize = gtk_file_list_realize;
  object_class->destroy = gtk_file_list_destroy; 
}

/* ************************************************************************* */
static void
gtk_file_list_init (GtkFileList	*filelist) {
  GtkFileListType	*types;
  guint8		*pixmap_data[GTK_FILE_LIST_CORE+1];
  gint 			n;

  filelist->path	 = NULL;
  filelist->filter	 = NULL;  
  filelist->show_folders = TRUE;
  filelist->show_hidden  = TRUE;
  filelist->sort_mode	 = GTK_FILE_LIST_SORT_TYPE;
  filelist->filter_num   = 0;

  pixmap_data[GTK_FILE_LIST_FILE]	= (guint8 *) file_pixbuf;
  pixmap_data[GTK_FILE_LIST_FOLDER]	= (guint8 *) folder_pixbuf;
  pixmap_data[GTK_FILE_LIST_EXEC] 	= (guint8 *) exec_pixbuf;
  pixmap_data[GTK_FILE_LIST_CPP] 	= (guint8 *) cpp_pixbuf;
  pixmap_data[GTK_FILE_LIST_C] 		= (guint8 *) c_pixbuf;
  pixmap_data[GTK_FILE_LIST_O] 		= (guint8 *) o_pixbuf;  
  pixmap_data[GTK_FILE_LIST_H] 		= (guint8 *) h_pixbuf;
  pixmap_data[GTK_FILE_LIST_F] 		= (guint8 *) f_pixbuf;
  pixmap_data[GTK_FILE_LIST_JAVA] 	= (guint8 *) java_pixbuf;
  pixmap_data[GTK_FILE_LIST_DOC] 	= (guint8 *) doc_pixbuf;
  pixmap_data[GTK_FILE_LIST_SOUND] 	= (guint8 *) sound_pixbuf;
  pixmap_data[GTK_FILE_LIST_PS] 	= (guint8 *) ps_pixbuf;
  pixmap_data[GTK_FILE_LIST_PDF] 	= (guint8 *) pdf_pixbuf;
  pixmap_data[GTK_FILE_LIST_TEXT] 	= (guint8 *) text_pixbuf;
  pixmap_data[GTK_FILE_LIST_MAKE] 	= (guint8 *) make_pixbuf;  
  pixmap_data[GTK_FILE_LIST_IMG] 	= (guint8 *) image_pixbuf;
  pixmap_data[GTK_FILE_LIST_ARCH] 	= (guint8 *) arch_pixbuf;
  pixmap_data[GTK_FILE_LIST_PKG] 	= (guint8 *) package_pixbuf;
  pixmap_data[GTK_FILE_LIST_DEB] 	= (guint8 *) deb_pixbuf;
  pixmap_data[GTK_FILE_LIST_RPM] 	= (guint8 *) rpm_pixbuf;
  pixmap_data[GTK_FILE_LIST_HTML] 	= (guint8 *) html_pixbuf;
  pixmap_data[GTK_FILE_LIST_MOVIE] 	= (guint8 *) movie_pixbuf;
  pixmap_data[GTK_FILE_LIST_CAT] 	= (guint8 *) cat_pixbuf;
  pixmap_data[GTK_FILE_LIST_CORE] 	= (guint8 *) core_pixbuf;

  filelist->ntypes = 0;

  for (n = 0; n <= GTK_FILE_LIST_CORE; n++) {
    gtk_file_list_add_type (filelist, (guint8 *) pixmap_data[n]);
  }
  types = default_types;
  while (types->extension) {
    gtk_file_list_add_type_filter (filelist, types->type, types->extension);
    types++;
  }
}

/* ************************************************************************* */
GtkWidget*
gtk_file_list_new (guint		icon_width,
		   GtkSelectionMode	mode,
		   const gchar 		*path,
		   gint			flags) {
  GtkWidget		*widget;
  GnomeFileIconList	*iconlist;
  GtkFileList		*filelist;

  widget = gtk_type_new (gtk_file_list_get_type ());

  iconlist = GNOME_FILE_ICON_LIST (widget);
  filelist = GTK_FILE_LIST (widget);

  gnome_file_icon_list_construct (GNOME_FILE_ICON_LIST (filelist),
				  DEFAULT_ICON_WIDTH,
				  DEFAULT_TEXT_WIDTH,
				  NULL,
				  GNOME_FILE_ICON_LIST_STATIC_TEXT);
  if (path) {
    filelist->path = g_strdup (path);
  } else {
    filelist->path = g_strdup (G_DIR_SEPARATOR_S);
  }
  return widget;
}

/* ************************************************************************* */
static void
gtk_file_list_realize (GtkWidget	*widget) {
  GtkFileList 	*filelist;

  GTK_WIDGET_CLASS (parent_class)->realize (widget);

  filelist = GTK_FILE_LIST (widget);
  gtk_file_list_open_dir (filelist, filelist->path); 
}

/* ************************************************************************* */
static void
gtk_file_list_destroy (GtkObject	*object) {
  gint	n;

  gnome_file_icon_list_clear (GNOME_FILE_ICON_LIST (object));
  
  for (n = 0; n < GTK_FILE_LIST (object)->filter_num; n++) {
    g_free (GTK_FILE_LIST (object)->filter[n]);
  }
  g_free (GTK_FILE_LIST (object)->filter);  
}

/* ************************************************************************* */
static gint
sort_list (gpointer	a,
	   gpointer	b) {
  GtkFileListItem 	*filea;
  GtkFileListItem 	*fileb;
  gint 			compare_value;

  filea = (GtkFileListItem *) a;
  fileb = (GtkFileListItem *) b;
#if 0
  switch (filelist->sort_mode) {
  case GTK_FILE_LIST_SORT_TYPE:
    compare_value = filea->type - fileb->type;
    if (compare_value == 0) {
      compare_value = strcmp (itema->label, itemb->label);
    }
    break;
  case GTK_FILE_LIST_SORT_NAME:
  default:
    compare_value = strcmp (itema->label, itemb->label);
    if (filea->type == GTK_FILE_LIST_FOLDER ||
	fileb->type == GTK_FILE_LIST_FOLDER) { 
      compare_value = filea->type - fileb->type;
      if (compare_value == 0) {
	compare_value = strcmp (itema->label, itemb->label);
      }
    }
    break;
  }
#else
  compare_value = filea->type - fileb->type;
  if (compare_value == 0) {
    compare_value = strcmp (filea->file_name, fileb->file_name);
  }
#endif
  return compare_value;
}

/* ************************************************************************* */
static gboolean
check_dir_extra (gchar		*dir_name,
		 struct stat 	*result,
		 gboolean 	*stat_subdirs) {
  /*
   * A list of directories that we know only contain other directories.
   * Trying to stat every file in these directories would be very
   * expensive.
   */
  static struct {
    gchar	*name;
    gboolean 	present;
    struct stat statbuf;
  } no_stat_dirs[] = {
    { "/afs", FALSE, { 0 } },
    { "/net", FALSE, { 0 } }
  };
  static gboolean 	initialized = FALSE;  
  static const gint 	n_no_stat_dirs
    = sizeof (no_stat_dirs) / sizeof(no_stat_dirs[0]);
  gint 			i;

  if (!initialized) {
    initialized = TRUE;
    for (i = 0; i < n_no_stat_dirs; i++) {
      if (stat (no_stat_dirs[i].name, &no_stat_dirs[i].statbuf) == 0) {
	no_stat_dirs[i].present = TRUE;
      }
    }
  }
  if (stat(dir_name, result) < 0) return FALSE;

  *stat_subdirs = TRUE;

  for (i = 0; i < n_no_stat_dirs; i++) {
    if (no_stat_dirs[i].present &&
	(no_stat_dirs[i].statbuf.st_dev == result->st_dev) &&
	(no_stat_dirs[i].statbuf.st_ino == result->st_ino)) {
      *stat_subdirs = FALSE;
      break;
    }
  }
  return TRUE;
}

/* ************************************************************************* */
static void
gtk_file_list_parse_filter (GtkFileList	*filelist,
			    gchar	*filter_text) {
  char	*ptr;
  char	*dst_ptr;
  char	buff[256];
  int	n;

  /* Ťե륿κ */
  if (filelist->filter) {
    for (n = 0; n < filelist->filter_num; n++) {
      g_free (filelist->filter[n]);
    }
    g_free (filelist->filter);
  }
  /* ڤʸ,򥹥åפ */
  ptr = filter_text;
  while ((*ptr == '|' || *ptr == ' ') && *ptr != '\0') ptr++;

  /* ե륿ο */
  if (*ptr == '\0') {
    filelist->filter     = NULL;
    filelist->filter_num = 0;
  } else {
    n = 1;
    while (*ptr != '\0') {
      if (*ptr == '|' && *(ptr+1) != '\0') {
	while ((*ptr == '|' || *ptr == ' ') && *ptr != '\0') ptr++;
	if (*ptr != '\0') n++;
      } else {
	ptr++;
      }
    }
    filelist->filter_num = n;

    /* ե륿μФ */
    {
      /* ΰ */
      filelist->filter = (gchar **) g_malloc (sizeof (gchar *) * n);

      /* ڤʸ,򥹥åפ */
      ptr = filter_text;
      while ((*ptr == '|' || *ptr == ' ') && *ptr != '\0') ptr++;
  
      /* ե륿Ф */
      n = 0;
      dst_ptr = buff;
      while (*ptr != '\0') {
	if (*ptr == '|' && *(ptr+1) != '\0') {
	  while ((*ptr == '|' || *ptr == ' ') && *ptr != '\0') ptr++;
	  if (*ptr != '\0') {
	    *dst_ptr = '\0';
	    filelist->filter[n] = (gchar *) g_strdup (buff);
	    dst_ptr = buff;
	    n++;
	  }
	} else {
	  *dst_ptr++ = *ptr++;
	}
      }
      *dst_ptr = '\0';
      filelist->filter[n] = (gchar *) g_strdup (buff);
    }
  }
}

/* ************************************************************************* *
 *
 * ************************************************************************* */

/* ************************************************************************* */
gboolean
gtk_file_list_open_dir (GtkFileList	*filelist,
			const gchar 	*dir_path) {
  DIR			*dir;
  struct dirent 	*dirent;
  struct stat 		fileinfo, linkinfo; 
  GnomeFileIconList 	*iconlist;
  GtkFileListItem 	*file_item;
  GtkFileListType 	*file_type;
  GtkWidget 		*widget;
  GdkPixbuf		*pixbuf;
  gboolean 		show_file;
  gboolean 		stat_subdirs = TRUE;
  gint 			type;
  guint			icon_num;
  gchar 		*full_name, *file, *real_path, *locale_name;
  GList 		*types, *files, *list;
  gint			n;

  widget   = GTK_WIDGET (filelist);
  iconlist = GNOME_FILE_ICON_LIST (widget);

  if (!slink) {
    slink = gdk_pixbuf_new_from_inline (-1, slink_pixbuf, FALSE, NULL);
  }
  real_path = g_get_absolute_path (dir_path);

  if (strcmp (real_path, filelist->path) == 0) return FALSE;
  
  if ((dir = opendir (real_path)) == NULL) {
    g_warning ("Can not open folder: %s", real_path);
    g_free (real_path);
    return FALSE;
  }
  if (!check_dir_extra (real_path, &fileinfo, &stat_subdirs)) {
    closedir (dir);
    g_warning ("Can not stat folder: %s", real_path);
    g_free (real_path);
    return FALSE;
  }
  if (filelist->path) g_free (filelist->path);
  filelist->path = g_strdup (real_path);

  /* UPDATE ICON LIST */
  gnome_file_icon_list_freeze (iconlist);
  icon_num = gnome_file_icon_list_get_num_icons (iconlist);
  for (n = 0; n < icon_num; n++) {
    file_item = (GtkFileListItem *)
      gnome_file_icon_list_get_icon_data (iconlist, n);
    g_free (file_item->file_name);
  }
  gnome_file_icon_list_clear (iconlist);

  gdk_threads_enter ();

  files = NULL;
  while ((dirent = readdir (dir)) != NULL) {
    full_name = g_build_filename (real_path, dirent->d_name, NULL);
    file = dirent->d_name;
    show_file = TRUE;
    if (stat_subdirs) {
      if (0 == stat (full_name, &fileinfo)) {
	if (S_ISDIR (fileinfo.st_mode)) {
	  type = GTK_FILE_LIST_FOLDER;
	} else {
	  type = GTK_FILE_LIST_FILE;
	  if (fileinfo.st_mode & 0111) type = GTK_FILE_LIST_EXEC;
	}
      } else {
	continue;
      }
    } else {
      type = GTK_FILE_LIST_FOLDER;
    }
    if (filelist->show_folders && type == GTK_FILE_LIST_FOLDER) {
      show_file = TRUE;
    }
    if (file[0] == '.') {
      if (filelist->show_hidden == TRUE) {
	show_file = TRUE;
      } else {
	show_file = FALSE;
      }
      if (filelist->show_folders &&
	  (strcmp (file, ".") == 0 || strcmp (file, "..") == 0)) {
	show_file = TRUE;
      }
    }
    if (strcmp (full_name, "/.") == 0 || strcmp (full_name, "/..") == 0) {
      show_file = FALSE;
    }
    if (!filelist->show_folders && type == GTK_FILE_LIST_FOLDER) {
      show_file = FALSE;
    }
    if (filelist->filter_num != 0) {
      show_file = FALSE;
      for (n = 0; n < filelist->filter_num; n++) {
	if (filelist->filter[n] &&
	    fnmatch (filelist->filter[n], file, (1 << 4)) == 0) {
	  show_file = TRUE;
	  break;
	}
      }
    }
    if (show_file) {
      file_item = (GtkFileListItem *) g_new (GtkFileListItem, 1);
      file_item->file_name = g_strdup (file);
      file_item->type      = type;
      file_item->is_link   = FALSE;
      if (stat_subdirs) {
	if (0 != lstat (full_name, &linkinfo)) {
	  g_warning ("Can not resolve link: %s",full_name);
	}
	if (S_ISLNK (linkinfo.st_mode)) file_item->is_link = TRUE;
      }
      files = g_list_append (files, file_item);
    }
    g_free (full_name);
  }
  closedir (dir);

  /* Υ */  
  list = files;
  while (list) {
    file_item = (GtkFileListItem *) list->data;
    type      = file_item->type;
    types     = filelist->types;
    while (types) {
      file_type = (GtkFileListType *) types->data;
      if (fnmatch ((gchar *) file_type->extension, file_item->file_name,
		   (1 << 4)) == 0) {
	type = file_type->type; 
	break;
      }
      types = types->next;
    }
    file_item->type = type;
    list = list->next;
  }
  files = g_list_sort (files, (GCompareFunc) sort_list);

  list = files;
  while (list) {
    file_item = (GtkFileListItem *) list->data;
    locale_name = g_strdup (file_item->file_name);
    g_free (file_item->file_name);
    file_item->file_name = g_filename_to_utf8 (locale_name, 
					       -1, NULL, NULL, NULL);
#if 0
    pixbuf = gdk_pixbuf_new_from_inline (-1,
					 g_list_nth_data (filelist->pixmaps,
							  file_item->type),
					 FALSE, NULL);
#else
    pixbuf = gdk_pixbuf_copy (g_list_nth_data (filelist->pixmaps,
					       file_item->type));
#endif
    if (file_item->is_link) {
      gdk_pixbuf_copy_area (slink,
			    0, 0,
			    gdk_pixbuf_get_width  (slink),
			    gdk_pixbuf_get_height (slink),
			    pixbuf,
			    gdk_pixbuf_get_width  (pixbuf)-7,
			    gdk_pixbuf_get_height (pixbuf)-7);
    }
    n = gnome_file_icon_list_append_pixbuf (iconlist, pixbuf, locale_name,
					    file_item->file_name);
    gnome_file_icon_list_set_icon_data (iconlist, n, file_item);
    g_free (locale_name);
    list = list->next;
  }
  gdk_threads_leave ();

  gnome_file_icon_list_thaw (iconlist);

  g_list_free (files);

  return TRUE;
}

/* ************************************************************************* */
gboolean
gtk_file_list_refresh (GtkFileList	*filelist) {
  DIR			*dir;
  struct dirent 	*dirent;
  struct stat 		fileinfo, linkinfo; 
  GnomeFileIconList 	*iconlist;
  GtkFileListItem 	*file_item;
  GtkFileListType 	*file_type;
  GtkWidget 		*widget;
  GdkPixbuf		*pixbuf;
  gboolean 		show_file;
  gboolean 		stat_subdirs = TRUE;
  gint 			type;
  guint			icon_num;
  gchar 		*full_name, *file, *locale_name;
  GList 		*types, *files, *list;
  gint			n;

  widget   = GTK_WIDGET (filelist);
  iconlist = GNOME_FILE_ICON_LIST (widget);

  if (!slink) {	
    slink = gdk_pixbuf_new_from_inline (-1, slink_pixbuf, FALSE, NULL);
  }
  if ((dir = opendir (filelist->path)) == NULL) {
    g_warning ("Can not open folder: %s", filelist->path);
    return FALSE;
  }
  if (!check_dir_extra (filelist->path, &fileinfo, &stat_subdirs)) {
    closedir (dir);
    g_warning ("Can not stat folder: %s", filelist->path);
    return FALSE;
  }
  /* UPDATE ICON LIST */
  gnome_file_icon_list_freeze (iconlist);
  icon_num = gnome_file_icon_list_get_num_icons (iconlist);
  for (n = 0; n < icon_num; n++) {
    file_item = (GtkFileListItem *)
      gnome_file_icon_list_get_icon_data (iconlist, n);
    g_free (file_item->file_name);
  }
  gnome_file_icon_list_clear (iconlist);

  gdk_threads_enter ();

  files = NULL;
  while ((dirent = readdir (dir)) != NULL) {
    full_name = g_build_filename (filelist->path, dirent->d_name, NULL);
    file = dirent->d_name;
    show_file = TRUE;
    if (stat_subdirs) {
      if (0 == stat (full_name, &fileinfo)) {
	if (S_ISDIR (fileinfo.st_mode)) {
	  type = GTK_FILE_LIST_FOLDER;
	} else {
	  type = GTK_FILE_LIST_FILE;
	  if (fileinfo.st_mode & 0111) type = GTK_FILE_LIST_EXEC;
	}
      } else {
	continue;
      }
    } else {
      type = GTK_FILE_LIST_FOLDER;
    }
    if (filelist->show_folders && type == GTK_FILE_LIST_FOLDER) {
      show_file = TRUE;
    }
    if (file[0] == '.') {
      if (filelist->show_hidden == TRUE) {
	show_file = TRUE;
      } else {
	show_file = FALSE;
      }
      if (filelist->show_folders &&
	  (strcmp (file, ".") == 0 || strcmp (file, "..") == 0)) {
	show_file = TRUE;
      }
    }
    if (strcmp (full_name, "/.") == 0 || strcmp (full_name, "/..") == 0) {
      show_file = FALSE;
    }
    if (!filelist->show_folders && type == GTK_FILE_LIST_FOLDER) {
      show_file = FALSE;
    }
    if (filelist->filter_num != 0) {
      show_file = FALSE;
      for (n = 0; n < filelist->filter_num; n++) {
	if (filelist->filter[n] &&
	    fnmatch (filelist->filter[n], file, (1 << 4)) == 0) {
	  show_file = TRUE;
	  break;
	}
      }
    }
    if (show_file) {
      file_item = (GtkFileListItem *) g_new (GtkFileListItem, 1);
      file_item->file_name = g_strdup (file);
      file_item->type      = type;
      file_item->is_link   = FALSE;
      if (stat_subdirs) {
	if (0 != lstat (full_name, &linkinfo)) {
	  g_warning ("Can not resolve link: %s", full_name);
	}
	if (S_ISLNK (linkinfo.st_mode)) file_item->is_link = TRUE;
      }
      files = g_list_append (files, file_item);
    }
    /* Υ */
    files = g_list_sort (files, (GCompareFunc) sort_list);
    
    g_free (full_name);
  }
  closedir (dir);
  
  list = files;
  while (list) {
    file_item = (GtkFileListItem *) list->data;
    type      = file_item->type;
    types     = filelist->types;
    while (types) {
      file_type = (GtkFileListType *) types->data;
      if (fnmatch ((gchar *) file_type->extension, file_item->file_name,
		   (1 << 4)) == 0) {
	type = file_type->type; 
	break;
      }
      types = types->next;
    }
#if 0
    pixbuf = gdk_pixbuf_new_from_inline (-1,
					 g_list_nth_data (filelist->pixmaps,
							  type),
					 FALSE, NULL);
#else
    pixbuf = gdk_pixbuf_copy (g_list_nth_data (filelist->pixmaps,
					       file_item->type));
#endif
    file_item->type = type;
    locale_name = g_strdup (file_item->file_name);
    g_free (file_item->file_name);
    file_item->file_name = g_filename_to_utf8 (locale_name, 
					       -1, NULL, NULL, NULL);
    if (file_item->is_link) {
      gdk_pixbuf_copy_area (slink,
			    0, 0,
			    gdk_pixbuf_get_width  (slink),
			    gdk_pixbuf_get_height (slink),
			    pixbuf,
			    gdk_pixbuf_get_width  (pixbuf)-7,
			    gdk_pixbuf_get_height (pixbuf)-7);
    }
    n = gnome_file_icon_list_append_pixbuf (iconlist, pixbuf, locale_name,
					    file_item->file_name);
    gnome_file_icon_list_set_icon_data (iconlist, n, file_item);
    g_free (locale_name);
    list = list->next;
  }
  gdk_threads_leave ();

  gnome_file_icon_list_thaw (iconlist);

  g_list_free (files);

  return TRUE;
}

/* ************************************************************************* */
void
gtk_file_list_set_filter (GtkFileList	*filelist,
			  const gchar	*filter) {
  gtk_file_list_parse_filter (filelist, (gchar *) filter);
  gtk_file_list_refresh (filelist);
}

/* ************************************************************************* */
gchar*
gtk_file_list_get_path (GtkFileList	*filelist) {
  return filelist->path;
}

/* ************************************************************************* */
gchar*
gtk_file_list_get_filename (GtkFileList *filelist) {
  GnomeFileIconTextItem	*iti = NULL;
  GList			*list;
  gpointer		selection = NULL;
  gchar			*file;

  file = NULL;

  list = gnome_file_icon_list_get_selection (GNOME_FILE_ICON_LIST (filelist));

  if (list) selection = list->data;

  g_print ("Id = %d\n", (int) selection);
  
  iti =
    gnome_file_icon_list_get_icon_text_item (GNOME_FILE_ICON_LIST (filelist),
					     (int) selection);
  if (iti) file = (gchar *) gnome_file_icon_text_item_get_text (iti);

  return file;
}

/* ************************************************************************* */
gint 
gtk_file_list_get_filetype (GtkFileList	*filelist) {
  GList		*list;
  gpointer 	selection = NULL;
  gint 		type = -1;

  list = gnome_file_icon_list_get_selection (GNOME_FILE_ICON_LIST (filelist));

  if (list) selection=list->data;
#if 0
  if (selection) {
    selection = ((GtkIconCanvasItem *) selection)->link;
    type = ((GtkFileListItem *) selection)->type;
  }
#endif
  return type;
}

/* ************************************************************************* */
gint
gtk_file_list_add_type (GtkFileList	*filelist,
			guint8		*pixmap_data) {
  GdkPixbuf	*pixbuf;

  pixbuf = gdk_pixbuf_new_from_inline (-1, pixmap_data, FALSE, NULL);
  filelist->pixmaps = g_list_append (filelist->pixmaps, pixbuf);
  filelist->ntypes++;

  return filelist->ntypes - 1;
}

/* ************************************************************************* */
void
gtk_file_list_add_type_filter (GtkFileList	*filelist,
			       gint 		type,
			       const gchar 	*filter) {
  GtkFileListType	*type_item;

  type_item = g_new0 (GtkFileListType, 1);
  type_item->type = type;
  type_item->extension = g_strdup (filter);

  filelist->types = g_list_append (filelist->types, type_item);

  return;
}

/* ************************************************************************* */
void
gtk_file_list_set_icon_from_file (GtkFileList	*filelist,
				  const gchar	*filename) {
  GdkPixbuf	*pixbuf;

  pixbuf = gdk_pixbuf_new_from_file (filename, NULL);
  if (pixbuf) {
    filelist->pixmaps = g_list_first (filelist->pixmaps);
    filelist->pixmaps->data = (gpointer) pixbuf;
  }
}

/* ************************************************************************* */
void
gtk_file_list_set_icon_from_pixbuf (GtkFileList		*filelist,
				    const GdkPixbuf	*pixbuf) {
  if (pixbuf) {
    filelist->pixmaps = g_list_first (filelist->pixmaps);
    filelist->pixmaps->data = (gpointer) pixbuf;
  }
}

/* ************************************************************************* */
void
gtk_file_list_set_icon_from_inline (GtkFileList		*filelist,
				    const guint8	*data) {
  GdkPixbuf	*pixbuf;

  pixbuf = gdk_pixbuf_new_from_inline (-1, data, FALSE, NULL);
  if (pixbuf) {
    filelist->pixmaps = g_list_first (filelist->pixmaps);
    filelist->pixmaps->data = (gpointer) pixbuf;
  }
}

/* ************************************************ End of gtkfilelist.c *** */
