/* ********************************************************** image_io.c *** *
 * $B2hA|F~=PNO4X?t(B
 *
 * Copyright (C) 2002-2003 Yasuyuki SUGAYA <sugaya@suri.it.okayama-u.ac.jp>
 * Okayama University
 *                                    Time-stamp: <04/06/21 15:21:04 sugaya>
 * ************************************************************************* */
#include <dlfcn.h>
#include "teoeyes.h"
#include "gfileutils.h"
#include "image.h"
#include "image_io.h"
#include "image_loader.h"

/* ************************************************************************* */


/* ************************************************************************* *
 * $BFbIt4X?t(B
 *
 * check_format
 * ************************************************************************* */

/* $B%U%)!<%^%C%H$N%A%'%C%/(B ************************************************** */
static gboolean
check_format (TeoeyesImageModule	*module,
	      guchar			*buffer,
	      gint			size) {
  ImageFormatPattern	*pattern;
  gint			n, m;

  for (pattern = module->info->signature; pattern->prefix; pattern++) {
    for (n = 0; n < size && pattern->prefix[n] != 0; n++) {
      m = pattern->mask ? pattern->mask[n] : ' ';
      if (m == ' ') {
	if (buffer[n] != pattern->prefix[n]) break;
      } else if (m == '!') {
	if (buffer[n] == pattern->prefix[n]) break;
      } else if (m == 'z') {
	if (buffer[n] != 0) break;
      } else if (m == 'n') {
	if (buffer[n] == 0) break;
      }
    }
    if (pattern->prefix[n] == 0) return TRUE;
  }
  return FALSE;
}

/* ************************************************************************* *
 * $B%0%m!<%P%k4X?t(B
 *
 * select_module_from_file
 * select_module_from_extension
 * loader_init
 * loader_close
 * check_image_supported
 * teoeyes_image_open
 * teoeyes_image_load
 * ************************************************************************* */

/* $B%U%!%$%k$+$i%m!<%@$rA*Br(B ************************************************ */
TeoeyesImageModule*
select_module_from_file (const gchar	*filename) {
  TeoeyesImageModule	*module;
  FILE			*fp;
  guchar		buffer[128];
  GList			*list;
  gint			size;

  fp =fopen (filename, "rb");
  if (!fp) {
    g_printerr (_("Failed to open file '%s'.\n"), filename);
    return FALSE;
  }
  size = fread (&buffer, 1, sizeof (buffer), fp);
  fclose (fp);
  if (size == 0) return FALSE;
  for (list = image_loader; list; list = g_list_next (list)) {
    module = dlsym (list->data, "module");
    if (check_format (module, buffer, size)) return module;      
  }
  return NULL;
}

/* $B3HD%;R$+$i%m!<%@$rA*Br(B ************************************************** */
TeoeyesImageModule*
select_module_from_extension (const gchar	*extension) {
  TeoeyesImageModule	*module;
  GList			*list;
  gint			n;
  
  for (list = image_loader; list; list = g_list_next (list)) {
    module = dlsym (list->data, "module");
    for (n = 0; module->info->extensions[n]; n++) {
      if (strcmp (extension, module->info->extensions[n]) == 0) {
	return module;
      }
    }
  }
  return NULL;
}

/* $B%b%8%e!<%k$NFI$_9~$_(B **************************************************** */
void
loader_init (GList	*loader_path) {
  GList			*loader_list;   
  GList			*loader;  
  GList			*list;
  void			*handle;
  gchar			*module_name;
  TeoeyesImageModule	*module;

  for (list = loader_path; list; list = g_list_next (list)) {
    loader_list = g_dir_ls (list->data);
    for (loader = loader_list; loader; loader = g_list_next (loader)) {
      module_name = g_strconcat (list->data, 
				 G_DIR_SEPARATOR_S, loader->data, NULL);
      handle = dlopen (module_name, RTLD_LAZY);
      if (handle) {
	module = dlsym (handle, "module");
	if (module) {
	  module->load_info (module);
	  image_loader = g_list_append (image_loader, handle);
	}
      }
      g_free (module_name);
    }
    g_list_foreach (loader_list, (GFunc) g_free, NULL);
  }
}

/* $B%b%8%e!<%k$N2rJ|(B ******************************************************** */
void
loader_close (GList	*loader) {
  TeoeyesImageModule	*module;
  GList			*list;

  for (list = loader; list; list = g_list_next (list)) {
    module = dlsym (list->data, "module");
    g_free (module->info);
    dlclose (list->data);
  }
  image_loader = NULL;
}

/* $B%5%]!<%H$5$l$F$$$k2hA|%?%$%W$+D4$Y$k(B ************************************ */
gboolean
check_image_supported (const gchar	*filename) {
  TeoeyesImageModule	*module = NULL, *m = NULL;
  gchar			*ext = NULL;
  
  module = select_module_from_file (filename);
  if (module && strcmp (module->module_name, "TEO") == 0) {
    ext = strrchr (filename, '.');
    if (!ext) return FALSE;
    ext++;
    m = select_module_from_extension (ext);
    return (m) ? TRUE : FALSE;
  }
  return (module) ? TRUE : FALSE;
}

/* $B2hA|%G!<%?$NFI$_9~$_(B **************************************************** *
 *
 * GdkPixbuf*
 * teoeyes_image_open (TEImage	*ti);
 *
 *	ti    : TEImage$B9=B$BN$X$N%]%$%s%?(B
 *
 * ************************************************************************* */
GdkPixbuf*
teoeyes_image_open (TEImage	*ti) {
  TeoeyesImageModule	*module;
  GdkPixbuf		*pixbuf = NULL;
  
  if ((module = select_module_from_file (ti->filename))) {
    if (strcmp (module->info->name, "TEO") == 0) {
      ti->user_data2 = (gpointer) to->load_all;
      return module->load (ti->filename, ti);
    } else {
      pixbuf = module->load (ti->filename, NULL);
      if (pixbuf) {
	ti->pixbuf_list = g_list_append (ti->pixbuf_list, pixbuf);
	if (ti->frame == -1) ti->frame = g_list_length (ti->pixbuf_list) - 1;
	return (GdkPixbuf *) g_list_nth_data (ti->pixbuf_list, ti->frame);
      }
    }
  }
  return NULL;
}

/* $B2hA|$NFI$_9~$_(B ********************************************************** *
 *
 * void
 * teoeyes_image_load (TEImage	*ti,	
 *		       gint	frame,
 *		       gdouble	min,
 *		       gdouble	max);
 *
 *	ti    : TEImage$B9=B$BN$X$N%]%$%s%?(B
 *	frame : $BFI$_9~$`%U%l!<%`HV9f(B
 * 	min   : $B@55,2=%Q%i%a!<%?$N:G>.CM(B
 * 	max   : $B@55,2=%Q%i%a!<%?$N:GBgCM(B
 *
 * ************************************************************************* */
gboolean
teoeyes_image_load (TEImage	*ti,
		    gint	frame,
		    gdouble	min,
		    gdouble	max) {
  ti->frame = frame;
  ti->min   = min;
  ti->max   = max;

  if (ti->pixbuf_list) {
    g_list_foreach (ti->pixbuf_list, (GFunc) gdk_pixbuf_unref, NULL);
    ti->pixbuf_list = NULL;
    ti->pixbuf      = NULL;
  }
  ti->pixbuf = teoeyes_image_open (ti);

  if (ti->pixbuf) {
    ti->has_alpha	= gdk_pixbuf_get_has_alpha  (ti->pixbuf);
    if (ti->nchannels == 3) {
      ti->nchannels	= gdk_pixbuf_get_n_channels (ti->pixbuf);    
    }
    ti->width		= gdk_pixbuf_get_width      (ti->pixbuf);
    ti->height		= gdk_pixbuf_get_height     (ti->pixbuf);
    ti->current_width	= gdk_pixbuf_get_width      (ti->pixbuf);
    ti->current_height	= gdk_pixbuf_get_height     (ti->pixbuf);
    
    return TRUE;
  } else {
    return FALSE;
  }
}

/* ************************************************************************* */
void
teoeyes_image_set (TEImage	*ti,
		   gint		frame,
		   gdouble	min,
		   gdouble	max) {
  ti->frame 	     = frame;
  ti->min   	     = min;
  ti->max   	     = max;

  if (!ti->pixbuf_list) {
    ti->pixbuf = teoeyes_image_open (ti);
  } else {
    if (frame == -1) ti->frame = g_list_length (ti->pixbuf_list) - 1;
    ti->pixbuf = (GdkPixbuf *) g_list_nth_data (ti->pixbuf_list, ti->frame);
  }
  ti->has_alpha      = gdk_pixbuf_get_has_alpha  (ti->pixbuf);
  ti->nchannels	     = gdk_pixbuf_get_n_channels (ti->pixbuf);
  ti->width	     = gdk_pixbuf_get_width      (ti->pixbuf);
  ti->height	     = gdk_pixbuf_get_height     (ti->pixbuf);
  ti->current_width  = gdk_pixbuf_get_width      (ti->pixbuf);
  ti->current_height = gdk_pixbuf_get_height     (ti->pixbuf);
}
		   
/* $B2hA|$NJ]B8(B ************************************************************** *
 *
 * void
 * teoeyes_image_save (const gchar 	*filename,
 *		       TEImage		*ti);
 *
 *	filename : $B2hA|%U%!%$%kL>(B 
 *	ti       : TEImage$B9=B$BN$X$N%]%$%s%?(B
 *
 * ************************************************************************* */
gboolean
teoeyes_image_save (const gchar	*filename,
		    TEImage	*ti) {
  TeoeyesImageModule	*module;
  gchar			*ext = NULL;

  /* $B3HD%;R$N3MF@(B */
  ext = strrchr (filename, '.');
  if (!ext || *(ext + 1) == G_STRING_TERMINATOR) {
    g_printerr (_("The extension does not exists.\n"));
    return FALSE;
  }
  ext++;

  /* $B3HD%;R$+$i%b%8%e!<%k$rA*Br(B */
  if ((module = select_module_from_extension (ext))) {
    if ((module->info->flags | IMAGE_FORMAT_WRITABLE) != 0) {
      /* $B2hA|$NJ]B8(B */
      if (strcmp (module->info->name, "TEO") == 0) {
	return module->save (filename, ti);
      } else {
	return module->save (filename, ti->pixbuf);
      }
    } else {
      /* $B%b%8%e!<%k$,=q$-9~$_$r%5%]!<%H$7$F$$$J$$>l9g(B  */
      g_printerr (_("The module for '%s' does not support save function.\n"),
		  ext);
      return FALSE;
    }
  } else {
    /* $B;XDj$7$?3HD%;R$r%5%]!<%H$9$k%b%8%e!<%k$,B8:_$7$J$$>l9g(B */
    g_printerr (_("The module for '%s' does not exists.\n"), ext);
    return FALSE;
  }
}

/* *************************************************** End of image_io.c *** */
