/* ********************************************************** property.c *** *
 * $B@_Dj%U%!%$%k$K4X$9$k4X?t(B
 *
 * Copyright (C) 2001-2003 Yasuyuki SUGAYA <sugaya@suri.it.okayama-u.ac.jp>
 * Okayama University
 *                                    Time-stamp: <04/06/22 01:44:38 sugaya>
 * ************************************************************************* */
#include <X11/keysym.h>
#include <gdk/gdk.h>
#include <gdk/gdkx.h>
#include <ctype.h>
#include "teoeyes.h"
#include "plugin_info.h"
#include "property.h"

#include "gtkiconbutton.h"
#include "gtkiconitemfactory.h"
#include "gtkcustomiconfilesel.h"
#include "xml.h"

#include "config_option.icon"
#include "config_browser.icon"
#include "config_assistant.icon"
#include "config_plugin.icon"
#include "config_shortcuts.icon"
#include "config_misc.icon"

#define	CONFIG_BASE_VERSION		"3.0.0"

#define	XML_TAG_TEOEYES_CONFIG		"teoeyes_config"
#define	XML_TAG_VERSION			"version"
#define	XML_TAG_NAME			"name"
#define	XML_TAG_VALUE			"value"
#define	XML_TAG_ITEM			"item"
#define	XML_TAG_VESTION			"version"
#define	XML_TAG_OPTION			"option"
#define	XML_TAG_OPTION_LOAD_ALL		"option_load_all"
#define	XML_TAG_OPTION_CLEAN_MEM	"option_clean_mem"
#define	XML_TAG_OPTION_NORMALIZE	"option_normalize"
#define	XML_TAG_CONFIRM			"confirm"
#define	XML_TAG_CONFIRM_SAVE		"confirm_save"
#define	XML_TAG_CONFIRM_OVERWRITE	"confirm_overwrite"
#define	XML_TAG_FOLDER_ICON		"folder_icon"
#define	XML_TAG_THUMBNAIL_DIR		"thumbnail_dir"
#define	XML_TAG_THUMBNAIL_WIDTH		"thumbnail_width"
#define	XML_TAG_THUMBNAIL_HEIGHT	"thumbnail_height"
#define	XML_TAG_LOADER_DIRS		"loader_dirs"
#define	XML_TAG_ASSISTANT_SKIN		"assistant_skin"
#define	XML_TAG_ASSISTANT_DOC_DIRS	"assistant_data_dirs"
#define	XML_TAG_PLUGIN_DIRS		"plugin_dirs"
#define	XML_TAG_UNUSE_PLUGIN_LIST	"unuse_plugin_list"
#define	XML_TAG_SHORTCUT		"shortcut"
#define	XML_TAG_LOOPE_SCALE		"loope_scale"
#define	XML_TAG_TITLE_IMAGE		"title_image"
#define	XML_TAG_PLAYER_SKIN		"player_skin"

enum {
  XML_VALUE_TYPE_INT,
  XML_VALUE_TYPE_STRING
};

/* ************************************************************************* */
static void	loope_scale_change 	(GtkWidget	*widget,
					 gpointer	data);
static gboolean	parse_option 		(TEConfig	*tc,
					 GNode		*node,
					 gint		level);
static gboolean	parse_confirm 		(TEConfig	*tc,
					 GNode		*node,
					 gint		level);
static gint	check_list_num 		(GNode		*node,
					 gint		level);
static gboolean	parse_list	 	(GList		**list,
					 GNode		*node,
					 gint		level);
static gchar*	create_shortcut_accl 	(gchar		*string);
static gboolean	parse_shortcut 		(TEConfig	*tc,
					 GNode		*node,
					 gint		level);
static void	print_tag 		(FILE		*fp,
					 const gchar	*tag,
					 guint		tab_level,
					 guint		ent_level);
static void	print_end_tag 		(FILE		*fp,
					 const gchar	*tag,
					 guint		tab_level,
					 guint		ent_level);
static void	print_element 		(FILE		*fp,
					 const gchar	*tag,
					 gpointer	value,
					 guint		value_type,
					 guint		tab_level,
					 guint		ent_level);
static void	cb_dialog_ok 		(GtkWidget	*widget,
					 GtkWidget	*dialog);
static void	cb_dialog_apply 	(GtkWidget	*widget,
					 GtkWidget	*dialog);
static void	cb_dialog_close_real 	(GtkWidget	*dialog,
					 gpointer	data);
static void	cb_dialog_close 	(GtkWidget	*widget,
					 GtkWidget	*dialog);
static void	cb_change_box 		(GtkWidget	*widget,
					 gpointer	data);
static GList*	get_list 		(GtkTreeView	*treeview);
static gchar*	create_shortcut_string 	(gchar		*accel);
static GList*	get_shortcut_list 	(GtkTreeView	*treeview);
static void	check_button_set 	(GtkWidget	*parent,
					 GtkWidget	*box,
					 const gchar	*label,
					 const gchar	*name,
					 gboolean	flag);
static GtkWidget* entry_box_new 	(GtkWidget	*parent,
					 const gchar	*title,
					 const gchar	*name,
					 const gchar	*value);
static GtkWidget* list_box_new 		(GtkWidget	*parent,
					 const gchar	*title,
					 const gchar	*list_title,
					 const gchar	*list_name,
					 const gchar	*entry_name,
					 GList		*value_list,	      
					 gboolean	header_visible);
static GtkWidget* loope_menu_new 	(void);
static GtkWidget* config_box_option 	(GtkWidget	*parent);
static GtkWidget* config_box_explorer 	(GtkWidget	*parent);
static GtkWidget* config_box_assistant 	(GtkWidget	*parent);
static GtkWidget* config_box_plugin 	(GtkWidget	*parent);
static GtkWidget* config_box_shortcut 	(GtkWidget	*parent);
static GtkWidget* config_box_misc 	(GtkWidget	*parent);

/* ************************************************************************* */
static gint		scale = 0;

/* $B%G%U%)%k%H%7%g!<%H%+%C%H@_Dj(B ******************************************** */
static gchar *shortcut_action[] =  {
  N_("New"),  
  N_("Open"),
  N_("Save"),
  N_("Save as"),
  N_("Print"),
  N_("Top Image"),
  N_("Previous Image"),
  N_("Next Image"),
  N_("Bottom Image"),
  N_("Top Frame"),
  N_("Previous Frame"),
  N_("Next Frame"),
  N_("Bottom Frame"),
  N_("Show Plane"),
  N_("Top Plane"),  
  N_("Previous Plane"),
  N_("Next Plane"),
  N_("Bottom Plane"),  
  N_("Double Size"),
  N_("Half Size"),
  N_("Scale Up 10\\%"),
  N_("Scale Down 10\\%"),
  N_("Show Menubar"),
  N_("Undo"),
  N_("Original"),
  N_("Grayscale"),
  N_("Reverse"),
  N_("Normalize"),
  N_("Flip Horizontal"),
  N_("Flip Vertical"),
  N_("Image List Viewer"),
  N_("Image Editor"),
  N_("Image Explorer"),
  N_("Image Player"),
  N_("Assistant"),  
  N_("Image Information"),
  N_("Property"),
  N_("About Teoeyes"),
  N_("Help"),
  N_("Quit")
};

/* $BAu>~;RJ8;zNs(B ************************************************************ */
static char *mod_str[] = {
  "", 
  "<Shift>",		"", "", 
  "<Control>",
  "<Shift><Control>",	"", "",
  "<Alt>", 
  "<Shift><Alt>",	"", "",
  "<Control><Alt>", 
  "<Shift><Control><Alt>"
};

/* $B%k!<%Z$N3HBgN(%a%K%e!<(B ************************************************** */
static GtkItemFactoryEntry loope_menu_items[] = {
  {N_("/x   2"), NULL, loope_scale_change,	2, "<Item>"},
  {N_("/x   4"), NULL, loope_scale_change, 	4, "<Item>"},
  {N_("/x   8"), NULL, loope_scale_change,	8, "<Item>"},
  {N_("/x  16"), NULL, loope_scale_change,      16, "<Item>"},
  {N_("/x  32"), NULL, loope_scale_change,      32, "<Item>"},
  {N_("/x  64"), NULL, loope_scale_change,      64, "<Item>"},
  {N_("/x 128"), NULL, loope_scale_change,     128, "<Item>"}
};

/* ************************************************************************* *
 * $B%3!<%k%P%C%/4X?t(B
 *
 * cb_toggle_active
 * cb_add_list
 * cb_delete_list
 * scale_to_index
 * loope_scale_change
 * cb_reference_dialog_ok
 * cb_reference_dialog_cancel
 * reference_dialog_new
 * cb_reference_dialog
 * ************************************************************************* */

/* ************************************************************************* */
static void
cb_toggle_active (GtkCellRendererToggle	*cell,
		  gchar			*path_str,
		  gpointer		data) {
  GtkTreeModel	*model = (GtkTreeModel *) data;
  GtkTreeIter	iter;
  GtkTreePath	*path = gtk_tree_path_new_from_string (path_str);
  gboolean	active;

  gtk_tree_model_get_iter (model, &iter, path);
  gtk_tree_model_get (model, &iter, 0, &active, -1);
  active ^= 1;
  gtk_list_store_set (GTK_LIST_STORE (model), &iter, 0, active, -1);
  gtk_tree_path_free (path);
}

/* ************************************************************************* */
static void
cb_add_list (GtkWidget	*widget,
	     gpointer	data) {
  GtkEntry		*entry;
  GtkTreeView		*treeview = GTK_TREE_VIEW (data);
  GtkTreeModel		*model;
  GtkTreeIter		iter;

  entry = GTK_ENTRY (G_GET_WIDGET (widget, "entry"));
  model = gtk_tree_view_get_model (treeview);
  if (gtk_entry_get_text (entry) &&
      strcmp (gtk_entry_get_text (entry), "") != 0) {
    gtk_list_store_append (GTK_LIST_STORE (model), &iter);
    gtk_list_store_set (GTK_LIST_STORE (model), &iter, 0,
			g_strdup (gtk_entry_get_text (entry)), -1);
  }
}

/* ************************************************************************* */
static void
cb_delete_list (GtkWidget	*widget,
		gpointer	data) {
  GtkTreeView		*treeview = GTK_TREE_VIEW (data);
  GtkTreeSelection	*selection;
  GtkTreeModel		*model;
  GtkTreeIter		iter;
  gint			success;
  
  selection = gtk_tree_view_get_selection (treeview);
  if (!selection) return;

  success = gtk_tree_selection_get_selected (selection, &model, &iter);
  if (!success) return;

  gtk_list_store_remove (GTK_LIST_STORE (model), &iter);
}

/* $B3HBgN($+$i%a%K%e!<$N%$%s%G%C%/%9$r5a$a$k(B ******************************** */
static gint
scale_to_index (gint	scale) {
  gint	n = 0;

  while (scale != 2) {
    scale = scale >> 1;
    n++;
  }
  return n;
}

/* $B3HBgN($NJQ99(B ************************************************************ */
static void
loope_scale_change (GtkWidget		*widget,
		    gpointer		data) {
  scale	= (gint) data;
}

/* $B;2>H%@%$%"%m%0$N!V(BOK$B!W%\%?%s$N%3!<%k%P%C%/4X?t(B ************************** */
static void
cb_reference_dialog_ok (GtkWidget		*widget,
			GtkCustomIconFileSel	*dialog) {
  GtkEntry	*entry;

  entry = GTK_ENTRY (G_GET_PARAMETER (dialog, "entry"));
  gtk_entry_set_text (entry,
		      gtk_custom_icon_file_selection_get_filename (dialog));
  gtk_widget_hide (GTK_WIDGET (dialog));
  gtk_widget_destroy (GTK_WIDGET (dialog));
}

/* $B;2>H%@%$%"%m%0$N!V(BOK$B!W%\%?%s$N%3!<%k%P%C%/4X?t(B ************************** */
static void
cb_reference_dialog_cancel (GtkWidget	*widget,
			    GtkWidget	*dialog) {
  gtk_widget_hide (dialog);
  gtk_widget_destroy (dialog);
}

/* $B;2>H%@%$%"%m%0$N@8@.(B **************************************************** */
static GtkWidget*
reference_dialog_new (const gchar	*title,
		      GtkEntry		*entry) {
  GtkWidget	*dialog;

  /* $B%@%$%"%m%0$N@8@.(B */
  dialog = gtk_custom_icon_file_selection_new (title);
  gtk_window_set_icon (GTK_WINDOW (dialog), te_icon);
  
  /* $B%3!<%k%P%C%/4X?t$N@_Dj(B */
  G_SET_PARAMETER (dialog, "entry", entry);
  g_signal_connect (G_OBJECT (GTK_CUSTOM_ICON_FILESEL (dialog)->ok_button),
		    "clicked", G_CALLBACK (cb_reference_dialog_ok),
		    GTK_CUSTOM_ICON_FILESEL (dialog));
  g_signal_connect (G_OBJECT (GTK_CUSTOM_ICON_FILESEL (dialog)->cancel_button),
		    "clicked", G_CALLBACK (cb_reference_dialog_cancel),
		    dialog);
  return dialog;
}

/* $B;2>H%@%$%"%m%0$N@8@.(B **************************************************** */
static void
cb_reference_dialog (GtkWidget	*widget,
		     GtkEntry	*entry) {
  GtkWidget	*dialog;

  dialog = reference_dialog_new (_("Reference"), entry);
  gtk_widget_show_all (dialog);
}

/* ************************************************************************* *
 * $B@_Dj%U%!%$%k$NFI$_9~$_MQ$N4X?t(B ($B%m!<%+%k4X?t(B)
 *
 * parse_option
 * parse_confirm
 * check_list_num
 * parse_list
 * create_shortcut_accl
 * parse_shortcut
 * ************************************************************************* */

/* $B%*%W%7%g%s$N9`L\(B ******************************************************** */
static gboolean
parse_option (TEConfig	*tc,
	      GNode	*node,
	      gint	level) {
  GNode		*work_node;
  XMLNode	*xnode;

  if (node == NULL) return FALSE;
  if (level < 0) return TRUE;
  
  xnode = (XMLNode *) node->data;

  if (xnode->element) {
    if (strcmp (xnode->element, XML_TAG_OPTION_LOAD_ALL) == 0) {
      work_node = node->next;
      xnode = (XMLNode *) work_node->data;
      tc->option_load_all = (gboolean) atoi (xnode->element);
    } else if (strcmp (xnode->element, XML_TAG_OPTION_CLEAN_MEM) == 0) {
#if 0
      work_node = node->next;
      xnode = (XMLNode *) work_node->data;
      tc->option_clean_mem = (gboolean) atoi (xnode->element);
#endif
    } else if (strcmp (xnode->element, XML_TAG_OPTION_NORMALIZE) == 0) {
      work_node = node->next;
      xnode = (XMLNode *) work_node->data;
      tc->option_normalize = (gboolean) atoi (xnode->element);
    }
  }
  if (node->children) {
    parse_option (tc, node->children, level+1);
  } else if (node->next) {
    parse_option (tc, node->next, level);
  } else {
    parse_option (tc, node->parent->next, level-1);
  }
  return TRUE;
}

/* $B3NG'$N9`L\(B ************************************************************** */
static gboolean
parse_confirm (TEConfig	*tc,
	       GNode	*node,
	       gint	level) {
  GNode		*work_node;	
  XMLNode	*xnode;

  if (node == NULL) return FALSE;
  if (level < 0) return TRUE;
  
  xnode = (XMLNode *) node->data;

  if (xnode->element) {
    if (strcmp (xnode->element, XML_TAG_CONFIRM_SAVE) == 0) {
      work_node = node->next;
      xnode = (XMLNode *) work_node->data;
      tc->confirm_save = (gboolean) atoi (xnode->element);
    } else if (strcmp (xnode->element, XML_TAG_CONFIRM_OVERWRITE) == 0) {
      work_node = node->next;
      xnode = (XMLNode *) work_node->data;
      tc->confirm_overwrite = (gboolean) atoi (xnode->element);
    }
  }
  if (node->children) {
    parse_confirm (tc, node->children, level+1);
  } else if (node->next) {
    parse_confirm (tc, node->next, level);
  } else {
    parse_confirm (tc, node->parent->next, level-1);
  }
  return TRUE;
}

/* Item$B$N?t$r?t$($k(B ******************************************************** */
static gint
check_list_num (GNode	*node,
		gint	level) {
  XMLNode	*xnode;
  static gint	nums = 0;
  
  if (node == NULL) return 0;
  if (level == -1) {
    nums = 0;
    level++;
  }
  if (level < 0) return nums;
  
  xnode = (XMLNode *) node->data;

  if (strcmp (xnode->tag->tag, XML_TAG_ITEM) == 0) {
    if (xnode->element) nums++;
  }
  if (node->children) check_list_num (node->children, level+1);

  return nums;
}

/* $B%j%9%H$N2r@O(B ************************************************************ */
static gboolean
parse_list (GList	**list,
	    GNode	*node,
	    gint	level) {
  XMLNode	*xnode;

  if (node == NULL) return FALSE;
  if (level == -1) level++;
  if (level < 0) return TRUE;
  
  xnode = (XMLNode *) node->data;

  if (strcmp (xnode->tag->tag, XML_TAG_ITEM) == 0) {
    *list = g_list_append (*list, g_strdup (xnode->element));
  }
  if (node->children) {
    parse_list (list, node->children, level+1);
  } else if (node->next) {
    parse_list (list, node->next, level);
  }
  return TRUE;
}

/* ************************************************************************* */
static gchar*
create_shortcut_accl (gchar	*string) {
  gchar	*ptr, *mod_ptr, *accl;
  gchar	buf[1024];
  gint	modifier = 0;
  
  ptr = string;
  while (*ptr != '+') {
    mod_ptr = buf;
    if (*ptr == '|') ptr++;
    while (*ptr != '|' && *ptr != '+') {
      *mod_ptr++ = *ptr++;
    }
    *mod_ptr = '\0';
    if (strcmp (buf, "Shift") == 0) {
      modifier |= 1;
    } else if (strcmp (buf, "Control") == 0) {
      modifier |= 4;
    } else if (strcmp (buf, "Alt") == 0) {
      modifier |= 8;
    }
  }
  ptr++;
  mod_ptr = buf;
  while (*ptr != '\0') *mod_ptr++ = *ptr++;
  *mod_ptr = '\0';

  accl = g_new (gchar, strlen (mod_str[modifier]) + strlen (buf) + 1);
  sprintf (accl, "%s%s", mod_str[modifier], buf);

  return accl;
}

/* $B%7%g!<%H%+%C%H$N9`L\(B **************************************************** */
static gboolean
parse_shortcut (TEConfig	*tc,
		GNode		*node,
		gint		level) {
  XMLNode	*xnode;
  
  if (node == NULL) return FALSE;
  if (level == -1) level++;
  if (level < 0) return TRUE;
  
  xnode = (XMLNode *) node->data;

  if (strcmp (xnode->tag->tag, XML_TAG_ITEM) == 0) {
    tc->shortcut = g_list_append (tc->shortcut, g_strdup (xnode->element));
    tc->shortcut_accel
      = g_list_append (tc->shortcut_accel,
		       create_shortcut_accl ((gchar *) xnode->element));
  }
  if (node->children) {
    parse_shortcut (tc, node->children, level+1);
  } else if (node->next) {
    parse_shortcut (tc, node->next, level);
  }
  return TRUE;
}

/* ************************************************************************* *
 * $B@_Dj%U%!%$%k=PNOMQ$N4X?t(B ($B%m!<%+%k4X?t(B)
 *
 * print_tag
 * print_end_tag
 * print_element
 * ************************************************************************* */

/* ************************************************************************* */
static void
print_tag (FILE		*fp,
	   const gchar	*tag,
	   guint	tab_level,
	   guint	ent_level) {
  gchar	*tab[] = {"", "\t", "\t\t", "\t\t\t"};
  gchar	*ent[] = {"", "\n", "\n\n", "\n\n\n"};

  fprintf (fp, "%s<%s>%s", tab[tab_level], tag, ent[ent_level]);
}

/* ************************************************************************* */
static void
print_end_tag (FILE		*fp,
	       const gchar	*tag,
	       guint		tab_level,
	       guint		ent_level) {
  gchar	*tab[] = {"", "\t", "\t\t", "\t\t\t"};
  gchar	*ent[] = {"", "\n", "\n\n", "\n\n\n"};

  fprintf (fp, "%s</%s>%s", tab[tab_level], tag, ent[ent_level]);
}

/* ************************************************************************* */
static void
print_element (FILE		*fp,
	       const gchar	*tag,
	       gpointer		value,
	       guint		value_type,
	       guint		tab_level,
	       guint		ent_level) {
  gchar	*tab[] = {"", "\t", "\t\t", "\t\t\t"};
  gchar	*ent[] = {"", "\n", "\n\n", "\n\n\n"};
  
  switch (value_type) {
  case XML_VALUE_TYPE_INT:
    fprintf (fp, "%s<%s>%d</%s>%s",
	     tab[tab_level], tag, (gint) value, tag, ent[ent_level]);
    break;
  case XML_VALUE_TYPE_STRING:
    fprintf (fp, "%s<%s>%s</%s>%s",
	     tab[tab_level], tag, (gchar *) value, tag, ent[ent_level]);
    break;
  }
}

/* ************************************************************************* *
 * $B%0%m!<%P%k4X?t(B
 *
 * config_check_version
 * config_load_file
 * config_save_file
 * config_free_file
 * ************************************************************************* */

/* $B@_Dj%U%!%$%k$N%P!<%8%g%s%A%'%C%/(B **************************************** */
gboolean
config_check_version (const gchar	*filename) {
  GNode		*root_node, *node;
  XMLNode	*xnode;
  gchar		*version;
  
  /* $B@_Dj%U%!%$%k$NFI$_9~$_(B */
  root_node = xml_parse_file (filename);
  if (!root_node) {
    g_printerr (_("Could not read teoeyes config file.\n"));
    return -1;
  }
  /* TeoEyes$B$N@_Dj%U%!%$%k$+$I$&$+3NG'(B */
  xnode = (XMLNode *) root_node->data;
  if (strcmp (xnode->tag->tag, XML_TAG_TEOEYES_CONFIG) != 0) {
    g_printerr (_("%s is not teoeyes_config_file.\n"), filename);
    g_node_destroy (root_node);
    return -1;
  }
  /* $B%P!<%8%g%s>pJs$NFI$_9~$_(B */
  node  = root_node->children;
  xnode = (XMLNode *) node->data;
  if (strcmp (xnode->tag->tag, XML_TAG_VERSION) != 0) {
    g_printerr (_("Could not read version infomation.\n"));
    g_node_destroy (root_node);
    return -1;
  }
  version = g_strdup (xnode->element);
  g_node_destroy (root_node);
  if (strcmp (version, CONFIG_BASE_VERSION) < 0) {
    g_free (version);
    return 0;
  } else {
    g_free (version);    
    return 1;
  }
}

/* $B@_Dj%U%!%$%k$NFI$_9~$_(B($B%P!<%8%g%s(B3.0.0$B0J9_BP1~(B) ************************* */
TEConfig*
config_load_file (const gchar	*filename) {
  TEConfig	*tc;
  GNode		*root_node, *node;
  XMLNode	*xnode;
  gint		nums;
  gint		count = 0;
  const gint	config_items = 15;
  
  /* $B@_Dj%U%!%$%k$NFI$_9~$_(B */
  root_node = xml_parse_file (filename);

  if (!root_node) {
    g_printerr (_("Could not read teoeyes config file.\n"));
    return NULL;
  }
  /* $B9=B$BN$NNN0h3NJ](B */
  tc = g_new (TEConfig, 1);
  if (!tc) {
    g_printerr (_("Could not allocate enough memory.\n"));
    g_node_destroy (root_node);
    return NULL;
  }
  /* $BJ8;zNsJQ?t$N=i4|2=(B */
  tc->folder_icon	  = NULL;
  tc->thumbnail_dir	  = NULL;
  tc->loader_dirs	  = NULL;	
  tc->assistant_skin	  = NULL;
  tc->assistant_doc_dirs  = NULL;
  tc->plugin_dirs	  = NULL;
  tc->unuse_plugin_list	  = NULL;
  tc->shortcut		  = NULL;
  tc->shortcut_accel	  = NULL;    
  tc->title_image	  = NULL;
  tc->player_skin	  = NULL;
  
  /* TeoEyes$B$N@_Dj%U%!%$%k$+$I$&$+3NG'(B */
  xnode = (XMLNode *) root_node->data;
  if (strcmp (xnode->tag->tag, XML_TAG_TEOEYES_CONFIG) != 0) {
    g_printerr (_("%s is not teoeyes_config_file.\n"), filename);
    g_node_destroy (root_node);
    g_free (tc);
    return NULL;
  }
  /* $B%P!<%8%g%s>pJs$NFI$_9~$_(B */
  node  = root_node->children;
  xnode = (XMLNode *) node->data;
  if (strcmp (xnode->tag->tag, XML_TAG_VERSION) != 0) {
    g_printerr (_("Could not read version infomation.\n"));
    g_node_destroy (root_node);
    g_free (tc);
    return NULL;
  }
  /* $B@_Dj9`L\$NFI$_9~$_(B */
  node  = node->next;
  xnode = (XMLNode *) node->data;

  while (count < config_items && node) {
    if (strcmp (xnode->tag->tag, XML_TAG_OPTION) == 0) {
      /* $B%*%W%7%g%s$N9`L\(B */
      if (!parse_option (tc, node, 0)) break;
      node = node->next;
      if (node) xnode = (XMLNode *) node->data;
      count++;
    } else if (strcmp (xnode->tag->tag, XML_TAG_CONFIRM) == 0) {
      /* $B3NG'$N9`L\(B */
      if (!parse_confirm (tc, node, 0)) break;
      node = node->next;
      if (node) xnode = (XMLNode *) node->data;
      count++;
    } else if (strcmp (xnode->tag->tag, XML_TAG_FOLDER_ICON) == 0) {
      /* $B2hA|%(%/%9%W%m!<%i$N%U%)%k%@%"%$%3%s$N>l=j(B */
      if (xnode->element) tc->folder_icon = g_strdup (xnode->element);
      node = node->next;
      if (node) xnode = (XMLNode *) node->data;
      count++;
    } else if (strcmp (xnode->tag->tag, XML_TAG_THUMBNAIL_DIR) == 0) {
      /* $B%5%`%M%$%k2hA|$NJ]B8>l=j(B */
      tc->thumbnail_dir = g_strdup (xnode->element);
      node = node->next;
      if (node) xnode = (XMLNode *) node->data;
      count++;
    } else if (strcmp (xnode->tag->tag, XML_TAG_THUMBNAIL_WIDTH) == 0) {
      /* $B%5%`%M%$%k2hA|$NI}(B */
      tc->thumbnail_width = atoi (xnode->element);
      node = node->next;      
      if (node) xnode = (XMLNode *) node->data;
      count++;
    } else if (strcmp (xnode->tag->tag, XML_TAG_THUMBNAIL_HEIGHT) == 0) {
      /* $B%5%`%M%$%k2hA|$N9b$5(B */
      tc->thumbnail_height = atoi (xnode->element);
      node = node->next;
      if (node) xnode = (XMLNode *) node->data;
      count++;
    } else if (strcmp (xnode->tag->tag, XML_TAG_LOADER_DIRS) == 0) {
      /* $B2hA|%m!<%@$N>l=j(B */
      if (!parse_list (&tc->loader_dirs, node, -1)) break;
      node = node->next;
      if (node) xnode = (XMLNode *) node->data;
      count++;
    } else if (strcmp (xnode->tag->tag, XML_TAG_ASSISTANT_SKIN) == 0) {
      /* $B%"%7%9%?%s%H$N%9%-%s(B */
      tc->assistant_skin = g_strdup (xnode->element);
      node = node->next;
      if (node) xnode = (XMLNode *) node->data;
      count++;
    } else if (strcmp (xnode->tag->tag, XML_TAG_ASSISTANT_DOC_DIRS) == 0) {
      /* $B%"%7%9%?%s%H$N%I%-%e%a%s%H$N>l=j(B */
      nums = check_list_num (node, -1);
      if (nums != 0) if (!parse_list (&tc->assistant_doc_dirs, node,-1)) break;
      node = node->next;
      if (node) xnode = (XMLNode *) node->data;
      count++;
    } else if (strcmp (xnode->tag->tag, XML_TAG_PLUGIN_DIRS) == 0) {
      /* $B%W%i%0%$%s$N>l=j(B */
      nums = check_list_num (node, -1);
      if (nums != 0) if (!parse_list (&tc->plugin_dirs, node, -1)) break;
      node = node->next;
      if (node) xnode = (XMLNode *) node->data;
      count++;
    } else if (strcmp (xnode->tag->tag, XML_TAG_UNUSE_PLUGIN_LIST) == 0) {
      /* $B;HMQ$7$J$$%W%i%0%$%s%j%9%H(B */
      nums = check_list_num (node, -1);
      if (nums != 0) if (!parse_list (&tc->unuse_plugin_list, node, -1)) break;
      node = node->next;
      if (node) xnode = (XMLNode *) node->data;
      count++;
    } else if (strcmp (xnode->tag->tag, XML_TAG_SHORTCUT) == 0) {
      /* $B%7%g!<%H%+%C%H(B */
      if (!parse_shortcut (tc, node, -1)) break;
      node = node->next;
      if (node) xnode = (XMLNode *) node->data;
      count++;
    } else if (strcmp (xnode->tag->tag, XML_TAG_LOOPE_SCALE) == 0) {
      /* $B%k!<%Z$N3HBgN((B */
      tc->loope_scale = atoi (xnode->element);            
      node = node->next;
      if (node) xnode = (XMLNode *) node->data;
      count++;
    } else if (strcmp (xnode->tag->tag, XML_TAG_TITLE_IMAGE) == 0) {
      /* $B%?%$%H%k2hA|(B */
      if (xnode->element) tc->title_image = g_strdup (xnode->element);
      node = node->next;
      if (node) xnode = (XMLNode *) node->data;
      count++;
    } else if (strcmp (xnode->tag->tag, XML_TAG_PLAYER_SKIN) == 0) {
      /* $B2hA|%W%l!<%d$N%9%-%s(B */
      if (xnode->element) tc->player_skin = g_strdup (xnode->element);      
      node = node->next;
      if (node) xnode = (XMLNode *) node->data;
      count++;
    }
  }
  if (count < config_items) {
    g_printerr (_("Could not read config file completely.\n"));
    g_node_destroy (root_node);
    config_free_file (tc);
    return NULL;
  }
  g_node_destroy (root_node);
  
  return tc;
}

/* $B@_Dj%U%!%$%k$NJ]B8(B ****************************************************** */
gboolean
config_save_file (TEConfig	*tc,
		  const gchar	*filename) {
FILE		*fp;
  GList		*list;

  /* $B%U%!%$%k$N%*!<%W%s(B */
  fp = fopen (filename, "w");
  if (!fp) {
    g_printerr (_("Could not create file %s.\n"), filename);
    return FALSE;
  }
  /* $B%U%!%$%k$X$N=PNO(B */
  fprintf (fp, "<?xml version=\"1.0\" encoding=\"EUC-JP\"?>\n\n");

  print_tag (fp, XML_TAG_TEOEYES_CONFIG, 0, 2);

  print_element (fp, XML_TAG_VERSION, VERSION, XML_VALUE_TYPE_STRING, 1, 2);

  /* $B%*%W%7%g%s$N9`L\(B */
  print_tag (fp, XML_TAG_OPTION, 1, 1);
  {
    print_tag (fp, XML_TAG_ITEM, 2, 1);
    {
      print_element (fp, XML_TAG_NAME, XML_TAG_OPTION_LOAD_ALL,
		     XML_VALUE_TYPE_STRING, 3, 0);
      print_element (fp, XML_TAG_VALUE, (gpointer) tc->option_load_all,
		     XML_VALUE_TYPE_INT, 0, 1);
    }
    print_end_tag (fp, XML_TAG_ITEM, 2, 1);

#if 0
    print_tag (fp, XML_TAG_ITEM, 2, 1);
    {
      print_element (fp, XML_TAG_NAME, XML_TAG_OPTION_CLEAN_MEM,
		     XML_VALUE_TYPE_STRING, 3, 0);
      print_element (fp, XML_TAG_VALUE, (gpointer) tc->option_clean_mem,
		     XML_VALUE_TYPE_INT, 0, 1);
    }
    print_end_tag (fp, XML_TAG_ITEM, 2, 1);
#endif
    print_tag (fp, XML_TAG_ITEM, 2, 1);
    {
      print_element (fp, XML_TAG_NAME, XML_TAG_OPTION_NORMALIZE,
		     XML_VALUE_TYPE_STRING, 3, 0);
      print_element (fp, XML_TAG_VALUE, (gpointer) tc->option_normalize,
		     XML_VALUE_TYPE_INT, 0, 1);
    }
    print_end_tag (fp, XML_TAG_ITEM, 2, 1);
  }
  print_end_tag (fp, XML_TAG_OPTION, 1, 2);

  /* $B3NG'$N9`L\(B */  
  print_tag (fp, XML_TAG_CONFIRM, 1, 1);
  {
    print_tag (fp, XML_TAG_ITEM, 2, 1);
    {
      print_element (fp, XML_TAG_NAME, XML_TAG_CONFIRM_SAVE,
		     XML_VALUE_TYPE_STRING, 3, 0);
      print_element (fp, XML_TAG_VALUE, (gpointer) tc->confirm_save,
		     XML_VALUE_TYPE_INT, 0, 1);
    }
    print_end_tag (fp, XML_TAG_ITEM, 2, 1);

    print_tag (fp, XML_TAG_ITEM, 2, 1);
    {
      print_element (fp, XML_TAG_NAME, XML_TAG_CONFIRM_OVERWRITE,
		     XML_VALUE_TYPE_STRING, 3, 0);
      print_element (fp, XML_TAG_VALUE, (gpointer) tc->confirm_overwrite,
		     XML_VALUE_TYPE_INT, 0, 1);
    }
    print_end_tag (fp, XML_TAG_ITEM, 2, 1);
  }
  print_end_tag (fp, XML_TAG_CONFIRM, 1, 2);

  /* $B%5%`%M%$%k$N9`L\(B */
  print_element (fp, XML_TAG_FOLDER_ICON,
		 (tc->folder_icon) ? tc->folder_icon : "",
		 XML_VALUE_TYPE_STRING, 1, 2);
  print_element (fp, XML_TAG_THUMBNAIL_DIR, (gpointer) tc->thumbnail_dir,
		 XML_VALUE_TYPE_STRING, 1, 2);
  print_element (fp, XML_TAG_THUMBNAIL_WIDTH, (gpointer) tc->thumbnail_width,
		 XML_VALUE_TYPE_INT, 1, 2);
  print_element (fp, XML_TAG_THUMBNAIL_HEIGHT, (gpointer) tc->thumbnail_height,
		 XML_VALUE_TYPE_INT, 1, 2);

  /* $B2hA|%m!<%@$N9`L\(B */
  print_tag (fp, XML_TAG_LOADER_DIRS, 1, 1);
  {
    for (list = tc->loader_dirs; list; list = g_list_next (list)) {
      print_element (fp, XML_TAG_ITEM, list->data,XML_VALUE_TYPE_STRING, 2, 1);
    }
  }
  print_end_tag (fp, XML_TAG_LOADER_DIRS, 1, 2);
  
  /* $B%"%7%9%?%s%H$N9`L\(B */
  print_element (fp, XML_TAG_ASSISTANT_SKIN, (gpointer) tc->assistant_skin,
		 XML_VALUE_TYPE_STRING, 1, 2);
  print_tag (fp, XML_TAG_ASSISTANT_DOC_DIRS, 1, 1);
  {
    for (list = tc->assistant_doc_dirs; list; list = g_list_next (list)) {
      print_element (fp, XML_TAG_ITEM, list->data,XML_VALUE_TYPE_STRING, 2, 1);
    }
  }
  print_end_tag (fp, XML_TAG_ASSISTANT_DOC_DIRS, 1, 2);

  /* $B%W%i%0%$%s$N9`L\(B */
  print_tag (fp, XML_TAG_PLUGIN_DIRS, 1, 1);
  {
    for (list = tc->plugin_dirs; list; list = g_list_next (list)) {
      print_element (fp, XML_TAG_ITEM, list->data,XML_VALUE_TYPE_STRING, 2, 1);
    }
  }
  print_end_tag (fp, XML_TAG_PLUGIN_DIRS, 1, 2);

  print_tag (fp, XML_TAG_UNUSE_PLUGIN_LIST, 1, 1);
  {
    for (list = tc->unuse_plugin_list; list; list = g_list_next (list)) {
      print_element (fp, XML_TAG_ITEM, list->data,XML_VALUE_TYPE_STRING, 2, 1);
    }
  }
  print_end_tag (fp, XML_TAG_UNUSE_PLUGIN_LIST, 1, 2);

  /* $B%7%g!<%H%+%C%H(B */
  print_tag (fp, XML_TAG_SHORTCUT, 1, 1);  
  {
    for (list = tc->shortcut; list; list = g_list_next (list)) {
      print_element (fp, XML_TAG_ITEM, list->data,
		     XML_VALUE_TYPE_STRING, 2, 1);
    }
  }
  print_end_tag (fp, XML_TAG_SHORTCUT, 1, 2);

  /* $B%k!<%Z$N3HBgN((B */
  print_element (fp, XML_TAG_LOOPE_SCALE, (gpointer) tc->loope_scale,
		 XML_VALUE_TYPE_INT, 1, 2);

  /* $B%?%$%H%k2hA|(B */
  print_element (fp, XML_TAG_TITLE_IMAGE, (gpointer) tc->title_image,
		 XML_VALUE_TYPE_STRING, 1, 2);

  /* $B2hA|%W%l!<%d$N%9%-%s(B */
  print_element (fp, XML_TAG_PLAYER_SKIN,
		 (tc->player_skin) ? tc->player_skin : "",
		 XML_VALUE_TYPE_STRING, 1, 2);

  print_end_tag (fp, XML_TAG_TEOEYES_CONFIG, 0, 1);

  fclose (fp);

  return TRUE;
}

/* $B@_Dj9`L\9=B$BN$NNN0h2rJ|(B ************************************************ */
void
config_free_file (TEConfig	*tc) {
  if (tc) {
    g_free (tc->folder_icon);
    g_free (tc->thumbnail_dir);
    g_free (tc->assistant_skin);
    g_free (tc->title_image);
    g_free (tc->player_skin);
    g_list_foreach (tc->loader_dirs,	    (GFunc) g_free, NULL);
    g_list_foreach (tc->assistant_doc_dirs, (GFunc) g_free, NULL);
    g_list_foreach (tc->plugin_dirs, 	    (GFunc) g_free, NULL);
    g_list_foreach (tc->unuse_plugin_list,  (GFunc) g_free, NULL);
    g_list_foreach (tc->shortcut,	    (GFunc) g_free, NULL);
    g_list_foreach (tc->shortcut_accel,	    (GFunc) g_free, NULL);
    g_free (tc);
  }
}

/* ************************************************************************* *
 * $B@_Dj%@%$%"%m%0$N%3!<%k%P%C%/4X?t(B ($B%m!<%+%k4X?t(B) callback
 *
 * cb_dialog_ok
 * cb_dialog_apply
 * cb_dialog_close_real
 * cb_dialog_close 
 * cb_change_box
 * ************************************************************************* */

/* $B@_Dj%@%$%"%m%0(B $B!V#O#K!W%\%?%s(B ******************************************* */
static void
cb_dialog_ok (GtkWidget	*widget,
	      GtkWidget	*dialog) {
  cb_dialog_apply    (widget, dialog);
  gtk_widget_hide    (dialog);
  gtk_widget_destroy (dialog);
  config_dialog = NULL;
}

/* $B@_Dj%@%$%"%m%0(B $BE,MQ(B ***************************************************** */
static void
cb_dialog_apply (GtkWidget	*widget,
		 GtkWidget	*dialog) {
  GtkWidget	*child;
  gchar		*dir;
  
  /* $B%*%W%7%g%s$N@_Dj$N99?7(B */
  child = G_GET_WIDGET (dialog, "option1");
  tc->option_load_all =
    gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (child));
#if 0
  child = gtk_object_get_data (GTK_OBJECT (dialog), "option2");
  tc->option_clean_mem =
    gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (child));
#endif
  child = G_GET_WIDGET (dialog, "option3");
  tc->option_normalize =
    gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (child));

  /* $B3NG'@_Dj$N99?7(B */
  child = G_GET_WIDGET (dialog, "confirm1");
  tc->confirm_save =
    gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (child));

  child = G_GET_WIDGET (dialog, "confirm2");
  tc->confirm_overwrite =
    gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (child));

  /* $B%V%i%&%6@_Dj$N99?7(B */
  child = G_GET_WIDGET (dialog, "folder_icon");
  g_free (tc->folder_icon);
  tc->folder_icon = NULL;  
  if (gtk_entry_get_text (GTK_ENTRY (child)) &&
      strcmp (gtk_entry_get_text (GTK_ENTRY (child)), "") != 0) {
    tc->folder_icon = g_strdup (gtk_entry_get_text (GTK_ENTRY (child)));
  }
  child = G_GET_WIDGET (dialog, "thumbnail_dir");
  g_free (tc->thumbnail_dir);
  tc->thumbnail_dir = NULL;
  if (gtk_entry_get_text (GTK_ENTRY (child)) &&
      strcmp (gtk_entry_get_text (GTK_ENTRY (child)), "") != 0) {
    tc->thumbnail_dir = g_strdup (gtk_entry_get_text (GTK_ENTRY (child)));
  } else {
    tc->thumbnail_dir = g_strconcat (g_get_home_dir (), G_DIR_SEPARATOR_S,
				     CONFIG_FILE_DIR,    G_DIR_SEPARATOR_S,
				     "icons", NULL);
  }
  child = G_GET_WIDGET (dialog, "thumbnail_width");  
  tc->thumbnail_width
    = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (child));
  child = gtk_object_get_data (GTK_OBJECT (dialog), "thumbnail_height");  
  tc->thumbnail_height
    = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (child));

  /* $B2hA|%m!<%@$N@_Dj(B */
  child = G_GET_WIDGET (dialog, "loader_dirs_list");
  g_list_free (tc->loader_dirs);
  tc->loader_dirs = get_list (GTK_TREE_VIEW (child));

  /* $B%"%7%9%?%s%H@_Dj$N99?7(B */
  child = G_GET_WIDGET (dialog, "assistant_skin");
  g_free (tc->assistant_skin);
  tc->assistant_skin = NULL;
  if (gtk_entry_get_text (GTK_ENTRY (child)) &&
      strcmp (gtk_entry_get_text (GTK_ENTRY (child)), "") != 0) {
    tc->assistant_skin = g_strdup (gtk_entry_get_text (GTK_ENTRY (child)));
  }
  child = G_GET_WIDGET (dialog, "assistant_doc_dirs_list");
  g_list_free (tc->assistant_doc_dirs);
  tc->assistant_doc_dirs = get_list (GTK_TREE_VIEW (child));
  
  /* $B%W%i%0%$%s$N>l=j(B */
  child = G_GET_WIDGET (dialog, "plugin_dirs_list");
  g_list_free (tc->plugin_dirs);
  tc->plugin_dirs = get_list (GTK_TREE_VIEW (child));

  /* $B;HMQ$7$J$$%W%i%0%$%s$N%j%9%H(B */
  child = G_GET_WIDGET (dialog, "unuse_plugin_list");
  {
    GtkIconItemFactoryEntry	*p;
    GtkTreeView			*treeview;
    GtkTreeModel		*model;
    GtkTreeIter			iter;
    GList			*list;
    gboolean			success;
    gboolean			active;

    g_list_foreach (tc->unuse_plugin_list, (GFunc) g_free, NULL); 
    tc->unuse_plugin_list = NULL;
    
    treeview = GTK_TREE_VIEW (G_GET_WIDGET (config_dialog, "plugin_treeview"));
    model    = gtk_tree_view_get_model (treeview);
    success  = gtk_tree_model_get_iter_first (model, &iter);
    list     = g_list_first (plugin_list);

    while (success) {
      gtk_tree_model_get (model, &iter, 0, &active, -1);
      ((PluginInfo *) list->data)->active = active;
      p = (GtkIconItemFactoryEntry *) list->data;
      if (!active) {
	tc->unuse_plugin_list
	  = g_list_append (tc->unuse_plugin_list, g_strdup (p->path));
      }
      list = g_list_next (list);
      success = gtk_tree_model_iter_next (model, &iter);
    }
  }
  /* $B%7%g!<%H%+%C%H(B */
  child = G_GET_WIDGET (dialog, "shortcut");
  g_list_free (tc->shortcut);
  tc->shortcut = get_shortcut_list (GTK_TREE_VIEW (child));
  
  /* $B%k!<%Z$N3HBgN((B */
  tc->loope_scale = scale;

  /* $B%?%$%H%k2hA|(B */
  child = G_GET_WIDGET (dialog, "title_image");
  g_free (tc->title_image);
  tc->title_image = NULL;
  if (gtk_entry_get_text (GTK_ENTRY (child)) ||
      strcmp (gtk_entry_get_text (GTK_ENTRY (child)), "") != 0) {
    tc->title_image = g_strdup (gtk_entry_get_text (GTK_ENTRY (child)));
  }
  /* $B2hA|%W%l!<%d$N%9%-%s(B */
  child = G_GET_WIDGET (dialog, "player_skin");
  g_free (tc->player_skin);
  tc->player_skin = NULL;
  if (gtk_entry_get_text (GTK_ENTRY (child)) &&
      strcmp (gtk_entry_get_text (GTK_ENTRY (child)), "") != 0) {
    tc->player_skin = g_strdup (gtk_entry_get_text (GTK_ENTRY (child)));
  }
  /* $B@_Dj%U%!%$%k$N99?7(B */
  dir = g_strconcat (g_get_home_dir (), 
		     G_DIR_SEPARATOR_S, CONFIG_FILE_DIR,   G_DIR_SEPARATOR_S,
		     CONFIG_FILE, NULL);
  config_save_file (tc, dir);

  gtk_widget_grab_focus (gtk_object_get_data (GTK_OBJECT (dialog),
					      "button_close"));
  g_free (dir);
}

/* $B@_Dj%@%$%"%m%0(B $B!VJD$8$k!W%\%?%s(B ***************************************** */
static void
cb_dialog_close_real (GtkWidget	*dialog,
		      gpointer	data) {
  gtk_widget_hide    (dialog);
  gtk_widget_destroy (dialog);
  config_dialog = NULL;
}

/* $B@_Dj%@%$%"%m%0(B $B!VJD$8$k!W%\%?%s(B ***************************************** */
static void
cb_dialog_close (GtkWidget	*widget,
		 GtkWidget	*dialog) {
  cb_dialog_close_real (dialog, NULL);
}

/* $B@_Dj%\%C%/%9$NI=<(@ZBX$((B ************************************************ */
static void
cb_change_box (GtkWidget	*widget,
	       gpointer		data) {
  GtkWidget	*box;
  static gchar	*box_str[] = {"box0", "box1", "box2", "box3", "box4", "box5"};
  int		n;
  
  for (n = 0; n < 6; n++) {
    box = G_GET_WIDGET (config_dialog, box_str[n]);
    gtk_widget_hide (box);
  }
  box = G_GET_WIDGET (config_dialog, box_str[(gint) data]);
  gtk_widget_show_all (box);
}

/* ************************************************************************* *
 * $B@_Dj%@%$%"%m%0$K4X$9$k4X?t(B ($B%m!<%+%k4X?t(B)
 *
 * get_list
 * get_shortcut_list
 * ************************************************************************* */

/* $B%j%9%H%\%C%/%9$+$i%j%9%H$r3MF@(B ****************************************** */
static GList*
get_list (GtkTreeView	*treeview) {
  GList		*list = NULL;
  GtkTreeModel	*model;
  GtkTreeIter	iter;
  gchar		*list_data;
  gboolean	success;
  
  model   = gtk_tree_view_get_model (treeview);
  success = gtk_tree_model_get_iter_first (model, &iter);
  while (success) {
    gtk_tree_model_get (model, &iter, 0, &list_data, -1);
    list    = g_list_append (list, list_data);
    success = gtk_tree_model_iter_next (model, &iter);
  }
  return list;
}

/* $B@_Dj%U%!%$%k=PNOMQ$N%7%g!<%H%+%C%HJ8;zNs$r@8@.(B ************************** */
static gchar*
create_shortcut_string (gchar	*accel) {
  gchar	*ptr, *last_ptr, *data_ptr;
  gchar	string[1024];

  ptr      = accel;
  last_ptr = strrchr (accel, '>');
  data_ptr = string;

  if (last_ptr) {
    last_ptr++;
    while (strcmp (ptr, last_ptr) != 0) {
      *ptr++;
      while (*ptr != '>') *data_ptr++ = *ptr++;
      *ptr++;
      if (*ptr == '<') *data_ptr++ = '|';
    }
  }
  *data_ptr++ = '+';
  while (*ptr != '\0') *data_ptr++ = *ptr++;
  *data_ptr = *ptr;

  return g_strdup (string);
}

/* $B%7%g!<%H%+%C%H$N%j%9%H%\%C%/%9$+$i%j%9%H$r3MF@(B ************************** */
static GList*
get_shortcut_list (GtkTreeView	*treeview) {
  GList		*list = NULL;
  GtkTreeModel	*model;
  GtkTreeIter	iter;
  gchar		*list_data;
  gboolean	success;
  
  model   = gtk_tree_view_get_model (treeview);
  success = gtk_tree_model_get_iter_first (model, &iter);
  while (success) {
    gtk_tree_model_get (model, &iter, 1, &list_data, -1);
    list    = g_list_append (list, create_shortcut_string (list_data));
    success = gtk_tree_model_iter_next (model, &iter);
    g_free (list_data);
  }
  return list;
}

/* $B%7%g!<%H%+%C%H$N%-!<@_Dj(B ************************************************ */
static void
modify_shortcut_key (GtkTreeView	*treeview,
		     const gchar	*key,
		     const guint	keycode,
		     const guint	state) {
  GtkTreeSelection	*selection;
  GtkTreeModel		*model;
  GtkTreeIter		iter;
  gchar			*old_text, *text, *accelerator;
  gboolean		success;

  selection = gtk_tree_view_get_selection (treeview);
  if (!selection) return;
  success = gtk_tree_selection_get_selected (selection, &model, &iter);

  gtk_tree_model_get (model, &iter, 1, &old_text, -1);
  accelerator = g_strconcat (mod_str[state], key, NULL);
  if (accelerator) {
    gtk_list_store_set (GTK_LIST_STORE (model), &iter, 1, accelerator, -1);
  }
  g_free (accelerator);

  success = gtk_tree_model_get_iter_first (model, &iter);
  while (success) {
    gtk_tree_model_get (model, &iter, 1, &text, -1);
    accelerator = create_shortcut_string (text);
    success     = gtk_tree_model_iter_next (model, &iter);
    g_free (text);
    g_free (accelerator);
  }
}

/* $B%7%g!<%H%+%C%H$N%-!<@_Dj(B ************************************************ */
static gchar*
grab_shortcut_key (guint	*keycode,
		   guint	*state) {
  GtkWidget	*window;
  GtkWidget	*label;
  GtkWidget	*frame;
  GtkWidget	*align;
  gchar		*key;
  XEvent	ev;
  
  /* $BF~NO%&%#%s%I%&$N@8@.(B */
  window = gtk_window_new (GTK_WINDOW_POPUP);
  gtk_window_set_policy (GTK_WINDOW (window), 0, 0, 1);
  gtk_window_set_position (GTK_WINDOW (window), GTK_WIN_POS_MOUSE);
  {
    /* $B%U%l!<%`$N@8@.(B */
    frame = gtk_frame_new (NULL);
    gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_OUT);
    align = gtk_alignment_new (0.0, 0.0, 0.0, 0.0);
    gtk_container_set_border_width (GTK_CONTAINER (align), 32);
    G_SET_WIDGET (window, "frame1", frame);
    {
      /* $B%i%Y%k$N@8@.(B */
      label = gtk_label_new (_("Please press the key on the keyboard\n"
			       "you wish to modify this keyboard-shortcut\n"
			       "to use from now on."));
      gtk_container_add (GTK_CONTAINER (window), frame);
      gtk_container_add (GTK_CONTAINER (frame),  align);
      gtk_container_add (GTK_CONTAINER (align),  label);
      G_SET_WIDGET (window, "label1", label);
    }
  }
  gtk_widget_show_all (window);

  /* $B%-!<F~NO(B */
  while (gtk_events_pending ()) gtk_main_iteration ();
  gdk_flush ();
  while (gtk_events_pending ()) gtk_main_iteration ();
  gdk_window_set_events (window->window, GDK_KEY_PRESS_MASK);
  *keycode = 0;
  
  while (*keycode == 0) {
    XSetInputFocus (GDK_DISPLAY (), GDK_WINDOW_XWINDOW (window->window),
		    RevertToPointerRoot, CurrentTime);
    gdk_keyboard_grab (window->window, TRUE, GDK_CURRENT_TIME);
    XWindowEvent (GDK_DISPLAY (), GDK_WINDOW_XWINDOW (window->window),
                  KeyReleaseMask, &ev);
    gdk_keyboard_ungrab (GDK_CURRENT_TIME);
    key = XKeysymToString (XKeycodeToKeysym(GDK_DISPLAY (), 
					    ev.xkey.keycode, 0));
    *keycode = XKeycodeToKeysym (GDK_DISPLAY (), ev.xkey.keycode, 1);
    if (*keycode >= 65 && *keycode <= 90) {
      key = XKeysymToString (XKeycodeToKeysym(GDK_DISPLAY (), 
					      ev.xkey.keycode, 1));
    }
  }
  /* $B8e;OKv(B */
  gtk_widget_destroy (window);
  *state = ev.xkey.state;

  return g_strdup (key);
}

/* ************************************************************************* */
static gboolean
shortcut_list_button_press (GtkWidget		*widget,
			    GdkEventButton	*event,
			    gpointer		data) {
  gchar	*key;
  guint	keycode, state;

  if (event->type == GDK_2BUTTON_PRESS) {
    key = grab_shortcut_key (&keycode, &state);	
    modify_shortcut_key (GTK_TREE_VIEW (widget), key, keycode, state);
  }
  return FALSE;
}

/* ************************************************************************* *
 * $B@_Dj%@%$%"%m%0$N@8@.$K4X$9$k4X?t(B ($B%m!<%+%k4X?t(B)
 *
 * check_button_set
 * entry_box_new
 * list_box_new
 * loope_menu_new
 * config_box_option
 * config_box_explorer
 * config_box_assistant
 * config_box_plugin
 * config_box_shortut
 * config_box_misc
 * ************************************************************************* */

/* $B%A%'%C%/%\%?%s$N@8@.(B **************************************************** */
static void
check_button_set (GtkWidget	*parent,
		  GtkWidget	*box,
		  const gchar	*label,
		  const gchar	*name,
		  gboolean	flag) {
  GtkWidget	*button;
  
  button = gtk_check_button_new_with_label (label);
  gtk_box_pack_start (GTK_BOX (box), button, FALSE, FALSE, 0);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button), flag);
  G_SET_WIDGET (parent, name, button);
}

/* $B;2>H%\%?%sIU$-$N%(%s%H%j%\%C%/%9$N@8@.(B ********************************** */
static GtkWidget*
entry_box_new (GtkWidget	*parent,
	       const gchar	*title,
	       const gchar	*name,
	       const gchar	*value) {
  GtkWidget	*frame;
  GtkWidget	*hbox;  
  GtkWidget	*entry;
  GtkWidget	*widget;

  /* $B%U%l!<%`$N@8@.(B */
  frame = gtk_frame_new (title);
  {
    hbox = gtk_hbox_new (FALSE, 0);
    gtk_container_border_width (GTK_CONTAINER (hbox), 8);      
    gtk_container_add (GTK_CONTAINER (frame), hbox);
    G_SET_WIDGET (frame, "hbox", hbox);
    {
      /* $B%(%s%H%j%\%C%/%9$N@8@.(B */
      entry = gtk_entry_new ();
      if (value) gtk_entry_set_text (GTK_ENTRY (entry), value);
      gtk_box_pack_start (GTK_BOX (hbox), entry, TRUE, TRUE, 3);
      G_SET_WIDGET (parent,  name,   entry);

      /* $B;2>H%\%?%s$N@8@.(B */
      widget = gtk_button_new_with_mnemonic (_("_Browse..."));
      g_signal_connect (GTK_OBJECT (widget), "clicked",
			G_CALLBACK (cb_reference_dialog), GTK_ENTRY (entry));
      gtk_box_pack_start (GTK_BOX (hbox), widget, FALSE, FALSE, 3);
      G_SET_WIDGET (frame, "button", widget);
    }
  }
  return frame;
}

/* $B%j%9%H%\%C%/%9$N@8@.(B **************************************************** */
static GtkWidget*
list_box_new (GtkWidget		*parent,
	      const gchar	*title,
	      const gchar	*list_title,
	      const gchar	*list_name,
	      const gchar	*entry_name,
	      GList		*value_list,	      
	      gboolean		header_visible) {
  GtkWidget		*hbox;
  GtkWidget		*vbox;  
  GtkWidget		*frame;
  GtkWidget		*entry;
  GtkWidget		*scrolled_window;
  GtkWidget		*widget;
  GtkWidget		*treeview;
  GtkListStore		*store;
  GtkTreeModel		*model;
  GtkCellRenderer	*renderer;
  GtkTreeViewColumn	*column;
  GtkTreeIter		iter;
  GList			*list;
  
  /* $B%"%7%9%?%s%H%G!<%?$NCV$->l=j(B */
  frame = gtk_frame_new (title);
  {
    vbox = gtk_vbox_new (FALSE, 3);
    gtk_container_border_width (GTK_CONTAINER (vbox), 8);
    gtk_container_add (GTK_CONTAINER (frame), vbox);
    G_SET_WIDGET (frame, "vbox1", vbox);
      
    scrolled_window = gtk_scrolled_window_new (NULL, NULL);
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window),
				    GTK_POLICY_AUTOMATIC,GTK_POLICY_AUTOMATIC);
    gtk_scrolled_window_set_shadow_type
      (GTK_SCROLLED_WINDOW (scrolled_window), GTK_SHADOW_ETCHED_OUT);
    gtk_box_pack_start (GTK_BOX (vbox), scrolled_window, TRUE, TRUE, 0);
    G_SET_WIDGET (frame, "scrolled_window", scrolled_window);
    {
      store    = gtk_list_store_new (1, G_TYPE_STRING);
      model    = GTK_TREE_MODEL (store);
      treeview = gtk_tree_view_new_with_model (model);
      G_SET_WIDGET (frame, "treeview", treeview);

      gtk_object_set_data (GTK_OBJECT (parent), list_name, treeview);
      gtk_container_add (GTK_CONTAINER (scrolled_window), treeview);
      
      renderer = gtk_cell_renderer_text_new ();
      column   = gtk_tree_view_column_new_with_attributes
	(list_title, renderer, "text", 0, NULL);
#if 0
      gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (treeview), TRUE);
#endif
      gtk_tree_view_set_headers_visible (GTK_TREE_VIEW (treeview),
					 header_visible);
      gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);
	
      for (list = value_list; list; list = g_list_next (list)) {
	gtk_list_store_append (store, &iter);
	gtk_list_store_set (store, &iter, 0, g_strdup (list->data), -1);
      }
    }
    if (entry_name) {
      /* $B%(%s%H%j%\%C%/%9$N@8@.(B */
      entry = gtk_entry_new ();
      gtk_object_set_data (GTK_OBJECT (parent), entry_name, entry);
      gtk_box_pack_start (GTK_BOX (vbox), entry, FALSE, FALSE, 3);
      G_SET_WIDGET (frame, "entry1", entry);
      
      hbox = gtk_hbox_new (FALSE, 0);
      gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
      G_SET_WIDGET (frame, "hbox1", hbox);
      {
	/* $B;2>H%\%?%s$N@8@.(B */
	widget = gtk_button_new_with_mnemonic (_("_Browse..."));
	g_signal_connect (GTK_OBJECT (widget), "clicked",
			  G_CALLBACK (cb_reference_dialog), GTK_ENTRY (entry));
	gtk_widget_set_size_request (widget, 80, -1);	
	gtk_box_pack_end (GTK_BOX (hbox), widget, FALSE, FALSE, 0);
	G_SET_WIDGET (frame, "button1", widget);

	/* $B:o=|%\%?%s$N@8@.(B */
	widget = gtk_button_new_with_mnemonic (_("_Delete"));
	g_signal_connect (GTK_OBJECT (widget), "clicked",
			  G_CALLBACK (cb_delete_list), treeview);
	gtk_widget_set_size_request (widget, 80, -1);	
	gtk_box_pack_end (GTK_BOX (hbox), widget, FALSE, FALSE, 0);
	G_SET_WIDGET (frame, "button2", widget);
	
	/* $BDI2C%\%?%s$N@8@.(B */
	widget = gtk_button_new_with_mnemonic (_("_Add"));
	g_signal_connect (GTK_OBJECT (widget), "clicked",
			  G_CALLBACK (cb_add_list), treeview);
	gtk_widget_set_size_request (widget, 80, -1);
	gtk_box_pack_end (GTK_BOX (hbox), widget, FALSE, FALSE, 0);
	G_SET_WIDGET (widget, "entry",   entry);
	G_SET_WIDGET (frame,  "button3", widget);
      }
    }
  }
  return frame;
}

/* $B%k!<%Z$N3HBgN(%a%K%e!<$N@8@.(B ******************************************** */
static GtkWidget*
loope_menu_new (void) {
  GtkItemFactory	*item_factory;
  gint			items;

  items = sizeof (loope_menu_items) / sizeof (loope_menu_items[0]);
  item_factory = gtk_item_factory_new (GTK_TYPE_MENU, "<sub>", NULL);
  gtk_item_factory_set_translate_func 
    (item_factory, (GtkTranslateFunc) gettext, NULL, NULL);
  gtk_item_factory_create_items (item_factory, items, loope_menu_items, NULL);
  return gtk_item_factory_get_widget (item_factory, "<sub>");
}

/* $B%*%W%7%g%s%\%C%/%9$N@_Dj(B ************************************************ */
static GtkWidget*
config_box_option (GtkWidget	*parent) {
  GtkWidget	*box;
  GtkWidget	*button_box;
  GtkWidget	*frame;

  /* BOX$B$N@8@.(B */
  box = gtk_vbox_new (FALSE, 16);
  gtk_container_border_width (GTK_CONTAINER (box), 5); 

  /* $B%G%U%)%k%H%*%W%7%g%s$N@_Dj(B */      
  frame = gtk_frame_new (_("Option"));
  gtk_box_pack_start (GTK_BOX (box), frame, FALSE, FALSE, 0);
  G_SET_WIDGET (box, "frame1", frame);

  /* $B%\%?%s%\%C%/%9$N@8@.(B */
  button_box = gtk_vbox_new (FALSE, 0);
  gtk_container_border_width (GTK_CONTAINER (button_box), 8);
  gtk_container_add (GTK_CONTAINER (frame), button_box);
  G_SET_WIDGET (box, "button_box1", button_box);
  {
    /* -l $B%*%W%7%g%s(B */
    check_button_set (parent, button_box,
		      _("Store all images in memory when start teoeyes."),
		      "option1", tc->option_load_all);
    /* -c $B%*%W%7%g%s(B */
#if 0
    check_button_set (parent, button_box,
		      _("Clean the old image memory when load a new image."),
		      "option2", tc->option_clean_mem);
#endif
    /* -N $B%*%W%7%g%s(B */
    check_button_set (parent, button_box,
		      _("Normalize the image in the optimal pixel range."),
		      "option3", tc->option_normalize);
#if 0
    /* -q $B%*%W%7%g%s(B */
    check_button_set (parent, button_box, _("Quick draw mode."),
		      "option4", tc->option_quick_draw);    
    /* -d $B%*%W%7%g%s(B */
    check_button_set (parent, button_box, _("Run teoeyes in display mode."),
		      "option5", tc->option_display_mode);    
#endif
  }
  /* $B%G%U%)%k%H%*%W%7%g%s$N@_Dj(B */      
  frame = gtk_frame_new (_("Confirm"));
  gtk_box_pack_start (GTK_BOX (box), frame, FALSE, FALSE, 0);
  G_SET_WIDGET (box, "frame2", frame);

  /* $B%\%?%s%\%C%/%9$N@8@.(B */
  button_box = gtk_vbox_new (FALSE, 0);
  gtk_container_border_width (GTK_CONTAINER (button_box), 8);  
  gtk_container_add (GTK_CONTAINER (frame), button_box);
  G_SET_WIDGET (box, "button_box2", button_box);
  {
    /* $B2hA|$rJ]B8$9$k$H$-$N3NG'(B */
    check_button_set (parent, button_box,
		      _("Confirm Are you sure you want to save."),
		      "confirm1", tc->confirm_save);
    /* $B2hA|$r>e=q$-$9$k$H$-$N3NG'(B */
    check_button_set (parent, button_box,
		      _("Confirm Are you sure you want to overwrite."),
		      "confirm2", tc->confirm_overwrite);
  }
  return box;
}

/* $B%V%i%&%6%\%C%/%9$N@_Dj(B ************************************************** */
static GtkWidget*
config_box_explorer (GtkWidget	*parent) {
  GtkWidget	*box;
  GtkWidget	*hbox;  
  GtkWidget	*frame;
  GtkWidget	*widget;
  GtkObject	*adjustment;

  /* BOX$B$N@8@.(B */
  box = gtk_vbox_new (FALSE, 16);
  gtk_container_border_width (GTK_CONTAINER (box), 5);
  {
    /* $B%U%)%k%@2hA|$N%G!<%?CV>l(B */
    frame = entry_box_new (parent, _("Folder Icon Theme"), "folder_icon",
			   tc->folder_icon);
    gtk_box_pack_start (GTK_BOX (box), frame, FALSE, FALSE, 3);
    G_SET_WIDGET (box, "frame1", frame);

    /* $B%5%`%M%$%k2hA|$N>l=j(B */
    frame = entry_box_new (parent, _("Thumbnail Folder"), "thumbnail_dir",
			   tc->thumbnail_dir);
    gtk_box_pack_start (GTK_BOX (box), frame, FALSE, FALSE, 3);
    G_SET_WIDGET (box, "frame2", frame);
    
    /* $B%5%`%M%$%k2hA|$NBg$-$5(B */    
    frame = gtk_frame_new (_("Thumbnail Size"));
    gtk_box_pack_start (GTK_BOX (box), frame, FALSE, FALSE, 0);
    G_SET_WIDGET (box, "frame3", frame);
    {
      hbox = gtk_hbox_new (FALSE, 0);
      gtk_container_border_width (GTK_CONTAINER (hbox), 8);      
      gtk_container_add (GTK_CONTAINER (frame), hbox);
      G_SET_WIDGET (box, "hbox1", hbox);

      /* $B%5%`%M%$%k2hA|$NI}(B */
      widget = gtk_label_new (_("Width :"));
      gtk_box_pack_start (GTK_BOX (hbox), widget, FALSE, FALSE, 3);
      G_SET_WIDGET (box, "label1", widget);

      adjustment = gtk_adjustment_new (16, 16, 320, 1, 10, 10);
      widget = gtk_spin_button_new (GTK_ADJUSTMENT (adjustment), 1, 0);
      gtk_spin_button_set_value (GTK_SPIN_BUTTON (widget),
				 (gfloat) tc->thumbnail_width);
      gtk_box_pack_start (GTK_BOX (hbox), widget, TRUE, TRUE, 3);
      G_SET_OBJECT (box, "adjustment1", adjustment);
      G_SET_WIDGET (parent, "thumbnail_width", widget);

      /* $B%5%`%M%$%k2hA|$N9b$5(B */      
      widget = gtk_label_new (_("Height :"));
      gtk_box_pack_start (GTK_BOX (hbox), widget, FALSE, FALSE, 3);
      G_SET_WIDGET (box, "label2", widget);

      adjustment = gtk_adjustment_new (16, 16, 320, 1, 10, 10);
      widget = gtk_spin_button_new (GTK_ADJUSTMENT (adjustment), 1, 0);
      gtk_spin_button_set_value (GTK_SPIN_BUTTON (widget),
				 (gfloat) tc->thumbnail_height);
      gtk_box_pack_start (GTK_BOX (hbox), widget, TRUE, TRUE, 3);
      G_SET_OBJECT (box, "adjustment2", adjustment);
      G_SET_WIDGET (parent, "thumbnail_height", widget);
    }
  }
  return box;
}

/* $B%"%7%9%?%s%H%\%C%/%9$N@_Dj(B ********************************************** */
static GtkWidget*
config_box_assistant (GtkWidget	*parent) {
  GtkWidget		*box;
  GtkWidget		*frame;
  
  /* BOX$B$N@8@.(B */
  box = gtk_vbox_new (FALSE, 16);
  gtk_container_border_width (GTK_CONTAINER (box), 5);
  {
    /* $B%"%7%9%?%s%H%9%-%s%G!<%?(B */
    frame = entry_box_new (parent, _("Assistant Skin Data"), "assistant_skin",
			   tc->assistant_skin);
    gtk_box_pack_start (GTK_BOX (box), frame, FALSE, FALSE, 3);
    G_SET_WIDGET (box, "frame1", frame);

    /* $B%"%7%9%?%s%H%G!<%?$NCV$->l=j(B */
    frame = list_box_new (parent, _("Assistant Document Folder"),
			  _("Assistant Document Folder"),
			  "assistant_doc_dirs_list",
			  "assistant_doc_dirs_entry",
			  tc->assistant_doc_dirs, FALSE);
    gtk_box_pack_start (GTK_BOX (box), frame, TRUE, TRUE, 3);
    G_SET_WIDGET (box, "frame2", frame);
  }
  return box;
}

/* $B%W%i%0%$%s$N@_Dj%\%C%/%9(B ************************************************ */
static gboolean
cb_show_plugin_info (GtkWidget		*widget,
		     GdkEventButton	*event,
		     gpointer		data) {
  GtkTreeView		*treeview = GTK_TREE_VIEW (widget);
  GtkTreeSelection	*selection;
  GtkTreeModel		*model;
  GtkTreeIter		iter;
  GtkWidget		*label;
  GList			*list;
  PluginInfo		*p;
  gchar			*plugin_name, *l_plugin_name;
  gchar			*utf8_name, *l_name;
  gint			success;

  selection = gtk_tree_view_get_selection (treeview);
  if (!selection) return FALSE;

  success = gtk_tree_selection_get_selected (selection, &model, &iter);
  if (!success) return FALSE;

  gtk_tree_model_get (model, &iter, 1, &plugin_name, -1);
  l_plugin_name = g_locale_from_utf8 (plugin_name, -1, NULL, NULL, NULL);
  g_free (plugin_name);

  for (list = g_list_first (plugin_list); list; list = g_list_next (list)) {
    p = (PluginInfo *) list->data;
    utf8_name = g_locale_to_utf8 (_(p->name), -1, NULL, NULL, NULL);
    if (utf8_name) {
      l_name = g_strdup (_(p->name));
      g_free (utf8_name);
    } else {
      l_name = g_locale_from_utf8 (_(p->name), -1, NULL, NULL, NULL);
    }
    if (strcmp (l_plugin_name, l_name) == 0) break;
    g_free (l_name);
  }
  g_free (l_plugin_name);

  if (list) {
    p = (PluginInfo *) list->data;

    label = G_GET_WIDGET (treeview, "plugin_name");
    if (label) gtk_label_set_label (GTK_LABEL (label), _(p->name));

    label = G_GET_WIDGET (treeview, "plugin_author");
    if (label) gtk_label_set_label (GTK_LABEL (label), _(p->author));

    label = G_GET_WIDGET (treeview, "plugin_description");
    if (label) gtk_label_set_label (GTK_LABEL (label), _(p->help));
  }
  return FALSE;
}

/* $B%W%i%0%$%s$N@_Dj%\%C%/%9(B ************************************************ */
static GtkWidget*
config_box_plugin (GtkWidget	*parent) {
  GtkWidget		*box;
  GtkWidget		*vbox;
  GtkWidget		*vbox2;
  GtkWidget		*hbox;
  GtkWidget		*frame;
  GtkWidget		*scrolled_window;
  GtkWidget		*label;
  GtkWidget		*treeview;
  GtkListStore		*store;
  GtkTreeModel		*model;
  GtkCellRenderer	*renderer;
  GtkTreeViewColumn	*column;
  GtkTreeIter		iter;
  GList			*list;
  gchar			*utf8_name;

  /* BOX$B$N@8@.(B */
  box = gtk_vbox_new (TRUE, 3);
  gtk_container_border_width (GTK_CONTAINER (box), 5);
  {
    /* $B%W%i%0%$%s$NCV$->l=j(B */
    frame = list_box_new (parent, _("Plug-In Folder"),
			  _("Plug-In Folder"),
			  "plugin_dirs_list", "plugin_dirs_entry",
			  tc->plugin_dirs, FALSE);
    gtk_box_pack_start (GTK_BOX (box), frame, TRUE, TRUE, 3);
    G_SET_WIDGET (box, "frame1", frame);

    /* $B%W%i%0%$%s%j%9%H(B */
    frame = gtk_frame_new (_("Plug-In List"));
    gtk_box_pack_start (GTK_BOX (box), frame, TRUE, TRUE, 3);
    G_SET_WIDGET (box, "frame2", frame);
    {
      vbox = gtk_vbox_new (FALSE, 3);
      gtk_container_border_width (GTK_CONTAINER (vbox), 8);
      gtk_container_add (GTK_CONTAINER (frame), vbox);
      G_SET_WIDGET (box, "vbox1", vbox);
     
      scrolled_window = gtk_scrolled_window_new (NULL, NULL);
      gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window),
				      GTK_POLICY_AUTOMATIC,
				      GTK_POLICY_AUTOMATIC);
      gtk_scrolled_window_set_shadow_type
	(GTK_SCROLLED_WINDOW (scrolled_window), GTK_SHADOW_ETCHED_OUT);
      gtk_box_pack_start (GTK_BOX (vbox), scrolled_window, TRUE, TRUE, 0);
      G_SET_WIDGET (box, "scrolled_window", scrolled_window);
      {
	store    = gtk_list_store_new (2, G_TYPE_BOOLEAN, G_TYPE_STRING);
	model    = GTK_TREE_MODEL (store);
	treeview = gtk_tree_view_new_with_model (model);
	g_signal_connect (G_OBJECT (treeview), "button_release_event",
			  G_CALLBACK (cb_show_plugin_info), NULL);
	gtk_container_add (GTK_CONTAINER (scrolled_window), treeview);
	G_SET_WIDGET (parent, "plugin_treeview", treeview);
	
	renderer = gtk_cell_renderer_toggle_new ();
	g_signal_connect (renderer, "toggled",
			  G_CALLBACK (cb_toggle_active), model);
	column   = gtk_tree_view_column_new_with_attributes
	  (_("Active"), renderer, "active", 0, NULL);
#if 0
	gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (treeview), TRUE);
#endif
	gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);

	renderer = gtk_cell_renderer_text_new ();
	column   = gtk_tree_view_column_new_with_attributes
	  (_("Plug-In"), renderer, "text", 1, NULL);
#if 0
	gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (treeview), TRUE);
#endif
	gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);
	for (list = plugin_list; list; list = g_list_next (list)) {
	  gtk_list_store_append (store, &iter);
	  utf8_name = g_locale_to_utf8 (_(((PluginInfo *) list->data)->name),
					-1, NULL, NULL, NULL);
	  if (utf8_name) {
	    gtk_list_store_set (store, &iter,
				0, ((PluginInfo *) list->data)->active,
				1, g_strdup (utf8_name), 
				-1);
	    g_free (utf8_name);
	  } else {
	    gtk_list_store_set (store, &iter,
				0, ((PluginInfo *) list->data)->active,
				1, g_strdup (_(((PluginInfo *) 
						list->data)->name)),
				-1);
	  }
	}
      }
    }
    /* $B%W%i%0%$%s%j%9%H(B */
    frame = gtk_frame_new (_("Plug-In Information"));
    gtk_box_pack_start (GTK_BOX (box), frame, TRUE, TRUE, 3);
    G_SET_WIDGET (box, "frame3", frame);
    {
      vbox = gtk_vbox_new (FALSE, 3);
      gtk_container_border_width (GTK_CONTAINER (vbox), 8);
      gtk_container_add (GTK_CONTAINER (frame), vbox);
      G_SET_WIDGET (box, "vbox2", vbox);

      scrolled_window = gtk_scrolled_window_new (NULL, NULL);
      gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window),
				      GTK_POLICY_AUTOMATIC,
				      GTK_POLICY_AUTOMATIC);
      gtk_scrolled_window_set_shadow_type
	(GTK_SCROLLED_WINDOW (scrolled_window), GTK_SHADOW_NONE);

      gtk_container_add (GTK_CONTAINER (vbox), scrolled_window);
      G_SET_WIDGET (box, "scrolled_window2", scrolled_window);
      
      vbox2 = gtk_vbox_new (FALSE, 0);
      gtk_scrolled_window_add_with_viewport (GTK_SCROLLED_WINDOW 
					     (scrolled_window), vbox2);
      G_SET_WIDGET (box, "vbox3", vbox2);
      {
	/* $B%?%$%H%k(B&$B%P!<%8%g%s(B */
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox2), hbox, TRUE, TRUE, 0);
	G_SET_WIDGET (box, "hbox1", hbox);
	{
	  label = gtk_label_new (_("Plugin Name"));
	  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
	  gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 5);
	  G_SET_WIDGET (box, "label1", label);
					       
	  label = gtk_label_new ("");
	  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
	  gtk_box_pack_start (GTK_BOX (hbox), label, TRUE, TRUE, 5);
	  G_SET_WIDGET (treeview, "plugin_name", label);
	}
	/* $B@):n<T(B */	    
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox2), hbox, TRUE, TRUE, 0);
	G_SET_WIDGET (box, "hbox2", hbox);
	{
	  label = gtk_label_new (_("Author"));
	  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
	  gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 5);
	  G_SET_WIDGET (box, "label2", label);

	  label = gtk_label_new ("");
	  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
	  gtk_box_pack_start (GTK_BOX (hbox), label, TRUE, TRUE, 5);
	  G_SET_WIDGET (treeview, "plugin_author", label);
	}	      
	/* $B%W%i%0%$%s$N@bL@(B */	    
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox2), hbox, TRUE, TRUE, 0);
	G_SET_WIDGET (box, "hbox3", hbox);
	{
	  label = gtk_label_new (_("Note"));
	  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
	  gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 5);
	  G_SET_WIDGET (box, "label3", label);

	  label = gtk_label_new ("");
	  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
	  gtk_box_pack_start (GTK_BOX (hbox), label, TRUE, TRUE, 5);
	  G_SET_WIDGET (treeview, "plugin_description", label);
	}	
      }
    }
  }
  return box;
}

/* $B%7%g!<%H%+%C%H%\%C%/%9$N@8@.(B ******************************************** */
static GtkWidget*
config_box_shortcut (GtkWidget	*parent) {
  GtkWidget		*box;
  GtkWidget		*vbox;  
  GtkWidget		*frame;
  GtkWidget		*scrolled_window;
  GtkWidget		*widget;
  GtkListStore		*store;
  GtkTreeModel		*model;
  GtkCellRenderer	*renderer;
  GtkTreeViewColumn	*column;
  GtkTreeIter		iter;
  GList			*list;
  gint			n;
  
  /* BOX$B$N@8@.(B */
  box = gtk_vbox_new (FALSE, 0);
  gtk_container_border_width (GTK_CONTAINER (box), 5);
  {
    /* $B%7%g!<%H%+%C%H%j%9%H(B */
    frame = gtk_frame_new (_("Shortcut"));
    gtk_box_pack_start (GTK_BOX (box), frame, TRUE, TRUE, 3);
    G_SET_WIDGET (box, "frame1", frame);
    {
      vbox = gtk_vbox_new (FALSE, 3);
      gtk_container_border_width (GTK_CONTAINER (vbox), 8);
      gtk_container_add (GTK_CONTAINER (frame), vbox);
      G_SET_WIDGET (box, "vbox1", vbox);

      scrolled_window = gtk_scrolled_window_new (NULL, NULL);
      gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window),
				      GTK_POLICY_AUTOMATIC,
				      GTK_POLICY_AUTOMATIC);
      gtk_scrolled_window_set_shadow_type
	(GTK_SCROLLED_WINDOW (scrolled_window), GTK_SHADOW_ETCHED_OUT);
      gtk_box_pack_start (GTK_BOX (vbox), scrolled_window, TRUE, TRUE, 0);
      G_SET_WIDGET (box, "scrolled_window", scrolled_window);
      {
	store    = gtk_list_store_new (3, 
				       G_TYPE_STRING, 
				       G_TYPE_STRING,
				       G_TYPE_BOOLEAN);
	model    = GTK_TREE_MODEL (store);
	widget   = gtk_tree_view_new_with_model (model);

	g_signal_connect (G_OBJECT (widget), "button_press_event",
			  G_CALLBACK (shortcut_list_button_press), model);

	gtk_container_add (GTK_CONTAINER (scrolled_window), widget);
	gtk_object_set_data (GTK_OBJECT (parent), "shortcut", widget);
	
	renderer = gtk_cell_renderer_text_new ();
	column   = gtk_tree_view_column_new_with_attributes
	  (_("Action"), renderer, "text", 0, NULL);
#if 0
	gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (widget), TRUE);
#endif
	gtk_tree_view_append_column (GTK_TREE_VIEW (widget), column);

	renderer = gtk_cell_renderer_text_new ();
	column   = gtk_tree_view_column_new_with_attributes
	  (_("Shortcut"), renderer, "text", 1, NULL);
#if 0
	gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (widget), TRUE);
#endif
	gtk_tree_view_append_column (GTK_TREE_VIEW (widget), column);

	for (list = tc->shortcut_accel, n = 0;
	     list; list = g_list_next (list), n++) {
	  gtk_list_store_append (store, &iter);
	  gtk_list_store_set (store, &iter,
			      0, _(shortcut_action[n]),
			      1, g_strdup (list->data),
			      -1);
	}
      }
    }
  }
  return box;
}

/* $B$=$NB>$N@_Dj(B ************************************************************ */
static GtkWidget*
config_box_misc (GtkWidget	*parent) {
  GtkWidget		*box;
  GtkWidget		*hbox;
  GtkWidget		*frame;
  GtkWidget		*widget;
  GtkWidget		*option_menu;
  
  /* BOX$B$N@8@.(B */
  box = gtk_vbox_new (FALSE, 0);
  gtk_container_border_width (GTK_CONTAINER (box), 5);
  {
    /* $B2hA|%m!<%@$NCV$->l=j(B */
    frame = list_box_new (parent, _("Image Loader Folder"),
			  _("Image Loader Folder"),
			  "loader_dirs_list", "loader_dirs_entry",
			  tc->loader_dirs, FALSE);
    gtk_box_pack_start (GTK_BOX (box), frame, FALSE, FALSE, 3);
    G_SET_WIDGET (box, "frame1", frame);

    /* $B%k!<%Z$N3HBgN((B */
    frame = gtk_frame_new (_("Loope"));
    gtk_box_pack_start (GTK_BOX (box), frame, FALSE, FALSE, 3);
    G_SET_WIDGET (box, "frame2", frame);
    {
      hbox = gtk_hbox_new (FALSE, 3);
      gtk_container_border_width (GTK_CONTAINER (hbox), 8);
      gtk_container_add (GTK_CONTAINER (frame), hbox);
      G_SET_WIDGET (box, "hbox1", hbox);
      {
	widget = gtk_label_new (_("Default Loope Scale"));
	gtk_box_pack_start (GTK_BOX (hbox), widget, FALSE, FALSE, 0);
	G_SET_WIDGET (box, "label1", widget);

	option_menu = gtk_option_menu_new ();
	gtk_box_pack_end (GTK_BOX (hbox), option_menu, FALSE, FALSE, 0);
	G_SET_WIDGET (box, "option_menu", option_menu);
	{
	  widget = loope_menu_new ();
	  gtk_option_menu_set_menu (GTK_OPTION_MENU (option_menu), widget);
	  G_SET_WIDGET (box, "loope_menu", widget);
	}
	gtk_option_menu_set_history (GTK_OPTION_MENU (option_menu),
				     scale_to_index (tc->loope_scale));
      }
    }
    /* $B%?%$%H%k2hA|(B */
    frame = entry_box_new (parent, _("Title Image"), "title_image",
			   tc->title_image);
    gtk_box_pack_start (GTK_BOX (box), frame, FALSE, FALSE, 3);
    G_SET_WIDGET (box, "frame3", frame);
    
    /* $B2hA|%W%l!<%d$N%9%-%s%G!<%?(B */
    frame = entry_box_new (parent, _("Image Player Skin"), "player_skin",
			   tc->player_skin);
    gtk_box_pack_start (GTK_BOX (box), frame, FALSE, FALSE, 3);    
    G_SET_WIDGET (box, "frame4", frame);    
  }
  return box;
}

/* ************************************************************************* *
 * $B@_Dj%@%$%"%m%0$N@8@.(B ($B%0%m!<%P%k4X?t(B)
 * ************************************************************************* */
GtkWidget*
config_dialog_new (void) {
  GtkWidget	*window;
  GtkWidget	*vbox;
  GtkWidget	*hbox;
  GtkWidget	*button_box;
  GtkWidget	*config_box[6];
  GtkWidget	*button;
  GtkWidget	*separator;
  
  /* $B%&%#%s%I%&$N@8@.(B */
  window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_window_set_wmclass (GTK_WINDOW (window), "teoeyes", "Teo Image Viewer");
  gtk_widget_realize (window);
  gtk_window_set_icon (GTK_WINDOW (window), te_icon);
  gdk_window_set_icon_name (window->window, _("TeoEyes Property Dialog"));
  g_signal_connect (window, "destroy", G_CALLBACK (cb_dialog_close_real),NULL);
  g_signal_connect (window, "delete_event",
		    G_CALLBACK (cb_dialog_close_real), NULL);
  gtk_window_set_title (GTK_WINDOW (window), _("Property Dialog")); 
  gtk_window_set_resizable (GTK_WINDOW (window), FALSE);
  gtk_widget_set_size_request (window, 560, -1);
  {
    /* $B?bD>%Q%C%/%\%C%/%9(B */
    vbox = gtk_vbox_new (FALSE, 0);
    gtk_widget_show (vbox);
    gtk_container_add (GTK_CONTAINER (window), vbox);
    gtk_container_border_width (GTK_CONTAINER (vbox), 5);
    G_SET_WIDGET (window, "vbox1", vbox);
    {
      /* $B?eJ?%Q%C%/%\%C%/%9(B */
      hbox = gtk_hbox_new (FALSE, 5);
      gtk_widget_show (hbox);
      gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
      G_SET_WIDGET (window, "hbox1", hbox);
      {
	/* $B%\%?%s%Q%C%/MQ$N%\%C%/%9(B */
	button_box = gtk_vbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (hbox), button_box, FALSE, FALSE, 0);
	G_SET_WIDGET (window, "button_box", button_box);
	{
	  /* $B!V%*%W%7%g%s!W%\%?%s(B */
	  button = gtk_icon_button_new_from_inline_with_label
	    (config_option, _("Option"), GTK_ICON_BUTTON_TEXT_BOTTOM, 2);
	  gtk_box_pack_start (GTK_BOX (button_box), button, FALSE, FALSE, 0);
	  gtk_widget_set_size_request (button, 120, -1);
	  g_signal_connect (G_OBJECT (button), "clicked",
			    G_CALLBACK (cb_change_box), (gpointer) 0);
	  G_SET_WIDGET (window, "button1", button);

	  /* $B!V2hA|%(%/%9%W%m!<%i!W%\%?%s(B */
	  button = gtk_icon_button_new_from_inline_with_label
	    (config_browser, _("Image Explorer"),
	     GTK_ICON_BUTTON_TEXT_BOTTOM, 2);
	  gtk_box_pack_start (GTK_BOX (button_box), button, FALSE, FALSE, 0);
	  gtk_widget_set_size_request (button, 120, -1);
	  g_signal_connect (G_OBJECT (button), "clicked",
			    G_CALLBACK (cb_change_box), (gpointer) 1);
	  G_SET_WIDGET (window, "button2", button);

	  /* $B!V%"%7%9%?%s%H!W%\%?%s(B */
	  button = gtk_icon_button_new_from_inline_with_label
	    (config_assistant, _("Assistant"),GTK_ICON_BUTTON_TEXT_BOTTOM, 2);
	  gtk_box_pack_start (GTK_BOX (button_box), button, FALSE, FALSE, 0);
	  gtk_widget_set_size_request (button, 120, -1);
	  g_signal_connect (G_OBJECT (button), "clicked",
			    G_CALLBACK (cb_change_box), (gpointer) 2);
	  G_SET_WIDGET (window, "button3", button);

	  /* $B!V%W%i%0%$%s!W%\%?%s(B */
	  button = gtk_icon_button_new_from_inline_with_label
	    (config_plugin, _("Plug-In"),GTK_ICON_BUTTON_TEXT_BOTTOM, 2);
	  gtk_box_pack_start (GTK_BOX (button_box), button, FALSE, FALSE, 0);
	  gtk_widget_set_size_request (button, 120, -1);
	  g_signal_connect (G_OBJECT (button), "clicked",
			    G_CALLBACK (cb_change_box), (gpointer) 3);
	  G_SET_WIDGET (window, "button4", button);

	  /* $B!V%7%g!<%H%+%C%H!W%\%?%s(B */
	  button = gtk_icon_button_new_from_inline_with_label
	    (config_shortcuts, _("ShortCut"),GTK_ICON_BUTTON_TEXT_BOTTOM, 2);
	  gtk_box_pack_start (GTK_BOX (button_box), button, FALSE, FALSE, 0);
	  gtk_widget_set_size_request (button, 120, -1);
	  g_signal_connect (G_OBJECT (button), "clicked",
			    G_CALLBACK (cb_change_box), (gpointer) 4);
	  G_SET_WIDGET (window, "button5", button);

	  /* $B!V$=$NB>!W%\%?%s(B */
	  button = gtk_icon_button_new_from_inline_with_label
	    (config_misc, _("Misc"),GTK_ICON_BUTTON_TEXT_BOTTOM, 2);
	  gtk_box_pack_start (GTK_BOX (button_box), button, FALSE, FALSE, 0);
	  gtk_widget_set_size_request (button, 120, -1);
	  g_signal_connect (G_OBJECT (button), "clicked",
			    G_CALLBACK (cb_change_box), (gpointer) 5);
	  G_SET_WIDGET (window, "button6", button);
	}
	gtk_widget_show_all (button_box);
      }
      /* $B$=$l$>$l$N@_Dj%\%C%/%9$N@8@.(B */
      config_box[0] = config_box_option (window);
      gtk_box_pack_start (GTK_BOX (hbox), config_box[0], TRUE, TRUE, 0);
      gtk_widget_show_all (config_box[0]);
      G_SET_WIDGET (window, "box0", config_box[0]);

      config_box[1] = config_box_explorer (window);
      gtk_box_pack_start (GTK_BOX (hbox), config_box[1], TRUE, TRUE, 0);
      G_SET_WIDGET (window, "box1", config_box[1]);

      config_box[2] = config_box_assistant (window);
      gtk_box_pack_start (GTK_BOX (hbox), config_box[2], TRUE, TRUE, 0);
      G_SET_WIDGET (window, "box2", config_box[2]);

      config_box[3] = config_box_plugin (window);
      gtk_box_pack_start (GTK_BOX (hbox), config_box[3], TRUE, TRUE, 0);
      G_SET_WIDGET (window, "box3", config_box[3]);

      config_box[4] = config_box_shortcut (window);
      gtk_box_pack_start (GTK_BOX (hbox), config_box[4], TRUE, TRUE, 0);
      G_SET_WIDGET (window, "box4", config_box[4]);

      config_box[5] = config_box_misc (window);
      gtk_box_pack_start (GTK_BOX (hbox), config_box[5], TRUE, TRUE, 0);
      G_SET_WIDGET (window, "box5", config_box[5]);
    }
    /* $B%;%Q%l!<%?(B */
    separator = gtk_hseparator_new ();
    gtk_box_pack_start (GTK_BOX (vbox), separator, FALSE, TRUE, 5);
    gtk_widget_show (separator);
    G_SET_WIDGET (window, "separator1", separator);

    /* $B?eJ?%Q%C%/%\%C%/%9(B */
    hbox = gtk_hbox_new (FALSE, 0);
    gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
    G_SET_WIDGET (window, "hbox2", hbox);
    {
      /* $B!VJD$8$k!W%\%?%s(B */
      button = gtk_button_new_from_stock (GTK_STOCK_CLOSE);
      g_signal_connect (G_OBJECT (button), "clicked",
			G_CALLBACK (cb_dialog_close), window);
      gtk_box_pack_end (GTK_BOX (hbox), button, FALSE, FALSE, 0);
      gtk_widget_set_size_request (button, 100, -1);
      GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
      gtk_widget_grab_default (button);
      G_SET_WIDGET (window, "button_close", button);

      /* $B!VE,MQ!W%\%?%s(B */
      button = gtk_button_new_from_stock (GTK_STOCK_APPLY);
      g_signal_connect (G_OBJECT (button), "clicked",
			G_CALLBACK (cb_dialog_apply), window);
      gtk_box_pack_end (GTK_BOX (hbox), button, FALSE, FALSE, 0);
      gtk_widget_set_size_request (button, 100, -1);
      G_SET_WIDGET (window, "button_apply", button);

      /* $B!V(BOK$B!W%\%?%s(B */
      button = gtk_button_new_from_stock (GTK_STOCK_OK);
      g_signal_connect (G_OBJECT (button), "clicked",
			G_CALLBACK (cb_dialog_ok), window);
      gtk_box_pack_end (GTK_BOX (hbox), button, FALSE, FALSE, 0);
      gtk_widget_set_size_request (button, 100, -1);
      G_SET_WIDGET (window, "button_ok", button);
    }
    gtk_widget_show_all (hbox);    
  }
  if (scale == 0) scale = tc->loope_scale;

  return window;
}

/* *************************************************** End of property.c *** */
