/*************************************************************************************************/
/*!
   	@file		ParserProcess.h
	@author 	Fanzo
*/
/*************************************************************************************************/
#pragma		once

///////////////////////////////////////////////////////////////////////////////////////////////////
//include files
#include	"ObjectAry.h"

#pragma pack( push , 8 )		//set align

namespace icubic
{
namespace parserprocess
{

///////////////////////////////////////////////////////////////////////////////////////////////////
// preprocessor deifne

///////////////////////////////////////////////////////////////////////////////////////////////////
// type define

///////////////////////////////////////////////////////////////////////////////////////////////////
// classes define


/**************************************************************************************************
 "IParserProcess" interface 
***************************************************************************************************/
cb_guid_define( IParserProcess_IID , 0xAE442377 , 0x5C164f52 , 0xA070E805 , 0x87DF4DA4 );
class IParserProcess;
typedef icubic::iface_object< IParserProcess , IParserProcess_IID >		iParserProcess;
typedef icubic::iface_reference< IParserProcess , IParserProcess_IID >	rParserProcess;
///////////////////////////////////////////////////////////////////////////////////////////////////
class IParserProcess
{
public:
	enum Result
	{
		Success , 
		ParseError , 
		StackOverflow , 
		NodeOverflow , 
	};
public:
//=================================================================================================
virtual
Result cb_call Execute
		(
		const wstring&		source , 
		int*				endpos
		) = 0;
//=================================================================================================
virtual
bool cb_call Save
		(
		IStreamWrite*	stream
		) = 0;
//=================================================================================================
virtual
bool cb_call Load
		(
		IStreamRead*	stream
		) = 0;
};
/**************************************************************************************************
 "IParserSurvice" interface 
***************************************************************************************************/
cb_guid_define( IParserSurvice_IID , 0x73B4DA17 , 0x4EB54990 , 0xB3C07EB4 , 0xB537FEBC );
class IParserSurvice;
typedef icubic::iface_object< IParserSurvice , IParserSurvice_IID >		iParserSurvice;
typedef icubic::iface_reference< IParserSurvice , IParserSurvice_IID >	rParserSurvice;
///////////////////////////////////////////////////////////////////////////////////////////////////
class IParserSurvice
{
public:
//=================================================================================================
virtual
void cb_call ClearTextBuffer() = 0;
//=================================================================================================
virtual
wstring cb_call GetTextBuffer() = 0;
//=================================================================================================
virtual
void cb_call SetTextBuffer
		(
		const wstring&	text
		) = 0;
//=================================================================================================
virtual
void cb_call AddTextBuffer
		(
		const wstring&	text
		) = 0;
//=================================================================================================
virtual
void cb_call PushTextBuffer() = 0;
//=================================================================================================
virtual
void cb_call PopTextBuffer() = 0;
};
/**************************************************************************************************
 "IParserCmd" interface 
***************************************************************************************************/
cb_guid_define( IParserCmd_IID , 0x7BC17DEB , 0xFD974aba , 0xA033BE57 , 0x8D769804 );
class IParserCmd;
typedef icubic::iface_object< IParserCmd , IParserCmd_IID >		iParserCmd;
typedef icubic::iface_reference< IParserCmd , IParserCmd_IID >	rParserCmd;
///////////////////////////////////////////////////////////////////////////////////////////////////
class IParserCmd
{
public:
//=================================================================================================
virtual
guid cb_call GetId() = 0;
//=================================================================================================
virtual
void cb_call Execute
		(
		IParserSurvice*	survice , 
		object*			r
		) = 0;
//=================================================================================================
virtual
bool cb_call Save
		(
		IStreamWrite*	stream
		) = 0;
//=================================================================================================
virtual
bool cb_call Load
		(
		IStreamRead*	stream
		) = 0;
};
/**************************************************************************************************
 "IParserCond" interface 
***************************************************************************************************/
cb_guid_define( IParserCond_IID , 0xC999E109 , 0xC92E4ec2 , 0xAAFDE227 , 0x810ECCB9 );
class IParserCond;
typedef icubic::iface_object< IParserCond , IParserCond_IID >		iParserCond;
typedef icubic::iface_reference< IParserCond , IParserCond_IID >	rParserCond;
///////////////////////////////////////////////////////////////////////////////////////////////////
class IParserCond
{
public:
//=================================================================================================
virtual
guid cb_call GetId() = 0;
//=================================================================================================
virtual
bool cb_call Execute
		(
		IParserSurvice*		survice , 
		const wstring&		source , 
		int*				pos
		) = 0;
//=================================================================================================
virtual
bool cb_call Save
		(
		IStreamWrite*	stream
		) = 0;
//=================================================================================================
virtual
bool cb_call Load
		(
		IStreamRead*	stream
		) = 0;
};
/**************************************************************************************************
 "IParserObjectFactory" interface 
***************************************************************************************************/
cb_guid_define( IParserObjectFactory_IID , 0x45BB5429 , 0x917B48be , 0x8AB2D4B1 , 0xDAB45622 );
class IParserObjectFactory;
typedef icubic::iface_object< IParserObjectFactory , IParserObjectFactory_IID >		iParserObjectFactory;
typedef icubic::iface_reference< IParserObjectFactory , IParserObjectFactory_IID >	rParserObjectFactory;
///////////////////////////////////////////////////////////////////////////////////////////////////
class IParserObjectFactory
{
public:
//=================================================================================================
virtual
object cb_call CreateCmd
		(
		const guid&		id
		) = 0;
//=================================================================================================
virtual
object cb_call CreateCond
		(
		const guid&		id
		) = 0;
};
/**************************************************************************************************
"ParserLink" class 
**************************************************************************************************/
class ParserLink
{
// variable member
public:
	iParserCond		m_cond;
	int				m_cond_parser;
	int				m_next_node;
	
// public functions
public:
//=================================================================================================
ParserLink() : m_cond_parser( -1 ) , m_next_node( -1 )
{
}
//=================================================================================================
bool cb_call Save
		(
		IStreamWrite*	stream
		)
{
	if( false == stream->Write( &m_cond_parser , sizeof(m_cond_parser) , 1 , Little_EndianType ) ) 
		return false;
	if( false == stream->Write( &m_next_node , sizeof(m_next_node) , 1 , Little_EndianType ) ) 
		return false;

	guid	id = m_cond == false ? null_guid() : m_cond->GetId();
	if( false == stream->Write( &id.m_Data[0] , sizeof(id.m_Data[0]) , _countof(id.m_Data) , Little_EndianType ) )
		return false;
	if( m_cond == true )
	{
		if( false == m_cond->Save( stream ) )
			return false;
	}
	return true;
}
//=================================================================================================
bool cb_call Load
		(
		IStreamRead*			stream , 
		IParserObjectFactory*	factory
		)
{
	if( 1 != stream->Read( &m_cond_parser , sizeof(m_cond_parser) , 1 , Little_EndianType ) ) 
		return false;
	if( 1 != stream->Read( &m_next_node , sizeof(m_next_node) , 1 , Little_EndianType ) ) 
		return false;

	guid	id;
	if( _countof(id.m_Data) != stream->Read( &id.m_Data[0] , sizeof(id.m_Data[0]) , _countof(id.m_Data) , Little_EndianType ) )
		return false;
	if( id != null_guid() )
	{
		m_cond	= factory->CreateCond( id );
		if( m_cond == false )
			return false;
		if( false == m_cond->Load( stream ) )
			return false;
	}
	return true;
}
};
/**************************************************************************************************
"ParserNode" class 
**************************************************************************************************/
class ParserNode
{
// variable member
public:
	iParserCmd			m_cmd;
	Array<ParserLink>	m_links;

// public functions
public:
//=================================================================================================
bool cb_call Save
		(
		IStreamWrite*	stream
		)
{
	guid	id = m_cmd == false ? null_guid() : m_cmd->GetId();
	if( false == stream->Write( &id.m_Data[0] , sizeof(id.m_Data[0]) , _countof(id.m_Data) , Little_EndianType ) )
		return false;
	if( m_cmd == true )
	{
		if( false == m_cmd->Save( stream ) )
			return false;
	}	
	int	linkoff , linknum = m_links.GetDatanum();
	if( false == stream->Write( &linknum , sizeof(linknum) , 1 , Little_EndianType ) )
		return false;
	for( linkoff = 0 ; linkoff < linknum ; linkoff++ )
	{
		if( false == m_links[linkoff].Save( stream ) )
			return false;
	}
	return true;
}
//=================================================================================================
bool cb_call Load
		(
		IStreamRead*			stream , 
		IParserObjectFactory*	factory
		)
{
	guid	id;
	if( _countof(id.m_Data) != stream->Read( &id.m_Data[0] , sizeof(id.m_Data[0]) , _countof(id.m_Data) , Little_EndianType ) )
		return false;
	if( id != null_guid() )
	{
		m_cmd	= factory->CreateCmd( id );
		if( m_cmd == false )
			return false;
		if( false == m_cmd->Load( stream ) )
			return false;		
	}

	int	linkoff , linknum = m_links.GetDatanum();
	if( 1 != stream->Read( &linknum , sizeof(linknum) , 1 , Little_EndianType ) )
		return false;
	m_links.Resize( linknum );
	for( linkoff = 0 ; linkoff < linknum ; linkoff++ )
	{
		if( false == m_links[linkoff].Load( stream , factory ) )
			return false;
	}
	return true;
}
};
/**************************************************************************************************
"Parser" class 
**************************************************************************************************/
class Parser
{
// variable member
public:
	Array<ParserNode>	m_nodes;
	int					m_start_node;
	int					m_end_node;

// public functions
public:
//=================================================================================================
Parser() : m_start_node( -1 ) , m_end_node( -1 )
{
}
//=================================================================================================
bool cb_call Save
		(
		IStreamWrite*	stream
		)
{
	if( false == stream->Write( &m_start_node , sizeof(m_start_node) , 1 , Little_EndianType ) )
		return false;
	if( false == stream->Write( &m_end_node , sizeof(m_end_node) , 1 , Little_EndianType ) )
		return false;

	int	nodeoff , nodenum = m_nodes.GetDatanum();
	if( false == stream->Write( &nodenum , sizeof(nodenum) , 1 , Little_EndianType ) )
		return false;
	for( nodeoff = 0 ; nodeoff < nodenum ; nodeoff++ )
	{
		if( false == m_nodes[nodeoff].Save( stream ) )
			return false;
	}
	return true;
}
//=================================================================================================
bool cb_call Load
		(
		IStreamRead*			stream , 
		IParserObjectFactory*	factory
		)
{
	if( 1 != stream->Read( &m_start_node , sizeof(m_start_node) , 1 , Little_EndianType ) )
		return false;
	if( 1 != stream->Read( &m_end_node , sizeof(m_end_node) , 1 , Little_EndianType ) )
		return false;

	int	nodeoff , nodenum;
	if( 1 != stream->Read( &nodenum , sizeof(nodenum) , 1 , Little_EndianType ) )
		return false;
	m_nodes.Resize( nodenum );
	for( nodeoff = 0 ; nodeoff < nodenum ; nodeoff++ )
	{
		if( false == m_nodes[nodeoff].Load( stream , factory ) )
			return false;
	}
	return true;
}
};

/**************************************************************************************************
"ParserProcess" class 
**************************************************************************************************/
class ParserProcess : 
	virtual public object_base , 
	public IParserProcess , 
	public IParserSurvice
{
	query_begin();
	iface_hook( IParserProcess , IParserProcess_IID )
	query_end( object_base );

	cb_guid_define( TextStack_IID , 0x16EAAB25 , 0x4026454b , 0x9656DB41 , 0x66110F5C );
	class TextStack;
	typedef icubic::iface_object< TextStack , TextStack_IID >		iCTextStack;
	typedef icubic::iface_reference< TextStack , TextStack_IID >	rCTextStack;
	class TextStack : 
		virtual public object_base , 
		public Stackdata<wstring>
	{
		query_begin();
		iface_hook( TextStack , TextStack_IID )
		query_end( object_base );
	};
	
// variable member
private:
	Stackdata<object>		m_gen_stack;
	iCTextStack				m_text_buffer;

	rParserObjectFactory	m_factory;
	int						m_start_parser;
	Array<Parser>			m_parsers;
	
	const int				m_parse_count_max;
	const int				m_node_count_max;
	
// "IParserSurvice" interface functions
private:
//=================================================================================================
void cb_call ClearTextBuffer()
{
	*m_text_buffer->Ptr() = L"";
}
//=================================================================================================
wstring cb_call GetTextBuffer()
{
	return *m_text_buffer->Ptr();
}
//=================================================================================================
void cb_call SetTextBuffer
		(
		const wstring&	text
		)
{
	*m_text_buffer->Ptr()	= text;
}
//=================================================================================================
void cb_call AddTextBuffer
		(
		const wstring&	text
		)
{
	*m_text_buffer->Ptr() += text;
}
//=================================================================================================
void cb_call PushTextBuffer()
{
	m_text_buffer->Push();
}
//=================================================================================================
void cb_call PopTextBuffer()
{
	if( m_text_buffer->GetDatanum() <= 1 )
		return;
	m_text_buffer->Pop();
}
// private functions
private:
//=================================================================================================
void Reset()
{
	m_text_buffer	= (iCTextStack)instance<TextStack>();
	m_text_buffer->Push();
	m_gen_stack.Reset();
}
//=================================================================================================
void ExecuteCmd
		(
		int		parser , 
		int		node
		)
{
	if( m_parsers[ parser ].m_nodes[ node ].m_cmd == false )
		return;
	object		r;
	m_parsers[ parser ].m_nodes[ node ].m_cmd->Execute( this , &r );
	if( r == true )
	{
		m_gen_stack.Push();
		*m_gen_stack	= r;
	}
}
//=================================================================================================
Result ExecuteCondition
		(
		int				parser , 
		int				node , 
		const wstring&	source , 
		int*			pos , 
		int*			perrpos , 
		int*			next_node , 
		int				parse_count
		)
{
	int		linkoff , linknum =m_parsers[parser].m_nodes[node].m_links.GetDatanum();
	for( linkoff = 0 ; linkoff < linknum ; linkoff++ )
	{
		ParserLink*	plink	= &m_parsers[parser].m_nodes[node].m_links[linkoff];
		int			t_pos	= *pos;
		Result		result	= Success;
		
		// cond
		if( plink->m_cond == true )
		{
			if( false == plink->m_cond->Execute( this , source , &t_pos ) )
				result = ParseError;
		}
		else if( plink->m_cond_parser != -1 )
		{
			result = ExecuteParser( plink->m_cond_parser , source , &t_pos , perrpos , parse_count );
		}
		// result
		if( result == Success )
		{
			*perrpos	= max( *perrpos , t_pos );
			*pos		= t_pos;
			*next_node	= plink->m_next_node;
			return Success;
		}
		else if( result != ParseError )
			return result;
	}
	return ParseError;
}
//=================================================================================================
Result ExecuteNode
		(
		int				parser , 
		int				node , 
		const wstring&	source , 
		int*			pos , 
		int*			perrpos , 
		int*			next_node , 
		int				parse_count
		)
{
	ExecuteCmd( parser , node );
	return ExecuteCondition( parser , node , source , pos , perrpos , next_node , parse_count );
}
//=================================================================================================
Result ExecuteParser
		(
		int				parser , 
		const wstring&	source , 
		int*			pos , 
		int*			perrpos , 
		int				parse_count
		)
{
	// parse count
	parse_count--;
	if( parse_count < 0 )
		return StackOverflow;
		
	// save data
	int			restore			= m_gen_stack.GetRestoreId();
	iCTextStack	text_restore	= m_text_buffer;
	m_text_buffer				= (iCTextStack)instance<TextStack>();
	m_text_buffer->Push();
	*m_text_buffer->Ptr()		= *text_restore->Ptr();
	
	// process
	int		node		= m_parsers[parser].m_start_node;
	int		node_count	= m_node_count_max;
	Result	result;
	while( true )
	{
		node_count--;
		if( node_count < 0 )
		{
			result = NodeOverflow;
			break;
		}
		
		int	next_node;
		result = ExecuteNode( parser , node , source , pos , perrpos , &next_node , parse_count );
		if( result != Success )
			break;
		if( next_node == m_parsers[parser].m_end_node )
			break;
		node	= next_node;
	}
	
	// restore data
	if( result == Success )
	{
		*text_restore->Ptr()	= *m_text_buffer->Ptr();
		m_text_buffer			= text_restore;
	}
	else
	{
		m_gen_stack.Restore( restore );
		m_text_buffer	= text_restore;
	}
	return result;
}	
// "IParserProcess" interface functions
public:
//=================================================================================================
Result cb_call Execute
		(
		const wstring&		source , 
		int*				endpos
		)
{
	Reset();
	int		t_pos		= 0;
	int		t_errpos	= 0;
	Result	result	= ExecuteParser( m_start_parser , source , &t_pos , &t_errpos , m_parse_count_max );
	if( result != Success )
	{
		store( endpos , t_errpos );
		return result;
	}
	else
	{
		store( endpos , t_pos );
		return result;
	}
}
//=================================================================================================
bool cb_call Save
		(
		IStreamWrite*	stream
		)
{
	{
		int	ver = 1;
		if( false == stream->Write( &ver , sizeof(ver) , 1 , Little_EndianType ) )
			return false;
	}
	{
		if( false == stream->Write( &m_start_parser , sizeof(m_start_parser) , 1 , Little_EndianType ) )
			return false;
	}
	int	parseoff , parsenum = m_parsers.GetDatanum();
	if( false == stream->Write( &parsenum , sizeof(parsenum) , 1 , Little_EndianType ) )
		return false;
	for( parseoff = 0 ; parseoff < parsenum ; parseoff++ )
	{
		if( false == m_parsers[parseoff].Save( stream ) )
			return false;
	}	
	return true;
}
//=================================================================================================
bool cb_call Load
		(
		IStreamRead*	stream
		)
{
	ResetCode();
	{
		int	ver;
		if( 1 != stream->Read( &ver , sizeof(ver) , 1 , Little_EndianType ) )
			return false;
		if( ver != 1 )
			return false;
	}
	{
		if( 1 != stream->Read( &m_start_parser , sizeof(m_start_parser) , 1 , Little_EndianType ) )
		{
			ResetCode();
			return false;
		}
	}
	int	parseoff , parsenum;
	if( 1 != stream->Read( &parsenum , sizeof(parsenum) , 1 , Little_EndianType ) )
	{
		ResetCode();
		return false;
	}
	m_parsers.Resize( parsenum );
	for( parseoff = 0 ; parseoff < parsenum ; parseoff++ )
	{
		if( false == m_parsers[parseoff].Load( stream , (iParserObjectFactory)m_factory.lock() ) )
			return false;
	}	
	return true;
}
// protected functions
protected:
//=================================================================================================
void* GetFirst()
{
	return m_gen_stack.GetFirst();
}
//=================================================================================================
void* GetNext
		(
		void*	prev
		)
{
	return m_gen_stack.GetNext( prev );
}
//=================================================================================================
void* GetPrev
		(
		void*	next
		)
{
	return m_gen_stack.GetPrev( next );
}
//=================================================================================================
object& GetData
		(
		void*	ptr
		)
{
	return m_gen_stack.GetData( ptr );
}
// public functions
public:
//=================================================================================================
ParserProcess() :
		m_start_parser( -1 ) , 
		m_parse_count_max( 256 ) , 
		m_node_count_max( 100000 )
{
}
//=================================================================================================
void SetFactory
		(
		rParserObjectFactory&	factory
		)
{
	m_factory	= factory;
}
//=================================================================================================
void ResetCode()
{
	m_start_parser = -1;
	m_parsers.Resize( 0 );
}
//=================================================================================================
bool SetParserNum
		(
		int		parsernum , 
		int		start_parser
		)
{
	if( start_parser < 0 
	||  start_parser > parsernum - 1 )
		return false;
	m_start_parser	= start_parser;
	m_parsers.Resize( parsernum );
	return true;
}
//=================================================================================================
bool SetParserProp
		(
		int		parser , 
		int		nodenum , 
		int		start_node , 
		int		end_node
		)
{
	if( start_node < 0 
	 || end_node < 0 
	 || start_node > nodenum - 1
	 || end_node > nodenum - 1 )
		return false;
	m_parsers[ parser ].m_start_node	= start_node;
	m_parsers[ parser ].m_end_node		= end_node;
	m_parsers[ parser ].m_nodes.Resize( nodenum );
	return true;
}
//=================================================================================================
bool SetNodeProp
		(
		int			parser , 
		int			node , 
		int			linknum , 
		iParserCmd&	cmd
		)
{
	m_parsers[ parser ].m_nodes[ node ].m_cmd = cmd;
	m_parsers[ parser ].m_nodes[ node ].m_links.Resize( linknum );
	return true;
}
//=================================================================================================
bool SetLinkProp
		(
		int				parser , 
		int				node , 
		int				link , 
		int				jmp_parser , 
		int				jmp_node , 
		iParserCond&	cond
		)
{
	m_parsers[ parser ].m_nodes[ node ].m_links[ link ].m_cond			= cond;
	m_parsers[ parser ].m_nodes[ node ].m_links[ link ].m_cond_parser	= jmp_parser;
	m_parsers[ parser ].m_nodes[ node ].m_links[ link ].m_next_node		= jmp_node;
	return true;
}
};
///////////////////////////////////////////////////////////////////////////////////////////////////
// global variable define

///////////////////////////////////////////////////////////////////////////////////////////////////
// global functions define

};	//namespace
};	//namespace

//using namespace icubic;		

#pragma pack( pop )			//release align
