# vim: tabstop=4 shiftwidth=4 softtabstop=4
# -*- coding: utf-8 -*-
#

__version__ = '1.1'

import sys
import os
import re
from linkpair.device import Device
from linkpair.port import Port
from linkpair.linkpair import LinkPair
from linkpair.formatter.graphviz_tb import GraphvizTBFormatter
from linkpair.utils.common import CommonUtils
from linkpair.collect.utils import CollectUtils
from linkpair.collect.collector_dataset import CollectorDataset
from linkpair.collect.agent.commandrunner import CommandRunner
from linkpair.collect.agent.linux_agent import LinuxAgent
from linkpair.collect.agent.linuxnet_agent import LinuxNetAgent
from linkpair.collect.agent.namespace_agent import NamespaceAgent
from linkpair.collect.agent.veth_agent import VethAgent
from linkpair.collect.agent.ovs_agent import OVSAgent
from linkpair.collect.agent.linuxbridge_agent import LinuxBridgeAgent
from linkpair.collect.agent.linuxvlan_agent import LinuxVlanAgent
from linkpair.collect.agent.sspvxlan_agent import SSPVxlanAgent
from linkpair.collect.agent.libvirt_agent import LibvirtAgent


class Collector(object):

    '''LinkpPair collector

    This class gets the LinkPair information from LinuxBridge and Open vSwitch and libvirt
    '''

    PEER_FOUND = 1

    def __init__(self, opts, util, formatter=GraphvizTBFormatter()):
        self._devices = {}
        self._ports = {}
        self._os_info = {}
        self._linuxnet = {}
        self._linkpairs = []
        self._port_to_br = {}
        self._iface_to_nss = {}
        self._veth_peer = {}
        self._opts = opts
        self._u = util
        self._remote_desc = opts.remote_desc
        self._ssh_username = ""
        self._ssh_hostname = ""
        self._ssh_hostport = 22
        self._remote_password = ""
        self._remote_sshkey = ""
        self._ssh_keyauth = False
        self._ssh_passauth = False

        [self._ssh_username, self._ssh_hostname,
            self._ssh_hostport] = self._u.parse_remote_desc(self._remote_desc)
        self._formatter = formatter
        self._runner = CommandRunner(self._remote_desc)
        self._u.debug_out("Collector initialized")

    def run(self):
        cd = CollectorDataset(
            self._devices, self._ports, self._os_info, self._linuxnet, self._linkpairs,
            self._port_to_br, self._iface_to_nss, self._veth_peer)
        cu = CollectUtils(
            cd, self._u, self._formatter)
        if self._opts.disable_linux_agent == False:
            os_agent = LinuxAgent(
                self._runner, cu, self._remote_desc, cd, self._formatter)
            self._u.debug_out("OSAgent running...")
            os_agent.run()
        if self._opts.disable_linuxnet_agent == False:
            linuxnet_agent = LinuxNetAgent(
                self._runner, cu, self._remote_desc, cd, self._formatter)
            self._u.debug_out("LinuxNetAgent running...")
            linuxnet_agent.run()
        if self._opts.disable_namespace_agent == False:
            namespace_agent = NamespaceAgent(
                self._runner, cu, self._remote_desc, cd, self._formatter)
            self._u.debug_out("NameSpaceAgent running...")
            namespace_agent.run()
        if self._opts.disable_veth_agent == False:
            veth_agent = VethAgent(
                self._runner, cu, self._remote_desc, cd, self._formatter)
            self._u.debug_out("VethAgent running...")
            veth_agent.run()
        if self._opts.disable_linuxvlan_agent == False:
            vlan_agent = LinuxVlanAgent(
                self._runner, cu, self._remote_desc, cd, self._formatter)
            self._u.debug_out("LinuxVlanAgent running...")
            vlan_agent.run()
        if self._opts.disable_sspvxlan_agent == False:
            vxlan_agent = SSPVxlanAgent(
                self._runner, cu, self._remote_desc, cd, self._formatter)
            self._u.debug_out("SSPVxlanAgent running...")
            vxlan_agent.run()
        if self._opts.disable_ovs_agent == False:
            ovs_agent = OVSAgent(
                self._runner, cu, self._remote_desc, cd, self._formatter)
            self._u.debug_out("OVSAgent running...")
            ovs_agent.run()
        if self._opts.disable_linuxbridge_agent == False:
            bridge_agent = LinuxBridgeAgent(
                self._runner, cu, self._remote_desc, cd, self._formatter)
            self._u.debug_out("LinuxBridgeAgent running...")
            bridge_agent.run()
        if self._opts.disable_libvirt_agent == False:
            libvirt_agent = LibvirtAgent(
                self._runner, cu, self._remote_desc, cd, self._formatter)
            self._u.debug_out("LibvirtAgent running...")
            libvirt_agent.run()

    def get_linkpairs(self):
        return self._linkpairs

    def connect_ssh(self):
        self._runner.connect_ssh()

    def set_remote_sshkey(self, remote_sshkey):
        self._runner.set_remote_sshkey(remote_sshkey)

    def set_remote_password(self, remote_password):
        self._runner.set_remote_password(remote_password)
