#!/usr/bin/env python
# vim: tabstop=4 shiftwidth=4 softtabstop=4
# -*- coding: utf-8 -*-
#
#   ti2.py : List informations of the TAP devices for running VMs
#
#

__version__ = '1.2'

import sys
from optparse import OptionParser, OptionGroup
from linkpair.collect.collector import Collector
from linkpair.formatter.grapheasy import GraphEasyFormatter
from linkpair.formatter.springy import SpringyFormatter
from linkpair.formatter.springy_raphael import SpringyRaphaelFormatter
from linkpair.formatter.graphviz_tb import GraphvizTBFormatter
from linkpair.utils.common import CommonUtils
from linkpair.exporter.JSON import JsonExporter
from linkpair.importer.JSON import JsonImporter
from linkpair.timer.collect_task import CollectTask
import api_server


def print_linkpair_records(linkpairs):
    util.debug_out("start print_linkpair_records()")
    formatter.set_linkpairs(linkpairs)
    for linkpair in formatter.format_linkpairs():
        util.message_out(linkpair)
    util.debug_out("end print_linkpair_records()")


def collect_linkpairs(opts, util, formatter):
    def collect_func():
        col = Collector(opts, util, formatter)
        if opts.remote_sshkey is not None:
            col.set_remote_sshkey(opts.remote_sshkey)
            col.connect_ssh()
        if opts.remote_password is not None:
            col.set_remote_password(opts.remote_password)
            col.connect_ssh()
        col.run()
        return col.get_linkpairs()
    return collect_func

if __name__ == "__main__":
    # init program options
    p = OptionParser(version="ver:%s" % __version__)
    g1 = OptionGroup(p, "Remote connection",
                     "connect to a remote system via SSH")
    g1.add_option('-r', '--remote', dest='remote_desc',
                  help="specify user's name and hostname, port number (ex. -r user@hostname:22)", metavar="REMOTE_DESC")
    g1.add_option('-p', '--password', dest='remote_password',
                  help="specify user's SSH password", metavar="PASSWORD")
    g1.add_option('-k', '--sshkey', dest='remote_sshkey',
                  help="specify user's SSH private-key file", metavar="FILE")
    p.add_option_group(g1)
    p.add_option('-f', '--format', dest='format',
                 choices=['g', 'graph-easy', 's', 'springy',
                          'r', 'springy-raphael', 't', 'graphviz-tb'],
                 default="graphviz-tb",
                 help="specify output format [FORMAT: g/graph-easy, s/springy, r/springy-raphael, t/graphviz-tb]",
                 metavar="FORMAT")
    p.add_option('-d', '--debug', dest='debug', action="store_true", default=False,
                 help="enable debug output")
    p.add_option('-e', '--export_file', dest='export_file',
                 help="export linkpairs to json file", metavar="FILE")
    p.add_option('-x', '--export_stdout', dest='export_stdout', action="store_true", default=False,
                 help="export linkpairs to stdout")
    p.add_option('-i', '--import_file', dest='import_file',
                 help="import json file as linkpairs", metavar="FILE")
    p.add_option('-a', '--api_server', dest='api_server', action="store_true", default=False,
                 help="enable Web API server")
    g2 = OptionGroup(p, "Collector Agent flags",
                     "disable Collector Agents")
    g2.add_option(
        '--disable_ovs_agent', dest='disable_ovs_agent', action="store_true", default=False,
        help="disable Open vSwitch collector agent")
    g2.add_option(
        '--disable_linux_agent', dest='disable_linux_agent', action="store_true", default=False,
        help="disable linux OS collector agent")
    g2.add_option(
        '--disable_linuxnet_agent', dest='disable_linuxnet_agent', action="store_true", default=False,
        help="disable linux Network collector agent")
    g2.add_option(
        '--disable_linuxbridge_agent', dest='disable_linuxbridge_agent', action="store_true", default=False,
        help="disable linux bridge collector agent")
    g2.add_option(
        '--disable_linuxvlan_agent', dest='disable_linuxvlan_agent', action="store_true", default=False,
        help="disable linux vlan collector agent")
    g2.add_option(
        '--disable_libvirt_agent', dest='disable_libvirt_agent', action="store_true", default=False,
        help="disable libvirt collector agent")
    g2.add_option(
        '--disable_namespace_agent', dest='disable_namespace_agent', action="store_true", default=False,
        help="disable network namespace collector agent")
    g2.add_option(
        '--disable_sspvxlan_agent', dest='disable_sspvxlan_agent', action="store_true", default=False,
        help="disable SSP VXLAN collector agent")
    g2.add_option(
        '--disable_veth_agent', dest='disable_veth_agent', action="store_true", default=False,
        help="disable veth collector agent")
    p.add_option_group(g2)
    p.add_option(
        '-v', '--verbose', dest='verbose', action="store_true", default=False,
        help="verbose output.")
    (opts, args) = p.parse_args()
    util = CommonUtils()
    util.set_opts(opts)
    util.debug_out("init program options")

    if opts.format == "g" or opts.format == "graph-easy":
        formatter = GraphEasyFormatter(util)
    elif opts.format == "s" or opts.format == "springy":
        formatter = SpringyFormatter(util)
    elif opts.format == "r" or opts.format == "springy-raphael":
        formatter = SpringyRaphaelFormatter(util)
    elif opts.format == "t" or opts.format == "graphviz-tb":
        formatter = GraphvizTBFormatter(util)
    else:
        formatter = GraphvizTBFormatter()
    util.debug_out("formatter: " + str(formatter))

    if opts.import_file is not None:
        importer = JsonImporter(util, opts.import_file, formatter)
        importer.run()
        linkpairs = importer.get_linkpairs()
    else:
        collect_func = collect_linkpairs(opts, util, formatter)
        linkpairs = collect_func()

    if opts.export_stdout:
        exporter = JsonExporter(util, linkpairs)
        util.message_out(exporter.run())
        sys.exit(0)
    elif opts.api_server:
        collect_task = CollectTask(collect_func, 30)
        collect_task.start()
        api_server.util = util
        api_server.collect_func = collect_func
        api_server.start_app()
    else:
        print_linkpair_records(linkpairs)
