/*
    TiMidity++ -- MIDI to WAVE converter and player
    Copyright (C) 1999-2002 Masanao Izumo <mo@goice.co.jp>
    Copyright (C) 1995 Tuukka Toivonen <tt@cgs.fi>

    This program is free software; you can redistribute it and/or modify
    it under the terms timip_of the GNU General Public License as published by
    the Free Software Foundation; either version 2 timip_of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty timip_of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy timip_of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_CONFIG_H
#include "timip_config.h"
#endif /* HAVE_CONFIG_H */
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#ifndef NO_STRING_H
#include <string.h>
#else
#include <strings.h>
#endif

#include "timip_timidity.h"
#include "timip_common.h"
#include "timip_url.h"

/* #define DEBUG */

int timip_url_errno;
static struct URL_module *url_mod_list = NULL;
char *timip_user_mailaddr = NULL;
char *timip_url_user_agent = NULL;
int timip_url_newline_code = '\n';
char *timip_url_lib_version = URL_LIB_VERSION;
int timip_uudecode_unquote_html = 0;

void timip_url_add_module(struct URL_module *m)
{
    m->chain = url_mod_list;
    url_mod_list = m;
}

void timip_url_add_modules(struct URL_module *m, ...)
{
    va_list ap;
    struct URL_module *mod;

    if(m == NULL)
	return;
    timip_url_add_module(m);
    va_start(ap, m);
    while((mod = va_arg(ap, struct URL_module *)) != NULL)
	timip_url_add_module(mod);
}

static int url_init_nop(void)
{
    /* Do nothing any more */
    return 1;
}

int timip_url_check_type(char *s)
{
    struct URL_module *m;

    for(m = url_mod_list; m; m = m->chain)
	if(m->type != URL_none_t && m->name_check && m->name_check(s))
	    return m->type;
    return -1;
}

URL timip_url_open(char *s)
{
    struct URL_module *m;

    for(m = url_mod_list; m; m = m->chain)
    {
#ifdef DEBUG
	fprintf(stderr, "Check URL type=%d\n", m->type);
#endif /* DEBUG */
	if(m->type != URL_none_t && m->name_check && m->name_check(s))
	{
#ifdef DEBUG
	    fprintf(stderr, "open url (type=%d, name=%s)\n", m->type, s);
#endif /* DEBUG */
	    if(m->url_init != url_init_nop)
	    {
		if(m->url_init && m->url_init() < 0)
		    return NULL;
		m->url_init = url_init_nop;
	    }

	    timip_url_errno = URLERR_NONE;
	    errno = 0;
	    return m->timip_url_open(s);
	}
    }

    timip_url_errno = URLERR_NOURL;
    errno = ENOENT;
    return NULL;
}

long timip_url_read(URL url, void *buff, long n)
{
    if(n <= 0)
	return 0;
    timip_url_errno = URLERR_NONE;
    errno = 0;
    if(url->nread >= url->readlimit) {
        url->eof = 1;
	return 0;
    }
    if(url->nread + n > url->readlimit)
	n = (long)(url->readlimit - url->nread);
    n = url->timip_url_read(url, buff, n);
    if(n > 0)
	url->nread += n;
    return n;
}

long timip_url_safe_read(URL url, void *buff, long n)
{
    long i;
    if(n <= 0)
	return 0;

    do /* Ignore signal intruption */
    {
	errno = 0;
	i = timip_url_read(url, buff, n);
    } while(i == -1 && errno == EINTR);
#if 0
    /* Already done in timip_url_read!! */
    if(i > 0)
	url->nread += i;
#endif
    return i;
}

long timip_url_nread(URL url, void *buff, long n)
{
    long insize = 0;
    char *s = (char *)buff;

    do
    {
	long i;
	i = timip_url_safe_read(url, s + insize, n - insize);
	if(i <= 0)
	{
	    if(insize == 0)
		return i;
	    break;
	}
	insize += i;
    } while(insize < n);

    return insize;
}

char *timip_url_gets(URL url, char *buff, int n)
{
    if(url->nread >= url->readlimit)
	return NULL;

    if(url->timip_url_gets == NULL)
    {
	int maxlen, i, c;
	int newline = timip_url_newline_code;

	maxlen = n - 1;
	if(maxlen == 0)
	    *buff = '\0';
	if(maxlen <= 0)
	    return buff;
	i = 0;

	do
	{
	    if((c = url_getc(url)) == EOF)
		break;
	    buff[i++] = c;
	} while(c != newline && i < maxlen);

	if(i == 0)
	    return NULL; /* EOF */
	buff[i] = '\0';
	return buff;
    }

    timip_url_errno = URLERR_NONE;
    errno = 0;

    if(url->nread + n > url->readlimit)
	n = (long)(url->readlimit - url->nread) + 1;

    buff = url->timip_url_gets(url, buff, n);
    if(buff != NULL)
	url->nread += strlen(buff);
    return buff;
}

int timip_url_readline(URL url, char *buff, int n)
{
    int maxlen, i, c;

    maxlen = n - 1;
    if(maxlen == 0)
	*buff = '\0';
    if(maxlen <= 0)
	return 0;
    do
    {
	i = 0;
	do
	{
	    if((c = url_getc(url)) == EOF)
		break;
	    buff[i++] = c;
	} while(c != '\r' && c != '\n' && i < maxlen);
	if(i == 0)
	    return 0; /* EOF */
    } while(i == 1 && (c == '\r' || c == '\n'));

    if(c == '\r' || c == '\n')
	i--;
    buff[i] = '\0';
    return i;
}

int timip_url_fgetc(URL url)
{
    if(url->nread >= url->readlimit)
	return EOF;

    url->nread++;
    if(url->timip_url_fgetc == NULL)
    {
	unsigned char c;
	if(timip_url_read(url, &c, 1) <= 0)
	    return EOF;
	return (int)c;
    }
    timip_url_errno = URLERR_NONE;
    errno = 0;
    return url->timip_url_fgetc(url);
}

long timip_url_seek(URL url, long offset, int whence)
{
    long pos, savelimit;

    if(url->timip_url_seek == NULL)
    {
	if(whence == SEEK_CUR && offset >= 0)
	{
	    pos = timip_url_tell(url);
	    if(offset == 0)
		return pos;
	    savelimit = (long)url->readlimit;
	    url->readlimit = URL_MAX_READLIMIT;
	    timip_url_skip(url, offset);
	    url->readlimit = savelimit;
	    url->nread = 0;
	    return pos;
	}

	if(whence == SEEK_SET)
	{
	    pos = timip_url_tell(url);
	    if(pos != -1 && pos <= offset)
	    {
		if(pos == offset)
		    return pos;
		savelimit = (long)url->readlimit;
		url->readlimit = URL_MAX_READLIMIT;
		timip_url_skip(url, offset - pos);
		url->readlimit = savelimit;
		url->nread = 0;
		return pos;
	    }
	}

	timip_url_errno = errno = EPERM;
	return -1;
    }
    timip_url_errno = URLERR_NONE;
    errno = 0;
    url->nread = 0;
    return url->timip_url_seek(url, offset, whence);
}

long timip_url_tell(URL url)
{
    timip_url_errno = URLERR_NONE;
    errno = 0;
    if(url->timip_url_tell == NULL)
	return (long)url->nread;
    return url->timip_url_tell(url);
}

void timip_url_skip(URL url, long n)
{
    char tmp[BUFSIZ];

    if(url->timip_url_seek != NULL)
    {
	long savenread;

	savenread = (long)url->nread;
	if(savenread >= url->readlimit)
	    return;
	if(savenread + n > url->readlimit)
	    n = (long)(url->readlimit - savenread);
	if(url->timip_url_seek(url, n, SEEK_CUR) != -1)
	{
	    url->nread = savenread + n;
	    return;
	}
	url->nread = savenread;
    }

    while(n > 0)
    {
	long c;

	c = n;
	if(c > sizeof(tmp))
	    c = sizeof(tmp);
	c = timip_url_read(url, tmp, c);
	if(c <= 0)
	    break;
	n -= c;
    }
}

void timip_url_rewind(URL url)
{
    if(url->timip_url_seek != NULL)
	url->timip_url_seek(url, 0, SEEK_SET);
    url->nread = 0;
}

void timip_url_set_readlimit(URL url, long readlimit)
{
    if(readlimit < 0)
	url->readlimit = URL_MAX_READLIMIT;
    else
	url->readlimit = (unsigned long)readlimit;
    url->nread = 0;
}

URL timip_alloc_url(int size)
{
    URL url;
#ifdef HAVE_SAFE_MALLOC
    url = (URL)timip_safe_malloc(size);
    memset(url, 0, size);
#else
    url = (URL)malloc(size);
    if(url != NULL)
	memset(url, 0, size);
    else
	timip_url_errno = errno;
#endif /* HAVE_SAFE_MALLOC */

    url->nread = 0;
    url->readlimit = URL_MAX_READLIMIT;
    url->eof = 0;
    return url;
}

void timip_url_close(URL url)
{
    int save_errno = errno;

    if(url == NULL)
    {
	fprintf(stderr, "URL timip_stream structure is NULL?\n");
#ifdef ABORT_AT_FATAL
	abort();
#endif /* ABORT_AT_FATAL */
    }
    else if(url->timip_url_close == NULL)
    {
	fprintf(stderr, "URL Error: Already URL is closed (type=%d)\n",
		url->type);
#ifdef ABORT_AT_FATAL
	abort();
#endif /* ABORT_AT_FATAL */
    }
    else
    {
	url->timip_url_close(url);
#if 0
	url->timip_url_close = NULL;
#endif /* unix */
    }
    errno = save_errno;
}

#if defined(TILD_SCHEME_ENABLE)
#include <pwd.h>
char *timip_url_expand_home_dir(char *fname)
{
    static char path[BUFSIZ];
    char *dir;
    int dirlen;

    if(fname[0] != '~')
	return fname;

    if(IS_PATH_SEP(fname[1])) /* ~/... */
    {
	fname++;
	if((dir = getenv("HOME")) == NULL)
	    if((dir = getenv("home")) == NULL)
		return fname;
    }
    else /* ~user/... */
    {
	struct passwd *pw;
	int i;

	fname++;
	for(i = 0; i < sizeof(path) - 1 && fname[i] && !IS_PATH_SEP(fname[i]); i++)
	    path[i] = fname[i];
	path[i] = '\0';
	if((pw = getpwnam(path)) == NULL)
	    return fname - 1;
	fname += i;
	dir = pw->pw_dir;
    }
    dirlen = strlen(dir);
    strncpy(path, dir, sizeof(path) - 1);
    if(sizeof(path) > dirlen)
	strncat(path, fname, sizeof(path) - dirlen - 1);
    path[sizeof(path) - 1] = '\0';
    return path;
}
char *timip_url_unexpand_home_dir(char *fname)
{
    static char path[BUFSIZ];
    char *dir, *p;
    int dirlen;

    if(!IS_PATH_SEP(fname[0]))
	return fname;

    if((dir = getenv("HOME")) == NULL)
	if((dir = getenv("home")) == NULL)
	    return fname;
    dirlen = strlen(dir);
    if(dirlen == 0 || dirlen >= sizeof(path) - 2)
	return fname;
    memcpy(path, dir, dirlen);
    if(!IS_PATH_SEP(path[dirlen - 1]))
	path[dirlen++] = PATH_SEP;

#ifndef TIMIP___W32__
    if(strncmp(path, fname, dirlen) != 0)
#else
    if(strncasecmp(path, fname, dirlen) != 0)
#endif /* TIMIP___W32__ */
	return fname;

    path[0] = '~';
    path[1] = '/';
    p = fname + dirlen;
    if(strlen(p) >= sizeof(path) - 3)
	return fname;
    path[2] = '\0';
    strcat(path, p);
    return path;
}
#else
char *timip_url_expand_home_dir(char *fname)
{
    return fname;
}
char *timip_url_unexpand_home_dir(char *fname)
{
    return fname;
}
#endif

static char *url_strerror_txt[] =
{
    "",				/* URLERR_NONE */
    "Unknown URL",		/* URLERR_NOURL */
    "Operation not permitted",	/* URLERR_OPERM */
    "Can't open a URL",		/* URLERR_CANTOPEN */
    "Invalid URL form",		/* URLERR_IURLF */
    "URL too long",		/* URLERR_URLTOOLONG */
    "No mail address",		/* URLERR_NOMAILADDR */
    ""
};

char *timip_url_strerror(int no)
{
    if(no <= URLERR_NONE)
	return strerror(no);
    if(no >= URLERR_MAXNO)
	return "Internal error";
    return url_strerror_txt[no - URLERR_NONE];
}

void *timip_url_dump(URL url, long nbytes, long *read_size)
{
    long allocated, offset, read_len;
    char *buff;

    if(read_size != NULL)
      *read_size = 0;
    if(nbytes == 0)
	return NULL;
    if(nbytes >= 0)
    {
	buff = (void *)timip_safe_malloc(nbytes);
	if(nbytes == 0)
	    return buff;
	read_len = timip_url_nread(url, buff, nbytes);
	if(read_size != NULL)
	  *read_size = read_len;
	if(read_len <= 0)
	{
	    free(buff);
	    return NULL;
	}
	return buff;
    }

    allocated = 1024;
    buff = (char *)timip_safe_malloc(allocated);
    offset = 0;
    read_len = allocated;
    while((nbytes = timip_url_read(url, buff + offset, read_len)) > 0)
    {
	offset += nbytes;
	read_len -= nbytes;
	if(offset == allocated)
	{
	    read_len = allocated;
	    allocated *= 2;
	    buff = (char *)timip_safe_realloc(buff, allocated);
	}
    }
    if(offset == 0)
    {
	free(buff);
	return NULL;
    }
    if(read_size != NULL)
      *read_size = offset;
    return buff;
}
