/*
    TiMidity++ -- MIDI to WAVE converter and player
    Copyright (C) 1999-2002 Masanao Izumo <mo@goice.co.jp>
    Copyright (C) 1995 Tuukka Toivonen <tt@cgs.fi>

    This program is free software; you can redistribute it and/or modify
    it under the terms timip_of the GNU General Public License as published by
    the Free Software Foundation; either version 2 timip_of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty timip_of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy timip_of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_CONFIG_H
#include "timip_config.h"
#endif /* HAVE_CONFIG_H */
#include <stdio.h>
#include <stdlib.h>

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif /* HAVE_UNISTD_H */

#ifndef NO_STRING_H
#include <string.h>
#else
#include <strings.h>
#endif
#include <signal.h> /* for SIGALRM */

#include "timip_timidity.h"
#include "timip_common.h"
#include "timip_url.h"
#include "timip_net.h"

typedef struct _NewsConnection
{
    char *host;
    unsigned short port;
    FILE *fp;
    SOCKET fd;
    struct _NewsConnection *next;
    int status; /* -1, 0, 1 */
} NewsConnection;

#define NNTP_OK_ID '2'
#define ALARM_TIMEOUT 10
/* #define DEBUG */

static VOLATILE int timeout_flag = 1;
static NewsConnection *connection_cache;
static int connection_cache_flag = URL_NEWS_CONN_NO_CACHE;

typedef struct _URL_news
{
    char common[sizeof(struct _URL)];

    NewsConnection *news;
    int status; /* for detection '\r?\n.\r?\n'
		 *                1  2 34  5
		 */
    int eof;
} URL_news;

enum
{
    ARTICLE_STATUS_0,
    ARTICLE_STATUS_1,
    ARTICLE_STATUS_2,
    ARTICLE_STATUS_3,
    ARTICLE_STATUS_4
};

static int name_news_check(char *url_string);
static long url_news_read(URL url, void *buff, long n);
static int url_news_fgetc(URL url);
static void url_news_close(URL url);

struct URL_module timip_URL_module_news =
{
    URL_news_t,
    name_news_check,
    NULL,
    timip_url_news_open,
    NULL
};

static int name_news_check(char *s)
{
    if(strncmp(s, "news://", 7) == 0 && strchr(s, '@') != NULL)
	return 1;
    return 0;
}

/*ARGSUSED*/
static void timip_timeout(int sig)
{
    timeout_flag = 1;
}

static void close_news_server(NewsConnection *news)
{
    if(news->fd != (SOCKET)-1)
    {
	timip_socket_write(news->fd, "QUIT\r\n", 6);
	closesocket(news->fd);
    }
    if(news->fp != NULL)
	timip_socket_fclose(news->fp);
    free(news->host);
    news->status = -1;
}

static NewsConnection *open_news_server(char *host, unsigned short port)
{
    NewsConnection *p;
    char buff[512];

    for(p = connection_cache; p != NULL; p = p->next)
    {
	if(p->status == 0 && strcmp(p->host, host) == 0 && p->port == port)
	{
	    p->status = 1;
	    return p;
	}
    }
    for(p = connection_cache; p != NULL; p = p->next)
	if(p->status == -1)
	    break;
    if(p == NULL)
    {
	if((p = (NewsConnection *)timip_safe_malloc(sizeof(NewsConnection))) == NULL)
	    return NULL;
	p->next = connection_cache;
	connection_cache = p;
	p->status = -1;
    }

    if((p->host = timip_safe_strdup(host)) == NULL)
	return NULL;
    p->port = port;

#ifdef TIMIP___W32__
    timeout_flag = 0;
    p->fd = timip_open_socket(host, port);
#else
    timeout_flag = 0;
    signal(SIGALRM, timip_timeout);
    alarm(ALARM_TIMEOUT);
    p->fd = timip_open_socket(host, port);
    alarm(0);
    signal(SIGALRM, SIG_DFL);
#endif /* TIMIP___W32__ */

    if(p->fd == (SOCKET)-1)
    {
	int save_errno;

	VOLATILE_TOUCH(timeout_flag);
#ifdef ETIMEDOUT
	if(timeout_flag)
	    errno = ETIMEDOUT;
#endif /* ETIMEDOUT */
	if(errno)
	    timip_url_errno = errno;
	else
	{
	    timip_url_errno = URLERR_CANTOPEN;
	    errno = ENOENT;
	}
#ifdef DEBUG
	perror(host);
#endif /* DEBUG */

	save_errno = errno;
	free(p->host);
	errno = save_errno;

	return NULL;
    }

    if((p->fp = timip_socket_fdopen(p->fd, "rb")) == NULL)
    {
	timip_url_errno = errno;
	closesocket(p->fd);
	free(p->host);
	errno = timip_url_errno;
	return NULL;
    }

    buff[0] = '\0';
    if(timip_socket_fgets(buff, sizeof(buff), p->fp) == NULL)
    {
	timip_url_errno = errno;
	closesocket(p->fd);
	timip_socket_fclose(p->fp);
	free(p->host);
	errno = timip_url_errno;
	return NULL;
    }

#ifdef DEBUG
    fprintf(stderr, "Connect status: %s", buff);
#endif /* DEBUG */

    if(buff[0] != NNTP_OK_ID)
    {
	closesocket(p->fd);
	timip_socket_fclose(p->fp);
	free(p->host);
	timip_url_errno = URLERR_CANTOPEN;
	errno = ENOENT;
	return NULL;
    }
    p->status = 1;
    return p;
}

int timip_url_news_connection_cache(int flag)
{
    NewsConnection *p;
    int oldflag;

    oldflag = connection_cache_flag;

    switch(flag)
    {
      case URL_NEWS_CONN_NO_CACHE:
      case URL_NEWS_CONN_CACHE:
	connection_cache_flag = flag;
	break;
      case URL_NEWS_CLOSE_CACHE:
	for(p = connection_cache; p != NULL; p = p->next)
	    if(p->status == 0)
		close_news_server(p);
	break;
      case URL_NEWS_GET_FLAG:
	break;
    }
    return oldflag;
}

URL timip_url_news_open(char *name)
{
    URL_news *url;
    char *host, *p;
    unsigned short port;
    char buff[BUFSIZ], messageID[256];
    int check_timeout;
    int i;

#ifdef DEBUG
    fprintf(stderr, "url_news_open(%s)\n", name);
#endif /* DEBUG */

    url = (URL_news *)timip_alloc_url(sizeof(URL_news));
    if(url == NULL)
    {
	timip_url_errno = errno;
	return NULL;
    }

    /* common members */
    URLm(url, type)      = URL_news_t;
    URLm(url, timip_url_read)  = url_news_read;
    URLm(url, timip_url_gets)  = NULL;
    URLm(url, timip_url_fgetc) = url_news_fgetc;
    URLm(url, timip_url_seek)  = NULL;
    URLm(url, timip_url_tell)  = NULL;
    URLm(url, timip_url_close) = url_news_close;

    /* private members */
    url->news = NULL;
    url->status = ARTICLE_STATUS_2;
    url->eof = 0;

    if(strncmp(name, "news://", 7) == 0)
	name += 7;

    strncpy(buff, name, sizeof(buff) - 1);
    buff[sizeof(buff) - 1] = '\0';

    host = buff;
    for(p = host; *p && *p != ':' && *p != '/'; p++)
	;
    if(*p == ':')
    {
	*p++ = '\0'; /* terminate `host' string */
	port = atoi(p);
	p = strchr(p, '/');
	if(p == NULL)
	{
	    timip_url_errno = URLERR_CANTOPEN;
	    errno = ENOENT;
	    url_news_close((URL)url);
	    return NULL;
	}
    }
    else
	port = 119;
    *p++ = '\0'; /* terminate `host' string */
    if(*p == '<')
	p++;
    strncpy(messageID, p, sizeof(messageID) - 1);
    messageID[sizeof(messageID) - 1] = '\0';
    i = strlen(messageID);
    if(i > 0 && messageID[i - 1] == '>')
	messageID[i - 1] = '\0';

#ifdef DEBUG
    fprintf(stderr, "messageID: <%s>\n", messageID);
#endif /* DEBUG */

#ifdef DEBUG
    fprintf(stderr, "open(host=`%s', port=`%d')\n", host, port);
#endif /* DEBUG */

    if((url->news = open_news_server(host, port)) == NULL)
    {
	url_news_close((URL)url);
	return NULL;
    }

    check_timeout = 1;
  retry_article:

    sprintf(buff, "ARTICLE <%s>\r\n", messageID);

#ifdef DEBUG
    fprintf(stderr, "CMD> %s", buff);
#endif /* DEBUG */

    timip_socket_write(url->news->fd, buff, (long)strlen(buff));
    buff[0] = '\0';
    if(timip_socket_fgets(buff, sizeof(buff), url->news->fp) == NULL)
    {
	if(check_timeout)
	{
	    check_timeout = 0;
	    close_news_server(url->news);
	    if((url->news = open_news_server(host, port)) != NULL)
		goto retry_article;
	}
	url_news_close((URL)url);
	timip_url_errno = URLERR_CANTOPEN;
	errno = ENOENT;
	return NULL;
    }

#ifdef DEBUG
    fprintf(stderr, "CMD< %s", buff);
#endif /* DEBUG */

    if(buff[0] != NNTP_OK_ID)
    {
	if(check_timeout && strncmp(buff, "503", 3) == 0)
	{
	    check_timeout = 0;
	    close_news_server(url->news);
	    if((url->news = open_news_server(host, port)) != NULL)
		goto retry_article;
	}
	url_news_close((URL)url);
	timip_url_errno = errno = ENOENT;
	return NULL;
    }
    return (URL)url;
}

static void url_news_close(URL url)
{
    URL_news *urlp = (URL_news *)url;
    NewsConnection *news = urlp->news;
    int save_errno = errno;

    if(news != NULL)
    {
	if(connection_cache_flag == URL_NEWS_CONN_CACHE)
	    news->status = 0;
	else
	    close_news_server(news);
    }
    free(url);

    errno = save_errno;
}

static long url_news_read(URL url, void *buff, long size)
{
    char *p = (char *)buff;
    long n;
    int c;

    n = 0;
    while(n < size)
    {
	if((c = url_news_fgetc(url)) == EOF)
	    break;
	p[n++] = c;
    }
    return n;
}

static int url_news_fgetc(URL url)
{
    URL_news *urlp = (URL_news *)url;
    NewsConnection *news = urlp->news;
    int c;

    if(urlp->eof)
	return EOF;
    if((c = timip_socket_fgetc(news->fp)) == EOF)
    {
	urlp->eof = 1;
	return EOF;
    }

    switch(urlp->status)
    {
      case ARTICLE_STATUS_0:
	if(c == '\r')
	    urlp->status = ARTICLE_STATUS_1;
	else if(c == '\n')
	    urlp->status = ARTICLE_STATUS_2;
	break;

      case ARTICLE_STATUS_1:
	if(c == '\n')
	    urlp->status = ARTICLE_STATUS_2;
	else
	    urlp->status = ARTICLE_STATUS_0;
	break;

      case ARTICLE_STATUS_2:
	if(c == '.')
	    urlp->status = ARTICLE_STATUS_3;
	else
	    urlp->status = ARTICLE_STATUS_0;
	break;

      case ARTICLE_STATUS_3:
	if(c == '\r')
	    urlp->status = ARTICLE_STATUS_4;
	else if(c == '\n')
	    urlp->eof = 1;
	else
	    urlp->status = ARTICLE_STATUS_0;
	break;

      case ARTICLE_STATUS_4:
	if(c == '\n')
	    urlp->eof = 1;
	break;
    }

    return c;
}

#ifdef NEWS_MAIN
void main(int argc, char **argv)
{
    URL url;
    char buff[BUFSIZ];
    int c;

    if(argc != 2)
    {
	fprintf(stderr, "Usage: %s news-URL\n", argv[0]);
	exit(1);
    }
    if((url = timip_url_news_open(argv[1])) == NULL)
    {
	fprintf(stderr, "Can't open news group: %s\n", argv[1]);
	exit(1);
    }

#if NEWS_MAIN
    while((c = url_getc(url)) != EOF)
	putchar(c);
#else
    while((c = timip_url_read(url, buff, sizeof(buff))) > 0)
	write(1, buff, c);
#endif
    timip_url_close(url);
    exit(0);
}
#endif /* NEWS_MAIN */
