/*
 * Copyright (c) 2007, 2008 University of Tsukuba
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 3. Neither the name of the University of Tsukuba nor the names of its
 *    contributors may be used to endorse or promote products derived from
 *    this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */
/*
 * Copyright (c) 2010-2014 Yuichi Watanabe
 */

/* address translation for pass-through */

#include <core/cpu.h>
#include "assert.h"
#include "callrealmode.h"
#include "constants.h"
#include "convert.h"
#include "cpu_seg.h"
#include "current.h"
#include "gmm_pass.h"
#include "guest_bioshook.h"
#include "initfunc.h"
#include "io_io.h"
#include "mm.h"
#include "panic.h"
#include "printf.h"
#include "string.h"
#include "uefi.h"

u64 int0x15_code;
u64 int0x15_data;

static int
gmm_pass_get_mem_map(int index, phys_t *base, phys_t *len, u32 *type,
		     bool *restrict_access);

u64 gmm_pass_gp2hp_2m (u64 gp);
u32 gmm_pass_getforcemap (u32 n, u64 *base, u64 *len);

static struct gmm_func func = {
	.gp2hp = gmm_pass_gp2hp,
	.gp2hp_2m = gmm_pass_gp2hp_2m,
	.getforcemap = gmm_pass_getforcemap,
	.get_mem_map = gmm_pass_get_mem_map,
};

/* translate a guest-physical address to a host-physical address */
/* (for pass-through) */
/* fakerom can be NULL if it is not needed */
/* return value: Host-physical address */
/*   fakerom=true:  the page is part of the VMM. treat as write protected */
/*   fakerom=false: writable */
u64
gmm_pass_gp2hp (u64 gp, bool *fakerom)
{
	bool f;
	u64 r;

	if (phys_in_vmm (gp)) {
		r = phys_blank_page;
		f = true;
	} else {
		r = gp;
		f = false;
	}
	if (fakerom)
		*fakerom = f;
	return r;
}

u64
gmm_pass_gp2hp_2m (u64 gp)
{
	if (gp & PAGESIZE2M_MASK)
		return GMM_GP2HP_2M_FAIL;
	if (phys_in_vmm (gp))	/* VMM is 4MiB aligned */
		return GMM_GP2HP_2M_FAIL;
	return gp;
}

u32
gmm_pass_getforcemap (u32 n, u64 *base, u64 *len)
{
#ifdef MAP_UEFI_MMIO
	if (uefi_mmio_space) {
		*base = uefi_mmio_space[n].base;
		*len = uefi_mmio_space[n].npages << PAGESIZE_SHIFT;
		return *len > 0 ? n + 1 : 0;
	}
#endif
	*base = 0;
	*len = 0;
	return 0;
}

static int
gmm_pass_get_mem_map(int index, phys_t *base, phys_t *len, u32 *type,
		      bool *restrict_access)
{
	struct resource *resource;
	struct vm *vm = current->vm;

	if (index < 0 || index >= vm->resource_count) {
		return 0;
	}

	resource = vm->resource + index;

	*base = resource->start;
	*len = resource->end - resource->start + 1;
	*type = resource->data;

	if (restrict_access) {
		if (resource->data == MEM_TYPE_RESERVED &&
		    resource->parent == NULL) {
			/*
			 * VMM or other VMs use this resource.
			 */
			*restrict_access = true;
		} else {
			*restrict_access = false;
		}
	}
	return 1;
}

static void
alloc_mem_for_int0x15_hook(void)
{
	int len1, len2;

	if (uefi_booted)
		return;

	len1 = guest_int0x15_hook_end - guest_int0x15_hook;
	int0x15_code = alloc_realmodemem (len1);

	len2 = MAXNUM_OF_SYSMEMMAP * sizeof (struct e820_data);
	int0x15_data = alloc_realmodemem (len2);
}

static void
gmm_update_e801_fake(void)
{
	u32 limit32 = 0x100000; /* 1M */
	phys_t gphys, len;
	u32 type;
	int index = 0;

	while (gmm_pass_get_mem_map(index++, &gphys, &len, &type, NULL)) {
		if (type != MEM_TYPE_AVAILABLE) {
			continue;
		}
		if (gphys == 0x100000) {
			limit32 = gphys + len;
			break;
		}
	}
	update_e801_fake(limit32);
}

static void
install_int0x15_hook (void)
{
	u64 int0x15_base;
	u64 int0x15_vector_phys = 0x15 * 4;
	int count, len1, len2, i;
	struct e820_data *q;
	u64 b1, l1;
	u32 n, nn1;
	u32 t1;
	void *p;

	if (uefi_booted)
		return;

	gmm_update_e801_fake();

	len1 = guest_int0x15_hook_end - guest_int0x15_hook;

	count = 0;
	for (n = 0, nn1 = 1; nn1; n = nn1) {
		nn1 = getfakesysmemmap (n, &b1, &l1, &t1);
		count++;
	}
	len2 = count * sizeof (struct e820_data);

	if (int0x15_data > int0x15_code)
		int0x15_base = int0x15_code;
	else
		int0x15_base = int0x15_data;
	int0x15_base &= 0xFFFF0;

	/* save old interrupt vector */
	read_hphys_l (int0x15_vector_phys, &guest_int0x15_orig, 0);

	/* write parameters properly */
	guest_int0x15_e801_fake_ax = e801_fake_ax;
	guest_int0x15_e801_fake_bx = e801_fake_bx;
	guest_int0x15_e820_data_minus0x18 = int0x15_data - int0x15_base - 0x18;
	guest_int0x15_e820_end = int0x15_data + len2 - int0x15_base;

	/* copy the program code */
  	p = mapmem_hphys (int0x15_code, len1, MAPMEM_WRITE);
	memcpy (p, guest_int0x15_hook, len1);
	unmapmem (p, len1);

	/* create e820_data */
	q = mapmem_hphys (int0x15_data, len2, MAPMEM_WRITE);
	i = 0;
	for (n = 0, nn1 = 1; nn1; n = nn1) {
		nn1 = getfakesysmemmap (n, &b1, &l1, &t1);
		q[i].n = n;
		q[i].nn = nn1;
		q[i].base = b1;
		q[i].len = l1;
		q[i].type = t1;
		i++;
	}
	unmapmem (q, len2);

	/* set interrupt vector */
	write_hphys_l (int0x15_vector_phys, (int0x15_code - int0x15_base) |
		       (int0x15_base << 12), 0);
}

static void
gmm_pass_init (void)
{
	if (vm_get_id() != 0) {
		return;
	}

	memcpy ((void *)&current->gmm, (void *)&func, sizeof func);
	if (cpu_is_bsp()) {
		install_int0x15_hook();
	}
}

INITFUNC ("bsp0", alloc_mem_for_int0x15_hook);
INITFUNC ("pass0", gmm_pass_init);
