/*
 * From BitVisor 1.0.1 core/gmm_pass.c
 */
/*
 * Copyright (c) 2007, 2008 University of Tsukuba
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 3. Neither the name of the University of Tsukuba nor the names of its
 *    contributors may be used to endorse or promote products derived from
 *    this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */
/*
 * Copyright (c) 2010-2016 Yuichi Watanabe
 */
/*
 * Guest Memory Manager whose guest-address is translated into
 * host-address via a mapping table.
 */

#include <common/calc.h>
#include <core/assert.h>
#include <core/initfunc.h>
#include <core/mm.h>
#include <core/panic.h>
#include <core/pci.h>
#include <core/printf.h>
#include <core/rm.h>
#include <core/string.h>
#include <core/vmmerr.h>
#include "current.h"
#include "mm.h"
#include "gbiosloader.h"
#include "gmm.h"
#include "gmm_trans.h"

/* #define GMM_DEBUG */
#ifdef GMM_DEBUG
#define GMM_DBG(...)						\
	do {							\
		printf("GMMTRANS: " __VA_ARGS__);		\
	} while (0)
#else
#define GMM_DBG(...)
#endif

#define HIGH_MEM_START	0x100000000
#define UEFI_AREA_START	0xffe00000
#define UEFI_AREA_SIZE	0x00200000

static struct gmm_trans_mem_map*
gmm_trans_search_mem_map(phys_t gphys)
{
	struct gmm_trans_mem_map* mem_map;
	int low, mid, high;

	mem_map = current->vm->gmm_trans.mem_map;
	low = 0;
	high = current->vm->gmm_trans.mem_map_count - 1;

	while (low <= high) {
		mid = (low + high) / 2;
		if (gphys < mem_map[mid].gphys_start) {
			high = mid - 1;
		} else if (gphys > mem_map[mid].gphys_end) {
			low = mid + 1;
		} else {
			return mem_map + mid;
		}
	}
	return NULL;
}

static u64
gmm_trans_gp2hp (u64 gphys, bool *fakerom)
{
	struct gmm_trans_mem_map* mem_map;
	phys_t start;

	if (fakerom) {
		*fakerom = false;
	}

	mem_map = gmm_trans_search_mem_map(gphys);
	if (mem_map) {
		return mem_map->hphys.start
			+ (gphys - mem_map->gphys_start);
	}
	start = gphys & ~PAGESIZE_MASK;
	if (pci_assigned_mmio_addr(start, start + PAGESIZE_MASK)) {
		return gphys;
	}
	if (fakerom) {
		*fakerom = true;
	}

	return phys_blank_page;
}

static u64
gmm_trans_gp2hp_2m(u64 gphys)
{
	struct gmm_trans_mem_map* mem_map;
	u64 hphys;
	phys_t start;

	if (gphys & PAGESIZE2M_MASK) {
		return GMM_GP2HP_2M_FAIL;
	}
	mem_map = gmm_trans_search_mem_map(gphys);
	if (mem_map) {
		if ((gphys + PAGESIZE2M_MASK) <= mem_map->gphys_end) {
			hphys = mem_map->hphys.start
				+ (gphys - mem_map->gphys_start);
			if (hphys & PAGESIZE2M_MASK) {
				return GMM_GP2HP_2M_FAIL;
			} else {
				return hphys;
			}
		} else {
			return GMM_GP2HP_2M_FAIL;
		}
	}
	start = gphys & ~PAGESIZE2M_MASK;
	if (pci_assigned_mmio_addr(start, start + PAGESIZE2M_MASK)) {
		return gphys;
	}
	return GMM_GP2HP_2M_FAIL;
}

static int
gmm_trans_get_mem_map(int index, phys_t *base, phys_t *len, u32 *type,
		      bool *restrict_access)
{
	struct gmm_trans *gmm_trans;
	struct gmm_trans_mem_map *mem_map;

	gmm_trans = &current->vm->gmm_trans;

	if (index < 0 || index >= gmm_trans->mem_map_count) {
		return 0;
	}

	ASSERT(gmm_trans->mem_map);
	mem_map = gmm_trans->mem_map + index;

	*base = mem_map->gphys_start;
	*len = mem_map->gphys_end - mem_map->gphys_start + 1;
	*type = mem_map->hphys.data;

	if (restrict_access) {
		/*
		 * This resource is not used by VMM or other VMs.
		 */
		*restrict_access = false;
	}
	return 1;
}

static void
gmm_trans_clear_mem(void)
{
	struct gmm_trans *gmm_trans;
	struct gmm_trans_mem_map *mem_map;
	phys_t hphys;
	void *vaddr;
	int i;

	gmm_trans = &current->vm->gmm_trans;

	ASSERT(gmm_trans->mem_map);
	for (i = 0; i < gmm_trans->mem_map_count; i++) {
		mem_map = gmm_trans->mem_map + i;
		GMM_DBG("clear hphys 0x%llx 0x%llx gphys 0x%llx 0x%llx\n",
			mem_map->hphys.start,
			mem_map->hphys.end,
			mem_map->gphys_start,
			mem_map->gphys_end);
		for (hphys = ROUND_UP(mem_map->hphys.start, PAGESIZE);
		     hphys < ROUND_DOWN(mem_map->hphys.end, PAGESIZE);
		     hphys += PAGESIZE) {
			vaddr = mapmem_hphys(hphys, PAGESIZE, MAPMEM_WRITE);
			memset(vaddr, 0, PAGESIZE);
			unmapmem(vaddr, PAGESIZE);
		}
	}
}

static struct gmm_func gmm_trans_func = {
	.gp2hp			= gmm_trans_gp2hp,
	.gp2hp_2m		= gmm_trans_gp2hp_2m,
	.get_mem_map		= gmm_trans_get_mem_map,
	.clear_mem		= gmm_trans_clear_mem
};

static phys_t
gmm_trans_mem_size(void)
{
	phys_t size = 0;
	struct vm *vm = current->vm;
	struct resource *resource;
	int i;

	for (i = 0; i < vm->resource_count; i++) {
		resource = vm->resource + i;
		size += (resource->end - resource->start + 1);
	}
	return size;
}

static void
gmm_trans_init_mem_map(void)
{
	struct gmm_trans* gmm_trans;
	struct gmm_trans_mem_map *new;
	phys_t top_of_low_avail_mem
		= ROUND_DOWN(mm_top_of_low_avail_mem() + 1, PAGESIZE) - 1;
	struct vm *vm = current->vm;
	struct resource *parent;
	int count;
	int i;
	vmmerr_t err;
	phys_t high_mem_start = (guest_bios_type() == BIOS_TYPE_UEFI ?
				 UEFI_AREA_START : HIGH_MEM_START);

	gmm_trans = &current->vm->gmm_trans;
	gmm_trans->mem_map
		= alloc(sizeof(struct resource) * (vm->resource_count + 1));
	if (gmm_trans->mem_map == NULL) {
		panic("gmm_trans_init_mem_map: Failed to alloc mem_map");
	}

	if (guest_bios_type() == BIOS_TYPE_UEFI) {
		phys_t mem_size = gmm_trans_mem_size();
		if (mem_size < top_of_low_avail_mem + 1 + UEFI_AREA_SIZE) {
			top_of_low_avail_mem = mem_size - UEFI_AREA_SIZE - 1;
		}
	}

	for (i = 0, count = 0; i < vm->resource_count; i++, count++) {
		parent = vm->resource + i;
		new = gmm_trans->mem_map + count;
		rm_init_resource(&new->hphys, parent->start, parent->end,
				 parent->type, parent->data, parent->name);
		if (new == gmm_trans->mem_map) {
			new->gphys_start = 0;
		} else {
			if ((new - 1)->gphys_end == top_of_low_avail_mem) {
				new->gphys_start = high_mem_start;
			} else {
				new->gphys_start = (new - 1)->gphys_end + 1;
			}
		}
		new->gphys_end = new->gphys_start
			+ (new->hphys.end - new->hphys.start);

		if (new->gphys_start < top_of_low_avail_mem
		    && new->gphys_end > top_of_low_avail_mem) {
			GMM_DBG("Shrink gphys_end 0x%llx to 0x%llx\n",
				new->gphys_end, top_of_low_avail_mem);
			new->hphys.end
				-= (new->gphys_end - top_of_low_avail_mem);
			new->gphys_end = top_of_low_avail_mem;

			err = rm_insert_resource(parent, &new->hphys);
			if (err != VMMERR_SUCCESS) {
				panic("Failed to insert a resource. "
				      "err 0x%x, hphys %016llx-%016llx\n",
				      err, new->hphys.start, new->hphys.end);
			}
			GMM_DBG("%s gphys %016llx-%016llx "
				"hphys %016llx-%016llx\n",
				vm->name, new->gphys_start, new->gphys_end,
				new->hphys.start, new->hphys.end);

			count++;
			new = gmm_trans->mem_map + count;
			rm_init_resource(&new->hphys, (new -1 )->hphys.end + 1,
					 parent->end, parent->type,
					 parent->data, parent->name);
			new->hphys.parent = parent;
			new->gphys_start = high_mem_start;
			new->gphys_end = new->gphys_start
				+ (new->hphys.end - new->hphys.start);
		}
		err = rm_insert_resource(parent, &new->hphys);
		if (err != VMMERR_SUCCESS) {
			panic("Failed to insert a resource. "
			      "err 0x%x, hphys %016llx-%016llx\n",
			      err, new->hphys.start, new->hphys.end);
		}
		GMM_DBG("%s gphys %016llx-%016llx hphys %016llx-%016llx\n",
			vm->name, new->gphys_start, new->gphys_end,
			new->hphys.start, new->hphys.end);
	}

	gmm_trans->mem_map_count = count;
}

static void
gmm_trans_init (void)
{
	if (vm_get_id() == 0) {
		return;
	}
	memcpy((void *)&current->gmm, (void *)&gmm_trans_func,
	       sizeof(gmm_trans_func));

	if (current->vbsp) {
		gmm_trans_init_mem_map();
	}
}

INITFUNC ("pass0", gmm_trans_init);
