/*
    Text maid
    copyright (c) 1998-2004 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include <locale.h>
#include "tmaid.h"
#include "charlist.h"
#include "command.h"
#include "file.h"
#include "find.h"
#include "format.h"
#include "general.h"
#include "jump.h"
#include "print.h"
#include "replace.h"
#include "sigfile.h"
#include "sigmain.h"
#include "toolbar.h"
#include "version.h"
#include "misc/argument.h"
#include "misc/fileio.h"
#include "misc/license.h"
#include "misc/misc_accel.h"
#include "misc/misc_comm.h"
#include "misc/profile.h"


gboolean caret = FALSE;             /* ja:キャレットの形 */
gboolean ins = TRUE;                /* ja:TRUE:挿入,FALSE:上書き */
gboolean newfile = FALSE;           /* ja:TRUE:新規ファイル,FALSE:なし */
gboolean second = FALSE;            /* ja:TRUE:新規,FALSE:既存 */
gchar *history_files[10];           /* ja:ファイルの履歴 */
gchar *open_path = NULL;            /* ja:開くのパス */
gchar *save_path = NULL;            /* ja:名前を付けて保存のパス */
gchar *clipboard_text = NULL;       /* ja:クリップボードのテキスト */
gint def_width, def_height;         /* ja:デフォルトのウインドウのサイズ */
gint ftnum = 0;                     /* ja:ファイルタイプの数 */
gint history = 4;                   /* ja:ファイルの履歴 */
gint tcount = 0;                    /* ja:ファイルタイプのカウンタ */
guint timer_id = 0;                 /* ja:タイマID */
FileType *ftype = NULL;             /* ja:ファイルタイプ */
GdkAtom atom_targets;               /* ja:ターゲット */
GdkAtom atom_text;                  /* ja:テキスト */
GdkAtom atom_compound;              /* ja:コンパウンド */
GdkAtom atom_utf8;                  /* ja:UTF-8 */
GtkWidget *window;                  /* ja:ウインドウ */
GtkWidget *notebook;                /* ja:ノートブック */
GtkWidget *status;                  /* ja:ステータスウインドウ */
GtkWidget *label_cursor;            /* ja:カーソル座標 */
GtkPositionType n_pos = GTK_POS_TOP;/* ja:ノートブックの位置 */
GtkItemFactory *ifactory_menu;      /* ja:メニュー */
GtkItemFactory *ifactory_popup;     /* ja:ポップアップ */
PangoFontDescription *system_font = NULL;
GdkColor system_color[12] = {
    {0, 0x0000, 0x0000, 0x0000}, {0, 0xffff, 0xffff, 0xffff},
    {0, 0x8000, 0x8000, 0x8000}, {0, 0x8000, 0x8000, 0x8000},
    {0, 0x8000, 0x8000, 0x8000}, {0, 0x8000, 0x8000, 0x8000},
    {0, 0x8000, 0x8000, 0x8000}, {0, 0xa000, 0xa000, 0xa000},
    {0, 0xffff, 0xffff, 0xffff}, {0, 0x0000, 0x0000, 0x8000},
    {0, 0x0000, 0x0000, 0x0000}, {0, 0xffff, 0xffff, 0xffff}};
GtkTargetEntry targets[4] = {       /* ja:Text maidが提供可能なセレクション */
    {"STRING",        0, TARGET_STRING},
    {"TEXT",          0, TARGET_TEXT}, 
    {"COMPOUND_TEXT", 0, TARGET_COMPOUND_TEXT},
    {"UTF8_STRING",   0, TARGET_UTF8}};


/******************************************************************************
*                                                                             *
* ja:メイン関数                                                               *
*                                                                             *
******************************************************************************/
static gchar *
menu_translate (const gchar *path,
                gpointer     func_data)
{
 return (gchar *)_(path);
}


int
main (int   argc,
      char *argv[])
{
  gboolean *arg_files, license = FALSE;
  gchar *section, *key, *tmp, *utf8str;
  const gchar *charset;
  gint i, count = 0, init_line = 0;
  Profile *profile;
  TmaidWindow *tmaid;
  GObject *accel;
  GdkBitmap *mask;
  GdkPixmap *pixmap;
  GtkAccelGroup *accel_group;
  GtkStyle *style;
  GtkWidget *menu_bar, *menu_shell, *menu_item;
  GtkWidget *handle, *label, *table, *toolbar, *vbox, *frame;
  GtkItemFactoryEntry menu_items[] = {
{N_("/_File"),                       NULL,         NULL,            0,"<Branch>"},
{N_("/File/_New"),                   "<control>N", command_new,     0,"<Item>"},
{N_("/File/_Open..."),               "<control>O", command_open,    0,"<Item>"},
{N_("/File/_Close"),                 "<control>W", command_close,   0,"<Item>"},
{N_("/File/_Save"),                  "<control>S", command_save,    0,"<Item>"},
{N_("/File/Save _As..."),            NULL,         command_saveas,  0,"<Item>"},
{N_("/File/_Reload..."),             "<control>R", command_reload,  0,"<Item>"},
{N_("/File/Sep1"),                   NULL,         NULL,            0,"<Separator>"},
{N_("/File/_Print..."),              "<control>P", command_print,   0,"<Item>"},
{N_("/File/Sep2"),                   NULL,         NULL,            0,"<Separator>"},
{N_("/File/Propert_ies..."),         "<alt>Return",command_property,0,"<Item>"},
{N_("/File/Sep3"),                   NULL,         NULL,            0,"<Separator>"},
{N_("/File/E_xit"),                  "<control>Q", command_exit,    0,"<Item>"},
{N_("/_Edit"),                       NULL,         NULL,            0,"<Branch>"},
{N_("/Edit/_Undo"),                  "<control>Z", command_undo,    0,"<Item>"},
{N_("/Edit/_Redo"),                  "F4",         command_redo,    0,"<Item>"},
{N_("/Edit/Sep1"),                   NULL,         NULL,            0,"<Separator>"},
{N_("/Edit/Cu_t"),                   "<control>X", command_cut,     0,"<Item>"},
{N_("/Edit/_Copy"),                  "<control>C", command_copy,    0,"<Item>"},
{N_("/Edit/_Paste"),                 "<control>V", command_paste,   0,"<Item>"},
{N_("/Edit/De_lete"),                NULL,         command_delete,  0,"<Item>"},
{N_("/Edit/Sep2"),                   NULL,         NULL,            0,"<Separator>"},
{N_("/Edit/Return at Right _Margin"),NULL,         command_margin,  0,"<Item>"},
{N_("/Edit/Convert Ta_b to Space"),  NULL,         command_tab,     0,"<Item>"},
{N_("/Edit/Sep3"),                   NULL,         NULL,            0,"<Separator>"},
{N_("/Edit/_Character by _Value..."),NULL,         command_valchr,  0,"<Item>"},
{N_("/Edit/_Jump..."),               "<control>J", command_jump,    0,"<Item>"},
{N_("/Edit/Sep4"),                   NULL,         NULL,            0,"<Separator>"},
{N_("/Edit/_All"),                   "<control>A", command_all,     0,"<Item>"},
{N_("/_Search"),                     NULL,         NULL,            0,"<Branch>"},
{N_("/Search/_Find..."),             "<control>F", command_find,    0,"<Item>"},
{N_("/Search/_Next"),                "F3",         command_next,    0,"<Item>"},
{N_("/Search/_Previous"),            "<shift>F3",  command_next,    1,"<Item>"},
{N_("/Search/_Replace..."),          "<control>H", command_replace, 0,"<Item>"},
{N_("/_Option"),                     NULL,         NULL,            0,"<Branch>"},
{N_("/Option/_Configuration..."),    NULL,         command_conf,    0,"<Item>"},
{N_("/Option/File _Type..."),        NULL,         format_dialog,   0,"<Item>"},
{N_("/_Window"),                     NULL,         NULL,            0,"<Branch>"},
{N_("/_Help"),                       NULL,         NULL,            0,"<Branch>"},
{N_("/Help/_About..."),              NULL,         version_dialog,  0,"<Item>"}};
    GtkItemFactoryEntry popup_items[] = {
{N_("/_Undo"),                       "<control>Z", command_undo,    0,"<Item>"},
{"/Sep1",                            NULL,         NULL,            0,"<Separator>"},
{N_("/Cu_t"),                        "<control>X", command_cut,     0,"<Item>"},
{N_("/_Copy"),                       "<control>C", command_copy,    0,"<Item>"},
{N_("/_Paste"),                      "<control>V", command_paste,   0,"<Item>"},
{N_("/De_lete"),                     NULL,         command_delete,  0,"<Item>"},
{"/Sep2",                            NULL,         NULL,            0,"<Separator>"},
{N_("/_All"),                        "<control>A", command_all,     0,"<Item>"},
{"/Sep3",                            NULL,         NULL,            0,"<Separator>"},
{N_("/_Input Methods"),              NULL,         NULL,            0,"<Item>"}};
  /* en:Drag&Drop */
  GtkTargetEntry drag_types[1] = {{"text/uri-list", 0, 0}};

  /* ja:初期化 */
  setlocale (LC_ALL, "");
#ifdef LOCALEDIR
  bindtextdomain (PACKAGE, LOCALEDIR);
#else /* not LOCALEDIR */
  {
    gchar *dirname, *path, *locale;

    path = fileio_get_full_path (argv[0]);
    dirname = g_path_get_dirname (path);
    g_free (path);
    locale = g_strconcat (dirname, G_DIR_SEPARATOR_S, "locale", NULL);
    g_free (dirname);
    bindtextdomain (PACKAGE, locale);
    g_free (locale);
  }
#endif /* not LOCALEDIR */
  bind_textdomain_codeset (PACKAGE, "UTF-8");
  textdomain (PACKAGE);
  gtk_set_locale ();
  gtk_init (&argc, &argv);

  def_width = gdk_screen_width () * 2 / 3;
  def_height = gdk_screen_height () * 2 / 3;
  g_memset (history_files, 0, sizeof (gchar *) * 10);
  /* ja:レジストリから読み出す */
  profile = profile_open (NULL);
  if (profile)
    {
      if (!profile_get_value (profile, "tmaid", "license",
            (gpointer)&license, sizeof (gboolean), PROFILE_VALUE_TYPE_BOOL))
        license = FALSE;
      if (!profile_get_value(profile, "tmaid", "ftnum",
            (gpointer)&ftnum, sizeof (gint), PROFILE_VALUE_TYPE_INT))
        ftnum = 0;
      if (!profile_get_value (profile, "tmaid","history",
            (gpointer)&history, sizeof (gint), PROFILE_VALUE_TYPE_INT))
        history = 4;
      if (!profile_get_value (profile, "tmaid", "newfile",
            (gpointer)&newfile, sizeof (gboolean), PROFILE_VALUE_TYPE_BOOL))
        newfile = FALSE;
      if (!profile_get_value (profile, "tmaid", "second",
            (gpointer)&second, sizeof (gboolean), PROFILE_VALUE_TYPE_BOOL))
        second = FALSE;
      if (!profile_get_value (profile, "tmaid", "def_width",
            (gpointer)&def_width, sizeof (gint), PROFILE_VALUE_TYPE_INT))
        def_width = gdk_screen_width () * 2 / 3;
      if (!profile_get_value (profile, "tmaid", "def_height",
            (gpointer)&def_height, sizeof (gint), PROFILE_VALUE_TYPE_INT))
        def_height = gdk_screen_height () * 2 / 3;
      if (!profile_get_value (profile, "tmaid", "n_pos",
            (gpointer)&n_pos, sizeof (gint), PROFILE_VALUE_TYPE_INT))
        n_pos = GTK_POS_TOP;
      utf8str = profile_get_string (profile, "tmaid", "openpath");
      open_path = utf8str
                ? g_filename_from_utf8 (utf8str, -1, NULL, NULL, NULL) : NULL;
      g_free (utf8str);
      utf8str = profile_get_string (profile, "tmaid", "savepath");
      save_path = utf8str
                ? g_filename_from_utf8 (utf8str, -1, NULL, NULL, NULL) : NULL;
      g_free (utf8str);
      /* ja:ファイルの履歴 */
      for (i = 0; i < history; i++)
        {
          key = g_strdup_printf ("history%d", i);
          utf8str = profile_get_string (profile, "tmaid", key);
          history_files[i] = utf8str
                ? g_filename_from_utf8 (utf8str, -1, NULL, NULL, NULL) : NULL;
          g_free (utf8str);
          g_free (key);
          if (!history_files[i])
            break;
          if (*history_files[i] == '\0')
            {
              g_free (history_files[i]);
              history_files[i] = NULL;
              break;
            }
        }
      /* ja:検索 */
      if (!profile_get_value (profile, "tmaid", "find_arrow",
            (gpointer)&find_arrow, sizeof (gboolean), PROFILE_VALUE_TYPE_BOOL))
        find_arrow = TRUE;
      if (!profile_get_value (profile, "tmaid", "find_ignorecase",
            (gpointer)&find_ignorecase, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL))
        find_ignorecase = FALSE;
      if (!profile_get_value (profile, "tmaid", "find_num",
            (gpointer)&find_num, sizeof (gint), PROFILE_VALUE_TYPE_INT))
        find_num = 0;
      for (i = 0; i < find_num; i++)
        {
          key = g_strdup_printf ("find%02d", i);
          find_text[i] = profile_get_string (profile, "tmaid", key);
          g_free (key);
          if (!find_text[i])
            break;
        }
      find_num = i;
      /* ja:置換 */
      if (!profile_get_value (profile, "tmaid", "replace_arrow",
            (gpointer)&replace_arrow, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL))
        replace_arrow = TRUE;
      if (!profile_get_value (profile, "tmaid", "replace_ignorecase",
            (gpointer)&replace_ignorecase, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL))
        replace_ignorecase = FALSE;
      if (!profile_get_value (profile, "tmaid", "replace_num",
            (gpointer)&replace_num, sizeof (gint), PROFILE_VALUE_TYPE_INT))
        replace_num = 0;
      for (i = 0; i < replace_num; i++)
        {
          key = g_strdup_printf ("replace%02d", i);
          replace_text[i] = profile_get_string (profile, "tmaid", key);
          g_free (key);
          if (!replace_text[i])
            break;
        }
      replace_num = i;
      /* ja:印刷 */
#ifdef G_OS_UNIX
      if (!profile_get_value (profile, "printer", "mode",
            (gpointer)&printer.mode, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL))
        printer.mode = TRUE;
      utf8str = profile_get_string (profile, "printer", "printer");
      if (utf8str)
        {
          printer.printer
                        = g_filename_from_utf8 (utf8str, -1, NULL, NULL, NULL);
          g_free (utf8str);
        }
      if (!printer.printer)
        printer.printer = g_strdup ("lpr");
#endif /* G_OS_UNIX */
#if defined G_OS_UNIX || defined G_OS_WIN32
      if (!profile_get_value (profile, "printer", "copies",
            (gpointer)&printer.copies, sizeof (gint), PROFILE_VALUE_TYPE_INT))
        printer.copies = 1;
#endif /* G_OS_UNIX or G_OS_WIN32 */
#ifndef G_OS_WIN32
      utf8str = profile_get_string (profile, "printer", "file");
      if (utf8str)
        {
          printer.file = g_filename_from_utf8 (utf8str, -1, NULL, NULL, NULL);
          g_free (utf8str);
        }
      if (!printer.file)
        printer.file = g_strdup ("tmaid.ps");
      if (!profile_get_value (profile, "printer", "width",
            (gpointer)&printer.width, sizeof (gint), PROFILE_VALUE_TYPE_INT))
        printer.width = 594;
      if (!profile_get_value (profile, "printer", "height",
            (gpointer)&printer.height, sizeof (gint), PROFILE_VALUE_TYPE_INT))
        printer.height = 840;
      if (!profile_get_value (profile, "printer", "left",
            (gpointer)&printer.left, sizeof (gint), PROFILE_VALUE_TYPE_INT))
        printer.left = 20;
      if (!profile_get_value (profile, "printer", "top",
            (gpointer)&printer.top, sizeof (gint), PROFILE_VALUE_TYPE_INT))
        printer.top = 20;
      if (!profile_get_value (profile, "printer", "right",
            (gpointer)&printer.right, sizeof (gint), PROFILE_VALUE_TYPE_INT))
        printer.right = 20;
      if (!profile_get_value (profile, "printer", "bottom",
            (gpointer)&printer.bottom, sizeof (gint), PROFILE_VALUE_TYPE_INT))
        printer.bottom = 20;
#endif /* not G_OS_WIN32 */
      /* ja:ファイルタイプ */
      if (ftnum > 0)
        ftype = g_malloc (ftnum * sizeof (FileType));
      for (i = 0; i < ftnum; i++)
        {
          section = g_strdup_printf ("filetype%04d", i);
          ftype[i].ft_id = tcount++;
          ftype[i].text = profile_get_string (profile, section, "text");
          if (!ftype[i].text)
            ftype[i].text = g_strdup("UNKNOW");
          ftype[i].ext = profile_get_string (profile, section, "ext");
          if (!ftype[i].ext)
            ftype[i].ext = g_strdup ("*");
          if (!profile_get_value (profile, section, "crlf",
                (gpointer)&ftype[i].crlf, sizeof (gint),
                                                    PROFILE_VALUE_TYPE_INT))
            ftype[i].crlf = 3;
          if (!profile_get_value (profile, section, "margin",
                (gpointer)&ftype[i]. margin, sizeof (gint),
                                                    PROFILE_VALUE_TYPE_INT))
            ftype[i].margin=80;
          if (!profile_get_value(profile,section,"tab",
                (gpointer)&ftype[i].tab, sizeof (gint),
                                                    PROFILE_VALUE_TYPE_INT))
            ftype[i].tab=8;
          if (!profile_get_value (profile, section, "writemode",
                (gpointer)&ftype[i].writemode, sizeof (gint),
                                                    PROFILE_VALUE_TYPE_INT))
            ftype[i].writemode = 0;
          if (!profile_get_value (profile, section, "associate",
                (gpointer)&ftype[i].associate, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL))
            ftype[i].associate = TRUE;
          if (!profile_get_value (profile, section, "autoindent",
                (gpointer)&ftype[i].autoindent, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL))
            ftype[i].autoindent = FALSE;
          if (!profile_get_value (profile, section, "code",
                (gpointer)&ftype[i].code, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL))
            ftype[i].code = FALSE;
          if (!profile_get_value(profile,section,"eof",
                (gpointer)&ftype[i].eof, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL))
            ftype[i].eof = FALSE;
          if (!profile_get_value (profile, section, "hexcase",
                (gpointer)&ftype[i].hexcase, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL))
            ftype[i].hexcase = FALSE;
          if (!profile_get_value (profile, section, "html",
                (gpointer)&ftype[i].html, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL))
            ftype[i].html = FALSE;
          if (!profile_get_value (profile, section, "esc",
                (gpointer)&ftype[i].esc, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL))
            ftype[i].esc = FALSE;
          if (!profile_get_value (profile, section, "limit",
                (gpointer)&ftype[i].limit, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL))
            ftype[i].limit = FALSE;
          if (!profile_get_value(profile,section,"overwrite",
                (gpointer)&ftype[i].overwrite, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL))
            ftype[i].overwrite = FALSE;
          if (!profile_get_value(profile,section,"space",
                (gpointer)&ftype[i].space, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL))
            ftype[i].space = FALSE;
          if (!profile_get_value(profile,section,"syscol",
                (gpointer)&ftype[i].syscol, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL))
            ftype[i].syscol = TRUE;
          if (!profile_get_value (profile, section, "tabconv",
                (gpointer)&ftype[i].tabconv, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL))
            ftype[i].tabconv = FALSE;
          if (!profile_get_value (profile, section, "gline",
                (gpointer)&ftype[i].gline, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL))
            ftype[i].gline = FALSE;
          if (!profile_get_value(profile,section,"mline",
                (gpointer)&ftype[i].mline, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL))
            ftype[i].mline = FALSE;
          if (!profile_get_value(profile,section,"nline",
                (gpointer)&ftype[i].nline, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL))
            ftype[i].nline = FALSE;
          if (!profile_get_value (profile, section, "uline",
                (gpointer)&ftype[i].uline, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL))
            ftype[i].uline = FALSE;
          if (!profile_get_value (profile, section, "vline",
                (gpointer)&ftype[i].vline, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL))
            ftype[i].vline = FALSE;
          /* ja:キャラクターセット */
          ftype[i].charset = profile_get_string (profile, section, "charset");
          if (ftype[i].charset)
            {
              tmp = charlist_strip (ftype[i].charset);
              g_free (ftype[i].charset);
              ftype[i].charset = tmp;
            }
          if (!ftype[i].charset)
            ftype[i].charset = g_get_charset (&charset)
                                        ? g_strdup_printf ("%s,UTF-8", charset)
                                                        : g_strdup ("UTF-8");
          /* ja:フォント */
          ftype[i].font_name = profile_get_string (profile, section, "font");
          /* ja:色 */
          if (!profile_get_value (profile, section, "color",
                    (gpointer)&ftype[i].color, sizeof (GdkColor) * 12,
                                                    PROFILE_VALUE_TYPE_ARRAY))
            g_memmove(ftype[i].color,system_color,sizeof(GdkColor)*12);
          g_free (section);
        }
      profile_close(profile);
    }
  else
    {
#ifdef G_OS_UNIX
      printer.mode = TRUE;
      printer.printer = g_strdup ("lpr");
#endif /* G_OS_UNIX */
#ifndef G_OS_WIN32
      printer.file = g_strdup ("tmaid.ps");
#if defined G_OS_UNIX || defined G_OS_WIN32
      printer.copies = 1;
#endif /* G_OS_UNIX or G_OS_WIN32 */
      printer.width = 594;
      printer.height = 840;
      printer.left = printer.top = printer.right = printer.bottom = 20;
#endif /* not G_OS_WIN32 */
    }
  if (!license)
    license = license_dialog ();
  if (!license)
    gtk_exit (0);

  /* ja:コマンドラインの解析 */
  {
    gint array[2], def[2];
    ArgumentBool arg_n, arg_x;
    ArgumentValue arg_h, arg_l;
    ArgumentArray arg_s;
    ArgumentExit arg_v;
    ArgumentHelp arg_q;
    ArgumentList arg_opts[] = {
{"history", 'h',  "NUM",          N_("History"),        ARGUMENT_TYPE_VALUE, &arg_h},
{"line",    'l',  "NUM",          N_("Caret Position"), ARGUMENT_TYPE_VALUE, &arg_l},
{"new",     'n',  "ON/OFF",       N_("New File"),       ARGUMENT_TYPE_BOOL,  &arg_n},
{"size",    's',  "WIDTH,HEIGHT", N_("Window Size"),    ARGUMENT_TYPE_ARRAY, &arg_s},
{"second",  'x',  "ON/OFF",       N_("Open Window"),    ARGUMENT_TYPE_BOOL,  &arg_x},
{"version", 'v',  NULL,           N_("About"),          ARGUMENT_TYPE_EXIT,  &arg_v},
{"help",    '?',  NULL,           N_("Help"),           ARGUMENT_TYPE_HELP,  &arg_q},
{NULL,      '\0', NULL,           NULL,                 0,                   NULL}};

    /* ja:ファイルの履歴 */
    arg_h.val = history;
    arg_h.def = 4;
    /* ja:初期のキャレット位置 */
    arg_l.val = init_line;
    arg_l.def = 0;
    /* ja:新規ファイル */
    arg_n.flag = newfile;
    arg_n.def = FALSE;
    /* ja:ウインドウサイズ */
    array[0] = def_width;
    array[1] = def_height;
    def[0] = gdk_screen_width ()  * 2 / 3;
    def[1] = gdk_screen_height () * 2 / 3;
    arg_s.leng = 2;
    arg_s.array = array;
    arg_s.def = def;
    /* ja:新規ウインドウ */
    arg_x.flag = second;
    arg_x.def = FALSE;
    /* ja:バージョン */
    arg_v.str =
TMAID_VERSION_NOTICE"\n"
TMAID_COPYRIGHT_NOTICE"\n"
"This is free software; see the source for copying conditions.  There is NO\n"
"warranty; not even for MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.\n";
    /* ja:ヘルプ */
    arg_q.str = TMAID_COPYRIGHT_NOTICE"\n\n";
    arg_files = arg_analyse (argv, arg_opts);
    if (0 <= arg_h.val && arg_h.val <= 10)
      history = arg_h.val;
    if (arg_l.val >= 0)
      init_line = arg_l.val;
    def_width = array[0];
    def_height = array[1];
    second = arg_x.flag;
  }

  /* ja:ファイルタイプ */
  if (ftnum <= 0)
    {
      ftnum = 1;
      ftype = g_malloc0 (ftnum * sizeof (FileType));
      ftype[0].text = g_strdup (_("All Files"));
      ftype[0].ext = g_strdup ("*");
      ftype[0].crlf = 3;
      ftype[0].margin = 80;
      ftype[0].tab = 8;
      ftype[0].associate = TRUE;
      ftype[0].syscol = TRUE;
      /* ja:キャラクターセット */
      ftype[0].charset = !g_get_charset (&charset) && charset
                ? g_strdup_printf ("%s,UTF-8", charset) : g_strdup ("UTF-8");
      /* ja:色 */
      g_memmove (ftype[0].color, system_color, sizeof (GdkColor) * 12);
    }

  /* ja:メインウインドウ */
  window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title (GTK_WINDOW (window), "Text maid");
  gtk_window_set_default_size (GTK_WINDOW (window), def_width, def_height);
  g_signal_connect (G_OBJECT(window), "delete-event",
                                            G_CALLBACK (signal_delete), NULL);
  g_signal_connect (G_OBJECT (window), "destroy",
                                    G_CALLBACK (signal_destroy_window), NULL);
  gtk_widget_realize (window);
  /* en:Drag and Drop */
  gtk_drag_dest_set (window, GTK_DEST_DEFAULT_MOTION
                        | GTK_DEST_DEFAULT_HIGHLIGHT | GTK_DEST_DEFAULT_DROP,
                                            drag_types, 1, GDK_ACTION_COPY);
  g_signal_connect (G_OBJECT (window), "drag-data-received",
                                G_CALLBACK (signal_drag_data_received), NULL);
  /* ja:クリップボード */
  atom_targets = gdk_atom_intern ("TARGETS", FALSE);
  atom_compound = gdk_atom_intern ("COMPOUND_TEXT", FALSE);
  atom_text = gdk_atom_intern ("TEXT", FALSE);
  atom_utf8 = gdk_atom_intern ("UTF8_STRING", FALSE);
  gtk_selection_add_targets (window, GDK_SELECTION_PRIMARY,
                        targets, sizeof (targets) / sizeof (GtkTargetEntry));
  gtk_selection_add_targets (window, GDK_SELECTION_CLIPBOARD,
                        targets, sizeof (targets) / sizeof (GtkTargetEntry));
  g_signal_connect (G_OBJECT (window), "selection-clear-event",
                                    G_CALLBACK (signal_selection_clear), NULL);
  g_signal_connect (G_OBJECT (window), "selection-get",
                                    G_CALLBACK (signal_selection_get), NULL);
  g_signal_connect (G_OBJECT (window), "selection-received",
                                G_CALLBACK (signal_selection_received), NULL);
  /* ja:ノートブック */
  notebook = gtk_notebook_new ();
  gtk_notebook_set_scrollable (GTK_NOTEBOOK (notebook), TRUE);
  gtk_notebook_set_tab_pos (GTK_NOTEBOOK (notebook), n_pos);
  g_signal_connect (G_OBJECT (notebook), "switch-page",
                                G_CALLBACK (signal_switch_page), NULL);
  g_signal_connect (G_OBJECT (notebook), "destroy",
                                G_CALLBACK (signal_destroy_notebook), NULL);
  /* ja:ポップアップ */
  accel_group = gtk_accel_group_new ();
  ifactory_popup = gtk_item_factory_new (GTK_TYPE_MENU, "<main>", accel_group);
  gtk_item_factory_set_translate_func (ifactory_popup, menu_translate,
                                                                NULL, NULL);
  gtk_item_factory_create_items (ifactory_popup,
                        sizeof (popup_items) / sizeof (GtkItemFactoryEntry),
                                                            popup_items, NULL);
  /* ja:メニュー */
  accel_group = gtk_accel_group_new ();
  ifactory_menu = gtk_item_factory_new (GTK_TYPE_MENU_BAR,
                                                        "<main>", accel_group);
  gtk_item_factory_set_translate_func (ifactory_menu, menu_translate,
                                                                NULL, NULL);
  gtk_item_factory_create_items (ifactory_menu,
                        sizeof (menu_items) / sizeof (GtkItemFactoryEntry),
                                                            menu_items, NULL);
  menu_bar = gtk_item_factory_get_widget (ifactory_menu, "<main>");
  /* ja:アクセルグループ */
  accel = misc_accel_new ();
  misc_accel_set_window (MISC_ACCEL (accel), GTK_WINDOW (window));
  misc_accel_add_accel_group (MISC_ACCEL (accel), accel_group);
  /* ja:ツールバー */
  toolbar = gtk_toolbar_new ();
  gtk_toolbar_set_orientation (GTK_TOOLBAR (toolbar),
                                                GTK_ORIENTATION_HORIZONTAL);
  gtk_toolbar_set_style (GTK_TOOLBAR (toolbar), GTK_TOOLBAR_ICONS);
  for (i = 0; toolbar_items[i].name != (gpointer)-1; i++)
    if (toolbar_items[i].name)
      {
        pixmap = gdk_pixmap_create_from_xpm_d (window->window,
                                            &mask, NULL, toolbar_items[i].xpm);
        toolbar_items[i].item = gtk_toolbar_append_item
                        (GTK_TOOLBAR (toolbar),
                        NULL, _(toolbar_items[i].name), NULL,
                        gtk_pixmap_new (pixmap, mask),
                        GTK_SIGNAL_FUNC (toolbar_items[i].callback),
                        toolbar_items[i].user_data);
      }
    else
      {
        gtk_toolbar_append_space(GTK_TOOLBAR(toolbar));
      }
  /* ja:テーブル */
  table = gtk_table_new (1, 5, FALSE);
  /* ja:ステータスバー */
  status = gtk_statusbar_new ();
  /* ja:カーソル座標 */
  label_cursor = gtk_label_new ("");
  gtk_label_set_justify (GTK_LABEL (label_cursor), GTK_JUSTIFY_CENTER);
  frame = gtk_frame_new (NULL);
  gtk_container_add( GTK_CONTAINER (frame), label_cursor);
  /* ja:テキスト */
  label = gtk_label_new ("");
  g_signal_connect (G_OBJECT (label), "style-set",
                                        G_CALLBACK (signal_style_set), NULL);
  /* ja:垂直ボックス/ハンドルボックス */
  vbox = gtk_vbox_new (FALSE, 0);
  handle = gtk_handle_box_new ();
  gtk_container_set_border_width (GTK_CONTAINER (handle), 2);
  gtk_container_add (GTK_CONTAINER (handle), menu_bar);
  gtk_box_pack_start (GTK_BOX (vbox), handle, FALSE, FALSE, 0);
  handle = gtk_handle_box_new ();
  gtk_container_set_border_width (GTK_CONTAINER (handle), 2);
  gtk_container_add (GTK_CONTAINER (handle), toolbar);
  gtk_box_pack_start (GTK_BOX (vbox), handle, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), notebook, TRUE, TRUE, 0);
  gtk_table_attach (GTK_TABLE (table), status, 0, 4, 0, 1,
            GTK_EXPAND | GTK_SHRINK | GTK_FILL, GTK_SHRINK | GTK_FILL, 0, 0);
  gtk_table_attach (GTK_TABLE (table), frame, 4, 5, 0, 1,
            GTK_EXPAND | GTK_SHRINK | GTK_FILL, GTK_SHRINK | GTK_FILL, 0, 0);
  gtk_box_pack_end (GTK_BOX (vbox), table, FALSE, FALSE, 0);
  gtk_box_pack_end (GTK_BOX (vbox), label, FALSE, FALSE, 0);
  gtk_container_add (GTK_CONTAINER (window), vbox);
  /* ja:ファイルの履歴 */
  menu_shell = gtk_item_factory_get_widget (ifactory_menu, "<main>/File");
  for (i = 0; i < history && history_files[i]; i++)
    {
      utf8str = g_filename_to_utf8 (history_files[i], -1, NULL, NULL, NULL);
      menu_item = gtk_menu_item_new_with_label (utf8str);
      g_free (utf8str);
      g_signal_connect (G_OBJECT (menu_item), "activate",
                G_CALLBACK (signal_activate_menu_history), history_files[i]);
      g_signal_connect (G_OBJECT (menu_item), "destroy",
                G_CALLBACK (signal_destroy_menu_history), history_files[i]);
      g_object_set_data (G_OBJECT (menu_item), "user_data", history_files[i]);
      gtk_menu_insert (GTK_MENU (menu_shell), menu_item, i + MENUFILE - 1);
    }
  if (i > 0)
    gtk_menu_insert (GTK_MENU (menu_shell), gtk_menu_item_new (),
                                                            i + MENUFILE - 1);
  /* ja:タイマ */
  timer_id = gtk_timeout_add (500, signal_timeout, NULL);

  /* ja:メニュー表示 */
  set_menu_bar (NULL);

  /* ja:既に起動しているかチェックする */
  if (!second)
    {
      GObject *comm;

      comm = misc_comm_new (window, "Text maid InterProcess communication");
      if (!misc_comm_is_first (MISC_COMM (comm)))
        {
          misc_comm_send_with_files (MISC_COMM (comm), argc, argv, arg_files);
          gtk_exit (0);
        }
      g_signal_connect (G_OBJECT (comm), "recv",
                                    G_CALLBACK (signal_selection_recv), NULL);
    }

  /* ja:表示 */
  gtk_widget_show_all (window);
  gtk_widget_hide_all (label);
  gtk_window_set_policy (GTK_WINDOW (window), TRUE, TRUE, TRUE);

  /* ja:システム色 */
  style = gtk_widget_get_style (label);
  system_color[0] = style->text[0];
  system_color[1] = style->base[0];
  system_color[2] = style->dark[0];
  system_color[3] = style->dark[0];
  system_color[4] = style->dark[0];
  system_color[5] = style->dark[0];
  system_color[6] = style->dark[0];
  system_color[7] = style->mid[0];
  system_color[8] = style->fg[3];
  system_color[9] = style->bg[3];
  /* ja:システムフォント */
  system_font = pango_font_description_copy
                                    (pango_context_get_font_description
                                    (gtk_widget_get_pango_context (label)));

  /* ja:新規ウインドウ */
  for (i = 1; i < argc; i++)
    if (arg_files[i])
      {
        tmaid = file_open_edit (argv[i]);
        if (init_line != 0)/* ja:指定行に移動 */
          jump_operation (tmaid, init_line);
        count++;
      }
  g_free (arg_files);
  if (newfile && count <= 0)
    file_open_edit (NULL);

  gtk_main ();

  /* ja:レジストリに書き込む */
  profile = profile_open (NULL);
  if (profile)
    {
      profile_set_value (profile, "tmaid", "license",
            (gpointer)&license, sizeof (gboolean), PROFILE_VALUE_TYPE_BOOL);
      profile_set_value (profile, "tmaid", "ftnum",
            (gpointer)&ftnum, sizeof (gint), PROFILE_VALUE_TYPE_INT);
      profile_set_value (profile, "tmaid", "history",
            (gpointer)&history, sizeof (gint), PROFILE_VALUE_TYPE_INT);
      profile_set_value (profile, "tmaid", "newfile",
            (gpointer)&newfile, sizeof (gboolean), PROFILE_VALUE_TYPE_BOOL);
      profile_set_value (profile, "tmaid", "second",
            (gpointer)&second, sizeof (gboolean), PROFILE_VALUE_TYPE_BOOL);
      profile_set_value (profile, "tmaid", "def_width",
            (gpointer)&def_width, sizeof (gint), PROFILE_VALUE_TYPE_INT);
      profile_set_value (profile, "tmaid", "def_height",
            (gpointer)&def_height, sizeof (gint), PROFILE_VALUE_TYPE_INT);
      profile_set_value (profile, "tmaid", "n_pos",
            (gpointer)&n_pos, sizeof (gint), PROFILE_VALUE_TYPE_INT);
      if (open_path)
        {
          utf8str = g_filename_to_utf8 (open_path, -1, NULL, NULL, NULL);
          if (utf8str)
            {
              profile_set_value (profile, "tmaid", "openpath",
                                    (gpointer)utf8str,
                                    (g_strlen (utf8str) + 1) * sizeof (gchar),
                                                    PROFILE_VALUE_TYPE_STRING);
              g_free (utf8str);
            }
        }
      if (save_path)
        {
          utf8str = g_filename_to_utf8 (save_path, -1, NULL, NULL, NULL);
          if (utf8str)
            {
              profile_set_value (profile, "tmaid", "savepath",
                                    (gpointer)utf8str,
                                    (g_strlen (utf8str) + 1) * sizeof (gchar),
                                                    PROFILE_VALUE_TYPE_STRING);
              g_free (utf8str);
            }
        }
      /* ja:ファイルの履歴 */
      for (i = 0; i < history && history_files[i] != NULL; i++)
        {
          key = g_strdup_printf ("history%d", i);
          utf8str = g_filename_to_utf8 (history_files[i], -1,
                                                            NULL, NULL, NULL);
          if (utf8str)
            {
              profile_set_value (profile, "tmaid", key, (gpointer)utf8str,
                                    (g_strlen (utf8str) + 1) * sizeof (gchar),
                                                    PROFILE_VALUE_TYPE_STRING);
              g_free (utf8str);
            }
          g_free (key);
        }
      while (i < 10)
        {
          key = g_strdup_printf ("history%d", i++);
          profile_delete_key (profile, "tmaid", key);
          g_free (key);
        }
      /* ja:検索 */
      profile_set_value (profile, "tmaid", "find_arrow",
            (gpointer)&find_arrow, sizeof (gboolean), PROFILE_VALUE_TYPE_BOOL);
      profile_set_value (profile, "tmaid", "find_ignorecase",
            (gpointer)&find_ignorecase, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL);
      profile_set_value (profile, "tmaid", "find_num",
            (gpointer)&find_num, sizeof (gint), PROFILE_VALUE_TYPE_INT);
      for (i = 0; i < find_num; i++)
        {
          key = g_strdup_printf ("find%02d", i);
          profile_set_value (profile, "tmaid", key, (gpointer)find_text[i],
                            (g_strlen (find_text[i]) + 1) * sizeof (gchar),
                                                    PROFILE_VALUE_TYPE_STRING);
          g_free (key);
        }
      while (i < 32)
        {
          key = g_strdup_printf ("find%02d", i++);
          profile_set_value (profile, "tmaid", key,
                            (gpointer)"", sizeof (gchar),
                                                    PROFILE_VALUE_TYPE_STRING);
          g_free (key);
        }
      /* ja:置換 */
      profile_set_value (profile, "tmaid", "replace_arrow",
            (gpointer)&replace_arrow, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL);
      profile_set_value (profile, "tmaid", "replace_ignorecase",
            (gpointer)&replace_ignorecase, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL);
      profile_set_value (profile, "tmaid", "replace_num",
            (gpointer)&replace_num, sizeof (gint), PROFILE_VALUE_TYPE_INT);
      for (i = 0; i < replace_num; i++)
        {
          key = g_strdup_printf ("replace%02d", i);
          profile_set_value (profile, "tmaid", key, (gpointer)replace_text[i],
                            (g_strlen (replace_text[i]) + 1) * sizeof (gchar),
                                                    PROFILE_VALUE_TYPE_STRING);
          g_free(key);
        }
      while (i < 32)
        {
          key = g_strdup_printf ("replace%02d", i++);
          profile_set_value (profile, "tmaid", key,
                            (gpointer)"", sizeof (gchar),
                                                    PROFILE_VALUE_TYPE_STRING);
          g_free (key);
        }
      /* ja:印刷 */
#ifdef G_OS_UNIX
      profile_set_value (profile, "printer", "mode",
            (gpointer)&printer.mode, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL);
      utf8str = g_filename_to_utf8 (printer.printer, -1, NULL, NULL, NULL);
      if (utf8str)
        {
          profile_set_value (profile, "printer", "printer", (gpointer)utf8str,
                                    (g_strlen (utf8str) + 1) * sizeof (gchar),
                                                    PROFILE_VALUE_TYPE_STRING);
          g_free (utf8str);
        }
#endif /* not G_OS_UNIX */
#if defined G_OS_UNIX || defined G_OS_WIN32
      profile_set_value (profile, "printer", "copies",
                                    (gpointer)&printer.copies, sizeof (gint),
                                                    PROFILE_VALUE_TYPE_INT);
#endif /* G_OS_UNIX or G_OS_WIN32 */
#ifndef G_OS_WIN32
      utf8str = g_filename_to_utf8 (printer.file, -1, NULL, NULL, NULL);
      if (utf8str)
        {
          profile_set_value (profile, "printer", "file", (gpointer)utf8str,
                                  (g_strlen (utf8str) + 1) * sizeof (gchar),
                                                    PROFILE_VALUE_TYPE_STRING);
          g_free (utf8str);
        }
      profile_set_value (profile, "printer", "width",
            (gpointer)&printer.width, sizeof (gint), PROFILE_VALUE_TYPE_INT);
      profile_set_value (profile, "printer", "height",
            (gpointer)&printer.height, sizeof (gint), PROFILE_VALUE_TYPE_INT);
      profile_set_value (profile, "printer", "left",
            (gpointer)&printer.left, sizeof (gint), PROFILE_VALUE_TYPE_INT);
      profile_set_value (profile, "printer", "top",
            (gpointer)&printer.top, sizeof (gint), PROFILE_VALUE_TYPE_INT);
      profile_set_value (profile, "printer", "right",
            (gpointer)&printer.right, sizeof (gint), PROFILE_VALUE_TYPE_INT);
      profile_set_value (profile, "printer", "bottom",
            (gpointer)&printer.bottom, sizeof (gint), PROFILE_VALUE_TYPE_INT);
#endif /* not G_OS_WIN32 */
      /* ja:ファイルタイプ */
      section = NULL;
      for (i = 0; i < ftnum; i++)
        {
          section = g_strdup_printf ("filetype%04d", i);
          profile_set_value (profile, section, "text", (gpointer)ftype[i].text,
                (g_strlen (ftype[i].text) + 1) * sizeof (gchar),
                                                    PROFILE_VALUE_TYPE_STRING);
          profile_set_value (profile, section, "ext", (gpointer)ftype[i].ext,
                (g_strlen (ftype[i].ext) + 1) * sizeof (gchar),
                                                    PROFILE_VALUE_TYPE_STRING);
          profile_set_value (profile, section, "crlf",
                (gpointer)&ftype[i].crlf, sizeof (gint),
                                                    PROFILE_VALUE_TYPE_INT);
          profile_set_value (profile,section, "margin",
                (gpointer)&ftype[i].margin, sizeof (gint),
                                                    PROFILE_VALUE_TYPE_INT);
          profile_set_value (profile, section, "tab",
                (gpointer)&ftype[i].tab, sizeof (gint),
                                                    PROFILE_VALUE_TYPE_INT);
          profile_set_value (profile,section, "writemode",
                (gpointer)&ftype[i].writemode, sizeof (gint),
                                                    PROFILE_VALUE_TYPE_INT);
          profile_set_value (profile,section, "associate",
                (gpointer)&ftype[i].associate, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL);
          profile_set_value (profile,section, "autoindent",
                (gpointer)&ftype[i].autoindent, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL);
          profile_set_value (profile, section, "code",
                (gpointer)&ftype[i].code, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL);
          profile_set_value (profile, section, "eof",
                (gpointer)&ftype[i].eof, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL);
          profile_set_value (profile, section, "hexcase",
                (gpointer)&ftype[i].hexcase, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL);
          profile_set_value (profile, section, "html",
                (gpointer)&ftype[i].html, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL);
          profile_set_value (profile, section, "esc",
                (gpointer)&ftype[i].esc, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL);
          profile_set_value (profile, section, "limit",
                (gpointer)&ftype[i].limit, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL);
          profile_set_value (profile, section, "overwrite",
                (gpointer)&ftype[i].overwrite, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL);
          profile_set_value (profile,  section,"space",
                (gpointer)&ftype[i].space, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL);
          profile_set_value (profile, section, "syscol",
                (gpointer)&ftype[i].syscol, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL);
          profile_set_value (profile, section, "tabconv",
                (gpointer)&ftype[i].tabconv, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL);
          profile_set_value (profile, section, "gline",
                (gpointer)&ftype[i].gline, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL);
          profile_set_value (profile, section, "mline",
                (gpointer)&ftype[i].mline, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL);
          profile_set_value (profile, section, "nline",
                (gpointer)&ftype[i].nline, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL);
          profile_set_value (profile, section, "uline",
                (gpointer)&ftype[i].uline, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL);
          profile_set_value (profile, section, "vline",
                (gpointer)&ftype[i].vline, sizeof (gboolean),
                                                    PROFILE_VALUE_TYPE_BOOL);
          /* ja:キャラクターセット */
          profile_set_value (profile, section, "charset",
                            (gpointer)ftype[i].charset,
                            (g_strlen (ftype[i].charset) + 1) * sizeof (gchar),
                                                    PROFILE_VALUE_TYPE_STRING);
          /* ja:フォント */
          if (ftype[i].font_name)
            profile_set_value (profile, section, "font",
                        (gpointer)ftype[i].font_name,
                        (g_strlen (ftype[i].font_name) + 1) * sizeof (gchar),
                                                    PROFILE_VALUE_TYPE_STRING);
          /* ja:色 */
          profile_set_value (profile, section, "color",
                        (gpointer)&ftype[i].color, sizeof (GdkColor) * 12,
                                                    PROFILE_VALUE_TYPE_ARRAY);
        }
      do
        {
          g_free (section);
          section = g_strdup_printf ("filetype%04d", i++);
        }
      while (profile_delete_section (profile, section));
      g_free (section);
      profile_close (profile);
    }
  pango_font_description_free (system_font);
  g_free (open_path);
  g_free (save_path);
  g_free (clipboard_text);
#ifdef G_OS_UNIX
  g_free (printer.printer);
#endif /* G_OS_UNIX */
#ifndef G_OS_WIN32
  g_free (printer.file);
#endif /* not G_OS_WIN32 */
  for (i = 0; i < find_num; i++)
    g_free (find_text[i]);
  for (i = 0; i < replace_num; i++)
    g_free (replace_text[i]);
  for (i = 0; i < history; i++)
    g_free (history_files[i]);
  for (i = 0; i < ftnum; i++)
    {
      g_free (ftype[i].text);
      g_free (ftype[i].ext);
      g_free (ftype[i].font_name);
    }
  g_free (ftype);
  return 0;
}
