/*
    Text maid
    copyright (c) 1998-2005 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include "abort.h"
#include "command.h"
#include "edit.h"
#include "file.h"
#include "find.h"
#include "general.h"
#include "jump.h"
#include "print.h"
#include "prop.h"
#include "reload.h"
#include "replace.h"
#include "repinfo.h"
#include "sigfile.h"
#include "valchr.h"
#include "misc/fileio.h"
#include "misc/misc.h"
#include "misc/misc_conf.h"
#include "misc/misc_mdi.h"
#ifdef G_OS_WIN32
# include <gdk/gdkwin32.h>
# include <tchar.h>
# include <windows.h>
#endif /* G_OS_WIN32 */


/******************************************************************************
*                                                                             *
* ja:メニュー関数群                                                           *
*                                                                             *
******************************************************************************/
void
command_new (gpointer   callback_data,
             guint      callback_action,
             GtkWidget *widget)
{
  file_open_edit (NULL);
}


#if ! defined G_OS_WIN32 && ! defined USE_GTK24
static void
command_open_ok (GtkWidget *widget,
                 GtkWidget *dialog)
{
  gchar **file;

  file = g_object_get_data (G_OBJECT (dialog), "user_data");
  *file = g_strdup (gtk_file_selection_get_filename
                                                (GTK_FILE_SELECTION (dialog)));
  gtk_widget_destroy (dialog);
}
#endif /* not defined G_OS_WIN32 and not defined USE_GTK24 */


void
command_open (gpointer   callback_data,
              guint      callback_action,
              GtkWidget *widget)
{
  gchar *file = NULL, *path;
#if defined USE_GTK24 || defined G_OS_WIN32
  gint i;
#endif /* USE_GTK24 or G_OS_WIN32 */
#ifdef G_OS_WIN32
  gchar *filter = NULL;
  gint length, pos = 0;
  LPTSTR lpszPath, lpszFilter;
  OPENFILENAME ofn;
  TCHAR szFile[MAX_PATH];
# ifdef UNICODE
  gchar *utf8file;
  gunichar2 *utf16file = NULL;

  utf8file = g_filename_to_utf8 (open_path, -1, NULL, NULL, NULL);
  if (utf8file)
    {
      utf16file = g_utf8_to_utf16 (utf8file, -1, NULL, NULL, NULL);
      g_free (utf8file);
    }
  lpszPath = utf16file;
# else /* not UNICODE */
  lpszPath = g_strdup (open_path);
# endif /* not UNICODE */
  /* ja:フィルタの設定 */
  for (i = 0; i < ftnum; i++)
    {
      length = g_strlen (ftype[i].text) + 1;
      filter = g_realloc (filter, (pos + length + 1) * sizeof (gchar));
      g_memmove (filter + pos, ftype[i].text, length * sizeof (gchar));
      pos += length;
      length = g_strlen (ftype[i].ext) + 1;
      filter = g_realloc (filter, (pos + length + 1) * sizeof (gchar));
      g_memmove (filter + pos, ftype[i].ext, length * sizeof (gchar));
      pos += length;
    }
  filter[pos] = '\0';
# ifdef UNICODE
  lpszFilter = g_utf8_to_utf16 (filter, pos + 1, NULL, NULL, NULL);
# else /* not UNICODE */
  lpszFilter = g_locale_from_utf8 (filter, pos + 1, NULL, NULL, NULL);
# endif /* not UNICODE */
  g_free (filter);
  szFile[0] = '\0';
  ofn.lStructSize = sizeof (OPENFILENAME);
  ofn.hwndOwner = GDK_WINDOW_HWND (window->window);
  ofn.lpstrFilter = lpszFilter;
  ofn.nFilterIndex = findex - 1;
  ofn.lpstrCustomFilter = NULL;
  ofn.lpstrFile = szFile;
  ofn.nMaxFile = MAX_PATH;
  ofn.lpstrFileTitle = NULL;
  ofn.lpstrInitialDir = lpszPath;
  ofn.lpstrTitle = NULL;
  ofn.Flags = OFN_HIDEREADONLY | OFN_NOCHANGEDIR;
  ofn.lpstrDefExt = NULL;
  if (GetOpenFileName (&ofn))
# ifdef UNICODE
    {
      utf16file = g_utf16_to_utf8 (szFile, -1, NULL, NULL, NULL);
      if (utf16file)
        {
          file = g_filename_from_utf8 (utf16file, -1, NULL, NULL, NULL);
          g_free (utf16file);
        }
      findex = ofn.nFilterIndex - 1;
    }
# else /* not UNICODE */
    file = g_strdup (szFile);
# endif /* not UNICODE */
  g_free (lpszPath);
  g_free (lpszFilter);
#else /* not G_OS_WIN32 */
  GtkWidget *dialog;

# ifdef USE_GTK24
  GtkFileFilter *filter;

  dialog = gtk_file_chooser_dialog_new (_("Open"),
                        GTK_WINDOW (window), GTK_FILE_CHOOSER_ACTION_OPEN,
                        GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
                        GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                        NULL);
  if (open_path)
    gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER (dialog), open_path);
  /* ja:フィルタの設定 */
  for (i = 0; i < ftnum; i++)
    {
      gchar **ext;
      gint j;

      filter = gtk_file_filter_new ();
      gtk_file_filter_set_name (filter, ftype[i].text);
      ext = g_strsplit (ftype[i].ext, G_SEARCHPATH_SEPARATOR_S, G_MAXINT);
      for (j = 0; ext[j]; j++)
        gtk_file_filter_add_pattern (filter, ext[j]);
      g_strfreev (ext);
      gtk_file_chooser_add_filter (GTK_FILE_CHOOSER (dialog), filter);
      if (i == findex)
        gtk_file_chooser_set_filter (GTK_FILE_CHOOSER (dialog), filter);
      g_object_set_data (G_OBJECT (filter), "user_data", (gpointer)i);
    }
  if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT)
    {
      file = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
      filter = gtk_file_chooser_get_filter (GTK_FILE_CHOOSER (dialog));
      findex = (gint)g_object_get_data (G_OBJECT (filter), "user_data");
    }
  gtk_widget_destroy (dialog);
# else /* not USE_GTK24 */
  dialog = gtk_file_selection_new (_("Open"));
  g_signal_connect (G_OBJECT (dialog), "destroy",
                                            G_CALLBACK (gtk_main_quit), NULL);
  g_signal_connect (G_OBJECT (GTK_FILE_SELECTION (dialog)->ok_button),
                            "clicked", G_CALLBACK (command_open_ok), dialog);
  g_signal_connect_swapped
                        (G_OBJECT (GTK_FILE_SELECTION (dialog)->cancel_button),
            "clicked", G_CALLBACK (gtk_widget_destroy), GTK_OBJECT (dialog));
  g_object_set_data (G_OBJECT (dialog), "user_data", &file);
  if (open_path)
    {
      path = g_strconcat (open_path, G_SEARCHPATH_SEPARATOR_S, NULL);
      gtk_file_selection_set_filename (GTK_FILE_SELECTION (dialog), path);
      g_free (path);
    }
  gtk_file_selection_hide_fileop_buttons (GTK_FILE_SELECTION (dialog));
  gtk_widget_show (dialog);
  gtk_grab_add (dialog);
  gtk_main ();
# endif /* not USE_GTK24 */
#endif /* not G_OS_WIN32 */
  if (file)
    {
      g_free (open_path);
      path = fileio_get_full_path (file);
      open_path = g_path_get_dirname (path);
      g_free (path);
      file_open_edit (file);
      g_free (file);
    }
}


void
command_save (gpointer   callback_data,
              guint      callback_action,
              GtkWidget *widget)
{
  gint page_num;
  TmaidWindow *tmaid;

  page_num = gtk_notebook_get_current_page (GTK_NOTEBOOK (mdi));
  tmaid = misc_mdi_get_data (MISC_MDI (mdi), page_num);
  if (misc_mdi_get_created (MISC_MDI (mdi), page_num))
    {
      command_saveas (callback_data, callback_action, widget);
    }
  else
    {
      gchar *text, *utf8str;
      const gchar *file;

      file = misc_mdi_get_file (MISC_MDI (mdi), page_num);
      utf8str = g_filename_to_utf8 (file, -1, NULL, NULL, NULL);
      text = g_strdup_printf (_("%s\nThis file is existed.\n\nOver write?"),
                                                                    utf8str);
      g_free (utf8str);
      if ((!tmaid->ft.overwrite || !fileio_isfile (file)
                                    || misc_message_box ("Text maid",text, 1,
                                            _("_Yes"), _("_No"), NULL) == 0)
                                            && file_save_text (file, tmaid))
        {
          misc_mdi_set_edited (MISC_MDI (mdi), page_num, FALSE);
          if (delete_list (&tmaid->undo) + delete_list (&tmaid->redo) > 0)
            set_menu_bar (tmaid);
        }
    }
}


void
command_saveas (gpointer   callback_data,
                guint      callback_action,
                GtkWidget *widget)
{
  gchar *file = NULL, *path;
#if defined USE_GTK24 || defined G_OS_WIN32
  gint i;
#endif /* USE_GTK24 or G_OS_WIN32 */
#ifdef G_OS_WIN32
  gchar *filter = NULL;
  gint length, pos = 0;
  LPTSTR lpszPath, lpszFilter;
  OPENFILENAME ofn;
  TCHAR szFile[MAX_PATH];
# ifdef UNICODE
  gchar *utf8file;
  gunichar2 *utf16file = NULL;

  utf8file = g_filename_to_utf8 (save_path, -1, NULL, NULL, NULL);
  if (utf8file)
    {
      utf16file = g_utf8_to_utf16 (utf8file, -1, NULL, NULL, NULL);
      g_free (utf8file);
    }
  lpszPath = utf16file;
# else /* not UNICODE */
  lpszPath = g_strdup (save_path);
# endif /* not UNICODE */
  /* ja:フィルタの設定 */
  for (i = 0; i < ftnum; i++)
    {
      length = g_strlen (ftype[i].text) + 1;
      filter = g_realloc (filter, (pos + length + 1) * sizeof (gchar));
      g_memmove (filter + pos, ftype[i].text, length * sizeof (gchar));
      pos += length;
      length = g_strlen (ftype[i].ext) + 1;
      filter = g_realloc (filter, (pos + length + 1) * sizeof (gchar));
      g_memmove (filter + pos, ftype[i].ext, length * sizeof (gchar));
      pos += length;
    }
  filter[pos] = '\0';
# ifdef UNICODE
  lpszFilter = g_utf8_to_utf16 (filter, pos + 1, NULL, NULL, NULL);
# else /* not UNICODE */
  lpszFilter = g_locale_from_utf8 (filter, pos + 1, NULL, NULL, NULL);
# endif /* not UNICODE */
  g_free (filter);
  szFile[0] = '\0';
  ofn.lStructSize = sizeof (OPENFILENAME);
  ofn.hwndOwner = GDK_WINDOW_HWND (window->window);
  ofn.lpstrFilter = lpszFilter;
  ofn.nFilterIndex = findex - 1;
  ofn.lpstrCustomFilter = NULL;
  ofn.lpstrFile = szFile;
  ofn.nMaxFile = MAX_PATH;
  ofn.lpstrFileTitle = NULL;
  ofn.lpstrInitialDir = lpszPath;
  ofn.lpstrTitle = NULL;
  ofn.Flags = OFN_HIDEREADONLY | OFN_NOCHANGEDIR | OFN_PATHMUSTEXIST;
  ofn.lpstrDefExt = NULL;
  if (GetSaveFileName (&ofn))
# ifdef UNICODE
    {
      utf16file = g_utf16_to_utf8 (szFile, -1, NULL, NULL, NULL);
      if (utf16file)
        {
          file = g_filename_from_utf8 (utf16file, -1, NULL, NULL, NULL);
          g_free (utf16file);
        }
      findex = ofn.nFilterIndex - 1;
    }
# else /* not UNICODE */
    file = g_strdup (szFile);
# endif /* not UNICODE */
  g_free (lpszPath);
  g_free (lpszFilter);
#else /* not G_OS_WIN32 */
  GtkWidget *dialog;

# ifdef USE_GTK24
  GtkFileFilter *filter;

  dialog = gtk_file_chooser_dialog_new (_("Save As"),
                        GTK_WINDOW (window), GTK_FILE_CHOOSER_ACTION_SAVE,
                        GTK_STOCK_SAVE_AS, GTK_RESPONSE_ACCEPT,
                        GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                        NULL);
  if (save_path)
    gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER (dialog), save_path);
  /* ja:フィルタの設定 */
  for (i = 0; i < ftnum; i++)
    {
      gchar **ext;
      gint j;
      GtkFileFilter *filter;

      filter = gtk_file_filter_new ();
      gtk_file_filter_set_name (filter, ftype[i].text);
      ext = g_strsplit (ftype[i].ext, G_SEARCHPATH_SEPARATOR_S, G_MAXINT);
      for (j = 0; ext[j]; j++)
        gtk_file_filter_add_pattern (filter, ext[j]);
      g_strfreev (ext);
      gtk_file_chooser_add_filter (GTK_FILE_CHOOSER (dialog), filter);
      if (i == findex)
        gtk_file_chooser_set_filter (GTK_FILE_CHOOSER (dialog), filter);
      g_object_set_data (G_OBJECT (filter), "user_data", (gpointer)i);
    }
  if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT)
    {
      file = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
      filter = gtk_file_chooser_get_filter (GTK_FILE_CHOOSER (dialog));
      findex = (gint)g_object_get_data (G_OBJECT (filter), "user_data");
    }
  gtk_widget_destroy (dialog);
# else /* not USE_GTK24 */
  dialog = gtk_file_selection_new (_("Save As"));
  g_signal_connect (G_OBJECT (dialog), "destroy",
                                            G_CALLBACK (gtk_main_quit), NULL);
  g_signal_connect (G_OBJECT (GTK_FILE_SELECTION (dialog)->ok_button),
                            "clicked", G_CALLBACK (command_open_ok), dialog);
  g_signal_connect_swapped
                        (G_OBJECT (GTK_FILE_SELECTION (dialog)->cancel_button),
            "clicked", G_CALLBACK (gtk_widget_destroy), GTK_OBJECT (dialog));
  g_object_set_data (G_OBJECT (dialog), "user_data", &file);
  if (save_path)
    {
      path = g_strconcat (save_path, G_SEARCHPATH_SEPARATOR_S, NULL);
      gtk_file_selection_set_filename (GTK_FILE_SELECTION (dialog), path);
      g_free (path);
    }
  gtk_file_selection_hide_fileop_buttons (GTK_FILE_SELECTION (dialog));
  gtk_widget_show (dialog);
  gtk_grab_add (dialog);
  gtk_main ();
# endif /* not USE_GTK24 */
#endif /* not G_OS_WIN32 */
  if (file)
    {
      gchar *text, *utf8str;
      gint page_num;
      TmaidWindow *tmaid;

      page_num = gtk_notebook_get_current_page (GTK_NOTEBOOK (mdi));
      tmaid = misc_mdi_get_data (MISC_MDI (mdi), page_num);
      utf8str = g_filename_to_utf8 (file, -1, NULL, NULL, NULL);
      text = g_strdup_printf (_("%s\nThis file is existed.\n\nOver write?"),
                                                                    utf8str);
      g_free (utf8str);
      if ((!tmaid->ft.overwrite || !fileio_isfile (file)
                                    || misc_message_box ("Text maid", text, 1,
                                            _("_Yes"), _("_No"), NULL) == 0)
                                            && file_save_text (file, tmaid))
        {
          g_free (save_path);
          path = fileio_get_full_path (file);
          save_path = g_path_get_dirname (path);
          g_free (path);
          /* ja:諸設定 */
          misc_mdi_set_created (MISC_MDI (mdi), page_num, FALSE);
          misc_mdi_set_edited (MISC_MDI (mdi), page_num, FALSE);
          misc_mdi_set_file (MISC_MDI (mdi), page_num, file);
          delete_list (&tmaid->undo);
          delete_list (&tmaid->redo);
          set_menu_bar (tmaid);
          file_set_history (file);
        }
      g_free (text);
      g_free (file);
  }
}


void
command_reload (gpointer   callback_data,
                guint      callback_action,
                GtkWidget *widget)
{
  gchar *text;
  const gchar *file;
  gint page_num;
  LineBuffer *p;
  TmaidWindow *tmaid;

  page_num = gtk_notebook_get_current_page (GTK_NOTEBOOK (mdi));
  tmaid = misc_mdi_get_data (MISC_MDI (mdi), page_num);
  file = misc_mdi_get_file (MISC_MDI (mdi), page_num);
  text = misc_mdi_get_edited (MISC_MDI (mdi), page_num) ? g_strdup_printf
            (_("File %s was edited.\nRestore to existed file?"), file) : NULL;
  if (reload_dialog (&tmaid->ft, text))
    {
      /* ja:バッファを解放する */
      while (tmaid->start->prev)
        tmaid->start = tmaid->start->prev;
      while (tmaid->start)
        {
          p = tmaid->start->next;
          if (tmaid->start->text)
            g_free(tmaid->start->text);
          g_free(tmaid->start);
          tmaid->start=p;
        }
      delete_list (&tmaid->undo);
      delete_list (&tmaid->redo);
      tmaid->cursor.x = tmaid->cursor.y = tmaid->top.x = tmaid->top.y = 0;
      tmaid->select.x = -1;
      /* ja:再読み込み */
      file_open_text (file, tmaid);
      /* ja:右マージン */
      if (tmaid->ft.limit)
        modify_margin (tmaid);
      misc_mdi_set_edited (MISC_MDI (mdi), page_num, FALSE);
      misc_set_scroll_bar (tmaid->hscroll,
                G_CALLBACK (signal_value_changed_hscroll), tmaid,
                0, edit_get_width_max (tmaid) + 1,
                MAX (tmaid->drawing->allocation.width / tmaid->font_width, 1),
                                                                tmaid->top.x);
      misc_set_scroll_bar (tmaid->vscroll,
            G_CALLBACK (signal_value_changed_vscroll), tmaid,
            0, tmaid->max,
            MAX (tmaid->drawing->allocation.height / tmaid->font_height, 1),
                                                                tmaid->top.y);
      set_menu_bar (tmaid);
      draw_caret (tmaid, NULL);
      gtk_widget_draw (tmaid->drawing, NULL);
      charlist_renewal_all (tmaid->ft.charset, tmaid->ft.ft_id);
    }
  g_free (text);
}


void
command_print (gpointer   callback_data,
               guint      callback_action,
               GtkWidget *widget)
{
  TmaidWindow *tmaid;

  tmaid = misc_mdi_get_data (MISC_MDI (mdi), -1);
  printer.select = tmaid->select.x >= 0;
  printer.column = tmaid->ft.margin;
  printer.font_width = tmaid->font_width;
  printer.font_height = tmaid->font_height;
  if (print_dialog (&printer))
    print_out (tmaid, &printer);
}


void
command_property (gpointer   callback_data,
                  guint      callback_action,
                  GtkWidget *widget)
{
  FileType ft;
  TmaidWindow *tmaid;

  tmaid = misc_mdi_get_data (MISC_MDI (mdi), -1);
  g_memset (&ft, 0, sizeof (FileType));
  property_copy (&ft, &tmaid->ft);
  ft.charset = g_strdup (tmaid->ft.charset);
  if (property_dialog(&ft))
    {
      if (ft.ft_id < 0)
        {
          /* ja:無所属のときにはこのウインドウだけ変更 */
          g_free (tmaid->ft.charset);
          tmaid->ft.charset = g_strdup (ft.charset);
          property_change (tmaid, &ft);
        }
      else
        {
          gint i;

          /* ja:同一ファイルタイプをすべて変更 */
          tmaid->ft.ft_id = ft.ft_id;
          for (i = 0; i < ftnum; i++)
            if (ftype[i].ft_id == ft.ft_id)
              {
                property_copy (ftype + i, &ft);
                break;
              }
          for (i = misc_mdi_get_n_pages (MISC_MDI (mdi)) - 1; i >= 0; i--)
            {
              tmaid = misc_mdi_get_data (MISC_MDI (mdi), i);
              if (tmaid->ft.ft_id == ft.ft_id)
                {
                  g_free (tmaid->ft.charset);
                  tmaid->ft.charset = g_strdup (ft.charset);
                  property_change (tmaid, &ft);
                }
            }
        }
      charlist_renewal_all (ft.charset, ft.ft_id);
  }
  g_free (ft.charset);
  g_free (ft.font_name);
}


void
command_close (gpointer   callback_data,
               guint      callback_action,
               GtkWidget *widget)
{
  gint page;

  if ((page = gtk_notebook_get_current_page (GTK_NOTEBOOK (mdi))) >= 0
    && prompt_close ((TmaidWindow *)misc_mdi_get_data (MISC_MDI (mdi), page)))
    gtk_notebook_remove_page (GTK_NOTEBOOK (mdi), page);
}


void
command_exit (gpointer   callback_data,
              guint      callback_action,
              GtkWidget *widget)
{
  GdkEvent event;

  event.any.type = GDK_DELETE;
  event.any.window = window->window;
  event.any.send_event = FALSE;
  gdk_event_put (&event);
}


void
command_undo (gpointer   callback_data,
              guint      callback_action,
              GtkWidget *widget)
{
  history_operation ((TmaidWindow *)misc_mdi_get_data (MISC_MDI (mdi), -1),
                                                                        FALSE);
}


void
command_redo (gpointer   callback_data,
              guint      callback_action,
              GtkWidget *widget)
{
  history_operation ((TmaidWindow *)misc_mdi_get_data (MISC_MDI (mdi), -1),
                                                                        TRUE);
}


void
command_cut (gpointer   callback_data,
             guint      callback_action,
             GtkWidget *widget)
{
  command_copy (callback_data, callback_action, widget);
  command_delete (callback_data, callback_action, widget);
}


void
command_copy (gpointer   callback_data,
              guint      callback_action,
              GtkWidget *widget)
{

  if (gtk_selection_owner_set (window, GDK_SELECTION_CLIPBOARD,
                                                            GDK_CURRENT_TIME))
    {
      gint length;
      TmaidWindow *tmaid;

      tmaid = misc_mdi_get_data (MISC_MDI (mdi), -1);
      g_free (clipboard_text);
      length = edit_get_sel_bytes (tmaid, &tmaid->select, &tmaid->cursor);
      clipboard_text = g_malloc ((length + 1) * sizeof (gchar));
      edit_cpy_sel_mem (tmaid, &tmaid->select, &tmaid->cursor, clipboard_text);
      clipboard_text[length] = '\0';
    }
}


void
command_paste (gpointer   callback_data,
               guint      callback_action,
               GtkWidget *widget)
{
  /* ja:Windowsのセレクションの実装があやしいので暫定的処理 */
#ifdef G_OS_WIN32
  if (OpenClipboard (GDK_WINDOW_HWND (window->window)))
    {
      gchar *text = NULL;
      HGLOBAL hGlobal;

      if (hGlobal = GetClipboardData (CF_UNICODETEXT))
        {
          LPWSTR lpszTextW;

          if (lpszTextW = GlobalLock (hGlobal))
            text = g_utf16_to_utf8 (lpszTextW, -1, NULL, NULL, NULL);
          GlobalUnlock (hGlobal);
        }
      if (!text && (hGlobal = GetClipboardData (CF_TEXT)))
        {
          LPSTR lpszTextA;

          if (lpszTextA = GlobalLock (hGlobal))
            text = g_locale_to_utf8 (lpszTextA, -1, NULL, NULL, NULL);
          GlobalUnlock (hGlobal);
        }
      CloseClipboard ();
      if (text)
        {
          gint page_num;
          GdkPoint select;
          TmaidHistory *d;
          TmaidWindow *tmaid;

          page_num = gtk_notebook_get_current_page (GTK_NOTEBOOK (mdi));
          tmaid = misc_mdi_get_data (MISC_MDI (mdi), page_num);
          select = tmaid->select;
          d = edit_operation (tmaid, text, g_strlen (text), TRUE, FALSE);
          g_free (text);
          d->next = tmaid->undo;
          tmaid->undo = d;
          if (delete_list (&tmaid->redo) > 0 || !d->next || select.x >= 0)
            set_menu_bar (tmaid);
          misc_mdi_set_edited (MISC_MDI (mdi), page_num, TRUE);
        }
    }
#else /* not G_OS_WIN32 */
  gtk_selection_convert (window, GDK_SELECTION_CLIPBOARD, atom_targets,
                                                            GDK_CURRENT_TIME);
#endif /* not G_OS_WIN32 */
}


void
command_delete (gpointer   callback_data,
                guint      callback_action,
                GtkWidget *widget)
{
  gint page_num;
  TmaidHistory *d;
  TmaidWindow *tmaid;

  page_num = gtk_notebook_get_current_page (GTK_NOTEBOOK (mdi));
  tmaid = misc_mdi_get_data (MISC_MDI (mdi), page_num);
  d = edit_operation (tmaid, NULL, 0, TRUE, FALSE);
  d->next = tmaid->undo;
  tmaid->undo = d;
  set_menu_bar (tmaid);
  misc_mdi_set_edited (MISC_MDI (mdi), page_num, TRUE);
}


void
command_margin (gpointer   callback_data,
                guint      callback_action,
                GtkWidget *widget)
{
  if (misc_message_box ("Text maid", _("Return at Right Margin?"), 1,
                                            _("_Yes"), _("_No"), NULL) == 0)
    margin_operation ((TmaidWindow *)misc_mdi_get_data (MISC_MDI (mdi), -1));
}


void
command_tab (gpointer   callback_data,
             guint      callback_action,
             GtkWidget *widget)
{
  if (misc_message_box ("Text maid", _("Convert Tab to Space?"), 1,
                                            _("_Yes"), _("_No"), NULL) == 0)
    tab_operation ((TmaidWindow *)misc_mdi_get_data (MISC_MDI (mdi), -1));
}


void
command_valchr (gpointer   callback_data,
                guint      callback_action,
                GtkWidget *widget)
{
  gchar valchr;

  if (valchr_dialog (&valchr))
    {
      gint page_num;
      GdkPoint select;
      TmaidHistory *d;
      TmaidWindow *tmaid;

      page_num = gtk_notebook_get_current_page (GTK_NOTEBOOK (mdi));
      tmaid = misc_mdi_get_data (MISC_MDI (mdi), page_num);
      select = tmaid->select;
      d = edit_operation (tmaid, &valchr, 1, TRUE, FALSE);
      d->next = tmaid->undo;
      tmaid->undo = d;
      if (delete_list (&tmaid->redo) > 0 || !d->next || select.x >= 0)
        set_menu_bar (tmaid);
      misc_mdi_set_edited (MISC_MDI (mdi), page_num, TRUE);
    }
}


void
command_jump (gpointer   callback_data,
              guint      callback_action,
              GtkWidget *widget)
{
  JumpDialog jmp;
  TmaidWindow *tmaid;

  tmaid = misc_mdi_get_data (MISC_MDI (mdi), -1);
  jmp.max = tmaid->max;
  jmp.cursor = tmaid->cursor.y + 1;
  if (jump_dialog (&jmp) && tmaid->cursor.y != jmp.cursor - 1)
    jump_operation (tmaid, jmp.cursor);
}


void
command_all (gpointer   callback_data,
             guint      callback_action,
             GtkWidget *widget)
{
  gint sx, sy;
  GdkPoint top;
  TmaidWindow *tmaid;

  tmaid = misc_mdi_get_data (MISC_MDI (mdi), -1);
  if (tmaid->max <= 1 && tmaid->start->length <= 0)
    return;
  top = tmaid->top;
  sx = MAX (tmaid->drawing->allocation.width / tmaid->font_width, 1);
  sy = MAX (tmaid->drawing->allocation.height / tmaid->font_height, 1);
  if (tmaid->select.x < 0)/* ja:新たに選択する */
    gtk_selection_owner_set (window, GDK_SELECTION_PRIMARY, GDK_CURRENT_TIME);
  tmaid->select.x = tmaid->select.y = 0;
  tmaid->cursor.x = edit_get_width (tmaid, tmaid->cursor.y = tmaid->max - 1);
  if (tmaid->cursor.x < tmaid->top.x)
    tmaid->top.x = tmaid->cursor.x;
  else if (tmaid->cursor.x - sx + 1 > tmaid->top.x)
    tmaid->top.x = tmaid->cursor.x - sx + 1;
  if (tmaid->cursor.y < tmaid->top.y)
    tmaid->top.y = tmaid->cursor.y;
  else if (tmaid->cursor.y - sy + 1 > tmaid->top.y)
    tmaid->top.y = tmaid->cursor.y - sy + 1;
  set_menu_bar (tmaid);
  if (tmaid->top.x != top.x)
    misc_set_scroll_bar (tmaid->hscroll,
                        G_CALLBACK (signal_value_changed_hscroll), tmaid,
                        0, edit_get_width_max (tmaid) + 1, sx, tmaid->top.x);
  if (tmaid->top.y != top.y)
    misc_set_scroll_bar(tmaid->vscroll,
                        G_CALLBACK (signal_value_changed_vscroll), tmaid,
                        0, tmaid->max, sy, tmaid->top.y);
  gtk_widget_draw (tmaid->drawing, NULL);
}


void
command_find (gpointer   callback_data,
              guint      callback_action,
              GtkWidget *widget)
{
  FindString fd;
  TmaidWindow *tmaid;

  tmaid = misc_mdi_get_data (MISC_MDI (mdi), -1);
  fd.text = NULL;
  fd.arrow = find_arrow;
  fd.ignorecase = find_ignorecase;
  if (tmaid->select.x >= 0)
    fd.arrow = tmaid->cursor.y < tmaid->select.y
                                        || (tmaid->cursor.y == tmaid->select.y
                                        && tmaid->cursor.x < tmaid->select.x);
  if (find_dialog (&fd))
    {
      gint i;

      set_menu_bar (tmaid);
      find_arrow = fd.arrow;
      find_ignorecase = fd.ignorecase;
      for (i = 0; i < find_num; i++)
        if (g_strcmp (fd.text, find_text[i]) == 0)
          break;
      if (i == find_num)
        {
          if (find_num >= 32)
            g_free (find_text[31]);
          else
            find_num++;
          g_memmove (find_text + 1, find_text,
                                            (find_num - 1) * sizeof (gchar *));
          find_text[0] = fd.text;
        }
      else
        {
          g_free (fd.text);
          if (i > 0)
            {
              gchar *text;

              text = find_text[i];
              g_memmove (find_text + 1, find_text, i * sizeof (gchar *));
              find_text[0] = text;
            }
        }
      command_next (callback_data, callback_action, widget);
    }
}


void
command_next (gpointer   callback_data,
              guint      callback_action,
              GtkWidget *widget)
{
  GtkWidget *dialog;
  TmaidWindow *tmaid;

  tmaid = misc_mdi_get_data (MISC_MDI (mdi), -1);
  /* en:Dialog Box */
  /* ja:ダイアログボックス */
  userbreak = TRUE;
  dialog = abort_dialog (_("Finding"));
  gtk_grab_add (dialog);
  find_operation (tmaid, &tmaid->cursor, &tmaid->select, find_text[0],
                callback_action ? !find_arrow : find_arrow, find_ignorecase);
  gtk_grab_remove (dialog);
  gtk_widget_destroy (dialog);
  if (userbreak && tmaid->select.x < 0)
    {
      gchar *text;

      text = g_strdup_printf (_("'%s' was not fount."), find_text[0]);
      misc_message_box ("Text maid", text, 0, _("OK"), NULL);
      g_free (text);
      set_menu_bar (tmaid);
    }
}


void
command_replace (gpointer   callback_data,
                 guint      callback_action,
                 GtkWidget *widget)
{
  gboolean replace = FALSE;
  gint page_num, result;
  TmaidWindow *tmaid;
  ReplaceString rp;

  page_num = gtk_notebook_get_current_page (GTK_NOTEBOOK (mdi));
  tmaid = misc_mdi_get_data (MISC_MDI (mdi), page_num);
  rp.src = rp.dst = NULL;
  rp.arrow = replace_arrow;
  rp.ignorecase = replace_ignorecase;
  if (tmaid->select.x >= 0)
    rp.arrow = tmaid->cursor.y < tmaid->select.y
                                        || (tmaid->cursor.y == tmaid->select.y
                                        && tmaid->cursor.x < tmaid->select.x);
  result = replace_dialog (&rp);
  if (result != REPLACE_CANCEL)
    {
      gchar *text;
      gint i, count = 0;
      GdkPoint select;
      GtkWidget *dialog = NULL;
      ReplaceInfo ri;

      set_menu_bar (tmaid);
      replace_arrow = rp.arrow;
      replace_ignorecase = rp.ignorecase;
      for (i = 0; i < find_num; i++)
        if (g_strcmp (rp.src, find_text[i]) == 0)
          break;
      if (i == find_num)
        {
          if (find_num >= 32)
            g_free (find_text[31]);
          else
            find_num++;
          g_memmove (find_text + 1, find_text,
                                            (find_num - 1) * sizeof (gchar *));
          find_text[0]=rp.src;
        }
      else
        {
          g_free (rp.src);
          text = find_text[i];
          g_memmove (find_text + 1, find_text, i * sizeof (gchar *));
          find_text[0] = text;
        }
      if (rp.dst && rp.dst[0] != '\0')
        {
          for (i = 0; i < replace_num; i++)
            if (g_strcmp (rp.dst, replace_text[i]) == 0)
              break;
          if (i == replace_num)
            {
              if (replace_num >= 32)
                g_free (replace_text[31]);
              else
                replace_num++;
              g_memmove (replace_text + 1, replace_text,
                                        (replace_num - 1) * sizeof (gchar *));
              replace_text[0] = rp.dst;
            }
          else
            {
              g_free (rp.dst);
              text = replace_text[i];
              g_memmove (replace_text + 1, replace_text, i * sizeof (gchar *));
              replace_text[0] = text;
            }
          ri.dst = replace_text[0];
        }
      else
        {
          g_free (rp.dst);
          ri.dst = NULL;
        }
      select = tmaid->select;
      ri.arrow = replace_arrow;
      ri.ignorecase = replace_ignorecase;
      userbreak = TRUE;
      do
        {
          gint bytes, data_pos;
          TmaidHistory *d;

          /* en:Dialog Box */
          /* ja:ダイアログボックス */
          if (!dialog)
            {
              dialog = abort_dialog (_("Replacing"));
              gtk_grab_add (dialog);
            }
          find_operation (tmaid, &tmaid->cursor, &select, find_text[0],
                                            replace_arrow, replace_ignorecase);
          if (tmaid->select.x < 0)
            break;
          replace = TRUE;
          bytes = edit_get_sel_bytes (tmaid, &tmaid->select, &tmaid->cursor);
          if (result != REPLACE_ALL)
            {
              /* en:Dialog Box */
              /* ja:ダイアログボックス */
              if (dialog)
                {
                  gtk_grab_remove (dialog);
                  gtk_widget_destroy (dialog);
                  dialog = NULL;
                }
              ri.src = g_malloc (bytes + 1);
              edit_cpy_sel_mem (tmaid, &tmaid->select, &tmaid->cursor, ri.src);
              ri.src[bytes] = '\0';
              ri.start = tmaid->select;
              ri.end = tmaid->cursor;
              result = repinfo_dialog (&ri);
              g_free (ri.src);
              if (result == REPLACE_CANCEL || result == REPLACE_NEXT)
                continue;
          }
          if (select.x >= 0 && tmaid->cursor.y == select.y && replace_arrow)
            data_pos = edit_get_data_pos (tmaid, select.x, select.y, FALSE);
          else
            data_pos = -1;
          d = edit_operation (tmaid, ri.dst, g_strlen (ri.dst),
                                                        replace_arrow, TRUE);
          if (data_pos >= 0)
            select.x = edit_get_screen_pos (tmaid,
                            data_pos + g_strlen (ri.dst) - bytes, select.y);
          if (result == REPLACE_OK || result == REPLACE_ALL)
            count++;
          d->next = tmaid->undo;
          tmaid->undo = d;
          if (delete_list (&tmaid->redo) > 0 || !d->next)
            set_menu_bar (tmaid);
          misc_mdi_set_edited (MISC_MDI (mdi), page_num, TRUE);
        }
      while (result != REPLACE_CANCEL);
      /* en:Dialog Box */
      /* ja:ダイアログボックス */
      if (dialog)
        {
          gtk_grab_remove (dialog);
          gtk_widget_destroy (dialog);
        }
      if (!replace)
        {
          text = g_strdup_printf (_("'%s' was not fount."), find_text[0]);
          misc_message_box ("Text maid", text, 0, _("OK"), NULL);
          g_free (text);
        }
      else if (result == REPLACE_ALL)
        {
          text = g_strdup_printf (_("%d strings were replaced"), count);
          misc_message_box ("Text maid", text, 0, _("OK"), NULL);
          g_free (text);
        }
      set_menu_bar (tmaid);
    }
}


static void
command_conf_ok (GtkWidget *widget,
                 GtkWidget *dialog)
{
  gint i;
  GList *glist;
  GtkWidget *menu_shell;

  history = misc_conf_get_history (MISC_CONF (dialog));
  newfile = misc_conf_get_newfile (MISC_CONF (dialog));
  second = misc_conf_get_second (MISC_CONF (dialog));
  misc_conf_get_size (MISC_CONF (dialog), &def_width, &def_height);
  n_pos = misc_conf_get_tab (MISC_CONF (dialog));
  gtk_widget_destroy (dialog);
  /* ja:履歴の整理 */
  menu_shell = gtk_item_factory_get_widget (ifactory_menu, "<main>/File");
  glist = gtk_container_children (GTK_CONTAINER (menu_shell));
  for (i = g_list_length (glist) - 3;
                            i > history- (history == 0) + MENUFILE - 2; i--)
    gtk_container_remove (GTK_CONTAINER (menu_shell),
                                                g_list_nth_data (glist, i));
  g_list_free (glist);
  /* ja:ノートの位置 */
  gtk_notebook_set_tab_pos (GTK_NOTEBOOK (mdi), n_pos);
}


void
command_conf (gpointer   callback_data,
              guint      callback_action,
              GtkWidget *widget)
{
  GtkWidget *dialog;

  dialog = misc_conf_new ();
  g_signal_connect (G_OBJECT (dialog), "destroy",
                                            G_CALLBACK (gtk_main_quit), NULL);
  g_signal_connect (G_OBJECT (MISC_CONF (dialog)->ok_button),
                            "clicked", G_CALLBACK (command_conf_ok), dialog);
  g_signal_connect_swapped (G_OBJECT (MISC_CONF (dialog)->cancel_button),
            "clicked", G_CALLBACK (gtk_widget_destroy), GTK_OBJECT (dialog));
  misc_conf_set_history (MISC_CONF(dialog), history);
  misc_conf_set_newfile (MISC_CONF(dialog), newfile);
  misc_conf_set_second (MISC_CONF(dialog), second);
  misc_conf_set_size (MISC_CONF(dialog), def_width, def_height);
  misc_conf_set_tab (MISC_CONF(dialog), n_pos);
  gtk_widget_show (dialog);
  gtk_grab_add (dialog);
  gtk_main ();
}
