/*
    Profile
    copyright (c) 1998-2005 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include "fileio.h"
#include "misc.h"
#include "profile.h"


#define PROFILE_DATA_TYPE_UNKNOW 0
#define PROFILE_DATA_TYPE_SPACE 1
#define PROFILE_DATA_TYPE_COMMENT 2
#define PROFILE_DATA_TYPE_SECTION 3
#define PROFILE_DATA_TYPE_KEY 4


typedef struct _ProfileList
{
  gchar *data, *section, *key, *value;
  guint type;
  struct _ProfileList *prev, *next;
} ProfileList;
struct _Profile
{
  gboolean edit;
  gchar *file;
  ProfileList *list, *sublist;
};


/******************************************************************************
*                                                                             *
* ja:初期化ファイル関数群(内部)                                               *
*                                                                             *
******************************************************************************/
/*  ja:リストをテキスト化する
    profile_list,リスト
             RET,テキストリスト                                             */
static gchar *
profile_list_to_text (ProfileList *profile_list)
{
  gchar *text = NULL;
  ProfileList *p;

  for (p = profile_list; p; p = p->next)
    if (text)
      {
        gchar *tmp;

        tmp = text;
        text = g_strconcat (text, "\n", p->data, NULL);
        g_free (tmp);
      }
    else
      {
        text = p->data ? g_strdup (p->data) : g_strdup ("");
      }
  return text;
}


/*  ja:テキストをリスト化する
    text,テキスト
     RET,リスト                                                             */
static ProfileList *
profile_list_from_text (const gchar *text)
{
  gchar **buf, *section = NULL;
  gint i;
  ProfileList *p = NULL, *q, *r;

  buf = g_strsplit (text, "\n", 0);
  for (i = 0; buf[i]; i++)
    {
      q = g_malloc0 (sizeof (ProfileList));
      q->data = g_strdup (buf[i]);
      q->prev = p;
      if (p)
        p->next = q;
      p = q;
    }
  g_strfreev (buf);
  if (p)
    while (p->prev)
      p = p->prev;
  r = p;
  /* ja:リストを整理する */
  while (p)
    {
      gsize n;
      gchar *data;

      data = p->data ? g_strstrip (g_strdup (p->data)) : NULL;
      n = data ? g_strlen (data) : 0;
      if (n <= 0)
        {
          p->type = PROFILE_DATA_TYPE_SPACE;
        }
      else if (data[0] == '#' || data[0] == ';')
        {
          p->type = PROFILE_DATA_TYPE_COMMENT;
        }
      else if (data[0] == '[' && data[n - 1] == ']')
        {
          p->type = PROFILE_DATA_TYPE_SECTION;
          g_free (section);
          section = g_strdup (data + 1);
          section[n - 2] = '\0';
          for (q = p->prev; q && q->type == PROFILE_DATA_TYPE_SPACE;
                                                                q = q->prev)
            {
              g_free (q->section);
              q->section = NULL;
            }
        }
      else if (g_strchr (data, '='))
        {
          p->type = PROFILE_DATA_TYPE_KEY;
          p->key = g_strdup (data);
          *g_strchr (p->key, '=') = '\0';
          p->value = g_strchr (p->data, '=') + 1;
        }
      else
        {
          p->type = PROFILE_DATA_TYPE_UNKNOW;
        }
      p->section = g_strdup (section);
      g_free (data);
      p = p->next;
    }
  g_free (section);
  return r;
}


/*  ja:ファイルをリスト化する
    file,ファイル名
     RET,リスト                                                             */
static ProfileList *
profile_list_from_file (const gchar *file)
{
  gchar *data;
  gsize length;
  ProfileList *p = NULL;

  data = fileio_load (file, &length);
  if (data)
    {
      data = g_realloc (data, length + 1);
      data[length] = '\0';
      p = profile_list_from_text (data);
      g_free (data);
    }
  return p;
}


/*  ja:初期化ファイルを解放する
    profile_list,リスト                                                     */
static void
profile_list_free (ProfileList *profile_list)
{
  ProfileList *p, *q;

  for (p = profile_list; p; p = q)
    {
      q = p->next;
      g_free (p->data);
      g_free (p->section);
      g_free (p->key);
      g_free (p);
    }
}


/******************************************************************************
*                                                                             *
* ja:初期化ファイル関数群(独自)                                               *
*                                                                             *
******************************************************************************/
/*  ja:初期化ファイルを開く
    file,ファイル名
     RET,プロファイル                                                       */
Profile *
profile_open (const gchar *file)
{
  gchar *base = NULL, *name;
  gint n;
  Profile *profile;

  profile = g_malloc0 (sizeof (Profile));
  name = g_get_prgname ();
  if (file)
    {
      base = g_path_get_basename (file);
      if (g_strfilecmp (file, base) != 0)
        {
          g_free (base);
          base = NULL;
        }
    }
  else
    {
      base = g_path_get_basename (name);
      n = g_strlen (base);
      if (n >= 5 && g_strfilecmp (base + n - 4, ".exe") == 0)
        base[n -= 4] = '\0';
    }
  if (base)
    {
      gchar *tmp;

      tmp = g_strdup (base);
      n = g_strlen (tmp);
      while (tmp[0] == '.')
        g_memmove (tmp, tmp + 1, n--);
      if (n > 0)
        {
          gchar *sys;

#ifdef SYSCONFDIR
          sys = g_strconcat (SYSCONFDIR, G_DIR_SEPARATOR_S,
                                                        tmp, ".conf", NULL);
#else /* not SYSCONFDIR */
          gchar *dir;

          dir = g_path_get_dirname (name);
          sys = g_strconcat (dir, G_DIR_SEPARATOR_S, tmp, ".conf", NULL);
          g_free (dir);
#endif /* not SYSCONFDIR */
          profile->sublist = profile_list_from_file (sys);
          g_free (sys);
        }
      g_free (tmp);
    }
  profile->file = !base ? g_strdup (file)
#ifdef G_OS_WIN32
      : g_strconcat (g_get_home_dir (), G_DIR_SEPARATOR_S, base, ".ini", NULL);
#else /* not G_OS_WIN32 */
      : g_strconcat (g_get_home_dir (), G_DIR_SEPARATOR_S, ".", base, NULL);
#endif /* not G_OS_WIN32 */
  g_free (base);
  profile->list = profile_list_from_file (profile->file);
  return profile;
}


/*  ja:初期化ファイルを閉じる
    profile,プロファイル
        RET,TRUE:正常終了,FALSE:エラー                                      */
gboolean
profile_close (Profile *profile)
{
  if (!profile)
    return TRUE;
  if (profile->edit)
    {
      /* ja:変更があったとき */
      gchar *text;

      text = profile_list_to_text (profile->list);
      if (text)
        {
          fileio_save (profile->file, text, g_strlen (text) * sizeof (gchar));
          g_free (text);
        }
    }
  profile_list_free (profile->list);
  profile_list_free (profile->sublist);
  g_free (profile->file);
  g_free (profile);
  return TRUE;
}


/*  ja:初期化ファイルから文字列を取得する
    profile,プロファイル
    section,セクション
        key,キー
        RET,文字列,NULL:エラー                                              */
gchar *
profile_get_string (Profile     *profile,
                    const gchar *section,
                    const gchar *key)
{
  ProfileList *p;

  if (!profile || !section || !key)
    return NULL;
  for (p = profile->list; p; p = p->next)
    if (p->type == PROFILE_DATA_TYPE_KEY && g_strcmp (p->section, section) == 0
                                                && g_strcmp (p->key, key) == 0)
      return g_strdup (p->value);
  for (p = profile->sublist; p; p = p->next)
    if (p->type == PROFILE_DATA_TYPE_KEY && g_strcmp (p->section, section) == 0
                                                && g_strcmp (p->key, key) == 0)
      return g_strdup (p->value);
  return NULL;
}


/*  ja:初期化ファイルから値のサイズ取得する
    profile,プロファイル
    section,セクション
        key,キー
       type,タイプ
        RET,バイト数,0:エラー                                               */
gsize
profile_get_size (Profile     *profile,
                  const gchar *section,
                  const gchar *key,
                  const guint  type)
{
  guint8 *array;
  gsize n;
  ProfileList *p;

  if (!profile || !section || !key)
    return 0;
  for (p = profile->list; p; p = p->next)
    if (p->type == PROFILE_DATA_TYPE_KEY && g_strcmp (p->section, section) == 0
                                                && g_strcmp (p->key, key) == 0)
      break;
  if (!p)
    for (p = profile->sublist; p; p = p->next)
      if (p->type == PROFILE_DATA_TYPE_KEY
        && g_strcmp (p->section, section) == 0 && g_strcmp (p->key, key) == 0)
        break;
  if (!p)
    return 0;
  switch (type)
    {
      case PROFILE_VALUE_TYPE_BOOL:
        return g_strcasecmp (p->value, "true") == 0
                || g_strcasecmp (p->value, "false") == 0
                || g_strcasecmp (p->value, "yes") == 0
                || g_strcasecmp (p->value, "no") == 0
                || g_strcasecmp (p->value, "on") == 0
                || g_strcasecmp (p->value, "off") == 0
                || g_strcasecmp (p->value, "ok") == 0
                || g_strcasecmp (p->value, "cancel") == 0
                ? sizeof (gboolean) : 0;
      case PROFILE_VALUE_TYPE_INT:
        return sizeof (gint);
      case PROFILE_VALUE_TYPE_STRING:
        return g_strlen (p->value) + 1;
      case PROFILE_VALUE_TYPE_ARRAY:
        array = misc_str_to_array (&n, 8, p->value, 10, FALSE);
        if (!array)
          return 0;
        g_free (array);
        return n;
    }
  return 0;
}


/*  ja:初期化ファイルから値を取得する
    profile,プロファイル
    section,セクション
        key,キー
      value,値を入れるバッファ
       size,値を入れるバッファのサイズ
       type,タイプ
        RET,TRUE:正常終了,FALSE:エラー                                      */
gboolean
profile_get_value (Profile     *profile,
                   const gchar *section,
                   const gchar *key,
                   gpointer     value,
                   const gsize  size,
                   const guint  type)
{
  guint8 *array;
  gint n;
  ProfileList *p;

  if (!profile || !section || !key || !value)
    return FALSE;
  for (p = profile->list; p; p = p->next)
    if (p->type == PROFILE_DATA_TYPE_KEY && g_strcmp (p->section, section) == 0
                                                && g_strcmp (p->key, key) == 0)
      break;
  if (!p)
    for (p = profile->sublist; p; p = p->next)
      if (p->type == PROFILE_DATA_TYPE_KEY
        && g_strcmp (p->section, section) == 0 && g_strcmp (p->key, key) == 0)
        break;
  if (!p)
    return FALSE;
  switch (type)
    {
      case PROFILE_VALUE_TYPE_BOOL:
        if (size < sizeof (gboolean))
          return FALSE;
        if (g_strcasecmp (p->value, "true") == 0
            || g_strcasecmp (p->value, "yes") == 0
            || g_strcasecmp (p->value, "on") == 0
            || g_strcasecmp (p->value, "ok") == 0)
          *((gboolean *)value) = TRUE;
        else if (g_strcmp (p->value, "false") == 0
            || g_strcasecmp (p->value, "no") == 0
            || g_strcasecmp (p->value, "off") == 0
            || g_strcasecmp (p->value, "cancel") == 0)
          *((gboolean *)value) = FALSE;
        else
          return FALSE;
        break;
      case PROFILE_VALUE_TYPE_INT:
        if (size < sizeof (gint))
          return FALSE;
        misc_str_to_val ((gint *)value, p->value, 10, TRUE);
        break;
      case PROFILE_VALUE_TYPE_STRING:
        if (size < g_strlen (p->value) + 1)
          return FALSE;
        g_strcpy ((gchar *)value, p->value);
        break;
      case PROFILE_VALUE_TYPE_ARRAY:
        array = misc_str_to_array (&n, 8, p->value, 10, FALSE);
        if (!array)
          return FALSE;
        if (size <= n)
          g_memmove (value, array, size);
        g_free (array);
        if (n < size)
          return FALSE;
        break;
      default:
        return FALSE;
    }
  return TRUE;
}


/*  ja:初期化ファイルに値を設定する
    profile,プロファイル
    section,セクション
        key,キー
      value,値が入っているバッファ
       size,値が入っているバッファのサイズ
       type,タイプ
        RET,TRUE:正常終了,FALSE:エラー                                      */
gboolean
profile_set_value (Profile       *profile,
                   const gchar   *section,
                   const gchar   *key,
                   gconstpointer  value,
                   const gsize    size,
                   const guint    type)
{
  gchar *data;
  gint i;
  ProfileList *p ,*q = NULL;

  if (!profile || !section || !key || !value)
    return FALSE;
  for (p = profile->list; p; q = p, p = p->next)
    if (p->type == PROFILE_DATA_TYPE_KEY && g_strcmp (p->section, section) == 0
                                                && g_strcmp (p->key, key) == 0)
      break;
  if (!p)
    {
      for (p = q; p; p = p->prev)
        if (p->section && g_strcmp (p->section, section) == 0)
          break;
      if (!p)
        {
          if (q)
            {
              /* ja:前のデータとの間にスペースを入れる */
              p = g_malloc (sizeof (ProfileList));
              p->type = PROFILE_DATA_TYPE_SPACE;
              p->data = p->section = p->key = p->value = NULL;
              p->prev = q;
              p->next = q->next;
              q->next = p;
              q = p;
            }
          /* ja:セクションもキーもないときにはセクションを作る */
          p = g_malloc (sizeof (ProfileList));
          p->type = PROFILE_DATA_TYPE_SECTION;
          p->data = g_strdup_printf ("[%s]", section);
          p->section = g_strdup (section);
          p->key = p->value = NULL;
          p->prev = q;
          if (q)
            {
              p->next = q->next;
              q->next = p;
            }
          else
            {
              p->next = NULL;
              profile->list = p;
            }
        }
      q = p;
      while (q->type == PROFILE_DATA_TYPE_SPACE && q->section
                            && g_strcmp (p->section, section) == 0 && q->prev)
        q = q->prev;
      /* ja:セクションの最後にキーを作る */
      p = g_malloc (sizeof (ProfileList));
      p->type = PROFILE_DATA_TYPE_KEY;
      p->data = g_strdup_printf ("%s=", key);
      p->section = g_strdup (section);
      p->key = g_strdup (key);
      p->value = g_strchr (p->data, '=') + 1;
      p->prev = q;
      p->next = q->next;
      q->next = p;
      if (p->next)
        p->next->prev = p;
    }
  switch (type)
    {
      case PROFILE_VALUE_TYPE_BOOL:
        g_free (p->data);
        p->data = g_strdup_printf ("%s=%s",
                            p->key, *((gboolean *)value) ? "true" : "false");
        break;
      case PROFILE_VALUE_TYPE_INT:
        g_free (p->data);
        p->data = g_strdup_printf ("%s=%d", p->key, *((gint *)value));
        break;
      case PROFILE_VALUE_TYPE_STRING:
        g_free (p->data);
        p->data = g_strdup_printf ("%s=%s", p->key, (gchar *)value);
        break;
      case PROFILE_VALUE_TYPE_ARRAY:
        g_free (p->data);
        p->data = g_strdup_printf ("%s=%u", p->key, ((guint8 *)value)[0]);
        for (i = 1; i < size; i++)
          {
            data = g_strdup_printf ("%s %u", p->data, ((guint8 *)value)[i]);
            g_free (p->data);
            p->data = data;
          }
        break;
      default:
        return FALSE;
    }
  p->type = type;
  p->value = g_strchr (p->data, '=') + 1;
  profile->edit = TRUE;
  return TRUE;
}


/*  ja:初期化ファイルのセクションを削除する
    profile,プロファイル
    section,セクション
        RET,TRUE:正常終了,FALSE:エラー                                      */
gboolean
profile_delete_section (Profile     *profile,
                        const gchar *section)
{
  gboolean result = FALSE;
  ProfileList *p, *q;

  if (!profile || !section)
    return FALSE;
  for (p = profile->list; p; p = q)
    {
      q = p->next;
      if (p->section && g_strcmp (p->section, section) == 0)
        {
          if (p->prev)
            p->prev->next = p->next;
          if (p->next)
            p->next->prev = p->prev;
          g_free (p->data);
          g_free (p->section);
          g_free (p->key);
          g_free (p);
          profile->edit = TRUE;
          result = TRUE;
        }
    }
  return result;
}


/*  ja:初期化ファイルのキーを削除する
    profile,プロファイル
    section,セクション
        key,キー
        RET,TRUE:正常終了,FALSE:エラー                                      */
gboolean
profile_delete_key (Profile     *profile,
                    const gchar *section,
                    const gchar *key)
{
  gboolean result = FALSE;
  ProfileList *p, *q;

  if (!profile || !section || !key)
    return FALSE;
  for (p = profile->list; p; p = q)
    {
      q = p->next;
      if (p->section && p->key && g_strcmp (p->section, section) == 0
                                                && g_strcmp (p->key, key) == 0)
        {
          if (p->prev)
            p->prev->next = p->next;
          if (p->next)
            p->next->prev = p->prev;
          g_free (p->data);
          g_free (p->section);
          g_free (p->key);
          g_free (p);
          profile->edit = TRUE;
          result = TRUE;
        }
    }
  return result;
}


/*  ja:初期化ファイルのセクションを列挙する
    profile,プロファイル
        RET,セクションのリスト,NULL:エラー                                  */
GList *
profile_enum_section (Profile *profile)
{
  GList *glist = NULL;
  ProfileList *p;

  if (!profile)
    return NULL;
  for (p = profile->list; p; p = p->next)
    if (p->section && (!glist || !g_list_find_custom (glist, p->section,
                                                        (GCompareFunc)strcmp)))
      glist = g_list_insert_sorted (glist, p->section, (GCompareFunc)strcmp);
  for (p = profile->sublist; p; p = p->next)
    if (p->section && (!glist || !g_list_find_custom (glist, p->section,
                                                        (GCompareFunc)strcmp)))
      glist = g_list_insert_sorted (glist, p->section, (GCompareFunc)strcmp);
  return glist;
}


/*  ja:初期化ファイルのキーを列挙する
    profile,プロファイル
    section,セクション
        RET,セクションのリスト,NULL:エラー                                  */
GList *
profile_enum_key (Profile     *profile,
                  const gchar *section)
{
  GList *glist = NULL;
  ProfileList *p;

  if (!profile)
    return NULL;
  for (p = profile->list; p; p = p->next)
    if (p->section && p->key && g_strcmp (p->section, section) == 0
                            && (!glist || !g_list_find_custom (glist, p->key,
                                                        (GCompareFunc)strcmp)))
      glist = g_list_insert_sorted (glist, p->key, (GCompareFunc)strcmp);
  for (p = profile->sublist; p; p = p->next)
    if (p->section && p->key && g_strcmp (p->section, section) == 0
                            && (!glist || !g_list_find_custom (glist, p->key,
                                                        (GCompareFunc)strcmp)))
      glist = g_list_insert_sorted (glist, p->key, (GCompareFunc)strcmp);
  return glist;
}


/******************************************************************************
*                                                                             *
* ja:初期化ファイル関数群(互換)                                               *
*                                                                             *
******************************************************************************/
#if ! GLIB_CHECK_VERSION(2,6,0)
GKeyFile *
g_key_file_new (void)
{
  return g_malloc0 (sizeof (Profile));
}


void
g_key_file_free (GKeyFile *key_file)
{
  if (key_file)
    {
      profile_list_free (key_file->list);
      profile_list_free (key_file->sublist);
      g_free (key_file->file);
      g_free (key_file);
    }
}


gboolean
g_key_file_load_from_file (GKeyFile       *key_file,
                           const gchar    *file,
                           GKeyFileFlags   flags,
                           GError        **error)
{
  gboolean result = FALSE;

  if (error)
    *error = NULL;
  if (key_file)
    {
      gchar *data;
      gsize length;

      data = fileio_load (file, &length);
      if (data)
        result = g_key_file_load_from_data (key_file, data, length,
                                                                flags, error);
      g_free (data);
    }
  return result;
}


gboolean
g_key_file_load_from_data (GKeyFile       *key_file,
                           const gchar    *data,
                           gsize           length,
                           GKeyFileFlags   flags,
                           GError        **error)
{
  if (error)
    *error = NULL;
  if (!key_file)
    return FALSE;
  /* ja:リスト解放 */
  profile_list_free (key_file->list);
  profile_list_free (key_file->sublist);
  key_file->list = key_file->sublist = NULL;
  if (data && length != 0)
    {
      gchar *text;

      text = length == (gsize)-1 ? g_strdup (data) : g_strndup (data, length);
      key_file->list = profile_list_from_text (text);
      g_free (text);
    }
  return TRUE;
}


gchar *
g_key_file_to_data (GKeyFile  *key_file,
                    gsize     *length,
                    GError   **error)
{
  gchar *data;

  if (error)
    *error = NULL;
  data = key_file ? profile_list_to_text (key_file->list) : NULL;
  if (data && length)
    *length = g_strlen (data);
  return data;
}


gchar **
g_key_file_get_groups (GKeyFile *key_file,
                       gsize    *length)
{
  gchar **str = NULL;
  GList *glist;

  glist = profile_enum_section (key_file);
  if (glist)
    {
      guint i, leng;

      leng = g_list_length (glist);
      str = g_malloc ((leng + 1) * sizeof (gchar *));
      for (i = 0; i < leng; i++)
        str[i] = g_strdup (g_list_nth_data (glist, i));
      str[i] = NULL;
      g_list_free (glist);
      if (length)
        *length = leng;
    }
  return str;
}


gchar **
g_key_file_get_keys (GKeyFile     *key_file,
                     const gchar  *group_name,
                     gsize        *length,
                     GError      **error)
{
  gchar **str = NULL;
  GList *glist;

  if (error)
    *error = NULL;
  glist = profile_enum_key (key_file, group_name);
  if (glist)
    {
      guint i, leng;

      leng = g_list_length (glist);
      str = g_malloc ((leng + 1) * sizeof (gchar *));
      for (i = 0; i < leng; i++)
        str[i] = g_strdup (g_list_nth_data (glist, i));
      str[i] = NULL;
      g_list_free (glist);
      if (length)
        *length = leng;
    }
  return str;
}


gboolean
g_key_file_has_group (GKeyFile    *key_file,
                      const gchar *group_name)
{
  if (key_file && group_name)
    {
      ProfileList *p;

      for (p = key_file->list; p; p = p->next)
        if (p->section && g_strcmp (p->section, group_name) == 0)
          return TRUE;
      for (p = key_file->sublist; p; p = p->next)
        if (p->section && g_strcmp (p->section, group_name) == 0)
          return TRUE;
    }
  return FALSE;
}


gboolean
g_key_file_has_key (GKeyFile     *key_file,
                    const gchar  *group_name,
                    const gchar  *key,
                    GError      **error)
{
  if (error)
    *error = NULL;
  if (key_file && group_name && key)
    {
      ProfileList *p;

      for (p = key_file->list; p; p = p->next)
        if (p->section && p->key && g_strcmp (p->section, group_name) == 0
                                                && g_strcmp (p->key, key) == 0)
          return TRUE;
      for (p = key_file->sublist; p; p = p->next)
        if (p->section && p->key && g_strcmp (p->section, group_name) == 0
                                                && g_strcmp (p->key, key) == 0)
          return TRUE;
    }
  return FALSE;
}


gchar *
g_key_file_get_string (GKeyFile     *key_file,
                       const gchar  *group_name,
                       const gchar  *key,
                       GError      **error)
{
  if (error)
    *error = NULL;
  return profile_get_string (key_file, group_name, key);
}


gboolean
g_key_file_get_boolean (GKeyFile     *key_file,
                        const gchar  *group_name,
                        const gchar  *key,
                        GError      **error)
{
  gboolean value;

  if (error)
    *error = NULL;
  if (!profile_get_value (key_file, group_name, key,
                (gpointer)&value, sizeof (gboolean), PROFILE_VALUE_TYPE_BOOL))
    value = FALSE;
  return value;
}


gint
g_key_file_get_integer (GKeyFile     *key_file,
                        const gchar  *group_name,
                        const gchar  *key,
                        GError      **error)
{
  gint value;

  if (error)
    *error = NULL;
  if (!profile_get_value (key_file, group_name, key,
                    (gpointer)&value, sizeof (gint), PROFILE_VALUE_TYPE_INT))
    value = 0;
  return value;
}


gint *
g_key_file_get_integer_list (GKeyFile     *key_file,
                             const gchar  *group_name,
                             const gchar  *key,
                             gsize        *length,
                             GError      **error)
{
  gsize leng;
  gint *value = NULL;

  if (error)
    *error = NULL;
  leng = profile_get_size (key_file, group_name, key,
                                                    PROFILE_VALUE_TYPE_ARRAY);
  if (leng > 0)
    {
      value = g_malloc (leng);
      if (!profile_get_value (key_file, group_name, key,
                            (gpointer)value, leng, PROFILE_VALUE_TYPE_ARRAY))
        {
          g_free (value);
          value = NULL;
          leng = 0;
        }
    }
  if (length)
    *length = leng / sizeof (gint);
  return value;
}


void
g_key_file_set_string (GKeyFile    *key_file,
                       const gchar *group_name,
                       const gchar *key,
                       const gchar *string)
{
  profile_set_value (key_file, group_name, key, (gpointer)string,
        (g_strlen (string) + 1) * sizeof (gchar), PROFILE_VALUE_TYPE_STRING);
}


void
g_key_file_set_boolean (GKeyFile    *key_file,
                        const gchar *group_name,
                        const gchar *key,
                        gboolean     value)
{
  profile_set_value (key_file, group_name, key,
                (gpointer)&value, sizeof (gboolean), PROFILE_VALUE_TYPE_BOOL);
}


void
g_key_file_set_integer (GKeyFile    *key_file,
                        const gchar *group_name,
                        const gchar *key,
                        gint         value)
{
  profile_set_value (key_file, group_name, key,
                    (gpointer)&value, sizeof (gint), PROFILE_VALUE_TYPE_INT);
}


void
g_key_file_set_integer_list (GKeyFile    *key_file,
                             const gchar *group_name,
                             const gchar *key,
                             gint         list[],
                             gsize        length)
{
  profile_set_value (key_file, group_name, key,
            (gpointer)list, length * sizeof (gint), PROFILE_VALUE_TYPE_ARRAY);
}


void
g_key_file_remove_group (GKeyFile     *key_file,
                         const gchar  *group_name,
                         GError      **error)
{
  if (error)
    *error = NULL;
  profile_delete_section (key_file, group_name);
}


void
g_key_file_remove_key (GKeyFile     *key_file,
                       const gchar  *group_name,
                       const gchar  *key,
                       GError      **error)
{
  if (error)
    *error = NULL;
  profile_delete_key (key_file, group_name, key);
}
#endif /* not GLIB_CHECK_VERSION(2,6,0) */
