/*
    misc
    copyright (c) 1998-2006 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include "misc.h"
#include <gdk/gdkkeysyms.h>


/******************************************************************************
*                                                                             *
* ja:メニュー/ツールバー関数群                                                *
*                                                                             *
******************************************************************************/
/*  ja:メニューを作成する
        entries,メニュー構造体
    accel_group,アクセルグループ                                            */
void
misc_create_menu (MiscCreateMenuEntry *entries,
                  GtkAccelGroup       *accel_group)
{
  gint i;

  for (i = 0; (entries[i].menu_type & MISC_CREATE_MENU_MASK)
                                            != MISC_CREATE_MENU_TERMINAL; i++)
    switch (entries[i].menu_type & MISC_CREATE_MENU_MASK)
      {
        case MISC_CREATE_MENU_BAR:
          entries[i].widget = gtk_menu_bar_new ();
          break;
        case MISC_CREATE_MENU_SHELL:
          {
            gchar *path, *p;
            gint j;

            entries[i].widget = gtk_menu_new ();
            path = g_strdup (entries[i].path);
            p = g_strrchr (path, '/');
            *p = '\0';
            for (j = 0; j < i; j++)
              if (g_strcmp (entries[j].path, path) == 0)
                {
                  gtk_menu_item_set_submenu (GTK_MENU_ITEM (entries[j].widget),
                                                            entries[i].widget);
                  break;
                }
            g_free (path);
          }
          break;
        default:
          {
            gchar *path, *p;
            gint j;

            switch (entries[i].menu_type & MISC_CREATE_MENU_MASK)
              {
                case MISC_CREATE_MENU_STOCK:
                  if (entries[i].menu_type & MISC_CREATE_MENU_NOIMAGE)
                    {
                      GtkStockItem stock_item;

                      gtk_stock_lookup (entries[i].name, &stock_item);
                      entries[i].widget = gtk_image_menu_item_new_with_mnemonic
                                                            (stock_item.label);
                    }
                  else
                    {
                      entries[i].widget = gtk_image_menu_item_new_from_stock
                                        (entries[i].name,
                                        entries[i].accel ? NULL : accel_group);
                    }
                  break;
                case MISC_CREATE_MENU_ITEM:
                  if (entries[i].xpm
                        && !(entries[i].menu_type & MISC_CREATE_MENU_NOIMAGE))
                    {
                      GdkPixbuf *pixbuf;

                      entries[i].widget = gtk_image_menu_item_new_with_mnemonic
                                                        (_(entries[i].name));
                      pixbuf = gdk_pixbuf_new_from_xpm_data (entries[i].xpm);
                      gtk_image_menu_item_set_image
                                    (GTK_IMAGE_MENU_ITEM (entries[i].widget),
                                        gtk_image_new_from_pixbuf (pixbuf));
                      g_object_unref (pixbuf);
                    }
                  else
                    {
                      entries[i].widget = gtk_menu_item_new_with_mnemonic
                                                        (_(entries[i].name));
                    }
                  break;
                case MISC_CREATE_MENU_CHECK:
                  entries[i].widget = gtk_check_menu_item_new_with_mnemonic
                                                        (_(entries[i].name));
                  break;
                case MISC_CREATE_MENU_RADIO:
                  entries[i].widget = gtk_radio_menu_item_new_with_mnemonic
                        (i > 0
                        && (entries[i - 1].menu_type & MISC_CREATE_MENU_MASK)
                                                     == MISC_CREATE_MENU_RADIO
                            ? gtk_radio_menu_item_get_group
                                (GTK_RADIO_MENU_ITEM (entries[i - 1].widget))
                            : NULL, _(entries[i].name));
                  break;
                default:
                  entries[i].widget = gtk_separator_menu_item_new ();
              }
            if (entries[i].menu_type & MISC_CREATE_MENU_ELLIPSIS)
              {
                gchar *mnemonic;
                GtkWidget *label;

                label = gtk_bin_get_child (GTK_BIN (entries[i].widget));
                mnemonic = g_strconcat
                        (gtk_label_get_label (GTK_LABEL (label)), "...", NULL);
                gtk_label_set_text_with_mnemonic (GTK_LABEL (label), mnemonic);
                g_free (mnemonic);
              }
            path = g_strdup (entries[i].path);
            p = g_strrchr (path, '/') + 1 ;
            *p = '\0';
            for (j = 0; j < i; j++)
              if (g_strcmp (entries[j].path, path) == 0)
                {
                  gtk_menu_shell_append (GTK_MENU_SHELL (entries[j].widget),
                                                            entries[i].widget);
                  break;
                }
            g_free (path);
            if (accel_group && entries[i].accel && entries[i].accel[0] != '\0')
              {
                guint key;
                GdkModifierType mods;

                gtk_accelerator_parse (entries[i].accel, &key, &mods);
                gtk_widget_add_accelerator (entries[i].widget, "activate",
                                        accel_group, key, mods,
                                        GTK_ACCEL_VISIBLE | GTK_ACCEL_LOCKED);
              }
            if (entries[i].func)
              g_signal_connect (G_OBJECT (entries[i].widget),
                            (entries[i].menu_type & MISC_CREATE_MENU_MASK)
                                                    == MISC_CREATE_MENU_CHECK
                            || (entries[i].menu_type & MISC_CREATE_MENU_MASK)
                                                    == MISC_CREATE_MENU_RADIO
                                                    ? "toggled" : "activate",
                        G_CALLBACK (entries[i].func), entries[i].user_data);
            if (entries[i].menu_type & MISC_CREATE_MENU_DISABLE)
              gtk_widget_set_sensitive (entries[i].widget, FALSE);
          }
      }
}


/*  ja:メニューを検索する
    entries,メニュー構造体
       path,パス
        RET,メニューアイテム                                                */
GtkWidget *
misc_find_menu (MiscCreateMenuEntry *entries,
                const gchar         *path)
{
  gint i;

  for (i = 0; entries[i].menu_type != MISC_CREATE_MENU_TERMINAL; i++)
    if (g_strcmp (entries[i].path, path) == 0)
      return entries[i].widget;
  return NULL;
}


/*  ja:ツールバーを作成する
    entries,ツールバー構造体
        RET,ツールバー                                                      */
GtkWidget *
misc_create_toolbar (MiscCreateToolbarEntry *entries)
{
  gint i;
  GtkTooltips *tooltips;
  GtkWidget *toolbar;

  toolbar = gtk_toolbar_new ();
  gtk_toolbar_set_orientation (GTK_TOOLBAR (toolbar),
                                                GTK_ORIENTATION_HORIZONTAL);
  gtk_toolbar_set_style (GTK_TOOLBAR (toolbar), GTK_TOOLBAR_ICONS);
  gtk_toolbar_set_tooltips (GTK_TOOLBAR (toolbar), TRUE);
  tooltips = gtk_tooltips_new ();
  for (i = 0; entries[i].path; i++)
    {
      if (entries[i].name)
        {
          if (entries[i].xpm)
            {
              GdkPixbuf *pixbuf;

              pixbuf = gdk_pixbuf_new_from_xpm_data (entries[i].xpm);
#if GTK_CHECK_VERSION(2,4,0)
              entries[i].tool_item = gtk_tool_button_new
                    (gtk_image_new_from_pixbuf (pixbuf), _(entries[i].name));
              gtk_tool_item_set_tooltip (entries[i].tool_item, tooltips,
                                    _(entries[i].name), _(entries[i].name));
#else /* not GTK_CHECK_VERSION(2,4,0) */
              entries[i].tool_item = gtk_toolbar_append_item
                                            (GTK_TOOLBAR (toolbar),
                                            _(entries[i].name),
                                            _(entries[i].name),
                                            _(entries[i].name),
                                            gtk_image_new_from_pixbuf (pixbuf),
                                            G_CALLBACK (entries[i].func),
                                            NULL);
#endif /* not GTK_CHECK_VERSION(2,4,0) */
              g_object_unref (pixbuf);
            }
          else
            {
              gchar *text;
              GtkStockItem stock_item;

              gtk_stock_lookup (entries[i].name, &stock_item);
              text = misc_mnemonic_to_text (stock_item.label);
#if GTK_CHECK_VERSION(2,4,0)
              entries[i].tool_item = gtk_tool_button_new_from_stock
                                                            (entries[i].name);
              gtk_tool_item_set_tooltip (entries[i].tool_item, tooltips,
                                                                text, text);
#else /* not GTK_CHECK_VERSION(2,4,0) */
              entries[i].tool_item = gtk_toolbar_insert_stock
                                                (GTK_TOOLBAR (toolbar),
                                                entries[i].name,
                                                text,
                                                text,
                                                G_CALLBACK (entries[i].func),
                                                NULL,
                                                -1);
#endif /* not GTK_CHECK_VERSION(2,4,0) */
              g_free (text);
            }
#if GTK_CHECK_VERSION(2,4,0)
          if (entries[i].func)
            g_signal_connect (G_OBJECT (entries[i].tool_item), "clicked",
                        G_CALLBACK (entries[i].func), entries[i].user_data);
#endif /* GTK_CHECK_VERSION(2,4,0) */
        }
      else
        {
#if GTK_CHECK_VERSION(2,4,0)
          entries[i].tool_item = gtk_separator_tool_item_new ();
#else /* not GTK_CHECK_VERSION(2,4,0) */
          gtk_toolbar_append_space (GTK_TOOLBAR (toolbar));
#endif /* not GTK_CHECK_VERSION(2,4,0) */
        }
#if GTK_CHECK_VERSION(2,4,0)
      gtk_toolbar_insert (GTK_TOOLBAR (toolbar), entries[i].tool_item, i);
#endif /* GTK_CHECK_VERSION(2,4,0) */
    }
  return toolbar;
}


/*  ja:ツールバーを検索する
    entries,ツールバー構造体
       path,パス
        RET,ツールアイテム                                                  */
GtkToolItem *
misc_find_toolbar (MiscCreateToolbarEntry *entries,
                   const gchar            *path)
{
  gint i;

  for (i = 0; entries[i].path; i++)
    if (g_strcmp (entries[i].path, path) == 0)
      return entries[i].tool_item;
  return NULL;
}


/******************************************************************************
*                                                                             *
* ja:低レベル関数群                                                           *
*                                                                             *
******************************************************************************/
/*  ja:mnemonicをテキストに変換する
    mnemonic,mnemonic
         RET,テキスト                                                       */
gchar *
misc_mnemonic_to_text (const gchar *mnemonic)
{
  gchar *text;
  gint i;

  if (!mnemonic)
    return NULL;
  text = g_strdup (mnemonic);
  for (i = 0; text[i] != '\0'; i++)
    if (text[i] ==  '(' && text[i + 1] ==  '_'
                    && g_ascii_isalpha (text[i + 2]) && text[i + 3] ==  ')')
      {
        g_memmove (text + i, text + i + 4, g_strlen (text + i) - 3);
        break;
      }
    else if (text[i] == '_' && g_ascii_isalpha (text[i + 1]))
      {
        g_memmove (text + i, text + i + 2, g_strlen (text + i) - 1);
        break;
      }
  return text;
}


/*  ja:スクロールバーを設定する
       scroll,スクロールウイジット
         func,シグナル関数
    func_data,データ
          min,最小値
          max,最大値
         page,ページ
          pos,位置                                                          */
void
misc_set_scroll_bar (GtkWidget     *scroll,
                     GtkSignalFunc  func,
                     gpointer       func_data,
                     const gint     min,
                     const gint     max,
                     const gint     page,
                     const gint     pos)
{
  GtkAdjustment *adjust;

  adjust = GTK_ADJUSTMENT (gtk_adjustment_new (pos, min, max, 1, page, page));
  gtk_range_set_adjustment (GTK_RANGE (scroll), adjust);
  if (func)
    g_signal_connect (G_OBJECT (adjust), "value-changed", func, func_data);
}


/*  ja:Windowの内容をスクロールする
    widget,ウイジット
        dx,X軸方向の移動
        dy,Y軸方向の移動
         x,スクロールする範囲の右上X座標
         y,スクロールする範囲の右上Y座標
     width,スクロールする範囲の幅
    height,スクロールする範囲の高さ                                         */
void
misc_scroll_window (GtkWidget  *widget,
                    const gint  dx,
                    const gint  dy,
                    const gint  x,
                    const gint  y,
                    const gint  width,
                    const gint  height)
{
  GdkGC *gc;
  GdkRectangle rc;

  if (ABS (dx) > width || ABS (dy) > height)
    {
      rc.x = x;
      rc.y = y;
      rc.width = width;
      rc.height = height;
      gtk_widget_draw (widget, &rc);
    }
  else if (dx != 0 || dy != 0)
    {
      gc = gdk_gc_new (widget->window);
      gdk_gc_set_exposures (gc, TRUE);
      gdk_window_copy_area (widget->window,
                            gc,
                            dx > 0 ? x + dx : x,
                            dy > 0 ? y + dy : y,
                            widget->window,
                            dx > 0 ? x : x - dx,
                            dy > 0 ? y : y - dy,
                            width - ABS (dx),
                            height - ABS (dy));
      gdk_gc_destroy (gc);
      if (dx != 0)
        {
          rc.x = dx > 0 ? x : x + width + dx;
          rc.y = y;
          rc.width = ABS (dx);
          rc.height = height;
          gtk_widget_draw (widget, &rc);
        }
      if (dy != 0)
        {
          rc.x = x;
          rc.y = dy > 0 ? y : y + height + dy;
          rc.width = width;
          rc.height = ABS (dy);
          gtk_widget_draw (widget, &rc);
        }
    }
}


/* ja:ESCが押されたとき */
gboolean
misc_dialog_key_press (GtkWidget   *widget,
                       GdkEventKey *event,
                       gpointer     user_data)
{
  if (event->keyval == GDK_Escape)
    gtk_dialog_response (GTK_DIALOG (widget), GTK_RESPONSE_NONE);
  return FALSE;
}


/******************************************************************************
*                                                                             *
* ja:数値文字列関数群                                                         *
*                                                                             *
******************************************************************************/
const static gchar hex[16]={'0', '1', '2', '3', '4', '5', '6', '7',
                            '8', '9', 'A', 'B', 'C', 'D', 'E', 'F'};


/*  ja:数値→文字列
    value,数値
    radix,基数
     wide,桁数
     flag,TRUE:符号あり,FALSE:符号なし
      RET,文字列                                                            */
gchar *
misc_str_from_val (const gint     value,
                   const gint     radix,
                   const gint     wide,
                   const gboolean flag)
{
  gchar c, *str = NULL;
  gint i = 0, j, t;

  t = value;
  if (value == 0)
    {
      str = g_malloc (sizeof (gchar));
      str[i++] = hex[0];
    }
  else
    {
      if (flag && value < 0)
        {
          while (t != 0)
            {
              str = g_realloc (str, (i + 1) * sizeof (gchar));
              str[i++] = hex[ABS (t % radix)];
              t /= radix;
            }
        }
      else
        {
          while (t != 0)
            {
              str = g_realloc (str, (i + 1) * sizeof (gchar));
              str[i++] = hex[(guint)t % radix];
              t = (guint)t / radix;
            }
        }
    }
  str = g_realloc (str, (i + 1) * sizeof (gchar));
  str[i] = '\0';
  for (j = 0; j < i / 2; j++)
    {
      c = str[j];
      str[j] = str[i - j - 1];
      str[i - j - 1] = c;
    }
  if (flag && value < 0)
    {
      for (j = i; j >= 0; j--)
        str[j + 1] = str[j];
      str[0] = '-';
      i++;
    }
  if (i < ABS (wide))
    {
      str = g_realloc (str, (ABS (wide) + 1) * sizeof (gchar));
      for (j = i; j >= 0; j--)
        str[j + ABS (wide) - i] = str[j];
      if (wide > 0)
        {
          for (j = 0; j < wide - i; j++)
            str[j] = ' ';
        }
      else
        {
          for (j = 0; j < -wide - i; j++)
            str[j] = hex[0];
          if (str[j] == '-')
            {
              str[0] = '-';
              str[j] = hex[0];
            }
        }
    }
  return str;
}


/*  ja:文字列→数値
    value,数値
      str,文字列
    radix,基数
     flag,TRUE:符号あり,FALSE:符号なし
      RET,TRUE:正常終了,FALSE:エラー                                        */
gboolean
misc_str_to_val (gint           *value,
                 const gchar    *str,
                 const gint      radix,
                 const gboolean  flag)
{
  gchar c;
  gint i, j, t;

  *value = 0;
  for (i = 0; str[i] != '\0' && str[i] == ' '; i++);
  if (str[i] == '\0')
    return FALSE;
  if (flag && str[i] == '-')
    {
      i++;
      while (str[i] != '\0')
        {
          t = *value;
          *value *= radix;
          c = g_ascii_toupper (str[i]);
          for (j = 0; j < radix; j++)
            if (hex[j] == c)
              break;
          *value += j;
          if (j == radix || *value < t)
            {
              *value = t;
              return FALSE;
            }
          i++;
        }
      if (*value < 0)
        {
          *value=0;
          return FALSE;
        }
      *value =- *value;
    }
  else
    {
      while (str[i] != '\0')
        {
          t = *value;
          *value *= radix;
          c = g_ascii_toupper (str[i]);
          for (j = 0; j < radix; j++)
            if (hex[j] == c)
              break;
          *value += j;
          if (j == radix || *value < t)
            {
              *value = t;
              return FALSE;
            }
          i++;
        }
    }
  return TRUE;
}


/*  ja:数値→文字列
     value,数値
    divide,除算値
       RET,文字列                                                           */
gchar *
misc_str_from_float (const gint value,
                     const gint divide)
{
    gchar *text = NULL;
    gint i = 0, n, t;

#ifdef G_HAVE_GINT64
  n = t = (gint64)value * 100000 / divide;
#else /* not G_HAVE_GINT64 */
  n = t = value * 100000 / divide;
#endif /* not G_HAVE_GINT64 */
  if (t == 0)
    {
      text = g_malloc (sizeof (gchar) * 4);
      text[i++] = hex[0];
    }
  else
    {
      t = ABS (t);
      while (t > 0)
        {
          text = g_realloc (text, (i + 4) * sizeof (gchar));
          text[i++] = hex[t % 10];
          t /= 10;
        }
    }
  text[i] = '\0';
  g_strreverse (text);
  if (g_strlen (text) < 6)
    {
      g_memmove (text + 2, text, (i + 1) * sizeof (gchar));
      text[0] = '0';
      text[1] = '.';
      i++;
    }
  else
    {
      g_memmove (text + i - 4, text + i - 5, sizeof (gchar) * 6);
      text[i - 5] = '.';
    }
  while (text[i] == '0')
    text[i--] = '\0';
  if (text[i] == '.')
    text[i--] = '\0';
  if (n >= 0)
    {
      text = g_realloc (text, (i + 2) * sizeof (gchar));
    }
  else
    {
      text = g_realloc (text, (i + 3) * sizeof (gchar));
      g_memmove (text + 1, text, (i + 2) * sizeof (gchar));
      text[0] = '-';
    }
  return text;
}


/*  ja:文字列→数値
     value,数値
    divide,除算値
      text,文字列
       RET,TRUE:正常終了,FALSE:エラー                                       */
gboolean
misc_str_to_float (gint        *value,
                   gint        *divide,
                   const gchar *text)
{
  gchar *p;
  gint i, j, t;

  *value = 0;
  *divide = 1;
  p = g_strdup (text);
  for (i = j = 0; p[i] != '\0'; i++)
    if (p[i] != ' ')
      p[j++] = p[i];
  if (j <= 0)
    return FALSE;
  p[j] = '\0';
  for (i = 0; p[i] != '\0'; i++)
    if (p[i] == '.')
      break;
  if (p[i] != '\0')
    {
      for (j = g_strlen (p) - 1; j > 0; j--)
        if (p[j] == '0')
          p[j] = '\0';
        else
          break;
      while (p[++i] != '\0')
        {
          p[i-1] = p[i];
          *divide *= 10;
        }
      p[i - 1] = '\0';
    }
  if (p[0] == '-' && p[1] == '\0')
    {
      g_free (p);
      return FALSE;
    }
  i = p[0] == '-' ? 1 : 0;
  while (p[i] != '\0')
    {
      t = *value;
      *value *= 10;
      for (j = 0; j < 10; j++)
        if (hex[j] == p[i])
          break;
      *value += j;
      if (j == 10 || *value < t)
        {
          *value = t;
          g_free (p);
          return FALSE;
        }
      i++;
  }
  while (*value % 2 == 0 && *divide % 2 == 0)
    *value /= 2, *divide /= 2;
  while (*value % 5 == 0 && *divide % 5 == 0)
    *value /= 5, *divide /= 5;
  *value *= (p[0] == '-' ? -1 : 1);
  g_free (p);
  return TRUE;
}


/*  ja:数列→文字列
    array,数列
     size,要素数
     bits,ビット数
    radix,基数
     wide,桁数
     flag,TRUE:符号あり,FALSE:符号なし
      RET,文字列                                                            */
gchar *
misc_str_from_array (gconstpointer  array,
                     const gsize    size,
                     const gint     bits,
                     const gint     radix,
                     const gint     wide,
                     const gboolean flag)
{
  gchar *p, *text = NULL;
  gint i, leng, length = 0;

  if (!array || size <= 0
                    || (bits != 0 && bits != 8 && bits != 16 && bits != 32))
    return NULL;
  for (i = 0; i < size; i++)
    {
      switch (bits)
        {
          case 8:
            p = flag
                ? misc_str_from_val (((gint8 *)array)[i],  radix, wide, flag)
                : misc_str_from_val (((guint8 *)array)[i], radix, wide, flag);
            break;
          case 16:
            p = flag
                ? misc_str_from_val (((gint16 *)array)[i],  radix, wide, flag)
                : misc_str_from_val (((guint16 *)array)[i], radix, wide, flag);
            break;
          case 32:
            p = flag
                ? misc_str_from_val (((gint32 *)array)[i],  radix, wide, flag)
                : misc_str_from_val (((guint32 *)array)[i], radix, wide, flag);
          default:
            p = flag
                ? misc_str_from_val (((gint *)array)[i],  radix, wide, flag)
                : misc_str_from_val (((guint *)array)[i], radix, wide, flag);
            break;
        }
      leng = g_strlen (p);
      text = g_realloc (text, (length + leng + 1) * sizeof (gchar));
      g_memmove (text + length, p, g_strlen (p));
      g_free (p);
      length += leng;
      text[length++] = ' ';
    }
  text[length - 1] = '\0';
  return text;
}


/*  ja:文字列→数列
     size,要素数
     bits,ビット数
     text,文字列
    radix,基数
     flag,TRUE:符号あり,FALSE:符号なし
      RET,数列                                                              */
gpointer
misc_str_to_array (gsize          *size,
                   const gint      bits,
                   const gchar    *text,
                   const gint      radix,
                   const gboolean  flag)
{
  gchar *p;
  gpointer array = NULL;
  gint i, j, value;

  if (!size)
    return NULL;
  *size = 0;
  if (!text || (bits != 0 && bits != 8 && bits != 16 && bits != 32))
    return NULL;
  p = g_malloc ((g_strlen (text) + 2) * sizeof (gchar));
  g_strcpy (p, text);
  for (i = 0; p[i] != '\0'; i++)
    {
      for (j = 0; j < radix; j++)
        if (p[i] == hex[j])
          break;
      if (j >= radix)
        p[i] = '\0';
    }
  p[i + 1] = '\0';
  i = 0;
  while (p[i] != '\0')
    {
      if (!misc_str_to_val (&value, p + i, radix, flag))
        {
          g_free (p);
          g_free (array);
          *size = 0;
          return NULL;
        }
      switch (bits)
        {
          case 8:
            array = g_realloc (array, (*size + 1) * sizeof (guint8));
            ((guint8 *)array)[(*size)++] = value;
            break;
          case 16:
            array = g_realloc (array, (*size + 1) * sizeof (guint16));
            ((guint16 *)array)[(*size)++] = value;
            break;
          case 32:
            array = g_realloc (array, (*size + 1) * sizeof (guint32));
            ((guint32 *)array)[(*size)++] = value;
          default:
            array = g_realloc (array, (*size + 1) * sizeof (guint));
            ((guint *)array)[(*size)++] = value;
        }
      i += g_strlen (p + i) + 1;
    }
  g_free (p);
  return array;
}
