/*
    Text maid
    copyright (c) 1998-2008 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include <gdk-pixbuf/gdk-pixbuf.h>
#include "abort.h"
#include "charset.h"
#include "edit.h"
#include "print.h"
#ifdef G_OS_WIN32
# include "orz/orzmdi.h"
# include <gdk/gdkwin32.h>
#else /* not G_OS_WIN32 */
# include "misc/fileio.h"
#endif /* not G_OS_WIN32 */


PrintConfig printer;                        /* ja:印刷の設定 */


/******************************************************************************
*                                                                             *
* ja:印刷関数群                                                               *
*                                                                             *
******************************************************************************/
#ifdef G_OS_WIN32
BOOL CALLBACK
PrintProc (HDC hdc,
           int nError)
{
  return abort_break;
}
#endif /* G_OS_WIN32 */


/*  印刷
     tmaid,ウインドウ情報
    prncfg,印刷情報
       RET,TRUE:正常終了,FALSE:エラー                                       */
gboolean
print_out (TmaidWindow *tmaid,
           PrintConfig *prncfg)
{
  guchar *c;
  gint i, x, y, sx, sy, data_pos, end_pos, line, width, channels, rowstride;
  LineBuffer *p;
  GdkGC *gc;
  GdkPixmap *pixmap;
  GdkPixbuf *pixbuf;
  GdkPoint start, end;
#ifdef G_OS_WIN32
  gchar *name, *utf8str;
  HBITMAP hBitmap;
  HDC hDC;
  DOCINFO di;
#else /* not G_OS_WIN32 */
  FILE *fp;
#endif /* not G_OS_WIN32 */

  sx = prncfg->column * prncfg->font_width;
#ifdef G_OS_WIN32
  sy = sx * prncfg->height / prncfg->width;
#else /* not G_OS_WIN32 */
  sy = sx * (prncfg->height - prncfg->left - prncfg->right)
                            / (prncfg->width - prncfg->top - prncfg->bottom);
#endif /* not G_OS_WIN32 */
  if (tmaid->select.x < 0 || !prncfg->select)
    {
      /* ja:選択範囲なし、すべて印刷 */
      start.x = start.y = 0;
      end.x = edit_get_width (tmaid, tmaid->max - 1);
      end.y = tmaid->max - 1;
    }
  else if (tmaid->select.y < tmaid->cursor.y
                                        || (tmaid->select.y == tmaid->cursor.y
                                        && tmaid->select.x < tmaid->cursor.x))
    {
      start = tmaid->select;
      end = tmaid->cursor;
    }
  else
    {
      start = tmaid->cursor;
      end = tmaid->select;
    }
#ifdef G_OS_WIN32
  hDC = CreateCompatibleDC (prncfg->pd.hDC);
  if (!hDC)
    {
      GtkWidget *dialog;

      dialog = gtk_message_dialog_new (GTK_WINDOW (window),
                            GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
                            GTK_MESSAGE_WARNING, GTK_BUTTONS_OK,
                            "CreateCompatibleDC");
      g_signal_connect (G_OBJECT (dialog), "key-press-event",
                                    G_CALLBACK (misc_dialog_key_press), NULL);
      gtk_dialog_run (GTK_DIALOG (dialog));
      gtk_widget_destroy (dialog);
      DeleteDC (prncfg->pd.hDC);
      return FALSE;
    }
  if (SetAbortProc (prncfg->pd.hDC, PrintProc) <= 0)
    {
      GtkWidget *dialog;

      dialog = gtk_message_dialog_new (GTK_WINDOW (window),
                            GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
                            GTK_MESSAGE_WARNING, GTK_BUTTONS_OK,
                            "SetAbortProc");
      g_signal_connect (G_OBJECT (dialog), "key-press-event",
                                    G_CALLBACK (misc_dialog_key_press), NULL);
      gtk_dialog_run (GTK_DIALOG (dialog));
      gtk_widget_destroy (dialog);
      DeleteDC (hDC);
      DeleteDC (prncfg->pd.hDC);
      return FALSE;
    }
  g_memset (&di, 0, sizeof (DOCINFO));
  di.cbSize = sizeof (DOCINFO);
#if GLIB_CHECK_VERSION(2,6,0)
  utf8str = g_filename_display_name (orz_mdi_get_file (ORZ_MDI (mdi),
                        orz_mdi_get_page_from_data (ORZ_MDI (mdi), tmaid)));
#else /* not GLIB_CHECK_VERSION(2,6,0) */
  utf8str = g_filename_to_utf8 (orz_mdi_get_file (ORZ_MDI (mdi),
                        orz_mdi_get_page_from_data (ORZ_MDI (mdi), tmaid)),
                                                        -1, NULL, NULL, NULL);
#endif /* not GLIB_CHECK_VERSION(2,6,0) */
# ifdef UNICODE
  name = g_utf8_to_utf16 (utf8str, -1, NULL, NULL, NULL);
# else /* not UNICODE */
  name = g_locale_from_utf8 (utf8str, -1, NULL, NULL, NULL);
# endif /* not UNICODE */
  g_free (utf8str);
  di.lpszDocName = name ? name : "Text maid";
  if (StartDoc (prncfg->pd.hDC, &di) <= 0)
    {
      GtkWidget *dialog;

      dialog = gtk_message_dialog_new (GTK_WINDOW (window),
                            GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
                            GTK_MESSAGE_WARNING, GTK_BUTTONS_OK,
                            "StartDoc");
      g_signal_connect (G_OBJECT (dialog), "key-press-event",
                                    G_CALLBACK (misc_dialog_key_press), NULL);
      gtk_dialog_run (GTK_DIALOG (dialog));
      gtk_widget_destroy (dialog);
      DeleteDC (hDC);
      DeleteDC (prncfg->pd.hDC);
      return FALSE;
    }
  g_free (name);
#else /* not G_OS_WIN32 */
# ifdef G_OS_UNIX
  if (prncfg->mode)
    {
      /* ja:プリンタ */
      fp = popen (prncfg->printer, "w");
      if (!fp)
        {
          GtkWidget *dialog;

          dialog = gtk_message_dialog_new (GTK_WINDOW (window),
                            GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
                            GTK_MESSAGE_WARNING, GTK_BUTTONS_OK,
                            _("Can not open printer"));
          g_signal_connect (G_OBJECT (dialog), "key-press-event",
                                    G_CALLBACK (misc_dialog_key_press), NULL);
          gtk_dialog_run (GTK_DIALOG (dialog));
          gtk_widget_destroy (dialog);
          return FALSE;
        }
    }
  else
    {
# endif /* G_OS_UNIX */
      /* ja:ファイル */
      fp = fopen (prncfg->file, "wt");
      if (!fp)
        {
          GtkWidget *dialog;

          dialog = gtk_message_dialog_new (GTK_WINDOW (window),
                            GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
                            GTK_MESSAGE_WARNING, GTK_BUTTONS_OK,
                            _("Can not open file"));
          g_signal_connect (G_OBJECT (dialog), "key-press-event",
                                    G_CALLBACK (misc_dialog_key_press), NULL);
          gtk_dialog_run (GTK_DIALOG (dialog));
          gtk_widget_destroy (dialog);
          return FALSE;
        }
# ifdef G_OS_UNIX
    }
# endif /* G_OS_UNIX */
  fputs ("%!PS-Adobe-3.0\n", fp);
#endif /* not G_OS_WIN32 */
  pixmap = gdk_pixmap_new (window->window, sx, sy, -1);
  pixbuf = gdk_pixbuf_new (GDK_COLORSPACE_RGB, FALSE, 8, sx, sy);
  gc = gdk_gc_new (pixmap);
  gdk_gc_set_line_attributes (gc, 1,
                                GDK_LINE_SOLID, GDK_CAP_BUTT, GDK_JOIN_MITER);
  /* ja:中断ボタン */
  abort_box_start ();
#if defined (G_OS_UNIX)
  for (i = 0; i < prncfg->mode ? prncfg->copies : 1; i++)
#elif defined (G_OS_WIN32)
  for (i = 0; i < prncfg->copies; i++)
#endif /* defined (G_OS_WIN32) */
    {
      p = edit_get_line_buf (&tmaid->start, &tmaid->off, start.y);
      end_pos = edit_get_data_pos (tmaid,end.x, end.y, FALSE);
      line = start.y;
      data_pos = edit_get_data_pos (tmaid, start.x, start.y, FALSE);
      while (line <= end.y)
        {
          /* ja:ページスタート */
          /* ja:背景 */
          gdk_gc_set_foreground (gc, (tmaid->ft.syscol
                                    ? system_color : tmaid->ft.color) + 11);
          gdk_draw_rectangle (pixmap, gc, TRUE, 0, 0, sx, sy);
          gdk_gc_set_foreground (gc, (tmaid->ft.syscol
                                    ? system_color : tmaid->ft.color) + 10);
          x = y = 0;
          while (y <= sy - tmaid->font_height && line <= end.y)
            {
              while (gtk_events_pending ())
                gtk_main_iteration ();
              while (x < sx)
                {
                  if (data_pos >= (line == end.y ? end_pos : p->length))
                    {
                      /* ja:行の終わり */
                      data_pos = 0;
                      line++;
                      p = p->next;
                      if (!p || !p->margin)
                        break;
                    }
                  if (p->text[data_pos] == '\t')
                    width = (x / tmaid->font_width / tmaid->ft.tab + 1)
                                    * tmaid->ft.tab * tmaid->font_width - x;
                  else if (data_pos + charset_length (p->text[data_pos])
                                                                <= p->length)
                    width = tmaid->font_width * charset_width
                                        (tmaid->layout, p->text + data_pos,
                                        tmaid->font_width, tmaid->font_buf);
                  else
                    width = tmaid->font_width;
                  if (x > 0 && sx < x + width)
                    break;
                  if (p->text[data_pos] == '\t')
                    {
                      /* ja:タブ */
                      data_pos++;
                    }
                  else if (data_pos + charset_length (p->text[data_pos])
                                                                <= p->length)
                    {
                      /* ja:半角/全角 */
                      pango_layout_set_text (tmaid->layout,
                                        p->text + data_pos,
                                        charset_length (p->text[data_pos]));
                      gdk_draw_layout (pixmap, gc, x, y, tmaid->layout);
                      data_pos += charset_length (p->text[data_pos]);
                    }
                  else
                    {
                      /* ja:強制半角 */
                      pango_layout_set_text (tmaid->layout,
                                                        p->text + data_pos, 1);
                      gdk_draw_layout (pixmap, gc, x, y, tmaid->layout);
                      data_pos++;
                    }
                  x += width;
                }
              x = 0;
              y += tmaid->font_height;
            }
#ifdef G_OS_WIN32
          hBitmap = SelectObject (hDC, (HBITMAP)GDK_WINDOW_HWND (pixmap));
          if (StartPage (prncfg->pd.hDC) <= 0)
            {
              GtkWidget *dialog;

              hBitmap = SelectObject (hDC, hBitmap);
              /* ja:中断ボタン */
              abort_box_end ();
              dialog = gtk_message_dialog_new (GTK_WINDOW (window),
                            GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
                            GTK_MESSAGE_WARNING, GTK_BUTTONS_OK,
                            "StartPage");
              g_signal_connect (G_OBJECT (dialog), "key-press-event",
                                    G_CALLBACK (misc_dialog_key_press), NULL);
              gtk_dialog_run (GTK_DIALOG (dialog));
              gtk_widget_destroy (dialog);
              goto loop;
            }
          if (!StretchBlt (prncfg->pd.hDC, 0, 0, prncfg->width, prncfg->height,
                                                hDC, 0, 0, sx, sy, SRCCOPY))
            {
              GtkWidget *dialog;

              EndPage (prncfg->pd.hDC);
              /* ja:中断ボタン */
              abort_box_end ();
              dialog = gtk_message_dialog_new (GTK_WINDOW (window),
                            GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
                            GTK_MESSAGE_WARNING, GTK_BUTTONS_OK,
                            "StretchBlt");
              g_signal_connect (G_OBJECT (dialog), "key-press-event",
                                    G_CALLBACK (misc_dialog_key_press), NULL);
              gtk_dialog_run (GTK_DIALOG (dialog));
              gtk_widget_destroy (dialog);
              hBitmap = SelectObject (hDC, hBitmap);
              goto loop;
            }
          if (EndPage (prncfg->pd.hDC) <= 0)
            {
              GtkWidget *dialog;

              hBitmap = SelectObject (hDC, hBitmap);
              /* ja:中断ボタン */
              abort_box_end ();
              dialog = gtk_message_dialog_new (GTK_WINDOW (window),
                            GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
                            GTK_MESSAGE_WARNING, GTK_BUTTONS_OK,
                            "EndPage");
              g_signal_connect (G_OBJECT (dialog), "key-press-event",
                                    G_CALLBACK (misc_dialog_key_press), NULL);
              gtk_dialog_run (GTK_DIALOG (dialog));
              gtk_widget_destroy (dialog);
              goto loop;
            }
          hBitmap = SelectObject (hDC, hBitmap);
#else /* not G_OS_WIN32 */
          fprintf (fp,
                "gsave\n"
                "%d %d translate\n"
                "%d %d scale\n"
                "%d %d 8 [%d 0 0 %d 0 0]\n"
                "{currentfile %d string readhexstring pop}\n"
                "image\n",
                prncfg->left, prncfg->bottom,
                prncfg->width - prncfg->left - prncfg->right,
                prncfg->height - prncfg->top - prncfg->bottom,
                sx, sy, sx, sy, sx);
          gdk_pixbuf_get_from_drawable (pixbuf, pixmap,
                            gdk_colormap_get_system (), 0, 0, 0, 0, sx, sy);
          channels = gdk_pixbuf_get_n_channels (pixbuf);
          rowstride = gdk_pixbuf_get_rowstride (pixbuf);
          c = gdk_pixbuf_get_pixels (pixbuf);
          i = 0;
          switch (channels)
            {
              case 1:
                for (y = sy - 1; y >= 0; y--)
                  {
                    for (x = 0; x < sx; x++)
                      fprintf (fp, "%02x", c[x * channels + y * rowstride]);
                    fputc ('\n', fp);
                  }
                break;
              case 3:
              case 4:
                for (y = sy - 1;y >= 0; y--)
                  {
                    for (x = 0; x < sx; x++)
                      fprintf (fp, "%02x",
                                    (c[x * channels + y * rowstride] * 77
                                    +c[x * channels + y * rowstride + 1] * 150
                                    +c[x * channels + y * rowstride + 2] * 29)
                                                                        / 256);
                    fputc ('\n', fp);
                  }
                break;
              default:
                for (y = 0; y < sy; y++)
                  {
                    for (x = 0; x < sx; x++)
                      fputs ("ff", fp);
                    fputc ('\n', fp);
                  }
            }
          fputs ("grestore\nshowpage\n", fp);
#endif /* not G_OS_WIN32 */
          if (!abort_break)
            {
              /* ja:中断ボタン */
              abort_box_end ();
              goto loop;
            }
        }
    }
  /* ja:中断ボタン */
  abort_box_end ();
  loop:
  /* ja:終了処理 */
  g_object_unref (gc);
  g_object_unref (pixbuf);
  g_object_unref (pixmap);
#ifdef G_OS_WIN32
  if (EndDoc (prncfg->pd.hDC) <= 0)
    {
      GtkWidget *dialog;

      dialog = gtk_message_dialog_new (GTK_WINDOW (window),
                            GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
                            GTK_MESSAGE_WARNING, GTK_BUTTONS_OK,
                            "EndDoc");
      g_signal_connect (G_OBJECT (dialog), "key-press-event",
                                    G_CALLBACK (misc_dialog_key_press), NULL);
      gtk_dialog_run (GTK_DIALOG (dialog));
      gtk_widget_destroy (dialog);
      DeleteDC (hDC);
      DeleteDC (prncfg->pd.hDC);
      return FALSE;
    }
  if (!DeleteDC (prncfg->pd.hDC))
    {
      GtkWidget *dialog;

      dialog = gtk_message_dialog_new (GTK_WINDOW (window),
                            GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
                            GTK_MESSAGE_WARNING, GTK_BUTTONS_OK,
                            "DeleteDC");
      g_signal_connect (G_OBJECT (dialog), "key-press-event",
                                    G_CALLBACK (misc_dialog_key_press), NULL);
      gtk_dialog_run (GTK_DIALOG (dialog));
      gtk_widget_destroy (dialog);
      DeleteDC (hDC);
      return FALSE;
    }
  if (!DeleteDC (hDC))
    {
      GtkWidget *dialog;

      dialog = gtk_message_dialog_new (GTK_WINDOW (window),
                            GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
                            GTK_MESSAGE_WARNING, GTK_BUTTONS_OK,
                            "DeleteDC");
      g_signal_connect (G_OBJECT (dialog), "key-press-event",
                                    G_CALLBACK (misc_dialog_key_press), NULL);
      gtk_dialog_run (GTK_DIALOG (dialog));
      gtk_widget_destroy (dialog);
      return FALSE;
    }
#else /* not G_OS_WIN32 */
# ifdef G_OS_UNIX
  if (prncfg->mode) /* ja:プリンタ */
    pclose (fp);
  else              /* ja:ファイル */
# endif /* G_OS_UNIX */
    fclose (fp);
#endif /* not G_OS_WIN32 */
  return TRUE;
}


/******************************************************************************
*                                                                             *
* ja:印刷ダイアログ関数群                                                     *
*                                                                             *
******************************************************************************/
typedef struct _PrintDialog
{
  gint font_width, font_height, width, height;
  GtkWidget *dialog, *label_lines, *spin_column;
#ifndef G_OS_WIN32
  GtkWidget *button, *check, *entry_file, *label_width, *label_height;
  GtkWidget *spin_width, *spin_height;
  GtkWidget *spin_top, *spin_left, *spin_right, *spin_bottom;
# ifdef G_OS_UNIX
  GtkWidget *entry_printer, *label_copies, *radio, *spin_copies;
# endif /* G_OS_UNIX */
#endif /* not G_OS_WIN32 */
} PrintDialog;


#ifndef G_OS_WIN32
static struct
{
  gchar *name;
  const gint width, height;
} paper[] = { {"A3", 840, 1188}, {"A4", 594, 840}, {"A5", 420, 594},
              {"B4", 730, 1030}, {"B5", 515, 730}, {NULL, 0, 0}};
#endif /* not G_OS_WIN32 */


/* ja:OKボタンの表示/非表示を判定 */
static void
print_dialog_ok_sensitive (GtkWidget   *widget,
                           PrintDialog *prndlg)
{
  gint column, row;
#ifndef G_OS_WIN32
  gint width, height, left, top, right, bottom;
# ifdef G_OS_UNIX
  gboolean mode;
  gint copies;
# endif /* G_OS_UNIX */
#endif /* not G_OS_WIN32 */

  column = gtk_spin_button_get_value_as_int
                                    (GTK_SPIN_BUTTON (prndlg->spin_column));
#ifdef G_OS_WIN32
  row = column * prndlg->font_width * prndlg->height
                                    / (prndlg->width * prndlg->font_height);
  gtk_dialog_set_response_sensitive (GTK_DIALOG (prndlg->dialog),
                                GTK_RESPONSE_ACCEPT, column >= 1 && row >= 1);
#else /* not G_OS_WIN32 */
# ifdef G_OS_UNIX
  mode = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (prndlg->radio));
  copies = gtk_spin_button_get_value_as_int
                                    (GTK_SPIN_BUTTON (prndlg->spin_copies));
# endif /* G_OS_UNIX */
  width = gtk_spin_button_get_value_as_int
                                    (GTK_SPIN_BUTTON (prndlg->spin_width));
  height = gtk_spin_button_get_value_as_int
                                    (GTK_SPIN_BUTTON (prndlg->spin_height));
  top = gtk_spin_button_get_value_as_int
                                    (GTK_SPIN_BUTTON (prndlg->spin_top));
  left = gtk_spin_button_get_value_as_int
                                    (GTK_SPIN_BUTTON (prndlg->spin_left));
  right = gtk_spin_button_get_value_as_int
                                    (GTK_SPIN_BUTTON (prndlg->spin_right));
  bottom = gtk_spin_button_get_value_as_int
                                    (GTK_SPIN_BUTTON (prndlg->spin_bottom));
  width -= left + right;
  height -= top + bottom;
  row = left >= 0 && top >= 0 && right >= 0 && bottom >= 0
                                                    && width > 0 && height > 0
                                    ? column * prndlg->font_width * height
                                        / (width * prndlg->font_height) : 0;
  gtk_dialog_set_response_sensitive (GTK_DIALOG (prndlg->dialog),
                                                        GTK_RESPONSE_ACCEPT,
# ifdef G_OS_UNIX
        column > 0 && row > 0 && (mode || copies > 0)
        && g_strlen (gtk_entry_get_text
                                (GTK_ENTRY (mode ? prndlg->entry_printer
                                                 : prndlg->entry_file))) > 0);
# else /* not G_OS_UNIX */
        column > 0 && row > 0
        && g_strlen (gtk_entry_get_text (GTK_ENTRY (prndlg->entry_file))) > 0);
# endif /* not G_OS_UNIX */
#endif /* not G_OS_WIN32 */
}


/* ja:スピンボタンが変化した */
static void
print_dialog_changed_spin (GtkWidget   *widget,
                           PrintDialog *prndlg)
{
  gchar *text;
  gint column, row;
#ifndef G_OS_WIN32
  gint width, height, left, top, right, bottom;
#endif /* not G_OS_WIN32 */

  column = gtk_spin_button_get_value_as_int
                                    (GTK_SPIN_BUTTON (prndlg->spin_column));
#ifdef G_OS_WIN32
  row = column * prndlg->font_width * prndlg->height
                                    / (prndlg->width * prndlg->font_height);
#else /* not G_OS_WIN32 */
  width = gtk_spin_button_get_value_as_int
                                    (GTK_SPIN_BUTTON (prndlg->spin_width));
  height = gtk_spin_button_get_value_as_int
                                    (GTK_SPIN_BUTTON (prndlg->spin_height));
  top = gtk_spin_button_get_value_as_int
                                    (GTK_SPIN_BUTTON (prndlg->spin_top));
  left = gtk_spin_button_get_value_as_int
                                    (GTK_SPIN_BUTTON (prndlg->spin_left));
  right = gtk_spin_button_get_value_as_int
                                    (GTK_SPIN_BUTTON (prndlg->spin_right));
  bottom = gtk_spin_button_get_value_as_int
                                    (GTK_SPIN_BUTTON (prndlg->spin_bottom));
  width -= left + right;
  height -= top + bottom;
  row = left >= 0 && top >= 0 && right >= 0 && bottom >= 0
                                                    && width > 0 && height > 0
                                    ? column * prndlg->font_width * height
                                        / (width * prndlg->font_height) : 0;
#endif /* not G_OS_WIN32 */
  text = g_strdup_printf (_("%d Lines"), MAX (row, 0));
  gtk_label_set_text (GTK_LABEL (prndlg->label_lines), text);
  g_free (text);
  print_dialog_ok_sensitive (widget, prndlg);
}


#ifndef G_OS_WIN32
/* ja:コンボボックス */
static void
print_dialog_changed_combo (GtkWidget   *widget,
                            PrintDialog *prndlg)
{
# if GTK_CHECK_VERSION(2,4,0)
  gchar *name;
# else /* not GTK_CHECK_VERSION(2,4,0) */
  const gchar *name;
# endif /* not GTK_CHECK_VERSION(2,4,0) */
  gint i;

# if GTK_CHECK_VERSION(2,6,0)
  name = gtk_combo_box_get_active_text (GTK_COMBO_BOX (widget));
# elif GTK_CHECK_VERSION(2,4,0)
  GtkTreeIter iter;

  if (gtk_combo_box_get_active_iter (GTK_COMBO_BOX (widget), &iter))
    gtk_tree_model_get (gtk_combo_box_get_model (GTK_COMBO_BOX (widget)),
                                                        &iter, 0, &name, -1);
  else
    name = NULL;
# else /* not GTK_CHECK_VERSION(2,4,0) */
  name = gtk_entry_get_text (GTK_ENTRY (widget));
# endif /* not GTK_CHECK_VERSION(2,4,0) */
  for (i = 0; paper[i].name; i++)
    if (g_strcmp (paper[i].name, name) == 0)
      break;
# if GTK_CHECK_VERSION(2,4,0)
  g_free (name);
# endif /* GTK_CHECK_VERSION(2,4,0) */
  if (paper[i].name)
    {
      gtk_spin_button_set_value (GTK_SPIN_BUTTON (prndlg->spin_width),
                                                            paper[i].width);
      gtk_spin_button_set_value (GTK_SPIN_BUTTON (prndlg->spin_height),
                                                            paper[i].height);
      gtk_widget_set_sensitive (prndlg->label_width, FALSE);
      gtk_widget_set_sensitive (prndlg->label_height, FALSE);
      gtk_widget_set_sensitive (prndlg->spin_width, FALSE);
      gtk_widget_set_sensitive (prndlg->spin_height, FALSE);
    }
  else
    {
      /* ja:ユーザー設定 */
      gtk_widget_set_sensitive (prndlg->label_width, TRUE);
      gtk_widget_set_sensitive (prndlg->label_height, TRUE);
      gtk_widget_set_sensitive (prndlg->spin_width, TRUE);
      gtk_widget_set_sensitive (prndlg->spin_height, TRUE);
    }
}
#endif /* not G_OS_WIN32 */


/* ja:ラジオボタン */
#ifdef G_OS_UNIX
static void
print_dialog_toggled (GtkWidget   *widget,
                      PrintDialog *prndlg)
{
  gboolean result;

  result = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (prndlg->radio));
  gtk_widget_set_sensitive (prndlg->entry_printer, result);
  gtk_widget_set_sensitive (prndlg->entry_file, !result);
  gtk_widget_set_sensitive (prndlg->button, !result);
  gtk_widget_set_sensitive (prndlg->label_copies, result);
  gtk_widget_set_sensitive (prndlg->spin_copies, result);
  print_dialog_ok_sensitive (widget, prndlg);
}
#endif /* G_OS_UNIX */


#ifndef G_OS_WIN32
/* ja:参照ボタンが押された */
static void
print_dialog_clicked_file (GtkWidget   *widget,
                           PrintDialog *prndlg)
{
  gchar *title;
  GtkStockItem stock_item;
  GtkWidget *dialog;
# if GTK_CHECK_VERSION(2,4,0)
  gchar *file;
  gint i;
  GtkFileFilter *filter;
# endif /* GTK_CHECK_VERSION(2,4,0) */

  gtk_stock_lookup (GTK_STOCK_SAVE_AS, &stock_item);
  title = misc_mnemonic_to_text (stock_item.label);
# if GTK_CHECK_VERSION(2,4,0)
  dialog = gtk_file_chooser_dialog_new (title,
                    GTK_WINDOW (prndlg->dialog), GTK_FILE_CHOOSER_ACTION_SAVE,
#ifdef G_OS_WIN32
                    GTK_STOCK_SAVE_AS, GTK_RESPONSE_ACCEPT,
                    GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
#else /* not G_OS_WIN32 */
                    GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                    GTK_STOCK_SAVE_AS, GTK_RESPONSE_ACCEPT,
#endif /* not G_OS_WIN32 */
                    NULL);
  file = g_filename_from_utf8 (gtk_entry_get_text
                    (GTK_ENTRY (prndlg->entry_file)), -1, NULL, NULL, NULL);
  if (g_file_test (file, G_FILE_TEST_EXISTS))
    {
      gtk_file_chooser_set_filename (GTK_FILE_CHOOSER (dialog), file);
    }
  else
    {
      gchar *dir, *path;

      dir = g_path_get_dirname (file);
      path = fileio_get_full_path (dir);
      g_free (dir);
      if (g_file_test (path, G_FILE_TEST_EXISTS))
        gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER (dialog), path);
      g_free (path);
    }
  g_free (file);
  /* ja:フィルタの設定 */
  for (i = 0; i < ftnum; i++)
    {
      gchar **ext;
      gint j;

      filter = gtk_file_filter_new ();
      gtk_file_filter_set_name (filter, ftype[i].text);
      ext = g_strsplit (ftype[i].ext, G_SEARCHPATH_SEPARATOR_S, 0);
      for (j = 0; ext[j]; j++)
        gtk_file_filter_add_pattern (filter, ext[j]);
      g_strfreev (ext);
      gtk_file_chooser_add_filter (GTK_FILE_CHOOSER (dialog), filter);
      if (i == findex)
        gtk_file_chooser_set_filter (GTK_FILE_CHOOSER (dialog), filter);
      g_object_set_data (G_OBJECT (filter), "user_data", GINT_TO_POINTER (i));
    }
  if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT)
    {
      file = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
      filter = gtk_file_chooser_get_filter (GTK_FILE_CHOOSER (dialog));
      findex = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (filter),
                                                                "user_data"));
      gtk_entry_set_text (GTK_ENTRY (prndlg->entry_file), file);
      g_free (file);
    }
# else /* not GTK_CHECK_VERSION(2,4,0) */
  dialog = gtk_file_selection_new (title);
  gtk_file_selection_set_filename (GTK_FILE_SELECTION (dialog),
                        gtk_entry_get_text (GTK_ENTRY (prndlg->entry_file)));
  gtk_file_selection_hide_fileop_buttons (GTK_FILE_SELECTION (dialog));
  if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_OK)
    gtk_entry_set_text (GTK_ENTRY (prndlg->entry_file),
                gtk_file_selection_get_filename (GTK_FILE_SELECTION (dialog)));
# endif /* not GTK_CHECK_VERSION(2,4,0) */
  g_free (title);
  gtk_widget_destroy (dialog);
}
#endif /* not G_OS_WIN32 */


/* ja:エントリー/スピンボタンでリターンが押された */
static void
print_dialog_activate (GtkWidget   *widget,
                       PrintDialog *prndlg)
{
#ifdef G_OS_WIN32
  if (gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (prndlg->spin_column))
                                                                        >= 1)
    gtk_dialog_response (GTK_DIALOG (prndlg->dialog), GTK_RESPONSE_ACCEPT);
#else /* not G_OS_WIN32 */
  gint left, top, right, bottom;
# ifdef G_OS_UNIX
  gboolean mode;
  gint copies;

  mode = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (prndlg->radio));
  copies = gtk_spin_button_get_value_as_int
                                    (GTK_SPIN_BUTTON (prndlg->spin_copies));
# endif /* G_OS_UNIX */
  top = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (prndlg->spin_top));
  left = gtk_spin_button_get_value_as_int
                                    (GTK_SPIN_BUTTON (prndlg->spin_left));
  right = gtk_spin_button_get_value_as_int
                                    (GTK_SPIN_BUTTON (prndlg->spin_right));
  bottom = gtk_spin_button_get_value_as_int
                                    (GTK_SPIN_BUTTON (prndlg->spin_bottom));
  if (
# ifdef G_OS_UNIX
    g_strlen (gtk_entry_get_text (GTK_ENTRY (mode ? prndlg->entry_printer
                                                  : prndlg->entry_file))) > 0
    && (mode || copies > 0)
# else /* not G_OS_UNIX */
    g_strlen (gtk_entry_get_text (GTK_ENTRY (prndlg->entry_file))) > 0
# endif /* not G_OS_UNIX */
    && gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (prndlg->spin_column))
                                                                        >= 1
    && left >= 0 && top >= 0 && right >= 0 && bottom >= 0
    && gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (prndlg->spin_width))
                                                                > left + right
    && gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (prndlg->spin_height))
                                                                > top + bottom)
    gtk_dialog_response (GTK_DIALOG (prndlg->dialog), GTK_RESPONSE_ACCEPT);
#endif /* not G_OS_WIN32 */
}


/*  ja:印刷ダイアログ表示
    prncfg,印刷情報
       RET,TRUE:OK,FALSE:Cancel                                             */
gboolean
print_dialog (PrintConfig *prncfg)
{
  gboolean result;
  gchar *title;
  GtkStockItem stock_item;
  GtkWidget *label_column, *hbox0, *hbox1, *vbox;
#ifndef G_OS_WIN32
  gint i;
# if ! GTK_CHECK_VERSION(2,4,0)
  GList *glist = NULL;
# endif /* not GTK_CHECK_VERSION(2,4,0) */
  GtkWidget *combo, *label_top, *label_left, *label_right, *label_bottom;
  GtkWidget *frame0, *frame1;
# ifdef G_OS_UNIX
  GtkWidget *radio;
# else /* not G_OS_UNIX */
  GtkWidget *label_file;
# endif /* not G_OS_UNIX */
#endif /* not G_OS_WIN32 */
  PrintDialog prndlg;

#ifdef G_OS_WIN32
  /* ja:プリンター */
  g_memset (&prncfg->pd, 0, sizeof (PRINTDLG));
  prncfg->pd.lStructSize = sizeof (PRINTDLG);
  prncfg->pd.hwndOwner = GDK_WINDOW_HWND (window->window);
  prncfg->pd.Flags = PD_HIDEPRINTTOFILE | PD_RETURNDC;
  if (prncfg->select)
    prncfg->pd.Flags |= PD_SELECTION;
  else
    prncfg->pd.Flags |= PD_NOSELECTION;
  prncfg->pd.nCopies = prncfg->copies;
  if (!PrintDlg (&prncfg->pd))
    return FALSE;
  prncfg->width = GetDeviceCaps (prncfg->pd.hDC, HORZRES);
  prncfg->height = GetDeviceCaps (prncfg->pd.hDC, VERTRES) * 24 / 25;
  if (prncfg->select)
    prncfg->select = (prncfg->pd.Flags & PD_SELECTION) != 0;
  prncfg->copies = prncfg->pd.nCopies;
#endif /* G_OS_WIN32 */

  prndlg.font_width = prncfg->font_width;
  prndlg.font_height = prncfg->font_height;
  prndlg.width = prncfg->width;
  prndlg.height = prncfg->height;
  /* ja:メインウインドウ */
  gtk_stock_lookup (GTK_STOCK_PRINT, &stock_item);
  title = misc_mnemonic_to_text (stock_item.label);
  prndlg.dialog = gtk_dialog_new_with_buttons
                            (title,
                            GTK_WINDOW (window),
                            GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
#ifdef G_OS_WIN32
                            GTK_STOCK_PRINT, GTK_RESPONSE_ACCEPT,
                            GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
#else /* not G_OS_WIN32 */
                            GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                            GTK_STOCK_PRINT, GTK_RESPONSE_ACCEPT,
#endif /* not G_OS_WIN32 */
                            NULL);
  g_free (title);
  g_signal_connect (G_OBJECT (prndlg.dialog), "key-press-event",
                                    G_CALLBACK (misc_dialog_key_press), NULL);
  /* ja:スピンボタン */
  prndlg.spin_column = gtk_spin_button_new (GTK_ADJUSTMENT (gtk_adjustment_new
                                (prncfg->column, 1, G_MAXINT, 1, 5, 0)), 0, 0);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (prndlg.spin_column), TRUE);
  g_signal_connect (G_OBJECT (prndlg.spin_column), "activate",
                                G_CALLBACK (print_dialog_activate), &prndlg);
  g_signal_connect (G_OBJECT (prndlg.spin_column), "changed",
                            G_CALLBACK (print_dialog_changed_spin), &prndlg);
  /* ja:ラベル */
  label_column = gtk_label_new_with_mnemonic (_("_Columns"));
  prndlg.label_lines = gtk_label_new ("");
  gtk_label_set_mnemonic_widget (GTK_LABEL (label_column), prndlg.spin_column);
#ifndef G_OS_WIN32
  /* ja:ボタン */
  prndlg.button = gtk_button_new_with_label (_("Browse..."));
  g_signal_connect (G_OBJECT (prndlg.button), "clicked",
                            G_CALLBACK (print_dialog_clicked_file), &prndlg);
  GTK_WIDGET_SET_FLAGS (prndlg.button, GTK_CAN_DEFAULT);
  /* ja:チェックボタン */
  prndlg.check = gtk_check_button_new_with_mnemonic (_("Print _Selection"));
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (prndlg.check),
                                                            prncfg->select);
  gtk_widget_set_sensitive (prndlg.check, prncfg->select);
  /* ja:エントリー */
  prndlg.entry_file = gtk_entry_new ();
  if (prncfg->file)
    gtk_entry_set_text (GTK_ENTRY (prndlg.entry_file), prncfg->file);
  g_signal_connect (G_OBJECT (prndlg.entry_file), "activate",
                                G_CALLBACK (print_dialog_activate), &prndlg);
  g_signal_connect (G_OBJECT (prndlg.entry_file), "changed",
                            G_CALLBACK (print_dialog_ok_sensitive), &prndlg);
  /* ja:スピンボタン */
  prndlg.spin_width = gtk_spin_button_new (GTK_ADJUSTMENT (gtk_adjustment_new
                                (prncfg->width, 1, G_MAXINT, 1, 5, 0)), 0, 0);
  prndlg.spin_height = gtk_spin_button_new (GTK_ADJUSTMENT (gtk_adjustment_new
                                (prncfg->height, 1, G_MAXINT, 1, 5, 0)), 0, 0);
  prndlg.spin_top = gtk_spin_button_new (GTK_ADJUSTMENT (gtk_adjustment_new
                                (prncfg->top, 0, G_MAXINT, 1, 5, 0)), 0, 0);
  prndlg.spin_left = gtk_spin_button_new (GTK_ADJUSTMENT (gtk_adjustment_new
                                (prncfg->left, 0, G_MAXINT, 1, 5, 0)), 0, 0);
  prndlg.spin_right = gtk_spin_button_new (GTK_ADJUSTMENT (gtk_adjustment_new
                                (prncfg->right, 0, G_MAXINT, 1, 5, 0)), 0, 0);
  prndlg.spin_bottom = gtk_spin_button_new (GTK_ADJUSTMENT (gtk_adjustment_new
                                (prncfg->bottom, 0, G_MAXINT, 1, 5, 0)), 0, 0);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (prndlg.spin_width), TRUE);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (prndlg.spin_height), TRUE);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (prndlg.spin_top), TRUE);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (prndlg.spin_left), TRUE);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (prndlg.spin_right), TRUE);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (prndlg.spin_bottom), TRUE);
  g_signal_connect (G_OBJECT (prndlg.spin_width), "activate",
                                G_CALLBACK (print_dialog_activate), &prndlg);
  g_signal_connect (G_OBJECT (prndlg.spin_height), "activate",
                                G_CALLBACK (print_dialog_activate), &prndlg);
  g_signal_connect (G_OBJECT (prndlg.spin_top), "activate",
                                G_CALLBACK (print_dialog_activate), &prndlg);
  g_signal_connect (G_OBJECT (prndlg.spin_left), "activate",
                                G_CALLBACK (print_dialog_activate), &prndlg);
  g_signal_connect (G_OBJECT (prndlg.spin_right), "activate",
                                G_CALLBACK (print_dialog_activate), &prndlg);
  g_signal_connect (G_OBJECT (prndlg.spin_bottom), "activate",
                                G_CALLBACK (print_dialog_activate), &prndlg);
  g_signal_connect (G_OBJECT (prndlg.spin_width), "changed",
                            G_CALLBACK (print_dialog_changed_spin), &prndlg);
  g_signal_connect (G_OBJECT (prndlg.spin_height), "changed",
                            G_CALLBACK (print_dialog_changed_spin), &prndlg);
  g_signal_connect (G_OBJECT (prndlg.spin_top), "changed",
                            G_CALLBACK (print_dialog_changed_spin), &prndlg);
  g_signal_connect (G_OBJECT (prndlg.spin_left), "changed",
                            G_CALLBACK (print_dialog_changed_spin), &prndlg);
  g_signal_connect (G_OBJECT (prndlg.spin_right), "changed",
                            G_CALLBACK (print_dialog_changed_spin), &prndlg);
  g_signal_connect (G_OBJECT (prndlg.spin_bottom), "changed",
                            G_CALLBACK (print_dialog_changed_spin), &prndlg);
  /* ja:ラベル */
  prndlg.label_width = gtk_label_new_with_mnemonic (_("_Width"));
  prndlg.label_height = gtk_label_new_with_mnemonic (_("_Height"));
  label_top = gtk_label_new_with_mnemonic (_("_Top"));
  label_left = gtk_label_new_with_mnemonic (_("_Left"));
  label_right = gtk_label_new_with_mnemonic (_("_Right"));
  label_bottom = gtk_label_new_with_mnemonic (_("_Bottom"));
  gtk_label_set_mnemonic_widget (GTK_LABEL (prndlg.label_width),
                                                        prndlg.spin_width);
  gtk_label_set_mnemonic_widget (GTK_LABEL (prndlg.label_height),
                                                        prndlg.spin_height);
  gtk_label_set_mnemonic_widget (GTK_LABEL (label_top), prndlg.spin_top);
  gtk_label_set_mnemonic_widget (GTK_LABEL (label_left), prndlg.spin_left);
  gtk_label_set_mnemonic_widget (GTK_LABEL (label_right), prndlg.spin_right);
  gtk_label_set_mnemonic_widget (GTK_LABEL (label_bottom), prndlg.spin_bottom);
# ifdef G_OS_UNIX
  /* ja:エントリー */
  prndlg.entry_printer = gtk_entry_new ();
  if (prncfg->printer)
    gtk_entry_set_text (GTK_ENTRY (prndlg.entry_printer), prncfg->printer);
  g_signal_connect (G_OBJECT (prndlg.entry_printer), "activate",
                                G_CALLBACK (print_dialog_activate), &prndlg);
  g_signal_connect (G_OBJECT (prndlg.entry_printer), "changed",
                            G_CALLBACK (print_dialog_ok_sensitive), &prndlg);
  /* ja:スピンボタン */
  prndlg.spin_copies = gtk_spin_button_new (GTK_ADJUSTMENT (gtk_adjustment_new
                                (prncfg->copies, 1, G_MAXINT, 1, 5, 0)), 0, 0);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (prndlg.spin_copies), TRUE);
  g_signal_connect (G_OBJECT (prndlg.spin_copies), "activate",
                                G_CALLBACK (print_dialog_activate), &prndlg);
  g_signal_connect (G_OBJECT (prndlg.spin_copies), "changed",
                            G_CALLBACK (print_dialog_changed_spin), &prndlg);
  /* ja:ラベル */
  prndlg.label_copies = gtk_label_new_with_mnemonic (_("C_opies"));
  gtk_label_set_mnemonic_widget (GTK_LABEL (prndlg.label_copies),
                                                        prndlg.spin_copies);
  /* ja:ラジオボタン */
  prndlg.radio = gtk_radio_button_new_with_mnemonic (NULL, _("_Printer"));
  radio = gtk_radio_button_new_with_mnemonic_from_widget
                                (GTK_RADIO_BUTTON (prndlg.radio), _("_File"));
  g_signal_connect (G_OBJECT (prndlg.radio), "toggled",
                                G_CALLBACK (print_dialog_toggled), &prndlg);
  g_signal_connect (G_OBJECT (radio), "toggled",
                                G_CALLBACK (print_dialog_toggled), &prndlg);
  if (!prncfg->mode)
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (radio), TRUE);
# else /* not G_OS_UNIX */
  /* ja:ラベル */
  label_file = gtk_label_new_with_mnemonic (_("_File"));
  gtk_label_set_mnemonic_widget (GTK_LABEL (label_file), prndlg.entry_file);
# endif /* not G_OS_UNIX */
  /* ja:コンボボックス */
# if GTK_CHECK_VERSION(2,4,0)
  combo = gtk_combo_box_new_text ();
  for (i = 0; paper[i].name; i++)
    gtk_combo_box_append_text (GTK_COMBO_BOX (combo), paper[i].name);
  gtk_combo_box_append_text (GTK_COMBO_BOX (combo), _("User Define"));
  g_signal_connect (G_OBJECT (combo), "changed",
                            G_CALLBACK (print_dialog_changed_combo), &prndlg);
# else /* not GTK_CHECK_VERSION(2,4,0) */
  for (i = 0; paper[i].name; i++)
    glist = g_list_append (glist, paper[i].name);
  glist = g_list_append (glist, _("User Define"));
  combo = gtk_combo_new ();
  gtk_combo_set_popdown_strings (GTK_COMBO (combo), glist);
  gtk_combo_set_value_in_list (GTK_COMBO (combo), TRUE, FALSE);
  gtk_editable_set_editable (GTK_EDITABLE (GTK_COMBO (combo)->entry), FALSE);
  g_signal_connect (G_OBJECT (GTK_COMBO (combo)->entry), "changed",
                            G_CALLBACK (print_dialog_changed_combo), &prndlg);
  g_list_free (glist);
# endif /* not GTK_CHECK_VERSION(2,4,0) */
  for (i = 0; paper[i].name; i++)
    if (paper[i].width == prncfg->width && paper[i].height == prncfg->height)
      break;
# if GTK_CHECK_VERSION(2,4,0)
  gtk_combo_box_set_active (GTK_COMBO_BOX (combo), i);
# else /* not GTK_CHECK_VERSION(2,4,0) */
  if (paper[i].name)
    gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (combo)->entry), paper[i].name);
  else
    gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (combo)->entry),
                                                            _("User Define"));
# endif /* not GTK_CHECK_VERSION(2,4,0) */

  /* ja:フレームとボックス */
  frame0 = gtk_frame_new (_("Paper"));
  vbox = gtk_vbox_new (FALSE, SPACING);
  gtk_container_set_border_width (GTK_CONTAINER (vbox), SPACING);
  gtk_box_pack_start (GTK_BOX (vbox), combo, TRUE, TRUE, 0);
  hbox0 = gtk_hbox_new (FALSE, SPACING);
  hbox1 = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox1), prndlg.label_width, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox1), prndlg.spin_width, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (hbox0), hbox1, TRUE, TRUE, 0);
  hbox1 = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox1), prndlg.label_height, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox1), prndlg.spin_height, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (hbox0), hbox1, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), hbox0, FALSE, FALSE, 0);
  gtk_container_add (GTK_CONTAINER (frame0), vbox);

  frame1 = gtk_frame_new (_("Margin"));
  vbox = gtk_vbox_new (FALSE, SPACING);
  gtk_container_set_border_width (GTK_CONTAINER (vbox), SPACING);
  hbox0 = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox0), label_top, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox0), prndlg.spin_top, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), hbox0, FALSE, FALSE, 0);
  hbox0 = gtk_hbox_new (FALSE, SPACING);
  hbox1 = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox1), label_left, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox1), prndlg.spin_left, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (hbox0), hbox1, TRUE, TRUE, 0);
  hbox1 = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox1), label_right, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox1), prndlg.spin_right, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (hbox0), hbox1, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), hbox0, FALSE, FALSE, 0);
  hbox0 = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox0), label_bottom, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox0), prndlg.spin_bottom, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), hbox0, FALSE, FALSE, 0);
  gtk_container_add (GTK_CONTAINER (frame1), vbox);
#endif /* not G_OS_WIN32 */

  /* ja:フレームとボックス */
  vbox = gtk_vbox_new (FALSE, SPACING);
  gtk_container_set_border_width (GTK_CONTAINER (vbox), SPACING);
  hbox0 = gtk_hbox_new (FALSE, SPACING);
  hbox1 = gtk_hbox_new (FALSE,0);
  gtk_box_pack_start (GTK_BOX (hbox1), prndlg.spin_column, TRUE, TRUE, 0);
  gtk_box_pack_end (GTK_BOX (hbox1), label_column, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox0), hbox1, TRUE, TRUE, 0);
  gtk_box_pack_end (GTK_BOX (hbox0), prndlg.label_lines, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), hbox0, FALSE, FALSE, 0);

#ifndef G_OS_WIN32
  gtk_box_pack_start (GTK_BOX (vbox), prndlg.check, FALSE, FALSE, 0);

# ifdef G_OS_UNIX
  hbox0 = gtk_hbox_new (FALSE, SPACING);
  gtk_box_pack_start (GTK_BOX (hbox0), prndlg.radio, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox0), prndlg.entry_printer, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), hbox0, FALSE, FALSE, 0);
# endif /* G_OS_UNIX */

  hbox0 = gtk_hbox_new (FALSE, SPACING);
# ifdef G_OS_UNIX
  gtk_box_pack_start (GTK_BOX (hbox0), radio, FALSE, FALSE, 0);
# else /* not G_OS_UNIX */
  gtk_box_pack_start (GTK_BOX (hbox0), label_file, FALSE, FALSE, 0);
# endif /* not G_OS_UNIX */
  hbox1 = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox1), prndlg.entry_file, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (hbox1), prndlg.button, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox0), hbox1, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), hbox0, FALSE, FALSE, 0);
#endif /* not G_OS_WIN32 */

#ifdef G_OS_UNIX
  hbox0 = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox0), prndlg.label_copies, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox0), prndlg.spin_copies, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), hbox0, FALSE, FALSE, 0);
#endif /* G_OS_UNIX */

#ifndef G_OS_WIN32
  gtk_box_pack_start (GTK_BOX (vbox), frame0, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), frame1, FALSE, FALSE, 0);
#endif /* not G_OS_WIN32 */

  gtk_container_add (GTK_CONTAINER (GTK_DIALOG (prndlg.dialog)->vbox), vbox);

  /* ja:表示 */
  print_dialog_changed_spin (NULL, &prndlg);
  gtk_widget_show_all (vbox);

  result = gtk_dialog_run (GTK_DIALOG (prndlg.dialog)) == GTK_RESPONSE_ACCEPT;
  if (result)
    {
#ifndef G_OS_WIN32
# ifdef G_OS_UNIX
      prncfg->mode = gtk_toggle_button_get_active
                                            (GTK_TOGGLE_BUTTON (prndlg.radio));
      prncfg->copies = gtk_spin_button_get_value_as_int
                                        (GTK_SPIN_BUTTON (prndlg.spin_copies));
      if (prncfg->mode)
        {
          g_free (prncfg->printer);
          prncfg->printer
            = g_strdup (gtk_entry_get_text (GTK_ENTRY (prndlg.entry_printer)));
        }
      else
        {
# endif /* G_OS_UNIX */
          g_free (prncfg->file);
          prncfg->file
            = g_strdup (gtk_entry_get_text (GTK_ENTRY (prndlg.entry_file)));
# ifdef G_OS_UNIX
        }
# endif /* G_OS_UNIX */
      prncfg->select = gtk_toggle_button_get_active
                                        (GTK_TOGGLE_BUTTON (prndlg.check));
      prncfg->width = gtk_spin_button_get_value_as_int
                                        (GTK_SPIN_BUTTON (prndlg.spin_width));
      prncfg->height = gtk_spin_button_get_value_as_int
                                        (GTK_SPIN_BUTTON (prndlg.spin_height));
      prncfg->top = gtk_spin_button_get_value_as_int
                                        (GTK_SPIN_BUTTON (prndlg.spin_top));
      prncfg->left = gtk_spin_button_get_value_as_int
                                        (GTK_SPIN_BUTTON (prndlg.spin_left));
      prncfg->right = gtk_spin_button_get_value_as_int
                                        (GTK_SPIN_BUTTON (prndlg.spin_right));
      prncfg->bottom = gtk_spin_button_get_value_as_int
                                        (GTK_SPIN_BUTTON (prndlg.spin_bottom));
#endif /* not G_OS_WIN32 */
      prncfg->column = gtk_spin_button_get_value_as_int
                                        (GTK_SPIN_BUTTON (prndlg.spin_column));
    }
  gtk_widget_destroy (prndlg.dialog);

#ifdef G_OS_WIN32
  if (!result)
    DeleteDC (prncfg->pd.hDC);
#endif /* G_OS_WIN32 */

  return result;
}
