/*
    Text maid for Windows
    copyright (c) 1998-2018 Kazuki Iwamoto https://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "general.h"
#include <commctrl.h>
#include "edit.h"
#include "find.h"
#include "replace.h"
#include "resource.h"
#include "valchr.h"
#include "wcommon.h"


/******************************************************************************
*                                                                             *
* ja:一般関数群                                                               *
*                                                                             *
******************************************************************************/
/*  ja:キャレットの描画
    hWnd,ウインドウ                                                         */
VOID
DrawCaret (HWND hWnd)
{
  LPTEXTWND ptw;

  ptw = (LPTEXTWND)GetWindowLongPtr (hWnd, GWLP_USERDATA);
  if (ptw)
    {
      int x, y;
      LPTSTR lpszFormat, lpszText;

      x = GetAlignPos (&ptw->lpStart, &ptw->nOff,
                        ptw->ptCursor.x, ptw->ptCursor.y, ptw->nTab, FALSE);
      y = ptw->ptCursor.y;
      lpszFormat = LoadText (GetModuleHandle (NULL),
                ptw->ptSelect.x < 0 ? IDS_STATUS_NOSEL : IDS_STATUS_SELECT);
      wasprintf (&lpszText, lpszFormat,
        x + 1, y + 1, ptw->ptSelect.x + 1, ptw->ptSelect.y + 1, x + 1, y + 1);
      MemoryFree (lpszFormat);
      StatusBar_SetText (hWndStat, 0, 0, lpszText);
      MemoryFree (lpszText);
      x = (x - ptw->ptTop.x) * ptw->nFontSize;
      y = (y - ptw->ptTop.y) * ptw->nFontSize * 2;
      if (x < 0 || ptw->siWnd.cx <= x || y < 0 || ptw->siWnd.cy <= y)
        x = y = -128;
      SetCaretPos (x, y);
      SetImePos (hWnd, x, y);
    }
  else
    {
      SetCaretPos (-128, -128);
    }
}


/*  ja:メニューを設定する
    ptw,TXTウインドウ情報                                                   */
VOID
SetMenuBar (LPTEXTWND ptw)
{
  int nMenu;
  HMENU hMenu;

  hMenu = GetMenu (hWndMain);
  nMenu = (GetMenuItemCount (hMenu) != MENUTOP);
  if (ptw)
    {
      EnableMenuItem (hMenu, nMenu + 1, MF_BYPOSITION | MF_ENABLED);
      EnableMenuItem (hMenu, nMenu + 2, MF_BYPOSITION | MF_ENABLED);
      EnableMenuItem (hMenu, nMenu + 4, MF_BYPOSITION | MF_ENABLED);
      EnableMenuItem (hMenu, CM_CLOSE, MF_ENABLED);
      EnableMenuItem (hMenu, CM_SAVE, MF_ENABLED);
      EnableMenuItem (hMenu, CM_SAVEAS, MF_ENABLED);
      EnableMenuItem (hMenu, CM_RELOAD, ptw->fCreate ? MF_GRAYED : MF_ENABLED);
      EnableMenuItem (hMenu, CM_PRINT, MF_ENABLED);
      EnableMenuItem (hMenu, CM_PROPERTY, MF_ENABLED);
      EnableMenuItem (hMenu, CM_UNDO, ptw->lpUndo ? MF_ENABLED : MF_GRAYED);
      EnableMenuItem (hMenu, CM_REDO, ptw->lpRedo ? MF_ENABLED : MF_GRAYED);
      EnableMenuItem (hMenu, CM_CUT, ptw->ptSelect.x >= 0
                                                    ? MF_ENABLED : MF_GRAYED);
      EnableMenuItem (hMenu, CM_COPY, ptw->ptSelect.x >= 0
                                                    ? MF_ENABLED : MF_GRAYED);
      EnableMenuItem (hMenu, CM_PASTE, fClipBoard ? MF_ENABLED : MF_GRAYED);
      EnableMenuItem (hMenu, CM_DELETE, ptw->ptSelect.x >= 0
                                                    ? MF_ENABLED : MF_GRAYED);
      EnableMenuItem (hMenu, CM_NEXT, ppszFind[0] ? MF_ENABLED : MF_GRAYED);
      ToolBar_EnableButton (hWndTool, CM_SAVE, TRUE);
      ToolBar_EnableButton (hWndTool, CM_PRINT, TRUE);
      ToolBar_EnableButton (hWndTool, CM_PROPERTY, TRUE);
      ToolBar_EnableButton (hWndTool, CM_UNDO, ptw->lpUndo != NULL);
      ToolBar_EnableButton (hWndTool, CM_REDO, ptw->lpRedo != NULL);
      ToolBar_EnableButton (hWndTool, CM_CUT, ptw->ptSelect.x >= 0);
      ToolBar_EnableButton (hWndTool, CM_COPY, ptw->ptSelect.x >= 0);
      ToolBar_EnableButton (hWndTool, CM_PASTE, fClipBoard);
      ToolBar_EnableButton (hWndTool, CM_DELETE, ptw->ptSelect.x >= 0);
      ToolBar_EnableButton (hWndTool, CM_FIND, TRUE);
      ToolBar_EnableButton (hWndTool, CM_REPLACE, TRUE);
    }
  else
    {
      EnableMenuItem (hMenu, nMenu + 1, MF_BYPOSITION | MF_GRAYED);
      EnableMenuItem (hMenu, nMenu + 2, MF_BYPOSITION | MF_GRAYED);
      EnableMenuItem (hMenu, nMenu + 4, MF_BYPOSITION | MF_GRAYED);
      EnableMenuItem (hMenu, CM_CLOSE, MF_GRAYED);
      EnableMenuItem (hMenu, CM_SAVE, MF_GRAYED);
      EnableMenuItem (hMenu, CM_SAVEAS, MF_GRAYED);
      EnableMenuItem (hMenu, CM_RELOAD, MF_GRAYED);
      EnableMenuItem (hMenu, CM_PRINT, MF_GRAYED);
      EnableMenuItem (hMenu, CM_PROPERTY, MF_GRAYED);
      ToolBar_EnableButton (hWndTool, CM_SAVE, FALSE);
      ToolBar_EnableButton (hWndTool, CM_PRINT, FALSE);
      ToolBar_EnableButton (hWndTool, CM_PROPERTY, FALSE);
      ToolBar_EnableButton (hWndTool, CM_UNDO, FALSE);
      ToolBar_EnableButton (hWndTool, CM_REDO, FALSE);
      ToolBar_EnableButton (hWndTool, CM_CUT, FALSE);
      ToolBar_EnableButton (hWndTool, CM_COPY, FALSE);
      ToolBar_EnableButton (hWndTool, CM_PASTE, FALSE);
      ToolBar_EnableButton (hWndTool, CM_DELETE, FALSE);
      ToolBar_EnableButton (hWndTool, CM_FIND, FALSE);
      ToolBar_EnableButton (hWndTool, CM_REPLACE, FALSE);
    }
  DrawMenuBar (hWndMain);
}


/*  ja:指定範囲を再描画する
         hWnd,ウインドウ
    lpptStart,開始
      lpptEnd,終了                                                          */
VOID
ClearSel (HWND    hWnd,
          LPPOINT lpptStart,
          LPPOINT lpptEnd)
{
  RECT rc;
  LPTEXTWND ptw;

  ptw = (LPTEXTWND)GetWindowLongPtr (hWnd, GWLP_USERDATA);
  if (lpptStart->y == lpptEnd->y)
    {
      /* ja:同じ行 */
      int nStart, nEnd;

      nStart = GetAlignPos (&ptw->lpStart, &ptw->nOff,
            min (lpptStart->x, lpptEnd->x), lpptStart->y, ptw->nTab, FALSE);
      nEnd = GetAlignPos (&ptw->lpStart, &ptw->nOff,
            max (lpptStart->x, lpptEnd->x), lpptStart->y, ptw->nTab, TRUE);
      rc.left = (nStart - ptw->ptTop.x) * ptw->nFontSize;
      rc.top = (lpptStart->y - ptw->ptTop.y) * ptw->nFontSize * 2;
      rc.right = (nEnd - ptw->ptTop.x + 1) * ptw->nFontSize;
      rc.bottom = rc.top + ptw->nFontSize * 2;
    }
  else
    {
      /* ja:違う行 */
      rc.left = 0;
      rc.top = (min (lpptStart->y, lpptEnd->y) - ptw->ptTop.y)
                                                        * ptw->nFontSize * 2;
      rc.right = ptw->siWnd.cx;
      rc.bottom = (max (lpptStart->y, lpptEnd->y) - ptw->ptTop.y + 1)
                                                        * ptw->nFontSize * 2;
    }
  if (rc.left < 0)
    rc.left = 0;
  if (rc.top < 0)
    rc.top = 0;
  if (rc.right > ptw->siWnd.cx)
    rc.right = ptw->siWnd.cx;
  if (rc.bottom > ptw->siWnd.cy)
    rc.bottom = ptw->siWnd.cy;
  if (rc.left < rc.right && rc.top < rc.bottom)
    InvalidateRect (hWnd, &rc, TRUE);
}


/*  ja:移動する
       hWnd,ウインドウ
    lpptTop,古い左上の座標                                                  */
VOID
MoveTextWindow (HWND    hWnd,
                LPPOINT lpptTop)
{
  LPTEXTWND ptw;

  ptw = (LPTEXTWND)GetWindowLongPtr (hWnd, GWLP_USERDATA);
  if (ptw->ptTop.x != lpptTop->x || ptw->ptTop.y != lpptTop->y)
    {
      int sx, sy;

      sx = max (ptw->siWnd.cx / ptw->nFontSize, 1);
      sy = max (ptw->siWnd.cy / (ptw->nFontSize * 2), 1);
      SetScrollBar (hWnd, SB_HORZ, 0, GetWidthMax (ptw), sx, ptw->ptTop.x);
      SetScrollBar (hWnd, SB_VERT, 0, ptw->nMax - 1, sy, ptw->ptTop.y);
      if (abs (ptw->ptTop.x - lpptTop->x) * ptw->nFontSize >= ptw->siWnd.cx
        || abs (ptw->ptTop.y - lpptTop->y) * ptw->nFontSize * 2
                                                            >= ptw->siWnd.cy)
        {
          /* ja:1画面を超える移動 */
          InvalidateRect (hWnd, NULL, TRUE);
        }
      else
        {
          /* ja:1画面以内の移動 */
          RECT rc;

          rc.left = rc.top = 0;
          rc.right = ptw->siWnd.cx;
          rc.bottom = ptw->siWnd.cy;
          ScrollWindowEx (hWnd, (lpptTop->x - ptw->ptTop.x) * ptw->nFontSize,
                            (lpptTop->y - ptw->ptTop.y) * ptw->nFontSize * 2,
                            NULL, &rc, NULL, NULL, SW_ERASE | SW_INVALIDATE);
        }
    }
}


/*  ja:キャレット位置にデータを挿入/上書きする
        hWnd,ウインドウ
    lpszText,データを格納するポインタ
     nLength,データのバイト数
      fCaret,TRUE:移動する,FALSE:移動しない
     fSelect,TRUE:選択する,FALSE:選択しない
         RET,逆の操作を行うDOING構造体                                      */
LPDOING
EditOperation (HWND   hWnd,
               LPCSTR lpszText,
               int    nLength,
               BOOL   fCaret,
               BOOL   fSelect)
{
  int sx, sy, nMax;
  LPDOING d;
  POINT ptTop;
  LPTEXTWND ptw;

  ptw = (LPTEXTWND)GetWindowLongPtr (hWnd, GWLP_USERDATA);
  ptw->ptCursor.x = GetAlignPos (&ptw->lpStart, &ptw->nOff,
                        ptw->ptCursor.x, ptw->ptCursor.y, ptw->nTab, FALSE);
  ptTop = ptw->ptTop;
  sx = max (ptw->siWnd.cx / ptw->nFontSize, 1);
  sy = max (ptw->siWnd.cy / (ptw->nFontSize * 2), 1);
  d = MemoryAlloc (sizeof (DOING));
  HideCaret (hWnd);
  if (ptw->ptSelect.x < 0)
    {
      d->lpszText = NULL;
      d->nLength = 0;
      d->fCaret = FALSE;
    }
  else
    {
      /* ja:削除 */
      int i, nDelete;
      LPLINEBUF p;
      POINT ptCursor;
      RECT rc1;

      d->nLength = GetSelByte (&ptw->lpStart, &ptw->nOff,
                                    &ptw->ptCursor, &ptw->ptSelect, ptw->nTab);
      d->lpszText = MemoryAlloc (d->nLength * sizeof (CHAR));
      CpySelMem (&ptw->lpStart, &ptw->nOff,
                    &ptw->ptCursor, &ptw->ptSelect, ptw->nTab, d->lpszText);
      nDelete = DelSelMem (&ptw->lpStart, &ptw->nOff,
                                    &ptw->ptCursor, &ptw->ptSelect, ptw->nTab,
                                            ptw->fLimit ? ptw->nMargin : 0);
      if (ptw->ptSelect.y < ptw->ptCursor.y
                                        || ptw->ptSelect.y == ptw->ptCursor.y
                                        && ptw->ptSelect.x < ptw->ptCursor.x)
        {
          ptCursor = ptw->ptSelect;
          d->fCaret = TRUE;
        }
      else
        {
          ptCursor = ptw->ptCursor;
          d->fCaret = FALSE;
        }
      /* ja:削除した行の削除した桁より右を初期化(右マージンのために1つ多い) */
      rc1.left = (ptCursor.x - ptw->ptTop.x - 1) * ptw->nFontSize;
      rc1.top = (ptCursor.y - ptw->ptTop.y) * ptw->nFontSize * 2;
      rc1.right = ptw->siWnd.cx;
      rc1.bottom = rc1.top + ptw->nFontSize * 2;
      InvalidateRect (hWnd, &rc1, TRUE);
      /* ja:減った行数分だけスクロール */
      if (nDelete > 0)
        if (nDelete * ptw->nFontSize * 2 >= ptw->siWnd.cy)
          {
            InvalidateRect (hWnd, NULL, TRUE);
          }
        else
          {
            RECT rc2;

            rc1.left = 0;
            rc1.top = (ptCursor.y - ptw->ptTop.y + 1) * ptw->nFontSize * 2;
            rc1.right = ptw->siWnd.cx;
            rc1.bottom = ptw->siWnd.cy;
            rc2 = rc1;
            ScrollWindowEx (hWnd, 0, -nDelete * ptw->nFontSize * 2,
                            &rc1, &rc2, NULL, NULL, SW_ERASE | SW_INVALIDATE);
          }
      ptw->nMax -= nDelete;/* ja:行数減少 */
      /* ja:選択範囲解除とキャレット移動 */
      ptw->ptCursor = ptCursor;
      ptw->ptSelect.x = -1;
      /* ja:削除を終えた位置から右マージンの影響のある行数だけ初期化 */
      for (i = 0, p = GetLineBuffer (&ptw->lpStart, &ptw->nOff,
                                        ptw->ptCursor.y); p; i++, p = p->next)
        if (!p->fMargin)
          break;
      if (i > 0)
        {
          rc1.left = 0;
          rc1.top = (ptw->ptCursor.y - ptw->ptTop.y + 1) * ptw->nFontSize * 2;
          rc1.right = ptw->siWnd.cx;
          rc1.bottom = rc1.top + i * ptw->nFontSize * 2;
          InvalidateRect (hWnd, &rc1, TRUE);
        }
    }
  if (nLength <= 0 || !lpszText)
    {
      d->ptCursor = ptw->ptCursor;
      d->ptSelect.x = -1;
    }
  else
    {
      /* ja:挿入 */
      int i, j, nPut;
      LPLINEBUF p, q;
      POINT ptCursor;
      RECT rc1;

      ptw->ptCursor.x = GetAlignPos (&ptw->lpStart, &ptw->nOff,
                        ptw->ptCursor.x, ptw->ptCursor.y, ptw->nTab, FALSE);
      nPut = PutMem (&ptw->lpStart, &ptw->nOff, &ptw->ptCursor, &ptCursor,
                ptw->nTab, lpszText, nLength, ptw->fLimit ? ptw->nMargin : 0);
      /* ja:挿入した行の挿入した桁より右を初期化 */
      rc1.left = (ptw->ptCursor.x - ptw->ptTop.x) * ptw->nFontSize;
      rc1.top = (ptw->ptCursor.y - ptw->ptTop.y) * ptw->nFontSize * 2;
      rc1.right = ptw->siWnd.cx;
      rc1.bottom = rc1.top + ptw->nFontSize * 2;
      InvalidateRect (hWnd, &rc1, TRUE);
      /* ja:増えた行数分だけスクロール */
      if (nPut > 0)
        if (nPut * ptw->nFontSize * 2 >= ptw->siWnd.cy)
          {
            InvalidateRect (hWnd, NULL, TRUE);
          }
        else
          {
            RECT rc2;

            rc1.left = 0;
            rc1.top = (ptw->ptCursor.y - ptw->ptTop.y + 1)
                                                        * ptw->nFontSize * 2;
            rc1.right = ptw->siWnd.cx;
            rc1.bottom = ptw->siWnd.cy;
            rc2 = rc1;
            ScrollWindowEx (hWnd, 0, nPut * ptw->nFontSize * 2,
                            &rc1, &rc2, NULL, NULL, SW_ERASE | SW_INVALIDATE);
          }
      /* ja:挿入を終えた位置から右マージンの影響のある行数だけ初期化 */
      p = q = GetLineBuffer (&ptw->lpStart, &ptw->nOff, ptCursor.y);
      for (i = ptCursor.y, j = 0;
                            p->prev && p->prev->fMargin && ptw->ptCursor.y < i;
                                                        i--, j++, p = p->prev);
      while (q->next && q->fMargin)
        j++, q = q->next;
      if (j > 0)
        {
          rc1.left = 0;
          rc1.top = (i - ptw->ptTop.y + 1) * ptw->nFontSize * 2;
          rc1.right = ptw->siWnd.cx;
          rc1.bottom = rc1.top + j * ptw->nFontSize * 2;
          InvalidateRect (hWnd, &rc1, TRUE);
        }
      ptw->nMax += nPut;/* ja:行数増加 */
      if (fCaret)
        {
          /* ja:キャレット移動 */
          d->ptCursor = ptCursor;
          d->ptSelect = ptw->ptCursor;
          ptw->ptCursor = ptCursor;
        }
      else
        {
          /* ja:キャレット移動なし */
          d->ptCursor = ptw->ptCursor;
          d->ptSelect = ptCursor;
        }
      if (fSelect)/* ja:選択する */
        ptw->ptSelect = d->ptSelect;
    }
  if (ptw->ptCursor.x < ptw->ptTop.x)
    ptw->ptTop.x = ptw->ptCursor.x;
  else if (ptw->ptCursor.x - sx + 1 > ptw->ptTop.x)
    ptw->ptTop.x = ptw->ptCursor.x - sx + 1;
  if (ptw->ptCursor.y < ptw->ptTop.y)
    ptw->ptTop.y = ptw->ptCursor.y;
  else if (ptw->ptCursor.y - sy + 1 > ptw->ptTop.y)
    ptw->ptTop.y = ptw->ptCursor.y - sy + 1;
  if (ptw->ptTop.y > ptw->nMax - sy)
    ptw->ptTop.y = max (ptw->nMax - sy, 0);
  nMax = GetWidthMax (ptw);
  if (ptw->ptTop.x > nMax - sx + 1)
    ptw->ptTop.x = max (nMax - sx + 1, 0);
  DrawCaret (hWnd);
  SetScrollBar (hWnd, SB_HORZ, 0, nMax, sx, ptw->ptTop.x);
  SetScrollBar (hWnd, SB_VERT, 0, ptw->nMax - 1, sy, ptw->ptTop.y);
  MoveTextWindow (hWnd, &ptTop);
  ShowCaret (hWnd);
  return d;
}


/*  ja:リストから削除する
    lpDo,リストの先頭
     RET,削除したリストの数                                                 */
int
DeleteList (DOING **lpDo)
{
  int nCount = 0;
  LPDOING d0, d1;

  for (d0 = *lpDo; d0; d0 = d1)
    {
      MemoryFree (d0->lpszText);
      d1 = d0->next;
      MemoryFree (d0);
      nCount++;
    }
  *lpDo = NULL;
  return nCount;
}


/*  ja:右マージンで折り返す
    ptw,TXTウインドウ情報                                                   */
VOID
ModifyMargin (LPTEXTWND ptw)
{
  LPLINEBUF p;

  /* ja:ラインバッファの先頭へ移動 */
  while (ptw->lpStart->prev)
    ptw->lpStart = ptw->lpStart->prev;
  ptw->nOff = 0;
  p = ptw->lpStart;
  if (ptw->fLimit)
    /* ja:右マージンによる改行あり */
    while (p)
      {
        int nDataPos = 0, nScreenPos = 0;

        while (nDataPos < p->nLength)
          if (p->lpszText[nDataPos] == '\t')
            {
              nScreenPos = (nScreenPos / ptw->nTab + 1) * ptw->nTab;
              if (ptw->nMargin < nScreenPos)
                break;
              nDataPos++;
            }
          else if (nDataPos == p->nLength - 1
                        || !IsDBCSLeadByteEx (CP_ACP, p->lpszText[nDataPos]))
            {
              nScreenPos++;
              if (ptw->nMargin < nScreenPos)
                break;
              nDataPos++;
            }
          else
            {
              nScreenPos += 2;
              if (ptw->nMargin < nScreenPos)
                break;
              nDataPos += 2;
            }
        if (ptw->nMargin < nScreenPos)
          {
            /* ja:右マージンを超えているとき、擬似改行する */
            LPLINEBUF q;

            q = MemoryAlloc (sizeof (LINEBUF));
            q->nLength = p->nLength - nDataPos;
            q->fMargin = p->fMargin;
            q->lpszText = MemoryAlloc (q->nLength * sizeof (CHAR));
            q->prev = p;
            q->next = p->next;
            p->next = q;
            if (q->next)
              q->next->prev = q;
            MemoryCopy (q->lpszText, p->lpszText + nDataPos,
                                                q->nLength * sizeof (CHAR));
            p->nLength = nDataPos;
            p->fMargin = TRUE;
            p->lpszText = MemoryReAlloc (p->lpszText,
                                                p->nLength * sizeof (CHAR));
            p = q;
            ptw->nMax++;
          }
        else if (nScreenPos < ptw->nMargin && p->fMargin && p->next)
          {
            /* ja:右マージンを下回り擬似改行のとき、次の行とあわせる */
            LPLINEBUF q;

            nDataPos = p->nLength;
            q = p->next;
            if (q->next)
              q->next->prev = p;
            p->next = q->next;
            p->nLength += q->nLength;
            p->fMargin = q->fMargin;
            p->lpszText = MemoryReAlloc (p->lpszText,
                                                p->nLength * sizeof (CHAR));
            MemoryCopy (p->lpszText + nDataPos, q->lpszText,
                                                q->nLength * sizeof (CHAR));
            MemoryFree (q->lpszText);
            MemoryFree (q);
            ptw->nMax--;
          }
        else
          {
            p = p->next;
          }
      }
  else
    /* ja:右マージンによる改行なし */
    while (p)
      if (p->fMargin)
        {
          /* ja:擬似改行のとき、次の行とあわせる */
          int nDataPos;
          LPLINEBUF q;

          nDataPos = p->nLength;
          q = p->next;
          if (q->next)
            q->next->prev = p;
          p->next = q->next;
          p->nLength += q->nLength;
          p->fMargin = q->fMargin;
          p->lpszText = MemoryReAlloc (p->lpszText,
                                                p->nLength * sizeof (CHAR));
          MemoryCopy (p->lpszText + nDataPos, q->lpszText,
                                                q->nLength * sizeof (CHAR));
          MemoryFree (q->lpszText);
          MemoryFree (q);
          ptw->nMax--;
        }
      else
        {
          p = p->next;
        }
}


/*  ja:プロパティを変更する
       hWnd,ウインドウ
    lpFType,ファイルタイプ                                                  */
VOID
ChangeProperty (HWND       hWnd,
                LPFILETYPE lpFType)
{
  int sx, sy, nMax, nMargin, nTab;
  BOOL fLimit;
  HDC hDC;
  LOGFONT LogFont;
  LPTEXTWND ptw;
  TEXTMETRIC TextMetric;

  ptw = (LPTEXTWND)GetWindowLongPtr (hWnd, GWLP_USERDATA);
  if (ptw->fLimit != lpFType->fLimit || ptw->nTab != lpFType->nTab
                            || ptw->fLimit && ptw->nMargin != lpFType->nMargin)
    {
      /* ja:折り返しの設定変更、タブ数変更 */
      DeleteList (&ptw->lpUndo);
      DeleteList (&ptw->lpRedo);
    }
  fLimit = ptw->fLimit;
  nMargin = ptw->nMargin;
  nTab = ptw->nTab;
  ptw->nID = lpFType->nID;
  ptw->nMargin = lpFType->nMargin;
  ptw->nTab = lpFType->nTab;
  ptw->fAssociate = lpFType->fAssociate;
  ptw->fAutoIndent = lpFType->fAutoIndent;
  ptw->fCode = lpFType->fCode;
  ptw->fCRLF = lpFType->fCRLF;
  ptw->fEOF = lpFType->fEOF;
  ptw->fLimit = lpFType->fLimit;
  ptw->fOverWrite = lpFType->fOverWrite;
  ptw->fRecycle = lpFType->fRecycle;
  ptw->fSpace = lpFType->fSpace;
  ptw->fSysColor = lpFType->fSysColor;
  ptw->fTabConv = lpFType->fTabConv;
  ptw->fGline = lpFType->fGline;
  ptw->fMline = lpFType->fMline;
  ptw->fUline = lpFType->fUline;
  ptw->fVline = lpFType->fVline;
  MemoryCopy (ptw->crColor, lpFType->crColor, sizeof (COLORREF) * 12);
  ptw->LogFont = lpFType->LogFont;
  if (fLimit != ptw->fLimit || fLimit
                    && (nTab != lpFType->nTab || nMargin != lpFType->nMargin))
    {
      /* ja:折り返しの設定変更、折り返しが有効でタブ数変更 */
      ptw->ptCursor.x = ptw->ptCursor.y = ptw->ptTop.x = ptw->ptTop.y = 0;
      ModifyMargin (ptw);
    }
  ptw->ptSelect.x = -1;
  if (ptw->hBrushWindow)
    DeleteObject (ptw->hBrushWindow);
  if (ptw->hBrushSpace)
    DeleteObject (ptw->hBrushSpace);
  if (ptw->hBrushCrlf)
    DeleteObject (ptw->hBrushCrlf);
  if (ptw->hBrushHighLight)
    DeleteObject (ptw->hBrushHighLight);
  if (ptw->hPenTab)
    DeleteObject (ptw->hPenTab);
  if (ptw->hPenMargin)
    DeleteObject (ptw->hPenMargin);
  if (ptw->hPenGrid)
    DeleteObject (ptw->hPenGrid);
  if (ptw->hFont)
    DeleteObject (ptw->hFont);
  if (ptw->hFontsm)
    DeleteObject (ptw->hFontsm);
  ptw->hBrushWindow = ptw->hBrushSpace = ptw->hBrushHighLight = NULL;
  ptw->hPenTab = ptw->hPenMargin = ptw->hPenGrid = NULL;
  ptw->hFont = ptw->hFontsm = NULL;

  if (!ptw->fSysColor)
    {
      ptw->hBrushWindow = CreateSolidBrush (ptw->crColor[1]);
      ptw->hBrushSpace = CreateSolidBrush (ptw->crColor[3]);
      ptw->hBrushCrlf = CreateSolidBrush (ptw->crColor[4]);
      ptw->hBrushHighLight = CreateSolidBrush (ptw->crColor[9]);
      ptw->hPenTab = CreatePen (PS_SOLID, 0, ptw->crColor[5]);
      ptw->hPenMargin = CreatePen (PS_SOLID, 0, ptw->crColor[6]);
      ptw->hPenGrid = CreatePen (PS_SOLID, 0, ptw->crColor[7]);
    }

  LogFont = ptw->LogFont;
  LogFont.lfHeight /= 2;
  ptw->hFont = CreateFontIndirect (&ptw->LogFont);
  ptw->hFontsm = CreateFontIndirect (&LogFont);
  hDC = GetDC (hWnd);
  ptw->hFont = SelectObject (hDC, ptw->hFont);
  GetTextMetrics (hDC, &TextMetric);
  ptw->hFont = SelectObject (hDC, ptw->hFont);
  ReleaseDC (hWnd, hDC);

  ptw->nFontSize = (TextMetric.tmHeight + 1) / 2;
  sx = max (ptw->siWnd.cx / ptw->nFontSize, 1);
  sy = max (ptw->siWnd.cy / (ptw->nFontSize * 2), 1);
  if (ptw->ptTop.y > ptw->nMax - sy)
    ptw->ptTop.y = max (ptw->nMax - sy, 0);
  nMax = GetWidthMax (ptw);
  if (ptw->ptTop.x > nMax - sx + 1)
    ptw->ptTop.x = max (nMax - sx + 1, 0);
  SetScrollBar (hWnd, SB_HORZ, 0, nMax, sx, ptw->ptTop.x);
  SetScrollBar (hWnd, SB_VERT, 0, ptw->nMax - 1, sy, ptw->ptTop.y);
  SetImeFont (hWnd, &ptw->LogFont);
  SetMenuBar (ptw);
  CreateCaret (hWnd, (HBITMAP)0,
                        uIns != 0 ? ptw->nFontSize : 2, ptw->nFontSize * 2);
  DrawCaret (hWnd);
  InvalidateRect (hWnd, NULL, TRUE);
  ShowCaret (hWnd);
}


BOOL CALLBACK
PropertyEnumProc (HWND   hWnd,
                  LPARAM lParam)
{
  LPTSTR lpszClass;

  lpszClass = GetClassNameNew (hWnd);
  if (lstrcmp (lpszClass, TEXTCLASS) == 0 && ((LPFILETYPE)lParam)->nID
                == ((LPTEXTWND)GetWindowLongPtr (hWnd, GWLP_USERDATA))->nID)
    ChangeProperty (hWnd, (LPFILETYPE)lParam);
  MemoryFree (lpszClass);
  return TRUE;
}


/*  ja:レジストリへ書き込む                                                 */
VOID
RegWrite (VOID)
{
  LONG lResult;
  HKEY hKey;

  lResult = RegOpenKeyEx (HKEY_CURRENT_USER,
                _T("Software\\maid.org\\Text maid"), 0, KEY_ALL_ACCESS, &hKey);
  if (lResult != ERROR_SUCCESS)
    {
      BOOL fCreate;

      fCreate = nRegStat & REG_STAT_CREATE;
      if (!fCreate)
        {
          LPTSTR lpszText;

          lpszText = LoadText (GetModuleHandle (NULL), IDS_OTHER_REG);
          fCreate = MessageBox (hWndMain, lpszText,
                                            APPLICATION, MB_OKCANCEL) == IDOK;
          MemoryFree (lpszText);
        }
      if (fCreate)
        {
          lResult = RegCreateKeyEx (HKEY_CURRENT_USER,
                            _T("Software\\maid.org\\Text maid"),
                            0, NULL, REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS,
                                                            NULL, &hKey, NULL);
          if (lResult != ERROR_SUCCESS)
            MessageBox (NULL, _T("RegCreateKeyEx"),
                                APPLICATION, MB_OK | MB_ICONEXCLAMATION);
        }
    }
  if (lResult == ERROR_SUCCESS)
    {
      int i, nMenu;
      DWORD cbMaxSubKeyLen, cbMaxValueNameLen, dwName;
      FILETIME FileTime;
      HKEY hKeySub;
      HMENU hMenu;
      LPTSTR lpszName;
      WINDOWPLACEMENT wndpl;

      wndpl.length = sizeof (WINDOWPLACEMENT);
      GetWindowPlacement (hWndMain, &wndpl);
      RegQueryInfoKey (hKey, NULL, NULL, NULL, NULL, &cbMaxSubKeyLen,
                            NULL, NULL, &cbMaxValueNameLen, NULL, NULL, NULL);

      /* ja:サブキーを削除 */
      lpszName = MemoryAlloc ((cbMaxSubKeyLen + 1) * sizeof (TCHAR));
      while (dwName = cbMaxSubKeyLen, RegEnumKeyEx (hKey, 0, lpszName, &dwName,
                                NULL, NULL, NULL, &FileTime) == ERROR_SUCCESS)
        RegDeleteKey (hKey, lpszName);
      /* ja:キーの値を削除 */
      lpszName = MemoryReAlloc (lpszName,
                                    (cbMaxValueNameLen + 1) * sizeof (TCHAR));
      while (dwName = MAX_PATH, RegEnumValue (hKey, 0, lpszName, &dwName,
                                    NULL, NULL, NULL, NULL) == ERROR_SUCCESS)
        RegDeleteValue (hKey, lpszName);
      MemoryFree (lpszName);
      RegSetDword (hKey, _T("CodeDef"), nCodeDef);
      RegSetDword (hKey, _T("Filter"), dwFilter);
      RegSetDword (hKey, _T("History"), nHistory);
      RegSetDword (hKey, _T("Zoomed"), fZoomed);
      RegSetString (hKey, _T("OpenPath"), lpszOpenPath);
      RegSetString (hKey, _T("SavePath"), lpszSavePath);
      RegSetBinary (hKey, _T("Rect"), &wndpl.rcNormalPosition, sizeof (RECT));
      RegSetDword (hKey, _T("FileType"), nFileType);
      RegSetDword (hKey, _T("FindNum"), nFind);
      RegSetDword (hKey, _T("FindArrow"), bFindArrow);
      RegSetDword (hKey, _T("FindCase"), bFindCase);
      RegSetDword (hKey, _T("FindWidth"), bFindWidth);
      RegSetDword (hKey, _T("ReplaceNum"), nReplace);
      RegSetDword (hKey, _T("ReplaceArrow"), bReplaceArrow);
      RegSetDword (hKey, _T("ReplaceCase"), bReplaceCase);
      RegSetDword (hKey, _T("ReplaceWidth"), bReplaceWidth);
      /* ja:ファイルタイプ */
      for (i = 0; i < nFileType; i++)
        {
          wasprintf (&lpszName, _T("FileType%04d"), i);
          RegCreateKeyEx (hKey, lpszName, 0, NULL, REG_OPTION_NON_VOLATILE,
                                        KEY_ALL_ACCESS, NULL, &hKeySub, NULL);
          MemoryFree (lpszName);
          RegSetString (hKeySub, _T("Text"), lpFileType[i].lpszText);
          RegSetString (hKeySub, _T("Ext"), lpFileType[i].lpszExt);
          RegSetDword (hKeySub, _T("Margin"), lpFileType[i].nMargin);
          RegSetDword (hKeySub, _T("Tab"), lpFileType[i].nTab);
          RegSetBinary (hKeySub, _T("LogFont"), &lpFileType[i].LogFont,
                                                    sizeof (LOGFONT));
          RegSetBinary (hKeySub, _T("Color"), lpFileType[i].crColor,
                                                    sizeof (COLORREF) * 12);
          RegSetDword (hKeySub, _T("Associate"), lpFileType[i].fAssociate);
          RegSetDword (hKeySub, _T("AutoIndent"), lpFileType[i].fAutoIndent);
          RegSetDword (hKeySub, _T("Code"), lpFileType[i].fCode);
          RegSetDword (hKeySub, _T("CRLF"), lpFileType[i].fCRLF);
          RegSetDword (hKeySub, _T("EOF"), lpFileType[i].fEOF);
          RegSetDword (hKeySub, _T("Limit"), lpFileType[i].fLimit);
          RegSetDword (hKeySub, _T("OverWrite"), lpFileType[i].fOverWrite);
          RegSetDword (hKeySub, _T("Recycle"), lpFileType[i].fRecycle);
          RegSetDword (hKeySub, _T("Space"), lpFileType[i].fSpace);
          RegSetDword (hKeySub, _T("SysColor"), lpFileType[i].fSysColor);
          RegSetDword (hKeySub, _T("TabConv"), lpFileType[i].fTabConv);
          RegSetDword (hKeySub, _T("Gline"), lpFileType[i].fGline);
          RegSetDword (hKeySub, _T("Mline"), lpFileType[i].fMline);
          RegSetDword (hKeySub, _T("Uline"), lpFileType[i].fUline);
          RegSetDword (hKeySub, _T("Vline"), lpFileType[i].fVline);
          RegSetDword (hKeySub, _T("Negotiate"), lpFileType[i].fNegotiate);
          RegSetDword (hKeySub, _T("RetCode"), lpFileType[i].uRetCode);
          RegSetDword (hKeySub, _T("CharSet"), lpFileType[i].dwCharSet);
          RegCloseKey (hKeySub);
        }
      /* ja:検索文字列 */
      RegCreateKeyEx (hKey, _T("Find"), 0, NULL, REG_OPTION_NON_VOLATILE,
                                        KEY_ALL_ACCESS, NULL, &hKeySub, NULL);
      for (i = 0; i < nFind; i++)
        {
          wasprintf (&lpszName, _T("Find%02d"), i);
          RegSetString (hKeySub, lpszName, ppszFind[i]);
          MemoryFree (lpszName);
        }
      RegCloseKey (hKeySub);
      /* ja:置換文字列 */
      RegCreateKeyEx (hKey, _T("Replace"), 0, NULL, REG_OPTION_NON_VOLATILE,
                                        KEY_ALL_ACCESS, NULL, &hKeySub, NULL);
      for (i = 0; i < nReplace; i++)
        {
          wasprintf (&lpszName, _T("Replace%02d"), i);
          RegSetString (hKeySub, lpszName, ppszReplace[i]);
          MemoryFree (lpszName);
        }
      RegCloseKey (hKeySub);
      /* ja:履歴 */
      hMenu = GetMenu (hWndMain);
      hMenu = GetSubMenu (hMenu, GetMenuItemCount (hMenu) != MENUTOP);
      nMenu = GetMenuItemCount (hMenu) - MENUFILE - 1;
      for (i = 0; i < nMenu; i++)
        {
          LPTSTR lpszText;

          lpszText = GetMenuStringNew (hMenu, i + MENUFILE - 1, MF_BYPOSITION);
          if (lstrlen (lpszText) > 3)
            {
              wasprintf (&lpszName, _T("History%d"), i);
              RegSetString (hKey, lpszName, lpszText + 3);
              MemoryFree (lpszName);
            }
          MemoryFree (lpszText);
        }
      RegCloseKey (hKey);
    }
}


/******************************************************************************
*                                                                             *
* ja:高レベル関数群                                                           *
*                                                                             *
******************************************************************************/
/*  ja:編集履歴操作
     hWnd,ウインドウ
    fKind,TRUE:redo,FALSE:undo
      RET,TRUE:正常終了,FALSE:エラー                                        */
VOID
HistoryOperation (HWND hWnd,
                  BOOL fKind)
{
  LPDOING d0, d1;
  LPTEXTWND ptw;

  ptw = (LPTEXTWND)GetWindowLongPtr (hWnd, GWLP_USERDATA);
  if (fKind)
    {
      d0 = ptw->lpRedo;
      ptw->lpRedo = d0->next;
    }
  else
    {
      d0 = ptw->lpUndo;
      ptw->lpUndo = d0->next;
    }
  if (ptw->ptSelect.x >= 0)
    ClearSel (hWnd, &ptw->ptSelect, &ptw->ptCursor);
  ptw->ptCursor = d0->ptCursor;
  ptw->ptSelect = d0->ptSelect;
  d1 = EditOperation (hWnd, d0->lpszText, d0->nLength, d0->fCaret, TRUE);
  if (d1)
    {
      if (fKind)
        {
          d1->next = ptw->lpUndo;
          ptw->lpUndo = d1;
        }
      else
        {
          d1->next = ptw->lpRedo;
          ptw->lpRedo = d1;
        }
      MemoryFree (d0->lpszText);
      MemoryFree (d0);
      HideCaret (hWnd);
      SetMenuBar (ptw);
      ShowCaret (hWnd);
    }
}


/*  ja:右マージンで改行する
    hWnd,ウインドウ                                                         */
VOID
MarginOperation (HWND hWnd)
{
  BOOL fLimit;
  LPTEXTWND ptw;
  LPLINEBUF p;

  ptw = (LPTEXTWND)GetWindowLongPtr (hWnd, GWLP_USERDATA);
  DeleteList (&ptw->lpUndo);
  DeleteList (&ptw->lpRedo);
  fLimit = ptw->fLimit;
  ptw->fLimit = TRUE;
  ModifyMargin (ptw);
  for (p = ptw->lpStart; p->prev; p = p->prev);
  while (p)
    {
      p->fMargin = FALSE;
      p = p->next;
    }
  ptw->fLimit = fLimit;
  ptw->fEdit = TRUE;
  ptw->ptCursor.x = ptw->ptCursor.y = ptw->ptTop.x = ptw->ptTop.y = 0;
  ptw->ptSelect.x = -1;
  SetScrollBar (hWnd, SB_HORZ, 0, GetWidthMax (ptw),
                max (ptw->siWnd.cx / ptw->nFontSize, 1), ptw->ptTop.x);
  SetScrollBar (hWnd, SB_VERT, 0, ptw->nMax - 1,
                max (ptw->siWnd.cy / (ptw->nFontSize * 2), 1), ptw->ptTop.y);
  SetMenuBar (ptw);
  DrawCaret (hWnd);
  InvalidateRect (hWnd, NULL, TRUE);
  ShowCaret (hWnd);
}


/*  ja:タブをスペースに変換する
    hWnd,ウインドウ                                                         */
VOID
TabOperation (HWND hWnd)
{
  LPTEXTWND ptw;
  LPLINEBUF p;

  ptw = (LPTEXTWND)GetWindowLongPtr (hWnd, GWLP_USERDATA);
  DeleteList (&ptw->lpUndo);
  DeleteList (&ptw->lpRedo);
  for (p = ptw->lpStart; p->prev; p = p->prev);
  while (p)
    {
      int nDataPos = 0, nScreenPos = 0;
      BOOL fTab = FALSE;

      while (nDataPos < p->nLength)
        if (p->lpszText[nDataPos] == '\t')
          {
            nScreenPos = (nScreenPos / ptw->nTab + 1) * ptw->nTab;
            nDataPos++;
            fTab = TRUE;
          }
        else if (nDataPos == p->nLength - 1
                        || !IsDBCSLeadByteEx (CP_ACP, p->lpszText[nDataPos]))
          {
            nScreenPos++;
            nDataPos++;
          }
        else
          {
            nScreenPos += 2;
            nDataPos += 2;
          }
      if (fTab)
        {
          LPSTR lpszTemp;

          lpszTemp = MemoryAlloc (nScreenPos * sizeof (CHAR));
          MemorySet (lpszTemp, ' ', nScreenPos * sizeof (CHAR));
          nDataPos = nScreenPos = 0;
          while (nDataPos < p->nLength)
            if (p->lpszText[nDataPos] == '\t')
              {
                nScreenPos = (nScreenPos / ptw->nTab + 1) * ptw->nTab;
                nDataPos++;
              }
            else if (nDataPos == p->nLength - 1
                        || !IsDBCSLeadByteEx (CP_ACP, p->lpszText[nDataPos]))
              {
                lpszTemp[nScreenPos] = p->lpszText[nDataPos];
                nScreenPos++;
                nDataPos++;
              }
            else
              {
                lpszTemp[nScreenPos] = p->lpszText[nDataPos];
                nScreenPos++;
                nDataPos++;
                lpszTemp[nScreenPos] = p->lpszText[nDataPos];
                nScreenPos++;
                nDataPos++;
              }
            MemoryFree (p->lpszText);
            p->nLength = nScreenPos;
            p->lpszText = lpszTemp;
        }
      p = p->next;
    }
  ptw->fEdit = TRUE;
  ptw->ptSelect.x = -1;
  SetMenuBar (ptw);
  DrawCaret (hWnd);
  InvalidateRect (hWnd, NULL, TRUE);
}
