/*
    wcommon
    copyright (c) 1998-2018 Kazuki Iwamoto https://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "wcommon.h"


/******************************************************************************
*                                                                             *
******************************************************************************/
typedef struct _LIST {
  struct _LIST *lpPrev, *lpNext;
  LPVOID lpValue;
} LIST, *LPLIST;


/*  ja:値を直接比較する
    lpValue1,値1
    lpValue2,値2
      lpData,ユーザデータ
         RET,負:値1が小さい,0:等しい,正:値1が大きい                         */
int WINAPI
ListCompareDirect (LPVOID lpValue1,
                   LPVOID lpValue2,
                   LPVOID lpData)
{
  return (UINT_PTR)lpValue1 - (UINT_PTR)lpValue2;
}


/*  ja:値を文字列として比較する
    lpValue1,値1
    lpValue2,値2
      lpData,ユーザデータ
         RET,負:値1が小さい,0:等しい,正:値1が大きい                         */
int WINAPI
ListCompareStringW (LPVOID lpValue1,
                    LPVOID lpValue2,
                    LPVOID lpData)
{
  return lstrcmpW (lpValue1, lpValue2);
}


/*  ja:値を文字列として比較する
    lpValue1,値1
    lpValue2,値2
      lpData,ユーザデータ
         RET,負:値1が小さい,0:等しい,正:値1が大きい                         */
int WINAPI
ListCompareStringA (LPVOID lpValue1,
                    LPVOID lpValue2,
                    LPVOID lpData)
{
  return lstrcmpA (lpValue1, lpValue2);
}


/*  ja:線形リストの値を解放する
    lpValue,値
     lpData,ユーザデータ                                                    */
VOID WINAPI
ListFree (LPVOID lpValue,
          LPVOID lpData)
{
  MemoryFree (lpValue);
}


/*  ja:線形リストを作る
    lpValue,値
        RET,線形リスト                                                      */
LPLIST WINAPI
ListNew (LPVOID lpValue)
{
  LPLIST lpList;

  lpList = MemoryAlloc (sizeof (LIST));
  lpList->lpValue = lpValue;
  return lpList;
}


/******************************************************************************
*                                                                             *
******************************************************************************/
/*  ja:線形リストの前を取得する
    lpList,線形リスト
       RET,線形リスト                                                       */
LPLIST WINAPI
ListPrev (LPLIST lpList)
{
  return lpList ? lpList->lpPrev : NULL;
}


/*  ja:線形リストの後を取得する
    lpList,線形リスト
       RET,線形リスト                                                       */
LPLIST WINAPI
ListNext (LPLIST lpList)
{
  return lpList ? lpList->lpNext : NULL;
}


/*  ja:線形リストの先頭を取得する
    lpList,線形リスト
       RET,線形リスト                                                       */
LPLIST WINAPI
ListFirst (LPLIST lpList)
{
  if (lpList)
    while (lpList->lpPrev)
      lpList = lpList->lpPrev;
  return lpList;
}


/*  ja:線形リストの末尾を取得する
    lpList,線形リスト
       RET,線形リスト                                                       */
LPLIST WINAPI
ListLast (LPLIST lpList)
{
  if (lpList)
    while (lpList->lpNext)
      lpList = lpList->lpNext;
  return lpList;
}


/*  ja:線形リストの中間を取得する
    lpList,線形リスト
       RET,線形リスト                                                       */
LPLIST WINAPI
ListMiddle (LPLIST lpList)
{
  if (lpList)
    {
      int nCount;

      nCount = 1 + ListLengthNext (lpList);
      while (lpList->lpPrev)
        {
          nCount++;
          lpList = lpList->lpPrev;
        }
      lpList = ListIndex (lpList, nCount / 2);
    }
  return lpList;
}


/*  ja:線形リストのn番目を求める
    lpList,線形リスト
    nIndex,n番(負:後方から,正or0:前方から)
       RET,線形リスト(NULL:要素なし)                                        */
LPLIST WINAPI
ListIndex (LPLIST lpList,
           int    nIndex)
{
  int nCount = 0;

  if (nIndex < 0)
    for (lpList = ListLast (lpList); lpList; lpList = lpList->lpPrev)
      {
        if (--nCount == nIndex)
          break;
      }
  else
    for (lpList = ListFirst (lpList); lpList; lpList = lpList->lpNext)
      {
        if (nCount++ == nIndex)
          break;
      }
  return lpList;
}


/******************************************************************************
*                                                                             *
******************************************************************************/
/*  ja:線形リストの前に線形リストを挿入する
      lpList,線形リスト
    lpInsert,挿入する線形リスト
         RET,線形リスト                                                     */
LPLIST WINAPI
ListLinkPrev (LPLIST lpList,
              LPLIST lpInsert)
{
  if (lpInsert)
    {
      if (lpList)
        {
          LPLIST lpFirst;

          lpFirst = ListFirst (lpInsert);
          lpFirst->lpPrev = lpList->lpPrev;
          if (lpList->lpPrev)
            lpList->lpPrev->lpNext = lpFirst;
          lpInsert = ListLast (lpInsert);
          lpInsert->lpNext = lpList;
          lpList->lpPrev = lpInsert;
        }
      else
        {
          lpList = lpInsert;
        }
    }
  return lpList;
}


/*  ja:線形リストの後に線形リストを挿入する
      lpList,線形リスト
    lpInsert,挿入する線形リスト
         RET,線形リスト                                                     */
LPLIST WINAPI
ListLinkNext (LPLIST lpList,
              LPLIST lpInsert)
{
  if (lpInsert)
    {
      if (lpList)
        {
          LPLIST lpLast;

          lpLast = ListLast (lpInsert);
          lpLast->lpNext = lpList->lpNext;
          if (lpList->lpNext)
            lpList->lpNext->lpPrev = lpLast;
          lpInsert = ListFirst (lpInsert);
          lpInsert->lpPrev = lpList;
          lpList->lpNext = lpInsert;
        }
      else
        {
          lpList = lpInsert;
        }
    }
  return lpList;
}


/*  ja:線形リストの先頭に線形リストを追加する
     lpList,線形リスト
    lpFirst,追加する線形リスト
        RET,線形リスト                                                      */
LPLIST WINAPI
ListLinkFirst (LPLIST lpList,
               LPLIST lpFirst)
{
  return ListLinkPrev (ListFirst (lpList), lpFirst);
}


/*  ja:線形リストの末尾に線形リストを追加する
    lpList,線形リスト
    lpLast,追加する線形リスト
       RET,線形リスト                                                       */
LPLIST WINAPI
ListLinkLast (LPLIST lpList,
              LPLIST lpLast)
{
  return ListLinkNext (ListLast (lpList), lpLast);
}


/*  ja:線形リストの前で分離する
    lpList,線形リスト
       RET,線形リスト                                                       */
LPLIST WINAPI
ListUnlinkPrev (LPLIST lpList)
{
  LPLIST lpPrev;

  if (lpList)
    {
      lpPrev = lpList->lpPrev;
      if (lpPrev)
        lpList->lpPrev = lpPrev->lpNext = NULL;
    }
  else
    {
      lpPrev = NULL;
    }
  return lpPrev;
}


/*  ja:線形リストの後で分離する
    lpList,線形リスト
       RET,線形リスト                                                       */
LPLIST WINAPI
ListUnlinkNext (LPLIST lpList)
{
  LPLIST lpNext;

  if (lpList)
    {
      lpNext = lpList->lpNext;
      if (lpNext)
        lpList->lpNext = lpNext->lpPrev = NULL;
    }
  else
    {
      lpNext = NULL;
    }
  return lpNext;
}


/*  ja:線形リストから取り除く
    lpList,線形リスト
       RET,線形リスト                                                       */
LPLIST WINAPI
ListRemove (LPLIST lpList)
{
  LPLIST lpPrev, lpNext;

  if (lpList)
    {
      lpPrev = lpList->lpPrev;
      lpNext = lpList->lpNext;
      if (lpPrev)
        {
          lpList->lpPrev = NULL;
          lpPrev->lpNext = lpNext;
        }
      if (lpNext)
        {
          lpList->lpNext = NULL;
          lpNext->lpPrev = lpPrev;
        }
    }
  else
    {
      lpPrev = lpNext = NULL;
    }
  return lpPrev ? lpPrev : lpNext;
}


/******************************************************************************
*                                                                             *
******************************************************************************/
/*  ja:線形リストを削除する
    lpList,線形リスト
    lpFree,解放関数
    lpData,ユーザデータ
       RET,線形リスト                                                       */
LPLIST WINAPI
ListDelete (LPLIST     lpList,
            ListFree_t lpFree,
            LPVOID     lpData)
{
  LPLIST lpResult;

  if (lpList)
    {
      lpResult = ListRemove (lpList);
      if (lpFree)
        lpFree (lpList->lpValue, lpData);
      MemoryFree (lpList);
    }
  else
    {
      lpResult = NULL;
    }
  return lpResult;
}


/*  ja:線形リストの前を破棄する
    lpList,線形リスト
    lpFree,解放関数
    lpData,ユーザデータ                                                     */
VOID WINAPI
ListDestroyPrev (LPLIST     lpList,
                 ListFree_t lpFree,
                 LPVOID     lpData)
{
  ListDestroy (ListUnlinkPrev (lpList), lpFree, lpData);
}


/*  ja:線形リストの後を破棄する
    lpList,線形リスト
    lpFree,解放関数
    lpData,ユーザデータ                                                     */
VOID WINAPI
ListDestroyNext (LPLIST     lpList,
                 ListFree_t lpFree,
                 LPVOID     lpData)
{
  ListDestroy (ListUnlinkNext (lpList), lpFree, lpData);
}


/*  ja:線形リストを破棄する
    lpList,線形リスト
    lpFree,解放関数
    lpData,ユーザデータ                                                     */
VOID WINAPI
ListDestroy (LPLIST     lpList,
             ListFree_t lpFree,
             LPVOID     lpData)
{
  while (lpList)
    lpList = ListDelete (lpList, lpFree, lpData);
}


/*  ja:線形リストに線形リストを追加する
      lpList1,線形リスト
      lpList2,線形リスト
    lpCompare,比較関数
       lpData,ユーザデータ
          RET,線形リスト                                                    */
LPLIST WINAPI
ListMerge (LPLIST        lpList1,
           LPLIST        lpList2,
           ListCompare_t lpCompare,
           LPVOID        lpData)
{
  LPLIST lpList = NULL;

  if (lpCompare)
    {
      lpList1 = ListFirst (lpList1);
      lpList2 = ListFirst (lpList2);
      while (lpList1 && lpList2)
        {
          LPLIST lpTemp;

          if (lpCompare (lpList1->lpValue, lpList2->lpValue, lpData) <= 0)
            {
              lpTemp = lpList1;
              lpList1 = ListRemove (lpTemp);
            }
          else
            {
              lpTemp = lpList2;
              lpList2 = ListRemove (lpTemp);
            }
          lpList = ListLinkLast (lpList, lpTemp);
        }
    }
  lpList = ListLinkLast (lpList, lpList1);
  lpList = ListLinkLast (lpList, lpList2);
  return lpList;
}


/*  ja:線形リストを並び替える
       lpList,線形リスト
    lpCompare,比較関数
       lpData,ユーザデータ
          RET,線形リスト                                                    */
LPLIST WINAPI
ListSort (LPLIST        lpList,
          ListCompare_t lpCompare,
          LPVOID        lpData)
{
  if (lpList && (lpList->lpPrev || lpList->lpNext))
    {
      LPLIST lpNext;

      lpList = ListMiddle (lpList);
      lpNext = ListUnlinkNext (lpList);
      lpList = ListSort (lpList, lpCompare, lpData);
      lpNext = ListSort (lpNext, lpCompare, lpData);
      lpList = ListMerge (lpList, lpNext, lpCompare, lpData);
    }
  return lpList;
}


/*  ja:線形リストのn番目まで拡張する
    lpList,線形リスト
    nIndex,n番(負:後方から,正or0:前方から)
       RET,線形リスト                                                       */
LPLIST WINAPI
ListExtend (LPLIST lpList,
            int    nIndex)
{
  int nCount = 0;

  if (!lpList)
    lpList = ListNew (NULL);
  if (nIndex < 0)
    {
      lpList = ListLast (lpList);
      while (--nCount > nIndex)
        if (lpList->lpPrev)
          {
            lpList = lpList->lpPrev;
          }
        else
          {
            LPLIST lpNew;

            lpNew = ListNew (NULL);
            lpNew->lpNext = lpList;
            lpList->lpPrev = lpNew;
            lpList = lpNew;
          }
    }
  else
    {
      lpList = ListFirst (lpList);
      while (nCount++ < nIndex)
        if (lpList->lpNext)
          {
            lpList = lpList->lpNext;
          }
        else
          {
            LPLIST lpNew;

            lpNew = ListNew (NULL);
            lpNew->lpPrev = lpList;
            lpList->lpNext = lpNew;
            lpList = lpNew;
          }
    }
  return lpList;
}


/*  ja:線形リストのn番目まで拡張または破棄する
    lpList,線形リスト
    nIndex,n番(負:後方から,正or0:前方から)
    lpFree,解放関数
    lpData,ユーザデータ
       RET,線形リスト                                                       */
LPLIST WINAPI
ListTruncate (LPLIST     lpList,
              int        nIndex,
              ListFree_t lpFree,
              LPVOID     lpData)
{
  lpList = ListExtend (lpList, nIndex);
  if (nIndex < 0)
    ListDestroyPrev (lpList, lpFree, lpData);
  else
    ListDestroyNext (lpList, lpFree, lpData);
  return lpList;
}


/******************************************************************************
*                                                                             *
******************************************************************************/
/*  ja:線形リストの値を取得する
    lpList,線形リスト
       RET,値                                                               */
LPVOID WINAPI
ListGetValue (LPLIST lpList)
{
  return lpList ? lpList->lpValue : NULL;
}


/*  ja:線形リストの値を設定する
     lpList,線形リスト
    lpValue,値
        RET,前の値(値:エラー)                                               */
LPVOID WINAPI
ListSetValue (LPLIST lpList,
              LPVOID lpValue)
{
  LPVOID lpOld;

  if (lpList)
    {
      lpOld = lpList->lpValue;
      lpList->lpValue = lpValue;
    }
  else
    {
      lpOld = lpValue;
    }
  return lpOld;
}


/*  ja:線形リストのn番目の値を取得する
    lpList,線形リスト
    nIndex,n番(負:後方から,正or0:前方から)
       RET,値                                                               */
LPVOID WINAPI
ListGetValueIndex (LPLIST lpList,
                   int    nIndex)
{
  return ListGetValue (ListIndex (lpList, nIndex));
}


/*  ja:線形リストのn番目の値を設定する
     lpList,線形リスト
     nIndex,n番(負:後方から,正or0:前方から)
    lpValue,値
        RET,前の値(値:エラー)                                               */
LPVOID WINAPI
ListSetValueIndex (LPLIST lpList,
                   int    nIndex,
                   LPVOID lpValue)
{
  return ListSetValue (lpList, ListIndex (lpList, nIndex));
}


/*  ja:線形リストのn番目の値を設定する(拡張)
     lpList,線形リスト
     nIndex,n番(負:後方から,正or0:前方から)
    lpValue,値
        RET,前の値                                                          */
LPVOID WINAPI
ListSetValueExtend (LPLIST lpList,
                    int    nIndex,
                    LPVOID lpValue)
{
  return ListSetValue (lpList, ListExtend (lpList, nIndex));
}


/*  ja:線形リストの前に値を挿入する
     lpList,線形リスト
    lpValue,挿入する値
        RET,線形リスト                                                      */
LPLIST WINAPI
ListInsertPrev (LPLIST lpList,
                LPVOID lpValue)
{
  return ListLinkPrev (lpList, ListNew (lpValue));
}


/*  ja:線形リストの後に値を挿入する
     lpList,線形リスト
    lpValue,挿入する値
        RET,線形リスト                                                      */
LPLIST WINAPI
ListInsertNext (LPLIST lpList,
                LPVOID lpValue)
{
  return ListLinkNext (lpList, ListNew (lpValue));
}


/*  ja:線形リストに値を追加する
       lpList,線形リスト
      lpValue,線形リスト
    lpCompare,比較関数
       lpData,ユーザデータ
          RET,線形リスト                                                    */
LPLIST WINAPI
ListInsert (LPLIST        lpList,
            LPVOID        lpValue,
            ListCompare_t lpCompare,
            LPVOID        lpData)
{
  BOOL f = TRUE;

  if (lpList && lpCompare)
    {
      f = lpCompare (lpList->lpValue, lpValue, lpData) <= 0;
      if (f)
        while (f && lpList->lpNext)
          {
            lpList = lpList->lpNext;
            f = lpCompare (lpList->lpValue, lpValue, lpData) <= 0;
          }
      else
        while (!f && lpList->lpPrev)
          {
            lpList = lpList->lpPrev;
            f = lpCompare (lpList->lpValue, lpValue, lpData) <= 0;
          }
    }
  return f ? ListInsertNext (lpList, lpValue)
           : ListInsertPrev (lpList, lpValue);
}


/*  ja:線形リストの先頭に値を追加する
     lpList,線形リスト
    lpValue,値
        RET,線形リスト                                                      */
LPLIST WINAPI
ListPrepend (LPLIST lpList,
             LPVOID lpValue)
{
  return ListLinkFirst (lpList, ListNew (lpValue));
}


/*  ja:線形リストの末尾に値を追加する
     lpList,線形リスト
    lpValue,値
        RET,線形リスト                                                      */
LPLIST WINAPI
ListAppend (LPLIST lpList,
            LPVOID lpValue)
{
  return ListLinkLast (lpList, ListNew (lpValue));
}


/*  ja:線形リストの要素を列挙する
    lpList,線形リスト
    lpProc,列挙関数
    lpData,ユーザデータ
       RET,TRUE:すべて列挙,FALSE:中断                                       */
BOOL WINAPI
ListEnum (LPLIST         lpList,
          ListEnumProc_t lpProc,
          LPVOID         lpData)
{
  BOOL fResult;

  fResult = (BOOL)lpProc;
  if (fResult)
    for (lpList = ListFirst (lpList); lpList; lpList = lpList->lpNext)
      {
        fResult = lpProc (lpList->lpValue, lpData);
        if (!fResult)
          break;
      }
  return fResult;
}


/******************************************************************************
*                                                                             *
******************************************************************************/
/*  ja:線形リストの前の一致する要素を取得する
       lpList,線形リスト
      lpValue,値
    lpCompare,比較関数
       lpData,ユーザデータ
          RET,線形リスト                                                    */
LPLIST WINAPI
ListFindPrev (LPLIST        lpList,
              LPVOID        lpValue,
              ListCompare_t lpCompare,
              LPVOID        lpData)
{
  if (!lpCompare)
    lpList = NULL;
  if (lpList)
    for (lpList = lpList->lpPrev; lpList; lpList = lpList->lpPrev)
      if (lpCompare (lpList->lpValue, lpValue, lpData) == 0)
        break;
  return lpList;
}


/*  ja:線形リストの後の一致する要素を取得する
       lpList,線形リスト
      lpValue,値
    lpCompare,比較関数
       lpData,ユーザデータ
          RET,線形リスト                                                    */
LPLIST WINAPI
ListFindNext (LPLIST        lpList,
              LPVOID        lpValue,
              ListCompare_t lpCompare,
              LPVOID        lpData)
{
  if (!lpCompare)
    lpList = NULL;
  if (lpList)
    for (lpList = lpList->lpNext; lpList; lpList = lpList->lpNext)
      if (lpCompare (lpList->lpValue, lpValue, lpData) == 0)
        break;
  return lpList;
}


/*  ja:線形リストの最初に一致する要素を取得する
       lpList,線形リスト
      lpValue,値
    lpCompare,比較関数
       lpData,ユーザデータ
          RET,線形リスト                                                    */
LPLIST WINAPI
ListFindFirst (LPLIST        lpList,
               LPVOID        lpValue,
               ListCompare_t lpCompare,
               LPVOID        lpData)
{
  if (!lpCompare)
    lpList = NULL;
  for (lpList = ListFirst (lpList); lpList; lpList = lpList->lpNext)
    if (lpCompare (lpList->lpValue, lpValue, lpData) == 0)
      break;
  return lpList;
}


/*  ja:線形リストの最後に一致する要素を取得する
       lpList,線形リスト
      lpValue,値
    lpCompare,比較関数
       lpData,ユーザデータ
          RET,線形リスト                                                    */
LPLIST WINAPI
ListFindLast (LPLIST        lpList,
              LPVOID        lpValue,
              ListCompare_t lpCompare,
              LPVOID        lpData)
{
  if (!lpCompare)
    lpList = NULL;
  for (lpList = ListLast (lpList); lpList; lpList = lpList->lpPrev)
    if (lpCompare (lpList->lpValue, lpValue, lpData) == 0)
      break;
  return lpList;
}


/******************************************************************************
*                                                                             *
******************************************************************************/
/*  ja:線形リストの前の要素数を求める
    lpList,線形リスト
       RET,要素数                                                           */
int WINAPI
ListLengthPrev (LPLIST lpList)
{
 int nCount = 0;

  if (lpList)
    for (lpList = lpList->lpPrev; lpList; lpList = lpList->lpPrev)
      nCount++;
  return nCount;
}


/*  ja:線形リストの後の要素数を求める
    lpList,線形リスト
       RET,要素数                                                           */
int WINAPI
ListLengthNext (LPLIST lpList)
{
 int nCount = 0;

  if (lpList)
    for (lpList = lpList->lpNext; lpList; lpList = lpList->lpNext)
      nCount++;
  return nCount;
}


/*  ja:線形リストの要素数を求める
    lpList,線形リスト
       RET,要素数                                                           */
int WINAPI
ListLength (LPLIST lpList)
{
  return lpList ? ListLengthPrev (lpList) + ListLengthNext (lpList) + 1 : 0;
}


/*  ja:線形リストの前の一致する要素の数を求める
       lpList,線形リスト
      lpValue,値
    lpCompare,比較関数
       lpData,ユーザデータ
          RET,一致数                                                        */
int WINAPI
ListHasValuePrev (LPLIST        lpList,
                  LPVOID        lpValue,
                  ListCompare_t lpCompare,
                  LPVOID        lpData)
{
 int nCount = 0;

  if (lpList && lpCompare)
    for (lpList = lpList->lpPrev; lpList; lpList = lpList->lpPrev)
      if (lpCompare (lpList->lpValue, lpValue, lpData) == 0)
        nCount++;
  return nCount;
}


/*  ja:線形リストの後の一致する要素の数を求める
       lpList,線形リスト
      lpValue,値
    lpCompare,比較関数
       lpData,ユーザデータ
          RET,一致数                                                        */
int WINAPI
ListHasValueNext (LPLIST        lpList,
                  LPVOID        lpValue,
                  ListCompare_t lpCompare,
                  LPVOID        lpData)
{
 int nCount = 0;

  if (lpList && lpCompare)
    for (lpList = lpList->lpNext; lpList; lpList = lpList->lpNext)
      if (lpCompare (lpList->lpValue, lpValue, lpData) == 0)
        nCount++;
  return nCount;
}


/*  ja:線形リストの一致する要素の数を求める
       lpList,線形リスト
      lpValue,値
    lpCompare,比較関数
       lpData,ユーザデータ
          RET,一致数                                                        */
int WINAPI
ListHasValue (LPLIST        lpList,
              LPVOID        lpValue,
              ListCompare_t lpCompare,
              LPVOID        lpData)
{
  return lpList && lpCompare
                ? ListHasValuePrev (lpList, lpValue, lpCompare, lpData)
                + ListHasValueNext (lpList, lpValue, lpCompare, lpData)
                + (lpCompare (lpList->lpValue, lpValue, lpData) == 0 ? 1 : 0)
                : 0;
}
