/*
    wcommon
    copyright (c) 1998-2018 Kazuki Iwamoto https://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "wcommon.h"


/******************************************************************************
*                                                                             *
******************************************************************************/
/*  ja:ドロップされたファイル名を取得する
    hDrop,ファイル名構造体のハンドル
    iFile,インデックス
      RET,ファイル名,NULL:エラー                                            */
LPWSTR WINAPI
DragQueryFileNewW (HDROP hDrop,
                   UINT  iFile)
{
  LPWSTR lpszFile = NULL;

  if (hDrop && iFile != -1)
    {
      int nLength;

      nLength = DragQueryFileW (hDrop, iFile, NULL, 0);
      if (nLength > 0)
        {
          nLength += 1;
          lpszFile = MemoryAlloc (nLength * sizeof (WCHAR));
          DragQueryFileW (hDrop, iFile, lpszFile, nLength);
        }
    }
  return lpszFile;
}


/*  ja:ドロップされたファイル名を取得する
    hDrop,ファイル名構造体のハンドル
    iFile,インデックス
      RET,ファイル名,NULL:エラー                                            */
LPSTR WINAPI
DragQueryFileNewA (HDROP hDrop,
                   UINT  iFile)
{
  LPSTR lpszFile = NULL;

  if (hDrop && iFile != -1)
    {
      int nLength;

      nLength = DragQueryFileA (hDrop, iFile, NULL, 0);
      if (nLength > 0)
        {
          nLength += 1;
          lpszFile = MemoryAlloc (nLength * sizeof (CHAR));
          DragQueryFileA (hDrop, iFile, lpszFile, nLength);
        }
    }
  return lpszFile;
}


/*  ja:ウインドウのクラス名を取得する
    hWnd,ウインドウ
     RET,クラス名,NULL:エラー                                               */
LPWSTR WINAPI
GetClassNameNewW (HWND hWnd)
{
  int nResult;
  SSIZE_T nSize = 0;
  LPWSTR lpszClass = NULL;

  do
    {
      nSize += WCOMMON_BUFFER_SIZE;
      lpszClass = MemoryReAlloc (lpszClass, nSize * sizeof (WCHAR));
      nResult = GetClassNameW (hWnd, lpszClass, nSize);
    }
  while (nResult >= nSize - 1);
  if (nResult > 0)
    {
      lpszClass = MemoryReAlloc (lpszClass, (nResult + 1) * sizeof (WCHAR));
    }
  else
    {
      MemoryFree (lpszClass);
      lpszClass = NULL;
    }
  return lpszClass;
}


/*  ja:ウインドウのクラス名を取得する
    hWnd,ウインドウ
     RET,クラス名,NULL:エラー                                               */
LPSTR WINAPI
GetClassNameNewA (HWND hWnd)
{
  int nResult;
  SSIZE_T nSize = 0;
  LPSTR lpszClass = NULL;

  do
    {
      nSize += WCOMMON_BUFFER_SIZE;
      lpszClass = MemoryReAlloc (lpszClass, nSize * sizeof (CHAR));
      nResult = GetClassNameA (hWnd, lpszClass, nSize);
    }
  while (nResult >= nSize - 1);
  if (nResult > 0)
    {
      lpszClass = MemoryReAlloc (lpszClass, (nResult + 1) * sizeof (CHAR));
    }
  else
    {
      MemoryFree (lpszClass);
      lpszClass = NULL;
    }
  return lpszClass;
}


/*  ja:ウインドウのテキストを取得する
    hWnd,ウインドウ
     RET,テキスト,NULL:エラー                                               */
LPWSTR WINAPI
GetWindowTextNewW (HWND hWnd)
{
  int nLength;
  LPWSTR lpszText;

  nLength = GetWindowTextLengthW (hWnd);
  if (nLength > 0)
    {
      nLength++;
      lpszText = MemoryAlloc (nLength * sizeof (WCHAR));
      GetWindowTextW (hWnd, lpszText, nLength);
    }
  else
    {
      lpszText = NULL;
    }
  return lpszText;
}


/*  ja:ウインドウのテキストを取得する
    hWnd,ウインドウ
     RET,テキスト,NULL:エラー                                               */
LPSTR WINAPI
GetWindowTextNewA (HWND hWnd)
{
  int nLength;
  LPSTR lpszText;

  nLength = GetWindowTextLengthA (hWnd);
  if (nLength > 0)
    {
      nLength++;
      lpszText = MemoryAlloc (nLength * sizeof (CHAR));
      GetWindowTextA (hWnd, lpszText, nLength);
    }
  else
    {
      lpszText = NULL;
    }
  return lpszText;
}


/*  ja:メニュー項目のテキストを取得する
      hMenu,メニューのハンドル
    uIDItem,メニュー項目の識別子
      uFlag,オプション
        RET,テキスト,NULL:エラー                                               */
LPWSTR WINAPI
GetMenuStringNewW (HMENU hMenu,
                   UINT  uIDItem,
                   UINT  uFlag)
{
  int nLength;
  LPWSTR lpszText;

  nLength = GetMenuStringW (hMenu, uIDItem, NULL, 0, uFlag);
  if (nLength > 0)
    {
      nLength++;
      lpszText = MemoryAlloc (nLength * sizeof (WCHAR));
      GetMenuStringW (hMenu, uIDItem, lpszText, nLength, uFlag);
    }
  else
    {
      lpszText = NULL;
    }
  return lpszText;
}


/*  ja:メニュー項目のテキストを取得する
      hMenu,メニューのハンドル
    uIDItem,メニュー項目の識別子
      uFlag,オプション
        RET,テキスト,NULL:エラー                                               */
LPSTR WINAPI
GetMenuStringNewA (HMENU hMenu,
                   UINT  uIDItem,
                   UINT  uFlag)
{
  int nLength;
  LPSTR lpszText;

  nLength = GetMenuStringA (hMenu, uIDItem, NULL, 0, uFlag);
  if (nLength > 0)
    {
      nLength++;
      lpszText = MemoryAlloc (nLength * sizeof (CHAR));
      GetMenuStringA (hMenu, uIDItem, lpszText, nLength, uFlag);
    }
  else
    {
      lpszText = NULL;
    }
  return lpszText;
}


/*  ja:グローバルアトムに関連付けられている文字列を取得する
    nAtom,アトム識別子
      RET,文字列,NULL:エラー                                                */
LPWSTR WINAPI
GlobalGetAtomNameNewW (ATOM nAtom)
{
  LPWSTR lpszText = NULL;
  int nSize = 0;
  UINT uResult;

  do
    {
      nSize += WCOMMON_BUFFER_SIZE;
      lpszText = MemoryReAlloc (lpszText, nSize * sizeof (WCHAR));
      uResult = GlobalGetAtomNameW (nAtom, lpszText, nSize);
    }
  while (uResult <= 0 && GetLastError () == ERROR_MORE_DATA);
  if (uResult > 0)
    {
      lpszText = MemoryReAlloc (lpszText, (uResult + 1) * sizeof (WCHAR));
    }
  else
    {
      MemoryFree (lpszText);
      lpszText = NULL;
    }
  return lpszText;
  
}


/*  ja:グローバルアトムに関連付けられている文字列を取得する
    nAtom,アトム識別子
      RET,文字列,NULL:エラー                                                */
LPSTR WINAPI
GlobalGetAtomNameNewA (ATOM nAtom)
{
  LPSTR lpszText = NULL;
  int nSize = 0;
  UINT uResult;

  do
    {
      nSize += WCOMMON_BUFFER_SIZE;
      lpszText = MemoryReAlloc (lpszText, nSize * sizeof (CHAR));
      uResult = GlobalGetAtomNameA (nAtom, lpszText, nSize);
    }
  while (uResult <= 0 && GetLastError () == ERROR_MORE_DATA);
  if (uResult > 0)
    {
      lpszText = MemoryReAlloc (lpszText, (uResult + 1) * sizeof (CHAR));
    }
  else
    {
      MemoryFree (lpszText);
      lpszText = NULL;
    }
  return lpszText;
  
}


/*  ja:リソースから文字列を読み込む
    hInstance,インスタンス
          uID,ID
          RET,文字列                                                        */
LPWSTR WINAPI
LoadTextW (HINSTANCE hInstance,
           UINT      uID)
{
  LPWSTR lpszText = NULL;
  int nResult, nSize = 0;

  do
    {
      nSize += WCOMMON_BUFFER_SIZE;
      lpszText = MemoryReAlloc (lpszText, nSize * sizeof (WCHAR));
      nResult = LoadStringW (hInstance, uID, lpszText, nSize);
    }
  while (nResult <= 0 || nResult >= nSize - 1);
  if (nResult > 0)
    {
      lpszText = MemoryReAlloc (lpszText, (nResult + 1) * sizeof (WCHAR));
    }
  else
    {
      MemoryFree (lpszText);
      lpszText = NULL;
    }
  return lpszText;
}


/*  ja:リソースから文字列を読み込む
    hInstance,インスタンス
          uID,ID
          RET,文字列                                                        */
LPSTR WINAPI
LoadTextA (HINSTANCE hInstance,
           UINT      uID)
{
  LPSTR lpszText = NULL;
  int nResult, nSize = 0;

  do
    {
      nSize += WCOMMON_BUFFER_SIZE;
      lpszText = MemoryReAlloc (lpszText, nSize * sizeof (CHAR));
      nResult = LoadStringA (hInstance, uID, lpszText, nSize);
    }
  while (nResult <= 0 || nResult >= nSize - 1);
  if (nResult > 0)
    {
      lpszText = MemoryReAlloc (lpszText, (nResult + 1) * sizeof (CHAR));
    }
  else
    {
      MemoryFree (lpszText);
      lpszText = NULL;
    }
  return lpszText;
}


/*  ja:スクロールバーを設定する
     hWnd,ウインドウ
     nBar,スクロールバータイプ
     nMin,最小値
     nMax,最大値
    nPage,ページ
     nPos,位置                                                              */
VOID WINAPI
SetScrollBar (HWND hWnd,
              int  nBar,
              int  nMin,
              int  nMax,
              UINT nPage,
              int  nPos)
{
  SCROLLINFO si;

  si.cbSize = sizeof (SCROLLINFO);
  si.fMask = SIF_PAGE | SIF_POS | SIF_RANGE | SIF_DISABLENOSCROLL;
  si.nMin = nMin;
  si.nMax = nMax;
  si.nPage = nPage;
  si.nPos = nPos;
  SetScrollInfo (hWnd, nBar, &si, TRUE);
}
