/*
    Text maid for Windows
    copyright (c) 1998-2018 Kazuki Iwamoto https://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "history.h"
#include <commctrl.h>
#include "resource.h"
#include "wcommon.h"


/******************************************************************************
*                                                                             *
* ja:ファイルの履歴関数                                                       *
*                                                                             *
******************************************************************************/
static BOOL CALLBACK
HistoryDlgProc (HWND   hDlg,
                UINT   uMsg,
                WPARAM wParam,
                LPARAM lParam)
{
  switch (uMsg)
    {
      case WM_INITDIALOG:
        SetWindowLongPtr (hDlg, DWLP_USER, lParam);
        /* ja:エディトコントロールの設定 */
        SetDlgItemInt (hDlg, IDC_EDITB1, *(int *)lParam, TRUE);
        /* ja:アップダウンコントロールの設定 */
        UpDownItem_SetRange (hDlg, IDC_UPDOWNB1, 0, 9);
        EnableDlgItem (hDlg, IDOK, FALSE);
        return TRUE;
      case WM_COMMAND:
        switch (LOWORD (wParam))
          {
            case IDOK:
              *(int *)GetWindowLongPtr (hDlg, DWLP_USER)
                                = GetDlgItemInt (hDlg, IDC_EDITB1, NULL, TRUE);
            case IDCANCEL:
              EndDialog (hDlg, LOWORD (wParam));
              return TRUE;
            case IDC_EDITB1:
              if (HIWORD (wParam) == EN_CHANGE)
                {
                  int nHistory;

                  nHistory = GetDlgItemInt (hDlg, IDC_EDITB1, NULL, TRUE);
                  EnableDlgItem (hDlg, IDOK, 0 <= nHistory && nHistory <= 9);
                }
          }
        return TRUE;
    }
  return FALSE;
}


/*  ja:ファイルの履歴に加える
     lpszFile,ファイル名
         hWnd,ウインドウ
     nHistory,最大履歴数
     nMenuTop,トップメニューの項目数
    nMenuFile,ファイルの項目の数                                            */
VOID
SetHistory (LPCTSTR lpszFile,
            HWND    hWnd,
            int     nHistory,
            int     nMenuTop,
            int     nMenuFile)
{
  if (1 < nHistory && nHistory <= 9)
    {
      int i, nCount;
      HMENU hMenu;
      LPTSTR lpszText;

      hMenu = GetMenu (hWnd);
      hMenu = GetSubMenu (hMenu, GetMenuItemCount (hMenu) != nMenuTop);
      nCount = GetMenuItemCount (hMenu);
      for (i = 0; i < nCount - nMenuFile - 1; i++)
        {
          BOOL fCompare;

          lpszText = GetMenuStringNew (hMenu, i + nMenuFile - 1,
                                                                MF_BYPOSITION);
          fCompare = lstrlen (lpszText) > 3;
          if (fCompare)
            {
              lpszText[1] = (TCHAR)(i + '2');
              ModifyMenu (hMenu, i + nMenuFile - 1,
                                            MF_BYPOSITION, i + 201, lpszText);
              fCompare = CompareString (LOCALE_USER_DEFAULT,
                        NORM_IGNORECASE, lpszFile, -1, lpszText + 3, -1) == 2;
            }
          MemoryFree (lpszText);
          if (fCompare)
            {
              DeleteMenu (hMenu, i + nMenuFile - 1, MF_BYPOSITION);
              nCount--;
              break;
            }
        }
      lpszText = StringJoin (NULL, _T("&1 "), lpszFile, NULL);
      InsertMenu (hMenu, nMenuFile - 1, MF_BYPOSITION, 200, lpszText);
      MemoryFree (lpszText);
      if (nCount == nMenuFile)
        InsertMenu (hMenu, nMenuFile, MF_BYPOSITION | MF_SEPARATOR, 0, NULL);
      if (nCount == nHistory + nMenuFile + 1)
        DeleteMenu (hMenu, nCount - 2, MF_BYPOSITION);
      DrawMenuBar (hWnd);
    }
}


/*  ja:ファイルの履歴を取得する
     hWnd,ウインドウ
    uItem,コマンド項目
      RET,ファイル名,NULL:エラー                                            */
LPTSTR
GetHistory (HWND hWnd,
            UINT uItem)
{
  int nLength;
  HMENU hMenu;
  LPTSTR lpszFile;

  hMenu = GetMenu (hWnd);
  lpszFile = GetMenuStringNew (hMenu, uItem, 0);
  nLength = lstrlen (lpszFile);
  if (nLength > 3)
    {
      MemoryCopy (lpszFile, lpszFile + 3, (nLength - 2) * sizeof (TCHAR));
    }
  else
    {
      MemoryFree (lpszFile);
      lpszFile = NULL;
    }
  return lpszFile;
}


/*  ja:最大履歴数を変更する
         hWnd,親ウインドウ
     nHistory,最大履歴数
     nMenuTop,トップメニューの項目数
    nMenuFile,ファイルの項目の数                                            */
VOID
ModifyHistory (HWND hWnd,
               int  nHistory,
               int  nMenuTop,
               int  nMenuFile)
{
  int i;
  HMENU hMenu;

  hMenu = GetMenu (hWnd);
  hMenu = GetSubMenu (hMenu, GetMenuItemCount (hMenu) != nMenuTop);
  for (i = GetMenuItemCount (hMenu) - 3;
                        i > nHistory - (nHistory == 0) + nMenuFile - 2; i--)
    DeleteMenu (hMenu, i, MF_BYPOSITION);
  DrawMenuBar (hWnd);
}


/*  ja:最大履歴数を変更する
             hInst,インスタンスハンドル
    lpTemplateName,テンプレート名
              hWnd,親ウインドウ
          nHistory,最大履歴数
          nMenuTop,トップメニューの項目数
         nMenuFile,ファイルの項目の数
               RET,最大履歴数                                               */
int
ModifyHistoryDlg (HINSTANCE hInst,
                  LPCTSTR   lpTemplateName,
                  HWND      hWnd,
                  int       nHistory,
                  int       nMenuTop,
                  int       nMenuFile)
{
  if (DialogBoxParamGUI (hInst, lpTemplateName,
                            hWnd, HistoryDlgProc, (LPARAM)&nHistory) == IDOK)
    ModifyHistory (hWnd, nHistory, nMenuTop, nMenuFile);
  return nHistory;
}
