/*
    wcommon
    copyright (c) 1998-2018 Kazuki Iwamoto https://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#ifndef __WLIST_H__
#define __WLIST_H__


#include "wcommon.h"


#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */


/******************************************************************************
*                                                                             *
******************************************************************************/
typedef struct _LIST *LPLIST;
typedef int (WINAPI *ListCompare_t)(LPVOID lpValue1, LPVOID lpValue2,
                                                                LPVOID lpData);
typedef VOID (WINAPI *ListFree_t)(LPVOID lpValue, LPVOID lpData);
typedef BOOL (CALLBACK *ListEnumProc_t)(LPVOID lpValue, LPVOID lpData);


/*  ja:値を直接比較する
    lpValue1,値1
    lpValue2,値2
      lpData,ユーザデータ
         RET,負:値1が小さい,0:等しい,正:値1が大きい                         */
int WINAPI
ListCompareDirect (LPVOID lpValue1,
                   LPVOID lpValue2,
                   LPVOID lpData);


/*  ja:値を文字列として比較する
    lpValue1,値1
    lpValue2,値2
      lpData,ユーザデータ
         RET,負:値1が小さい,0:等しい,正:値1が大きい                         */
int WINAPI
ListCompareStringW (LPVOID lpValue1,
                    LPVOID lpValue2,
                    LPVOID lpData);


/*  ja:値を文字列として比較する
    lpValue1,値1
    lpValue2,値2
      lpData,ユーザデータ
         RET,負:値1が小さい,0:等しい,正:値1が大きい                         */
int WINAPI
ListCompareStringA (LPVOID lpValue1,
                    LPVOID lpValue2,
                    LPVOID lpData);


/*  ja:線形リストの値を解放する
    lpValue,値
     lpData,ユーザデータ                                                    */
VOID WINAPI
ListFree (LPVOID lpValue,
          LPVOID lpData);


/*  ja:線形リストを作る
    lpValue,値
        RET,線形リスト                                                      */
LPLIST WINAPI
ListNew (LPVOID lpValue);


#ifdef UNICODE
# define ListCompareString ListCompareStringW
#else /* not UNICODE */
# define ListCompareString ListCompareStringA
#endif /* not UNICODE */


/******************************************************************************
*                                                                             *
******************************************************************************/
/*  ja:線形リストの前を取得する
    lpList,線形リスト
       RET,線形リスト                                                       */
LPLIST WINAPI
ListPrev (LPLIST lpList);


/*  ja:線形リストの後を取得する
    lpList,線形リスト
       RET,線形リスト                                                       */
LPLIST WINAPI
ListNext (LPLIST lpList);


/*  ja:線形リストの先頭を取得する
    lpList,線形リスト
       RET,線形リスト                                                       */
LPLIST WINAPI
ListFirst (LPLIST lpList);


/*  ja:線形リストの末尾を取得する
    lpList,線形リスト
       RET,線形リスト                                                       */
LPLIST WINAPI
ListLast (LPLIST lpList);


/*  ja:線形リストの中間を取得する
    lpList,線形リスト
       RET,線形リスト                                                       */
LPLIST WINAPI
ListMiddle (LPLIST lpList);


/*  ja:線形リストのn番目を求める
    lpList,線形リスト
    nIndex,n番(負:後方から,正or0:前方から)
       RET,線形リスト(NULL:要素なし)                                        */
LPLIST WINAPI
ListIndex (LPLIST lpList,
           int    nIndex);


/******************************************************************************
*                                                                             *
******************************************************************************/
/*  ja:線形リストの前に線形リストを挿入する
      lpList,線形リスト
    lpInsert,挿入する線形リスト
         RET,線形リスト                                                     */
LPLIST WINAPI
ListLinkPrev (LPLIST lpList,
              LPLIST lpInsert);


/*  ja:線形リストの後に線形リストを挿入する
      lpList,線形リスト
    lpInsert,挿入する線形リスト
         RET,線形リスト                                                     */
LPLIST WINAPI
ListLinkNext (LPLIST lpList,
              LPLIST lpInsert);


/*  ja:線形リストの先頭に線形リストを追加する
     lpList,線形リスト
    lpFirst,追加する線形リスト
        RET,線形リスト                                                      */
LPLIST WINAPI
ListLinkFirst (LPLIST lpList,
               LPLIST lpFirst);


/*  ja:線形リストの末尾に線形リストを追加する
    lpList,線形リスト
    lpLast,追加する線形リスト
       RET,線形リスト                                                       */
LPLIST WINAPI
ListLinkLast (LPLIST lpList,
              LPLIST lpLast);


/*  ja:線形リストの前で分離する
    lpList,線形リスト
       RET,線形リスト                                                       */
LPLIST WINAPI
ListUnlinkPrev (LPLIST lpList);


/*  ja:線形リストの後で分離する
    lpList,線形リスト
       RET,線形リスト                                                       */
LPLIST WINAPI
ListUnlinkNext (LPLIST lpList);


/*  ja:線形リストから取り除く
    lpList,線形リスト
       RET,線形リスト                                                       */
LPLIST WINAPI
ListRemove (LPLIST lpList);


/******************************************************************************
*                                                                             *
******************************************************************************/
/*  ja:線形リストを削除する
    lpList,線形リスト
    lpFree,解放関数
    lpData,ユーザデータ
       RET,線形リスト                                                       */
LPLIST WINAPI
ListDelete (LPLIST     lpList,
            ListFree_t lpFree,
            LPVOID     lpData);


/*  ja:線形リストの前を破棄する
    lpList,線形リスト
    lpFree,解放関数
    lpData,ユーザデータ                                                     */
VOID WINAPI
ListDestroyPrev (LPLIST     lpList,
                 ListFree_t lpFree,
                 LPVOID     lpData);


/*  ja:線形リストの後を破棄する
    lpList,線形リスト
    lpFree,解放関数
    lpData,ユーザデータ                                                     */
VOID WINAPI
ListDestroyNext (LPLIST     lpList,
                 ListFree_t lpFree,
                 LPVOID     lpData);


/*  ja:線形リストを破棄する
    lpList,線形リスト
    lpFree,解放関数
    lpData,ユーザデータ                                                     */
VOID WINAPI
ListDestroy (LPLIST     lpList,
             ListFree_t lpFree,
             LPVOID     lpData);


/*  ja:線形リストに線形リストを追加する
      lpList1,線形リスト
      lpList2,線形リスト
    lpCompare,比較関数
       lpData,ユーザデータ
          RET,線形リスト                                                    */
LPLIST WINAPI
ListMerge (LPLIST        lpList1,
           LPLIST        lpList2,
           ListCompare_t lpCompare,
           LPVOID        lpData);


/*  ja:線形リストを並び替える
       lpList,線形リスト
    lpCompare,比較関数
       lpData,ユーザデータ
          RET,線形リスト                                                    */
LPLIST WINAPI
ListSort (LPLIST        lpList,
          ListCompare_t lpCompare,
          LPVOID        lpData);


/*  ja:線形リストのn番目まで拡張する
    lpList,線形リスト
    nIndex,n番(負:後方から,正or0:前方から)
       RET,線形リスト                                                       */
LPLIST WINAPI
ListExtend (LPLIST lpList,
            int    nIndex);


/*  ja:線形リストのn番目まで拡張または破棄する
    lpList,線形リスト
    nIndex,n番(負:後方から,正or0:前方から)
    lpFree,解放関数
    lpData,ユーザデータ
       RET,線形リスト                                                       */
LPLIST WINAPI
ListTruncate (LPLIST     lpList,
              int        nIndex,
              ListFree_t lpFree,
              LPVOID     lpData);


/******************************************************************************
*                                                                             *
******************************************************************************/
/*  ja:線形リストの値を取得する
    lpList,線形リスト
       RET,値                                                               */
LPVOID WINAPI
ListGetValue (LPLIST lpList);


/*  ja:線形リストの値を設定する
     lpList,線形リスト
    lpValue,値
        RET,前の値(値:エラー)                                               */
LPVOID WINAPI
ListSetValue (LPLIST lpList,
              LPVOID lpValue);


/*  ja:線形リストのn番目の値を取得する
    lpList,線形リスト
    nIndex,n番(負:後方から,正or0:前方から)
       RET,値                                                               */
LPVOID WINAPI
ListGetValueIndex (LPLIST lpList,
                   int    nIndex);


/*  ja:線形リストのn番目の値を設定する
     lpList,線形リスト
     nIndex,n番(負:後方から,正or0:前方から)
    lpValue,値
        RET,前の値(値:エラー)                                               */
LPVOID WINAPI
ListSetValueIndex (LPLIST lpList,
                   int    nIndex,
                   LPVOID lpValue);


/*  ja:線形リストのn番目の値を設定する(拡張)
     lpList,線形リスト
     nIndex,n番(負:後方から,正or0:前方から)
    lpValue,値
        RET,前の値                                                          */
LPVOID WINAPI
ListSetValueExtend (LPLIST lpList,
                    int    nIndex,
                    LPVOID lpValue);


/*  ja:線形リストの前に値を挿入する
     lpList,線形リスト
    lpValue,挿入する値
        RET,線形リスト                                                      */
LPLIST WINAPI
ListInsertPrev (LPLIST lpList,
                LPVOID lpValue);


/*  ja:線形リストの後に値を挿入する
     lpList,線形リスト
    lpValue,挿入する値
        RET,線形リスト                                                      */
LPLIST WINAPI
ListInsertNext (LPLIST lpList,
                LPVOID lpValue);


/*  ja:線形リストに値を追加する
       lpList,線形リスト
      lpValue,線形リスト
    lpCompare,比較関数
       lpData,ユーザデータ
          RET,線形リスト                                                    */
LPLIST WINAPI
ListInsert (LPLIST        lpList,
            LPVOID        lpValue,
            ListCompare_t lpCompare,
            LPVOID        lpData);


/*  ja:線形リストの先頭に値を追加する
     lpList,線形リスト
    lpValue,値
        RET,線形リスト                                                      */
LPLIST WINAPI
ListPrepend (LPLIST lpList,
             LPVOID lpValue);


/*  ja:線形リストの末尾に値を追加する
     lpList,線形リスト
    lpValue,値
        RET,線形リスト                                                      */
LPLIST WINAPI
ListAppend (LPLIST lpList,
            LPVOID lpValue);


/*  ja:線形リストの要素を列挙する
    lpList,線形リスト
    lpProc,列挙関数
    lpData,ユーザデータ
       RET,TRUE:すべて列挙,FALSE:中断                                       */
BOOL WINAPI
ListEnum (LPLIST         lpList,
          ListEnumProc_t lpProc,
          LPVOID         lpData);


/******************************************************************************
*                                                                             *
******************************************************************************/
/*  ja:線形リストの前の一致する要素を取得する
       lpList,線形リスト
      lpValue,値
    lpCompare,比較関数
       lpData,ユーザデータ
          RET,線形リスト                                                    */
LPLIST WINAPI
ListFindPrev (LPLIST        lpList,
              LPVOID        lpValue,
              ListCompare_t lpCompare,
              LPVOID        lpData);


/*  ja:線形リストの後の一致する要素を取得する
       lpList,線形リスト
      lpValue,値
    lpCompare,比較関数
       lpData,ユーザデータ
          RET,線形リスト                                                    */
LPLIST WINAPI
ListFindNext (LPLIST        lpList,
              LPVOID        lpValue,
              ListCompare_t lpCompare,
              LPVOID        lpData);


/*  ja:線形リストの最初に一致する要素を取得する
       lpList,線形リスト
      lpValue,値
    lpCompare,比較関数
       lpData,ユーザデータ
          RET,線形リスト                                                    */
LPLIST WINAPI
ListFindFirst (LPLIST        lpList,
               LPVOID        lpValue,
               ListCompare_t lpCompare,
               LPVOID        lpData);


/*  ja:線形リストの最後に一致する要素を取得する
       lpList,線形リスト
      lpValue,値
    lpCompare,比較関数
       lpData,ユーザデータ
          RET,線形リスト                                                    */
LPLIST WINAPI
ListFindLast (LPLIST        lpList,
              LPVOID        lpValue,
              ListCompare_t lpCompare,
              LPVOID        lpData);


/******************************************************************************
*                                                                             *
******************************************************************************/
/*  ja:線形リストの前の要素数を求める
    lpList,線形リスト
       RET,要素数                                                           */
int WINAPI
ListLengthPrev (LPLIST lpList);


/*  ja:線形リストの後の要素数を求める
    lpList,線形リスト
       RET,要素数                                                           */
int WINAPI
ListLengthNext (LPLIST lpList);


/*  ja:線形リストの要素数を求める
    lpList,線形リスト
       RET,要素数                                                           */
int WINAPI
ListLength (LPLIST lpList);


/*  ja:線形リストの前の一致する要素の数を求める
       lpList,線形リスト
      lpValue,値
    lpCompare,比較関数
       lpData,ユーザデータ
          RET,一致数                                                        */
int WINAPI
ListHasValuePrev (LPLIST        lpList,
                  LPVOID        lpValue,
                  ListCompare_t lpCompare,
                  LPVOID        lpData);


/*  ja:線形リストの後の一致する要素の数を求める
       lpList,線形リスト
      lpValue,値
    lpCompare,比較関数
       lpData,ユーザデータ
          RET,一致数                                                        */
int WINAPI
ListHasValueNext (LPLIST        lpList,
                  LPVOID        lpValue,
                  ListCompare_t lpCompare,
                  LPVOID        lpData);


/*  ja:線形リストの一致する要素の数を求める
       lpList,線形リスト
      lpValue,値
    lpCompare,比較関数
       lpData,ユーザデータ
          RET,一致数                                                        */
int WINAPI
ListHasValue (LPLIST        lpList,
              LPVOID        lpValue,
              ListCompare_t lpCompare,
              LPVOID        lpData);


#ifdef __cplusplus
}
#endif /* __cplusplus */


#endif /* __WLIST_H__ */
