/*
    Text maid
    copyright (c) 1998-2004 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include <gdk/gdkkeysyms.h>
#include "charlist.h"
#include "charset.h"
#include "edit.h"
#include "ftype.h"
#include "general.h"
#include "prop.h"
#include "sigfile.h"
#include "misc/misc.h"


/******************************************************************************
*                                                                             *
* プロパティ関数群                                                            *
*                                                                             *
******************************************************************************/
/*  プロパティをコピーする
    ft_dst,コピー先
    ft_src,コピー元                                                         */
void
property_copy (FileType *ft_dst, FileType *ft_src)
{
    ft_dst->ft_id = ft_src->ft_id;
    if (ft_dst->crlf < 3 && ft_src->crlf < 3)
        ft_dst->crlf = ft_src->crlf;
    ft_dst->margin = ft_src->margin;
    ft_dst->tab = ft_src->tab;
    ft_dst->writemode = ft_src->writemode;
    ft_dst->autoindent = ft_src->autoindent;
    ft_dst->code = ft_src->code;
    ft_dst->eof = ft_src->eof;
    ft_dst->hexcase = ft_src->hexcase;
    ft_dst->html = ft_src->html;
    ft_dst->esc = ft_src->esc;
    ft_dst->limit = ft_src->limit;
    ft_dst->overwrite = ft_src->overwrite;
    ft_dst->space = ft_src->space;
    ft_dst->syscol = ft_src->syscol;
    ft_dst->tabconv = ft_src->tabconv;
    ft_dst->gline = ft_src->gline;
    ft_dst->mline = ft_src->mline;
    ft_dst->nline = ft_src->nline;
    ft_dst->uline = ft_src->uline;
    ft_dst->vline = ft_src->vline;
    g_free (ft_dst->font_name);
    ft_dst->font_name = ft_src->font_name != NULL
                                        ? g_strdup (ft_src->font_name) : NULL;
    g_memmove (ft_dst->color, ft_src->color, sizeof (GdkColor) * 12);
}


/*  プロパティを変更する
    tmaid,TXTウインドウ情報
       ft,ファイルタイプ                                                    */
void property_change(TmaidWindow *tmaid,FileType *ft)
{
    gboolean modify;
    gint max,sx,sy;

    if (tmaid->ft.limit != ft->limit || tmaid->ft.tab != ft->tab
                        || (tmaid->ft.code && tmaid->ft.margin != ft->margin))
      {
        /* 折り返しの設定変更、タブ数変更 */
        delete_list(&tmaid->undo);
        delete_list(&tmaid->redo);
      }
    modify = tmaid->ft.limit != ft->limit || (tmaid->ft.limit
                && (tmaid->ft.tab != ft->tab || tmaid->ft.margin != ft->margin
                    || g_strcmp (tmaid->ft.font_name, ft->font_name) != 0));
    property_copy (&tmaid->ft, ft);
    tmaid->select.x=-1;
    /* フォント */
    gtk_widget_modify_font (window, tmaid->ft.font_name != NULL
                    ? pango_font_description_from_string (tmaid->ft.font_name)
                    : pango_font_description_copy (system_font));
    tmaid->layout = gtk_widget_create_pango_layout (window, NULL);
    charset_size (tmaid->layout, &tmaid->font_width, &tmaid->font_height,
                                                            tmaid->font_buf);
    if (modify) {
        /* 折り返しの設定変更、
                            折り返しが有効でタブ数や右マージンやフォント変更 */
        tmaid->cursor.x=tmaid->cursor.y=tmaid->top.x=tmaid->top.y=0;
        modify_margin(tmaid);
    }
    /* スクロールバー */
    sx=MAX(tmaid->drawing->allocation.width/tmaid->font_width,1);
    sy=MAX(tmaid->drawing->allocation.height/tmaid->font_height,1);
    if (tmaid->top.y>tmaid->max-sy)
        tmaid->top.y=MAX(tmaid->max-sy,0);
    max=edit_get_width_max(tmaid);
    if (tmaid->top.x>max-sx+1)
        tmaid->top.x=MAX(max-sx+1,0);
    misc_set_scroll_bar (tmaid->hscroll,
                            G_CALLBACK (signal_value_changed_hscroll), tmaid,
                                            0, max + 1, sx, tmaid->top.x);
    misc_set_scroll_bar (tmaid->vscroll,
                            G_CALLBACK (signal_value_changed_vscroll), tmaid,
                                            0, tmaid->max, sy, tmaid->top.y);
    /* 再描画 */
    gtk_widget_draw(tmaid->drawing,NULL);
    draw_caret(tmaid,NULL);
}


/******************************************************************************
*                                                                             *
* プロパティダイアログ関数群                                                  *
*                                                                             *
******************************************************************************/
static gboolean property_clicked_ok,color_list[12];
static gchar *charset, *font_name;
static gint ft_id;
static GdkColor gdk_color[12];
static GtkListStore *store;
static GtkWidget *dialog;
static GtkWidget *button0, *button1, *button4, *combo0, *combo1;
static GtkWidget *check0, *check1, *check2, *check3, *check4;
static GtkWidget *check5, *check6, *check7, *check8, *check9;
static GtkWidget *check10, *check11, *check12, *check13, *check14, *check15;
static GtkWidget *radio0, *radio1, *radio2, *radio3, *radio4, *radio5, *radio6;
static GtkWidget *tview, *entry0, *entry1, *label, *spin0, *spin1;


/* OKボタンの表示/非表示を判定 */
static void
property_dialog_ok_sensitive (GtkWidget *widget, gpointer user_data)
{
    gtk_widget_set_sensitive (button4,
                (entry0 == NULL
                    || g_strlen (gtk_entry_get_text (GTK_ENTRY (entry0))) > 0)
                && (entry1 == NULL
                    || g_strlen (gtk_entry_get_text (GTK_ENTRY (entry1))) > 0)
                && (combo1 == NULL || charlist_is_valid (gtk_entry_get_text
                                    (GTK_ENTRY (GTK_COMBO (combo1)->entry)))));
}


/* システム色 */
static void
property_dialog_toggled_check (GtkWidget *widget, gpointer user_data)
{
    gtk_tree_selection_unselect_all
                        (gtk_tree_view_get_selection (GTK_TREE_VIEW (tview)));
    gtk_widget_set_sensitive (tview,
                !gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (check12)));
    gtk_widget_set_sensitive (button0, FALSE);
    property_dialog_ok_sensitive (widget, user_data);
}


/* ラジオボタン */
static void
property_dialog_toggled_radio (GtkWidget *widget, gpointer user_data)
{
    gboolean result;

    result = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(radio3))
                    || gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(radio4));
    gtk_widget_set_sensitive (radio5, result);
    gtk_widget_set_sensitive (radio6, result);
    property_dialog_ok_sensitive (widget, user_data);
}


/* リストボックスの選択 */
static void
property_dialog_changed (GtkTreeSelection *select, gpointer user_data)
{
    gint i, row;
    GtkTreeIter iter;

    row = -1;
    for (i = 0; i < 12; i++)
        if (gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (store),
                                                                &iter, NULL, i)
                        && gtk_tree_selection_iter_is_selected (select, &iter))
          {
            row = i;
            break;
          }
    gtk_widget_set_sensitive (button0, row >= 0);
}


/* 色の設定でOKが押されたとき */
static void property_dialog_clicked_color_ok(GtkWidget *widget,
                                                    GtkWidget *dialog_color)
{
    gchar *xpm[18];
    gint i;
    gdouble color[4];
    GtkTreeIter iter;

    gtk_color_selection_get_color(GTK_COLOR_SELECTION(
                    GTK_COLOR_SELECTION_DIALOG(dialog_color)->colorsel),color);
    xpm[0]="24 16 1 1";
    for (i=2;i<18;i++)
        xpm[i]="XXXXXXXXXXXXXXXXXXXXXXXX";
    for (i = 0; i < 12; i++)
        if (gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (store),
                                                                &iter, NULL, i)
            && gtk_tree_selection_iter_is_selected
                (gtk_tree_view_get_selection (GTK_TREE_VIEW (tview)), &iter))
          {
            gdk_color[i].red = color[0] * 0xffff;
            gdk_color[i].green = color[1] * 0xffff;
            gdk_color[i].blue = color[2] * 0xffff;
            xpm[1] = g_strdup_printf ("X\tc #%02X%02X%02X",
                                                    gdk_color[i].red / 256,
                                                    gdk_color[i].green / 256,
                                                    gdk_color[i].blue / 256);
            gtk_list_store_set (store, &iter,
                    0, gdk_pixbuf_new_from_xpm_data ((const char **)xpm), -1);
            g_free (xpm[1]);
          }
    property_dialog_ok_sensitive(NULL,NULL);
    gtk_widget_destroy(dialog_color);
}


/* 色の設定ボタンが押された */
static void property_dialog_clicked_color(GtkWidget *widget,gpointer user_data)
{
    gint i;
    gdouble color[4];
    GtkWidget *dialog_color;

    dialog_color=gtk_color_selection_dialog_new(_("Color Configuration"));
    g_signal_connect (G_OBJECT (dialog_color), "destroy",
                                            G_CALLBACK (gtk_main_quit), NULL);
    g_signal_connect (G_OBJECT (
        GTK_COLOR_SELECTION_DIALOG (dialog_color)->ok_button),
        "clicked",G_CALLBACK (property_dialog_clicked_color_ok), dialog_color);
    g_signal_connect_swapped (
        G_OBJECT (GTK_COLOR_SELECTION_DIALOG (dialog_color)->cancel_button),
        "clicked", G_CALLBACK (gtk_widget_destroy), G_OBJECT (dialog_color));
    for (i=0;i<12;i++)
        if (color_list[i]) {
            color[0]=(gdouble)gdk_color[i].red/0xffff;
            color[1]=(gdouble)gdk_color[i].green/0xffff;
            color[2]=(gdouble)gdk_color[i].blue/0xffff;
            color[3]=0;
            gtk_color_selection_set_color(GTK_COLOR_SELECTION(
                    GTK_COLOR_SELECTION_DIALOG(dialog_color)->colorsel),color);
            break;
        }

    gtk_widget_show(dialog_color);
    gtk_grab_add(dialog_color);
    gtk_main();
}


/* キャラクターセットの設定ボタンが押された */
static void
property_dialog_clicked_charset (GtkWidget *widget, gpointer user_data)
{
    gchar *result;

    result = charlist_dialog (charset);
    if (result != NULL)
      {
        g_free (charset);
        charset = result;
        property_dialog_ok_sensitive (NULL, NULL);
      }
}


/* フォントの設定でOKが押されたとき */
static void property_dialog_clicked_font_ok(GtkWidget *widget,
                                                        GtkWidget *dialog_font)
{
    gchar *name;

    name=gtk_font_selection_dialog_get_font_name(
                                    GTK_FONT_SELECTION_DIALOG(dialog_font));
    if (name!=NULL && name[0]!='\0') {
        g_free(font_name);
        font_name=name;
        property_dialog_ok_sensitive(NULL,NULL);
    }
    gtk_widget_destroy(dialog_font);
}


/* フォントボタンが押された */
static void property_dialog_clicked_font(GtkWidget *widget,gpointer user_data)
{
    GtkWidget *dialog_font;

    dialog_font=gtk_font_selection_dialog_new(_("Font"));
    g_signal_connect (G_OBJECT (dialog_font), "destroy",
                                            G_CALLBACK (gtk_main_quit), NULL);
    g_signal_connect (G_OBJECT (
        GTK_FONT_SELECTION_DIALOG (dialog_font)->ok_button),
        "clicked", G_CALLBACK (property_dialog_clicked_font_ok), dialog_font);
    g_signal_connect_swapped (
        G_OBJECT (GTK_FONT_SELECTION_DIALOG (dialog_font)->cancel_button),
        "clicked", G_CALLBACK (gtk_widget_destroy), G_OBJECT (dialog_font));
    if (font_name!=NULL)
        gtk_font_selection_dialog_set_font_name(
                            GTK_FONT_SELECTION_DIALOG(dialog_font),font_name);
    gtk_widget_show(dialog_font);
    gtk_grab_add(dialog_font);
    gtk_main();
}


/* ファイルタイプボタンが押された */
static void property_dialog_clicked_ftype(GtkWidget *widget,gpointer user_data)
{
    gint pos;

    if (ftype_dialog(&pos)) {
        if (pos>=0) {
            ft_id=ftype[pos].ft_id;
            gtk_label_set_text(GTK_LABEL(label),ftype[pos].text);
            gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check0),
                                                        ftype[pos].gline);
            gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check1),
                                                        ftype[pos].space);
            gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check2),
                                                        ftype[pos].nline);
            gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check3),
                                                        ftype[pos].mline);
            gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check4),
                                                        ftype[pos].limit);
            gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check5),
                                                        ftype[pos].code);
            gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check6),
                                                        ftype[pos].uline);
            gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check7),
                                                        ftype[pos].vline);
            gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check8),
                                                        ftype[pos].autoindent);
            gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (check9),
                                                        ftype[pos].tabconv);
            gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check10),
                                                        ftype[pos].eof);
            gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check11),
                                                        ftype[pos].overwrite);
            gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check12),
                                                        ftype[pos].syscol);
            switch (ftype[pos].writemode)
              {
                case  1:gtk_toggle_button_set_active
                                    (GTK_TOGGLE_BUTTON (radio1), TRUE); break;
                case  2:gtk_toggle_button_set_active
                                    (GTK_TOGGLE_BUTTON (radio2), TRUE); break;
                case  3:gtk_toggle_button_set_active
                                    (GTK_TOGGLE_BUTTON (radio3), TRUE); break;
                case  4:gtk_toggle_button_set_active
                                    (GTK_TOGGLE_BUTTON (radio4), TRUE); break;
                default:gtk_toggle_button_set_active
                                    (GTK_TOGGLE_BUTTON (radio0), TRUE);
              }
            gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (ftype[pos].hexcase
                                                    ? radio5 : radio6), TRUE);
            if (ftype[pos].crlf < 3)
                gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (combo0)->entry),
                    gtk_label_get_text (GTK_LABEL (gtk_bin_get_child (GTK_BIN
                    (g_list_nth_data (gtk_container_get_children
                        (GTK_CONTAINER (GTK_COMBO (combo0)->list)),
                                                ftype[pos].crlf))))));
            gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin0),
                                                        ftype[pos].margin);
            gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin1),ftype[pos].tab);
            g_memmove(gdk_color,ftype[pos].color,sizeof(GdkColor)*12);
            g_free(font_name);
            font_name=g_strdup(ftype[pos].font_name);
            gtk_widget_set_sensitive(button4,TRUE);
        } else {
            ft_id=-1;
            gtk_label_set_text(GTK_LABEL(label),_("Independent File Type"));
            property_dialog_ok_sensitive(NULL,NULL);
        }
    }
}


/* OKボタンが押された */
static void property_dialog_clicked_ok(GtkWidget *widget,FileType *ft)
{
    ft->ft_id=ft_id;
    if (entry0!=NULL) {
        g_free(ft->text);
        ft->text=g_strdup(gtk_entry_get_text(GTK_ENTRY(entry0)));
    }
    if (entry1!=NULL) {
        g_free(ft->ext);
        ft->ext=g_strdup(gtk_entry_get_text(GTK_ENTRY(entry1)));
    }
    ft->margin=gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(spin0));
    ft->tab=gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(spin1));
    ft->gline=gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(check0));
    ft->space=gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(check1));
    ft->nline=gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(check2));
    ft->mline=gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(check3));
    ft->limit=gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(check4));
    ft->code=gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(check5));
    ft->uline=gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(check6));
    ft->vline=gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(check7));
    ft->autoindent=gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(check8));
    ft->tabconv = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (check9));
    ft->eof=gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(check10));
    ft->overwrite=gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(check11));
    ft->syscol=gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(check12));
    if (check13 != NULL)
        ft->html = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (check13));
    if (check14 != NULL)
        ft->esc = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (check14));
    if (check15!=NULL)
        ft->associate=gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(check15));
    if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (radio1)))
        ft->writemode = 1;
    else if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (radio2)))
        ft->writemode = 2;
    else if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (radio3)))
        ft->writemode = 3;
    else if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (radio4)))
        ft->writemode = 4;
    else
        ft->writemode = 0;
    ft->hexcase = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (radio5));
    if (combo0 != NULL)
      {
        const gchar *text;
        GList *glist;

        text = gtk_entry_get_text (GTK_ENTRY (GTK_COMBO (combo0)->entry));
        glist = gtk_container_children
                                    (GTK_CONTAINER (GTK_COMBO (combo0)->list));
        for (ft->crlf = g_list_length (glist) - 1; ft->crlf >= 0; ft->crlf--)
            if (g_strcmp (gtk_label_get_text (GTK_LABEL (gtk_bin_get_child
                (GTK_BIN (g_list_nth_data (glist, ft->crlf))))), text) == 0)
                break;
        g_list_free (glist);
        if (ft->crlf < 0)
            ft->crlf = 0;
      }
    g_free (ft->charset);
    if (combo1 != NULL)
      {
        ft->charset = g_strdup (gtk_entry_get_text
                                    (GTK_ENTRY (GTK_COMBO (combo1)->entry)));
      }
    if (button1 != NULL)
      {
        ft->charset = charset;
        charset = NULL;
      }
    g_memmove(ft->color,gdk_color,sizeof(GdkColor)*12);
    g_free(ft->font_name);
    ft->font_name=font_name;
    font_name=NULL;
    gtk_widget_destroy(dialog);
    property_clicked_ok=TRUE;
}


/* スピンボタン/コンボボックスでリターンが押された */
static void property_dialog_activate(GtkWidget *widget,FileType *ft)
{
    if (GTK_WIDGET_IS_SENSITIVE (button4))
        property_dialog_clicked_ok(widget,ft);
}


/* ESCが押された */
static gboolean property_dialog_key_press(GtkWidget *widget,GdkEventKey *event,
                                                            gpointer user_data)
{
    if (event->keyval==GDK_Escape)
        gtk_widget_destroy(widget);
    return FALSE;
}


gboolean property_dialog(FileType *ft)
{
    gchar **name, *xpm[18];
    gint i, j;
    GList *glist;
    GtkCellRenderer *renderer;
    GtkTreeIter iter;
    GtkTreeSelection *select;
    GtkTreeViewColumn *column;
    GtkWidget *button2, *button3, *button5, *scroll, *frame_label;
    GtkWidget *hbox,*tbox,*vbox;
    GtkWidget *frame0, *frame1, *frame2, *frame3, *frame4, *frame5, *frame6;
    gchar *color_name[12] = {
            N_("Foreground"), N_("Background"), N_("Control Code"),
            N_("Full Width Space"), N_("Return Code"), N_("Tab Under Line"),
            N_("Right Margin"), N_("Grid"), N_("Selected Foreground"),
            N_("Selected Background"), N_("Foreground(Print)"),
            N_("Background(Print)")};

    /* メインウインドウ */
    dialog=gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title(GTK_WINDOW(dialog),ft->text==NULL || ft->ext==NULL
                                    ?_("Property"):_("Details of File Type"));
    g_signal_connect_after (G_OBJECT(dialog), "key-press-event",
                                G_CALLBACK (property_dialog_key_press), NULL);
    g_signal_connect (G_OBJECT (dialog), "destroy",
                                            G_CALLBACK (gtk_main_quit), NULL);
    /* ボタン */
    button0 = gtk_button_new_with_label (_("Palette"));
    button2 = gtk_button_new_with_mnemonic (_("_Font"));
    button4=gtk_button_new_with_label(_("OK"));
    button5=gtk_button_new_with_label(_("Cancel"));
    g_signal_connect (G_OBJECT (button0), "clicked",
                            G_CALLBACK (property_dialog_clicked_color), NULL);
    g_signal_connect (G_OBJECT (button2), "clicked",
                            G_CALLBACK (property_dialog_clicked_font), NULL);
    g_signal_connect (G_OBJECT (button4), "clicked",
                                G_CALLBACK (property_dialog_clicked_ok), ft);
    g_signal_connect_swapped (G_OBJECT (button5), "clicked",
                        G_CALLBACK (gtk_widget_destroy), G_OBJECT (dialog));
    GTK_WIDGET_SET_FLAGS(button0,GTK_CAN_DEFAULT);
    GTK_WIDGET_SET_FLAGS(button2,GTK_CAN_DEFAULT);
    GTK_WIDGET_SET_FLAGS(button4,GTK_CAN_DEFAULT);
    GTK_WIDGET_SET_FLAGS(button5,GTK_CAN_DEFAULT);
    if (ft->text != NULL && ft->ext != NULL)
      {
        button1 = gtk_button_new_with_label (_("Auto Negotiation"));
        g_signal_connect (G_OBJECT (button1), "clicked",
                        G_CALLBACK (property_dialog_clicked_charset), NULL);
        GTK_WIDGET_SET_FLAGS (button1, GTK_CAN_DEFAULT);
      }
    else
      {
        button1 = NULL;
      }
    if (ft->text == NULL && ft->ext == NULL)
      {
        button3 = gtk_button_new_with_mnemonic (_("_Associated File Type"));
        g_signal_connect (G_OBJECT (button3), "clicked",
                            G_CALLBACK (property_dialog_clicked_ftype), NULL);
        GTK_WIDGET_SET_FLAGS (button3, GTK_CAN_DEFAULT);
      }
    else
      {
        button3 = NULL;
      }
    /* チェックボタン */
    check0 = gtk_check_button_new_with_label (_("Grid"));
    check1 = gtk_check_button_new_with_label (_("Full Width Space"));
    check2 = gtk_check_button_new_with_label (_("Return Code"));
    check3 = gtk_check_button_new_with_label (_("Show Right Margin"));
    check4 = gtk_check_button_new_with_label (_("Turn"));
    check5 = gtk_check_button_new_with_label (_("Control Mark"));
    check6 = gtk_check_button_new_with_label (_("Under Line"));
    check7 = gtk_check_button_new_with_label (_("Vertical Line"));
    check8 = gtk_check_button_new_with_label (_("Auto Indent"));
    check9 = gtk_check_button_new_with_label (_("Convert Tab to Space"));
    check10 = gtk_check_button_new_with_label (_("Append ^Z at EOF"));
    check11 = gtk_check_button_new_with_label (_("Over Write with Agreement"));
    check12 = gtk_check_button_new_with_label (_("Use System Color"));
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check0),ft->gline);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check1),ft->space);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check2),ft->nline);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check3),ft->mline);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check4),ft->limit);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check5),ft->code);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check6),ft->uline);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check7),ft->vline);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check8),ft->autoindent);
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (check9), ft->tabconv);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check10),ft->eof);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check11),ft->overwrite);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check12),ft->syscol);
    g_signal_connect (G_OBJECT (check0), "toggled",
                            G_CALLBACK (property_dialog_ok_sensitive), NULL);
    g_signal_connect (G_OBJECT (check1), "toggled",
                            G_CALLBACK (property_dialog_ok_sensitive), NULL);
    g_signal_connect (G_OBJECT (check2), "toggled",
                            G_CALLBACK (property_dialog_ok_sensitive), NULL);
    g_signal_connect (G_OBJECT (check3), "toggled",
                            G_CALLBACK (property_dialog_ok_sensitive), NULL);
    g_signal_connect (G_OBJECT (check4), "toggled",
                            G_CALLBACK (property_dialog_ok_sensitive), NULL);
    g_signal_connect (G_OBJECT (check5), "toggled",
                            G_CALLBACK (property_dialog_ok_sensitive), NULL);
    g_signal_connect (G_OBJECT (check6), "toggled",
                            G_CALLBACK (property_dialog_ok_sensitive), NULL);
    g_signal_connect (G_OBJECT (check7), "toggled",
                            G_CALLBACK (property_dialog_ok_sensitive), NULL);
    g_signal_connect (G_OBJECT (check8), "toggled",
                            G_CALLBACK (property_dialog_ok_sensitive), NULL);
    g_signal_connect (G_OBJECT (check9), "toggled",
                            G_CALLBACK (property_dialog_ok_sensitive), NULL);
    g_signal_connect (G_OBJECT (check10), "toggled",
                            G_CALLBACK (property_dialog_ok_sensitive), NULL);
    g_signal_connect (G_OBJECT (check11), "toggled",
                            G_CALLBACK (property_dialog_ok_sensitive), NULL);
    g_signal_connect (G_OBJECT (check12), "toggled",
                            G_CALLBACK (property_dialog_toggled_check), NULL);
    if (ft->text != NULL && ft->ext != NULL)
      {
        check13 = gtk_check_button_new_with_label
                                    (_("HTML Numeric character reference"));
        check14 = gtk_check_button_new_with_label (_("Escape sequence"));
        check15 = gtk_check_button_new_with_mnemonic
                                                (_("_Associate File Type"));
        gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (check13), ft->html);
        gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (check14), ft->esc);
        gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (check15),
                                                                ft->associate);
        g_signal_connect (G_OBJECT (check13), "toggled",
                            G_CALLBACK (property_dialog_ok_sensitive), NULL);
        g_signal_connect (G_OBJECT (check14), "toggled",
                            G_CALLBACK (property_dialog_ok_sensitive), NULL);
        g_signal_connect (G_OBJECT (check15), "toggled",
                            G_CALLBACK (property_dialog_ok_sensitive), NULL);
    } else {
        check13 = check14 = check15 = NULL;
    }
    /* コンボボックス */
    glist = g_list_append (NULL, "CR+LF");
    glist = g_list_append (glist, "CR");
    glist = g_list_append (glist, "LF");
    if (ft->text != NULL && ft->ext != NULL)
        glist = g_list_append (glist, _("Auto"));
    combo0 = gtk_combo_new();
    gtk_combo_set_popdown_strings (GTK_COMBO (combo0), glist);
    gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (combo0)->entry),
                                            g_list_nth_data (glist, ft->crlf));
    gtk_editable_set_editable
                            (GTK_EDITABLE (GTK_COMBO (combo0)->entry), FALSE);
    g_signal_connect (G_OBJECT (GTK_COMBO (combo0)->entry), "activate",
                                    G_CALLBACK (property_dialog_activate), ft);
    g_signal_connect (G_OBJECT (GTK_COMBO (combo0)->entry),"changed",
                            G_CALLBACK (property_dialog_ok_sensitive), NULL);
    if (ft->text == NULL && ft->ext == NULL)
      {
        glist = NULL;
        for (i = 0; i < ftnum; i++)
          {
            name = g_strsplit (ftype[i].charset, ",", G_MAXINT);
            for (j = 0; name[j] != NULL; j++)
                if (g_list_find_custom (glist, name[j],
                                        (GCompareFunc)g_strcasecmp) == NULL)
                    glist = g_list_insert_sorted (glist, g_strdup (name[j]),
                                                (GCompareFunc)g_strcasecmp);
            g_strfreev (name);
          }
        combo1 = gtk_combo_new();
        gtk_combo_set_popdown_strings (GTK_COMBO (combo1), glist);
        gtk_combo_set_value_in_list (GTK_COMBO (combo1), FALSE, FALSE);
        gtk_combo_set_case_sensitive (GTK_COMBO (combo1), TRUE);
        gtk_entry_set_text
                        (GTK_ENTRY (GTK_COMBO (combo1)->entry), ft->charset);
        gtk_editable_select_region
                            (GTK_EDITABLE (GTK_COMBO (combo1)->entry), 0, -1);
        g_signal_connect (G_OBJECT (GTK_COMBO (combo1)->entry), "activate",
                                    G_CALLBACK (property_dialog_activate), ft);
        g_signal_connect (G_OBJECT (GTK_COMBO (combo1)->entry), "changed",
                            G_CALLBACK (property_dialog_ok_sensitive), NULL);
        for (i = g_list_length (glist) - 1; i >= 0; i--)
            g_free (g_list_nth_data (glist, i));
      }
    else
      {
        combo1 = NULL;
      }
    /* エントリー/ラベル */
    label=NULL;
    if (ft->text!=NULL && ft->ext!=NULL) {
        entry0=gtk_entry_new_with_max_length(256);
        entry1=gtk_entry_new_with_max_length(256);
        gtk_entry_set_text(GTK_ENTRY(entry0),ft->text);
        gtk_entry_set_text(GTK_ENTRY(entry1),ft->ext);
        g_signal_connect (G_OBJECT (entry0), "changed",
                            G_CALLBACK (property_dialog_ok_sensitive), NULL);
        g_signal_connect (G_OBJECT (entry1), "changed",
                            G_CALLBACK (property_dialog_ok_sensitive), NULL);
        g_signal_connect (G_OBJECT (entry0), "activate",
                                    G_CALLBACK (property_dialog_activate), ft);
        g_signal_connect (G_OBJECT (entry1), "activate",
                                    G_CALLBACK (property_dialog_activate), ft);
    } else {
        if (ft->ft_id>=0) {
            /* ファイルタイプあり */
            for (i=0;i<ftnum;i++)
                if (ft->ft_id==ftype[i].ft_id) {
                    label=gtk_label_new(ftype[i].text);
                    break;
                }
            if (label==NULL)
                label=gtk_label_new(_("Unknow File Type"));
        } else {
            /* ファイルタイプなし */
            label=gtk_label_new(_("Independent File Type"));
        }
        gtk_misc_set_alignment(GTK_MISC(label),0,0.5);
        entry0=entry1=NULL;
    }
    /* ラジオボタン */
    radio0 = gtk_radio_button_new_with_label (NULL, _("Abort"));
    radio1 = gtk_radio_button_new_with_label_from_widget
                            (GTK_RADIO_BUTTON (radio0), _("Question Mark"));
    radio2 = gtk_radio_button_new_with_label_from_widget
                            (GTK_RADIO_BUTTON (radio1),
                            _("HTML Numeric character reference (Decimal)"));
    radio3 = gtk_radio_button_new_with_label_from_widget
                        (GTK_RADIO_BUTTON (radio2),
                        _("HTML Numeric character reference (Hexadecimal)"));
    radio4 = gtk_radio_button_new_with_label_from_widget
                            (GTK_RADIO_BUTTON (radio3), _("Escape sequence"));
    radio5 = gtk_radio_button_new_with_label (NULL, _("Upper case"));
    radio6 = gtk_radio_button_new_with_label_from_widget
                                (GTK_RADIO_BUTTON (radio5), _("Lower case"));
    switch (ft->writemode)
      {
        case  1:gtk_toggle_button_set_active
                                    (GTK_TOGGLE_BUTTON (radio1), TRUE); break;
        case  2:gtk_toggle_button_set_active
                                    (GTK_TOGGLE_BUTTON (radio2), TRUE); break;
        case  3:gtk_toggle_button_set_active
                                    (GTK_TOGGLE_BUTTON (radio3), TRUE); break;
        case  4:gtk_toggle_button_set_active
                                    (GTK_TOGGLE_BUTTON (radio4), TRUE); break;
        default:gtk_toggle_button_set_active
                                    (GTK_TOGGLE_BUTTON (radio0), TRUE);
      }
    gtk_toggle_button_set_active
                    (GTK_TOGGLE_BUTTON (ft->hexcase ? radio5 : radio6), TRUE);
    g_signal_connect (G_OBJECT (radio0), "toggled",
                            G_CALLBACK (property_dialog_toggled_radio), NULL);
    g_signal_connect (G_OBJECT (radio1), "toggled",
                            G_CALLBACK (property_dialog_toggled_radio), NULL);
    g_signal_connect (G_OBJECT (radio2), "toggled",
                            G_CALLBACK (property_dialog_toggled_radio), NULL);
    g_signal_connect (G_OBJECT (radio3), "toggled",
                            G_CALLBACK (property_dialog_toggled_radio), NULL);
    g_signal_connect (G_OBJECT (radio4), "toggled",
                            G_CALLBACK (property_dialog_toggled_radio), NULL);
    g_signal_connect (G_OBJECT (radio5), "toggled",
                            G_CALLBACK (property_dialog_ok_sensitive), NULL);
    g_signal_connect (G_OBJECT (radio6), "toggled",
                            G_CALLBACK (property_dialog_ok_sensitive), NULL);
    /* リストボックス */
    store = gtk_list_store_new (2, GDK_TYPE_PIXBUF, G_TYPE_STRING);
    tview = gtk_tree_view_new_with_model (GTK_TREE_MODEL (store));
    gtk_tree_view_set_headers_visible (GTK_TREE_VIEW (tview), FALSE);
    gtk_tree_view_set_enable_search (GTK_TREE_VIEW (tview), FALSE);

    renderer = gtk_cell_renderer_pixbuf_new ();
    column = gtk_tree_view_column_new_with_attributes
                                        (NULL, renderer, "pixbuf", 0, NULL);
    gtk_tree_view_append_column (GTK_TREE_VIEW (tview), column);

    renderer = gtk_cell_renderer_text_new ();
    column = gtk_tree_view_column_new_with_attributes
                                        (NULL, renderer, "text", 1, NULL);
    gtk_tree_view_append_column (GTK_TREE_VIEW (tview), column);

    select = gtk_tree_view_get_selection (GTK_TREE_VIEW (tview));
    gtk_tree_selection_set_mode (select, GTK_SELECTION_MULTIPLE);
    g_signal_connect (G_OBJECT (select), "changed",
                                G_CALLBACK (property_dialog_changed), NULL);
    xpm[0]="24 16 1 1";
    for (i=2;i<18;i++)
        xpm[i]="XXXXXXXXXXXXXXXXXXXXXXXX";
    for (i = 0; i < 12; i++)
      {
        xpm[1] = g_strdup_printf ("X\tc #%02X%02X%02X",
                                                    ft->color[i].red / 256,
                                                    ft->color[i].green / 256,
                                                    ft->color[i].blue / 256);
        gtk_list_store_append (store, &iter);
        gtk_list_store_set (store, &iter,
                        0, gdk_pixbuf_new_from_xpm_data ((const char **)xpm),
                        1, _(color_name[i]), -1);
        g_free (xpm[1]);
      }
    /* スクロールウインドウ */
    scroll=gtk_scrolled_window_new(NULL,NULL);
    gtk_container_add (GTK_CONTAINER (scroll), tview);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scroll),
                                        GTK_POLICY_NEVER,GTK_POLICY_AUTOMATIC);
    /* スピンボタン */
    spin0=gtk_spin_button_new(
        GTK_ADJUSTMENT(gtk_adjustment_new(ft->margin,1,G_MAXINT,1,5,0)),0,0);
    spin1=gtk_spin_button_new(
        GTK_ADJUSTMENT(gtk_adjustment_new(ft->tab,1,G_MAXINT,1,5,0)),0,0);
    gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(spin0),TRUE);
    gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(spin1),TRUE);
    g_signal_connect (G_OBJECT (spin0), "activate",
                                    G_CALLBACK (property_dialog_activate), ft);
    g_signal_connect (G_OBJECT (spin1), "activate",
                                    G_CALLBACK (property_dialog_activate), ft);
    g_signal_connect (G_OBJECT (spin0), "changed",
                            G_CALLBACK (property_dialog_ok_sensitive), NULL);
    g_signal_connect (G_OBJECT (spin1), "changed",
                            G_CALLBACK (property_dialog_ok_sensitive), NULL);
    /* フレームとボックス */
    frame0 = gtk_frame_new ("");
    frame_label = gtk_frame_get_label_widget (GTK_FRAME (frame0));
    gtk_label_set_text_with_mnemonic (GTK_LABEL (frame_label), _("_Display"));
    gtk_label_set_mnemonic_widget (GTK_LABEL (frame_label), check0);
    hbox = gtk_hbox_new (FALSE, SPACING);
    gtk_container_set_border_width (GTK_CONTAINER (hbox), SPACING);
    gtk_box_pack_start (GTK_BOX (hbox), check0, TRUE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (hbox), check1, TRUE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (hbox), check2, TRUE, TRUE, 0);
    gtk_container_add(GTK_CONTAINER(frame0),hbox);

    frame1 = gtk_frame_new ("");
    frame_label = gtk_frame_get_label_widget (GTK_FRAME (frame1));
    gtk_label_set_text_with_mnemonic (GTK_LABEL (frame_label),
                                      _("_Right Margin"));
    gtk_label_set_mnemonic_widget (GTK_LABEL (frame_label), spin0);
    hbox = gtk_hbox_new (FALSE, SPACING);
    gtk_container_set_border_width (GTK_CONTAINER (hbox), SPACING);
    gtk_box_pack_start(GTK_BOX(hbox),spin0,TRUE,TRUE,0);
    gtk_box_pack_start(GTK_BOX(hbox),check3,FALSE,FALSE,0);
    gtk_box_pack_start(GTK_BOX(hbox),check4,FALSE,FALSE,0);
    gtk_container_add(GTK_CONTAINER(frame1),hbox);

    frame2 = gtk_frame_new ("");
    frame_label = gtk_frame_get_label_widget (GTK_FRAME (frame2));
    gtk_label_set_text_with_mnemonic (GTK_LABEL (frame_label), _("_Tab"));
    gtk_label_set_mnemonic_widget (GTK_LABEL (frame_label), spin1);
    hbox = gtk_hbox_new (FALSE, SPACING);
    gtk_container_set_border_width (GTK_CONTAINER (hbox), SPACING);
    gtk_box_pack_start(GTK_BOX(hbox),spin1,TRUE,TRUE,0);
    gtk_box_pack_start(GTK_BOX(hbox),check5,FALSE,FALSE,0);
    gtk_box_pack_start(GTK_BOX(hbox),check6,FALSE,FALSE,0);
    gtk_box_pack_start(GTK_BOX(hbox),check7,FALSE,FALSE,0);
    gtk_container_add(GTK_CONTAINER(frame2),hbox);

    frame3=gtk_frame_new("");
    frame_label = gtk_frame_get_label_widget (GTK_FRAME (frame3));
    gtk_label_set_text_with_mnemonic (GTK_LABEL (frame_label), _("Ed_it"));
    gtk_label_set_mnemonic_widget (GTK_LABEL (frame_label), check8);
    hbox = gtk_hbox_new (FALSE, SPACING);
    gtk_container_set_border_width (GTK_CONTAINER (hbox), SPACING);
    gtk_box_pack_start (GTK_BOX(hbox), check8, FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (hbox), check9, FALSE, FALSE, 0);
    gtk_container_add (GTK_CONTAINER (frame3), hbox);

    frame4 = gtk_frame_new ("");
    frame_label = gtk_frame_get_label_widget (GTK_FRAME (frame4));
    gtk_label_set_text_with_mnemonic (GTK_LABEL (frame_label), _("_Save"));
    gtk_label_set_mnemonic_widget (GTK_LABEL (frame_label), check10);
    tbox = gtk_vbox_new (FALSE, SPACING);
    gtk_container_set_border_width (GTK_CONTAINER (tbox), SPACING);
    hbox = gtk_hbox_new (FALSE, SPACING);
    gtk_box_pack_start (GTK_BOX (hbox), check10, TRUE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (hbox), check11, TRUE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (tbox), hbox, FALSE, FALSE, 0);
    hbox = gtk_hbox_new (FALSE, SPACING);
    vbox = gtk_vbox_new (FALSE, SPACING);
    gtk_box_pack_start (GTK_BOX (vbox), radio0, FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (vbox), radio1, FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (vbox), radio2, FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (vbox), radio3, FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (vbox), radio4, FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (hbox), vbox, TRUE, TRUE, 0);
    vbox = gtk_vbox_new (FALSE, SPACING);
    gtk_box_pack_start (GTK_BOX (vbox), radio5, FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (vbox), radio6, FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (hbox), vbox, TRUE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (tbox), hbox, FALSE, FALSE, 0);
    gtk_container_add (GTK_CONTAINER (frame4), tbox);

    frame5 = gtk_frame_new ("");
    frame_label = gtk_frame_get_label_widget (GTK_FRAME (frame5));
    gtk_label_set_text_with_mnemonic (GTK_LABEL (frame_label), _("_Color"));
    gtk_label_set_mnemonic_widget (GTK_LABEL (frame_label), check12);
    vbox = gtk_vbox_new (FALSE, SPACING);
    gtk_container_set_border_width (GTK_CONTAINER (vbox), SPACING);
    gtk_box_pack_start(GTK_BOX(vbox),scroll,TRUE,TRUE,0);
    gtk_box_pack_start(GTK_BOX(vbox),check12,FALSE,FALSE,0);
    hbox=gtk_hbox_new(FALSE,0);
    gtk_box_pack_end(GTK_BOX(hbox),button0,FALSE,FALSE,0);
    gtk_box_pack_start(GTK_BOX(vbox),hbox,FALSE,FALSE,0);
    gtk_container_add (GTK_CONTAINER (frame5), vbox);

    frame6 = gtk_frame_new ("");
    frame_label = gtk_frame_get_label_widget (GTK_FRAME (frame6));
    gtk_label_set_text_with_mnemonic (GTK_LABEL (frame_label),
                                                    _("Character _Encoding"));
    gtk_label_set_mnemonic_widget (GTK_LABEL (frame_label),
                                   GTK_COMBO (combo0)->entry);
    vbox = gtk_vbox_new (FALSE, SPACING);
    gtk_container_set_border_width (GTK_CONTAINER (vbox), SPACING);
    gtk_box_pack_start (GTK_BOX (vbox), combo0, FALSE, FALSE, 0);
    if (button1 != NULL)
      {
        hbox = gtk_hbox_new (FALSE, 0);
        gtk_box_pack_end (GTK_BOX (hbox), button1, FALSE, FALSE, 0);
        gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
      }
    if (combo1 != NULL)
        gtk_box_pack_start (GTK_BOX (vbox), combo1, FALSE, FALSE, 0);
    if (check13 != NULL)
        gtk_box_pack_start (GTK_BOX (vbox), check13, FALSE, FALSE, 0);
    if (check14 != NULL)
        gtk_box_pack_start (GTK_BOX (vbox), check14, FALSE, FALSE, 0);
    gtk_container_add (GTK_CONTAINER (frame6), vbox);

    tbox = gtk_hbox_new (FALSE, SPACING);
    if (entry0 != NULL && entry1 != NULL)
      {
        hbox = gtk_hbox_new (FALSE, 0);
        gtk_box_pack_start (GTK_BOX (hbox),
                                gtk_label_new (_("Title")), FALSE, FALSE, 0);
        gtk_box_pack_start (GTK_BOX (hbox), entry0, TRUE, TRUE, 0);
        gtk_box_pack_start (GTK_BOX (tbox), hbox, TRUE, TRUE, 0);
        hbox = gtk_hbox_new (FALSE, 0);
        gtk_box_pack_start (GTK_BOX (hbox),
                            gtk_label_new (_("Extension")), FALSE, FALSE, 0);
        gtk_box_pack_start (GTK_BOX (hbox), entry1, TRUE, TRUE, 0);
        gtk_box_pack_start (GTK_BOX (tbox), hbox, TRUE, TRUE, 0);
      }
    else
      {
        gtk_box_pack_start (GTK_BOX (tbox), label, TRUE, TRUE, 0);
      }

    hbox = gtk_hbox_new (FALSE, SPACING);
    vbox = gtk_vbox_new (FALSE, SPACING);
    gtk_box_pack_start(GTK_BOX(vbox),frame0,FALSE,FALSE,0);
    gtk_box_pack_start(GTK_BOX(vbox),frame1,FALSE,FALSE,0);
    gtk_box_pack_start(GTK_BOX(vbox),frame2,FALSE,FALSE,0);
    gtk_box_pack_start(GTK_BOX(vbox),frame3,FALSE,FALSE,0);
    gtk_box_pack_start (GTK_BOX (vbox), frame4, FALSE, FALSE, 0);
    gtk_box_pack_start(GTK_BOX(hbox),vbox,TRUE,TRUE,0);
    vbox = gtk_vbox_new (FALSE, SPACING);
    gtk_box_pack_start (GTK_BOX (vbox), frame5, TRUE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (vbox), frame6, FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (hbox), vbox, TRUE, TRUE, 0);

    vbox = gtk_vbox_new (FALSE, SPACING);
    gtk_container_set_border_width (GTK_CONTAINER (vbox), SPACING);
    gtk_box_pack_start(GTK_BOX(vbox),tbox,FALSE,FALSE,0);
    gtk_box_pack_start(GTK_BOX(vbox),hbox,TRUE,TRUE,0);
    hbox = gtk_hbox_new (FALSE, SPACING);
    gtk_box_pack_start(GTK_BOX(hbox),button2,FALSE,FALSE,0);
    if (button3!=NULL)
        gtk_box_pack_start(GTK_BOX(hbox),button3,FALSE,FALSE,0);
    if (check15!=NULL)
        gtk_box_pack_start(GTK_BOX(hbox),check15,FALSE,FALSE,0);
    gtk_box_pack_end(GTK_BOX(hbox),button5,FALSE,FALSE,0);
    gtk_box_pack_end(GTK_BOX(hbox),button4,FALSE,FALSE,0);
    gtk_box_pack_end(GTK_BOX(vbox),hbox,FALSE,FALSE,0);

    gtk_container_add(GTK_CONTAINER(dialog),vbox);

    /* 表示 */
    gtk_widget_set_sensitive(button0,FALSE);
    gtk_widget_set_sensitive(button4,FALSE);
    gtk_widget_set_sensitive (radio5, ft->writemode >= 3);
    gtk_widget_set_sensitive (radio6, ft->writemode >= 3);
    gtk_widget_set_sensitive (tview, !ft->syscol);
    gtk_widget_grab_focus(button5);

    ft_id=ft->ft_id;
    g_memmove(gdk_color,ft->color,sizeof(GdkColor)*12);
    charset = button1 != NULL ? g_strdup (ft->charset) : NULL;
    font_name=g_strdup(ft->font_name);
    property_clicked_ok=FALSE;

    gtk_window_set_policy(GTK_WINDOW(dialog),FALSE,TRUE,FALSE);
    gtk_grab_add(dialog);
    gtk_widget_show_all(dialog);
    gtk_main();

    g_free (charset);
    g_free(font_name);

    return property_clicked_ok;
}
