/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 *  Copyright (C) 2000 - 2004 Hiroyuki Komatsu <komatsu@taiyaki.org>
 *  Copyright (C) 2004 Takuro Ashie <ashie@homa.ne.jp>
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this program; if not, write to the
 *  Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 *  Boston, MA  02111-1307  USA
 *
 *  $Id: uim-tomoe-gtk.c,v 1.5 2005/02/09 03:14:28 makeinu Exp $
 */

#ifdef HAVE_CONFIG_H
#include <config.h> 
#endif /* HAVE_CONFIG_H */
#include <gtk/gtk.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <uim/uim.h>
#include <uim/uim-helper.h>
#include "uim-tomoe-gtk.h"
#include "intl.h"

/*  for uim */
static void helper_disconnect_cb    (void);
static void check_helper_connection (void);

/* candidates table */
static void create_table (void);

static GtkWidget *draw_window;
static GtkWidget *drawing_area;
static GtkWidget *clear_button;

static GtkWidget *cand_window = NULL;
static GtkWidget *cand_entry = NULL;
static GList *cand_list = NULL;

static int uim_fd = -1;

static GPid pid;
static gint in_fd, out_fd, err_fd;

/* ΰΤΥХå󥰥ԥåޥå */
static GdkPixmap *pixmap  = NULL;
static gint prev_x = -1;
static gint prev_y = -1;

static GList *stroke      = NULL;
static GList *stroke_list = NULL;


/* ŬڤʥοХå󥰥ԥåޥåפ */
static gint
configure_event (GtkWidget *widget, GdkEventConfigure *event)
{
    if (pixmap)
        gdk_pixmap_unref(pixmap);

    pixmap = gdk_pixmap_new(widget->window,
                            widget->allocation.width,
                            widget->allocation.height,
                            -1);
    gdk_draw_rectangle (pixmap,
                        widget->style->white_gc,
                        TRUE,
                        0, 0,
                        widget->allocation.width,
                        widget->allocation.height);

    return TRUE;
}

/* Хå󥰥ԥåޥåפ饹꡼ */
static gint
expose_event (GtkWidget *widget, GdkEventExpose *event)
{
    gdk_draw_pixmap(widget->window,
                    widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
                    pixmap,
                    event->area.x, event->area.y,
                    event->area.x, event->area.y,
                    event->area.width, event->area.height);

    return FALSE;
}

static void
draw_brush (GtkWidget *widget, gint x, gint y)
{
    GdkRectangle update_rect;

    point *p = g_malloc(sizeof (point));
    p->x = x;
    p->y = y;
    stroke = g_list_append(stroke, p);

    if (prev_x == -1)
    {
        prev_x  = x;
        prev_y  = y;
    }
  
    update_rect.x = MIN(x, prev_x);
    update_rect.y = MIN(y, prev_y);
    update_rect.width  = abs(x - prev_x) + 1;
    update_rect.height = abs(y - prev_y) + 1;

    gdk_draw_line(pixmap, widget->style->black_gc, x, y, prev_x, prev_y);
    gtk_widget_draw(widget, &update_rect);

    prev_x = x;
    prev_y = y;
}

static gint
button_press_event (GtkWidget *widget, GdkEventButton *event)
{
    if (event->button == 1 && pixmap != NULL)
    {
        draw_brush(widget, event->x, event->y);
    }
    return TRUE;
}

static gint
button_release_event (GtkWidget *widget, GdkEventButton *event)
{
    stroke_list = g_list_append(stroke_list, stroke);
    stroke = NULL;
    prev_x = prev_y = -1;
    return TRUE;
}

static gint
motion_notify_event (GtkWidget *widget, GdkEventMotion *event)
{
    int x, y;
    GdkModifierType state;

    if (event->is_hint)
    {
        gdk_window_get_pointer(event->window, &x, &y, &state);
    }
    else
    {
        x = event->x;
        y = event->y;
        state = event->state;
    }
    
    if (state & GDK_BUTTON1_MASK && pixmap != NULL)
    {
        draw_brush (widget, x, y);
    }
    return TRUE;
}

static gint
get_distance (GList *first_node, GList *last_node, GList **most_node)
{
    /* Getting distance 
     * MAX( |aw - bv + c| )
     * a = x-p : b = y-q : c = py - qx
     * first = (p, q) : last = (x, y) : other = (v, w)
     */

    GList *dot;
    gint a, b, c;
    gint dist = 0;
    gint max  = 0;
    point *first = first_node->data;
    point *last  = last_node->data;
    point *p;

    *most_node = NULL;
    if (first_node == last_node)
    {
        return 0;
    }

    a = last->x - first->x;
    b = last->y - first->y;
    c = last->y * first->x - last->x * first->y;

    for (dot = first_node; dot != last_node; dot = dot->next)
    {
        p = dot->data;
        dist = abs((a * p->y) - (b * p->x) + c);
        if (dist > max)
        {
            max = dist;
            *most_node = dot;
        }
    }
    return max * max / (a*a + b*b);
}

static GList*
get_vertex (GList *first_node, GList *last_node)
{
    GList *rv = NULL;
    GList *most_node;
    gint dist;
    gint error = 300 * 300 / 400; /* 5% */

    dist = get_distance(first_node, last_node, &most_node);

    if (dist > error)
    {
        rv = g_list_concat(get_vertex(first_node, most_node),
                           get_vertex(most_node, last_node));
    }
    else
    {
        rv = g_list_append(rv, last_node->data);
    }
    return rv;
}

static gboolean
io_dialog (GIOChannel * channel, GIOCondition condition, gpointer data)
{
    if (condition & G_IO_IN)
    {
        GIOStatus status;
        gchar buf[1024], *tmp, **chars;
        gsize bytes_read, bytes_written, end;

        status = g_io_channel_read (channel, buf, 1023,
                                    &bytes_read);

        end = bytes_read > 1023 ? 1023 : bytes_read;
        buf[end] = '\0';

        /* FIXME! check return value */
        tmp = g_convert(buf, strlen(buf),
                        "UTF-8", "EUC-JP",
                        &bytes_read, &bytes_written,
                        NULL);

        if (tmp)
        {
            gint i;

            chars = g_strsplit(tmp, " ", -1);

            for (i = 0; chars[i]; i++)
            {
                g_strstrip(chars[i]);
                if (*chars[i])
                    cand_list = g_list_append(cand_list, g_strdup(chars[i]));
            }
            g_strfreev(chars);

            g_free(tmp);
        }

        if (status == G_IO_STATUS_EOF)
        {
            goto END;
        }

        return TRUE;
    }

    if (condition & G_IO_ERR)
    {
        goto END;
    }

    if (condition & G_IO_HUP)
    {
        goto END;
    }

    if (condition & G_IO_NVAL)
    {
        goto END;
    }

    return TRUE;

END:
    close(in_fd);
    close(out_fd);
    close(err_fd);
    g_spawn_close_pid(pid);
    pid = 0;

    create_table();

    return FALSE;
}

static gint
create_tomoe_process (void)
{
    gchar *tomoe_argv[2];
    GError *error = NULL;
    gboolean success;

    /* run tomoe */
    tomoe_argv[0] = "tomoe";
    tomoe_argv[1] = NULL;
    success = g_spawn_async_with_pipes(NULL,
                                       tomoe_argv,
                                       NULL,
                                       G_SPAWN_SEARCH_PATH,
                                       NULL,
                                       NULL,
                                       &pid,
                                       &in_fd,
                                       &out_fd,
                                       &err_fd,
                                       &error);
    if (success)
    {
        GIOChannel *channel;
        channel = g_io_channel_unix_new (out_fd);
        g_io_add_watch (channel, G_IO_IN | G_IO_HUP | G_IO_ERR |
                        G_IO_NVAL, io_dialog, NULL);
        g_io_channel_unref (channel);
    }

    if (error)
    {
        g_printerr("%s\n", error->message);
        g_error_free(error);
    }

    return in_fd;
}

static void
clear (GtkWidget *widget)
{
    GdkRectangle update_rect;
    GList *line, *dots, *dot;
    gint x, y, px, py;
    GString *string;
    gchar text[1024];

    string = g_string_new("");

    create_tomoe_process();

    update_rect.x = 0;
    update_rect.y = 0;
    update_rect.width  = widget->allocation.width;
    update_rect.height = widget->allocation.height;

    gdk_draw_rectangle(pixmap, widget->style->white_gc, TRUE, 0, 0,
                       widget->allocation.width,
                       widget->allocation.height);

    if(g_list_length(stroke_list) == 0) {
        stroke_list = NULL;
        gtk_widget_draw (widget, &update_rect);
        return;
    }

    g_snprintf(text, sizeof(text), ":%d\n", g_list_length(stroke_list));
    g_string_append(string, text);
    for (line = stroke_list; line; line = line->next)
    {
        dots = g_list_prepend(get_vertex(line->data, g_list_last(line->data)),
                              ((GList *)(line->data))->data);
        px = py = -1;
        g_snprintf(text, sizeof(text), "%d ", g_list_length(dots));
        g_string_append(string, text);
        for (dot = dots; dot; dot = dot->next)
        {
            x = ((point *)(dot->data))->x;
            y = ((point *)(dot->data))->y;
            g_snprintf(text, sizeof(text), "(%d %d) ", x, y);
            g_string_append(string, text);
            if (px != -1)
            {
                gdk_draw_line(pixmap, widget->style->black_gc, px, py, x, y);
            }
            px = x;
            py = y;
        }
        g_string_append(string, "\n");
    }
    g_string_append(string, "\n[EOF]\n");
    /* FIXME! check return value */
    write(in_fd, string->str, string->len);
    g_string_free(string, TRUE);

    stroke_list = NULL;
    gtk_widget_draw (widget, &update_rect);

    gtk_widget_set_sensitive(drawing_area, FALSE);
    gtk_widget_set_sensitive(clear_button, FALSE);
}



/*
 *  candidates table
 */
static void
cand_win_response_cb(GtkDialog *dialog, gint action, gpointer data)
{
    switch (action) {
    case GTK_RESPONSE_CLOSE:
        gtk_widget_destroy(GTK_WIDGET(dialog));
        break;
    default:
        break;
    }
}

static void
table_destroy_cb(GtkWidget *widget)
{
    gtk_widget_set_sensitive(drawing_area, TRUE);
    gtk_widget_set_sensitive(clear_button, TRUE);
    cand_window = NULL;
    cand_entry = NULL;
}

static void
char_clicked_cb(GtkButton *button)
{
    const char *str = gtk_label_get_text(GTK_LABEL(GTK_BIN(button)->child));
    GString *tmp;

    gtk_entry_set_text(GTK_ENTRY(cand_entry), str);

    tmp = g_string_new("commit_string\n");
    g_string_append(tmp, str);
    g_string_append(tmp, "\n");

    check_helper_connection();
    uim_helper_send_message(uim_fd, tmp->str);

    g_string_free(tmp, TRUE);
}

static void
create_table (void)
{
    GtkWidget *vbox, *table, *button;
    gint i;
    GList *node;

    if (!cand_list) {
        gtk_widget_set_sensitive(drawing_area, TRUE);
        gtk_widget_set_sensitive(clear_button, TRUE);
        return;
    }

    cand_window = gtk_dialog_new_with_buttons(
        _("Candidates"), GTK_WINDOW(draw_window), GTK_DIALOG_MODAL,
        GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE,
        NULL);
    gtk_window_set_resizable(GTK_WINDOW(cand_window), FALSE);
    g_signal_connect(G_OBJECT(cand_window), "response",
                     G_CALLBACK(cand_win_response_cb), NULL);
    g_signal_connect(G_OBJECT(cand_window), "destroy",
                     G_CALLBACK(table_destroy_cb), NULL);

    vbox = gtk_vbox_new(FALSE, 2);
    gtk_container_set_border_width(GTK_CONTAINER(vbox), 2);
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(cand_window)->vbox),
                       vbox, TRUE, TRUE, 0);
    gtk_widget_show(vbox);

    table = gtk_table_new(10, 10, FALSE);
    gtk_table_set_row_spacings(GTK_TABLE(table), 3);
    gtk_table_set_col_spacings(GTK_TABLE(table), 3);
    gtk_box_pack_start(GTK_BOX(vbox), table, TRUE, TRUE, 2);
    gtk_widget_show(table);

    for (node = cand_list, i = 0;
         node && i < 100;
         node = g_list_next(node), i++)
    {
        GtkWidget *label;
        gchar *str = node->data, *text;

        if (!*str) continue;
        text  = g_markup_printf_escaped("<span size=\"x-large\">%s</span>",
                                        str);
        label = gtk_label_new("");
        gtk_label_set_markup(GTK_LABEL(label), text);
        g_free(text);
        button = gtk_button_new();
        gtk_container_add(GTK_CONTAINER(button), label);
        g_signal_connect(G_OBJECT(button), "clicked",
                         G_CALLBACK(char_clicked_cb), NULL);
        gtk_table_attach_defaults(GTK_TABLE(table), button,
                                  i % 10, i % 10 + 1,
                                  i / 10, i / 10 + 1);
        gtk_widget_show(button);
        gtk_widget_show(label);
    }

    cand_entry = gtk_entry_new();
    gtk_entry_set_editable(GTK_ENTRY(cand_entry), FALSE);
    gtk_box_pack_start(GTK_BOX(vbox), cand_entry, FALSE, FALSE, 2);
    gtk_widget_show(cand_entry);

    gtk_window_set_position(GTK_WINDOW(cand_window),
                            GTK_WIN_POS_CENTER_ON_PARENT);
    gtk_widget_show(cand_window);

    g_list_foreach(cand_list, (GFunc) g_free, NULL);
    g_list_free(cand_list);
    cand_list = NULL;
}


/*
 *  for uim
 */
static void
helper_disconnect_cb(void)
{
    uim_fd = -1;
}

static void
check_helper_connection(void)
{
    if (uim_fd < 0)
    {
        uim_fd = uim_helper_init_client_fd(helper_disconnect_cb);
        if (uim_fd < 0)
            return;
    }
}


/*
 *  main
 */
static void
quit (void)
{
    gtk_exit (0);
}

int
main (int argc, char *argv[])
{
    GtkWidget *vbox;

    setlocale(LC_ALL, "");
    bindtextdomain(PACKAGE, LOCALEDIR);
    textdomain(PACKAGE);
    bind_textdomain_codeset(PACKAGE, "UTF-8");

    gtk_init (&argc, &argv);

    draw_window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
    gtk_window_set_position(GTK_WINDOW(draw_window), GTK_WIN_POS_CENTER);

    vbox = gtk_vbox_new (FALSE, 0);
    gtk_container_add (GTK_CONTAINER (draw_window), vbox);
    gtk_widget_show (vbox);

    gtk_signal_connect (GTK_OBJECT (draw_window), "destroy",
                        GTK_SIGNAL_FUNC (quit), NULL);

    /* ΰ  */

    drawing_area = gtk_drawing_area_new ();
    gtk_drawing_area_size (GTK_DRAWING_AREA (drawing_area), 300, 300);
    gtk_box_pack_start (GTK_BOX (vbox), drawing_area, TRUE, TRUE, 0);

    gtk_widget_show (drawing_area);

    /* Хå󥰥ԥåޥåפν˻Ѥ륷ʥ */

    gtk_signal_connect (GTK_OBJECT (drawing_area), "expose_event",
                        (GtkSignalFunc) expose_event, NULL);
    gtk_signal_connect (GTK_OBJECT(drawing_area),"configure_event",
                        (GtkSignalFunc) configure_event, NULL);

    /* ٥ȥʥ */

    gtk_signal_connect (GTK_OBJECT (drawing_area), "motion_notify_event",
                        (GtkSignalFunc) motion_notify_event, NULL);
    gtk_signal_connect (GTK_OBJECT (drawing_area), "button_press_event",
                        (GtkSignalFunc) button_press_event, NULL);
    gtk_signal_connect (GTK_OBJECT (drawing_area), "button_release_event",
                        (GtkSignalFunc) button_release_event, NULL);

    gtk_widget_set_events (drawing_area, GDK_EXPOSURE_MASK
                           | GDK_LEAVE_NOTIFY_MASK
                           | GDK_BUTTON_PRESS_MASK
                           | GDK_BUTTON_RELEASE_MASK
                           | GDK_POINTER_MOTION_MASK
                           | GDK_POINTER_MOTION_HINT_MASK);

    clear_button = gtk_button_new_with_label (_("Clear"));
    gtk_box_pack_start (GTK_BOX (vbox), clear_button, FALSE, FALSE, 0);

    gtk_signal_connect_object (GTK_OBJECT (clear_button), "clicked",
                               GTK_SIGNAL_FUNC (clear),
                               GTK_OBJECT (drawing_area));
    gtk_widget_show (clear_button);

    gtk_widget_show (draw_window);

    /* confirm helper connection */
    check_helper_connection();

    gtk_main ();

    return 0;
}
