/*
 *  Copyright (C) 2006 Juernjakob Harder
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
  #include <config.h>
#endif

#include <glib/gprintf.h>
#include <glib/gi18n.h>
#include <gtk/gtk.h>

#include "tomoe-reading-search.h"
#include "tomoe-char-table.h"

enum {
    LAST_SIGNAL
};

enum {
    CHAR_COLUMN,
    STROKECOUNT_COLUMN,
    READING_COLUMN,
    COLUMN_COUNT
};

typedef struct _TomoeReadingSearchPrivate	TomoeReadingSearchPrivate;
struct _TomoeReadingSearchPrivate
{
    TomoeContext *context;
    GtkListStore *result_store;
    GtkWidget    *input;
};

#define TOMOE_READING_SEARCH_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), TOMOE_TYPE_READING_SEARCH, TomoeReadingSearchPrivate))

G_DEFINE_TYPE (TomoeReadingSearch, tomoe_reading_search, GTK_TYPE_TABLE)

static void dispose (GObject *object);

static void on_find_button_clicked    (GtkButton           *button,
                                       gpointer             user_data);
static void on_input_entry_activate   (GtkEntry            *entry,
                                       gpointer            *user_data);

/*static guint reading_search_signals[LAST_SIGNAL] = { 0 };*/

GtkWidget *
tomoe_reading_search_new (void)
{
    return GTK_WIDGET(g_object_new (TOMOE_TYPE_READING_SEARCH,
                                    NULL));
}

void
tomoe_reading_search_set_context (TomoeReadingSearch *page, TomoeContext *context)
{
    TomoeReadingSearchPrivate *priv = TOMOE_READING_SEARCH_GET_PRIVATE (page);
    priv->context = g_object_ref (context);
}

static void
tomoe_reading_search_class_init (TomoeReadingSearchClass *klass)
{
    GObjectClass *gobject_class = G_OBJECT_CLASS (klass);

    gobject_class->dispose = dispose;

    g_type_class_add_private (gobject_class, sizeof (TomoeReadingSearchPrivate));
}

static void
tomoe_reading_search_init (TomoeReadingSearch *page)
{
    GtkWidget *hbox, *vbox, *input, *button, *list, *label;
    GtkCellRenderer   *renderer;
    GtkTreeViewColumn *column;
    GtkTreeSelection *result_sel;
    GtkWidget *scrolled_window;
    TomoeReadingSearchPrivate *priv = TOMOE_READING_SEARCH_GET_PRIVATE (page);

    gtk_table_resize (GTK_TABLE (page), 1, 1);
    gtk_table_set_homogeneous (GTK_TABLE (page), FALSE);

    vbox = gtk_vbox_new (FALSE, 0);
    gtk_container_set_border_width (GTK_CONTAINER (vbox), 8);
    gtk_table_attach_defaults (GTK_TABLE (page), vbox, 0, 1, 0, 1);
    hbox = gtk_hbox_new (FALSE, 0);
    gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

    label = gtk_label_new (_("Reading:"));
    gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 4);
    gtk_widget_show (label);

    input = gtk_entry_new ();
    priv->input = input;
    gtk_box_pack_start (GTK_BOX (hbox), input, FALSE, FALSE, 4);
    g_signal_connect (G_OBJECT (input), "activate",
                      G_CALLBACK (on_input_entry_activate),
                      (gpointer) page);
    gtk_widget_show (input);

    button = gtk_button_new_from_stock (GTK_STOCK_FIND);
    gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, FALSE, 4);
    g_signal_connect (G_OBJECT (button), "clicked",
                      G_CALLBACK (on_find_button_clicked),
                      (gpointer) page);
    gtk_widget_show (button);

    /* result view */
    priv->result_store = gtk_list_store_new (COLUMN_COUNT, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING);

    list = gtk_tree_view_new_with_model (GTK_TREE_MODEL (priv->result_store));
    result_sel = gtk_tree_view_get_selection (GTK_TREE_VIEW (list));
    gtk_tree_selection_set_mode (result_sel, GTK_SELECTION_SINGLE);
    
    scrolled_window = gtk_scrolled_window_new (NULL, NULL);
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window), 
                                    GTK_POLICY_AUTOMATIC,
                                    GTK_POLICY_AUTOMATIC);
    gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scrolled_window),
					 GTK_SHADOW_ETCHED_IN);
    gtk_container_add (GTK_CONTAINER (scrolled_window), list);
    gtk_box_pack_start (GTK_BOX (vbox), scrolled_window, TRUE, TRUE, 4);
    gtk_widget_show (scrolled_window);

    renderer = gtk_cell_renderer_text_new ();
    column = gtk_tree_view_column_new_with_attributes (_("Character"),
                                                       renderer,
                                                       "text", CHAR_COLUMN,
                                                       NULL);
    gtk_tree_view_append_column (GTK_TREE_VIEW (list), column);
    renderer = gtk_cell_renderer_text_new ();
    column = gtk_tree_view_column_new_with_attributes (_("Stroke count"),
                                                       renderer,
                                                       "text", STROKECOUNT_COLUMN,
                                                       NULL);
    gtk_tree_view_append_column (GTK_TREE_VIEW (list), column);
    renderer = gtk_cell_renderer_text_new ();
    column = gtk_tree_view_column_new_with_attributes (_("Reading"),
                                                       renderer,
                                                       "text", READING_COLUMN,
                                                       NULL);
    gtk_tree_view_append_column (GTK_TREE_VIEW (list), column);
    gtk_widget_set_size_request (list, 400, 320);
    /*
    g_signal_connect (G_OBJECT (page->char_sel), "changed",
                      G_CALLBACK (on_selection_changed),
                      (gpointer) page);
    */
    gtk_widget_show (list);

    gtk_widget_show (hbox);
    gtk_widget_show (vbox);
}

static void
dispose (GObject *object)
{
    TomoeReadingSearchPrivate *priv = TOMOE_READING_SEARCH_GET_PRIVATE (object);

    if (priv->result_store) {
        g_object_unref (priv->result_store);
        priv->result_store = NULL;
    }
    if (priv->context) {
        g_object_unref (priv->context);
        priv->context = NULL;
    }

    if (G_OBJECT_CLASS(tomoe_reading_search_parent_class)->dispose)
        G_OBJECT_CLASS(tomoe_reading_search_parent_class)->dispose(object);
}

static void
tomoe_reading_search_start_search (TomoeReadingSearch *page)
{
    GList *result, *list;
    const gchar *reading_text;
    TomoeQuery *query;
    TomoeReading *reading;
    TomoeReadingSearchPrivate *priv = TOMOE_READING_SEARCH_GET_PRIVATE (page);

    g_return_if_fail (GTK_IS_LIST_STORE (priv->result_store));
    g_return_if_fail (GTK_IS_ENTRY (priv->input));

    gtk_list_store_clear (priv->result_store);

    reading_text = gtk_entry_get_text (GTK_ENTRY (priv->input));

    reading = tomoe_reading_new (TOMOE_READING_UNKNOWN, reading_text);
    query = tomoe_query_new ();
    tomoe_query_add_reading (query, reading);
    result = tomoe_context_search (priv->context, query);
    g_object_unref (query);
    g_object_unref (reading);

    for (list = result; list; list = g_list_next (list)) {
        TomoeCandidate *cand = TOMOE_CANDIDATE (list->data);
        TomoeChar *c = tomoe_candidate_get_char (cand);
        const GList *readings = tomoe_char_get_readings (c);
        gchar *strokes_text, *readings_text;
        gint strokes = 0;
        GtkTreeIter iter;

        if (tomoe_char_get_writing (c))
            strokes = tomoe_writing_get_n_strokes (tomoe_char_get_writing (c));

        if (strokes > 0)
            strokes_text = g_strdup_printf ("%d", strokes);
        else
            strokes_text = g_strdup ("?");

        if (readings) {
            gchar **str_array;
            guint reading_num, i;
            reading_num = g_list_length ((GList *) readings);
            str_array = g_new0 (gchar *, reading_num + 1);
            str_array[reading_num] = NULL;
            for (i = 0; i < reading_num; i++) {
    	        TomoeReading *reading = TOMOE_READING (g_list_nth_data ((GList *) readings, i));
                str_array[i]  = (gchar *) tomoe_reading_get_reading (reading);
            }
            readings_text = g_strjoinv (" ", str_array);
            g_free (str_array);
        } else {
            readings_text = g_strdup ("?");
        }
        gtk_list_store_append (priv->result_store, &iter);
        gtk_list_store_set (priv->result_store, &iter,
                            CHAR_COLUMN, tomoe_char_get_utf8 (c),
                            STROKECOUNT_COLUMN, strokes_text,
                            READING_COLUMN, readings_text,
                            -1);
        g_free (readings_text);
        g_free (strokes_text);
    }

    g_list_foreach (result, (GFunc) g_object_unref, NULL);
    g_list_free (result);
}

static void
on_input_entry_activate (GtkEntry *entry, gpointer *user_data)
{
    TomoeReadingSearch *page = TOMOE_READING_SEARCH (user_data);

    g_return_if_fail (TOMOE_IS_READING_SEARCH (page));

    tomoe_reading_search_start_search (page);
}

static void
on_find_button_clicked (GtkButton *button, gpointer user_data)
{
    TomoeReadingSearch *page = TOMOE_READING_SEARCH (user_data);

    g_return_if_fail (TOMOE_IS_READING_SEARCH (page));

    tomoe_reading_search_start_search (page);
}
/*
 * vi:ts=4:nowrap:ai:expandtab
 */
