/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 *  Copyright (C) 2005 Takuro Ashie
 *  Copyright (C) 2006 Juernjakob Harder
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
  #include <config.h>
#endif

#include <glib/gi18n.h>

#include "tomoe-stroke-search.h"
#include "tomoe-canvas.h"
#include "tomoe-char-table.h"

enum {
    SELECTED_SIGNAL,
    LAST_SIGNAL
};

typedef struct _TomoeStrokeSearchPrivate	TomoeStrokeSearchPrivate;
struct _TomoeStrokeSearchPrivate
{
    GtkWidget *canvas;
    GtkWidget *button_area;
    GtkWidget *find_button;
    GtkWidget *go_back_button;
    GtkWidget *clear_button;
    GtkWidget *normalize_button;
    GtkWidget *candidates_view;
};

#define TOMOE_STROKE_SEARCH_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), TOMOE_TYPE_STROKE_SEARCH, TomoeStrokeSearchPrivate))

G_DEFINE_TYPE (TomoeStrokeSearch, tomoe_stroke_search, GTK_TYPE_TABLE)

static void dispose (GObject *object);
static void tomoe_stroke_search_set_sensitive    (TomoeStrokeSearch         *strokeSearch);

/* callbacks for child widgets */
static void on_canvas_stroke_added        (TomoeCanvas         *canvas,
                                           gpointer             user_data);
static void on_find_button_clicked        (GtkButton           *button,
                                           gpointer             user_data);
static void on_go_back_button_clicked     (GtkButton           *button,
                                           gpointer             user_data);
static void on_clear_button_clicked       (GtkButton           *button,
                                           gpointer             user_data);
static void on_normalize_button_clicked   (GtkButton           *button,
                                           gpointer             user_data);
#ifdef ENABLE_DUMPSTROKE
static void on_dump_button_clicked        (GtkButton           *button,
                                           gpointer             user_data);
#endif
static gboolean on_candidate_selected     (TomoeCharTable      *table,
                                           GdkEventButton      *event,
                                           gpointer             user_data);

static guint stroke_search_signals[LAST_SIGNAL] = { 0 };

static void
tomoe_stroke_search_class_init (TomoeStrokeSearchClass *klass)
{
    GObjectClass *gobject_class = G_OBJECT_CLASS (klass);

    stroke_search_signals[SELECTED_SIGNAL] =
      g_signal_new ("selected",
  		  G_TYPE_FROM_CLASS (klass),
  		  G_SIGNAL_RUN_LAST,
  		  G_STRUCT_OFFSET (TomoeStrokeSearchClass, selected),
  		  NULL, NULL,
  		  g_cclosure_marshal_VOID__VOID,
  		  G_TYPE_NONE, 0);

    gobject_class->dispose = dispose;

    klass->selected        = NULL;

    g_type_class_add_private (gobject_class, sizeof (TomoeStrokeSearchPrivate));
}

static void
tomoe_stroke_search_init (TomoeStrokeSearch *strokeSearch)
{
    GtkWidget *main_vbox, *hbox, *vbox, *frame, *alignment;
    GtkWidget *canvas, *button, *table;
    TomoeStrokeSearchPrivate *priv = TOMOE_STROKE_SEARCH_GET_PRIVATE (strokeSearch);

    gtk_table_resize (GTK_TABLE (strokeSearch), 1, 1);
    gtk_table_set_homogeneous (GTK_TABLE (strokeSearch), TRUE);

    main_vbox = gtk_vbox_new (FALSE, 0);
    gtk_container_set_border_width (GTK_CONTAINER (main_vbox), 8);
    gtk_table_attach_defaults (GTK_TABLE (strokeSearch), main_vbox, 
                               0, 1, 0, 1);
    gtk_widget_show (main_vbox);

    hbox = gtk_hbox_new (FALSE, 0);
    gtk_box_pack_start (GTK_BOX (main_vbox), hbox,
                        TRUE, TRUE, 0);
    gtk_widget_show (hbox);

    /* canvas */
    frame = gtk_frame_new (NULL);
    gtk_box_pack_start (GTK_BOX (hbox), frame, TRUE, TRUE, 0);
    gtk_widget_show (frame);

    canvas = tomoe_canvas_new ();
    priv->canvas = canvas;
    gtk_widget_set_size_request (canvas, 300, 300);
    gtk_container_add (GTK_CONTAINER (frame), canvas);
    g_signal_connect (G_OBJECT (canvas), "stroke-added",
                      G_CALLBACK (on_canvas_stroke_added),
                      (gpointer) strokeSearch);
    gtk_widget_show (canvas);

    /* button area */
    alignment = gtk_alignment_new (0.5, 0.5, 1.0, 1.0);
    gtk_alignment_set_padding (GTK_ALIGNMENT (alignment), 0, 0, 8, 0);
    gtk_box_pack_start (GTK_BOX (hbox), alignment, FALSE, FALSE, 0);
    gtk_widget_show (alignment);

    vbox = gtk_vbox_new (FALSE, 0);
    priv->button_area = vbox;
    gtk_container_add (GTK_CONTAINER (alignment), vbox);
    gtk_widget_show (vbox);

    button = gtk_button_new_from_stock (GTK_STOCK_FIND);
    priv->find_button = button;
    gtk_box_pack_start (GTK_BOX (vbox), button, FALSE, FALSE, 4);
    g_signal_connect (G_OBJECT (button), "clicked",
                      G_CALLBACK (on_find_button_clicked),
                      (gpointer) strokeSearch);
    gtk_widget_show (button);

    button = gtk_button_new_from_stock (GTK_STOCK_GO_BACK);
    priv->go_back_button = button;
    gtk_box_pack_start (GTK_BOX (vbox), button, FALSE, FALSE, 4);
    g_signal_connect (G_OBJECT (button), "clicked",
                      G_CALLBACK (on_go_back_button_clicked),
                      (gpointer) strokeSearch);
    gtk_widget_show (button);

    button = gtk_button_new_from_stock (GTK_STOCK_CLEAR);
    priv->clear_button = button;
    gtk_box_pack_start (GTK_BOX (vbox), button, FALSE, FALSE, 4);
    g_signal_connect (G_OBJECT (button), "clicked",
                      G_CALLBACK (on_clear_button_clicked),
                      (gpointer) strokeSearch);
    gtk_widget_show (button);

    button = gtk_button_new_from_stock (GTK_STOCK_ZOOM_FIT);
    priv->normalize_button = button;
    gtk_box_pack_start (GTK_BOX (vbox), button, FALSE, FALSE, 4);
    g_signal_connect (G_OBJECT (button), "clicked",
                      G_CALLBACK (on_normalize_button_clicked),
                      (gpointer) strokeSearch);
    gtk_widget_show (button);
#ifdef ENABLE_DUMPSTROKE
    button = gtk_button_new_from_stock (GTK_STOCK_SAVE);
    gtk_box_pack_start (GTK_BOX (vbox), button, FALSE, FALSE, 4);
    g_signal_connect (G_OBJECT (button), "clicked",
                      G_CALLBACK (on_dump_button_clicked),
                      (gpointer) strokeSearch);
    gtk_widget_show (button);
#endif

    /* candidates view */
    frame = gtk_frame_new (NULL);
    gtk_box_pack_start (GTK_BOX (main_vbox), frame, TRUE, TRUE, 4);
    gtk_widget_show (frame);

    table = tomoe_char_table_new ();
    priv->candidates_view = table;
    tomoe_char_table_set_canvas (TOMOE_CHAR_TABLE (table),
                                 TOMOE_CANVAS (canvas));
    g_signal_connect (G_OBJECT (table), "button-release-event",
                      G_CALLBACK (on_candidate_selected),
                      (gpointer) strokeSearch);
    gtk_container_add (GTK_CONTAINER (frame), table);
    gtk_widget_show (table);

    tomoe_stroke_search_set_sensitive (strokeSearch);
}

static void
dispose (GObject *object)
{
    if (G_OBJECT_CLASS(tomoe_stroke_search_parent_class)->dispose)
        G_OBJECT_CLASS(tomoe_stroke_search_parent_class)->dispose(object);
}

static void
tomoe_stroke_search_set_sensitive (TomoeStrokeSearch *strokeSearch)
{
    TomoeStrokeSearchPrivate *priv = TOMOE_STROKE_SEARCH_GET_PRIVATE (strokeSearch);
    gboolean editable = tomoe_canvas_has_stroke (TOMOE_CANVAS (priv->canvas));

    gtk_widget_set_sensitive (priv->find_button,      editable);
    gtk_widget_set_sensitive (priv->go_back_button,   editable);
    gtk_widget_set_sensitive (priv->clear_button,     editable);
    gtk_widget_set_sensitive (priv->normalize_button, editable);
}

GtkWidget *
tomoe_stroke_search_new (void)
{
    return GTK_WIDGET(g_object_new (TOMOE_TYPE_STROKE_SEARCH,
                                    NULL));
}

GtkWidget *
tomoe_stroke_search_get_canvas (TomoeStrokeSearch *strokeSearch)
{
    g_return_val_if_fail (TOMOE_IS_STROKE_SEARCH (strokeSearch), NULL);

    return TOMOE_STROKE_SEARCH_GET_PRIVATE (strokeSearch)->canvas;
}

static void
on_canvas_stroke_added (TomoeCanvas *canvas, gpointer user_data)
{
    TomoeStrokeSearch *strokeSearch = TOMOE_STROKE_SEARCH (user_data);

    tomoe_stroke_search_set_sensitive (strokeSearch);
}

static void
on_find_button_clicked (GtkButton *button, gpointer user_data)
{
    TomoeStrokeSearch *strokeSearch = TOMOE_STROKE_SEARCH (user_data);
    TomoeStrokeSearchPrivate *priv = TOMOE_STROKE_SEARCH_GET_PRIVATE (strokeSearch);

    g_return_if_fail (TOMOE_IS_STROKE_SEARCH (strokeSearch));
    g_return_if_fail (TOMOE_IS_CANVAS (priv->canvas));

    tomoe_canvas_find (TOMOE_CANVAS (priv->canvas));
}

static void
on_go_back_button_clicked (GtkButton *button, gpointer user_data)
{
    TomoeStrokeSearch *strokeSearch = TOMOE_STROKE_SEARCH (user_data);
    TomoeStrokeSearchPrivate *priv = TOMOE_STROKE_SEARCH_GET_PRIVATE (strokeSearch);

    g_return_if_fail (TOMOE_IS_STROKE_SEARCH (strokeSearch));
    g_return_if_fail (TOMOE_IS_CANVAS (priv->canvas));

    tomoe_canvas_revert (TOMOE_CANVAS (priv->canvas));
    tomoe_canvas_find (TOMOE_CANVAS (priv->canvas));
    tomoe_stroke_search_set_sensitive (strokeSearch);
}

static void
on_clear_button_clicked (GtkButton *button, gpointer user_data)
{
    TomoeStrokeSearch *strokeSearch = TOMOE_STROKE_SEARCH (user_data);
    TomoeStrokeSearchPrivate *priv = TOMOE_STROKE_SEARCH_GET_PRIVATE (strokeSearch);

    g_return_if_fail (TOMOE_IS_STROKE_SEARCH (strokeSearch));
    g_return_if_fail (TOMOE_IS_CANVAS (priv->canvas));

    tomoe_canvas_clear (TOMOE_CANVAS (priv->canvas));
    tomoe_stroke_search_set_sensitive (strokeSearch);
}

static void
on_normalize_button_clicked (GtkButton *button, gpointer user_data)
{
    TomoeStrokeSearch *strokeSearch = TOMOE_STROKE_SEARCH (user_data);
    TomoeStrokeSearchPrivate *priv = TOMOE_STROKE_SEARCH_GET_PRIVATE (strokeSearch);

    g_return_if_fail (TOMOE_IS_STROKE_SEARCH (strokeSearch));
    g_return_if_fail (TOMOE_IS_CANVAS (priv->canvas));

    tomoe_canvas_normalize (TOMOE_CANVAS (priv->canvas));
    tomoe_stroke_search_set_sensitive (strokeSearch);
}


/* TODO does not work */
static gboolean
on_candidate_selected (TomoeCharTable *table,
                       GdkEventButton *event,
                       gpointer user_data)
{
    TomoeStrokeSearch *strokeSearch = TOMOE_STROKE_SEARCH (user_data);

    g_return_val_if_fail (TOMOE_IS_STROKE_SEARCH (strokeSearch), FALSE);

    tomoe_stroke_search_set_sensitive (strokeSearch);
    return FALSE;
}

GtkWidget *
tomoe_stroke_search_get_button_area (TomoeStrokeSearch *stroke_search)
{
    TomoeStrokeSearchPrivate *priv;

    g_return_val_if_fail (TOMOE_IS_STROKE_SEARCH (stroke_search), NULL);
    priv = TOMOE_STROKE_SEARCH_GET_PRIVATE (stroke_search);

    return priv->button_area;
}

GtkWidget *
tomoe_stroke_search_get_candidates_view (TomoeStrokeSearch *stroke_search)
{
    TomoeStrokeSearchPrivate *priv;

    g_return_val_if_fail (TOMOE_IS_STROKE_SEARCH (stroke_search), NULL);
    priv = TOMOE_STROKE_SEARCH_GET_PRIVATE (stroke_search);

    return priv->candidates_view;
}

#ifdef ENABLE_DUMPSTROKE
static void
on_dump_button_clicked (GtkButton *button, gpointer user_data)
{
    TomoeStrokeSearch *strokeSearch = TOMOE_STROKE_SEARCH (user_data);
    TomoeStrokeSearchPrivate *priv = TOMOE_STROKE_SEARCH_GET_PRIVATE (strokeSearch);
    TomoeWriting *writing;
    GtkWidget *dialog;
    GList *strokes, *list;
    GList *candidates;
    GString *dump_string;
    gchar *filename;

    g_return_if_fail (TOMOE_IS_STROKE_SEARCH (strokeSearch));
    g_return_if_fail (TOMOE_IS_CANVAS (priv->canvas));

    dialog = gtk_file_chooser_dialog_new (_("Select the file name for dump"),
                                          NULL,
                                          GTK_FILE_CHOOSER_ACTION_SAVE,
                                          GTK_STOCK_SAVE, GTK_RESPONSE_ACCEPT,
                                          NULL);

    if (gtk_dialog_run (GTK_DIALOG (dialog)) != GTK_RESPONSE_ACCEPT) {
        gtk_widget_destroy (dialog);
        return;
    }
    filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
    gtk_widget_destroy (dialog);

    dump_string = g_string_new (NULL);
    writing = tomoe_canvas_get_writing (TOMOE_CANVAS (priv->canvas));

    /* set candidate data */
    candidates = (GList *) tomoe_canvas_get_candidates (TOMOE_CANVAS (priv->canvas));
    for (list = candidates; list; list = g_list_next (list)) {
        TomoeCandidate *cand = TOMOE_CANDIDATE (list->data);
        TomoeChar *c = tomoe_candidate_get_char (cand);
        if (list != candidates) 
            g_string_append_c (dump_string, ' ');
        g_string_append (dump_string, tomoe_char_get_utf8 (c));
    }
    g_string_append_c (dump_string, '\n');

    /* set stroke data */
    strokes = (GList *) tomoe_writing_get_strokes (writing);
    for (list = strokes; list; list = g_list_next (list)) {
        GList *points = (GList *) list->data;
        GList *point;

        for (point = points; point; point = g_list_next (point)) {
            TomoePoint *p = (TomoePoint *) point->data;
            if (point != points)
                g_string_append (dump_string, ", ");
            g_string_append_printf (dump_string, "%d %d", p->x, p->y);
        }
        g_string_append_c (dump_string, '\n');
    }

    g_file_set_contents (filename, dump_string->str, dump_string->len, NULL);

    g_object_unref (writing);
    g_string_free (dump_string, TRUE);
    g_free (filename);
}
#endif

/*
 * vi:ts=4:nowrap:ai:expandtab
 */
