/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 *  Copyright (C) 2005 Takuro Ashie
 *  Copyright (C) 2006 Juernjakob Harder
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
  #include <config.h>
#endif

#include <glib/gi18n.h>
#include <tomoe.h>

#include "tomoe-window.h"
#include "tomoe-canvas.h"
#include "tomoe-handwriting.h"
#include "tomoe-char-table.h"
#include "tomoe-reading-search.h"
#include "tomoe-gucharmap.h"

#define TOMOE_DATA_VIEW_ICON LIBTOMOEGTKDATADIR"/tomoe-data-view.png"
#define TOMOE_DICTIONARY_ICON LIBTOMOEGTKDATADIR"/tomoe-dictionary.png"
#define TOMOE_HANDWRITING_ICON LIBTOMOEGTKDATADIR"/tomoe-handwriting.png"
#define TOMOE_SEARCHING_ICON LIBTOMOEGTKDATADIR"/tomoe-search.png"
#define TOMOE_GUCHARMAP_ICON LIBTOMOEGTKDATADIR"/tomoe-gucharmap.png"

enum {
    SELECTED_SIGNAL,
    LAST_SIGNAL
};

enum {
    PROP_0,
    PROP_HANDWRITING_PAGE,
    PROP_READING_SEARCH_PAGE,
    PROP_GUCHARMAP_PAGE
};

typedef struct _TomoeWindowPrivate	TomoeWindowPrivate;
struct _TomoeWindowPrivate
{
    TomoeContext *context;

    GtkWidget    *notebook;

    /* child pages */
    GtkWidget    *handwriting;
    GtkWidget    *reading;
    GtkWidget    *gucharmap;
    GtkTooltips  *tooltips;
};

#define TOMOE_WINDOW_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), TOMOE_TYPE_WINDOW, TomoeWindowPrivate))

G_DEFINE_TYPE (TomoeWindow, tomoe_window, GTK_TYPE_WINDOW)

static void dispose                           (GObject            *object);
static void set_property                      (GObject            *object,
                                               guint               prop_id,
                                               const GValue       *value,
                                               GParamSpec         *pspec);
static void get_property                      (GObject            *object,
                                               guint               prop_id,
                                               GValue             *value,
                                               GParamSpec         *pspec);

static void on_notebook_switch_page           (GtkNotebook        *notebook,
                                               GtkNotebookPage    *page,
                                               guint               page_num,
                                               gpointer            user_data);
static void on_handwriting_candidate_selected (TomoeHandwriting   *view,
                                               gpointer            user_data);
static void on_reading_search_selected        (TomoeReadingSearch *search,
                                               gpointer            user_data);
static void on_gucharmap_selected             (TomoeGucharmap     *gucharmap,
                                               gpointer            data);

static guint window_signals[LAST_SIGNAL] = { 0 };

static void
tomoe_window_class_init (TomoeWindowClass *klass)
{
    GObjectClass *gobject_class = G_OBJECT_CLASS (klass);
    GParamSpec *spec;

    gobject_class->dispose      = dispose;
    gobject_class->set_property = set_property;
    gobject_class->get_property = get_property;

    window_signals[SELECTED_SIGNAL] =
      g_signal_new ("selected",
  		  G_TYPE_FROM_CLASS (klass),
  		  G_SIGNAL_RUN_LAST,
  		  G_STRUCT_OFFSET (TomoeWindowClass, selected),
  		  NULL, NULL,
  		  g_cclosure_marshal_VOID__VOID,
  		  G_TYPE_NONE, 0);

    spec = g_param_spec_object (
        "handwriting-page",
        N_("TomoeHandwriting"),
        N_("The TomoeHandwriting widget inside of the TomoeWindow."),
        GTK_TYPE_WIDGET, G_PARAM_READABLE);
    g_object_class_install_property (gobject_class,
                                     PROP_HANDWRITING_PAGE, spec);

    spec = g_param_spec_object (
        "reading-search-page",
        N_("TomoeReadingSearch"),
        N_("The TomoeReadingSearch widget inside of the TomoeWindow."),
        GTK_TYPE_WIDGET, G_PARAM_READABLE);
    g_object_class_install_property (gobject_class,
                                     PROP_READING_SEARCH_PAGE, spec);

    spec = g_param_spec_object (
        "gucharmap-page",
        N_("TomoeGucharmap"),
        N_("The TomoeGucharmap widget inside of the TomoeWindow."),
        GTK_TYPE_WIDGET, G_PARAM_READABLE);
    g_object_class_install_property (gobject_class,
                                     PROP_GUCHARMAP_PAGE, spec);

    g_type_class_add_private (gobject_class, sizeof (TomoeWindowPrivate));
}

static void
tomoe_window_init (TomoeWindow *window)
{
    TomoeHandwriting *handwriting_page;
    GtkWidget *widget, *notebook, *vbox;
    TomoeWindowPrivate *priv = TOMOE_WINDOW_GET_PRIVATE (window);

#ifdef ENABLE_NLS
    bindtextdomain (GETTEXT_PACKAGE, LIBTOMOE_GTK_LOCALEDIR);
    textdomain (GETTEXT_PACKAGE);
    bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
#endif

    /* load libtomoe */
    tomoe_init ();
    
    priv->context = tomoe_context_new ();
    /*
    tomoe_context_load_config (priv->context,
                               g_getenv ("TOMOE_CONFIG_FILE"));
    */
    tomoe_context_load_config (priv->context, NULL);

    gtk_window_set_title (GTK_WINDOW (window),
                          _("Handwriting recognition"));

    vbox = gtk_vbox_new (FALSE, 0);
    gtk_container_add (GTK_CONTAINER (window), vbox);
    gtk_widget_show (vbox);

    // tooltips
    priv->tooltips = gtk_tooltips_new ();
    g_object_ref (G_OBJECT (priv->tooltips));
    gtk_object_sink (GTK_OBJECT (priv->tooltips));

    /* create search pages view notebook */
    notebook = gtk_notebook_new ();
    priv->notebook = notebook;
    gtk_notebook_set_tab_pos (GTK_NOTEBOOK (notebook), GTK_POS_LEFT);
    gtk_notebook_set_show_tabs (GTK_NOTEBOOK (notebook), TRUE);
    gtk_notebook_set_show_border (GTK_NOTEBOOK (notebook), FALSE);
    g_signal_connect (G_OBJECT (notebook), "switch-page",
                      G_CALLBACK (on_notebook_switch_page), window);
    gtk_container_add (GTK_CONTAINER (vbox), notebook);
    gtk_widget_show (notebook);

    /* handwriting page */
    widget = tomoe_handwriting_new (priv->context);
    priv->handwriting = widget;
    handwriting_page = TOMOE_HANDWRITING (widget);
    g_signal_connect (G_OBJECT (widget), "selected",
                      G_CALLBACK (on_handwriting_candidate_selected),
                      (gpointer) window);
    gtk_widget_show (widget);
    tomoe_window_append_page (window, widget, 
                              gtk_image_new_from_file (TOMOE_HANDWRITING_ICON),
                              _("Handwriting recognition"));

    /* reading search page */
    widget = tomoe_reading_search_new (priv->context);
    priv->reading = widget;
    g_signal_connect (G_OBJECT (widget), "selected",
                      G_CALLBACK (on_reading_search_selected),
                      (gpointer) window);
    gtk_widget_show (widget);
    tomoe_window_append_page (window, widget,
                              gtk_image_new_from_file (TOMOE_SEARCHING_ICON),
                              _("Search with reading"));

    /* gucharmap page */
    widget = tomoe_gucharmap_new ();
    priv->gucharmap = widget;
    g_signal_connect (G_OBJECT (widget), "selected",
                      G_CALLBACK (on_gucharmap_selected),
                      (gpointer) window);
    gtk_widget_show (widget);
    tomoe_window_append_page (window, widget,
                              gtk_image_new_from_file (TOMOE_GUCHARMAP_ICON),
                              _("Unicode table"));
}

static void
dispose (GObject *object)
{
    TomoeWindowPrivate *priv = TOMOE_WINDOW_GET_PRIVATE (object);

    if (priv->context) {
        g_object_unref (priv->context);
        priv->context = NULL;
    }

    if (priv->tooltips) {
        g_object_unref (priv->tooltips);
        priv->tooltips = NULL;
    }

    tomoe_quit ();

    if (G_OBJECT_CLASS(tomoe_window_parent_class)->dispose)
        G_OBJECT_CLASS(tomoe_window_parent_class)->dispose(object);
}

GtkWidget *
tomoe_window_new (void)
{
    return GTK_WIDGET(g_object_new (TOMOE_TYPE_WINDOW,
                                    NULL));
}

static void
set_property (GObject *object,
              guint prop_id,
              const GValue *value,
              GParamSpec *pspec)
{
#if 0
    TomoeWindow *window = TOMOE_WINDOW (object);
    TomoeWindowPrivate *priv = TOMOE_WINDOW_GET_PRIVATE (window);
#endif

    switch (prop_id) {
    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
        break;
    }
}

static void
get_property (GObject *object,
              guint prop_id,
              GValue *value,
              GParamSpec *pspec)
{
    TomoeWindow *window = TOMOE_WINDOW (object);
    TomoeWindowPrivate *priv = TOMOE_WINDOW_GET_PRIVATE (window);

    switch (prop_id) {
    case PROP_HANDWRITING_PAGE:
        g_value_set_object (value, priv->handwriting);
        break;
    case PROP_READING_SEARCH_PAGE:
        g_value_set_object (value, priv->reading);
        break;
    case PROP_GUCHARMAP_PAGE:
        g_value_set_object (value, priv->gucharmap);
        break;
    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
        break;
    }
}

void
tomoe_window_append_page (TomoeWindow *window,
                          GtkWidget   *page,
                          GtkWidget   *label,
                          const gchar *label_text)
{
    TomoeWindowPrivate *priv;
    GtkWidget *event_box = NULL;

    g_return_if_fail (TOMOE_IS_WINDOW (window));

    if (!page || !GTK_IS_WIDGET (page)) return;

    priv = TOMOE_WINDOW_GET_PRIVATE (window);

    g_object_set_data_full (G_OBJECT (page), "TomoeWindow::page-title",
                            g_strdup (label_text), (GDestroyNotify) g_free);

    event_box = gtk_event_box_new ();
    gtk_event_box_set_visible_window (GTK_EVENT_BOX (event_box), FALSE);
    gtk_container_add (GTK_CONTAINER (event_box), label);
    gtk_widget_show (event_box);
    gtk_widget_show (label);

    gtk_notebook_append_page (GTK_NOTEBOOK (priv->notebook), page, event_box);

    if (label_text)
        gtk_tooltips_set_tip (priv->tooltips, event_box, label_text, NULL);
}

GtkWidget *
tomoe_window_get_handwriting_page (TomoeWindow *window)
{
    g_return_val_if_fail (TOMOE_IS_WINDOW (window), NULL);

    return TOMOE_WINDOW_GET_PRIVATE (window)->handwriting;
}

GtkWidget *
tomoe_window_get_notebook (TomoeWindow *window)
{
    g_return_val_if_fail (TOMOE_IS_WINDOW (window), NULL);

    return TOMOE_WINDOW_GET_PRIVATE (window)->notebook;
}

GtkWidget *
tomoe_window_get_reading_search_page (TomoeWindow *window)
{
    g_return_val_if_fail (TOMOE_IS_WINDOW (window), NULL);

    return TOMOE_WINDOW_GET_PRIVATE (window)->reading;
}

GtkWidget *
tomoe_window_get_gucharmap_page (TomoeWindow *window)
{
    g_return_val_if_fail (TOMOE_IS_WINDOW (window), NULL);

    return TOMOE_WINDOW_GET_PRIVATE (window)->gucharmap;
}

const gchar *
tomoe_window_get_selected_char (TomoeWindow *window)
{
    TomoeWindowPrivate *priv;
    gint page_num;
    gpointer widget;

    g_return_val_if_fail (TOMOE_IS_WINDOW (window), NULL);
    priv = TOMOE_WINDOW_GET_PRIVATE (window);

    page_num = gtk_notebook_get_current_page (GTK_NOTEBOOK (priv->notebook));
    widget = gtk_notebook_get_nth_page (GTK_NOTEBOOK (priv->notebook), page_num);

    if (TOMOE_IS_HANDWRITING (widget))
        return tomoe_handwriting_get_selected_char (widget);
    else if (TOMOE_IS_READING_SEARCH (widget))
        return tomoe_reading_search_get_selected_char (widget);
    else if (TOMOE_IS_GUCHARMAP (widget))
        return tomoe_gucharmap_get_selected_char (widget);
    return NULL;
}

TomoeChar *
tomoe_window_get_selected_tomoe_char (TomoeWindow *window)
{
    TomoeWindowPrivate *priv;
    gint page_num;
    gpointer widget;

    g_return_val_if_fail (TOMOE_IS_WINDOW (window), NULL);
    priv = TOMOE_WINDOW_GET_PRIVATE (window);

    page_num = gtk_notebook_get_current_page (GTK_NOTEBOOK (priv->notebook));
    widget = gtk_notebook_get_nth_page (GTK_NOTEBOOK (priv->notebook), page_num);

    if (TOMOE_IS_HANDWRITING (widget))
        return tomoe_handwriting_get_selected_tomoe_char (widget);
    else if (TOMOE_IS_READING_SEARCH (widget))
        return tomoe_reading_search_get_selected_tomoe_char (widget);
    else if (TOMOE_IS_GUCHARMAP (widget))
        return tomoe_gucharmap_get_selected_tomoe_char (widget);
    return NULL;
}

static void
on_notebook_switch_page (GtkNotebook *notebook, GtkNotebookPage *page,
                         guint page_num, gpointer user_data)
{
    TomoeWindow *window = TOMOE_WINDOW (user_data);
    GtkWidget *widget = gtk_notebook_get_nth_page (notebook, page_num);

    const gchar *title = g_object_get_data (G_OBJECT (widget),
                                            "TomoeWindow::page-title");
    if (title && *title)
        gtk_window_set_title (GTK_WINDOW (window), title);
    else
        gtk_window_set_title (GTK_WINDOW (window), _("Tomoe"));
}

static void
on_handwriting_candidate_selected (TomoeHandwriting *view, gpointer user_data)
{
    TomoeWindow *window;
    g_return_if_fail (TOMOE_IS_WINDOW (user_data));

    window = TOMOE_WINDOW (user_data);
    g_signal_emit (G_OBJECT (window),
                   window_signals[SELECTED_SIGNAL], 0);
}

static void
on_reading_search_selected (TomoeReadingSearch *search, gpointer user_data)
{
    TomoeWindow *window;
    g_return_if_fail (TOMOE_IS_WINDOW (user_data));

    window = TOMOE_WINDOW (user_data);
    g_signal_emit (G_OBJECT (window),
                   window_signals[SELECTED_SIGNAL], 0);
}

static void
on_gucharmap_selected (TomoeGucharmap *gucharmap, gpointer user_data)
{
    TomoeWindow *window;
    g_return_if_fail (TOMOE_IS_WINDOW (user_data));

    window = TOMOE_WINDOW (user_data);
    g_signal_emit (G_OBJECT (window),
                   window_signals[SELECTED_SIGNAL], 0);
}

/*
 * vi:ts=4:nowrap:ai:expandtab
 */
