/*******************************************************************************
  TPI - flexible but useless plug-in framework.
  Copyright (C) 2002-2009 Silky

  This library is free software; you can redistribute it and/or modify it under
  the terms of the GNU Lesser General Public License as published by the Free
  Software Foundation; either version 2.1 of the License, or (at your option)
  any later version.

  This library is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License
  for more details.

  You should have received a copy of the GNU Lesser General Public License along
  with this library; if not, write to the Free Software Foundation, Inc.,
  51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA

  $Id: sqxArc.cpp 589 2012-01-15 10:47:03Z sirakaba $
*******************************************************************************/

//******************************************************************************
//    Includes
//******************************************************************************

#include "../../common/header/plugin.h"
#include "../../common/header/plugin-extra.h"
#include "../../common/library/library.h"
#include <windows.h>
#include "sqxArc.h"

//******************************************************************************
//    Global varients
//******************************************************************************

HMODULE g_hLib;
TPI_PROC g_prProc;

//******************************************************************************
//    Inside Functions
//******************************************************************************

#define GetAPIAddress(name) GetProcAddress(g_hLib, "Sqx" name)

int ErrorCodeConvert(int nErrorCode)
{
	switch (nErrorCode)
	{
	case SQX_ERR_SUCCESS:                       return TPI_ERROR_SUCCESS;
	case SQX_ERR_ERROR:                         return TPI_ERROR_UNDEFINED;
	case SQX_ERR_FILE_NOT_FOUND:                return TPI_ERROR_IO_NOTFOUND;
	case SQX_ERR_PATH_NOT_FOUND:                return TPI_ERROR_IO_NOTFOUND;
	case SQX_ERR_TOO_MANY_FILES:                return TPI_ERROR_IO_OPEN;
	case SQX_ERR_ACCESS_DENIED:                 return TPI_ERROR_IO_ACCESS;
	case SQX_ERR_INVALID_HANDLE:                return TPI_ERROR_D_PARAMETER;
	case SQX_ERR_DISK_FULL:                     return TPI_ERROR_IO_WRITE;
	case SQX_ERR_OUT_OF_MEMORY:                 return TPI_ERROR_D_OUTOFMEMORY;
	case SQX_ERR_CANT_ACCESS_TEMP_DIR:          return TPI_ERROR_IO_TMP_ACCESS;
	case SQX_ERR_TEMP_DIR_FULL:                 return TPI_ERROR_IO_TMP_WRITE;
	case SQX_ERR_USER_ABORT:                    return TPI_ERROR_D_SKIPPED;
	case SQX_ERR_INVALID_ARC_HANDLE:            return TPI_ERROR_D_PARAMETER;
	case SQX_ERR_CANT_FIND_LANG_DATA:           return TPI_ERROR_UNDEFINED;
	case SQX_ERR_UNKNOWN_SUBSTREAM:             return TPI_ERROR_UNDEFINED;
	case SQX_ERR_BAD_SUBSTREAM_CRC:             return TPI_ERROR_ARC_BROKEN_SUM;
	case SQX_ERR_UNKNOWN_METHOD:                return TPI_ERROR_ARC_UNSUPPORTED;
	case SQX_ERR_FILE_ENCRYPTED:                return TPI_ERROR_ARC_ENCRYPTED;
	case SQX_ERR_BAD_CRC:                       return TPI_ERROR_ARC_BROKEN_SUM;
	case SQX_ERR_CANT_CREATE_FILE:              return TPI_ERROR_IO_OPEN;
	case SQX_ERR_BAD_FILE_FORMAT:               return TPI_ERROR_ARC_BROKEN_HEADER;
	case SQX_ERR_FUNCTION_NOT_SUPPORTED:        return TPI_ERROR_D_UNSUPPORTED;
	case SQX_ERR_FUNC_NOT_SUPPORTED_BY_ARCHIVE: return TPI_ERROR_D_UNSUPPORTED;
	case SQX_ERR_CANT_CREATE_ARC_DIR:           return TPI_ERROR_IO_ARC_WRITE;
	case SQX_ERR_INVALID_DIR_NAME:              return TPI_ERROR_IO_ARC_WRITE;
	case SQX_ERR_INVALID_FILE_NAME:             return TPI_ERROR_IO_ARC_WRITE;
	case SQX_ERR_TOO_MANY_BROKEN_FBLOCKS:       return TPI_ERROR_ARC_BROKEN_MISC;
	case SQX_ERR_ARCHIVE_OK_RDATA_NOT:          return TPI_ERROR_ARC_BROKEN_MISC;
	case SQX_ERR_RDATA_DAMAGED:                 return TPI_ERROR_ARC_BROKEN_MISC;
	case SQX_ERR_RDATA_NEW_VERSION:             return TPI_ERROR_ARC_UNSUPPORTED;
	case SQX_ERR_RDATA_DOES_NOT_MATCH:          return TPI_ERROR_ARC_BROKEN_MISC;
	case SQX_ERR_CANT_FIND_RDATA:               return TPI_ERROR_ARC_MISC;
	case SQX_ERR_ARCHIVE_IS_LOCKED:             return TPI_ERROR_ARC_MISC;
	case SQX_ERR_CANT_ADD_TO_MV:                return TPI_ERROR_ARC_UNSUPPORTED;
	case SQX_ERR_CANT_DELETE_FROM_MV:           return TPI_ERROR_ARC_UNSUPPORTED;
	case SQX_ERR_NEED_1ST_VOLUME:               return TPI_ERROR_ARC_MISC;
	case SQX_ERR_MISSING_VOLUME:                return TPI_ERROR_ARC_MISC;
	case SQX_ERR_VOLUME_LIMIT_REACHED:          return TPI_ERROR_ARC_MISC;
	case SQX_ERR_SFXSTUB_NOT_INSTALLED:         return TPI_ERROR_UNDEFINED;
	case SQX_ERR_BACKUP_READ_ACCESS_DENIED:     return TPI_ERROR_IO_ACCESS;
	case SQX_ERR_BACKUP_WRITE_ACCESS_DENIED:    return TPI_ERROR_IO_WRITE;
	case SQX_ERR_ACL_READ_ACCESS_DENIED:        return TPI_ERROR_IO_READ;
	case SQX_ERR_ACL_WRITE_ACCESS_DENIED:       return TPI_ERROR_IO_WRITE;
	case SQX_ERR_WRONG_ARCHIVER_VERSION:        return TPI_ERROR_ARC_UNSUPPORTED;
	case SQX_ERR_CANT_COPY_SOURCE_TO_SOURCE:    return TPI_ERROR_IO_ARC_COPY;
	case SQX_ERR_VOLUMES_TOO_SMALL:             return TPI_ERROR_ARC_MISC;
	case SQX_ERR_ARCHIVE_VERSION_TOO_HIGH:      return TPI_ERROR_ARC_UNSUPPORTED;
	case SQX_ERR_EXT_RDATA_DOES_NOT_MATCH:      return TPI_ERROR_ARC_MISC;
	case SQX_ERR_BAD_PARAMETER:                 return TPI_ERROR_D_PARAMETER;
	case SQX_ERR_EQUAL_PASSWORDS:               return TPI_ERROR_ARC_MISC;
	case SQX_ERR_REQUIRES_ENCRYPTION:           return TPI_ERROR_ARC_MISC;
	case SQX_ERR_MISSING_HEADER_PASSWORD:       return TPI_ERROR_ARC_MISC;
	case SQX_ERR_MISSING_SQX_PRIVATE_KEY:       return TPI_ERROR_ARC_MISC;
	case SQX_ERR_MISSING_SQX_AVKEY:             return TPI_ERROR_ARC_MISC;
	case SQX_ERR_MISSING_EXTERNAL_AVKEY:        return TPI_ERROR_ARC_MISC;
	case SQX_ERR_INVALID_SQX_AVKEY:             return TPI_ERROR_UNDEFINED;
	case SQX_ERR_SQX_AVKEY_VERSION:             return TPI_ERROR_UNDEFINED;
	case SQX_ERR_SQX_AVDATA_VERSION:            return TPI_ERROR_UNDEFINED;
	case SQX_ERR_SQX_BROKEN_AVRECORD:           return TPI_ERROR_ARC_BROKEN_HEADER;
	case SQX_ERR_RIJNDAEL_RSA:                  return TPI_ERROR_UNDEFINED;
	case SQX_ERR_REQUIRES_NTFS:                 return TPI_ERROR_UNDEFINED;
	case SQX_ERR_REQUIRES_WINNT:                return TPI_ERROR_UNDEFINED;
	case SQX_ERR_REQUIRES_W2K:                  return TPI_ERROR_UNDEFINED;
	case SQX_ERR_REQUIRES_WINXP:                return TPI_ERROR_UNDEFINED;
	case SQX_ERR_REQUIRES_WINXP_SP1:            return TPI_ERROR_UNDEFINED;
	case SQX_ERR_REQUIRES_WINXP_SP2:            return TPI_ERROR_UNDEFINED;
	case SQX_ERR_REQUIRES_LONGHORN:             return TPI_ERROR_UNDEFINED;
	case SQX_ERR_NO_RESOURCES_FOUND:            return TPI_ERROR_UNDEFINED;
	case SQX_ERR_UNKNOWN_ICON_FORMAT:           return TPI_ERROR_UNDEFINED;
	case SQX_ERR_NO_MATCHING_ICON_SIZE:         return TPI_ERROR_UNDEFINED;
	case SQX_ERR_UNKNOWN_EXE_FORMAT:            return TPI_ERROR_UNDEFINED;
	case SQX_ERR_REQUIRES_SOURCE_PATH:          return TPI_ERROR_D_PARAMETER;
	case SQX_ERR_FILE_DATA_NOT_EQUAL:           return TPI_ERROR_ARC_BROKEN_MISC;
	case SQX_ERR_COMMENT_BIGGER_4K:             return TPI_ERROR_ARC_MISC;
	case SQX_ERR_CANT_CREATE_SFX_FROM_MV:       return TPI_ERROR_ARC_UNSUPPORTED;
	default:                                    return TPI_ERROR_UNDEFINED;
	}
}

//******************************************************************************
//    Callback Wrapper
//******************************************************************************

int CALLBACK CallbackProc(void *, SQX_CALLBACKINFO * pCallbackInfo)
{
	static TPI_PROCESSINFO piInfo;
	switch (pCallbackInfo->dwCallbackType)
	{
	case SQX_CALLBACK_FILENAME:
		piInfo.eMessage             = TPI_MESSAGE_STATUS;
		piInfo.eStatus              = TPI_STATUS_BEGINPROCESS;
		piInfo.fiInfo.szStoredName  = WC2String(pCallbackInfo->pszSourceFileName);
		piInfo.fiInfo.fnFileName    = wxFileName(piInfo.fiInfo.szStoredName);
		piInfo.fnDestination        = wxFileName(WC2String(pCallbackInfo->pszTargetFileName));
		return
			g_prProc == NULL || g_prProc(TPI_NOTIFY_COMMON, & piInfo) != TPI_CALLBACK_CANCEL ?
				SQX_PROGRESS_OK : SQX_PROGRESS_CANCEL;

	case SQX_CALLBACK_PROGRESS:
		piInfo.eMessage = TPI_MESSAGE_STATUS;
		piInfo.eStatus  = TPI_STATUS_INPROCESS;
		piInfo.nProcessedSize = pCallbackInfo->iProgress;
		return
			g_prProc == NULL || g_prProc(TPI_NOTIFY_COMMON, & piInfo) != TPI_CALLBACK_CANCEL ?
				SQX_PROGRESS_OK : SQX_PROGRESS_CANCEL;

	case SQX_CALLBACK_REPLACE:
		piInfo.eMessage             = TPI_MESSAGE_ASK;
		piInfo.eStatus              = TPI_PARAM_DEST;
		piInfo.fiInfo.szStoredName  = WC2String(pCallbackInfo->pFindDataReplace->cFileName);
		piInfo.fiInfo.fnFileName    = wxFileName(piInfo.fiInfo.szStoredName);
		piInfo.fiInfo.dwAttribute   = pCallbackInfo->pFindDataReplace->dwFileAttributes;
		piInfo.fiInfo.nUnpackedSize = pCallbackInfo->pFindDataReplace->nFileSizeLow | ((wxULongLong_t) pCallbackInfo->pFindDataReplace->nFileSizeHigh) << 32;
		piInfo.fiInfo.tmAccess      = FileTimeToWxDateTime(& pCallbackInfo->pFindDataReplace->ftLastAccessTime);
		piInfo.fiInfo.tmModify      = FileTimeToWxDateTime(& pCallbackInfo->pFindDataReplace->ftLastWriteTime);
		piInfo.fiInfo.tmCreate      = FileTimeToWxDateTime(& pCallbackInfo->pFindDataReplace->ftCreationTime);
		return
			g_prProc == NULL ? SQX_REPLACE_OVERWRITE :
			g_prProc(TPI_NOTIFY_COMMON, & piInfo) == TPI_CALLBACK_CANCEL ? SQX_PROGRESS_CANCEL :
			piInfo.fnDestination.IsOk() ? SQX_REPLACE_OVERWRITE : SQX_REPLACE_SKIP;

	case SQX_CALLBACK_PASSWORD:
	case SQX_CALLBACK_PASSWORD_HEADER:
		piInfo.eMessage = TPI_MESSAGE_ASK;
		piInfo.eStatus  = TPI_PARAM_PASSWORD;
		piInfo.szParam.Empty();
		if (g_prProc == NULL || g_prProc(TPI_NOTIFY_COMMON, & piInfo) != TPI_CALLBACK_CONTINUE)
		{
			return SQX_PASSWORD_CANCEL;
		}
		wcsncpy(pCallbackInfo->szCryptKey, piInfo.szParam.c_str(), sizeof(pCallbackInfo->szCryptKey));
		return SQX_PASSWORD_OK;

	case SQX_CALLBACK_SKIP:
		return SQX_SKIPFILE_SKIP;

	case SQX_CALLBACK_NEXTDISK:
		piInfo.eMessage = TPI_MESSAGE_ASK;
		piInfo.eStatus  = TPI_PARAM_NEXTVOLUME;
		piInfo.szParam.Empty();
		if (g_prProc == NULL || g_prProc(TPI_NOTIFY_COMMON, & piInfo) != TPI_CALLBACK_CONTINUE)
		{
			return SQX_NEXTDISK_CANCEL;
		}
		wcsncpy(pCallbackInfo->szNextDiskPath, piInfo.szParam.c_str(), sizeof(pCallbackInfo->szNextDiskPath));
		return SQX_NEXTDISK_OK;

	case SQX_CALLBACK_SIGNAL:
		piInfo.eMessage = TPI_MESSAGE_STATUS;
		switch (pCallbackInfo->dwSignal)
		{
		case SQX_SIGNAL_COMPRESS:
		case SQX_SIGNAL_UNCOMPRESS:
		case SQX_SIGNAL_DELETE:
		case SQX_SIGNAL_REPAIR_ARCHIVE:
		case SQX_SIGNAL_TEST_ARCHIVE:
			piInfo.eStatus  = TPI_STATUS_OPENARCHIVE;
			break;
		case SQX_SIGNAL_TEMP_ARC_COPY:
			piInfo.eStatus  = TPI_STATUS_COPYARCHIVE;
			break;
		default:
			return 0;
		}

		// コールバック関数に送信。
		if (g_prProc != NULL)
		{
			g_prProc(TPI_NOTIFY_COMMON, & piInfo);
		}
		return 0;

	default:
		return 0;
	}
}

//******************************************************************************
//    Functions
//******************************************************************************

#ifdef __cplusplus
extern "C"
{
#endif

int __stdcall GetPluginInformation
(
	unsigned int _uInfoId,
	wxULongLong_t,
	void * _pPtr
)
{
	if (_pPtr == NULL)
	{
		return TPI_ERROR_D_PARAMETER;
	}
	switch (_uInfoId)
	{
	case TPI_INFO_VERSION_MAJOR:
	case TPI_INFO_VERSION_MINOR:
		* (int *) _pPtr = 0;
		break;
	case TPI_INFO_VERSION_API:
		* (int *) _pPtr = 2;
		break;
	case TPI_INFO_HANDLE_ON_COMMAND:
		* (int *) _pPtr = 1;
		break;
	default:
		return TPI_ERROR_D_UNSUPPORTED;
	}
	return TPI_ERROR_SUCCESS;
}

int __stdcall GetFormatInformation(TPI_FORMATINFO * _fiInfo, bool _bFirst)
{
	static int s_uFileId = 0;
	if (s_uFileId == 2)
	{
		return TPI_ERROR_S_ENDOFDATA;
	}
	if (_bFirst)
	{
		s_uFileId = 0;
	}

	_fiInfo->szTypeName   = _bFirst ? wxT("SQX1") : wxT("SQX2");
	_fiInfo->szSuffix     = wxT("sqx");
	_fiInfo->szEngineName = wxT("sqx20u.dll");
	_fiInfo->szTPIName    = wxT("sqxArc");
	_fiInfo->nTypeId      = s_uFileId++;
	_fiInfo->eSupportedCommand = TPI_COMMAND_CREATE | TPI_COMMAND_EXTRACT | TPI_COMMAND_DELETE | TPI_COMMAND_TEST | TPI_COMMAND_REPAIR;
	_fiInfo->fArchive     = true;
	_fiInfo->fComment     = true;
	_fiInfo->fSFX         = true;
	_fiInfo->fSolid       = true;
	_fiInfo->fEncryptPassword = true;
	_fiInfo->fEncryptHeader = true;
	_fiInfo->fMultiVolume = true;
	_fiInfo->fMMOptimize  = true;
	_fiInfo->nCompressLevelMin  = 0;
	_fiInfo->nCompressLevelMax  = 5;
	_fiInfo->nRecoveryRecordMin = 0;
	_fiInfo->nRecoveryRecordMax = 5;

	return TPI_ERROR_SUCCESS;
}

int __stdcall LoadPlugin
(
	const wxString &,
	TPI_PROC _prProc,
	wxULongLong_t
)
{
	::RemoveCwdFromSearchPath();
	g_hLib = ::LoadLibrary(L"sqx20u.dll");
	if (g_hLib == NULL)
	{
		::FreeLibrary(g_hLib);
		return TPI_ERROR_U_LOAD_LIBRARY;
	}

	// コールバック関数を設定。
	if (_prProc != NULL)
	{
		g_prProc = * _prProc;
	}
	return TPI_ERROR_SUCCESS;
}

int __stdcall FreePlugin
(
	void * // _pReserved
)
{
	::FreeLibrary(g_hLib);
	return TPI_ERROR_SUCCESS;
}

int __stdcall OpenArchive
(
	const wxString & _szArcName,
	void * * _hArchive,
	wxULongLong_t * _nFileCount
)
{
	FARPROC fpProc = ::GetAPIAddress("InitArchive");
	if (fpProc == NULL)
	{
		return TPI_ERROR_U_USE_LIBRARY;
	}

	int nErrorCode = ErrorCodeConvert(((int (__stdcall *)(const wchar_t *, SQXCALLBACK, void *, void **)) fpProc)(_szArcName.wchar_str(), CallbackProc, NULL, _hArchive));
	if (nErrorCode != TPI_ERROR_SUCCESS)
	{
		return nErrorCode;
	}

	if (_nFileCount != NULL)
	{
		fpProc = ::GetAPIAddress("InitArcFileList");
		if (fpProc == NULL)
		{
			CloseArchive(* _hArchive);
			return TPI_ERROR_U_USE_LIBRARY;
		}
		SQX_ARCLIST * al = ((SQX_ARCLIST * (__stdcall *)(void *)) fpProc)(* _hArchive);
		if (al == NULL)
		{
			CloseArchive(* _hArchive);
			return TPI_ERROR_UNDEFINED;
		}
		* _nFileCount = al->dwItemCount;

		fpProc = ::GetAPIAddress("DoneArcFileList");
		if (fpProc == NULL)
		{
			CloseArchive(* _hArchive);
			return TPI_ERROR_U_USE_LIBRARY;
		}
		((void (__stdcall *)(void *, SQX_ARCLIST *)) fpProc)(* _hArchive, al);
	}

	return TPI_ERROR_SUCCESS;
}

int __stdcall CloseArchive
(
	void * _hArchive
)
{
	FARPROC fpProc = ::GetAPIAddress("DoneArchive");
	if (fpProc == NULL || _hArchive == NULL)
	{
		return TPI_ERROR_U_USE_LIBRARY;
	}
	((void (__stdcall *)(void *)) fpProc)(_hArchive);
	return TPI_ERROR_SUCCESS;
}

int __stdcall GetFileInformation
(
	void * _hArchive,
	TPI_FILEINFO * _fiInfo,
	bool _bFirst
)
{
	static wxULongLong_t s_nFileId;
	static SQX_ARCLISTNODE * aln;
	static SQX_ARCLIST * al;
	static void * fl;
	if (_bFirst)
	{
		s_nFileId = 0;
		FARPROC fpProc = ::GetAPIAddress("InitArcFileList");
		if (fpProc == NULL)
		{
			return TPI_ERROR_U_USE_LIBRARY;
		}
		al = ((SQX_ARCLIST * (__stdcall *)(void *)) fpProc)(_hArchive);
		if (al == NULL)
		{
			return TPI_ERROR_UNDEFINED;
		}

		fpProc = ::GetAPIAddress("InitFileList");
		if (fpProc == NULL)
		{
			return TPI_ERROR_U_USE_LIBRARY;
		}
		fl = ((void * (__stdcall *)(void *)) fpProc)(_hArchive);
		if (fl == NULL)
		{
			return TPI_ERROR_UNDEFINED;
		}

		fpProc = ::GetAPIAddress("AppendFileList");
		if (fpProc == NULL)
		{
			return TPI_ERROR_U_USE_LIBRARY;
		}
		((void (__stdcall *)(void *, void *, const wchar_t *)) fpProc)(_hArchive, fl, L"*");

		fpProc = ::GetAPIAddress("ListFiles");
		if (fpProc == NULL)
		{
			return TPI_ERROR_U_USE_LIBRARY;
		}

		SQX_ARCINFO ai;
		memset(& ai, 0, sizeof(SQX_ARCINFO));
		ai.cbSize = sizeof(SQX_ARCINFO);
		int nErrorCode = ErrorCodeConvert(((int (__stdcall *)(void *, void *, SQX_ARCLIST *, SQX_ARCINFO *)) fpProc)(_hArchive, fl, al, & ai));
		if (nErrorCode != TPI_ERROR_SUCCESS)
		{
			return nErrorCode;
		}
		aln = al->pHead;
	}

	if (aln == NULL)
	{
		FARPROC fpProc = ::GetAPIAddress("DoneArcFileList");
		if (fpProc == NULL)
		{
			return TPI_ERROR_U_USE_LIBRARY;
		}
		((void (__stdcall *)(void *, SQX_ARCLIST *)) fpProc)(_hArchive, al);

		fpProc = ::GetAPIAddress("DoneFileList");
		if (fpProc == NULL)
		{
			return TPI_ERROR_U_USE_LIBRARY;
		}
		((void (__stdcall *)(void *, void *)) fpProc)(_hArchive, fl);

		return TPI_ERROR_S_ENDOFDATA;
	}

	SQX_ARCNODE * an = aln->pArcNode;
	if (! an->fTagged)
	{
		aln = aln->pNext;
		return GetFileInformation(_hArchive, _fiInfo, false);
	}

	_fiInfo->eOSType        = an->dwHostOS;
	_fiInfo->dwAttribute    = an->dwAttributes;
	if (an->fEncrypted)
	{
		_fiInfo->dwAttribute |= TPI_ATTRIBUTE_ENCRYPTED;
	}
	_fiInfo->dwCRC32        = an->dwFileCRC;
	_fiInfo->nPackedSize    = an->dwlSize;
	_fiInfo->nUnpackedSize  = an->dwlSizeOrig;
	_fiInfo->nFileId        = s_nFileId++;
	if (an->win32FileTime.fBlockPresent)
	{
		_fiInfo->tmCreate   = FileTimeToWxDateTime(& an->win32FileTime.ftCreationTime);
		_fiInfo->tmAccess   = FileTimeToWxDateTime(& an->win32FileTime.ftLastAccessTime);
		_fiInfo->tmModify   = FileTimeToWxDateTime(& an->win32FileTime.ftLastWriteTime);
	}
	else
	{
		_fiInfo->tmModify.SetFromDOS(an->dwDosFileTime);
	}
	_fiInfo->szComment      = WC2String(an->pszComment);
	_fiInfo->szStoredName   = WC2String(an->pszFileName);
	_fiInfo->fnFileName     = wxFileName(_fiInfo->szStoredName);
	_fiInfo->szMethod       =
		an->dwMethod == SQX_METHOD_STORED ? wxT("store") :
		an->dwMethod == SQX_METHOD_NORMAL ? wxT("lzh-1") :
		an->dwMethod == SQX_METHOD_GOOD   ? wxT("lzh-2") :
		an->dwMethod == SQX_METHOD_HIGH   ? wxT("lzh-3") :
		an->dwMethod == SQX_METHOD_BEST   ? wxT("lzh-4") :
		an->dwMethod == SQX_METHOD_AUDIO  ? wxT("audio") :
		an->dwMethod == SQX_METHOD_TEXT   ? wxT("text")  :
		an->dwMethod == SQX_METHOD_ULTRA  ? wxT("ultra") : wxT("unknown");
	aln = aln->pNext;
	return TPI_ERROR_SUCCESS;
}

int __stdcall GetArchiveInformation
(
	void * _hArchive,
	TPI_ARCHIVEINFO * _aiInfo
)
{
	FARPROC fpProc = ::GetAPIAddress("InitArcFileList");
	if (fpProc == NULL)
	{
		return TPI_ERROR_U_USE_LIBRARY;
	}
	SQX_ARCLIST * al = ((SQX_ARCLIST * (__stdcall *)(void *)) fpProc)(_hArchive);
	if (al == NULL)
	{
		return TPI_ERROR_UNDEFINED;
	}

	fpProc = ::GetAPIAddress("InitFileList");
	if (fpProc == NULL)
	{
		return TPI_ERROR_U_USE_LIBRARY;
	}
	void * fl = ((void * (__stdcall *)(void *)) fpProc)(_hArchive);
	if (fl == NULL)
	{
		return TPI_ERROR_UNDEFINED;
	}

	fpProc = ::GetAPIAddress("AppendFileList");
	if (fpProc == NULL)
	{
		return TPI_ERROR_U_USE_LIBRARY;
	}
	((void (__stdcall *)(void *, void *, const wchar_t *)) fpProc)(_hArchive, fl, L"*");

	fpProc = ::GetAPIAddress("ListFiles");
	if (fpProc == NULL)
	{
		return TPI_ERROR_U_USE_LIBRARY;
	}

	SQX_ARCINFO ai;
	memset(& ai, 0, sizeof(SQX_ARCINFO));
	ai.cbSize = sizeof(SQX_ARCINFO);
	int nErrorCode = ErrorCodeConvert(((int (__stdcall *)(void *, void *, SQX_ARCLIST *, SQX_ARCINFO *)) fpProc)(_hArchive, fl, al, & ai));
	if (nErrorCode != TPI_ERROR_SUCCESS)
	{
		return nErrorCode;
	}

	fpProc = ::GetAPIAddress("DoneArcFileList");
	if (fpProc == NULL)
	{
		return TPI_ERROR_U_USE_LIBRARY;
	}
	((void (__stdcall *)(void *, SQX_ARCLIST *)) fpProc)(_hArchive, al);

	fpProc = ::GetAPIAddress("DoneFileList");
	if (fpProc == NULL)
	{
		return TPI_ERROR_U_USE_LIBRARY;
	}
	((void (__stdcall *)(void *, void *)) fpProc)(_hArchive, fl);

	_aiInfo->fSolid         = ai.fSolid           == TRUE;
	_aiInfo->fEncryptHeader = ai.fHeaderEncrypted == TRUE;
	_aiInfo->fEncryptData   = ai.dwEncryption != SQX_ENCRYPTION_NONE;
	_aiInfo->eOSType        = ai.dwHostOS;
	_aiInfo->nRecoveryRecord= ai.fRecoveryData;
	_aiInfo->wCompressRatio = ai.iRatio * 10;
	_aiInfo->nFileSize      = ai.dwlUncompressedSize;
	_aiInfo->nPackedSize    = ai.dwlCompressedSize;
	_aiInfo->nUnpackedSize  = ai.dwlUncompressedSize;
	if (ai.fArchiveComment)
	{
		fpProc = ::GetAPIAddress("GetArchiveComment");
		if (fpProc == NULL)
		{
			return TPI_ERROR_U_USE_LIBRARY;
		}
		wchar_t sz[4097];
		memset(sz, 0, sizeof(sz));
		((void (__stdcall *)(void *, wchar_t *, DWORD)) fpProc)(_hArchive, sz, sizeof(sz) - 1);
		_aiInfo->szComment = WC2String(sz);
	}
	if (ai.avInfo.fAVInfoPresent)
	{
		wxDateTime tm = FileTimeToWxDateTime(& ai.avInfo.ftCreationTime);
		_aiInfo->szComment += wxT("\nSignatured by ") + WC2String(ai.avInfo.szAV_ID) + wxT(" at ") + tm.Format(wxT("%Y/%m/%d %H:%M:%S"));
	}
	GetFormatInformation(& _aiInfo->fiInfo, ai.dwFileFormat == SQX_FILEFORMAT_10);
	return TPI_ERROR_SUCCESS;
}

int __stdcall Command
(
	wxULongLong_t _eCommand,
	TPI_SWITCHES * _swInfo,
	void * _hArchive,
	const wxArrayString & _szFiles
)
{
	int nErrorCode = TPI_ERROR_SUCCESS;
	if (_eCommand == TPI_COMMAND_CREATE)
	{
		nErrorCode = OpenArchive(_swInfo->szArcName, & _hArchive);
		if (nErrorCode != TPI_ERROR_SUCCESS)
		{
			return nErrorCode;
		}
	}

	FARPROC fpProc;
	void * fl = NULL;
	if (_eCommand != TPI_COMMAND_TEST && _eCommand != TPI_COMMAND_REPAIR)
	{
		fpProc = ::GetAPIAddress("InitFileList");
		if (fpProc == NULL)
		{
			if (_eCommand == TPI_COMMAND_CREATE)
			{
				CloseArchive(_hArchive);
			}
			return TPI_ERROR_U_USE_LIBRARY;
		}
		fl = ((void * (__stdcall *)(void *)) fpProc)(_hArchive);
		if (fl == NULL)
		{
			if (_eCommand == TPI_COMMAND_CREATE)
			{
				CloseArchive(_hArchive);
			}
			return TPI_ERROR_UNDEFINED;
		}

		fpProc = ::GetAPIAddress("AppendFileList");
		if (fpProc == NULL)
		{
			if (_eCommand == TPI_COMMAND_CREATE)
			{
				CloseArchive(_hArchive);
			}
			return TPI_ERROR_U_USE_LIBRARY;
		}
		for (size_t i = 0; i < _szFiles.Count(); i++)
		{
			((void (__stdcall *)(void *, void *, const wchar_t *)) fpProc)(_hArchive, fl, _szFiles[i].c_str());
		}
	}

	// グローバル変数にポインタを保存。
	fpProc =
		_eCommand == TPI_COMMAND_CREATE  ? ::GetAPIAddress("CompressFiles") :
//		_eCommand == TPI_COMMAND_ADD     ? ::GetAPIAddress("CompressFiles") :
		_eCommand == TPI_COMMAND_EXTRACT ? ::GetAPIAddress("ExtractFiles")  :
		_eCommand == TPI_COMMAND_DELETE  ? ::GetAPIAddress("DeleteFiles")   :
		_eCommand == TPI_COMMAND_TEST    ? ::GetAPIAddress("TestArchive")   :
		_eCommand == TPI_COMMAND_REPAIR  ? ::GetAPIAddress("RepairArchive") : NULL;
	if (fpProc == NULL)
	{
		if (_eCommand == TPI_COMMAND_CREATE)
		{
			CloseArchive(_hArchive);
		}
		return TPI_ERROR_U_USE_LIBRARY;
	}

	switch (_eCommand)
	{
	case TPI_COMMAND_CREATE:
//	case TPI_COMMAND_ADD:
	{
		SQX_COMPRESSOPTIONS co;
		memset(& co, 0, sizeof(co));
		co.cbSize                       = sizeof(co);
		co.dwFileFormat                 = _swInfo->nArchiveType;
		co.compOptions.dwCompRate       = _swInfo->nCompressLevel;
		co.compOptions.fSolidFlag       = true;
		co.compOptions.dwMultimediaCompression = _swInfo->fMMOptimize;
		co.sfxCommand.fCreateSfx        = _swInfo->fMakeSFX;
		co.dwRecoveryData               = _swInfo->nRecoveryRecord;
		co.pszMainComment               = _swInfo->szComment.c_str();
		co.dwEncryption                 = _swInfo->szPassword.IsEmpty() ? SQX_ENCRYPTION_NONE : SQX_ENCRYPTION_AES_256BIT;
		co.fEncryptHeaders              = _swInfo->fEncryptHeader;
		co.dwFileNames                  = SQX_FILENAME_UNICODE;
		co.fRetainFolderStructure       = _swInfo->fStoreDirectoryPathes;
		co.fStoreACL                    = true;
		co.fStoreStreams                = true;
		co.fStoreWin32FileTime          = true;
		co.fAutoSaveComments            = true;
		co.dwlVolumeSize                = _swInfo->nSplitSize;
		wcsncpy(co.szInputPath, _swInfo->fnDestinationDirectory.GetFullPath().c_str(), sizeof(co.szInputPath) / sizeof(wchar_t) - 1);
		wcsncpy(co.szPassword,  _swInfo->szPassword.c_str(), sizeof(co.szPassword) / sizeof(wchar_t) - 1);
		if (_swInfo->fEncryptHeader)
		{
			wcsncpy(co.szPasswordHeader, _swInfo->szPassword.c_str(), sizeof(co.szPasswordHeader) / sizeof(wchar_t) - 1);
		}
		nErrorCode = ErrorCodeConvert(((int (__stdcall *)(void *, SQXCALLBACK, void *, void *, SQX_COMPRESSOPTIONS *)) fpProc)(_hArchive, CallbackProc, NULL, fl, & co));
		break;
	}
	case TPI_COMMAND_EXTRACT:
	{
		SQX_EXTRACTOPTIONS eo;
		memset(& eo, 0, sizeof(eo));
		eo.cbSize                       = sizeof(eo);
		eo.fCreateFolders               = _swInfo->fStoreDirectoryPathes;
		eo.fKeepBrokenFiles             = true;
		eo.fRestoreACLs                 = true;
		eo.fRestoreStreams              = true;
		eo.fRestoreUnicodeNames         = true;
		eo.fRestoreWin32FileTimes       = true;
		eo.fRestoreDirectoryTimeStamps  = true;
		eo.fAutoRestoreComments         = true;
		eo.fSetZoneID                   = true;
		wcsncpy(eo.szPassword,       _swInfo->szPassword.c_str(), sizeof(eo.szPassword) / sizeof(wchar_t) - 1);
		wcsncpy(eo.szPasswordHeader, _swInfo->szPassword.c_str(), sizeof(eo.szPasswordHeader) / sizeof(wchar_t) - 1);
		wcsncpy(eo.szOutputPath,     _swInfo->fnDestinationDirectory.GetFullPath().c_str(), sizeof(eo.szOutputPath) / sizeof(wchar_t) - 1);
		nErrorCode = ErrorCodeConvert(((int (__stdcall *)(void *, SQXCALLBACK, void *, void *, SQX_EXTRACTOPTIONS *)) fpProc)(_hArchive, CallbackProc, NULL, fl, & eo));
		break;
	}
	case TPI_COMMAND_DELETE:
		nErrorCode = ErrorCodeConvert(((int (__stdcall *)(void *, SQXCALLBACK, void *, void *)) fpProc)(_hArchive, CallbackProc, NULL, fl));
		break;
	case TPI_COMMAND_TEST:
		nErrorCode = ErrorCodeConvert(((int (__stdcall *)(void *, SQXCALLBACK, void *)) fpProc)(_hArchive, CallbackProc, NULL));
		break;
	case TPI_COMMAND_REPAIR:
	{
		wxString sz = _swInfo->szArcName + wxT("_repaired");
		nErrorCode = ErrorCodeConvert(((int (__stdcall *)(void *, const wchar_t *, SQX_ARCLIST *, SQX_ARCLIST *)) fpProc)(_hArchive, sz.c_str(), NULL, NULL));
		break;
	}
	}

	if (nErrorCode != TPI_ERROR_SUCCESS)
	{
		if (_eCommand == TPI_COMMAND_CREATE)
		{
			CloseArchive(_hArchive);
		}
		return nErrorCode;
	}

	if (_eCommand != TPI_COMMAND_TEST && _eCommand != TPI_COMMAND_REPAIR)
	{
		fpProc = ::GetAPIAddress("DoneFileList");
		if (fpProc == NULL)
		{
			if (_eCommand == TPI_COMMAND_CREATE)
			{
				CloseArchive(_hArchive);
			}
			return TPI_ERROR_U_USE_LIBRARY;
		}
		((void (__stdcall *)(void *, void *)) fpProc)(_hArchive, fl);
	}

	return _eCommand == TPI_COMMAND_CREATE ? CloseArchive(_hArchive) : TPI_ERROR_SUCCESS;
}

#ifdef __cplusplus
}
#endif
