package jp.snowgoose.treno.component;

import java.lang.annotation.Annotation;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import jp.snowgoose.treno.util.ClassCollector;
import jp.snowgoose.treno.util.ReflectionUtils;
import jp.snowgoose.treno.util.ResourceUtils;

/**
 * @author snowgoose
 */
public class Scanner {

    private Collection<ClassCollector> classCollectors;
    private InstanceProvider instanceProvider;

    public Scanner(Collection<ClassCollector> classCollectors, InstanceProvider instanceProvider) {
        this(classCollectors);
        this.instanceProvider = instanceProvider;
    }

    public Scanner(Collection<ClassCollector> classCollectors) {
        this.classCollectors = classCollectors;
    }

    public Scanned scan(Collection<String> packageNames) {
        return scan(packageNames.toArray(new String[packageNames.size()]));
    }

    public Scanned scan(String[] packageNames) {
        Collection<Class<?>> collectedClasses = new ArrayList<Class<?>>();
        for (String packageName : packageNames) {
            for (URL resourceURL : getPackageResources(packageName)) {
                for (ClassCollector collector : classCollectors) {
                    collectedClasses.addAll(collector.collect(packageName, resourceURL));
                }
            }
        }
        return new Scanned(collectedClasses, instanceProvider);
    }

    private List<URL> getPackageResources(String packageName) {
        return ResourceUtils.findResources(packageName.replace('.', '/'));
    }

    public static class Scanned {
        private Collection<Class<?>> collectedClasses;
        private InstanceProvider instanceProvider;

        protected Scanned(Collection<Class<?>> collectedClasses, InstanceProvider instanceProvider) {
            this(collectedClasses);
            this.instanceProvider = instanceProvider;
        }

        protected Scanned(Collection<Class<?>> collectedClasses) {
            this.collectedClasses = collectedClasses;
        }

        @SuppressWarnings("unchecked")
        public <T> Collection<T> getInstances(Condition<T> condition) {
            List<T> result = new ArrayList<T>();
            for (Class<?> clazz : this.collectedClasses) {
                if (condition.filter((Class<T>) clazz)) {
                    T instance = (T) instanceProvider.resolveInstance(clazz);
                    if (instance != null) {
                        result.add(instance);
                    }
                }
            }
            return result;
        }

        @SuppressWarnings("unchecked")
        public <T extends UniqueAddon> T getInstance(Condition<T> condition, String id) {
            for (Class<?> clazz : collectedClasses) {
                if (condition.filter(clazz)
                        && ReflectionUtils.isImplements(clazz, UniqueAddon.class)
                        && clazz.isInterface() == false) {
                    T unique = (T) instanceProvider.resolveInstance(clazz);
                    if (unique.getUniqueId().equals(id)) {
                        return unique;
                    }
                }
            }
            return null;
        }

        public <T> Collection<Class<?>> getClasses(Condition<T> condition) {
            List<Class<?>> result = new ArrayList<Class<?>>();
            for (Class<?> clazz : collectedClasses) {
                if (condition.filter(clazz)) {
                    result.add(clazz);
                }
            }
            return result;
        }

        public interface Condition<T> {
            boolean filter(Class<?> clazz);
        }

        public static class Conditions {
            public static <T extends Annotation> Condition<T> annotateWith(Class<T> annotateWith) {
                return new AnnotateWith<T>(annotateWith);
            }

            public static <T> Condition<T> implementsInterface(Class<T> clazz) {
                return new Implements<T>(clazz);
            }

            public static <T> Condition<T> extendsClass(Class<T> clazz) {
                return new ExtendsClass<T>(clazz);
            }
        }

        public static class AnnotateWith<T extends Annotation> implements Condition<T> {
            private Class<T> annotateWith;

            public AnnotateWith(Class<T> annotateClass) {
                this.annotateWith = annotateClass;
            }

            public boolean filter(Class<?> clazz) {
                return ReflectionUtils.annotateWith(annotateWith, clazz);
            }
        }

        public static class Implements<T extends Object> implements Condition<T> {
            private Class<T> implementsIf;

            public Implements(Class<T> implementsIf) {
                this.implementsIf = implementsIf;
            }

            public boolean filter(Class<?> clazz) {
                return ReflectionUtils.isImplements(clazz, implementsIf);
            }

        }

        public static class ExtendsClass<T extends Object> implements Condition<T> {
            private Class<T> extendsIf;

            public ExtendsClass(Class<T> extendsIf) {
                this.extendsIf = extendsIf;
            }

            public boolean filter(Class<?> clazz) {
                Class<?> superClass = clazz.getSuperclass();
                if (superClass == null || superClass.equals(Object.class)) {
                    return false;
                } else if (superClass.equals(extendsIf)) {
                    return true;
                } else {
                    return filter(superClass);
                }
            }

        }

    }

}
