package jp.snowgoose.treno.context;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;

import jp.snowgoose.treno.component.ActionInstance;
import jp.snowgoose.treno.metadata.BindDescriptor;
import jp.snowgoose.treno.metadata.BindDescriptors;
import jp.snowgoose.treno.metadata.Scope;
import jp.snowgoose.treno.metadata.BindDescriptor.BindElementType.BindType;
import jp.snowgoose.treno.metadata.BindDescriptor.BindElementType.FieldBindElement;
import jp.snowgoose.treno.metadata.BindDescriptor.BindElementType.MethodBindElement;
import jp.snowgoose.treno.util.ReflectionUtils;
import jp.snowgoose.treno.util.ThreadLocalMap;

/**
 * @author snowgoose
 */
public class SimpleValueMapper extends AbstractRequestValueMapper {

    private Map<Object, Object> responseMap = new ThreadLocalMap<Object, Object>();

    public SimpleValueMapper(RequestContext request, ParameterConverters converters) {
        super(request, converters);
    }

    public boolean requestSupported(HttpServletRequest request) {
        return false;
    }

    public Object get(Scope scope, String name) {
        if (Scope.PARAMETER == scope) {
            return getRequest().getParameter(name);
        } else if (Scope.REQUEST == scope) {
            Object value = getRequest().getParameter(name);
            if (value == null) {
                return getRequest().getAttribute(name);
            } else {
                return value;
            }
        } else if (Scope.SESSION == scope) {
            HttpSession session = getRequest().getSession(false);
            if (session == null) {
                return null;
            }
            return session.getAttribute(name);
        } else if (Scope.RESPONSE == scope) {
            return responseMap.get(name);
        }
        return null;
    }

    public Map<?, ?> getAll(Scope scope) {
        if (scope == Scope.RESPONSE) {
            return (Map<?, ?>) responseMap;
        } else {
            return null;
        }
    }

    public void put(String name, Object value, Scope scope) {
        if (Scope.REQUEST == scope) {
            getRequest().setAttribute(name, value);
        } else if (Scope.SESSION == scope) {
            getRequest().getSession(true).setAttribute(name, value);
        } else if (Scope.RESPONSE == scope) {
            responseMap.put(name, value);
        }
    }

    public void map(ActionInstance instance) {
        List<Object> args = new ArrayList<Object>();
        BindDescriptors bindDescriptors = instance.getActionDescriptor().getBindDescriptors();
        for (BindDescriptor bindDescriptor : bindDescriptors.getAll()) {
            ParameterConverter converter = getParameterConverters().getParameterConverter(
                    bindDescriptor.getConverterType());
            Object rowValue = get(bindDescriptor.getScope(), bindDescriptor.getName());
            Object value = converter.convert(rowValue, bindDescriptor);
            if (bindDescriptor.bindTypeIs(BindType.METHOD)) {
                bindToMethod(instance.getInstance(), bindDescriptor, value);
            } else if (bindDescriptor.bindTypeIs(BindType.FIELD)) {
                bindToField(instance.getInstance(), bindDescriptor, value);
            } else if (bindDescriptor.bindTypeIs(BindType.PARAMETER)) {
                args.add(converter.convert(
                        get(bindDescriptor.getScope(), bindDescriptor.getName()), bindDescriptor));
            }
        }
        mapArgs(instance);
    }

    private void bindToField(Object instance, BindDescriptor bindDescriptor, Object value) {
        FieldBindElement element = (FieldBindElement) bindDescriptor.getBindElementType();
        ReflectionUtils.setValueToField(element.getFieldName(), element.getModefier(), instance,
                value);
    }

    private void bindToMethod(Object instance, BindDescriptor bindDescriptor, Object value) {
        MethodBindElement element = (MethodBindElement) bindDescriptor.getBindElementType();
        ReflectionUtils.invokeMethodSilently(instance, element.getMethodName(),
                getArgTypes(bindDescriptor), value);
    }

    private Class<?>[] getArgTypes(BindDescriptor bindDescriptor) {
        return new Class<?>[] { bindDescriptor.getParameterType() };
    }

    public void unmap(ActionInstance instance) {
        BindDescriptors bindDescriptors = instance.getActionDescriptor().getBindDescriptors();
        for (BindDescriptor bindDescriptor : bindDescriptors.getAll()) {
            if (bindDescriptor.bindTypeIs(BindType.METHOD)) {
                Object value = getValueViaMethod(instance.getInstance(),
                        (MethodBindElement) bindDescriptor.getBindElementType());
                put(bindDescriptor.getName(), value, bindDescriptor.getScope());
            } else if (bindDescriptor.bindTypeIs(BindType.TYPE)) {
                put(bindDescriptor.getName(), instance.getInstance(), bindDescriptor.getScope());
            } else if (bindDescriptor.bindTypeIs(BindType.FIELD)) {
                Object value = getValueViaField(instance.getInstance(),
                        (FieldBindElement) bindDescriptor.getBindElementType());
                put(bindDescriptor.getName(), value, bindDescriptor.getScope());
            }
        }
    }

    private Object getValueViaField(Object invokedInstance, FieldBindElement elementType) {
        return ReflectionUtils.getValueViaField(elementType.getFieldName(), elementType
                .getModefier(), invokedInstance);
    }

    private Object getValueViaMethod(Object invokedInstance, MethodBindElement bindDescriptor) {
        return ReflectionUtils.invokeMethodSilently(invokedInstance,
                bindDescriptor.getMethodName(), new Class<?>[0], new Object[0]);
    }

    private void mapArgs(ActionInstance instance) {
        Collection<BindDescriptor> parameterDescriptors = instance.getActionDescriptor()
                .getBindDescriptors().getParameterBindDescriptors();
        if (parameterDescriptors.isEmpty() == false) {
            List<Object> args = new ArrayList<Object>();
            ParameterConverters converters = getParameterConverters();
            for (BindDescriptor bindDesc : parameterDescriptors) {
                ParameterConverter converter = converters.getParameterConverter(bindDesc
                        .getConverterType());
                Object argValue = get(bindDesc.getScope(), bindDesc.getName());
                args.add(converter.convert(argValue, bindDesc));
            }
            instance.setPreparedArgs(args);
        }
    }

}
