package jp.snowgoose.treno.metadata;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import jp.snowgoose.treno.exception.ActionNotFoundException;
import jp.snowgoose.treno.util.Maps;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author snowgoose
 */
public class ActionDescriptors {

    private static final Logger log = LoggerFactory.getLogger(ActionDescriptors.class);

    private final Map<MappedPath, ActionDescriptor> actionDescriptorsMap = Maps.newLinkedHashMap();

    public ActionDescriptors() {
        super();
    }

    public ActionDescriptors(List<String> packageNames, ActionDescriptorFactory factory,
            Collection<Class<?>> actionClasses) {
        init(packageNames, factory, actionClasses);
    }

    protected void init(List<String> packageNames, ActionDescriptorFactory factory,
            Collection<Class<?>> actionClasses) {
        putToActionDescriptorsMap(actionClasses, factory);
    }

    private void putToActionDescriptorsMap(Collection<Class<?>> collectedClasses,
            ActionDescriptorFactory factory) {
        for (Class<?> clazz : collectedClasses) {
            for (ActionDescriptor actionDesc : factory.createActionDescriptors(clazz)) {
                put(actionDesc);
            }
        }
    }

    public ActionDescriptor getActionDescriptor(MappedPath mappedPath)
            throws ActionNotFoundException {
        if (actionDescriptorsMap.containsKey(mappedPath) == false) {
            throw new ActionNotFoundException(mappedPath);
        }
        return this.actionDescriptorsMap.get(mappedPath);
    }

    protected Map<MappedPath, ActionDescriptor> getActionDescriptorsMap() {
        return this.actionDescriptorsMap;
    }

    public void put(ActionDescriptor actionDescriptor) {
        if (this.actionDescriptorsMap.containsKey(actionDescriptor.getMappedPath())) {
            log.warn("action registration ignored.");
        } else {
            this.actionDescriptorsMap.put(actionDescriptor.getMappedPath(), actionDescriptor);
        }
    }

    public int size() {
        return this.actionDescriptorsMap.keySet().size();
    }

}
