package jp.snowgoose.treno.metadata;

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import jp.snowgoose.treno.annotation.Direction;
import jp.snowgoose.treno.annotation.InvokeAction;
import jp.snowgoose.treno.annotation.InvokeActions;
import jp.snowgoose.treno.annotation.BindValue;
import jp.snowgoose.treno.metadata.ActionDescriptor.ActionMethod;
import jp.snowgoose.treno.metadata.BindDescriptor.BindElementType;
import jp.snowgoose.treno.metadata.BindDescriptor.BindElementType.ParameterBindElement;
import jp.snowgoose.treno.util.Assertion;
import jp.snowgoose.treno.util.Maps;
import jp.snowgoose.treno.util.ReflectionUtils;
import jp.snowgoose.treno.util.StringUtils;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author snowgoose
 */
public class AnnotationActionDescriptorFactory implements ActionDescriptorFactory {

    private static final Logger log = LoggerFactory
            .getLogger(AnnotationActionDescriptorFactory.class);
    private final BindDescriptorFactory bindFactory = new AnnotationBindDescriptorFactory();
    private final ResultDescriptorFactory resultFactory = new AnnotationResultDescriptorFactory();

    public String getUniqueId() {
        return "annotation";
    }

    public Collection<ActionDescriptor> createActionDescriptors(Class<?> clazz) {
        Assertion.notNull(clazz, "ActionClass");
        InvokeActions actions;
        if ((actions = findAction(clazz)) != null) {
            return findActionDescriptors(clazz, actions);
        } else {
            log.debug(String.format("action not found at class %s", clazz));
            return Collections.emptyList();
        }
    }

    private List<ActionDescriptor> findActionDescriptors(Class<?> clazz, InvokeActions actions) {
        Method[] methods = clazz.getDeclaredMethods();
        List<ActionDescriptor> result = new ArrayList<ActionDescriptor>();
        for (Method method : methods) {
            InvokeAction action = method.getAnnotation(InvokeAction.class);
            if (action != null) { // is action.
                result.add(createActionDescriptor(clazz, method, actions, action));
            }
        }
        return result;
    }

    private ActionDescriptor createActionDescriptor(Class<?> clazz, Method method,
            InvokeActions actions, InvokeAction action) {
        MappedPath mappedPath = new MappedPath(actions.value(), action.path());
        ActionMethod actionMethod = new ActionMethod();
        actionMethod.actionClass = clazz;
        actionMethod.argumentTypes = method.getParameterTypes();
        actionMethod.methodName = method.getName();
        ActionDescriptor actionDesc = new ActionDescriptor();
        actionDesc.actionMethod = actionMethod;
        actionDesc.bindDescriptors = bindFactory.createBindDescriptors(clazz, method);
        actionDesc.mappedPath = mappedPath;
        actionDesc.resultDescriptors = resultFactory.createResultDescriptors(clazz, action);
        log.info(String.format("action created on %s", mappedPath.getPath()));
        return actionDesc;
    }

    private InvokeActions findAction(Class<?> clazz) {
        return clazz.getAnnotation(InvokeActions.class);
    }

    public static class AnnotationBindDescriptorFactory implements BindDescriptorFactory {

        public BindDescriptors createBindDescriptors(Class<?> clazz, Method actionMethod) {
            BindDescriptors bindDescriptors = new BindDescriptors();
            bindDescriptors.put(createClassBindDescriptor(clazz));
            bindDescriptors.putAll(createMethodBindDescriptors(clazz));
            bindDescriptors.putAll(createFieldBindDescriptors(clazz));
            for (int i = 0, length = actionMethod.getParameterTypes().length; i < length; i++) {
                bindDescriptors.put(createParameterBindDescriptor(actionMethod, i));
            }
            return bindDescriptors;
        }

        private BindDescriptor createClassBindDescriptor(Class<?> clazz) {
            BindValue classBind = clazz.getAnnotation(BindValue.class);
            if (classBind != null) {
                BindDescriptor bindDesc = new BindDescriptor();
                String name = classBind.value();
                if (StringUtils.isEmpty(name)) {
                    name = StringUtils.decapitalize(clazz.getSimpleName());
                }
                bindDesc.name = name;
                bindDesc.bindElementType = new BindElementType.TypeBindElement(clazz);
                bindDesc.parameterType = clazz;
                bindDesc.scope = classBind.scope();
                return bindDesc;
            }
            return null;
        }

        private Collection<BindDescriptor> createFieldBindDescriptors(Class<?> clazz) {
            // TODO : implement
            List<BindDescriptor> bindDescriptors = new ArrayList<BindDescriptor>();
            for (Field field : clazz.getDeclaredFields()) {
                field.setAccessible(true);
                BindValue bind = field.getAnnotation(BindValue.class);
                if (bind != null) {
                    BindDescriptor bindDesc = new BindDescriptor();
                    String name = bind.value();
                    if (StringUtils.isEmpty(name)) {
                        name = field.getName();
                    }
                    bindDesc.name = name;
                    bindDesc.bindElementType = new BindElementType.FieldBindElement(field
                            .getModifiers(), field.getName());
                    bindDesc.parameterType = field.getType();
                    bindDesc.scope = bind.scope();
                    bindDesc.format = bind.format();
                    bindDesc.converterType = bind.converter();
                    bindDescriptors.add(bindDesc);
                }
            }
            return bindDescriptors;
        }

        private Collection<BindDescriptor> createMethodBindDescriptors(Class<?> clazz) {
            List<BindDescriptor> bindDescriptors = new ArrayList<BindDescriptor>();
            for (Method method : clazz.getDeclaredMethods()) {
                if (method.getName().startsWith("get")) { // TODO
                    BindValue bind = method.getAnnotation(BindValue.class);
                    if (bind != null && method.getAnnotation(InvokeAction.class) == null) {
                        BindDescriptor bindDesc = new BindDescriptor();
                        String name = bind.value();
                        if (StringUtils.isEmpty(name)) {
                            name = removeGet(method.getName());
                        }
                        bindDesc.name = name;
                        bindDesc.bindElementType = new BindElementType.MethodBindElement(method
                                .getName());
                        bindDesc.parameterType = method.getReturnType();
                        bindDesc.scope = bind.scope();
                        bindDesc.format = bind.format();
                        bindDesc.converterType = bind.converter();
                        bindDescriptors.add(bindDesc);
                    }
                }
            }
            return bindDescriptors;
        }

        private String removeGet(String methodName) {
            if (methodName.startsWith("get")) {
                return methodName.substring(3, methodName.length())
                        .toLowerCase(Locale.getDefault());
            }
            return methodName;
        }

        private BindDescriptor createParameterBindDescriptor(Method method, int parameterIndex) {
            BindValue bind = ReflectionUtils.getMethodParameterAnnotation(method,
                    BindValue.class, parameterIndex);
            if (bind != null) {
                BindDescriptor bindDesc = new BindDescriptor();
                bindDesc.bindElementType = new ParameterBindElement(parameterIndex);
                bindDesc.name = bind.value();
                bindDesc.scope = bind.scope();
                Class<?> type = method.getParameterTypes()[parameterIndex];
                if (type.isPrimitive()) {
                    bindDesc.parameterType = wrapPrimitive(type);
                } else {
                    bindDesc.parameterType = type;
                }
                bindDesc.format = bind.format();
                bindDesc.converterType = bind.converter();
                return bindDesc;
            }
            return null;
        }

        private static final Map<Class<?>, Class<?>> primitiveWrapMap;

        static {
            primitiveWrapMap = Maps.newLinkedHashMap();
            primitiveWrapMap.put(Boolean.TYPE, Boolean.class);
            primitiveWrapMap.put(Byte.TYPE, Byte.class);
            primitiveWrapMap.put(Character.TYPE, Character.class);
            primitiveWrapMap.put(Short.TYPE, Short.class);
            primitiveWrapMap.put(Float.TYPE, Float.class);
            primitiveWrapMap.put(Double.TYPE, Double.class);
            primitiveWrapMap.put(Integer.TYPE, Integer.class);
            primitiveWrapMap.put(Long.TYPE, Long.class);
        }

        private Class<?> wrapPrimitive(Class<?> type) {
            Class<?> result;
            if ((result = primitiveWrapMap.get(type)) != null) {
                return result;
            } else {
                return type;
            }
        }
    }

    public static class AnnotationResultDescriptorFactory implements ResultDescriptorFactory {

        public ResultDescriptors createResultDescriptors(Class<?> clazz, InvokeAction action) {
            ResultDescriptors resultDescs = new ResultDescriptors();
            Direction[] results = action.directions();
            for (Direction result : results) {
                ResultDescriptor resultDesc = new ResultDescriptor();
                resultDesc.resultType = result.type();
                resultDesc.to = result.to();
                resultDesc.when = result.when();
                resultDescs.put(resultDesc);
            }
            return resultDescs;
        }

    }

}
