package jp.snowgoose.treno.util;

import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.security.AccessController;
import java.security.PrivilegedAction;

import jp.snowgoose.treno.metadata.ActionDescriptor.ActionMethod;

/**
 * @author snowgoose
 */
public class ReflectionUtils {

    @SuppressWarnings("unchecked")
    public static <T extends Annotation> T getMethodParameterAnnotation(Method method,
            Class<T> annotationClass, int parameterIndex) {
        for (Annotation annotation : method.getParameterAnnotations()[parameterIndex]) {
            if (annotation.annotationType().equals(annotationClass)) {
                return (T) annotation;
            }
        }
        return null;
    }

    public static Object invokeMethodSilently(Object instance, String methodName,
            Class<?>[] argTypes, Object... args) {
        try {
            return invokeMethod(instance, methodName, argTypes, args);
        } catch (IllegalAccessException e) {
            return null;
        } catch (IllegalArgumentException e) {
            return null;
        } catch (SecurityException e) {
            return null;
        } catch (InvocationTargetException e) {
            return null;
        } catch (NoSuchMethodException e) {
            return null;
        }
    }

    public static Object invokeMethod(Object instance, String methodName, Class<?>[] argTypes,
            Object... args) throws IllegalArgumentException, IllegalAccessException,
            InvocationTargetException, SecurityException, NoSuchMethodException {
        Method method = instance.getClass().getMethod(methodName, argTypes);
        return method.invoke(instance, args);
    }

    public static Object invokeMethod(Object instance, ActionMethod actionMethod, Object... args)
            throws IllegalArgumentException, IllegalAccessException, InvocationTargetException,
            SecurityException, NoSuchMethodException {
        return invokeMethod(instance, actionMethod.getMethodName(),
                actionMethod.getArgumentTypes(), args);
    }

    public static <T> T getInstanceSilently(Class<T> clazz) {
        try {
            return clazz.newInstance();
        } catch (InstantiationException e) {
            return null;
        } catch (IllegalAccessException e) {
            return null;
        }
    }

    public static synchronized boolean isImplements(Class<?> obj, Class<?> clazz) {
        if (obj.equals(clazz)) {
            return true;
        }
        for (Class<?> implIf : obj.getInterfaces()) {
            Class<?>[] ifaces;
            if ((ifaces = implIf.getInterfaces()) != null && ifaces.length != 0) {
                for (Class<?> iface : ifaces) {
                    if (isImplements(iface, clazz)) {
                        return true;
                    }
                }
            }
            if (implIf.equals(clazz)) {
                return true;
            }
        }
        return false;
    }

    public static boolean annotateWith(Class<? extends Annotation> annotationClass, Class<?> clazz) {
        return (clazz.getAnnotation(annotationClass) != null);
    }

    public static boolean setValueToField(String fieldName, final int modifier, Object instance,
            Object value) {
        try {
            final Field field = instance.getClass().getDeclaredField(fieldName);
            AccessController.doPrivileged(new PrivilegedAction<Object>() {
                public Object run() {
                    if (Modifier.isPrivate(modifier)) {
                        field.setAccessible(true);
                    }
                    return null;
                }
            });
            field.set(instance, value);
            return true;
        } catch (SecurityException e) {
            return false;
        } catch (NoSuchFieldException e) {
            return false;
        } catch (IllegalArgumentException e) {
            return false;
        } catch (IllegalAccessException e) {
            return false;
        }
    }

    public static Object getValueViaField(String fieldName, final int modifier, Object instance) {
        final Field field;
        try {
            field = instance.getClass().getDeclaredField(fieldName);
            AccessController.doPrivileged(new PrivilegedAction<Object>() {
                public Object run() {
                    if (Modifier.isPrivate(modifier)) {
                        field.setAccessible(true);
                    }
                    return null;
                }
            });
            return field.get(instance);
        } catch (SecurityException e) {
            return null;
        } catch (NoSuchFieldException e) {
            return null;
        } catch (IllegalArgumentException e) {
            return null;
        } catch (IllegalAccessException e) {
            return null;
        }
    }

}
