/**
 * \file demodulator.h
 * \date 2015/03/10
 * \brief 復調器関連インクルードファイル
 */

#ifndef DEMODULATOR_H_
#define DEMODULATOR_H_


#ifndef _MACRO_ONLY

    // Following mode macro can be used only the demodulator

    /** \brief 受信機はワイドFMモードである */
#define radio_mode_WFM  2
    /** \brief 受信機はナローFMモードである */
#define radio_mode_NFM  4
    /** \brief 受信機は振幅変調モードである */
#define radio_mode_AM   5
    /** \brief 受信機は振幅変調モードの同期復調モードである */
#define radio_mode_SAM  6
    /** \brief 受信機はLSB復調モードである */
#define radio_mode_LSB  7
    /** \brief 受信機はUSB復調モードである */
#define radio_mode_USB  8
    /** \brief 受信機はCW復調モードである */
#define radio_mode_CW   9


/**
 * \defgroup radioAPI ラジオAPI
 * \brief 復調器がSHマイコン/FPGAと情報の授受を行うためのAPI群
 * \details
 * このモジュールに含まれるAPIは、復調器の中でユーザー復調アルゴリズムが使うために
 * 公開されている。制御情報取得用APIはTRX-305からの制御情報を受け取るためのものである。
 *
 * 制御情報は周期的に送られてくるため、必ずしも即座に反応する必要はない。具体的には
 * TRX-305からは16種類の情報が16周期で送られてくる。これらは情報の中身に変更がない場合も
 * かわらない。
 */
/*@{*/

/**
 * \defgroup getInfo 制御情報取得API
 * \brief 復調器がSHマイコン/FPGAから受け取る制御情報取得用のAPI
 * \details
 * ここに列挙されるAPIは、いずれも復調器内のユーザーアルゴリズムから自由に使ってよい。
 *
 * TRX-305から送られてくる制御情報は、ほとんどの場合TRX-305A基板にシリアルポートから
 * 送られてくるコマンドに対応している。コマンドの一覧に関しては、トランジスタ技術誌
 * 2014年11月号208ページの表を参照。なお、同記事のコマンドはTRX-305AのSHマイコンで
 * 解釈後にDSPに送られるため、ここで使える命令とは必ずしも一対一対応していない。
 *
 * TRX-305Aが受け取るシリアル命令は、SHマイコンにより解釈されたあとにFPGA内部に
 * 貯めこまれ、繰り返し送り出される。DSPはこの命令を受信し、制御情報として内部に
 * さらに蓄える。
 *
 * なお、これらのAPIが返すのは、あくまで解釈のない「マイコンから送られてきた値」である。
 * その値の物理的な解釈については、本プロジェクトは最小限に留める。言い換えると、
 * ここに書いていない解釈は行わない。
 *
 * 命令に関してはすべて実際に解析したものであり、参考とした文献は存在しない。
 * 命令の解析は framework.h の DEBUG_COMMAND のコメントアウトを外すことによって、
 * DSPからのシリアル出力を監視しながらSHにシリアルコマンドを投入して行う。
 *
 *
 */
/*@{*/


    // These APIs are published for the demodulator programmer
    /**
     * \def radio_api_getARM_mode
     * \brief 受信モードの取得
     * \details
     * このAPIは現在の受信モード（復調モード）を返す。シリアルコマンドARMに対応する。
     * 復調モードは整数である。
     * ユーザーに公開されているAPIであるが、2つの理由から使用を推奨しない。
     *
     * まず、復調器hののモード管理は radio_demodulate_wide_FM() および
     * radio_demodulate_wide_FM() がフレームワークと強調する管理で行っており、
     * プログラマは復調器がどのモードにあるかひと目でわかるようになっている。
     * そのため、モード管理にこの関数を呼ぶ必要はない。
     *
     * また、USBとLSBの復調を概ね共用する場合も、radio_api_is_USB() を呼べば
     * 事が足りる。したがって、この関数を呼ぶ必要はない。
     *
     * どうしてもこのAPIを呼び出すときには、 radio_mode_WFM() をはじめとする専用
     * 定数と比較してモードを調べること。SHマイコンは外部のシリアルから与えられた
     * パラメータと違う値を送ってくるため、この注意が必要となる。
     */
#define radio_api_getARM_mode() (radio.comdata[1])

    /**
     * \brief IFシフト量の取得
     * \details
     * シリアルコマンドのAIS命令に対応する。
     * 16bitの符号付き整数でシフト量を返す。
     */
#define radio_api_getAIS_if_shift() (signed short)(radio.comdata[2])

    /**
     * \brief AGCアタック値の取得
     * \details
     * シリアルコマンドのAAT命令に対応する。
     * 16bitの符号無し整数でアタック値を返す。
     */
#define radio_api_getAAT_attack() (radio.comdata[3])

    /**
     * \brief AGCフェード値の取得
     * \details
     * シリアルコマンドのAAT命令に対応する。
     * 16bitの符号無し整数でフェード値を返す。
     */
#define radio_api_getAFA_fade() (radio.comdata[4])

    /**
     * \brief AGCフェード値の取得
     * \details
     * シリアルコマンドのAAT命令に対応する。
     * 16bitの符号無し整数でフェード値を返す。
     */
#define radio_api_getAST_noise_squelch_level() (radio.comdata[5])

    /**
     * \brief 再生トーンピッチの取得
     * \details
     * シリアルコマンドのACP命令に対応する。
     *
     * SHマイコンはシリアルコマンド ACP hhhh (hは16進数)に対して、
     * hhhhを十進数xに変換し x*16916/4096 の値をDSPに送り込む。
     * このAPIが返すのは送り込まれてきた変換後の値である。
     *
     * 16bitの符号付き整数でピッチ値を返す。
     */
#define radio_api_getACP_cw_pitch()  (signed short)(radio.comdata[6])

    /**
     * \brief デノイザー値の取得
     * \details
     * シリアルコマンドADN命令に対応する。
     * 16bitの符号無し整数でデノイザー値を返す。
     */
#define radio_api_getADN_denoiser() (radio.comdata[7])

/**
     * \brief オートノッチ値の取得
     * \details
     * シリアルコマンドAAN命令に対応する。
     * 16bitの符号無し整数でオートノッチ値を返す。
     */
#define radio_api_getAAN_auto_notch() (radio.comdata[8])

/**
     * \brief ゲインレベル値の取得
     * \details
     * シリアルコマンドAGL命令に対応する。
     * 16bitの符号無し整数でゲインレベル値を返す。
     */
#define radio_api_getAGL_gain_level() (radio.comdata[11])
// Flag 0
/**
     * \brief IFフィルタ値の取得
     * \details
     * シリアルコマンドAIF命令に対応する。
     * 16bitの符号無し整数でフィルタ情報を返す。値の範囲は0から6である。
     *
     * （この情報がDSPへの命令なのか通知情報なのかは、現時点では不明である）
     */
#define radio_api_getAIF_filter() ((radio.comdata[0]>>1)&0x7)

/**
     * \brief ノイズブランカ制御命令の取得
     * \details
     * シリアルコマンドABN命令に対応する。
     * 16bitの符号無し整数でノイズブランカ制御値値を返す。
     * \li 0 : オン
     * \li 1 : オフ
     */
#define radio_api_getABN_noise_blanker_on() ((~radio.comdata[0]>>11)&0x1)


/**
     * \brief AGC制御情報の取得
     * \details
     * シリアルコマンドARG命令に対応する。
     * 16bitの符号無し整数でAGC制御情報値を返す。
     *
     * \li 0 : オン
     * \li 1 : オフ
     */
#define radio_api_getARG_agc_on() (~(radio.comdata[0]>>13)&0x1)

/**
     * \brief 復調モード情報の取得
     * \details
     * SSB復調時に必要に応じて参照する。。
     * 16bitの符号無し整数でSSBのモードを返す。
     *
     * \li 0 : LSB
     * \li 非0 : USB
     */
#define radio_api_is_USB()  ((radio.comdata[0]>>6)&0x1)     // 1 if USB

// Flag10
/**
     * \brief ノイズスケルチ制御情報の取得
     * \details
     * シリアルコマンドNE命令に対応する。
     * 16bitの符号無し整数でスケルチ制御情報値を返す。
     *
     * \li 1 : オン
     * \li 0 : オフ
     *
     * なお、以上の設定はトランジスタ技術誌の2014年11月号の表に基づくが、
     * この表には混乱があり、反転している可能性もある。
     */
#define radio_api_getANE_noise_squelch_on() ((radio.comdata[10]>>1)&0x1)


/**
     * \brief オーディオHPF情報の取得
     * \details
     * シリアルコマンドAAF nm命令のnに対応する。
     * 16bitの符号無し整数でオーディオHPF制御情報値を返す。
     *
     * 値の範囲は0-2
     */
#define radio_api_getAAF_hpf() ((radio.comdata[10]>>4)&0x3)

/**
     * \brief オーディオLPF情報の取得
     * \details
     * シリアルコマンドAAF nm命令のmに対応する。
     * 16bitの符号無し整数でオーディオLPF制御情報値を返す。
     *
     * 値の範囲は0-2
     */
#define radio_api_getAAF_lpf() ((radio.comdata[10]>>2)&0x3)

// Flag 12
/**
     * \brief ボイススケルチ値の取得
     * \details
     * シリアルコマンドAVP命令に対応する。
     * 16bitの符号無し整数でボイススケルチ値を返す。
     * 値の範囲は0-127
     *
     * シリアルコマンドAVPの引数は0000-FFFF (65556)までの値をとるが、
     * DSPに渡される値はbit7がAVSによって破壊される。したがって、
     * AVPコマンドに渡す値は 0000から 007Fに制限すべきである。
     */
#define radio_api_getAVP_voice_squelch_level() ((radio.comdata[12])&0x3F)

/**
     * \brief ボイススケルチ情報の取得
     * \details
     * シリアルコマンドAVS命令に対応する。
     * 16bitの符号無し整数でボイススケルチ制御情報値を返す。
     *
     * \li 0 : オン
     * \li 1 : オフ
     */

#define radio_api_getAVS_voice_squelch_on() ((~radio.comdata[12]>>7)&0x1)
/*@}*/
/*  end of defgroup getInfo */

/**
 * \defgroup setStat 状態報告API
 * \brief DSPがFPGA/SHマイコンに情報を送るためのAPI群
 */
/*@{*/

/**
 * \def radio_api_set_S_meter
 * \brief Sメータ値のセット
 * \details
 * このAPIはDSPからFPGA/SHマイコンに渡すSメータ値を設定する。値の範囲は
 * 15bit符号なし整数である。APIは受け取った値をそのままFPGAに送信する。
 */
#define radio_api_set_S_meter(s) radio.smeter = s

	/**
	 * \def radio_api_set_stereo
	 * \brief FMステレオ受信状態の報告
	 * \details
	 * FM復調がモノラルかステレオを報告する。引数は以下のように与える
	 * \li 0 : モノラル
	 * \li 非1 : ステレオ
	 */
#define radio_api_set_stereo(s) s ? radio.flags |= (1<<1) : radio.flags &= ~(1<<1)

    /**
     * \def radio_api_set_voice_squelch
     * \brief ボイススケルチ状態の報告
     * \details
     * 復調器がボイススケルチを持っている場合、その状態を報告する。
     * \li 0 : ボイススケルチは開いていない
     * \li 1 : ボイススケルチは開いている
     */
#define radio_api_set_voice_squelch(s)  s ? radio.flags |= (1<<2) : radio.flags &= ~(1<<2)

/**
 * \def radio_api_set_noise_squelch
 * \brief ノイズスケルチ状態の報告
 * \details
 * 復調器がノイズスケルチを持っている場合、その状態を報告する。
     * \li 0 : ノイズスケルチは開いていない
     * \li 1 : ノイズスケルチは開いている
 */
#define radio_api_set_noise_squelch(s) s ? radio.flags |= (1<<3) : radio.flags &= ~(1<<3)


/*@}*/
/*  end of defgroup setStat */

/*@}*/
/*  end of defgroup radioAPI */

/**
 * \defgroup callbacks 復調用コールバック関数
 * \brief フレームワークが復調ように呼び出すコールバック関数群
 * \details
 * 復調器のためにフレームワークから呼び出すコールバック関数。復調アルゴリズムは全てこの中に記述する。
 *
 * 関数は大きく分けて2種類に分けられる。ひとつは初期化関数で \ref init_demodulator() がこれである。
 * その他の関数は実際の復調を行う関数である。
 */

/*@{*/


     /**
      * \brief 復調器の初期化
      * \details
      * 復調アルゴリズムの初期化を行う。アルゴリズムが変数を初期化しなければならないようなときには、
      * この関数の中に初期化プログラムを書く。
      *
      * この関数は、\ref radio_demodulate_wide_FM() や \ref radio_demodulate_non_wide_FM() が
      * 呼ばれる前に一度だけ呼ばれる。
      *
      */
void init_demodulator(void);


     /**
      * \brief ワイドFMの復調コールバック関数
      * \param idata 受信IFのI(in phase)データ配列。32bit符号付き固定小数点数。フォーマットはQ1.31
      * \param qdata 受信IFのQ(Quadratural phase)データ配列。32bit符号付き固定小数点数。フォーマットはQ1.31
      * \param left 復調オーディオ信号の左チャンネルデータ。16bit符号付き固定小数点数。フォーマットはQ1.15
      * \param right 復調オーディオ信号の右チャンネルデータ。16bit符号付き固定小数点数。フォーマットはQ1.15
      * \details
     * ワイドFM以外の信号を復調するために呼ばれるコールバック関数。この関数と \ref radio_demodulate_non_wide_FM() の
     * 切り替えはフレームワークが自動的に行うため、ユーザー側は気にしなくていい。
     *
     *
     * IF入力(idata[], qdata[])、AF出力(*left, *right)ともFsは31.7kHzである(トランジスタ技術誌2015年5月号pp183)。
     * 1サンプル毎に呼び出されるので、このコールバックは1秒間に31,700回呼び出される。
     *
     * IF入力はワイドFMではオーバーサンプルされた値が入力する。そのため、1サンプルあたりのデータ数は1ではない。
     * 具体的なオーバーサンプル値は、マクロ WIDE_FM_OVERSAMPE で知ることができる。まとめると、この関数の中では
     * idata/qdataそれぞれ WIDE_FM_OVERSAMPE 個のデータを復調処理して、1個のleft/right データを生成する。
     *
     * IF入力データとAF出力データの語長が違うことに注意。データのパック形式は固定小数点型であるため、いずれも
     * データの値の範囲は[-1,1)となる。しかし、C言語としての取り扱いは整数なので、それぞれの最大値は異なる（IFデータの
     * 最大値はAFデータの65536倍）。
     *
      */

void radio_demodulate_wide_FM( short idata[], short qdata[], short* left, short* right );

/**
 * \brief ワイドFM以外の復調コールバック関数
 * \param idata 受信IFのI(in phase)データ。32bit符号付き固定小数点数。フォーマットはQ1.31
 * \param qdata 受信IFのQ(Quadratural phase)データ。32bit符号付き固定小数点数。フォーマットはQ1.31
 * \param left 復調オーディオ信号の左チャンネルデータ。16bit符号付き固定小数点数。フォーマットはQ1.15
 * \param right 復調オーディオ信号の右チャンネルデータ。16bit符号付き固定小数点数。フォーマットはQ1.15
 * \details
 * ワイドFM以外の信号を復調するために呼ばれるコールバック関数。この関数と \ref radio_demodulate_wide_FM() の
 * 切り替えはフレームワークが自動的に行うため、ユーザー側は気にしなくていい。
 *
 * 非ワイドFM以外のモードにはAM, SAM, LSB, USB, Narrow FM, CW の受信モードがある。これらの受信モードの
 * 検出は、コールバック関数内部のスケルトンが自動的に切り分けているので、それぞれの場合ごとに復調アルゴリズムを
 * 書けばよい。
 *
 * LSBとUSBの復調コードの大半を共有したい場合には、一部でLSB/USBの判断を行われなければならない。この場合は
 * \ref radio_api_is_USB() APIを使うことで判別を行う。
 *
 * なお、ワイドFMモードから非ワイドFMモードへの切り替えがSHマイコンから出された場合、実際のFMモードからの
 * 復調器の切り替えは二段階になりうる。最初の段階では radio_demodulate_wide_FM() の呼び出しが終了し、
 * radio_demodulate_non_wide_FM() の呼び出しへと切り替わる。次の段階では、\ref radio_demodulate_non_wide_FM()
 * の中で正しい復調モードが検出され、スケルトンの動作が切り替わる。
 *
 * この二段階切り替えは必ず起こるというわけではなく、FPGAの実装形態に依存する。単にフレームワークは二段階切り替えにも
 * 対応できるということである。
 *
 * IF入力(idata, qdata)、AF出力(*left, *right)ともFsは31.7kHzである(トランジスタ技術誌2015年5月号pp183)。
 * 1サンプル毎に呼び出されるので、このコールバックは1秒間に31,700回呼び出される。
 *
 * IF入力データとAF出力データの語長が違うことに注意。データのパック形式は固定小数点型であるため、いずれも
 * データの値の範囲は[-1,1)となる。しかし、C言語としての取り扱いは整数なので、それぞれの最大値は異なる（IFデータの
 * 最大値はAFデータの65536倍）。
 *
 */

void radio_demodulate_non_wide_FM( int idata, int qdata, short* left, short* right );

/*@}*/
/*  end of defgroupt callbacks */

#endif /* _MACRO_ONLY */

#endif /* DEMODULATOR_H_ */
