/*
 * FileOperation class.
 *
 * Copyright (C) 2007 SATOH Takayuki All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package ts.util.file;

import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.FileNotFoundException;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.LinkedList;
import ts.util.Duo;
import ts.util.text.StringSequence;

/**
 * t@CNXB
 * <br>
 * t@C̈ꗗARs[A폜Aړs邽߂̃\bhpӂB
 *
 * @author  V. 
 * @version $Revision: 1.3 $, $Date: 2007/05/27 16:13:10 $
 */
public final class FileOperation
{
  /**
   * ftHgRXgN^B
   */
  protected FileOperation()
  {}

  /**
   * ړRs[ƂȂt@C{@link java.io.File File}IuWFNg
   * 擾B
   *
   * @param  srcFile ړΏۂ̃t@CB
   * @param  baseDir ̊fBNgB
   * @param  destDir ړ̃fBNgB
   * @return ړRs[ƂȂ{@link java.io.File File}IuWFNgB
   * @throws IOException ɓo͗OꍇB
   * @throws AssertionError k̏ꍇA͈ړΏۂ̃t@C̊
   *        fBNgȉ̃t@CłȂꍇifobOE[ĥ݁jB
   */
  protected static File getDestinationFile(
    File srcFile, File baseDir, File destDir) throws IOException
  {
    assert (srcFile != null) : "@param:srcFile is null.";
    assert (baseDir != null) : "@param:baseDir is null.";
    assert (destDir != null) : "@param:destDir is null.";

    StringSequence srcPath = new StringSequence(srcFile.getCanonicalPath());
    assert srcPath.startsWith(baseDir.getCanonicalPath());

    StringSequence basePath = new StringSequence(
      baseDir.getCanonicalPath() + File.separator);

    if (srcPath.restLength() > basePath.restLength()) {
      return new File(destDir, srcPath.next(basePath.restLength()).substring());
    }
    else {
      return destDir;
    }
  }

  /**
   * w肳ꂽfBNg쐬B
   * <br>
   * k̏ꍇ́Â܂܃kԂB
   *
   * @param  dir fBNgB
   * @return 쐬ꂽŏʂ̃fBNgBw肳ꂽfBNgɍ쐬
   *           Ăꍇ̓kԂB
   * @throws IOException 쐬ɗOꍇB
   */
  private static File createDir(File dir) throws IOException
  {
    if (dir == null) {
      return null;
    }

    File topDir = null;
    for (File d = dir; d != null && ! d.exists(); d = d.getParentFile()) {
      topDir = d;
    }
    if (! dir.mkdirs()) {
      if (dir.exists()) {
        if (! dir.isDirectory()) {
          throw new FileAlreadyExistsException(dir.getPath());
        }
      }
      else {
        throw new IOException(dir.getPath());
      }
    }
    return topDir;
  }

  /** 
   * w肳ꂽpX̃t@C쐬B
   * <br>
   * w肳ꂽt@C̐efBNg݂Ȃꍇ́AefBNg쐬
   * B
   * w肳ꂽpX̃t@Cɑ݂ꍇ́AOX[B
   *
   * @param  path 쐬t@C̃pXB
   * @throws IOException t@C̍쐬ɎsꍇB
   * @throws AssertionError k̏ꍇifobOE[ĥ݁jB
   */
  public static File createNewFile(String path)
    throws FileAlreadyExistsException, IOException
  {
    assert (path != null) : "@param:path is null.";

    return createNewFile(new File(path));
  }

  /** 
   * w肳ꂽt@C쐬B
   * <br>
   * w肳ꂽt@C̐efBNg݂Ȃꍇ́AefBNg
   * 쐬B
   * w肳ꂽt@Cɑ݂ꍇ́AOX[B
   *
   * @param  file 쐬t@CB
   * @throws IOException t@C̍쐬ɎsꍇB
   * @throws AssertionError k̏ꍇifobOE[ĥ݁jB
   */
  public static File createNewFile(File file)
    throws IOException
  {
    assert (file != null) : "@param:file is null.";

    File createdTopDir = createDir(file.getParentFile());

    if (! file.createNewFile()) {
      if (createdTopDir != null) {
        deleteInner(createdTopDir);
      }

      if (file.exists()) {
        throw new FileAlreadyExistsException(file.getPath());
      }
      else {
        throw new IOException(file.getPath());
      }
    }

    return file;
  }

  /**
   * w肳ꂽfBNg̉ɋ̃t@Cj[NȖOŐB
   * <br>
   * t@Cɂ́Aw肳ꂽړyѐڔgpB
   * ړɂ3ȏオKvłB
   * ڔɂ́Ak܂ޔCӂ̕񂪎w\łAkw肳ꂽꍇ
   * <tt>".tmp"</tt>gpB
   * <br>
   * fBNgɃkw肳ꂽꍇ́A
   * VXeˑ̈ꎞt@CfBNggpB
   * <br>
   * ̃\bh͓{@link java.io.File#createTempFile(String,String,File)
   * createTempFile(String,String,File)}\bhĂяoĂB
   *
   * @param  prefix t@CɎgpړB
   * @param  suffix t@CɎgpڔB
   * @param  dir t@CfBNgB
   * @return VKɐꂽt@C{@link java.io.File File}IuWFNgB
   * @throws NullPointerException ړꂪ3ɖȂꍇB
   * @throws IllegalArgumentException ړꂪ3ɖȂꍇB
   * @throws IOException t@CłȂꍇB
   * @throws SecurityException ZLeB}l[Wɂt@C̐
   *           ȂꍇB
   * @see java.io.File#createTempFile(String, String, File)
   */
  public static File createTempFile(String prefix, String suffix, File dir)
    throws IllegalArgumentException, IOException
  {
    return File.createTempFile(prefix, suffix, dir);
  }

  /**
   * w肳ꂽfBNg̉ɋ̃fBNgj[NȖOŐB
   * <br>
   * t@Cɂ́Aw肳ꂽړyѐڔgpB
   * ړɂ3ȏオKvłB
   * ڔɂ́Ak܂ޔCӂ̕񂪎w\łAkw肳ꂽꍇ
   * 󕶎񂪎gpB
   * <br>
   * fBNgɃkw肳ꂽꍇ́A
   * VXeˑ̈ꎞt@CfBNggpB
   * <br>
   * ̃\bh͓{@link java.io.File#createTempFile(String,String,File)
   * createTempFile(String,String,File)}\bhĂяoāA
   * j[NȃfBNg肵ĂB
   *
   * @param  prefix t@CɎgpړB
   * @param  suffix t@CɎgpڔB
   * @param  dir t@CfBNgB
   * @return VKɐꂽfBNg{@link java.io.File
   *           File}IuWFNgB
   * @throws IllegalArgumentException ړꂲ3ɖȂꍇB
   * @throws IOException fBNgłȂꍇB
   * @throws SecurityException ZLeB}l[WɂfBNg̐
   *           ȂꍇB
   * @see java.io.File#createTempFile(String, String, File)
   */
  public static File createTempDirectory(String prefix, String suffix, File dir)
    throws IllegalArgumentException, IOException
  {
    File file = File.createTempFile(prefix, suffix, dir);
    file.delete();
    file.mkdirs();
    return file;
  }

  /**
   * w肳ꂽfBNg̃t@C̈ꗗ擾B
   * <br>
   * w肳ꂽfBNg̏ꍇ́ÃXgԂB
   * fBNgł͂Ȃꍇ́Ai[XgԂB
   * ɃG[ꍇ͗OX[B
   *
   * @param  baseDir x[XfBNgB
   * @return x[XfBNgɂt@Ci[XgB
   * @throws FileNotFoundException w肳ꂽfBNg̓t@C
   *           ȂꍇB
   * @throws IOException ɓo̓G[ꍇB
   * @throws SecurityException ZLeB}l[Wɂt@Cւ̓Ǎ
   *           ANZXȂꍇB
   * @throws AssertionError k̏ꍇifobOE[ĥ)B
   */
  public static List<File> list(File baseDir)
    throws FileNotFoundException, IOException
  {
    assert (baseDir != null) : "@param:baseDir is null.";

    if (! baseDir.exists()) {
      throw new FileNotFoundException(baseDir.getPath());
    }

    List<File> lst = new LinkedList<File>();

    File[] ret = baseDir.listFiles();
    if (ret == null) {
      if (! baseDir.isDirectory()) {
        lst.add(baseDir);
        return lst;
      }
      else {
        throw new IOException(baseDir.getPath());
      }
    }

    for (int i=0; i<ret.length; i++) {
      lst.add(ret[i]);
    }

    return lst;
  }

  /**
   * w肳ꂽfBNg̃t@ĈAtB^̏ɊY
   * t@C̈ꗗ擾B
   * <br>
   * w肳ꂽfBNg̏ꍇ́ÃXgԂB
   * tB^̏ɊYt@C݂Ȃꍇ́ÃXg
   * ԂB
   * fBNgłȂꍇ́Ãt@CtB^̏Ŕ肵
   * XgɊi[ĕԂB
   * ɃG[ꍇ͗OX[B
   *
   * @param  baseDir x[XEfBNgB
   * @param  filter t@C̑IʂɎgptB^B
   * @return x[XEfBNgɂt@Ci[XgB
   * @throws FileNotFoundException w肳ꂽfBNg̓t@C
   *           ȂꍇB
   * @throws IOException ɓo̓G[ꍇB 
   * @throws SecurityException ZLeBE}l[Wɂt@Cւ
   *           ǍANZXȂꍇB
   * @throws AssertionError k̏ꍇifobOE[ĥ݁jB
   */
  public static List<File> list(File baseDir, FileFilter filter)
    throws FileNotFoundException, IOException
  {
    assert (filter != null) : "@param:filter is null.";

    List<File> lst = list(baseDir);

    Iterator<File> it = lst.iterator();
    while (it.hasNext()) {
      if (! filter.accept(it.next())) {
        it.remove();
      }
    }
    return lst;
  }

  /**
   * w肳ꂽfBNgȉ̑SẴt@C̈ꗗ擾B
   * <br>
   * w肳ꂽfBNgɃfBNg΁ẢɊi[Ă
   * t@CꗗɊ܂߂ĕԂB
   * w肳ꂽfBNg̏ꍇ́A̔zԂB
   * fBNgłȂꍇ́Ai[XgԂB
   * ɃG[ꍇ͗OX[B
   *
   * @param  baseDir x[XfBNgB
   * @return x[XIuWFNgȉɂt@C{@link java.io.File File}
   *           IuWFNg̃XgB
   * @throws FileNotFoundException w肳ꂽfBNg̓t@C
   *           ȂꍇB
   * @throws IOException ɓo̓G[ꍇB
   * @throws SecurityException ZLeB}l[Wɂt@Cւ̓Ǎ
   *           ANZXȂꍇB
   * @throws AssertionError k̏ꍇifobOE[ĥ)B
   */
  public static List<File> listRecursive(File baseDir)
    throws FileNotFoundException, IOException
  {
    assert (baseDir != null) : "@param:baseDir is null.";

    if (! baseDir.exists()) {
      throw new FileNotFoundException(baseDir.getPath());
    }

    List<File> lst = new LinkedList<File>();
    if (! baseDir.isDirectory()) {
      lst.add(baseDir);
      return lst;
    }

    return listInner(baseDir, lst);
  }

  /**
   * w肳ꂽfBNgȉ̃t@ĈAtB^̏ɊY
   * t@C̈ꗗ擾B
   * <br>
   * w肳ꂽfBNgɃfBNg΁ẢɊi[Ă
   * t@CtB^̏Ŕ肵ĈꗗɊ܂߂ĕԂB
   * tB^̏ɊYt@C݂Ȃꍇ́ÃXg
   * ԂB
   * w肳ꂽfBNg̏ꍇ́ÃXgԂB
   * fBNgłȂꍇ́Ãt@CtB^̏Ŕ肵
   * XgɊi[ĕԂB
   * ɃG[ꍇ͗OX[B
   *
   * @param  baseDir x[XfBNgB
   * @param  filter t@C̑IʂɎgptB^B
   * @return x[XIuWFNgȉɂt@C{@link java.io.File File}
   *           IuWFNg̃XgB
   * @throws FileNotFoundException w肳ꂽfBNg̓t@C
   *           ȂꍇB
   * @throws IOException ɓo̓G[ꍇB
   * @throws SecurityException ZLeB}l[Wɂt@Cւ̓Ǎ
   *           ANZXȂꍇB
   * @throws AssertionError k̏ꍇifobOE[ĥ)B
   */
  public static List<File> listRecursive(File baseDir, FileFilter filter)
    throws FileNotFoundException, IOException
  {
    assert (filter != null) : "@param:filter is null.";

    List<File> lst = listRecursive(baseDir);
    Iterator<File> it = lst.iterator();
    while (it.hasNext()) {
      if (! filter.accept(it.next())) {
        it.remove();
      }
    }
    return lst;
  }

  /**
   * w肳ꂽt@C폜B
   * <br>
   * w肳ꂽt@CfBNg̏ꍇÃfBNgȂ
   * 폜͐B
   *
   * @param  file 폜Ώۂ̃t@CB
   * @throws FileNotFoundException w肳ꂽt@CȂꍇB
   * @throws DirectoryNotEmptyException w肳ꂽt@C̃fBNg
   *           ȂꍇB
   * @throws IOException w肳ꂽt@CbN̏ꍇA͏
   *           o̓G[ꍇB
   * @throws SecurityException ZLeB}l[Wɂt@Cւ̓Ǎ
   *           ANZX͍폜ANZXȂꍇB
   * @throws AssertionError k̏ꍇifobOE[ĥ)B
   */
  public static void delete(File file)
    throws FileNotFoundException, DirectoryNotEmptyException, IOException
  {
    assert (file != null) : "@param:file is null.";

    if (! file.delete()) {
      if (! file.exists()) {
        throw new FileNotFoundException(file.getPath());
      }

      if (file.isDirectory()) {
        if (file.list().length > 0)
          throw new DirectoryNotEmptyException(file.getPath());
      }

      throw new IOException(file.getPath());
    }
  }

  /**
   * w肳ꂽt@Ct@CtB^̏ɍvꍇɍ폜B
   * <br>
   * w肳ꂽt@CfBNg̏ꍇÃfBNgłȂȂ
   * 폜ȂB
   *
   * @param  file 폜Ώۂ̃t@CB
   * @param  filter 폜Ώۂ̃t@C̑IʂɎgptB^B
   * @throws FileNotFoundException w肳ꂽt@CȂꍇB
   * @throws DirectoryNotEmptyException w肳ꂽt@C̃fBNg
   *           ȂꍇB
   * @throws IOException ɓo̓G[ꍇB
   * @throws SecurityException ZLeB}l[Wɂt@Cւ̓Ǎ
   *           ANZX͍폜ANZXȂꍇB
   * @throws AssertionError k̏ꍇifobOE[ĥ)B
   */
  public static void delete(File file, FileFilter filter)
    throws FileNotFoundException, DirectoryNotEmptyException, IOException
  {
    assert (file != null) : "@param:file is null.";
    assert (filter != null) : "@param:filter is null.";

    if (! file.exists()) {
      throw new FileNotFoundException(file.getPath());
    }

    if (filter.accept(file)) {
      delete(file);
    }
  }

  /**
   * w肳ꂽt@C폜B
   * <br>
   * w肳ꂽt@CfBNg̏ꍇÃfBNgȉ̑SĂ
   * t@C폜B
   *
   * @param  file 폜Ώۂ̃t@CB
   * @throws FileNotFoundException w肳ꂽt@CȂꍇB
   * @throws IOException ɓo̓G[ꍇB
   * @throws SecurityException ZLeB}l[Wɂt@Cւ̓Ǎ
   *           ANZX͍폜ANZXȂꍇB
   * @throws AssertionError k̏ꍇifobOE[ĥ)B
   */
  public static void deleteRecursive(File file)
    throws FileNotFoundException, IOException
  {
    assert (file != null) : "@param:file is null.";

    if (! file.exists()) {
      throw new FileNotFoundException(file.getPath());
    }

    File tmpDir = createTempDirectory("__del", "", file.getParentFile());
    try {
      File dstDir = new File(tmpDir, file.getName());
      if (! file.renameTo(dstDir)) {
        throw new IOException(file.getPath());
      }
    }
    finally {
      deleteInner(tmpDir);
    }
  }

  /**
   * w肳ꂽt@CtB^̏ɍvꍇÃt@C폜B
   * <br>
   * w肳ꂽt@CfBNg̏ꍇ́ÃfBNgȉ̃t@C
   * tB^̏Ŕ肵č폜B
   *
   * @param  file 폜Ώۂ̃t@CB
   * @param  filter 폜Ώۂ̃t@C̑IʂɎgptB^B
   * @throws FileNotFoundException w肳ꂽt@CȂꍇB
   * @throws IOException ɓo̓G[ꍇB
   * @throws SecurityException ZLeB}l[Wɂt@Cւ̓Ǎ
   *           ANZX͍폜ANZXȂꍇB
   * @throws AssertionError k̏ꍇifobOE[ĥ)B
   */
  public static void deleteRecursive(File file, FileFilter filter)
    throws FileNotFoundException, IOException
  {
    assert (file != null) : "@param:file is null.";
    assert (filter != null) : "@param:filter is null.";

    if (! file.exists()) {
      throw new FileNotFoundException(file.getPath());
    }

    List<File> lst = new LinkedList<File>();
    lst.add(file);
    if (file.isDirectory()) {
      listInner(file, lst);
    }

    File tmpDir = createTempDirectory("__del", "", file.getParentFile());
    File dstDir = new File(tmpDir, file.getName());
    boolean complete = false;
    try {
      for (File f : lst) {
        if (! f.exists()) { // already moved with the parent directory.
          continue;
        }
        if (! filter.accept(f)) {
          continue;
        }
        File d = getDestinationFile(f, file, dstDir);
        File p = d.getParentFile();
        if (p != null) {
          p.mkdirs();
        }
        if (! f.renameTo(d)) {
          throw new IOException(f.getPath());
        }
      }
      complete = true;
    }
    finally {
      if (! complete) {
        moveInner(dstDir, file);
      }
      deleteInner(tmpDir);
    }
  }

  /**
   * P̃t@CQ̃t@CɈړB
   * <br>
   * Pʏ̃t@C̏ꍇ͂QɈړB
   * PfBNg̏ꍇ͂̃fBNgʂ̃t@CƈꏏɈړ
   * B
   * AAQɑ݂ꍇ́Aꂪʏ̃t@CłfBNg
   * łAړs킸ɗOX[B
   *
   * @param  src ړ̃t@CB
   * @param  dst ړ̃t@CB
   * @throws FileNotFoundException ړ̃t@C݂ȂꍇB
   * @throws FileAlreadyExistsExcdeption ړ̃t@Cɑ݂Ă
   *           ꍇB
   * @throws IOException ɓo̓G[ꍇB
   * @throws SecurityException ZLeB}l[Wɂt@Cւ̓Ǎ
   *           ANZXA̓t@CȂꍇB
   * @throws AssertionError k̏ꍇifobOE[ĥ)B
   */
  public static void move(File src, File dst)
    throws FileNotFoundException, FileAlreadyExistsException, IOException
  {
    assert (src != null) : "@param:src is null.";
    assert (dst != null) : "@param:dst is null.";

    if (! src.exists()) {
      throw new FileNotFoundException(src.getPath());
    }

    File createdTopDir = createDir(dst.getParentFile());

    if (! src.renameTo(dst)) {
      if (createdTopDir != null) {
        deleteInner(createdTopDir);
      }

      if (dst.exists()) {
        throw new FileAlreadyExistsException(dst.getPath());
      }
      else {
        throw new IOException(src.getPath());
      }
    }
  }

  /**
   * P̃t@CtB^ɊYꍇɁAQ̃t@C
   * ړB
   * <br>
   * Pʏ̃t@C̏ꍇ͂QɈړB
   * PfBNg̏ꍇ͂̃fBNgʂ̃t@CƈꏏɈړ
   * B
   * AAQɑ݂ꍇ́Aꂪʏ̃t@CłfBNg
   * łAړs킸ɗOX[B
   *
   *
   * @param  src ړ̃t@CB
   * @param  dst ړ̃t@CB
   * @param  filter ړΏۂ̃t@C̑IʂɎgptB^B
   * @throws FileNotFoundException ړ̃t@C݂ȂꍇB
   * @throws FileAlreadyExistsExcdeption ړ̃t@Cɑ݂Ă
   *           ꍇB
   * @throws IOException ɓo̓G[ꍇB
   * @throws SecurityException ZLeB}l[Wɂt@Cւ̓Ǎ
   *           ANZXA̓t@CȂꍇB
   * @throws AssertionError k̏ꍇifobOE[ĥ)B
   */
  public static void move(File src, File dst, FileFilter filter)
    throws FileNotFoundException, FileAlreadyExistsException, IOException
  {
    assert (src != null) : "@param:src is null.";
    assert (dst != null) : "@param:dst is null.";
    assert (filter != null) : "@param:filter is null.";

    if (! src.exists()) {
      throw new FileNotFoundException(src.getPath());
    }

    if (! filter.accept(src)) {
      return;
    }

    File createdTopDir = createDir(dst.getParentFile());

    if (! src.renameTo(dst)) {
      if (createdTopDir != null) {
        deleteInner(createdTopDir);
      }

      if (dst.exists()) {
        throw new FileAlreadyExistsException(dst.getPath());
      }
      else {
        throw new IOException(src.getPath());
      }
    }
  }

  /**
   * P̃t@CQ̃t@CɈړB
   * <br>
   * PfBNg̏ꍇ́Aȉ̃t@CSĈړB
   * <br>
   * ړ̃t@Cɑ݂Ăꍇ́Ãt@CfBNgłȂ
   * ȂΗOX[AfBNgȂ΂̉̃t@C̈ړsB
   * <br>
   * ړɗOꍇ́Aړt@Cɖ߂B
   *
   * @param  src ړ̃t@CB
   * @param  dst ړ̃t@CB
   * @throws FileNotFoundException ړ̃t@C݂ȂꍇB
   * @throws FileAlreadyExistsExcdeption ړ̃t@Cɑ݂Ă
   *           ꍇiʏt@Ĉ݁jB
   * @throws IOException ɓo̓G[ꍇB
   * @throws SecurityException ZLeB}l[Wɂt@Cւ̓Ǎ
   *           ANZXA̓t@CȂꍇB
   * @throws AssertionError k̏ꍇifobOE[ĥ)B
   */
  public static void moveRecursive(File src, File dst)
    throws FileNotFoundException, FileAlreadyExistsException,
           DirectoryNotEmptyException, IOException
  {
    moveInner(src, dst);
  }

  /**
   * P̃t@CtB^̏ɊYꍇɁAQ̃t@C
   * ړB
   * <br>
   * PfBNg̏ꍇAȉ̃t@CŃtB^̏ɊY
   * ̂ړB
   * <br>
   * ړ̃t@Cɑ݂Ăꍇ́Ãt@CfBNgłȂ
   * ȂΗOX[AfBNgȂ΂̉̃t@C̈ړsB
   * <br>
   * ړɗOꍇ́Aړt@Cɖ߂B
   *
   * @param  src ړ̃t@CB
   * @param  dst ړ̃t@CB
   * @throws FileNotFoundException ړ̃t@C݂ȂꍇB
   * @throws FileAlreadyExistsExcdeption ړ̃t@Cɑ݂Ă
   *           ꍇiʏt@Ĉ݁jB
   * @throws IOException ɓo̓G[ꍇB
   * @throws SecurityException ZLeB}l[Wɂt@Cւ̓Ǎ
   *           ANZXA̓t@CȂꍇB
   * @throws AssertionError k̏ꍇifobOE[ĥ)B
   */
  public static void moveRecursive(File src, File dst, FileFilter filter)
    throws FileNotFoundException, FileAlreadyExistsException,
           DirectoryNotEmptyException, IOException
  {
    moveInner(src, dst, filter);
  }

  /**
   * P̃t@CQ̃t@CɃRs[B
   * <br>
   * P̃t@C݂Ȃꍇ͗OX[B
   * Q̃t@Cɑ݂ꍇ͗OX[B
   * P̃t@CfBNg̏ꍇÃfBNgłȂꍇ
   * OX[B
   *
   * @param  src Rs[̃t@CB
   * @param  dst Rs[̃t@CB
   * @throws FileNotFoundException Rs[̃t@C݂ȂꍇB
   * @throws DirectoryNotEmptyException Rs[̃fBNgłȂꍇB
   * @throws FileAlreadyExistsException Rs[̃t@Cɑ݂ꍇB
   * @throws IOException ɓo̓G[ꍇB
   * @throws SecurityException ZLeB}l[Wɂt@Cւ̓Ǎ
   *           ANZXA݊mFANZXAfBNgAt@C
   *           ȂꍇB
   * @throws AssertionError k̏ꍇifobOE[ĥ)B
   */
  public static void copy(File src, File dst)
    throws FileNotFoundException, DirectoryNotEmptyException,
           FileAlreadyExistsException, IOException
  {
    assert (src != null) : "@param:src is null.";
    assert (dst != null) : "@param:dst is null.";

    if (! src.exists()) {
      throw new FileNotFoundException(dst.getPath());
    }
    else if (dst.exists()) {
      throw new FileAlreadyExistsException(dst.getPath());
    }
    
    if (! src.isDirectory()) {
      boolean complete = false;
      File createdTopDir = createDir(dst.getParentFile());
      try {
        copyFile(src, dst);
        complete = true;
      }
      finally {
        if (! complete) {
          if (createdTopDir != null) {
            deleteInner(createdTopDir);
          }
        }
      }
    }
    else {
      if (src.list().length > 0) {
        throw new DirectoryNotEmptyException(src.getPath());
      }
      dst.mkdirs();
    }
  }

  /**
   * P̃t@CtB^̏ɍvꍇɁAQ̃t@C
   * Rs[B
   * <br>
   * P̃t@C݂Ȃꍇ͗OX[B
   * Q̃t@Cɑ݂ꍇ͗OX[B
   * P̃t@CfBNg̏ꍇÃfBNgłȂꍇ
   * OX[B
   *
   * @param  src Rs[̃t@CB
   * @param  dst Rs[̃t@CB
   * @param  filter Rs[Ώۂ̃t@C̑IʂɎgptB^B
   * @throws FileNotFoundException Rs[̃t@C݂ȂꍇB
   * @throws DirectoryNotEmptyException Rs[̃fBNgłȂ
   *           ꍇB
   * @throws FileAlreadyExistsException Rs[̃t@Cɑ݂Ă
   *           ꍇB
   * @throws IOException ɓo̓G[ꍇB
   * @throws SecurityException ZLeB}l[Wɂt@Cւ̓Ǎ
   *           ANZXA݊mFANZXAfBNgAt@C
   *           ȂꍇB
   * @throws AssertionError k̏ꍇifobOE[ĥ)B
   */
  public static void copy(File src, File dst, FileFilter filter)
    throws FileNotFoundException, DirectoryNotEmptyException, 
           FileAlreadyExistsException, IOException
  {
    assert (src != null) : "@param:src is null.";
    assert (dst != null) : "@param:dst is null.";
    assert (filter != null) : "@param:filter is null.";

    if (! src.exists()) {
      throw new FileNotFoundException(dst.getPath());
    }

    if (! filter.accept(src)) {
      return;
    }

     if (dst.exists()) {
      throw new FileAlreadyExistsException(dst.getPath());
    }
    
    if (! src.isDirectory()) {
      boolean complete = false;
      File createdTopDir = createDir(dst.getParentFile());
      try {
        copyFile(src, dst);
        complete = true;
      }
      finally {
        if (! complete) {
          if (createdTopDir != null) {
            deleteInner(createdTopDir);
          }
        }
      }
    }
    else {
      if (src.list().length > 0) {
        throw new DirectoryNotEmptyException(src.getPath());
      }
      dst.mkdirs();
    }
  }

  /**
   * P̃t@CQ̃t@CɃRs[B
   * <br>
   * P̃t@CfBNg̏ꍇÃfBNgȉ̃t@C
   * SăRs[B
   * <br>
   * P̃t@C݂Ȃꍇ͗OX[B
   * Q̃t@Cɑ݂ꍇ͗OX[B
   *
   * @param  src Rs[̃t@CB
   * @param  dst Rs[̃t@CB
   * @throws FileNotFoundException Rs[̃t@C݂ȂꍇB
   * @throws FileAlreadyExistsException Rs[́AfBNgłȂt@C
   *           ɑ݂ĂꍇB
   * @throws IOException ɓo̓G[ꍇB
   * @throws SecurityException ZLeB}l[Wɂt@Cւ̓Ǎ
   *           ANZXA݊mFANZXAfBNgAt@C
   *           ȂꍇB
   * @throws AssertionError k̏ꍇifobOE[ĥ)B
   */
  public static void copyRecursive(File src, File dst)
    throws FileNotFoundException, FileAlreadyExistsException, IOException
  {
    copyInner(src, dst);
  }

  /**
   * Rs[̃t@CtB^̏ɍvꍇɁARs[ɃRs[B
   * <br>
   * P̃t@CfBNg̏ꍇÃfBNgȉ̃t@C
   * tB^̏Ŕ肵ăRs[B
   *
   * @param  src Rs[̃t@CB
   * @param  dst Rs[̃t@CB
   * @param  filter Rs[Ώۂ̃t@C̑IʂɎgptB^B
   * @throws FileNotFoundException Rs[̃t@C݂ȂꍇB
   * @throws FileAlreadyExistsException Rs[̃t@Cɑ݂Ă
   *           ꍇB
   * @throws IOException ɓo̓G[ꍇB
   * @throws SecurityException ZLeB}l[Wɂt@Cւ̓Ǎ
   *           ANZXA݊mFANZXAfBNgAt@C
   *           ȂꍇB
   * @throws AssertionError k̏ꍇifobOE[ĥ)B
   */
  public static void copyRecursive(File src, File dst, FileFilter filter)
    throws FileNotFoundException, FileAlreadyExistsException, IOException
  {
    copyInner(src, dst, filter);
  }


  /**
   * w肳ꂽfBNgȉ̑SẴt@Ci[Xg擾B
   * <br>
   * XgɊi[t@C̏́AefBNg̕Ả̃t@C
   * Ɋi[B
   * <br>
   * w肳ꂽt@CfBNgłȂꍇ́ÃXg
   * ̂܂ܕԂB
   * <br>
   * w肳ꂽfBNg̏ꍇ́ÃXgɉǉ̂܂
   * ԂB
   *
   * @param  dir fBNgB
   * @param  fileLst t@C̈ꗗi[郊XgB
   * @return t@C̈ꗗi[XgB<code>fileLst</code>Ɠ
   *           łB
   * @throws IOException ɓo̓G[ꍇB
   * @throws SecurityException ZLeB}l[Wɂt@Cւ̓Ǎ
   *           ANZXȂꍇB
   * @throws AssertionError k̏ꍇA
   *           <code>dir</code>݂ȂꍇA
   *           <code>dir</code>fBNgłȂꍇ
   *           ifobOE[ĥ)B
   */
  private static List<File> listInner(File dir, List<File> fileLst)
    throws IOException
  {
    assert (dir != null) : "@param:dir is null.";
    assert (fileLst != null) : "@param:fileLst is null.";
    assert (dir.exists()) : "@param:dir does not exist.";

    if (! dir.isDirectory()) {
      return fileLst;
    }
    
    File[] files = dir.listFiles();
    if (files == null) {
      throw new IOException(dir.getPath());
    }

    for (File f : files) {
      fileLst.add(f);
      listInner(f, fileLst);
    }
    return fileLst;
  }

  /**
   * w肳ꂽfBNgȉ̑SẴt@C폜B
   * <br>
   * w肳ꂽt@CfBNgłȂꍇ폜B
   *
   * @param  dir fBNgB
   * @throws IOException ɓo̓G[ꍇB
   * @throws SecurityException ZLeB}l[Wɂt@Cւ̓Ǎ
   *           ANZX͍폜ANZXȂꍇB
   * @throws AssertionError k̏ꍇA
   *           <code>dir</code>fBNgłȂꍇ
   *          ifobOE[ĥ)B
   */
  private static void deleteInner(File dir) throws IOException
  {
    assert (dir != null) : "@param:dir is null.";
    assert (dir.exists()) : "@param:dir does not exist.";
    
    if (dir.isDirectory()) {
      File[] files = dir.listFiles();
      if (files == null) {
        throw new IOException(dir.getPath());
      }

      for (int i=0; i<files.length; i++) {
        if (files[i].isDirectory()) {
          deleteInner(files[i]);
        }
        else {
          if (! files[i].delete()) {
            throw new IOException(dir.getPath());
          }
        }
      }
    }

    if (! dir.delete()) {
      throw new IOException(dir.getPath());
    }
  }

  /**
   * w肳ꂽfBNgȉ̑SẴt@CړB
   * <br>
   * ړ̃t@Cɑ݂Ăꍇ́Ãt@CfBNgłȂ
   * ȂΗOX[AfBNgȂ΂̉̃t@C̈ړsB
   * <br>
   * ړɗOꍇ́Aړt@Cɖ߂B
   *
   * @param  srcFile ړ̃t@CB
   * @param  dstFile ړ̃t@CB
   *
   * @throws FileAlreadyExistsException ړ̃t@Cɑ݂ꍇB
   * @throws FileNotFoundException ړ̃t@C݂ȂꍇB
   * @throws IOException ړɗOꍇB
   * @throws AssertionError k̏ꍇifobOE[ĥ݁jB
   */
  private static void moveInner(File src, File dst) throws IOException
  {
    assert (src != null) : "@param:src is null.";
    assert (dst != null) : "@param:dst is null.";

    checkFailureReason(src, dst, false);

    boolean complete = false;
    File createdTopDir = null;
    try {
      createdTopDir = createDir(dst.getParentFile());

      if (src.renameTo(dst)) {
        complete = true;
        return;
      }

      checkFailureReason(src, dst, true);
    } 
    finally {
      if (! complete && createdTopDir != null) {
        deleteInner(createdTopDir);
      }
    }

    LinkedList<Duo<File,File>> rollbackLst = new LinkedList<Duo<File,File>>();
    try {
      List<File> fL = new LinkedList<File>();
      fL.add(src);
      if (src.isDirectory()) {
        listInner(src, fL);
      }

      for (File f0 : fL) {
        File f1 = getDestinationFile(f0, src, dst);
        if (! f0.exists()) {  // already moved with parent directory.
          continue;
        }
        else if (f0.renameTo(f1)) {
          rollbackLst.addFirst(new Duo<File,File>(f0, f1));
        }
        else {
          checkFailureReason(f0, f1, true);
        }//if
      }//for

      deleteInner(src);
      complete = true;
    }
    finally {
      if (! complete) {
        for (Duo<File,File> duo : rollbackLst) {
          duo.getSecond().renameTo(duo.getFirst());
        }
      }//if
    }
  }

  /**
   * w肳ꂽfBNgȉ̃t@ĈAtB^ɊỸt@C
   * ړB
   * <br>
   * ړ̃t@Cɑ݂Ăꍇ́Ãt@CfBNg
   * ȂȂΗOX[AfBNgȂ΂̉̃t@C̈ړ
   * ċNIɎsB
   * <br>
   * ړɗOꍇ́Aړt@Cɖ߂B
   *
   * @param  srcFile ړ̃fBNgB
   * @param  dstFile ړ̃fBNgB
   * @param  filter ړΏۂ̃t@C̑IʂɎgptB^B
   * @throws IOException ɓo̓G[ꍇB
   * @throws AssertionError k̏ꍇifobOE[ĥ݁jB
   */
  private static void moveInner(File src, File dst, FileFilter filter)
    throws IOException
  {
    assert (src != null) : "@param:src is null.";
    assert (dst != null) : "@param:dst is null.";
    assert (filter != null) : "@param:filter is null.";

    checkFailureReason(src, dst, false);

    boolean complete = false;
    LinkedList<Duo<File,File>> rollbackLst = new LinkedList<Duo<File,File>>();

    try {
      List<File> fL = new LinkedList<File>();
      fL.add(src);
      if (src.isDirectory()) {
        listInner(src, fL);
      }

      for (File f0 : fL) {
        if (! f0.exists()) {  // already moved with the parent directory.
          continue;
        }
        if (! filter.accept(f0)) {
          continue;
        }

        File f1 = getDestinationFile(f0, src, dst);
        File d = createDir(f1.getParentFile());
        if (d != null) {
          rollbackLst.addFirst(new Duo<File,File>(null, d));
        }

        if (f0.renameTo(f1)) {
          rollbackLst.addFirst(new Duo<File,File>(f0, f1));
        }
        else {
          checkFailureReason(f0, f1, true);
        }//if
      }//for

      complete = true;
    }
    finally {
      if (! complete) {
        for (Duo<File,File> duo : rollbackLst) {
          if (duo.getFirst() != null) {
            duo.getSecond().renameTo(duo.getFirst());
          }
          else {
            deleteInner(duo.getSecond());
          }
        }
      }//if
    }
  }

  /**
   * t@C̈ړRs[Ɏs𒲂ׁAɑ΂OX[B
   *
   * @param  src ړERs[̃t@CB
   * @param  dst ړERs[̃t@CB
   * @param  force IɗOX[ꍇ<tt>true</tt>B
   * @throws IOException sɑΉOB
   */
  private static void checkFailureReason(File src, File dst, boolean force)
    throws IOException
  {
    if (! src.exists()) {
      throw new FileNotFoundException(src.getPath());
    }
    else if (dst.exists()) {
      if (src.isDirectory() != dst.isDirectory()) {
        throw new FileAlreadyExistsException(dst.getPath());
      }
      else if (! src.isDirectory()) {
        throw new FileAlreadyExistsException(dst.getPath());
      }
    }
    else if (force) {
      throw new IOException(src.getPath());
    }
  }

  /**
   * t@C̎̂Rs[B
   * <br>
   * Rs[̃t@Cɑ݂ꍇ́A㏑B
   *
   * @param  src Rs[̃t@CB
   * @param  dst Rs[̃t@CB
   * @throws FileNotFoundException Rs[̃t@C݂ȂAʂ
   *          t@Cł͂ȂfBNgł邩A܂͉炩̗RŊJ
   *          ƂłȂꍇB
   * @throws IOException Rs[ɗOꍇB
   * @throws AssertionError k̏ꍇA̓Rs[̃t@C
   *           fBNgw肳ꂽꍇifobOE[ĥ݁jB
   */
  private static void copyFile(File src, File dst)
    throws FileNotFoundException, IOException
  {
    assert (src != null) : "@param:src is null.";
    assert (dst != null) : "@param:dst is null.";
    assert (! src.isDirectory()) : "@param:src is a directory.";

    boolean dstCreated = !dst.exists();
    boolean complete = false;

    FileInputStream fis = null;
    FileOutputStream fos = null;
    try {
      fis = new FileInputStream(src);
      fos = new FileOutputStream(dst);

      int n;
      byte[] bytes = new byte[1024];
      while ((n = fis.read(bytes, 0, bytes.length)) > 0) {
        fos.write(bytes, 0, n);
      }

      complete = true;
    }
    finally {
      if (fos != null) {
        try {
          fos.close();
        }
        catch (Exception e) {}
      }
      if (fis != null) {
        try {
          fis.close();
        }
        catch (Exception e) {}
      }

      if (! complete && dstCreated) {
        dst.delete();
      }
    }
  }

  /**
   * w肳ꂽfBNgȉ̑SẴt@CRs[B
   * <br>
   * Rs[̃t@Cɑ݂Ăꍇ́Ãt@CfBNg
   * ȂȂΗOX[AfBNgȂ΂̉̃t@C̈ړ
   * ċNIɎsB
   *
   * @param  src Rs[̃fBNgB
   * @param  dst Rs[̃fBNgB
   * @throws IOException ɓo̓G[ꍇB
   * @throws AssertionError k̏ꍇifobOE[ĥ)B
   */
  private static void copyInner(File src, File dst) throws IOException
  {
    assert (src != null) : "@param:src is null.";
    assert (dst != null) : "@param:dst is null.";

    checkFailureReason(src, dst, false);

    boolean complete = false;
    LinkedList<File> rollbackLst = new LinkedList<File>();
    File createdTopDir = null;
    try {
      createdTopDir = createDir(dst.getParentFile());

      List<File> fL = new LinkedList<File>();
      fL.add(src);
      if (src.isDirectory()) {
        listInner(src, fL);
      }

      for (File f0 : fL) {
        File f1 = getDestinationFile(f0, src, dst);

        checkFailureReason(f0, f1, false);

        if (! f0.isDirectory()) {
          copyFile(f0, f1);
          rollbackLst.addFirst(f1);
        }
        else if (! f1.exists()) {
          f1.mkdir();
          rollbackLst.addFirst(f1);
        }
      }
      complete = true;
    }
    finally {
      if (! complete) {
        if (createdTopDir != null) {
          deleteInner(createdTopDir);
        }
        else {
          for (File f1 : rollbackLst) {
            f1.delete();
          }
        }
      }//if (! complete)
    }
  }

  /**
   * w肳ꂽfBNgȉ̃t@C̓AtB^ɊỸt@C
   * Rs[B
   * <br>
   * Rs[̃t@Cɑ݂Ăꍇ́Ãt@CfBNg
   * ȂȂΗOX[AfBNgȂ΂̉̃t@C̈ړ
   * ċNIɎsB
   *
   * @param  src Rs[̃fBNgB
   * @param  dst Rs[̃fBNgB
   * @param  filter Rs[Ώۂ̃t@C̑IʂɎgptB^B
   * @throws IOException ɓo̓G[ꍇB
   */
  private static void copyInner(File src, File dst, FileFilter filter)
    throws IOException
  {
    assert (src != null) : "@param:src is null.";
    assert (dst != null) : "@param:dst is null.";
    assert (filter != null) : "@param:filter is null.";

    checkFailureReason(src, dst, false);

    boolean complete = false;
    LinkedList<File> rollbackLst = new LinkedList<File>();

    File createdTopDir = null;
    try {
      List<File> fL = new LinkedList<File>();
      fL.add(src);
      if (src.isDirectory()) {
        listInner(src, fL);
      }

      for (File f0 : fL) {
        if (! filter.accept(f0)) {
          continue;
        }

        File f1 = getDestinationFile(f0, src, dst);

        checkFailureReason(f0, f1, false);

        if (! f0.isDirectory()) {
          File d = createDir(f1.getParentFile());
          if (d != null) {
            rollbackLst.addFirst(d);
          }
          copyFile(f0, f1);
          rollbackLst.addFirst(f1);
        }
        else if (! f1.exists()) {
          File d = createDir(f1);
          if (d != null) {
            rollbackLst.addFirst(d);
          }
        }
      }
      complete = true;
    }
    finally {
      if (! complete) {
        if (createdTopDir != null) {
          deleteInner(createdTopDir);
        }
        else {
          for (File f1 : rollbackLst) {
            if (f1.exists()) {
              deleteInner(f1);
            }
          }
        }
      }//if (! complete)
    }
  }
}

