/*
 * FileOperation05CopyTest class.
 *
 * Copyright (C) 2007 SATOH Takayuki All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package ts.util.file;

import ts.tester.function.FunctionTester;
import ts.tester.function.print.*;
import java.io.*;
import java.util.*;

/**
 * {@link ts.util.file.FileOperationy FileOperation}NX̋@\NXB
 *
 * @author  V.
 * @version $Revision: 1.2 $, $Date: 2007/05/27 16:13:11 $
 */
public class FileOperation05CopyTest extends FunctionTester
{
  public static void main(String[] args)
  {
    try {
      run(FileOperation05CopyTest.class, (args.length == 0) ? null : args[0]);
    }
    catch (Exception e) {
      e.printStackTrace();
    }
  }

  final File WORKDIR = new File("test/data/ts/util/file/FileOperation");
  File[] DIRS = new File[6];
  File[] FILES = new File[10];

  protected void preInvocation(String method)
  {
    WORKDIR.mkdirs();
    DIRS[0] = new File(WORKDIR, "DIRS0");
    DIRS[1] = new File(DIRS[0], "DIRS1");
    DIRS[2] = new File(DIRS[0], "DIRS2");
    DIRS[3] = new File(DIRS[1], "DIRS3");
    DIRS[4] = new File(DIRS[2], "DIRS4");
    DIRS[5] = new File(DIRS[2], "DIRS5");
    FILES[0] = new File(DIRS[0], "FILES0");
    FILES[1] = new File(DIRS[1], "FILES1");
    FILES[2] = new File(DIRS[2], "FILES2");
    FILES[3] = new File(DIRS[3], "FILES3");
    FILES[4] = new File(DIRS[4], "FILES4");
    FILES[5] = new File(DIRS[0], "FILES5");
    FILES[6] = new File(DIRS[1], "FILES6");
    FILES[7] = new File(DIRS[2], "FILES7");
    FILES[8] = new File(DIRS[3], "FILES8");
    FILES[9] = new File(DIRS[4], "FILES9");
    try {
      for (File d : DIRS) d.mkdirs();
      for (File f : FILES) f.createNewFile();
    } catch (Exception e) {}

    /*
       DIRS[0]
        +- DIRS[1]
          +- DIRS[3]
            +- FILES[3]
            +- FILES[8]
          +- FILES[1]
          +- FILES[6]
        +- DIRS[2]
          +- DIRS[4]
            +- FILES[4]
            +- FILES[9]
          +- DIRS[5]
          +- FILES[2]
          +- FILES[7]
        +- FILES[0]
        +- FILES[5]
    */
  }

  protected void postInvocation(String method)
  {
    try {
      FileOperation.deleteRecursive(WORKDIR);
    } catch (Exception e) {}
  }

  /* -- test case -- */

  public void copy()
  {
    MSG("ʏ̃t@C𓯂fBNgɕʖŃRs[ꍇB");
    
    File src = FILES[4];
    File dst = new File(DIRS[4], "aaa");
    TRUE (src.exists());
    FALSE(dst.exists());

    try {
      FileOperation.copy(src, dst);
    } catch (Exception e) {
      NG(e);
    }

    TRUE (src.exists());
    TRUE (dst.exists());
  }

  public void copy_001()
  {
    MSG("ʏ̃t@Cʂ̃fBNgɃRs[ꍇB");

    File src = FILES[3];
    File dst = new File(WORKDIR, "aaa/bbb/f3");

    TRUE(src.exists());
    FALSE(dst.exists());
    FALSE(dst.getParentFile().exists());

    try {
      FileOperation.copy(src, dst);
    } catch (Exception e) {
      NG(e);
    }
    
    TRUE(src.exists());
    TRUE(dst.exists());
  }

  public void copy_002()
  {
    MSG("̃fBNg𓯂fBNgɕʖŃRs[ꍇB");

    File src = DIRS[5];
    File dst = new File(DIRS[2], "sss");

    TRUE(src.exists());
    TRUE(src.isDirectory());
    TRUE(src.listFiles().length == 0);
    FALSE(dst.exists());

    try {
      FileOperation.copy(src, dst);
    } catch (Exception e) {
      NG(e);
    }

    TRUE(src.exists());
    TRUE(src.isDirectory());
    TRUE(src.listFiles().length == 0);
    TRUE(dst.exists());
    TRUE(dst.isDirectory());
    TRUE(dst.listFiles().length == 0);
  }

  public void copy_003()
  {
    MSG("̃fBNgʂ̃fBNgɃRs[ꍇB");

    File src = DIRS[5];
    File dst = new File(DIRS[3], "f5");

    TRUE(src.exists());
    TRUE(src.isDirectory());
    TRUE(src.listFiles().length == 0);
    FALSE(dst.exists());

    try {
      FileOperation.copy(src, dst);
    } catch (Exception e) {
      NG(e);
    }

    TRUE(src.exists());
    TRUE(src.isDirectory());
    TRUE(src.listFiles().length == 0);
    TRUE(dst.exists());
    TRUE(dst.isDirectory());
    TRUE(dst.listFiles().length == 0);
  }

  public void copy_004()
  {
    MSG("łȂfBNgRs[悤ƂꍇB");

    File src = DIRS[3];
    File dst = new File(DIRS[2], "xxx");

    TRUE(src.exists());
    TRUE(src.isDirectory());
    TRUE(src.listFiles().length > 0);
    FALSE(dst.exists());

    try {
      FileOperation.copy(src, dst);
      NG();
    } catch (DirectoryNotEmptyException e) {
      EQUAL(e.getPath(), src.getPath());
    } catch (Exception e) {
      NG(e);
    }

    TRUE(src.exists());
    TRUE(src.isDirectory());
    TRUE(src.listFiles().length > 0);
    FALSE(dst.exists());
  }

  public void copy_005()
  {
    MSG("Rs[̃t@Ciʏ̃t@Cjɑ݂ꍇB");

    File src = FILES[1];
    File dst = FILES[6];

    TRUE(src.exists());
    FALSE(src.isDirectory());
    TRUE(dst.exists());
    FALSE(dst.isDirectory());

    try {
      FileOperation.copy(src, dst);
      NG();
    } catch (FileAlreadyExistsException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    src = FILES[1];
    dst = FILES[2];

    TRUE(src.exists());
    FALSE(src.isDirectory());
    TRUE(dst.exists());
    FALSE(dst.isDirectory());

    try {
      FileOperation.copy(src, dst);
      NG();
    } catch (FileAlreadyExistsException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    src = DIRS[5];
    dst = FILES[2];

    TRUE(src.exists());
    TRUE(src.isDirectory());
    TRUE(dst.exists());
    FALSE(dst.isDirectory());

    try {
      FileOperation.copy(src, dst);
      NG();
    } catch (FileAlreadyExistsException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void copy_006()
  {
    MSG("Rs[̃t@CifBNgjɑ݂ꍇB");

    File src = DIRS[5];
    File dst = DIRS[3];

    TRUE(src.exists());
    TRUE(src.isDirectory());
    TRUE(dst.exists());
    TRUE(dst.isDirectory());

    try {
      FileOperation.copy(src, dst);
      NG();
    } catch (FileAlreadyExistsException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    src = FILES[1];
    dst = DIRS[5];

    TRUE(src.exists());
    FALSE(src.isDirectory());
    TRUE(dst.exists());
    TRUE(dst.isDirectory());

    try {
      FileOperation.copy(src, dst);
      NG();
    } catch (FileAlreadyExistsException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void copy_007()
  {
    MSG("k̏ꍇB");

    File f = FILES[1];

    try {
      FileOperation.copy(null, f);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    try {
      FileOperation.copy(f, null);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void copy_008()
  {
    MSG("Rs[̃t@CLbNĂꍇB");

    File src = FILES[1];
    File dst = new File(WORKDIR, "ffff");

    TRUE(src.exists());
    FALSE(src.isDirectory());
    FALSE(dst.exists());

    try { 
      FileOperationTest.lockFile("shared", src);
      Thread.sleep(1000);
    } catch (Exception e) {
      NG(e);
    }

    try {
      FileOperation.copy(src, dst);
      Thread.sleep(1000);
    } catch (Exception e) {
      NG(e);
    } finally {
      FileOperationTest.releaseLock();
    }

    TRUE(src.exists());
    FALSE(src.isDirectory());
    TRUE(dst.exists());
    FALSE(dst.isDirectory());
  }

  public void copy_009()
  {
    MSG("Rs[̃t@CrbNĂꍇB");

    File src = FILES[1];
    File dst = new File(WORKDIR, "gggg");

    TRUE(src.exists());
    FALSE(src.isDirectory());
    FALSE(dst.exists());

    try {
      FileOperationTest.lockFile("exclusive", src);
      Thread.sleep(1000);
    } catch (Exception e) {
      NG(e);
    }

    try {
      FileOperation.copy(src, dst);
      Thread.sleep(1000);
      NG();
    } catch (IOException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    } finally {
      FileOperationTest.releaseLock();
    }

    TRUE(src.exists());
    FALSE(src.isDirectory());
    FALSE(dst.exists());
  }

  public void copy_010()
  {
    MSG("Rs[̃t@C݂ȂꍇB");

    File src = new File(WORKDIR, "aaa");
    File dst = new File(WORKDIR, "bbb");

    try {
      FileOperation.copy(src, dst);
      NG();
    } catch (FileNotFoundException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void copy_withFilter()
  {
    MSG("ʏ̃t@C𓯂fBNgɕʖŃRs[ꍇB");
    
    File src = FILES[4];
    File dst = new File(DIRS[4], "aaa");
    TRUE (src.exists());
    FALSE(dst.exists());

    FileFilter filterX = new FileTypeFilter(FileType.Directory);
    try {
      FileOperation.copy(src, dst, filterX);
    } catch (Exception e) {
      NG(e);
    }

    TRUE (src.exists());
    FALSE(dst.exists());

    FileFilter filter = new FileTypeFilter(FileType.File);
    try {
      FileOperation.copy(src, dst, filter);
    } catch (Exception e) {
      NG(e);
    }

    TRUE (src.exists());
    TRUE (dst.exists());
  }

  public void copy_withFilter001()
  {
    MSG("ʏ̃t@Cʂ̃fBNgɃRs[ꍇB");

    File src = FILES[3];
    File dst = new File(WORKDIR, "aaa/bbb/f3");

    TRUE(src.exists());
    FALSE(dst.exists());
    FALSE(dst.getParentFile().exists());

    FileFilter filter = new FileTypeFilter(FileType.File);
    try {
      FileOperation.copy(src, dst, filter);
    } catch (Exception e) {
      NG(e);
      e.printStackTrace();
    }
    
    TRUE(src.exists());
    TRUE(dst.exists());
  }

  public void copy_withFilter002()
  {
    MSG("̃fBNg𓯂fBNgɕʖŃRs[ꍇB");

    File src = DIRS[5];
    File dst = new File(DIRS[2], "sss");

    TRUE(src.exists());
    TRUE(src.isDirectory());
    TRUE(src.listFiles().length == 0);
    FALSE(dst.exists());

    FileFilter filterX = new FileTypeFilter(FileType.File);
    try {
      FileOperation.copy(src, dst, filterX);
    } catch (Exception e) {
      NG(e);
    }

    TRUE(src.exists());
    TRUE(src.isDirectory());
    TRUE(src.listFiles().length == 0);
    FALSE(dst.exists());

    FileFilter filter = new FileTypeFilter(FileType.Directory);
    try {
      FileOperation.copy(src, dst, filter);
    } catch (Exception e) {
      NG(e);
    }

    TRUE(src.exists());
    TRUE(src.isDirectory());
    TRUE(src.listFiles().length == 0);
    TRUE(dst.exists());
    TRUE(dst.isDirectory());
    TRUE(dst.listFiles().length == 0);
  }

  public void copy_withFilter003()
  {
    MSG("̃fBNgʂ̃fBNgɃRs[ꍇB");

    File src = DIRS[5];
    File dst = new File(DIRS[3], "f5");

    TRUE(src.exists());
    TRUE(src.isDirectory());
    TRUE(src.listFiles().length == 0);
    FALSE(dst.exists());

    FileFilter filterX = new FileTypeFilter(FileType.File);
    try {
      FileOperation.copy(src, dst, filterX);
    } catch (Exception e) {
      NG(e);
    }

    TRUE(src.exists());
    TRUE(src.isDirectory());
    TRUE(src.listFiles().length == 0);
    FALSE(dst.exists());

    FileFilter filter = new FileTypeFilter(FileType.Directory);
    try {
      FileOperation.copy(src, dst, filter);
    } catch (Exception e) {
      NG(e);
    }

    TRUE(src.exists());
    TRUE(src.isDirectory());
    TRUE(src.listFiles().length == 0);
    TRUE(dst.exists());
    TRUE(dst.isDirectory());
    TRUE(dst.listFiles().length == 0);
  }

  public void copy_withFilter004()
  {
    MSG("łȂfBNgRs[悤ƂꍇB");

    File src = DIRS[3];
    File dst = new File(DIRS[2], "xxx");

    TRUE(src.exists());
    TRUE(src.isDirectory());
    TRUE(src.listFiles().length > 0);
    FALSE(dst.exists());

    FileFilter filter = new FileTypeFilter(FileType.Directory);
    try {
      FileOperation.copy(src, dst, filter);
    } catch (DirectoryNotEmptyException e) {
      EQUAL(e.getPath(), src.getPath());
    } catch (Exception e) {
      NG(e);
    }

    TRUE(src.exists());
    TRUE(src.isDirectory());
    TRUE(src.listFiles().length > 0);
    FALSE(dst.exists());
  }

  public void copy_withFilter005()
  {
    MSG("Rs[̃t@Ciʏ̃t@Cjɑ݂ꍇB");

    File src = FILES[1];
    File dst = FILES[6];

    TRUE(src.exists());
    FALSE(src.isDirectory());
    TRUE(dst.exists());
    FALSE(dst.isDirectory());

    FileFilter filter = new FileTypeFilter(FileType.File);
    try {
      FileOperation.copy(src, dst, filter);
      NG();
    } catch (FileAlreadyExistsException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    src = FILES[1];
    dst = FILES[2];

    TRUE(src.exists());
    FALSE(src.isDirectory());
    TRUE(dst.exists());
    FALSE(dst.isDirectory());

    try {
      FileOperation.copy(src, dst, filter);
      NG();
    } catch (FileAlreadyExistsException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    src = DIRS[5];
    dst = FILES[2];

    TRUE(src.exists());
    TRUE(src.isDirectory());
    TRUE(dst.exists());
    FALSE(dst.isDirectory());

    filter = new FileTypeFilter(FileType.Directory);
    try {
      FileOperation.copy(src, dst, filter);
      NG();
    } catch (FileAlreadyExistsException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void copy_withFilter006()
  {
    MSG("Rs[̃t@CifBNgjɑ݂ꍇB");

    File src = DIRS[5];
    File dst = DIRS[3];

    TRUE(src.exists());
    TRUE(src.isDirectory());
    TRUE(dst.exists());
    TRUE(dst.isDirectory());

    FileFilter filter = new FileTypeFilter(FileType.Directory);
    try {
      FileOperation.copy(src, dst, filter);
      NG();
    } catch (FileAlreadyExistsException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    src = FILES[1];
    dst = DIRS[5];

    TRUE(src.exists());
    FALSE(src.isDirectory());
    TRUE(dst.exists());
    TRUE(dst.isDirectory());

    filter = new FileTypeFilter(FileType.File);
    try {
      FileOperation.copy(src, dst, filter);
      NG();
    } catch (FileAlreadyExistsException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void copy_withFilter007()
  {
    MSG("k̏ꍇB");

    File f = FILES[1];

    FileFilter filter = new FileTypeFilter(FileType.File);
    try {
      FileOperation.copy(null, f, filter);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    try {
      FileOperation.copy(f, null, filter);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    try {
      FileOperation.copy(f, f, null);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void copy_withFilter008()
  {
    MSG("Rs[̃t@CLbNĂꍇB");

    File src = FILES[1];
    File dst = new File(WORKDIR, "ffff");

    TRUE(src.exists());
    FALSE(src.isDirectory());
    FALSE(dst.exists());

    try {
      FileOperationTest.lockFile("shared", src);
      Thread.sleep(1000);
    } catch (Exception e) {
      NG(e);
    }

    FileFilter filter = new FileTypeFilter(FileType.File);
    try {
      FileOperation.copy(src, dst, filter);
      Thread.sleep(1000);
    } catch (Exception e) {
      NG(e);
    } finally {
      FileOperationTest.releaseLock();
    }

    TRUE(src.exists());
    FALSE(src.isDirectory());
    TRUE(dst.exists());
    FALSE(dst.isDirectory());
  }

  public void copy_withFilter009()
  {
    MSG("Rs[̃t@CrbNĂꍇB");

    File src = FILES[1];
    File dst = new File(WORKDIR, "gggg");

    TRUE(src.exists());
    FALSE(src.isDirectory());
    FALSE(dst.exists());

    try {
      FileOperationTest.lockFile("exclusive", src);
      Thread.sleep(1000);
    } catch (Exception e) {
      NG(e);
    }

    FileFilter filter = new FileTypeFilter(FileType.File);
    try {
      FileOperation.copy(src, dst, filter);
      Thread.sleep(1000);
      NG();
    } catch (IOException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    } finally {
      FileOperationTest.releaseLock();
    }

    TRUE(src.exists());
    FALSE(src.isDirectory());
    FALSE(dst.exists());
  }

  public void copy_withFilter010()
  {
    MSG("Rs[̃t@C݂ȂꍇB");

    File src = new File(WORKDIR, "aaa");
    File dst = new File(WORKDIR, "bbb");
    FileFilter filter = new FileTypeFilter(FileType.Directory);

    try {
      FileOperation.copy(src, dst, filter);
      NG();
    } catch (FileNotFoundException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void copyRecursive()
  {
    MSG("łȂfBNgRs[ꍇB");

    File src = DIRS[0];
    File dst = new File(WORKDIR, "aaa/bbb/ccc");

    for (File d : DIRS) TRUE(d.exists());
    for (File f : FILES) TRUE(f.exists());
    FALSE(new File(WORKDIR, "aaa").exists());

    try {
      FileOperation.copyRecursive(src, dst);
    } catch (Exception e) {
      NG(e);
    }

    for (File d : DIRS) TRUE(d.exists());
    for (File f : FILES) TRUE(f.exists());
    for (File d : DIRS) {
      try {
        TRUE(FileOperation.getDestinationFile(d, src, dst).exists());
      } catch (Exception e) {
        NG(e);
      }
    }
    for (File f : FILES) {
      try {
        TRUE(FileOperation.getDestinationFile(f, src, dst).exists());
      } catch (Exception e) {
        NG(e);
      }
    }
  }

  public void copyRecursive_withFilter()
  {
    MSG("łȂfBNgRs[ꍇB");

    File src = DIRS[0];
    File dst = new File(WORKDIR, "aaa/bbb/ccc");

    for (File d : DIRS) TRUE(d.exists());
    for (File f : FILES) TRUE(f.exists());
    FALSE(new File(WORKDIR, "aaa").exists());

    FileFilter filter = new FileTypeFilter(FileType.File);
    try {
      FileOperation.copyRecursive(src, dst, filter);
    } catch (Exception e) {
      NG(e);
    }

    for (File d : DIRS) TRUE(d.exists());
    for (File f : FILES) TRUE(f.exists());
    for (File d : DIRS) {
      // DIRS[5] is an empty directory, so was picked out by the filter.
      if (d.equals(DIRS[5])) continue;

      try {
        OK(d.getPath());
        TRUE(FileOperation.getDestinationFile(d, src, dst).exists());
      } catch (Exception e) {
        NG(e);
      }
    }
    for (File f : FILES) {
      try {
        TRUE(FileOperation.getDestinationFile(f, src, dst).exists());
      } catch (Exception e) {
        NG(e);
      }
    }
  }

  public void copyRecursive_withFilter001()
  {
    MSG("łȂfBNgRs[ꍇB");

    File src = DIRS[0];
    File dst = new File(WORKDIR, "aaa/bbb/ccc");

    for (File d : DIRS) TRUE(d.exists());
    for (File f : FILES) TRUE(f.exists());
    FALSE(new File(WORKDIR, "aaa").exists());

    FileFilter filter = new FileTypeFilter(FileType.Directory);
    try {
      FileOperation.copyRecursive(src, dst, filter);
    } catch (Exception e) {
      NG(e);
    }

    for (File d : DIRS) TRUE(d.exists());
    for (File f : FILES) TRUE(f.exists());
    for (File d : DIRS) {
      try {
        OK(d.getPath());
        TRUE(FileOperation.getDestinationFile(d, src, dst).exists());
      } catch (Exception e) {
        NG(e);
      }
    }
    for (File f : FILES) {
      try {
        FALSE(FileOperation.getDestinationFile(f, src, dst).exists());
      } catch (Exception e) {
        NG(e);
      }
    }
  }
}

