/*
 * FileNewerThanTargetFilter class.
 *
 * Copyright (C) 2007 SATOH Takayuki All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package ts.util.file;

import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.util.List;
import java.util.LinkedList;
import ts.util.Duo;
import ts.util.Trio;
import ts.util.text.StringOperation;

/**
 * ^[Qbg̃t@CŏIXVVƂƂ
 * t@CEtB^ENXB
 * <br>
 * ^[Qbg̃t@ĆA{@link #accept(File) accept(File)}\bhŎw
 * ꂽt@C̃pXÃIuWFNgɐݒ肳ꂽϊ@ɂĕϊ
 * ꂽpXɂē肳B
 * <br>
 * t@CEpX̕ΐÃIuWFNgɒǉԂɏ]āAϊ@
 * KpB
 * <br>
 * ^[Qbg̃t@CƁÃt@C̊ԂōŏIXV̔rsA
 * ̃t@C̕Vꍇɐ^ƂB
 * <br>
 * ^[Qbg̃t@C݂Ȃꍇ͐^ԂB
 *
 * @author  V.
 * @version $Revision: 1.1.1.1 $, $Date: 2010-10-16 00:03:46 $
 */
public class FileNewerThanTargetFilter implements FileFilter
{
  /** t@CEpX̕ϊɎgpϊ̊fBNgB */
  private File baseDir_ ;

  /** t@CEpX̕ϊɎgpϊ̊fBNgB */
  private File destDir_ ;

  /** t@C̕ϊɎgp鐳K\ƒȗgi[郊XgB */
  private List<Trio<String,String,Boolean>> nameReplaceLst_ =
    new LinkedList<Trio<String,String,Boolean>>();

  /** fBNgEpX̕ϊɎgp鐳K\ƒȗgi[
   * XgB */
  private List<Trio<String,String,Boolean>> dirReplaceLst_ =
    new LinkedList<Trio<String,String,Boolean>>();

  /**
   * ^[QbgEt@C̃pX߂邽߂Ɏgpϊƕϊ
   * fBNgɂƂRXgN^B
   * <br>
   * ϊƂ́A{@link #accept(File) accept(File)}\bhŎw肳t@C
   * ̂ƂłAϊƂ̓^[QbgEt@ĈƂłB
   * <br>
   * ϊ̊fBNg݂Ȃꍇ́A{@link #accept(File) 
   * accept(File)}\bhsʂUƂȂB
   *
   * @param  baseDir ϊ̊fBNgB
   * @param  destDir ϊ̊fBNgB
   * @throws AssertionError k̏ꍇifobOE[ĥ݁jB
   */
  public FileNewerThanTargetFilter(File baseDir, File destDir)
  {
    assert (baseDir != null) : "@param:baseDir is null.";
    assert (destDir != null) : "@param:destDir is null.";

    baseDir_ = baseDir;
    destDir_ = destDir;
  }

  /**
   * t@C̕ϊɎgp鐳K\ƒuǉB
   * <br>
   * K\ɍv̕ΐAŏɍv̂ϊB
   *
   * @param  regex K\B
   * @param  replacement uB
   * @throws AssertionError k̏ꍇifobOE[ĥ݁jB
   */
  public void addNameReplaceFirst(String regex, String replacement)
  {
    assert (regex != null) : "@param:regex is null.";
    assert (replacement != null) : "@param:replacement is null.";

    nameReplaceLst_.add(new Trio<String,String,Boolean>(
      regex, replacement, Boolean.FALSE));
  }

  /**
   * t@C̕ϊɎgp鐳K\ƒuǉB
   * <br>
   * RɐK\ɍvSĂ̕ϊB
   *
   * @param  regex K\B
   * @param  replacement uB
   * @throws AssertionError k̏ꍇifobOE[ĥ݁jB
   */
  public void addNameReplaceAll(String regex, String replacement)
  {
    assert (regex != null) : "@param:regex is null.";
    assert (replacement != null) : "@param:replacement is null.";

    nameReplaceLst_.add(new Trio<String,String,Boolean>(
      regex, replacement, Boolean.TRUE));
  }

  /**
   * fBNgEpX̕ϊɎgp鐳K\ƒuǉB
   * <br>
   * K\ɍv̕ΐAŏɍv̂ϊB
   * <br>
   * ϊpX́Aϊ̊fBNg̃pX牺݂̂̕łB
   *
   * @param  regex K\B
   * @param  replacement uB
   * @throws AssertionError k̏ꍇifobOE[ĥ݁jB
   */
  public void addDirReplaceFirst(String regex, String replacement)
  {
    assert (regex != null) : "@param:regex is null.";
    assert (replacement != null) : "@param:replacement is null.";

    dirReplaceLst_.add(new Trio<String,String,Boolean>(
      regex, replacement, Boolean.FALSE));
  }

  /**
   * fBNgEpX̕ϊɎgp鐳K\ƒuǉB
   * <br>
   * K\ɍvSĂ̕ϊB
   * <br>
   * ϊpX́Aϊ̊fBNg̃pX牺݂̂̕łB
   *
   * @param  regex K\B
   * @param  replacement uB
   * @throws AssertionError k̏ꍇifobOE[ĥ݁jB
   */
  public void addDirReplaceAll(String regex, String replacement)
  {
    assert (regex != null) : "@param:regex is null.";
    assert (replacement != null) : "@param:replacement is null.";

    dirReplaceLst_.add(new Trio<String,String,Boolean>(
      regex, replacement, Boolean.TRUE));
  }

  /**
   * w肳ꂽt@C̍ŏIXVA^[QbgEt@C̍ŏIXV
   * Vǂ𔻒肷B
   * <br>
   * ^[QbgEt@C̃pX́Ãt@C̃pXɁÃIuWFNg
   * ɐݒ肳Ăϊ@gċ߂B
   * <br>
   * ^[QbgEt@C݂Ȃꍇ<tt>true</tt>ԂB
   * <br>
   * ̃t@C݂Ȃꍇ<tt>false</tt>ԂB
   * <br>
   * ^[QbgEt@C̃pXւ̕ϊɎsꍇ<tt>false</tt>ԂB
   *
   * @param file  Ώۂ̃t@CB
   * @return w肳ꂽt@C̍ŏIXV^[QbgEt@CV
   *         A^[QbgEt@C݂Ȃꍇ<tt>true</tt>ԂB
   * @throws AssertionError k̏ꍇifobOE[ĥ݁jB
   */
  public boolean accept(File file)
  {
    assert (file != null) : "@param:file is null.";

    try {
      if (! file.exists()) {
        return false;
      }

      File target = getTargetFile(file);
      if (! target.exists()) {
        return true;
      }
      else if (target.lastModified() >= file.lastModified()) {
        return false;
      }
      else {
        return true;
      }
    }
    catch (Exception e) {
      return false;
    }
  }

  /**
   * ^[QbgEt@C߂B
   *
   * @param  file Ώۂ̃t@CB
   * @throws IOException ^[QbgEt@C߂邱ƂɎsꍇB
   * @throws AssertionError k̏ꍇB
   */
  protected File getTargetFile(File file) throws IOException
  {
    Duo<String,String> duo = getRelativePaths(file, baseDir_);

    String name = duo.getSecond();
    for (Trio<String,String,Boolean> trio : nameReplaceLst_) {
      if (trio.getThird().booleanValue()) {
        name = name.replaceAll(trio.getFirst(), trio.getSecond());
      }
      else {
        name = name.replaceFirst(trio.getFirst(), trio.getSecond());
      }
    }

    String dir = duo.getFirst();
    for (Trio<String,String,Boolean> trio : dirReplaceLst_) {
      if (trio.getThird().booleanValue()) {
        dir = dir.replaceAll(trio.getFirst(), trio.getSecond());
      }
      else {
        dir = dir.replaceFirst(trio.getFirst(), trio.getSecond());
      }
    }

    File target = destDir_;
    if (! StringOperation.isEmpty(dir)) {
      target = new File(target, dir);
    }
    if (! StringOperation.isEmpty(name)) {
      target = new File(target, name);
    }
    return target;
  }

  /**
   * w肳ꂽt@ĆAt@C̑΃pX擾A
   * efBNgEpXƃt@CɕĎ擾B
   * <br>
   * Ԃ{@link ts.util.Duo Duo}IuWFNg
   * PvfɐefBNgEpXAQvfɃt@Ci[B
   *
   * @param  file t@CB
   * @param  baseFile t@CB
   * @return ΃pX̐efBNgEpXƃt@Ci[
   *           {@link ts.util.Duo Duo}IuWFNgB
   * @throws IOException ΃pX̎擾ɎsꍇB
   * @throws AssertionError k̏ꍇB
   */
  protected Duo<String,String> getRelativePaths(File file, File baseFile)
    throws IOException
  {
    assert (file != null) : "@param:file is null.";
    assert (baseFile != null) : "@param:baseFile is null.";

    String path = file.getCanonicalPath();
    String basePath = baseFile.getCanonicalPath();

    if (StringOperation.startsWith(path, basePath)) {
      int basePathLen = StringOperation.length(basePath);
      String relPath = StringOperation.substring(path, basePathLen);

      if (StringOperation.isEmpty(relPath)) {
        return new Duo<String,String>("", "");
      }
      else if (StringOperation.startsWith(relPath, File.separator)) {
        int sepLen = StringOperation.length(File.separator);
        int endIndex = StringOperation.lastIndexOf(relPath, File.separator);
        if (endIndex < 0) {
          return new Duo<String,String>("", relPath);
        }
        else if (endIndex == 0) {
          return new Duo<String,String>(
            "", StringOperation.substring(relPath, sepLen));
        }
        else {
          return new Duo<String,String>(
            StringOperation.substring(relPath, sepLen, endIndex),
            StringOperation.substring(relPath, endIndex + sepLen)
          );
        }
      }
    }

    throw new IllegalArgumentException(
      "@param:file is not a child path of @param:baseFile.");
  }
}

