/*
 * StringCutterTest class.
 *
 * Copyright (C) 2007 SATOH Takayuki All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package ts.util.text;

import ts.tester.function.coverage.FunctionTester;
import ts.tester.function.print.*;
import java.util.List;

/**
 * {@link ts.util.text.StringCutter StringCutter}NX̋@\NXB
 *
 * @author  V.
 * @version $Revision: 1.1.1.1 $, $Date: 2010-10-16 00:03:53 $
 */
public class StringCutterTest extends FunctionTester
{
  public static void main(String[] args)
  {
    try {
      PrinterGroup group = new PrinterGroup();
      group.addPrinter(new ConsolePrinter());
      group.addPrinter(new HtmlPrinter("SATOH Takayuki"));
      setPrinter(group);

      run(StringCutterTest.class, (args.length == 0) ? null : args[0]);
    }
    catch (Exception e) {
      e.printStackTrace();
    }
  }

  protected void preTesting()
  {
    MSG("StringCutter NX̋@\sB");
  }
  
  /* -- test case -- */

  public void constructor()
  {
    MSG("ftHgERXgN^̊mFB");

    StringCutter cutter = new StringCutter();
    NOTNULL(cutter);
  }

  public void split_string_chStr()
  {
    MSG("؂蕶ŕB");

    List<String> lst;

    lst = StringCutter.split("ABCDEFGHIJKLMNOPQRSTUVWXYZ", ","); 
    EQUAL(lst.size(), 1);
    EQUAL(lst.get(0), "ABCDEFGHIJKLMNOPQRSTUVWXYZ");

    lst = StringCutter.split("ABCDEFGHIJKLM,NOPQRSTUVWXYZ", ","); 
    EQUAL(lst.size(), 2);
    EQUAL(lst.get(0), "ABCDEFGHIJKLM");
    EQUAL(lst.get(1), "NOPQRSTUVWXYZ");

    lst = StringCutter.split(",ABCDEFGHIJKLMNOPQRSTUVWXYZ,", ","); 
    EQUAL(lst.size(), 3);
    EQUAL(lst.get(0), "");
    EQUAL(lst.get(1), "ABCDEFGHIJKLMNOPQRSTUVWXYZ");
    EQUAL(lst.get(2), "");

    lst = StringCutter.split("AB/CDEFGH//IJKLMNO///PQRSTUVW/XYZ", "/"); 
    EQUAL(lst.size(), 8);
    EQUAL(lst.get(0), "AB");
    EQUAL(lst.get(1), "CDEFGH");
    EQUAL(lst.get(2), "");
    EQUAL(lst.get(3), "IJKLMNO");
    EQUAL(lst.get(4), "");
    EQUAL(lst.get(5), "");
    EQUAL(lst.get(6), "PQRSTUVW");
    EQUAL(lst.get(7), "XYZ");

    lst = StringCutter.split("", "/");
    EQUAL(lst.size(), 1);
    EQUAL(lst.get(0), "");
  }

  public void split_string_chStr_1()
  {
    MSG("k̏ꍇB");

    try {
      StringCutter.split(null, ",");
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void split_string_string()
  {
    MSG("Aw肳ꂽ؂蕶ŕB");

    String str;
    List<String> lst;

    str = "";
    lst = StringCutter.split(str, "<br>");
    EQUAL(lst.size(), 1);
    EQUAL(lst.get(0), "");

    str = "<br><br>";
    lst = StringCutter.split(str, "<br>");
    EQUAL(lst.size(), 3);
    EQUAL(lst.get(0), "");
    EQUAL(lst.get(1), "");
    EQUAL(lst.get(2), "");

    str = "<br><br><br><br>";
    lst = StringCutter.split(str, "<br>");
    EQUAL(lst.size(), 5);
    EQUAL(lst.get(0), "");
    EQUAL(lst.get(1), "");
    EQUAL(lst.get(2), "");
    EQUAL(lst.get(3), "");
    EQUAL(lst.get(4), "");

    str = "b";
    lst = StringCutter.split(str, "<br>");
    EQUAL(lst.size(), 1);
    EQUAL(lst.get(0), "b");

    str = "";
    lst = StringCutter.split(str, "<br>");
    EQUAL(lst.size(), 1);
    EQUAL(lst.get(0), "");
  }

  public void split_string_string_1()
  {
    MSG("k̏ꍇB");

    try {
      StringCutter.split(null, ",");
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    try {
      StringCutter.split("", (String) null);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    } 
  }

  public void split_string_string_2()
  {
    MSG("؂蕶󕶎̏ꍇB");

    try {
      StringCutter.split("", "");
      NG();
    } catch (IllegalArgumentException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void splitIgnoreCase_string_string()
  {
    MSG("Aw肵؂蕶ő啶lɕB");

    String str;
    List<String> lst;

    str = "KaKiKuKeKo";
    lst = StringCutter.splitIgnoreCase(str, "<br>");
    EQUAL(lst.size(), 1);
    EQUAL(lst.get(0), "KaKiKuKeKo");

    str = "<BR>KaKiKuKeKo<Br>";
    lst = StringCutter.splitIgnoreCase(str, "<br>");
    EQUAL(lst.size(), 3);
    EQUAL(lst.get(0), "");
    EQUAL(lst.get(1), "KaKiKuKeKo");
    EQUAL(lst.get(2), "");

    str = "<bR><br>KaKiKuKeKo<BR><BR>";
    lst = StringCutter.splitIgnoreCase(str, "<br>");
    EQUAL(lst.size(), 5);
    EQUAL(lst.get(0), "");
    EQUAL(lst.get(1), "");
    EQUAL(lst.get(2), "KaKiKuKeKo");
    EQUAL(lst.get(3), "");
    EQUAL(lst.get(4), "");

    str = "b";
    lst = StringCutter.splitIgnoreCase(str, "<br>");
    EQUAL(lst.size(), 1);
    EQUAL(lst.get(0), "b");

    str = "";
    lst = StringCutter.splitIgnoreCase(str, "<br>");
    EQUAL(lst.size(), 1);
    EQUAL(lst.get(0), "");
  }

  public void splitIgnoreCase_string_string_1()
  {
    MSG("k̏ꍇB");

    try {
      StringCutter.splitIgnoreCase(null, ",");
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    try {
      StringCutter.splitIgnoreCase("", (String) null);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    } 
  }

  public void splitIgnoreCase_string_string_2()
  {
    MSG("؂蕶󕶎̏ꍇB");

    try {
      StringCutter.splitIgnoreCase("", "");
      NG();
    } catch (IllegalArgumentException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void split_string_chars()
  {
    MSG("𕡐̋؂蕶ŕB");

    List lst;
    String[] delims = { ",", "/" };

    lst = StringCutter.split("ABCDEFGHIJKLMNOPQRSTUVWXYZ", delims); 
    EQUAL(lst.size(), 1);
    EQUAL(lst.get(0), "ABCDEFGHIJKLMNOPQRSTUVWXYZ");

    lst = StringCutter.split("ABCDEFGHIJKLM,NOPQRSTUVWXYZ", delims);
    EQUAL(lst.size(), 2);
    EQUAL(lst.get(0), "ABCDEFGHIJKLM");
    EQUAL(lst.get(1), "NOPQRSTUVWXYZ");

    lst = StringCutter.split("ABC/DEFGHIJKLMNOP/QRSTUVWXYZ", delims);
    EQUAL(lst.size(), 3);
    EQUAL(lst.get(0), "ABC");
    EQUAL(lst.get(1), "DEFGHIJKLMNOP");
    EQUAL(lst.get(2), "QRSTUVWXYZ");

    lst = StringCutter.split("/ABCDEFGHIJKLMNOPQRSTUVWXYZ,", delims); 
    EQUAL(lst.size(), 3);
    EQUAL(lst.get(0), "");
    EQUAL(lst.get(1), "ABCDEFGHIJKLMNOPQRSTUVWXYZ");
    EQUAL(lst.get(2), "");

    lst = StringCutter.split("/AB/C,DEFG,/HIJKL,MNO/,P/QRSTU/VWXYZ,", delims); 
    EQUAL(lst.size(), 12);
    EQUAL(lst.get(0), "");
    EQUAL(lst.get(1), "AB");
    EQUAL(lst.get(2), "C");
    EQUAL(lst.get(3), "DEFG");
    EQUAL(lst.get(4), "");
    EQUAL(lst.get(5), "HIJKL");
    EQUAL(lst.get(6), "MNO");
    EQUAL(lst.get(7), "");
    EQUAL(lst.get(8), "P");
    EQUAL(lst.get(9), "QRSTU");
    EQUAL(lst.get(10), "VWXYZ");
    EQUAL(lst.get(11), "");

    lst = StringCutter.split("", delims);
    EQUAL(lst.size(), 1);
    EQUAL(lst.get(0), "");
  }

  public void split_string_chars_1()
  {
    MSG("k̏ꍇB");

    String[] delims = { ",", "/" };
    try {
      StringCutter.split(null, delims);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    try {
      StringCutter.split("", (String[]) null);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void split_string_strings()
  {
    MSG("A̋؂蕶ŕB");

    String str;
    String[] delims = { "<p>", "<br>", "</p>" };
    List<String> lst;

    str = "";
    lst = StringCutter.split(str, delims);
    EQUAL(lst.size(), 1);
    EQUAL(lst.get(0), "");

    str = "<br><br>";
    lst = StringCutter.split(str, delims);
    EQUAL(lst.size(), 3);
    EQUAL(lst.get(0), "");
    EQUAL(lst.get(1), "");
    EQUAL(lst.get(2), "");

    str = "<p><br><br></p>";
    lst = StringCutter.split(str, delims);
    EQUAL(lst.size(), 5);
    EQUAL(lst.get(0), "");
    EQUAL(lst.get(1), "");
    EQUAL(lst.get(2), "");
    EQUAL(lst.get(3), "");
    EQUAL(lst.get(4), "");

    str = "<p><bq><br></p>";
    lst = StringCutter.split(str, delims);
    EQUAL(lst.size(), 4);
    EQUAL(lst.get(0), "");
    EQUAL(lst.get(1), "<bq>");
    EQUAL(lst.get(2), "");
    EQUAL(lst.get(3), "");

    str = "b";
    lst = StringCutter.split(str, delims);
    EQUAL(lst.size(), 1);
    EQUAL(lst.get(0), "b");

    str = "";
    lst = StringCutter.split(str, delims);
    EQUAL(lst.size(), 1);
    EQUAL(lst.get(0), "");
  }

  public void split_string_strings_1()
  {
    MSG("k̏ꍇB");

    String str;
    String delims[] = { "<p>", "<br>", "</p>" };
    List<String> lst;

    try {
      StringCutter.split(null, delims);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    try {
      StringCutter.split("", (String[]) null);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    } 
  }

  public void split_string_strings_2()
  {
    MSG("؂蕶̔z̒Ƀk̗vfĂꍇB");

    String str;
    String[] delims = { "<p>", null, "</p>" };

    str = "";
    try {
      StringCutter.split(str, delims);
      NG();
    } catch (IllegalArgumentException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void split_string_strings_3()
  {
    MSG("؂蕶̔z̒ɋ󕶎̗vfĂꍇB");

    String str;
    String[] delims = { "<p>", "", "</p>" };

    str = "";
    try {
      StringCutter.split(str, delims);
      NG();
    } catch (IllegalArgumentException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void splitIgnoreCase_string_strings()
  {
    MSG("A̋؂蕶i啶͖jŕB");

    String str;
    String[] delims = { "<p>", "<br>", "</p>" };
    List<String> lst;

    str = "KaKiKuKeKo";
    lst = StringCutter.splitIgnoreCase(str, delims);
    EQUAL(lst.size(), 1);
    EQUAL(lst.get(0), "KaKiKuKeKo");

    str = "<Br>KaKiKuKeKo<bR>";
    lst = StringCutter.splitIgnoreCase(str, delims);
    EQUAL(lst.size(), 3);
    EQUAL(lst.get(0), "");
    EQUAL(lst.get(1), "KaKiKuKeKo");
    EQUAL(lst.get(2), "");

    str = "<P><bq>KaKiKuKeKo<BR></p>";
    lst = StringCutter.splitIgnoreCase(str, delims);
    EQUAL(lst.size(), 4);
    EQUAL(lst.get(0), "");
    EQUAL(lst.get(1), "<bq>KaKiKuKeKo");
    EQUAL(lst.get(2), "");
    EQUAL(lst.get(3), "");

    str = "<P><br>KaKiKuKeKo<BR></p>";
    lst = StringCutter.splitIgnoreCase(str, delims);
    EQUAL(lst.size(), 5);
    EQUAL(lst.get(0), "");
    EQUAL(lst.get(1), "");
    EQUAL(lst.get(2), "KaKiKuKeKo");
    EQUAL(lst.get(3), "");
    EQUAL(lst.get(4), "");

    str = "b";
    lst = StringCutter.splitIgnoreCase(str, delims);
    EQUAL(lst.size(), 1);
    EQUAL(lst.get(0), "b");

    str = "";
    lst = StringCutter.splitIgnoreCase(str, delims);
    EQUAL(lst.size(), 1);
    EQUAL(lst.get(0), "");
  }

  public void splitIgnoreCase_string_strings_1()
  {
    MSG("k̏ꍇB");

    String str;
    String delims[] = { "<p>", "<br>", "</p>" };
    List<String> lst;

    try {
      StringCutter.splitIgnoreCase(null, delims);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    try {
      StringCutter.splitIgnoreCase("", (String[]) null);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    } 
  }

  public void splitIgnoreCase_string_strings_2()
  {
    MSG("؂蕶̔z̒Ƀk̗vfĂꍇB");

    String str;
    String[] delims = { "<p>", null, "</p>" };

    str = "";
    try {
      StringCutter.splitIgnoreCase(str, delims);
      NG();
    } catch (IllegalArgumentException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void splitIgnoreCase_string_strings_3()
  {
    MSG("؂蕶̔z̒ɋ󕶎̗vfĂꍇB");

    String str;
    String[] delims = { "<p>", "", "</p>" };

    str = "";
    try {
      StringCutter.splitIgnoreCase(str, delims);
      NG();
    } catch (IllegalArgumentException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void split_string_string_9()
  {
    MSG("؂蕶AđꍇB");

    List lst;

    lst = StringCutter.split("/////", "/"); 
    EQUAL(lst.size(), 6);
    EQUAL(lst.get(0), "");
    EQUAL(lst.get(1), "");
    EQUAL(lst.get(2), "");
    EQUAL(lst.get(3), "");
    EQUAL(lst.get(4), "");
    EQUAL(lst.get(5), "");

    lst = StringCutter.split("/////", "//"); 
    EQUAL(lst.size(), 3);
    EQUAL(lst.get(0), "");
    EQUAL(lst.get(1), "");
    EQUAL(lst.get(2), "/");
  }

  public void splitIgnoreCase_string_string_9()
  {
    MSG("؂蕶AđꍇB");

    List lst;

    lst = StringCutter.splitIgnoreCase("/////", "/"); 
    EQUAL(lst.size(), 6);
    EQUAL(lst.get(0), "");
    EQUAL(lst.get(1), "");
    EQUAL(lst.get(2), "");
    EQUAL(lst.get(3), "");
    EQUAL(lst.get(4), "");
    EQUAL(lst.get(5), "");

    lst = StringCutter.splitIgnoreCase("/////", "//"); 
    EQUAL(lst.size(), 3);
    EQUAL(lst.get(0), "");
    EQUAL(lst.get(1), "");
    EQUAL(lst.get(2), "/");
  }

  public void split_string_strings_9()
  {
    MSG("؂蕶AđꍇB");

    List lst;

    String[] delims = { "\\", ".", "/" };
    lst = StringCutter.split("/////", delims); 
    EQUAL(lst.size(), 6);
    EQUAL(lst.get(0), "");
    EQUAL(lst.get(1), "");
    EQUAL(lst.get(2), "");
    EQUAL(lst.get(3), "");
    EQUAL(lst.get(4), "");
    EQUAL(lst.get(5), "");

    String[] delims2 = { "\\", ".", "//" };
    lst = StringCutter.split("/////", delims2); 
    EQUAL(lst.size(), 3);
    EQUAL(lst.get(0), "");
    EQUAL(lst.get(1), "");
    EQUAL(lst.get(2), "/");
  }

  public void splitIgnoreCase_string_strings_9()
  {
    MSG("؂蕶AđꍇB");

    List lst;

    String[] delims = { "\\", ".", "/" };
    lst = StringCutter.splitIgnoreCase("/////", delims); 
    EQUAL(lst.size(), 6);
    EQUAL(lst.get(0), "");
    EQUAL(lst.get(1), "");
    EQUAL(lst.get(2), "");
    EQUAL(lst.get(3), "");
    EQUAL(lst.get(4), "");
    EQUAL(lst.get(5), "");

    String[] delims2 = { "\\", ".", "//" };
    lst = StringCutter.splitIgnoreCase("/////", delims2); 
    EQUAL(lst.size(), 3);
    EQUAL(lst.get(0), "");
    EQUAL(lst.get(1), "");
    EQUAL(lst.get(2), "/");
  }
}

