package ts.query;

import ts.tester.UnitTest;
import ts.util.resource.*;
import ts.util.*;
import java.io.*;
import java.util.*;

public class DefaultQueryTransactionTest extends UnitTest
{
  public static void main(String args[])
  {
    run(DefaultQueryTransactionTest.class, args);
  }

  static StringWriter SW = null;
  static PrintWriter PW = null;
  static String OUTPUT_LOG() {
    return SW.toString();
  }
  static void OPEN_LOG() {
    SW = new StringWriter();
    PW = new PrintWriter(SW);
  }
  static void CLOSE_LOG() {
    PW.close();
  }
  static void CLEAR_LOG() {
    CLOSE_LOG();
    OPEN_LOG();
  }

  @Override
  protected void preInvocation(String method)
  {
    OPEN_LOG();
  }

  @Override
  protected void postInvocation(String method)
  {
    CLOSE_LOG();
  }

  static class MyConnection implements QueryConnection {
    private long limitTm = -1L;
    private final String connId;
    public MyConnection(String connId) { this.connId = connId; }
    @Override
    public String getId() { return this.connId; }
    @Override
    public void setLimitTimeMillis(long tm) { this.limitTm = tm; }
    @Override
    public long getLimitTimeMillis() { return this.limitTm; }
    @Override
    public Query getQuery(Resource res) { return null; }
    @Override
    public void open() { PW.print("[" + connId + " is opened!]"); }
    @Override
    public void commit() { PW.print("[" + connId + " is committed!]"); }
    @Override
    public void rollback() { PW.print("[" + connId + " is rollbacked!]"); }
    @Override
    public void close() { PW.print("[" + connId + " is closed!]"); }
  }

  static class MyConnectionFactory implements QueryConnectionFactory {
    public QueryConnection create(String connId, QueryConfig config)
    throws ReasonedException {
      if ("FAIL".equals(connId)) {
        throw new ReasonedException(
          QueryConnectionFactory.Error.FailToCreate, connId);
      }
      return new MyConnection(connId);
    }
  }


  public void constructor()
  {
    MSG("デフォルト・コンストラクタの確認。");

    DefaultQueryTransaction tr = new DefaultQueryTransaction();
    TRUE(tr instanceof QueryTransaction);
    EQUAL(tr.getState(), QueryTransaction.State.CREATED);
    EQUAL(tr.getTimeoutMillis(), -1L);
    EQUAL(tr.getBeginTimeMillis(), -1L);
    EQUAL(tr.getLimitTimeMillis(), -1L);
  }

  public void setTimeoutMillis_timeout()
  {
    MSG("デフォルトのタイムアウト値を設定するメソッドの確認。");

    DefaultQueryTransaction tr;
    long tm0, tm1, tm;

    tr = new DefaultQueryTransaction();
    EQUAL(tr.getState(), QueryTransaction.State.CREATED);
    EQUAL(tr.getTimeoutMillis(), -1L);
    EQUAL(tr.getBeginTimeMillis(), -1L);
    EQUAL(tr.getLimitTimeMillis(), -1L);

    tm0 = System.currentTimeMillis();
    tr.begin();
    tm1 = System.currentTimeMillis();
    EQUAL(tr.getState(), QueryTransaction.State.BEGINED);
    EQUAL(tr.getTimeoutMillis(), -1L);
    TRUE(tm0 <= tr.getBeginTimeMillis() && tr.getBeginTimeMillis() <= tm1);
    tm = tr.getBeginTimeMillis();
    EQUAL(tr.getLimitTimeMillis(), -1L);

    tr.commit();
    EQUAL(tr.getState(), QueryTransaction.State.COMMITTED);
    EQUAL(tr.getTimeoutMillis(), -1L);
    EQUAL(tr.getBeginTimeMillis(), tm);
    EQUAL(tr.getLimitTimeMillis(), -1L);

    tr.end();
    EQUAL(tr.getState(), QueryTransaction.State.ENDED);
    EQUAL(tr.getTimeoutMillis(), -1L);
    EQUAL(tr.getBeginTimeMillis(), tm);
    EQUAL(tr.getLimitTimeMillis(), -1L);


    tr = new DefaultQueryTransaction();
    tr.setTimeoutMillis(100L);
    EQUAL(tr.getState(), QueryTransaction.State.CREATED);
    EQUAL(tr.getTimeoutMillis(), 100L);
    EQUAL(tr.getBeginTimeMillis(), -1L);
    EQUAL(tr.getLimitTimeMillis(), -1L);

    tm0 = System.currentTimeMillis();
    tr.begin();
    tm1 = System.currentTimeMillis();
    EQUAL(tr.getState(), QueryTransaction.State.BEGINED);
    EQUAL(tr.getTimeoutMillis(), 100L);
    TRUE(tm0 <= tr.getBeginTimeMillis() && tr.getBeginTimeMillis() <= tm1);
    tm = tr.getBeginTimeMillis();
    EQUAL(tr.getLimitTimeMillis(), tr.getBeginTimeMillis() + 100L);

    tr.commit();
    EQUAL(tr.getState(), QueryTransaction.State.COMMITTED);
    EQUAL(tr.getTimeoutMillis(), 100L);
    EQUAL(tr.getBeginTimeMillis(), tm);
    EQUAL(tr.getLimitTimeMillis(), tm + 100L);

    tr.end();
    EQUAL(tr.getState(), QueryTransaction.State.ENDED);
    EQUAL(tr.getTimeoutMillis(), 100L);
    EQUAL(tr.getBeginTimeMillis(), tm);
    EQUAL(tr.getLimitTimeMillis(), tm + 100L);
  }

  public void setTimeoutMillis_timeout_BadState()
  {
    MSG("トランザクションの状態が適切でない場合。");

    DefaultQueryTransaction tr;

    tr = new DefaultQueryTransaction();
    EQUAL(tr.getState(), QueryTransaction.State.CREATED);

    tr.begin();
    EQUAL(tr.getState(), QueryTransaction.State.BEGINED);

    try {
      tr.setTimeoutMillis(100L);
      NG();
    } catch (ReasonedRuntimeException e) {
      OK(e);
    }

    tr.commit();
    EQUAL(tr.getState(), QueryTransaction.State.COMMITTED);

    try {
      tr.setTimeoutMillis(100L);
      NG();
    } catch (ReasonedRuntimeException e) {
      OK(e);
    }

    tr.end();
    EQUAL(tr.getState(), QueryTransaction.State.ENDED);

    try {
      tr.setTimeoutMillis(100L);
      NG();
    } catch (ReasonedRuntimeException e) {
      OK(e);
    }


    tr = new DefaultQueryTransaction();
    EQUAL(tr.getState(), QueryTransaction.State.CREATED);

    tr.begin();
    EQUAL(tr.getState(), QueryTransaction.State.BEGINED);

    try {
      tr.setTimeoutMillis(100L);
      NG();
    } catch (ReasonedRuntimeException e) {
      OK(e);
    }

    tr.rollback();
    EQUAL(tr.getState(), QueryTransaction.State.ROLLBACKED);

    try {
      tr.setTimeoutMillis(100L);
      NG();
    } catch (ReasonedRuntimeException e) {
      OK(e);
    }

    tr.end();
    EQUAL(tr.getState(), QueryTransaction.State.ENDED);

    try {
      tr.setTimeoutMillis(100L);
      NG();
    } catch (ReasonedRuntimeException e) {
      OK(e);
    }
  }

  public void begin_timeout()
  {
    MSG("トランザクションを開始するメソッドの確認。");

    DefaultQueryTransaction tr = new DefaultQueryTransaction();
    EQUAL(tr.getState(), QueryTransaction.State.CREATED);
    EQUAL(tr.getTimeoutMillis(), -1L);
    EQUAL(tr.getBeginTimeMillis(), -1L);
    EQUAL(tr.getLimitTimeMillis(), -1L);

    long bgnTm;

    tr.begin(1000L);
    EQUAL(tr.getState(), QueryTransaction.State.BEGINED);
    EQUAL(tr.getTimeoutMillis(), -1L);
    bgnTm = tr.getBeginTimeMillis();
    EQUAL(tr.getLimitTimeMillis(), bgnTm + 1000L);

    try {
      tr.begin(1000L);
      NG();
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), QueryTransaction.Error.IllegalState);
    }

    tr.commit();
    EQUAL(tr.getState(), QueryTransaction.State.COMMITTED);
    EQUAL(tr.getTimeoutMillis(), -1L);
    EQUAL(tr.getBeginTimeMillis(), bgnTm);
    EQUAL(tr.getLimitTimeMillis(), bgnTm + 1000L);

    try {
      tr.begin(1000L);
      NG();
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), QueryTransaction.Error.IllegalState);
    }

    tr.end();
    EQUAL(tr.getState(), QueryTransaction.State.ENDED);
    EQUAL(tr.getTimeoutMillis(), -1L);
    EQUAL(tr.getBeginTimeMillis(), bgnTm);
    EQUAL(tr.getLimitTimeMillis(), bgnTm + 1000L);

    try {
      tr.begin(1000L);
      NG();
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), QueryTransaction.Error.IllegalState);
    }
  }

  public void begin_timeout_Negative()
  {
    MSG("引数が負の場合。");

    DefaultQueryTransaction tr = new DefaultQueryTransaction();
    EQUAL(tr.getState(), QueryTransaction.State.CREATED);
    EQUAL(tr.getTimeoutMillis(), -1L);
    EQUAL(tr.getBeginTimeMillis(), -1L);
    EQUAL(tr.getLimitTimeMillis(), -1L);

    long bgnTm;

    tr.begin(-1000L);
    EQUAL(tr.getState(), QueryTransaction.State.BEGINED);
    EQUAL(tr.getTimeoutMillis(), -1L);
    bgnTm = tr.getBeginTimeMillis();
    EQUAL(tr.getLimitTimeMillis(), -1L);

    try {
      tr.begin(-1000L);
      NG();
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), QueryTransaction.Error.IllegalState);
    }

    tr.commit();
    EQUAL(tr.getState(), QueryTransaction.State.COMMITTED);
    EQUAL(tr.getTimeoutMillis(), -1L);
    EQUAL(tr.getBeginTimeMillis(), bgnTm);
    EQUAL(tr.getLimitTimeMillis(), -1L);

    try {
      tr.begin(-1000L);
      NG();
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), QueryTransaction.Error.IllegalState);
    }

    tr.end();
    EQUAL(tr.getState(), QueryTransaction.State.ENDED);
    EQUAL(tr.getTimeoutMillis(), -1L);
    EQUAL(tr.getBeginTimeMillis(), bgnTm);
    EQUAL(tr.getLimitTimeMillis(), -1L);

    try {
      tr.begin(-1000L);
      NG();
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), QueryTransaction.Error.IllegalState);
    }
  }

  public void commit()
  {
    MSG("トランザクションの結果を確定するメソッドの確認。");

    DefaultQueryTransaction tr = new DefaultQueryTransaction();
    EQUAL(tr.getState(), QueryTransaction.State.CREATED);
    EQUAL(tr.getTimeoutMillis(), -1L);
    EQUAL(tr.getBeginTimeMillis(), -1L);
    EQUAL(tr.getLimitTimeMillis(), -1L);

    try {
      tr.commit();
      NG();
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), QueryTransaction.Error.IllegalState);
    }

    long bgnTm;

    tr.begin(1000L);
    EQUAL(tr.getState(), QueryTransaction.State.BEGINED);
    EQUAL(tr.getTimeoutMillis(), -1L);
    bgnTm = tr.getBeginTimeMillis();
    EQUAL(tr.getLimitTimeMillis(), bgnTm + 1000L);

    tr.commit();
    EQUAL(tr.getState(), QueryTransaction.State.COMMITTED);
    EQUAL(tr.getTimeoutMillis(), -1L);
    EQUAL(tr.getBeginTimeMillis(), bgnTm);
    EQUAL(tr.getLimitTimeMillis(), bgnTm + 1000L);

    try {
      tr.commit();
      NG();
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), QueryTransaction.Error.IllegalState);
    }

    tr.end();
    EQUAL(tr.getState(), QueryTransaction.State.ENDED);
    EQUAL(tr.getTimeoutMillis(), -1L);
    EQUAL(tr.getBeginTimeMillis(), bgnTm);
    EQUAL(tr.getLimitTimeMillis(), bgnTm + 1000L);

    try {
      tr.commit();
      NG();
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), QueryTransaction.Error.IllegalState);
    }
  }

  public void rollback()
  {
    MSG("トランザクションの結果を取り消すメソッドの確認。");

    DefaultQueryTransaction tr = new DefaultQueryTransaction();
    EQUAL(tr.getState(), QueryTransaction.State.CREATED);
    EQUAL(tr.getTimeoutMillis(), -1L);
    EQUAL(tr.getBeginTimeMillis(), -1L);
    EQUAL(tr.getLimitTimeMillis(), -1L);

    try {
      tr.rollback();
      NG();
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), QueryTransaction.Error.IllegalState);
    }

    long bgnTm;

    tr.begin(1000L);
    EQUAL(tr.getState(), QueryTransaction.State.BEGINED);
    EQUAL(tr.getTimeoutMillis(), -1L);
    bgnTm = tr.getBeginTimeMillis();
    EQUAL(tr.getLimitTimeMillis(), bgnTm + 1000L);

    tr.rollback();
    EQUAL(tr.getState(), QueryTransaction.State.ROLLBACKED);
    EQUAL(tr.getTimeoutMillis(), -1L);
    EQUAL(tr.getBeginTimeMillis(), bgnTm);
    EQUAL(tr.getLimitTimeMillis(), bgnTm + 1000L);

    try {
      tr.rollback();
      NG();
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), QueryTransaction.Error.IllegalState);
    }

    tr.end();
    EQUAL(tr.getState(), QueryTransaction.State.ENDED);
    EQUAL(tr.getTimeoutMillis(), -1L);
    EQUAL(tr.getBeginTimeMillis(), bgnTm);
    EQUAL(tr.getLimitTimeMillis(), bgnTm + 1000L);

    try {
      tr.rollback();
      NG();
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), QueryTransaction.Error.IllegalState);
    }
  }

  public void end()
  {
    MSG("トランザクションを終了するメソッドの確認。");

    DefaultQueryTransaction tr;

    tr = new DefaultQueryTransaction();
    EQUAL(tr.getState(), QueryTransaction.State.CREATED);
    EQUAL(tr.getTimeoutMillis(), -1L);
    EQUAL(tr.getBeginTimeMillis(), -1L);
    EQUAL(tr.getLimitTimeMillis(), -1L);

    tr.end();
    EQUAL(tr.getState(), QueryTransaction.State.ENDED);
    EQUAL(tr.getTimeoutMillis(), -1L);
    EQUAL(tr.getBeginTimeMillis(), -1L);
    EQUAL(tr.getLimitTimeMillis(), -1L);

    long bgnTm;

    tr = new DefaultQueryTransaction();
    tr.begin(1000L);
    EQUAL(tr.getState(), QueryTransaction.State.BEGINED);
    EQUAL(tr.getTimeoutMillis(), -1L);
    bgnTm = tr.getBeginTimeMillis();
    EQUAL(tr.getLimitTimeMillis(), bgnTm + 1000L);

    tr.end();
    EQUAL(tr.getState(), QueryTransaction.State.ENDED);
    EQUAL(tr.getTimeoutMillis(), -1L);
    bgnTm = tr.getBeginTimeMillis();
    EQUAL(tr.getLimitTimeMillis(), bgnTm + 1000L);

    tr = new DefaultQueryTransaction();
    tr.begin(1000L);
    tr.commit();
    EQUAL(tr.getState(), QueryTransaction.State.COMMITTED);
    EQUAL(tr.getTimeoutMillis(), -1L);
    bgnTm = tr.getBeginTimeMillis();
    EQUAL(tr.getLimitTimeMillis(), bgnTm + 1000L);

    tr.end();
    EQUAL(tr.getState(), QueryTransaction.State.ENDED);
    EQUAL(tr.getTimeoutMillis(), -1L);
    EQUAL(tr.getBeginTimeMillis(), bgnTm);
    EQUAL(tr.getLimitTimeMillis(), bgnTm + 1000L);

    tr.end();
    EQUAL(tr.getState(), QueryTransaction.State.ENDED);
    EQUAL(tr.getTimeoutMillis(), -1L);
    EQUAL(tr.getBeginTimeMillis(), bgnTm);
    EQUAL(tr.getLimitTimeMillis(), bgnTm + 1000L);
  }

  public void getConnection_connId()
  {
    MSG("コネクションを取得するメソッドの確認。");

    DefaultQueryTransaction tr;

    tr = new DefaultQueryTransaction();
    EQUAL(OUTPUT_LOG(), "");
    tr.addConnectionFactory("c0", new MyConnectionFactory());
    EQUAL(OUTPUT_LOG(), "");
    tr.begin();
    EQUAL(OUTPUT_LOG(), "");
    tr.commit();
    EQUAL(OUTPUT_LOG(), "");
    tr.end();
    EQUAL(OUTPUT_LOG(), "");

    CLEAR_LOG();

    tr = new DefaultQueryTransaction();
    EQUAL(OUTPUT_LOG(), "");
    tr.addConnectionFactory("c0", new MyConnectionFactory());
    EQUAL(OUTPUT_LOG(), "");
    tr.begin();
    EQUAL(OUTPUT_LOG(), "");
    try {
      tr.getConnection("c0");
      EQUAL(OUTPUT_LOG(), "[c0 is opened!]");
      tr.getConnection("c0");
      EQUAL(OUTPUT_LOG(), "[c0 is opened!]");
      tr.getConnection("c0");
      EQUAL(OUTPUT_LOG(), "[c0 is opened!]");
      tr.commit();
      EQUAL(OUTPUT_LOG(), "[c0 is opened!][c0 is committed!]");
    } catch (Exception e) {
      tr.rollback();
      EQUAL(OUTPUT_LOG(), "[c0 is opened!][c0 is rollbacked!]");
    }
    finally {
      tr.end();
      EQUAL(OUTPUT_LOG(), "[c0 is opened!][c0 is committed!][c0 is closed!]");
    }

    CLEAR_LOG();

    tr = new DefaultQueryTransaction();
    EQUAL(OUTPUT_LOG(), "");
    tr.addConnectionFactory("c0", new MyConnectionFactory());
    tr.addConnectionFactory("c1", new MyConnectionFactory());
    tr.addConnectionFactory("c2", new MyConnectionFactory());
    EQUAL(OUTPUT_LOG(), "");
    tr.begin();
    EQUAL(OUTPUT_LOG(), "");
    try {
      tr.getConnection("c1");
      EQUAL(OUTPUT_LOG(), "[c1 is opened!]");
      tr.getConnection("c0");
      EQUAL(OUTPUT_LOG(), "[c1 is opened!][c0 is opened!]");
      tr.getConnection("c0");
      EQUAL(OUTPUT_LOG(), "[c1 is opened!][c0 is opened!]");
      tr.getConnection("c1");
      EQUAL(OUTPUT_LOG(), "[c1 is opened!][c0 is opened!]");
      tr.commit();
      EQUAL(OUTPUT_LOG(), "[c1 is opened!][c0 is opened!][c1 is committed!][c0 is committed!]");
    } catch (Exception e) {
      tr.rollback();
      EQUAL(OUTPUT_LOG(), "[c1 is opened!][c0 is opened!][c1 is rollbacked!][c0 is rollbacked!]");
    }
    finally {
      tr.end();
      EQUAL(OUTPUT_LOG(), "[c1 is opened!][c0 is opened!][c1 is committed!][c0 is committed!][c1 is closed!][c0 is closed!]");
    }
  }

  public void getConnection_connId_NoConnection()
  {
    MSG("引数のIDに対するコネクションが存在しない場合。");

    DefaultQueryTransaction tr;
    tr = new DefaultQueryTransaction();
    tr.begin();
    try {
      tr.getConnection("c0");
      NG();
    } catch (ReasonedException e) {
      EQUAL(e.getReason(), QueryTransaction.Error.ConnectionNotFound);
      EQUAL(e.getMessage(), "c0");
    }
    tr.end();

    tr = new DefaultQueryTransaction();
    tr.addConnectionFactory("c0", new MyConnectionFactory());
    tr.begin();
    try {
      EQUAL(tr.getConnection("c0").getId(), "c0");
    } catch (ReasonedException e) {
      NG();
    }
    try {
      tr.getConnection("c1");
      NG();
    } catch (ReasonedException e) {
      EQUAL(e.getReason(), QueryTransaction.Error.ConnectionNotFound);
      EQUAL(e.getMessage(), "c1");
    }
    tr.end();
  }

  public void getConnection_connId_BadState()
  {
    MSG("トランザクションの状態が不正な場合。");

    DefaultQueryTransaction tr = new DefaultQueryTransaction();
    tr.addConnectionFactory("c0", new MyConnectionFactory());

    try {
      tr.getConnection("c0");
      NG();
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), QueryTransaction.Error.IllegalState);
    } catch (ReasonedException e) {
      NG(e);
    }

    tr.begin();
    try {
      EQUAL(tr.getConnection("c0").getId(), "c0");
    } catch (ReasonedException e) {
      NG(e);
    }

    tr.rollback();
    try {
      tr.getConnection("c0");
      NG();
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), QueryTransaction.Error.IllegalState);
    } catch (ReasonedException e) {
      NG(e);
    }

    tr.end();
    try {
      tr.getConnection("c0");
      NG();
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), QueryTransaction.Error.IllegalState);
    } catch (ReasonedException e) {
      NG(e);
    }
  }

  public void getConnection_connId_FailToCreate()
  {
    MSG("コネクションの作成に失敗した場合。");

    DefaultQueryTransaction tr = new DefaultQueryTransaction();
    tr.addConnectionFactory("FAIL", new MyConnectionFactory());
    tr.begin();

    try {
      tr.getConnection("FAIL");
      NG();
    } catch (ReasonedException e) {
      EQUAL(e.getReason(), QueryConnectionFactory.Error.FailToCreate);
      EQUAL(e.getMessage(), "FAIL");
    }

    tr.end();
  }

  public void addConnectionFactory_connId_factory_Null()
  {
    MSG("引数がヌルの場合。");

    DefaultQueryTransaction tr = new DefaultQueryTransaction();
    try {
      tr.addConnectionFactory(null, new MyConnectionFactory());
      NG();
    } catch (AssertionError e) {
      OK(e);
    }
  }

  public void addConnectionFactory_connId_factory_BadState()
  {
    MSG("トランザクションの状態が適切でない場合。");

    DefaultQueryTransaction tr = new DefaultQueryTransaction();
    tr.addConnectionFactory("c0", new MyConnectionFactory());

    tr.begin();
    try {
      tr.addConnectionFactory("c1", new MyConnectionFactory());
      NG();
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), QueryTransaction.Error.IllegalState);
    }

    tr.rollback();
    try {
      tr.addConnectionFactory("c2", new MyConnectionFactory());
      NG();
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), QueryTransaction.Error.IllegalState);
    }

    tr.end();
    try {
      tr.addConnectionFactory("c3", new MyConnectionFactory());
      NG();
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), QueryTransaction.Error.IllegalState);
    }
  }
}
