/*
 * QueryContext class.
 *
 * Copyright (C) 2011 SATOH Takayuki All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package ts.query;

import ts.util.AbstractTypedGetter;
import ts.util.resource.Resource;
import java.io.Serializable;
import java.util.Collection;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.Map;
import java.util.HashMap;
import java.util.Set;
import java.util.LinkedHashSet;
import java.util.TreeSet;
import java.util.Collections;

/**
 * クエリの実行内容を格納するためのクラス。
 * <br>
 * 接続先を識別するコネクションID、実行内容を識別するクエリID、
 * クエリ・リソース・オブジェクトを取得するためのメソッドを備えている。
 * また、入力データ名の列挙、出力データ名の列挙を行うメソッドも用意している。
 * <br>
 * さらに、このクラスは{@link ts.util.TypedGetter}インターフェイスを
 * インプリメントしており、クエリに応じたコンテキスト情報を基本データ型や
 * 文字列、{@link ts.util.DateTime}オブジェクトに変換して取得することが可能
 * である。
 * <br>
 * このクラスのインスタンスは、実行結果オブジェクトの属性としても使用される。
 *
 * @author 佐藤隆之
 * @version $Id: QueryContext.java,v 1.7 2011-08-09 14:24:18 tayu Exp $
 */
public class QueryContext extends AbstractTypedGetter<Enum,Object>
{
  /** シリアル・バージョン番号。 */
  static final long serialVersionUID = 9063342353250916896L;

  /** クエリID。 */
  private final String queryId;

  /** コネクションID。 */
  private final String connectionId;

  /** クエリに応じたコンテキスト情報を格納するマップ。 */
  private final Map<Enum,Object> innerMap = new HashMap<Enum,Object>();

  /** 入力データ名のセット。 */
  private final Set<String> inputNameSet = new TreeSet<String>();

  /** 出力データ名のセット。 */
  private final Set<QueryOutput> outputSet = new LinkedHashSet<QueryOutput>();

  /**
   * コネクションIDとクエリIDを引数にとるコンストラクタ。
   *
   * @param connId コネクションID。
   * @param queryId クエリID。
   */
  public QueryContext(String connId, String queryId)
  {
    assert (connId != null && queryId != null) : 
      (connId  == null) ? "@param:connId is null." :
      (queryId == null) ? "@param:queryId is null." : "";

    this.connectionId = connId;
    this.queryId = queryId;
  }
 
  /**
   * クエリIDを取得する。
   *
   * @return クエリID。
   */
  public String getQueryId()
  {
    return this.queryId;
  }

  /**
   * コネクションIDを取得する。
   *
   * @return コネクションID。
   */
  public String getConnectionId()
  {
    return this.connectionId;
  }

  /**
   * 指定されたキーに結びつけられたコンテキスト情報データを取得する。
   *
   * @param key キー。
   * @return 引数のキーに結びつけられたコンテキスト情報データ。
   */
  public Object get(Enum key)
  {
    assert (key != null) : "@param:key is null.";
    return this.innerMap.get(key);
  }

  /**
   * 指定されたキーに対するコンテキスト情報データを設定する。
   * <br>
   * 引数の値がヌルの場合は、指定されたキーのコンテキスト情報データのエントリ
   * を削除する。
   *
   * @param key キー。
   * @param value 引数のキーに結びつけられるコンテキスト情報データ。
   * @throws AssertionError 引数がヌルの場合（デバッグ・モードのみ）。
   */
  public void set(Enum key, Serializable value)
  {
    assert (key != null) : "@param:key is null.";

    if (value == null) {
      this.innerMap.remove(key);
    }
    else {
      this.innerMap.put(key, value);
    }
  }

  /**
   * クエリが結果テーブルを返すかどうかを取得する。
   * <br>
   * クエリの内容が結果テーブルを返すものであるかどうかを判別し、結果テーブルを
   * 返す場合は<tt>true</tt>を返す。
   *
   * @return 結果テーブルを返す場合は<tt>true</tt>。
   */
  public boolean hasResultTable()
  {
    return (countOutputNames() > 0) ? true : false;
  }

  /**
   * クエリ・リソースの中に含まれる全ての入力パラメータ名を列挙する。
   * <br>
   * 入力パラメータ名の重複は除かれて出力される。
   *
   * @return 入力パラメータ名を格納した列挙オブジェクト。
   */
  public Enumeration<String> enumInputNames()
  {
    return Collections.enumeration(this.inputNameSet);
  }

  /**
   * クエリ・リソースの中に含まれる全ての入力パラメータ名の数を取得する。
   * <br>
   * 入力パラメータ名の重複は除かれて数えられる。
   *
   * @return 入力パラメータ名の数。
   */
  public int countInputNames()
  {
    return this.inputNameSet.size();
  }

  /**
   * クエリ・リソースの中に含まれる全ての出力パラメータ名を列挙する。
   * <br>
   * 出力パラメータ名の重複は除かれて出力される。
   *
   * @return 出力パラメータ名を格納した列挙オブジェクト。
   */
  public Enumeration<String> enumOutputNames()
  {
    final Iterator<QueryOutput> itr = this.outputSet.iterator();

    return new Enumeration<String>() {
      public boolean hasMoreElements() {
        return itr.hasNext();
      }
      public String nextElement() {
        return itr.next().getName();
      }
    };
  }

  /**
   * クエリ・リソースの中に含まれる全ての出力パラメータ名の数を取得する。
   * <br>
   * 出力パラメータ名の重複は除かれて数えられる。
   *
   * @return 出力パラメータ名の数。
   */
  public int countOutputNames()
  {
    return this.outputSet.size();
  }

  /**
   * 入力パラメータ名を追加する。
   *
   * @param name 入力パラメータ名。
   * @throws AssertionError 引数がヌルの場合（デバッグ・モードのみ）。
   */
  public void addInputName(String name)
  {
    assert (name != null) : "@param:name is null.";
    this.inputNameSet.add(name);
  }

  /**
   * 複数の入力パラメータ名を追加する。
   *
   * @param names 入力パラメータ名のコレクション。
   * @throws AssertionError 引数や引数のコレクションの要素がヌルの場合
   *           （デバッグ・モードのみ）。
   */
  public void addInputNames(Collection<String> names)
  {
    assert (names != null) : "@param:names is null.";

    for (String s : names) {
      assert (s != null) : "@param:names has null elements.";
      this.inputNameSet.add(s);
    }
  }

  /**
   * 複数の入力パラメータ名を追加する。
   *
   * @param names 入力パラメータ名の配列。
   * @throws AssertionError 引数や引数の配列の要素がヌルの場合
   *           （デバッグ・モードのみ）。
   */
  public void addInputNames(String[] names)
  {
    assert (names != null) : "@param:names is null.";

    for (String s : names) {
      assert (s != null) : "@param:names has null elements.";
      this.inputNameSet.add(s);
    }
  }

  /**
   * 出力項目を追加する。
   *
   * @param output 出力項目オブジェクト。
   * @throws AssertionError 引数がヌルの場合（デバッグ・モードのみ）。
   */
  public void addOutput(QueryOutput output)
  {
    assert (output != null) : "@param:output is null.";
    this.outputSet.add(output);
  }

  /**
   * 複数の出力項目を追加する。
   *
   * @param outputs 出力項目オブジェクトのコレクション。
   * @throws AssertionError 引数や引数のコレクションの要素がヌルの場合
   *           （デバッグ・モードのみ）。
   */
  public void addOutputs(Collection<QueryOutput> outputs)
  {
    assert (outputs != null) : "@param:outputs is null.";

    for (QueryOutput output : outputs) {
      assert (output != null) : "@param:outputs has null elements.";
      this.outputSet.add(output);
    }
  }

  /**
   * 複数の出力項目を追加する。
   *
   * @param outputs 出力項目オブジェクトの配列k。
   * @throws AssertionError 引数や引数の配列の要素がヌルの場合
   *           （デバッグ・モードのみ）。
   */
  public void addOutputs(QueryOutput[] outputs)
  {
    assert (outputs != null) : "@param:outputs is null.";

    for (QueryOutput output : outputs) {
      assert (output != null) : "@param:outputs has null elements.";
      this.outputSet.add(output);
    }
  }
}
