/*
 * QueryParam class.
 *
 * Copyright (C) 2011 SATOH Takayuki All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package ts.query;

import ts.util.AbstractTypedGetter;
import java.util.Map;
import java.util.HashMap;

/**
 * クエリ・パラメータ・クラス。
 * <br>
 * クエリの実行内容に適用するパラメータを表すクラス。
 * パラメータの名前、値を属性に持ち、その他にデータ型などの情報を任意の属性
 * として保持することができる。
 *
 * @author 佐藤隆之
 * @version $Id: QueryParam.java,v 1.4 2011-08-09 14:08:20 tayu Exp $
 */
public class QueryParam extends AbstractTypedGetter<String,Object>
{
  /** シリアル・バージョン番号。 */
  static final long serialVersionUID = -7316624337998775137L;

  /** 入出力区分を表す列挙型。 */
  public enum IO {
    /** 入出力区分が入力であることを表す列挙値。 */
    IN,
    /** 入出力区分が出力であることを表す列挙値。 */
    OUT,
    /** 入出力区分が入力兼出力であることを表す列挙値。 */
    INOUT,
  }

  /** パラメータ名。 */
  private final String name;

  /** 入出力区分。 */
  private final IO io;

  /** パラメータ値を格納するイテレータ。 */
  private QueryParamValueIterator iterator = null;

  /** 任意の属性を格納するマップ。 */
  private Map<String,Object> attrMap = null;

  /**
   * パラメータ名を引数にとるコンストラクタ。
   * <br>
   * 入出力区分は入力、値はヌルに指定される。
   *
   * @param name パラメータ名。
   * @throws AssertionError 引数がヌルの場合（デバッグ・モードのみ）。
   */
  public QueryParam(String name)
  {
    this(name, IO.IN);
  }

  /**
   * パラメータ名と入出力区分を引数にとるコンストラクタ。
   * <br>
   * 値はヌルに指定される。
   *
   * @param name パラメータ名。
   * @param io 入出力区分。
   * @throws AssertionError 引数がヌルの場合（デバッグ・モードのみ）。
   */
  public QueryParam(String name, IO io)
  {
    assert (name != null && io != null) : 
      (name == null) ? "@param:name is null." :
      (io   == null) ? "@param:io is null." : "";

    this.name = name;
    this.io = io;
  }

  /**
   * パラメータ名を取得する。
   *
   * @return パラメータ名。
   */
  public String getName()
  {
    return this.name;
  }

  /**
   * パラメータの入出力区分を取得する。
   *
   * @return パラメータの入出力区分。
   */
  public IO getIO()
  {
    return this.io;
  }

  /** 
   * 指定されたキーに結びつけられた属性値を取得する。
   *
   * @param key キー。
   * @return 属性値。
   */
  @Override
  public Object get(String key)
  {
    return (this.attrMap == null) ? null : this.attrMap.get(key);
  }

  /** 
   * 指定されたキーを属性名とする属性値を設定する。
   *
   * @param key キー。
   * @param attrValue 属性値。
   * @return 以前このキーに結びつけられていた属性値。
   */
  public Object put(String key, Object attrValue)
  {
    if (this.attrMap == null) {
      this.attrMap = new HashMap<String,Object>();
    }

    if (attrValue == null) {
      return this.attrMap.remove(key);
    }
    else {
      return this.attrMap.put(key, attrValue);
    }
  }

  /**
   * パラメータの値を格納するイテレータを設定する。
   *
   * @param itr パラメータの値。
   */
  public void setValueIterator(QueryParamValueIterator itr)
  {
    this.iterator = itr;
  }

  /**
   * パラメータの値を取得する。
   *
   * @return パラメータの値。
   */
  public Object getValue()
  {
    return (this.iterator == null) ? null : this.iterator.next();
  }

  /**
   * パラメータの値を取得する。
   *
   * @return パラメータの値。
   */
  public Object getValueCount()
  {
    return (this.iterator == null) ? 1 : this.iterator.count();
  }
}
