/*
 * QueryResult class.
 *
 * Copyright (C) 2011 SATOH Takayuki All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package ts.query;

import ts.util.ReasonedException;
import ts.util.ReasonedRuntimeException;
import ts.util.table.Table;
import ts.util.table.MapIterator;
import java.io.Serializable;
import java.util.TreeSet;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Enumeration;

/**
 * クエリの実行結果を格納するクラス。
 * <br>
 * クエリの実行結果として、実行成否、処理件数、処理時間、結果テーブル、クエリの
 * 実行に失敗した時の例愛オブジェクトを保持する。
 * また、実行されたクエリのコンテキスト情報を保持する。
 *
 * @author 佐藤隆之
 * @version $Id: QueryResult.java,v 1.4 2011-08-01 15:25:03 tayu Exp $
 */
public class QueryResult implements Serializable
{
  /** シリアル・バージョン番号。 */
  static final long serialVersionUID = 8309940887092598304L;

  /** コンテキスト情報オブジェクト。 */
  private final QueryContext context;

  /** クエリの実行成否フラグ。 */
  private boolean isSuccess = false;

  /** クエリの処理件数。 */
  private int resultCount = 0;

  /** クエリの処理時間。 */
  private long spentTimeMillis = 0L;

  /** クエリの結果を格納するテーブル・オブジェクト。 */
  private Table<String,Object> resultTable = null;

  /** クエリの実行が失敗した時に発生した例外オブジェクト。 */
  private Exception exception = null;

  /**
   * クエリのコンテキスト情報をにとるコンストラクタ。
   *
   * @param context コンテキスト情報オブジェクト。
   */
  public QueryResult(QueryContext context)
  {
    assert (context != null) : "@param:context is null.";
    this.context = context;
  }

  /**
   * クエリIDを取得する。
   *
   * @return クエリID。
   */
  public String getQueryId()
  {
    return this.context.getQueryId();
  }

  /**
   * クエリのコンテキスト情報を取得する。
   *
   * @return クエリのコンテキスト情報。
   */
  public QueryContext getContext()
  {
    return this.context;
  }

  /**
   * クエリの実行の成否を設定する。
   *
   * @param b クエリの実行が成功した場合は<tt>true</tt>を設定する。
   */
  protected void setSuccess(boolean b)
  {
    this.isSuccess = b;
  }

  /**
   * クエリの実行の成否を取得する。
   *
   * @return クエリの実行が成功した場合は<tt>true</tt>を返す。
   */
  public boolean isSuccess()
  {
    return this.isSuccess;
  }

  /**
   * クエリの処理件数を設定する。
   *
   * @param resultCount クエリの処理件数。
   */
  protected void setResultCount(int resultCount)
  {
    assert (resultCount >= 0) : "@param:resultCount is negative.";
    this.resultCount = resultCount;
  }

  /**
   * クエリの処理件数を取得する。
   *
   * @return クエリの処理件数。
   */
  public int getResultCount()
  {
    return this.resultCount;
  }

  /**
   * クエリの処理時間をミリ秒単位で設定する。
   *
   * @param spentTime クエリの処理時間[msec]。
   */
  protected void setSpentTimeMillis(long spentTime)
  {
    assert (spentTime >= 0) : "@param:spentTime is negative.";
    this.spentTimeMillis = spentTime;
  }

  /**
   * クエリの処理時間をミリ秒単位で取得する。
   *
   * @return クエリの処理時間[msec]。
   */
  public long getSpentTimeMillis()
  {
    return this.spentTimeMillis;
  }

  /**
   * クエリの実行が失敗した時の理由付き例外オブジェクトを設定する。
   * <br>
   * クエリの実行が成功した場合はヌルを設定する。
   *
   * @param exc クエリの実行が失敗した時の理由付き例外オブジェクト。
   */
  protected void setException(ReasonedException exc)
  {
    this.exception = exc;
  }

  /**
   * クエリの実行が失敗した時の理由付きランタイム例外オブジェクトを設定する。
   * <br>
   * クエリの実行が成功した場合はヌルを設定する。
   *
   * @param exc クエリの実行が失敗した時の理由付きランタイム例外オブジェクト。
   */
  protected void setException(ReasonedRuntimeException exc)
  {
    this.exception = exc;
  }

  /**
   * クエリの実行が失敗した時の例外オブジェクトを取得する。
   * <br>
   * クエリの実行が成功した場合はヌルを返す。
   *
   * @return クエリの実行が失敗した時の例外オブジェクト。
   */
  public Exception getException()
  {
    return this.exception;
  }

  /**
   * クエリの実行が失敗した時の例外の理由を取得する。
   * <br>
   * クエリの実行が成功した場合はヌルを返す。
   *
   * @return クエリの実行が失敗した時の例外の理由。
   */
  public Enum getExceptionReason()
  {
    if (this.exception != null) {
      if (this.exception instanceof ReasonedException) {
        return ReasonedException.class.cast(this.exception).getReason();
      }
      else if (this.exception instanceof ReasonedRuntimeException) {
        return ReasonedRuntimeException.class.cast(this.exception).getReason();
      }
    }
    return null;
  }

  /**
   * 結果テーブルを取得する。
   *
   * @return 結果テーブル。
   */
  public Table<String,Object> getResultTable()
  {
    return this.resultTable;
  }

  /**
   * 結果テーブルを設定する。
   *
   * @param table 結果テーブル。
   */
  protected void setResultTable(Table<String,Object> table)
  {
    this.resultTable = table;
  }
}
