/*
 * DefaultQuery class.
 *
 * Copyright (C) 2011 SATOH Takayuki All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package ts.query;

import ts.util.ReasonedException;
import ts.util.ReasonedRuntimeException;
import ts.util.table.Header;
import ts.util.table.Table;
import ts.util.table.ArrayListTable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Arrays;

/**
 * クエリを実行するクラスのデフォルト実装クラス。
 *
 * @author 佐藤隆之
 * @version $Id: DefaultQuery.java,v 1.3 2011-09-18 16:15:15 tayu Exp $
 */
public class DefaultQuery implements Query
{
  /** クエリを実行する際の接続先を示す{@link QueryConnection}オブジェクト。 */
  private final QueryConnection connection;

  /**
   * クエリの実行内容の基となる情報を保持する{@link QueryResource}オブジェクト。
   */
  private final QueryResource resource;

  /**
   * 接続先を示す{@link QueryConnection}オブジェクトと、実行内容の元になる情報
   * を保持する{@link QueryResource}オブジェクトを引数にとるコンストラクタ。
   *
   * @param conn {@link QueryConnection}オブジェクト。
   * @param res  {@link QueryResource}オブジェクト。
   */
  public DefaultQuery(QueryConnection conn, QueryResource res)
  {
    assert (conn != null && res != null) :
      (conn == null) ? "@param:conn is null." :
      (res  == null) ? "@param:res is null." : "";

    this.connection = conn;
    this.resource = res;
  }

  /**
   * クエリを実行する接続先を示す{@link QueryConnection}オブジェクトを取得する。
   *
   * @return クエリを実行する接続先を示す{@link QueryConnection}オブジェクト。
   */
  protected QueryConnection getConnection()
  {
    return this.connection;
  }

  /**
   * クエリの実行内容の元になる情報を保持する{@link QueryResource}
   * オブジェクトを取得する。
   *
   * @return クエリの実行内容の元になる情報を保持する{@link QueryResource}
   *           オブジェクト。
   */
  protected QueryResource getResource()
  {
    return this.resource;
  }

  /**
   * {@inheritDoc}
   */
  public QueryResult execute(String queryId, Map<String,Object> inputMap)
    throws ReasonedException
  {
    return executeQuery(queryId, inputMap, null);
  }

  /**
   * {@inheritDoc}
   */
  public QueryResultList executeSet(
    String[] qidArr, Map<String,Object> inputMap) throws ReasonedException
  {
    assert (qidArr != null && inputMap != null) :
      (qidArr   == null) ? "@param:qidArr is null." :
      (inputMap == null) ? "@param:inputMap is null." : "";

    long stime = System.currentTimeMillis();
    QueryResultList rsltLst = new QueryResultList();

    for (String qid : qidArr) {
      QueryResult rslt = executeQuery(qid, inputMap, rsltLst);
      rsltLst.addResult(rslt);
    }

    long etime = System.currentTimeMillis();
    rsltLst.setSpentTimeMillis(etime - stime);
    return rsltLst;
  }

  /**
   * {@inheritDoc}
   */
  public QueryResultList executeSet(
    List<String> qidLst, Map<String,Object> inputMap) throws ReasonedException
  {
    assert (qidLst != null && inputMap != null) :
      (qidLst   == null) ? "@param:qidLst is null." :
      (inputMap == null) ? "@param:inputMap is null." : "";

    long stime = System.currentTimeMillis();
    QueryResultList rsltLst = new QueryResultList();

    for (String qid : qidLst) {
      QueryResult rslt = executeQuery(qid, inputMap, rsltLst);
      rsltLst.addResult(rslt);
    }

    long etime = System.currentTimeMillis();
    rsltLst.setSpentTimeMillis(etime - stime);
    return rsltLst;
  }

  /**
   * {@inheritDoc}
   */
  public QueryResultList executeAll(Map<String,Object> inputMap)
    throws ReasonedException
  {
    assert (inputMap != null) : "@param:inputMap is null.";

    long stime = System.currentTimeMillis();
    QueryResultList rsltLst = new QueryResultList();

    for (String qid : getResource().listAllQueryIds()) {
      QueryResult rslt = executeQuery(qid, inputMap, rsltLst);
      rsltLst.addResult(rslt);
    }

    long etime = System.currentTimeMillis();
    rsltLst.setSpentTimeMillis(etime - stime);
    return rsltLst;
  }

  /**
   * クエリの結果データを格納するテーブルを作成する。
   *
   * @return クエリの結果データを格納するテーブル。
   */
  protected Table<String,Object> newResultTable(QueryContent cont)
  {
    assert (cont != null) : "@param:cont is null.";

    Header<String> hdr = new ArrayListTable.Header<String>(cont.countOutputs());
    for (QueryOutput out : cont.getOutputs()) {
      hdr.addColumn(out.getName());
    }
    return new ArrayListTable<String,Object>(hdr);
  }

  /**
   * 指定された一つのクエリIDに対応づけられたクエリを実行する。
   *
   * @param queryId クエリID。
   * @param inputMap 入力パラメータ・マップ。
   * @param rsltLst 以前に実行されたクエリの実行結果を格納するリスト。
   * @return クエリ結果オブジェクト。
   * @throws ReasonedException クエリの実行に失敗した場合。
   */
  protected QueryResult executeQuery(
    String queryId, Map<String,Object> inputMap, QueryResultList rsltLst
  ) throws ReasonedException
  {
    assert (queryId != null && inputMap != null) :
      (queryId  == null) ? "@param:query is null." :
      (inputMap == null) ? "@param:inputMap is null." : "";

    long stime = System.currentTimeMillis();
    QueryResult rslt = new QueryResult(getConnection().getId(), queryId);

    try {
      checkTimeout(queryId);

      QueryContent cont = getResource().createContent(queryId,inputMap,rsltLst);
      rslt.setContent(cont);

      checkTimeout(queryId);

      if (cont.hasResultTable()) {
        Table<String,Object> tbl = newResultTable(cont);
        int num = getConnection().executeContent(cont, tbl);
        rslt.setResultTable(tbl);
        rslt.setResultCount(num);
      }
      else {
        int num = getConnection().executeContent(cont);
        rslt.setResultCount(num);
      }
      rslt.setSuccess(true);
    }
    catch (Exception e) {
      rslt.setSuccess(false);
      rslt.setException(e);
      throw rslt.getReasonedException();
    }
    finally {
      long etime = System.currentTimeMillis();
      rslt.setSpentTimeMillis(etime - stime);
      getConnection().getTransactionQueryResultList().addResult(rslt);
    }

    return rslt;
  }

  /**
   * タイムアウトの判定を実行する。
   *
   * @param queryId 実行されるクエリのクエリID。
   * @throws ReasonedException 現在の時刻がタイムアウト時刻を超えた場合。
   */
  protected void checkTimeout(String queryId) throws ReasonedException
  {
    QueryConnection conn = getConnection();
    long limitTime = conn.getLimitTimeMillis();
    if (limitTime > 0L) {
      if (limitTime < System.currentTimeMillis()) {
        throw new ReasonedException(Error.Timeout,
          "[connection:" + conn.getId() + "][query:" + queryId + "]");
      }
    }
  }
}
