package ts.query;

import ts.tester.UnitTest;
import ts.util.*;
import ts.util.file.*;
import ts.util.resource.*;
import ts.util.table.*;
import ts.util.text.*;
import java.io.*;
import java.util.*;

public class DefaultQueryTest extends UnitTest
{
  public static void main(String[] args)
  {
    run(DefaultQueryTest.class, args);
  }

  /*
   * データ・ファイルを元に戻す。
   */
  void rollbackDataFile()
  {
    try {
      File dir = new File("test/data/ts/query");
      File file0 = new File(dir, "SampleData.dat");
      File file1 = new File(dir, "SampleData.xml");
      if (file1.exists()) {
        FileOperation.delete(file1);
      }
      FileOperation.copy(file0, file1);
    }
    catch (Exception e) {
      NG(e);
    }
  }


  /*
   * XMLファイルに対してアクセスを行うクエリ・コネクションのファクトリ・クラス。
   */
  static class XmlFileQueryConnectionFactory implements QueryConnectionFactory{
    public QueryConnection create(String connId, QueryTransaction tran) {
      return new XmlFileQueryConnection(connId, tran);
    }
  }


  static boolean FAIL = false;

  /*
   * XMLファイルに対して値の取得・更新を行うサンプル・クエリ・コネクション・
   * クラス。
   */
  static class XmlFileQueryConnection implements QueryConnection {
    private final String connId;
    private final XmlResource resource;
    private long limitTime;
    private boolean isModified = false;
    private boolean isClosed = true;
    private QueryResultList rsltLst;
    public enum Error { FileNotFound, FailToLoadFile, FailToSaveFile, }
    public XmlFileQueryConnection(String connId, QueryTransaction tran) {
      this.connId = connId;
      this.resource = new XmlResource();
      this.resource.setValidating(false);
      this.rsltLst = tran.getQueryResultList();
      this.limitTime = tran.getLimitTimeMillis();
    }
    protected void setModified(boolean b) { this.isModified = b; }
    @Override
    public String getId() { return this.connId; }
    @Override
    public boolean isClosed() { return this.isClosed; }
    @Override
    public long getLimitTimeMillis() { return this.limitTime; }
    @Override
    public void open() {
      String path = QueryConfig.getInstance().getString("sample.dat.path");
      try {
        this.resource.load(path);
        this.isClosed = false;
      } catch (FileNotFoundException e) {
        throw new ReasonedRuntimeException(Error.FileNotFound, path, e);
      } catch (Exception e) {
        throw new ReasonedRuntimeException(Error.FailToLoadFile, path, e);
      }
    }
    @Override
    public void commit() {
      if (! this.isModified) return;
      String path = QueryConfig.getInstance().getString("sample.dat.path");
      try {
        this.resource.save(path + ".tmp");
        FileOperation.delete(new File(path));
        FileOperation.move(new File(path + ".tmp"), new File(path));
        this.isModified = false;
      } catch (Exception e) {
        throw new ReasonedRuntimeException(Error.FailToSaveFile, path, e);
      }
    }
    @Override
    public void rollback() {
      String path = QueryConfig.getInstance().getString("sample.dat.path");
      try {
        this.resource.load(path);
        this.isModified = false;
      } catch (Exception e) {
        throw new RuntimeException(e);
      }
    }
    @Override
    public void close() {
      try {
        this.resource.removeChildren("");
        this.isModified = false;
        this.isClosed = true;
      } catch (Exception e) {
        throw new RuntimeException(e);
      }
    }
    @Override
    public Query getQuery(QueryResource res) {
      return new DefaultQuery(this, res);
    }
    @Override
    public int executeContent(QueryContent cont, Table<String,Object> table)
    throws ReasonedException {
      try { Thread.sleep(10L); } catch (InterruptedException e) {}
      String nodePath = cont.getString(XmlFileQueryResource.Key.NodePath);
      if (nodePath.isEmpty()) throw new NullPointerException();
      for (Resource child : this.resource.getChildren(nodePath)) {
        Map<String,Object> rec = table.appendNew();
        for (Resource col : child.getChildren("*")) {
          rec.put(col.getBaseElementName(), col.getFirstValue(""));
        }
      }
      try { Thread.sleep(10L); } catch (InterruptedException e) {}
      return table.recordCount();
    }
    @Override
    public int executeContent(QueryContent cont) throws ReasonedException {
      try { Thread.sleep(10L); } catch (InterruptedException e) {}
      if (FAIL) throw new ReasonedException(Query.Error.FailToExecute);
      String nodePath = cont.getString(XmlFileQueryResource.Key.NodePath);
      if (nodePath.isEmpty()) throw new NullPointerException();
      List<Resource> children = this.resource.getChildren(nodePath);
      List<QueryParam> paramLst = cont.listParams();
      for (Resource child : children) {
        for (QueryParam param : paramLst) {
          if (param.getIO() != QueryParam.IO.IN) continue;
          Object obj = param.getValue();
          String val = (obj != null) ? obj.toString() : "";
          child.setFirstValue(param.getName(), val);
        }
      }
      this.isModified = true;
      try { Thread.sleep(10L); } catch (InterruptedException e) {}
      return children.size();
    }
    @Override
    public QueryResultList getTransactionQueryResultList() {
      return this.rsltLst;
    }
  }

  /*
   * XMLファイルに対して値の取得・更新を行うサンプル・クエリ・リソース・
   * クラス。
   */
  static class XmlFileQueryResource implements QueryResource
  {
    private PropertyResource res = new PropertyResource();
    public enum Key { NodePath, }
    public enum Error { FileNotFound, FailToLoad, }
    public XmlFileQueryResource() {
      String path = QueryConfig.getInstance().getString("sample.res.path");
      try {
        res.load(path);
      } catch (FileNotFoundException e) {
        throw new ReasonedRuntimeException(Error.FileNotFound);
      } catch (Exception e) {
        throw new ReasonedRuntimeException(Error.FailToLoad);
      }
    }
    @Override
    public QueryContent createContent(
      String queryId, Map<String,Object> inputMap, QueryResultList rsltLst
    ) throws ReasonedException {
      try { Thread.sleep(10L); } catch (InterruptedException e) {}
      QueryContent cont = new QueryContent();
      if ("select".equalsIgnoreCase(queryId)) {
        String s = this.res.getFirstValue("query.select");
        if (inputMap.containsKey("ID")) {
          s += "(ID=" + inputMap.get("ID") + ")";
        }
        cont.set(Key.NodePath, s);
        cont.addOutput(new QueryOutput("CCC"));
        cont.addOutput(new QueryOutput("DDD"));
        cont.addOutput(new QueryOutput("EEE"));
        cont.addOutput(new QueryOutput("FFF"));
        try { Thread.sleep(10L); } catch (InterruptedException e) {}
      }
      else if ("update".equalsIgnoreCase(queryId)) {
        String s = this.res.getFirstValue("query.update");
        if (inputMap.containsKey("ID")) {
          s += "(ID=" + inputMap.get("ID") + ")";
        }
        cont.set(Key.NodePath, s);
        for (Map.Entry<String,Object> me : inputMap.entrySet()) {
          if (me.getKey().equals("ID")) continue;
          QueryParam param = new QueryParam(me.getKey());
          param.setValue(me.getValue());
          cont.addParam(param);
        }
        try { Thread.sleep(10L); } catch (InterruptedException e) {}
      }
      else if ("".equalsIgnoreCase(queryId)) {
        cont.set(Key.NodePath, null);
        try { Thread.sleep(10L); } catch (InterruptedException e) {}
      }
      else if ("fail".equalsIgnoreCase(queryId)) {
        try { Thread.sleep(10L); } catch (InterruptedException e) {}
        throw new ReasonedException(
          QueryResource.Error.FailToCreateContent, queryId);
      }
      else {
        try { Thread.sleep(10L); } catch (InterruptedException e) {}
        throw new ReasonedException(
          QueryResource.Error.QueryIdNotFound, queryId);
      }
      try { Thread.sleep(100L); } catch (InterruptedException e) {}
      return cont;
    }
    @Override
    public List<String> listAllQueryIds() {
      List<String> lst = new ArrayList<String>(2);
      lst.add("select");
      lst.add("update");
      return lst;
    }
  }

  /* -- test -- */ 
  public void constructor_conn_res()
  {
    MSG("コンストラクタの確認。");

    QueryTransaction tran = new DefaultQueryTransaction();
    tran.addConnectionFactory("c0", new XmlFileQueryConnectionFactory());

    try {
      tran.begin();

      QueryConnection conn = tran.getConnection("c0");
      XmlFileQueryResource res = new XmlFileQueryResource();

      DefaultQuery query = new DefaultQuery(conn, res);
      TRUE(query instanceof Query);
      EQUAL(query.getConnection(), conn);
      EQUAL(query.getResource(), res);
    }
    catch (ReasonedException e) {
      NG(e);
    }
    finally {
      tran.end();
    }
  }

  public void constructor_conn_res_Null()
  {
    MSG("引数がヌルの場合。");

    QueryTransaction tran = new DefaultQueryTransaction();
    tran.addConnectionFactory("c0", new XmlFileQueryConnectionFactory());

    try {
      tran.begin();
      QueryConnection conn = tran.getConnection("c0");
      XmlFileQueryResource res = new XmlFileQueryResource();

      try {
        new DefaultQuery(null, res);
        NG();
      } catch (AssertionError e) {
        OK(e);
      }

      try {
        new DefaultQuery(conn, null);
        NG();
      } catch (AssertionError e) {
        OK(e);
      }
    }
    catch (ReasonedException e) {
      NG(e);
    }
    finally {
      tran.end();
    }
  }

  public void newResultTable_cont()
  {
    MSG("クエリの結果データを格納するテーブルを作成するメソッドの確認。");

    QueryTransaction tran = new DefaultQueryTransaction();
    tran.addConnectionFactory("c0", new XmlFileQueryConnectionFactory());

    try {
      tran.begin();

      QueryConnection conn = tran.getConnection("c0");
      XmlFileQueryResource res = new XmlFileQueryResource();

      DefaultQuery query = DefaultQuery.class.cast(conn.getQuery(res));

      QueryOutput out0, out1, out2;
      out0 = new QueryOutput("O1");
      out1 = new QueryOutput("OA");
      out2 = new QueryOutput("Oa");

      QueryContent cont = new QueryContent();
      cont.addOutput(out0);
      cont.addOutput(out1);
      cont.addOutput(out2);

      Table<String,Object> table = query.newResultTable(cont);
      EQUAL(table.columnCount(), 3);
      EQUAL(table.recordCount(), 0);
      Set<String> columnSet = new HashSet<String>();
      columnSet.add(out0.getName());
      columnSet.add(out1.getName());
      columnSet.add(out2.getName());
      Enumeration<String> enm = table.columns();
      TRUE(enm.hasMoreElements());
      TRUE(columnSet.contains(enm.nextElement()));
      TRUE(enm.hasMoreElements());
      TRUE(columnSet.contains(enm.nextElement()));
      TRUE(enm.hasMoreElements());
      TRUE(columnSet.contains(enm.nextElement()));
      FALSE(enm.hasMoreElements());
    }
    catch (ReasonedException e) {
      NG(e);
    }
    finally {
      tran.end();
    }
  }

  public void newResultTable_cont_Null()
  {
    MSG("引数がヌルの場合。");

    QueryTransaction tran = new DefaultQueryTransaction();
    tran.addConnectionFactory("c0", new XmlFileQueryConnectionFactory());

    try {
      tran.begin();

      QueryConnection conn = tran.getConnection("c0");
      XmlFileQueryResource res = new XmlFileQueryResource();

      DefaultQuery query = DefaultQuery.class.cast(conn.getQuery(res));

      try {
        query.newResultTable(null);
        NG();
      } catch (AssertionError e) {
        OK(e);
      }
    }
    catch (ReasonedException e) {
      NG(e);
    }
    finally {
      tran.end();
    }
  }

  public void checkTimeout_qid()
  {
    MSG("タイムアウト時間を超えたかどうかを判定するメソッドの確認。");

    DefaultQueryTransaction tran = new DefaultQueryTransaction();
    tran.addConnectionFactory("c0", new XmlFileQueryConnectionFactory());

    try {
      tran.begin();

      QueryConnection conn = tran.getConnection("c0");
      XmlFileQueryResource res = new XmlFileQueryResource();

      DefaultQuery query = DefaultQuery.class.cast(conn.getQuery(res));
      query.checkTimeout("q0");
      OK();
    }
    catch (ReasonedException e) {
      NG(e);
    }
    finally {
      tran.end();
    }

    tran = new DefaultQueryTransaction();
    tran.addConnectionFactory("c0", new XmlFileQueryConnectionFactory());
    tran.setTimeoutMillis(1000L);

    try {
      tran.begin();

      OK("Begin Time = " + tran.getBeginTimeMillis());
      OK("Limit Time = " + tran.getLimitTimeMillis());
      OK("This  Time = " + System.currentTimeMillis());
  
      XmlFileQueryResource res = new XmlFileQueryResource();
      QueryConnection conn = tran.getConnection("c0");
      OK("Limit Time = " + conn.getLimitTimeMillis());

      DefaultQuery query = DefaultQuery.class.cast(conn.getQuery(res));
      try {
        query.checkTimeout("q0");
        OK();
      } catch (ReasonedException e) {
        NG(e);
      }

      try {
        Thread.sleep(1000L);
      } catch (InterruptedException e) {
        NG(e);
      }
      OK("This  Time = " + System.currentTimeMillis());

      try {
        query.checkTimeout("q1");
        NG();
      } catch (ReasonedException e) {
        OK(e);
        EQUAL(e.getReason(), Query.Error.Timeout);
      }
    }
    catch (Exception e) {
      NG(e);
    }
    finally {
      tran.end();
    }
  }

  public void checkTimeout_qid_Null()
  {
    MSG("引数がヌルの場合。");

    DefaultQueryTransaction tran = new DefaultQueryTransaction();
    tran.addConnectionFactory("c0", new XmlFileQueryConnectionFactory());

    try {
      tran.begin();

      QueryConnection conn = tran.getConnection("c0");
      XmlFileQueryResource res = new XmlFileQueryResource();

      DefaultQuery query = DefaultQuery.class.cast(conn.getQuery(res));
      query.checkTimeout(null);
      OK();
    }
    catch (ReasonedException e) {
      NG(e);
    }
    finally {
      tran.end();
    }

    tran = new DefaultQueryTransaction();
    tran.addConnectionFactory("c0", new XmlFileQueryConnectionFactory());
    tran.setTimeoutMillis(1000L);

    try {
      tran.begin();
      OK("Begin Time = " + tran.getBeginTimeMillis());
      OK("Limit Time = " + tran.getLimitTimeMillis());
      OK("This  Time = " + System.currentTimeMillis());

      XmlFileQueryResource res = new XmlFileQueryResource();
      QueryConnection conn = tran.getConnection("c0");
      OK("Limit Time = " + conn.getLimitTimeMillis());

      DefaultQuery query = DefaultQuery.class.cast(conn.getQuery(res));
      try {
        query.checkTimeout(null);
        OK();
      } catch (ReasonedException e) {
        NG(e);
      }
      try {
        Thread.sleep(1000L);
      } catch (InterruptedException e) {
        NG(e);
      }
      OK("This  Time = " + System.currentTimeMillis());
      try {
        query.checkTimeout(null);
        NG();
      } catch (ReasonedException e) {
        OK(e);
        EQUAL(e.getReason(), Query.Error.Timeout);
      }
    }
    catch (ReasonedException e) {
      NG(e.toString());
    }
    finally {
      tran.end();
    }
  }

  public void execute_qid_inputMap_Select()
  {
    MSG("検索を行う一つのクエリを実行した場合。");

    DefaultQueryTransaction tran = new DefaultQueryTransaction();
    tran.addConnectionFactory("c0", new XmlFileQueryConnectionFactory());

    Map<String,Object> map = new LinkedHashMap<String,Object>();

    try {
      tran.begin();

      QueryConnection conn = tran.getConnection("c0");
      Query query = conn.getQuery(new XmlFileQueryResource());

      QueryResult rslt = query.execute("select", map);
      TRUE(rslt.isSuccess());
      EQUAL(rslt.getResultCount(), 3);
      POSITIVE(rslt.getSpentTimeMillis());
      NULL(rslt.getReasonedException());
      EQUAL(rslt.getQueryId(), "select");
      EQUAL(rslt.getConnectionId(), "c0");
      EQUAL(rslt.getContent().get(XmlFileQueryResource.Key.NodePath),
        "AAA.BBB");
      EQUAL(rslt.getContent().listParams().size(), 0);
      Table<String,Object> table = rslt.getResultTable();
      MapIterator<String,Object> recIt = table.records();
      TRUE(recIt.hasNext());
      Map<String,Object> rec0 = recIt.next();
      EQUAL(rec0.get("CCC"), "あいうえお");
      EQUAL(rec0.get("DDD"), "１２３");
      EQUAL(rec0.get("EEE"), "999");
      TRUE(recIt.hasNext());
      Map<String,Object> rec1 = recIt.next();
      EQUAL(rec1.get("CCC"), "かきくけこ");
      EQUAL(rec1.get("DDD"), "４５６");
      EQUAL(rec1.get("EEE"), "888");
      TRUE(recIt.hasNext());
      Map<String,Object> rec2 = recIt.next();
      EQUAL(rec2.get("CCC"), "さしすせそ");
      EQUAL(rec2.get("DDD"), "７８９");
      EQUAL(rec2.get("EEE"), "777");
      FALSE(recIt.hasNext());
      tran.commit();
    }
    catch (ReasonedException e) {
      tran.rollback();
      NG(e);
    }
    finally {
      tran.end();
    }
  }

  public void execute_qid_inputMap_Update()
  {
    MSG("更新を行う一つのクエリを実行した場合。");

    DefaultQueryTransaction tran = new DefaultQueryTransaction();
    tran.addConnectionFactory("c0", new XmlFileQueryConnectionFactory());

    Map<String,Object> map = new LinkedHashMap<String,Object>();
    map.put("ID", "b2");
    map.put("CCC", "たちつてと");
    map.put("DDD", "８８８");
    map.put("EEE", "123456789");

    try {
      tran.begin();

      QueryConnection conn = tran.getConnection("c0");
      Query query = conn.getQuery(new XmlFileQueryResource());

      QueryResult rslt = query.execute("select", map);
      TRUE(rslt.isSuccess());
      EQUAL(rslt.getResultCount(), 1);
      POSITIVE(rslt.getSpentTimeMillis());
      NULL(rslt.getReasonedException());
      EQUAL(rslt.getQueryId(), "select");
      EQUAL(rslt.getConnectionId(), "c0");
      EQUAL(rslt.getContent().get(XmlFileQueryResource.Key.NodePath),
        "AAA.BBB(ID=b2)");
      EQUAL(rslt.getContent().listParams().size(), 0);
      Table<String,Object> table = rslt.getResultTable();
      MapIterator<String,Object> recIt = table.records();
      TRUE(recIt.hasNext());
      Map<String,Object> rec1 = recIt.next();
      EQUAL(rec1.get("CCC"), "かきくけこ");
      EQUAL(rec1.get("DDD"), "４５６");
      EQUAL(rec1.get("EEE"), "888");
      FALSE(recIt.hasNext());

      rslt = query.execute("update", map);
      TRUE(rslt.isSuccess());
      EQUAL(rslt.getResultCount(), 1);
      POSITIVE(rslt.getSpentTimeMillis());
      NULL(rslt.getReasonedException());
      EQUAL(rslt.getQueryId(), "update");
      EQUAL(rslt.getConnectionId(), "c0");
      EQUAL(rslt.getContent().get(XmlFileQueryResource.Key.NodePath),
        "AAA.BBB(ID=b2)");
      List<QueryParam> paramL = rslt.getContent().listParams();
      EQUAL(paramL.size(), 3);
      EQUAL(paramL.get(0).getName(), "CCC");
      EQUAL(paramL.get(0).getValue(), "たちつてと");
      EQUAL(paramL.get(1).getName(), "DDD");
      EQUAL(paramL.get(1).getValue(), "８８８");
      EQUAL(paramL.get(2).getName(), "EEE");
      EQUAL(paramL.get(2).getValue(), "123456789");
      NULL(rslt.getResultTable());

      tran.commit();
    }
    catch (ReasonedException e) {
      tran.rollback();
      NG(e);
    }
    finally {
      tran.end();
    }

    tran = new DefaultQueryTransaction();
    tran.addConnectionFactory("c0", new XmlFileQueryConnectionFactory());

    map = new LinkedHashMap<String,Object>();
    map.put("ID", "b2");

    try {
      tran.begin();

      QueryConnection conn = tran.getConnection("c0");
      Query query = conn.getQuery(new XmlFileQueryResource());

      QueryResult rslt = query.execute("select", map);
      TRUE(rslt.isSuccess());
      EQUAL(rslt.getResultCount(), 1);
      POSITIVE(rslt.getSpentTimeMillis());
      NULL(rslt.getReasonedException());
      EQUAL(rslt.getQueryId(), "select");
      EQUAL(rslt.getConnectionId(), "c0");
      EQUAL(rslt.getContent().get(XmlFileQueryResource.Key.NodePath),
        "AAA.BBB(ID=b2)");
      EQUAL(rslt.getContent().listParams().size(), 0);
      Table<String,Object> table = rslt.getResultTable();
      MapIterator<String,Object> recIt = table.records();
      TRUE(recIt.hasNext());
      Map<String,Object> rec1 = recIt.next();
      EQUAL(rec1.get("CCC"), "たちつてと");
      EQUAL(rec1.get("DDD"), "８８８");
      EQUAL(rec1.get("EEE"), "123456789");
      FALSE(recIt.hasNext());

      tran.commit();
    }
    catch (ReasonedException e) {
      tran.rollback();
      NG(e);
    }
    finally {
      tran.end();
      rollbackDataFile();
    }
  }

  public void execute_qid_inputMap_Null()
  {
    MSG("引数がヌルの場合。");

    DefaultQueryTransaction tran = new DefaultQueryTransaction();
    tran.addConnectionFactory("c0", new XmlFileQueryConnectionFactory());

    Map<String,Object> map = new LinkedHashMap<String,Object>();

    try {
      tran.begin();

      QueryConnection conn = tran.getConnection("c0");
      Query query = conn.getQuery(new XmlFileQueryResource());

      try {
        query.execute(null, map);
        NG();
      } catch (AssertionError e) {
        OK(e);
      }

      try {
        query.execute("q0", null);
        NG();
      } catch (AssertionError e) {
        OK(e);
      }
    }
    catch (ReasonedException e) {
      tran.rollback();
    }
    finally {
      tran.end();
    }
  }

  public void execute_qid_inputMap_QueryIdNotFound()
  {
    MSG("指定されたクエリIDが見つからなかった場合。");

    DefaultQueryTransaction tran = new DefaultQueryTransaction();
    tran.addConnectionFactory("c0", new XmlFileQueryConnectionFactory());

    Map<String,Object> map = new LinkedHashMap<String,Object>();

    try {
      tran.begin();

      QueryConnection conn = tran.getConnection("c0");
      Query query = conn.getQuery(new XmlFileQueryResource());

      query.execute("xxxx", map);
      NG();
    }
    catch (ReasonedException e) {
      tran.rollback();
      EQUAL(e.getReason(), QueryResource.Error.QueryIdNotFound);
    }
    finally {
      tran.end();
    }
  }

  public void execute_qid_inputMap_IllegalTransactionState()
  {
    MSG("トランザクションの状態が不正な場合。");

    DefaultQueryTransaction tran = new DefaultQueryTransaction();
    tran.addConnectionFactory("c0", new XmlFileQueryConnectionFactory());

    Map<String,Object> map = new LinkedHashMap<String,Object>();

    try {
      QueryConnection conn = tran.getConnection("c0");
      Query query = conn.getQuery(new XmlFileQueryResource());
      query.execute("select", map);
      NG();
    }
    catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), QueryTransaction.Error.IllegalState);
    }
    catch (ReasonedException e) {
      tran.rollback();
      NG();
    }
    finally {
      tran.end();
    }

    tran = new DefaultQueryTransaction();
    tran.addConnectionFactory("c0", new XmlFileQueryConnectionFactory());

    try {
      tran.begin();
      QueryConnection conn = tran.getConnection("c0");
      tran.commit();
      Query query = conn.getQuery(new XmlFileQueryResource());
      tran.commit();
      query.execute("select", map);
      tran.commit();
    }
    catch (ReasonedRuntimeException e) {
      tran.rollback();
      NG(e);
    }
    catch (ReasonedException e) {
      tran.rollback();
      NG(e);
    }
    finally {
      tran.end();
    }
  }

  public void execute_qid_inputMap_FailToExecute()
  {
    MSG("クエリ実行時にエラーが発生した場合。");

    DefaultQueryTransaction tran = new DefaultQueryTransaction();
    tran.addConnectionFactory("c0", new XmlFileQueryConnectionFactory());

    Map<String,Object> map = new LinkedHashMap<String,Object>();

    try {
      tran.begin();

      QueryConnection conn = tran.getConnection("c0");
      Query query = conn.getQuery(new XmlFileQueryResource());

      query.execute("fail", map);
      NG();
    }
    catch (ReasonedException e) {
      tran.rollback();
      EQUAL(e.getReason(), QueryResource.Error.FailToCreateContent);
    }
    finally {
      tran.end();
    }

    tran = new DefaultQueryTransaction();
    tran.addConnectionFactory("c0", new XmlFileQueryConnectionFactory());

    map = new LinkedHashMap<String,Object>();

    try {
      tran.begin();

      QueryConnection conn = tran.getConnection("c0");
      Query query = conn.getQuery(new XmlFileQueryResource());

      query.execute("", map);
      NG();
    }
    catch (ReasonedException e) {
      tran.rollback();
      EQUAL(e.getReason(), Query.Error.FailToExecute);
      EQUAL(e.getCause().getClass(), NullPointerException.class);
    }
    finally {
      tran.end();
    }
  }

  public void executeSet_qidArr_inputMap()
  {
    MSG("複数のクエリIDの配列を指定してクエリを実行した場合。");

    DefaultQueryTransaction tran = new DefaultQueryTransaction();
    tran.addConnectionFactory("c0", new XmlFileQueryConnectionFactory());

    Map<String,Object> map = new LinkedHashMap<String,Object>();
    map.put("ID", "b2");
    map.put("CCC", "たちつてと");
    map.put("DDD", "８８８");
    map.put("EEE", "123456789");

    QueryResultList rsltLst;
    QueryResult rslt;
    List<QueryResult> rL;
    Table<String,Object> table;
    MapIterator<String,Object> recIt;
    Map<String,Object> rec;
    List<QueryParam> paramL;

    try {
      tran.begin();

      QueryConnection conn = tran.getConnection("c0");
      Query query = conn.getQuery(new XmlFileQueryResource());

      rsltLst = query.executeSet(new String[]{"select","update","select"}, map);
      tran.commit();

      TRUE(rsltLst.isSuccess());
      POSITIVE(rsltLst.getSpentTimeMillis());
      EQUAL(rsltLst.countResults(), 3);

      rL = rsltLst.getAllResults();
      EQUAL(rL.size(), 3);

      rslt = rL.get(0);
      TRUE(rslt.isSuccess());
      EQUAL(rslt.getResultCount(), 1);
      POSITIVE(rslt.getSpentTimeMillis());
      NULL(rslt.getReasonedException());
      EQUAL(rslt.getQueryId(), "select");
      EQUAL(rslt.getConnectionId(), "c0");
      EQUAL(rslt.getContent().get(XmlFileQueryResource.Key.NodePath),
        "AAA.BBB(ID=b2)");
      EQUAL(rslt.getContent().listParams().size(), 0);
      table = rslt.getResultTable();
      recIt = table.records();
      TRUE(recIt.hasNext());
      rec = recIt.next();
      EQUAL(rec.get("CCC"), "かきくけこ");
      EQUAL(rec.get("DDD"), "４５６");
      EQUAL(rec.get("EEE"), "888");
      FALSE(recIt.hasNext());

      rslt = rL.get(1);
      TRUE(rslt.isSuccess());
      EQUAL(rslt.getResultCount(), 1);
      POSITIVE(rslt.getSpentTimeMillis());
      NULL(rslt.getReasonedException());
      EQUAL(rslt.getQueryId(), "update");
      EQUAL(rslt.getConnectionId(), "c0");
      EQUAL(rslt.getContent().get(XmlFileQueryResource.Key.NodePath),
        "AAA.BBB(ID=b2)");
      paramL = rslt.getContent().listParams();
      EQUAL(paramL.size(), 3);
      EQUAL(paramL.get(0).getName(), "CCC");
      EQUAL(paramL.get(0).getValue(), "たちつてと");
      EQUAL(paramL.get(1).getName(), "DDD");
      EQUAL(paramL.get(1).getValue(), "８８８");
      EQUAL(paramL.get(2).getName(), "EEE");
      EQUAL(paramL.get(2).getValue(), "123456789");
      NULL(rslt.getResultTable());

      rslt = rL.get(2);
      TRUE(rslt.isSuccess());
      EQUAL(rslt.getResultCount(), 1);
      POSITIVE(rslt.getSpentTimeMillis());
      NULL(rslt.getReasonedException());
      EQUAL(rslt.getQueryId(), "select");
      EQUAL(rslt.getConnectionId(), "c0");
      EQUAL(rslt.getContent().get(XmlFileQueryResource.Key.NodePath),
        "AAA.BBB(ID=b2)");
      EQUAL(rslt.getContent().listParams().size(), 0);
      table = rslt.getResultTable();
      recIt = table.records();
      TRUE(recIt.hasNext());
      rec = recIt.next();
      EQUAL(rec.get("CCC"), "たちつてと");
      EQUAL(rec.get("DDD"), "８８８");
      EQUAL(rec.get("EEE"), "123456789");
      FALSE(recIt.hasNext());
    }
    catch (ReasonedException e) {
      tran.rollback();
      NG(e);
    }
    finally {
      tran.end();
      rollbackDataFile();
    }

    rsltLst = tran.getQueryResultList();
    TRUE(rsltLst.isSuccess());
    POSITIVE(rsltLst.getSpentTimeMillis());
    EQUAL(rsltLst.countResults(), 3);

    rL = rsltLst.getAllResults();
    EQUAL(rL.size(), 3);

    rslt = rL.get(0);
    TRUE(rslt.isSuccess());
    EQUAL(rslt.getResultCount(), 1);
    POSITIVE(rslt.getSpentTimeMillis());
    NULL(rslt.getReasonedException());
    EQUAL(rslt.getQueryId(), "select");
    EQUAL(rslt.getConnectionId(), "c0");
    EQUAL(rslt.getContent().get(XmlFileQueryResource.Key.NodePath),
      "AAA.BBB(ID=b2)");
    EQUAL(rslt.getContent().listParams().size(), 0);
    table = rslt.getResultTable();
    recIt = table.records();
    TRUE(recIt.hasNext());
    rec = recIt.next();
    EQUAL(rec.get("CCC"), "かきくけこ");
    EQUAL(rec.get("DDD"), "４５６");
    EQUAL(rec.get("EEE"), "888");
    FALSE(recIt.hasNext());

    rslt = rL.get(1);
    TRUE(rslt.isSuccess());
    EQUAL(rslt.getResultCount(), 1);
    POSITIVE(rslt.getSpentTimeMillis());
    NULL(rslt.getReasonedException());
    EQUAL(rslt.getQueryId(), "update");
    EQUAL(rslt.getConnectionId(), "c0");
    EQUAL(rslt.getContent().get(XmlFileQueryResource.Key.NodePath),
      "AAA.BBB(ID=b2)");
    paramL = rslt.getContent().listParams();
    EQUAL(paramL.size(), 3);
    EQUAL(paramL.get(0).getName(), "CCC");
    EQUAL(paramL.get(0).getValue(), "たちつてと");
    EQUAL(paramL.get(1).getName(), "DDD");
    EQUAL(paramL.get(1).getValue(), "８８８");
    EQUAL(paramL.get(2).getName(), "EEE");
    EQUAL(paramL.get(2).getValue(), "123456789");
    NULL(rslt.getResultTable());

    rslt = rL.get(2);
    TRUE(rslt.isSuccess());
    EQUAL(rslt.getResultCount(), 1);
    POSITIVE(rslt.getSpentTimeMillis());
    NULL(rslt.getReasonedException());
    EQUAL(rslt.getQueryId(), "select");
    EQUAL(rslt.getConnectionId(), "c0");
    EQUAL(rslt.getContent().get(XmlFileQueryResource.Key.NodePath),
      "AAA.BBB(ID=b2)");
    EQUAL(rslt.getContent().listParams().size(), 0);
    table = rslt.getResultTable();
    recIt = table.records();
    TRUE(recIt.hasNext());
    rec = recIt.next();
    EQUAL(rec.get("CCC"), "たちつてと");
    EQUAL(rec.get("DDD"), "８８８");
    EQUAL(rec.get("EEE"), "123456789");
    FALSE(recIt.hasNext());
  }

  public void executeSet_qidArr_inputMap_Null()
  {
    MSG("引数がヌルの場合。");

    DefaultQueryTransaction tran = new DefaultQueryTransaction();
    tran.addConnectionFactory("c0", new XmlFileQueryConnectionFactory());

    Map<String,Object> map = new LinkedHashMap<String,Object>();

    try {
      tran.begin();

      QueryConnection conn = tran.getConnection("c0");
      Query query = conn.getQuery(new XmlFileQueryResource());

      try {
        query.executeSet((String[])null, map);
        NG();
      } catch (AssertionError e) {
        OK(e);
      }

      try {
        query.executeSet(new String[]{"select","update"}, null);
        NG();
      } catch (AssertionError e) {
        OK(e);
      }
    }
    catch (ReasonedException e) {
      tran.rollback();
      NG();
    }
    finally {
      tran.end();
    }
  }

  public void executeSet_qidArr_inputMap_HasNullElement()
  {
    MSG("引数の配列にヌルの要素が含まれていた場合。");

    DefaultQueryTransaction tran = new DefaultQueryTransaction();
    tran.addConnectionFactory("c0", new XmlFileQueryConnectionFactory());

    Map<String,Object> map = new LinkedHashMap<String,Object>();

    try {
      tran.begin();

      QueryConnection conn = tran.getConnection("c0");
      Query query = conn.getQuery(new XmlFileQueryResource());
      try {
        query.executeSet(new String[]{"select",null,"update"}, map);
        NG();
      }
      catch (AssertionError e) {
        OK(e);
      }
    }
    catch (ReasonedException e) {
      tran.rollback();
      NG(e);
    }
    finally {
      tran.end();
    }
  }

  public void executeSet_qidArr_inputMap_QueryIdNotFound()
  {
    MSG("引数の配列に存在しないクエリIDが含まれていた場合。");

    DefaultQueryTransaction tran = new DefaultQueryTransaction();
    tran.addConnectionFactory("c0", new XmlFileQueryConnectionFactory());

    Map<String,Object> map = new LinkedHashMap<String,Object>();

    try {
      tran.begin();

      QueryConnection conn = tran.getConnection("c0");
      Query query = conn.getQuery(new XmlFileQueryResource());
      query.executeSet(new String[]{"select","xxx","update"}, map);
      NG();
    }
    catch (ReasonedException e) {
      tran.rollback();
      EQUAL(e.getReason(), QueryResource.Error.QueryIdNotFound);
      EQUAL(e.getMessage(), "xxx");
    }
    finally {
      tran.end();
    }
  }

  public void executeSet_qidArr_inputMap_IllegalTransactionState()
  {
    MSG("トランザクションの状態が不正な場合。");

    QueryTransaction tran =
      QueryTransactionManager.createThreadSafeTransaction("t0");
    tran.addConnectionFactory("c0", new XmlFileQueryConnectionFactory());

    Map<String,Object> map = new LinkedHashMap<String,Object>();
    map.put("ID", "b2");
    map.put("CCC", "たちつてと");
    map.put("DDD", "８８８");
    map.put("EEE", "123456789");

    QueryResultList rsltLst;
    QueryResult rslt;
    List<QueryResult> rL;
    Table<String,Object> table;
    MapIterator<String,Object> recIt;
    Map<String,Object> rec;
    List<QueryParam> paramL;

    String[] arr = {"select", "update"};

    try {
      tran.begin();

      Thread th = new Thread() {
        public void run() {
          try {
            Thread.sleep(1000L);
          } catch (InterruptedException e) {}
          QueryTransactionManager.getThreadSafeTransaction("t0").end();
        }
      };
      th.start();

      QueryConnection conn = tran.getConnection("c0");
      Query query = conn.getQuery(new XmlFileQueryResource());

      rsltLst = query.executeSet(arr, map);
      try {
        th.join();
      } catch (InterruptedException e) {}
      tran.commit();
      NG();
    }
    catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), QueryTransaction.Error.IllegalState);
      EQUAL(e.getMessage(), "Ended");
    }
    catch (ReasonedException e) {
      tran.rollback();
      NG();
    }
    finally {
      tran.end();
    }

    rsltLst = tran.getQueryResultList();
    TRUE(rsltLst.isSuccess());
    POSITIVE(rsltLst.getSpentTimeMillis());
    EQUAL(rsltLst.countResults(), 2);

    rL = rsltLst.getAllResults();
    EQUAL(rL.size(), 2);

    rslt = rL.get(0);
    TRUE(rslt.isSuccess());
    EQUAL(rslt.getResultCount(), 1);
    POSITIVE(rslt.getSpentTimeMillis());
    NULL(rslt.getReasonedException());
    EQUAL(rslt.getQueryId(), "select");
    EQUAL(rslt.getConnectionId(), "c0");
    EQUAL(rslt.getContent().get(XmlFileQueryResource.Key.NodePath),
      "AAA.BBB(ID=b2)");
    EQUAL(rslt.getContent().listParams().size(), 0);
    table = rslt.getResultTable();
    recIt = table.records();
    TRUE(recIt.hasNext());

    rec = recIt.next();
    EQUAL(rec.get("CCC"), "かきくけこ");
    EQUAL(rec.get("DDD"), "４５６");
    EQUAL(rec.get("EEE"), "888");
    FALSE(recIt.hasNext());

    rslt = rL.get(1);
    TRUE(rslt.isSuccess());
    EQUAL(rslt.getResultCount(), 1);
    POSITIVE(rslt.getSpentTimeMillis());
    NULL(rslt.getReasonedException());
    EQUAL(rslt.getQueryId(), "update");
    EQUAL(rslt.getConnectionId(), "c0");
    NULL(rslt.getResultTable());

    rollbackDataFile();
  }

  public void executeSet_qidArr_inputMap_FailToExecute()
  {
    MSG("クエリ実行時にエラーが発生した場合。");

    QueryTransaction tran =
      QueryTransactionManager.createThreadSafeTransaction("t0");
    tran.addConnectionFactory("c0", new XmlFileQueryConnectionFactory());

    Map<String,Object> map = new LinkedHashMap<String,Object>();
    map.put("ID", "b2");
    map.put("CCC", "たちつてと");
    map.put("DDD", "８８８");
    map.put("EEE", "123456789");

    QueryResultList rsltLst;
    QueryResult rslt;
    List<QueryResult> rL;
    Table<String,Object> table;
    MapIterator<String,Object> recIt;
    Map<String,Object> rec;
    List<QueryParam> paramL;

    try {
      tran.begin();

      QueryConnection conn = tran.getConnection("c0");
      Query query = conn.getQuery(new XmlFileQueryResource());

      rsltLst = query.executeSet(new String[]{"select","fail"}, map);
      NG();
    }
    catch (ReasonedException e) {
      tran.rollback();
      EQUAL(e.getReason(), QueryResource.Error.FailToCreateContent);
    }
    finally {
      tran.end();
    }

    rsltLst = tran.getQueryResultList();
    FALSE(rsltLst.isSuccess());
    POSITIVE(rsltLst.getSpentTimeMillis());
    EQUAL(rsltLst.countResults(), 2);

    rL = rsltLst.getAllResults();
    EQUAL(rL.size(), 2);

    rslt = rL.get(0);
    TRUE(rslt.isSuccess());
    EQUAL(rslt.getResultCount(), 1);
    POSITIVE(rslt.getSpentTimeMillis());
    NULL(rslt.getReasonedException());
    EQUAL(rslt.getQueryId(), "select");
    EQUAL(rslt.getConnectionId(), "c0");
    EQUAL(rslt.getContent().get(XmlFileQueryResource.Key.NodePath),
      "AAA.BBB(ID=b2)");
    EQUAL(rslt.getContent().listParams().size(), 0);
    table = rslt.getResultTable();
    recIt = table.records();
    TRUE(recIt.hasNext());

    rec = recIt.next();
    EQUAL(rec.get("CCC"), "かきくけこ");
    EQUAL(rec.get("DDD"), "４５６");
    EQUAL(rec.get("EEE"), "888");
    FALSE(recIt.hasNext());

    rslt = rL.get(1);
    FALSE(rslt.isSuccess());
    EQUAL(rslt.getResultCount(), 0);
    POSITIVE(rslt.getSpentTimeMillis());
    EQUAL(rslt.getReasonedException().getReason(),
      QueryResource.Error.FailToCreateContent);
    EQUAL(rslt.getQueryId(), "fail");
    EQUAL(rslt.getConnectionId(), "c0");
    NULL(rslt.getResultTable());

    rollbackDataFile();
  }

  public void executeSet_qidLst_inputMap()
  {
    MSG("複数のクエリIDのリストを指定してクエリを実行した場合。");

    DefaultQueryTransaction tran = new DefaultQueryTransaction();
    tran.addConnectionFactory("c0", new XmlFileQueryConnectionFactory());

    Map<String,Object> map = new LinkedHashMap<String,Object>();
    map.put("ID", "b2");
    map.put("CCC", "たちつてと");
    map.put("DDD", "８８８");
    map.put("EEE", "123456789");

    QueryResultList rsltLst;
    QueryResult rslt;
    List<QueryResult> rL;
    Table<String,Object> table;
    MapIterator<String,Object> recIt;
    Map<String,Object> rec;
    List<QueryParam> paramL;

    List<String> qidLst = new ArrayList<String>(3);
    qidLst.add("select");
    qidLst.add("update");
    qidLst.add("select");

    try {
      tran.begin();

      QueryConnection conn = tran.getConnection("c0");
      Query query = conn.getQuery(new XmlFileQueryResource());

      rsltLst = query.executeSet(qidLst, map);
      tran.commit();

      TRUE(rsltLst.isSuccess());
      POSITIVE(rsltLst.getSpentTimeMillis());
      EQUAL(rsltLst.countResults(), 3);

      rL = rsltLst.getAllResults();
      EQUAL(rL.size(), 3);

      rslt = rL.get(0);
      TRUE(rslt.isSuccess());
      EQUAL(rslt.getResultCount(), 1);
      POSITIVE(rslt.getSpentTimeMillis());
      NULL(rslt.getReasonedException());
      EQUAL(rslt.getQueryId(), "select");
      EQUAL(rslt.getConnectionId(), "c0");
      EQUAL(rslt.getContent().get(XmlFileQueryResource.Key.NodePath),
        "AAA.BBB(ID=b2)");
      EQUAL(rslt.getContent().listParams().size(), 0);
      table = rslt.getResultTable();
      recIt = table.records();
      TRUE(recIt.hasNext());
      rec = recIt.next();
      EQUAL(rec.get("CCC"), "かきくけこ");
      EQUAL(rec.get("DDD"), "４５６");
      EQUAL(rec.get("EEE"), "888");
      FALSE(recIt.hasNext());

      rslt = rL.get(1);
      TRUE(rslt.isSuccess());
      EQUAL(rslt.getResultCount(), 1);
      POSITIVE(rslt.getSpentTimeMillis());
      NULL(rslt.getReasonedException());
      EQUAL(rslt.getQueryId(), "update");
      EQUAL(rslt.getConnectionId(), "c0");
      EQUAL(rslt.getContent().get(XmlFileQueryResource.Key.NodePath),
        "AAA.BBB(ID=b2)");
      paramL = rslt.getContent().listParams();
      EQUAL(paramL.size(), 3);
      EQUAL(paramL.get(0).getName(), "CCC");
      EQUAL(paramL.get(0).getValue(), "たちつてと");
      EQUAL(paramL.get(1).getName(), "DDD");
      EQUAL(paramL.get(1).getValue(), "８８８");
      EQUAL(paramL.get(2).getName(), "EEE");
      EQUAL(paramL.get(2).getValue(), "123456789");
      NULL(rslt.getResultTable());

      rslt = rL.get(2);
      TRUE(rslt.isSuccess());
      EQUAL(rslt.getResultCount(), 1);
      POSITIVE(rslt.getSpentTimeMillis());
      NULL(rslt.getReasonedException());
      EQUAL(rslt.getQueryId(), "select");
      EQUAL(rslt.getConnectionId(), "c0");
      EQUAL(rslt.getContent().get(XmlFileQueryResource.Key.NodePath),
        "AAA.BBB(ID=b2)");
      EQUAL(rslt.getContent().listParams().size(), 0);
      table = rslt.getResultTable();
      recIt = table.records();
      TRUE(recIt.hasNext());
      rec = recIt.next();
      EQUAL(rec.get("CCC"), "たちつてと");
      EQUAL(rec.get("DDD"), "８８８");
      EQUAL(rec.get("EEE"), "123456789");
      FALSE(recIt.hasNext());
    }
    catch (ReasonedException e) {
      tran.rollback();
      NG(e);
    }
    finally {
      tran.end();
      rollbackDataFile();
    }

    rsltLst = tran.getQueryResultList();
    TRUE(rsltLst.isSuccess());
    POSITIVE(rsltLst.getSpentTimeMillis());
    EQUAL(rsltLst.countResults(), 3);

    rL = rsltLst.getAllResults();
    EQUAL(rL.size(), 3);

    rslt = rL.get(0);
    TRUE(rslt.isSuccess());
    EQUAL(rslt.getResultCount(), 1);
    POSITIVE(rslt.getSpentTimeMillis());
    NULL(rslt.getReasonedException());
    EQUAL(rslt.getQueryId(), "select");
    EQUAL(rslt.getConnectionId(), "c0");
    EQUAL(rslt.getContent().get(XmlFileQueryResource.Key.NodePath),
      "AAA.BBB(ID=b2)");
    EQUAL(rslt.getContent().listParams().size(), 0);
    table = rslt.getResultTable();
    recIt = table.records();
    TRUE(recIt.hasNext());
    rec = recIt.next();
    EQUAL(rec.get("CCC"), "かきくけこ");
    EQUAL(rec.get("DDD"), "４５６");
    EQUAL(rec.get("EEE"), "888");
    FALSE(recIt.hasNext());

    rslt = rL.get(1);
    TRUE(rslt.isSuccess());
    EQUAL(rslt.getResultCount(), 1);
    POSITIVE(rslt.getSpentTimeMillis());
    NULL(rslt.getReasonedException());
    EQUAL(rslt.getQueryId(), "update");
    EQUAL(rslt.getConnectionId(), "c0");
    EQUAL(rslt.getContent().get(XmlFileQueryResource.Key.NodePath),
      "AAA.BBB(ID=b2)");
    paramL = rslt.getContent().listParams();
    EQUAL(paramL.size(), 3);
    EQUAL(paramL.get(0).getName(), "CCC");
    EQUAL(paramL.get(0).getValue(), "たちつてと");
    EQUAL(paramL.get(1).getName(), "DDD");
    EQUAL(paramL.get(1).getValue(), "８８８");
    EQUAL(paramL.get(2).getName(), "EEE");
    EQUAL(paramL.get(2).getValue(), "123456789");
    NULL(rslt.getResultTable());

    rslt = rL.get(2);
    TRUE(rslt.isSuccess());
    EQUAL(rslt.getResultCount(), 1);
    POSITIVE(rslt.getSpentTimeMillis());
    NULL(rslt.getReasonedException());
    EQUAL(rslt.getQueryId(), "select");
    EQUAL(rslt.getConnectionId(), "c0");
    EQUAL(rslt.getContent().get(XmlFileQueryResource.Key.NodePath),
      "AAA.BBB(ID=b2)");
    EQUAL(rslt.getContent().listParams().size(), 0);
    table = rslt.getResultTable();
    recIt = table.records();
    TRUE(recIt.hasNext());
    rec = recIt.next();
    EQUAL(rec.get("CCC"), "たちつてと");
    EQUAL(rec.get("DDD"), "８８８");
    EQUAL(rec.get("EEE"), "123456789");
    FALSE(recIt.hasNext());
  }

  public void executeSet_qidLst_inputMap_Null()
  {
    MSG("引数がヌルの場合。");

    DefaultQueryTransaction tran = new DefaultQueryTransaction();
    tran.addConnectionFactory("c0", new XmlFileQueryConnectionFactory());

    Map<String,Object> map = new LinkedHashMap<String,Object>();
    List<String> qidLst = new LinkedList<String>();
    qidLst.add("select");
    qidLst.add("update");

    try {
      tran.begin();

      QueryConnection conn = tran.getConnection("c0");
      Query query = conn.getQuery(new XmlFileQueryResource());

      try {
        query.executeSet((List<String>)null, map);
        NG();
      } catch (AssertionError e) {
        OK(e);
      }

      try {
        query.executeSet(qidLst, null);
        NG();
      } catch (AssertionError e) {
        OK(e);
      }
    }
    catch (ReasonedException e) {
      tran.rollback();
      NG();
    }
    finally {
      tran.end();
    }
  }

  public void executeSet_qidLst_inputMap_HasNullElement()
  {
    MSG("引数のリストにヌルの要素が含まれていた場合。");

    DefaultQueryTransaction tran = new DefaultQueryTransaction();
    tran.addConnectionFactory("c0", new XmlFileQueryConnectionFactory());

    Map<String,Object> map = new LinkedHashMap<String,Object>();

    List<String> qidLst = new LinkedList<String>();
    qidLst.add("select");
    qidLst.add(null);
    qidLst.add("update");

    try {
      tran.begin();

      QueryConnection conn = tran.getConnection("c0");
      Query query = conn.getQuery(new XmlFileQueryResource());
      try {
        query.executeSet(qidLst, map);
        NG();
      } catch (AssertionError e) {
        OK(e);
      }
    }
    catch (ReasonedException e) {
      tran.rollback();
      NG(e);
    }
    finally {
      tran.end();
    }
  }

  public void executeSet_qidLst_inputMap_QueryIdNotFound()
  {
    MSG("引数のリストに存在しないクエリIDが含まれていた場合。");

    DefaultQueryTransaction tran = new DefaultQueryTransaction();
    tran.addConnectionFactory("c0", new XmlFileQueryConnectionFactory());

    Map<String,Object> map = new LinkedHashMap<String,Object>();

    List<String> qidLst = new LinkedList<String>();
    qidLst.add("select");
    qidLst.add("xxx");
    qidLst.add("update");

    try {
      tran.begin();

      QueryConnection conn = tran.getConnection("c0");
      Query query = conn.getQuery(new XmlFileQueryResource());
      query.executeSet(qidLst, map);
      NG();
    }
    catch (ReasonedException e) {
      tran.rollback();
      EQUAL(e.getReason(), QueryResource.Error.QueryIdNotFound);
      EQUAL(e.getMessage(), "xxx");
    }
    finally {
      tran.end();
    }
  }

  public void executeSet_qidLst_inputMap_IllegalTransactionState()
  {
    MSG("トランザクションの状態が不正な場合。");

    QueryTransaction tran =
      QueryTransactionManager.createThreadSafeTransaction("t0");
    tran.addConnectionFactory("c0", new XmlFileQueryConnectionFactory());

    Map<String,Object> map = new LinkedHashMap<String,Object>();
    map.put("ID", "b2");
    map.put("CCC", "たちつてと");
    map.put("DDD", "８８８");
    map.put("EEE", "123456789");

    QueryResultList rsltLst;
    QueryResult rslt;
    List<QueryResult> rL;
    Table<String,Object> table;
    MapIterator<String,Object> recIt;
    Map<String,Object> rec;
    List<QueryParam> paramL;

    List<String> qidLst = new LinkedList<String>();
    qidLst.add("select");
    qidLst.add("update");

    try {
      tran.begin();

      Thread th = new Thread() {
        public void run() {
          try {
            Thread.sleep(1000L);
          } catch (InterruptedException e) {}
          QueryTransactionManager.getThreadSafeTransaction("t0").end();
        }
      };
      th.start();

      QueryConnection conn = tran.getConnection("c0");
      Query query = conn.getQuery(new XmlFileQueryResource());

      rsltLst = query.executeSet(qidLst, map);
      try {
        th.join();
      } catch (InterruptedException e) {}
      tran.commit();
      NG();
    }
    catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), QueryTransaction.Error.IllegalState);
      EQUAL(e.getMessage(), "Ended");
    }
    catch (ReasonedException e) {
      tran.rollback();
      NG();
    }
    finally {
      tran.end();
    }

    rsltLst = tran.getQueryResultList();
    TRUE(rsltLst.isSuccess());
    POSITIVE(rsltLst.getSpentTimeMillis());
    EQUAL(rsltLst.countResults(), 2);

    rL = rsltLst.getAllResults();
    EQUAL(rL.size(), 2);

    rslt = rL.get(0);
    TRUE(rslt.isSuccess());
    EQUAL(rslt.getResultCount(), 1);
    POSITIVE(rslt.getSpentTimeMillis());
    NULL(rslt.getReasonedException());
    EQUAL(rslt.getQueryId(), "select");
    EQUAL(rslt.getConnectionId(), "c0");
    EQUAL(rslt.getContent().get(XmlFileQueryResource.Key.NodePath),
      "AAA.BBB(ID=b2)");
    EQUAL(rslt.getContent().listParams().size(), 0);
    table = rslt.getResultTable();
    recIt = table.records();
    TRUE(recIt.hasNext());

    rec = recIt.next();
    EQUAL(rec.get("CCC"), "かきくけこ");
    EQUAL(rec.get("DDD"), "４５６");
    EQUAL(rec.get("EEE"), "888");
    FALSE(recIt.hasNext());

    rslt = rL.get(1);
    TRUE(rslt.isSuccess());
    EQUAL(rslt.getResultCount(), 1);
    POSITIVE(rslt.getSpentTimeMillis());
    NULL(rslt.getReasonedException());
    EQUAL(rslt.getQueryId(), "update");
    EQUAL(rslt.getConnectionId(), "c0");
    NULL(rslt.getResultTable());

    rollbackDataFile();
  }

  public void executeSet_qidLst_inputMap_FailToExecute()
  {
    MSG("クエリの実行中にエラーが発生した場合。");

    QueryTransaction tran =
      QueryTransactionManager.createThreadSafeTransaction("t0");
    tran.addConnectionFactory("c0", new XmlFileQueryConnectionFactory());

    Map<String,Object> map = new LinkedHashMap<String,Object>();
    map.put("ID", "b2");
    map.put("CCC", "たちつてと");
    map.put("DDD", "８８８");
    map.put("EEE", "123456789");

    QueryResultList rsltLst;
    QueryResult rslt;
    List<QueryResult> rL;
    Table<String,Object> table;
    MapIterator<String,Object> recIt;
    Map<String,Object> rec;
    List<QueryParam> paramL;

    List<String> qidLst = new LinkedList<String>();
    qidLst.add("select");
    qidLst.add("fail");

    try {
      tran.begin();

      QueryConnection conn = tran.getConnection("c0");
      Query query = conn.getQuery(new XmlFileQueryResource());

      rsltLst = query.executeSet(qidLst, map);
      NG();
    }
    catch (ReasonedException e) {
      tran.rollback();
      EQUAL(e.getReason(), QueryResource.Error.FailToCreateContent);
    }
    finally {
      tran.end();
    }

    rsltLst = tran.getQueryResultList();
    FALSE(rsltLst.isSuccess());
    POSITIVE(rsltLst.getSpentTimeMillis());
    EQUAL(rsltLst.countResults(), 2);

    rL = rsltLst.getAllResults();
    EQUAL(rL.size(), 2);

    rslt = rL.get(0);
    TRUE(rslt.isSuccess());
    EQUAL(rslt.getResultCount(), 1);
    POSITIVE(rslt.getSpentTimeMillis());
    NULL(rslt.getReasonedException());
    EQUAL(rslt.getQueryId(), "select");
    EQUAL(rslt.getConnectionId(), "c0");
    EQUAL(rslt.getContent().get(XmlFileQueryResource.Key.NodePath),
      "AAA.BBB(ID=b2)");
    EQUAL(rslt.getContent().listParams().size(), 0);
    table = rslt.getResultTable();
    recIt = table.records();
    TRUE(recIt.hasNext());

    rec = recIt.next();
    EQUAL(rec.get("CCC"), "かきくけこ");
    EQUAL(rec.get("DDD"), "４５６");
    EQUAL(rec.get("EEE"), "888");
    FALSE(recIt.hasNext());

    rslt = rL.get(1);
    FALSE(rslt.isSuccess());
    EQUAL(rslt.getResultCount(), 0);
    POSITIVE(rslt.getSpentTimeMillis());
    EQUAL(rslt.getReasonedException().getReason(),
      QueryResource.Error.FailToCreateContent);
    EQUAL(rslt.getQueryId(), "fail");
    EQUAL(rslt.getConnectionId(), "c0");
    NULL(rslt.getResultTable());

    rollbackDataFile();
  }

  public void executeAll_inputMap()
  {
    MSG("クエリ・リソースの中に定義されている全てのクエリIDのクエリを実行。");

    DefaultQueryTransaction tran = new DefaultQueryTransaction();
    tran.addConnectionFactory("c0", new XmlFileQueryConnectionFactory());

    Map<String,Object> map = new LinkedHashMap<String,Object>();
    map.put("ID", "b2");
    map.put("CCC", "たちつてと");
    map.put("DDD", "８８８");
    map.put("EEE", "123456789");

    QueryResultList rsltLst;
    QueryResult rslt;
    List<QueryResult> rL;
    Table<String,Object> table;
    MapIterator<String,Object> recIt;
    Map<String,Object> rec;
    List<QueryParam> paramL;

    try {
      tran.begin();

      QueryConnection conn = tran.getConnection("c0");
      Query query = conn.getQuery(new XmlFileQueryResource());

      rsltLst = query.executeAll(map);
      tran.commit();

      TRUE(rsltLst.isSuccess());
      POSITIVE(rsltLst.getSpentTimeMillis());
      EQUAL(rsltLst.countResults(), 2);

      rL = rsltLst.getAllResults();
      EQUAL(rL.size(), 2);

      rslt = rL.get(0);
      TRUE(rslt.isSuccess());
      EQUAL(rslt.getResultCount(), 1);
      POSITIVE(rslt.getSpentTimeMillis());
      NULL(rslt.getReasonedException());
      EQUAL(rslt.getQueryId(), "select");
      EQUAL(rslt.getConnectionId(), "c0");
      EQUAL(rslt.getContent().get(XmlFileQueryResource.Key.NodePath),
        "AAA.BBB(ID=b2)");
      EQUAL(rslt.getContent().listParams().size(), 0);
      table = rslt.getResultTable();
      recIt = table.records();
      TRUE(recIt.hasNext());
      rec = recIt.next();
      EQUAL(rec.get("CCC"), "かきくけこ");
      EQUAL(rec.get("DDD"), "４５６");
      EQUAL(rec.get("EEE"), "888");
      FALSE(recIt.hasNext());

      rslt = rL.get(1);
      TRUE(rslt.isSuccess());
      EQUAL(rslt.getResultCount(), 1);
      POSITIVE(rslt.getSpentTimeMillis());
      NULL(rslt.getReasonedException());
      EQUAL(rslt.getQueryId(), "update");
      EQUAL(rslt.getConnectionId(), "c0");
      EQUAL(rslt.getContent().get(XmlFileQueryResource.Key.NodePath),
        "AAA.BBB(ID=b2)");
      paramL = rslt.getContent().listParams();
      EQUAL(paramL.size(), 3);
      EQUAL(paramL.get(0).getName(), "CCC");
      EQUAL(paramL.get(0).getValue(), "たちつてと");
      EQUAL(paramL.get(1).getName(), "DDD");
      EQUAL(paramL.get(1).getValue(), "８８８");
      EQUAL(paramL.get(2).getName(), "EEE");
      EQUAL(paramL.get(2).getValue(), "123456789");
      NULL(rslt.getResultTable());
    }
    catch (ReasonedException e) {
      tran.rollback();
      NG(e);
    }
    finally {
      tran.end();
    }

    rsltLst = tran.getQueryResultList();
    TRUE(rsltLst.isSuccess());
    POSITIVE(rsltLst.getSpentTimeMillis());
    EQUAL(rsltLst.countResults(), 2);

    rL = rsltLst.getAllResults();
    EQUAL(rL.size(), 2);

    rslt = rL.get(0);
    TRUE(rslt.isSuccess());
    EQUAL(rslt.getResultCount(), 1);
    POSITIVE(rslt.getSpentTimeMillis());
    NULL(rslt.getReasonedException());
    EQUAL(rslt.getQueryId(), "select");
    EQUAL(rslt.getConnectionId(), "c0");
    EQUAL(rslt.getContent().get(XmlFileQueryResource.Key.NodePath),
      "AAA.BBB(ID=b2)");
    EQUAL(rslt.getContent().listParams().size(), 0);
    table = rslt.getResultTable();
    recIt = table.records();
    TRUE(recIt.hasNext());
    rec = recIt.next();
    EQUAL(rec.get("CCC"), "かきくけこ");
    EQUAL(rec.get("DDD"), "４５６");
    EQUAL(rec.get("EEE"), "888");
    FALSE(recIt.hasNext());

    rslt = rL.get(1);
    TRUE(rslt.isSuccess());
    EQUAL(rslt.getResultCount(), 1);
    POSITIVE(rslt.getSpentTimeMillis());
    NULL(rslt.getReasonedException());
    EQUAL(rslt.getQueryId(), "update");
    EQUAL(rslt.getConnectionId(), "c0");
    EQUAL(rslt.getContent().get(XmlFileQueryResource.Key.NodePath),
      "AAA.BBB(ID=b2)");
    paramL = rslt.getContent().listParams();
    EQUAL(paramL.size(), 3);
    EQUAL(paramL.get(0).getName(), "CCC");
    EQUAL(paramL.get(0).getValue(), "たちつてと");
    EQUAL(paramL.get(1).getName(), "DDD");
    EQUAL(paramL.get(1).getValue(), "８８８");
    EQUAL(paramL.get(2).getName(), "EEE");
    EQUAL(paramL.get(2).getValue(), "123456789");
    NULL(rslt.getResultTable());

    tran = new DefaultQueryTransaction();
    tran.addConnectionFactory("c0", new XmlFileQueryConnectionFactory());

    map = new LinkedHashMap<String,Object>();
    map.put("ID", "b2");

    try {
      tran.begin();

      QueryConnection conn = tran.getConnection("c0");
      Query query = conn.getQuery(new XmlFileQueryResource());

      rslt = query.execute("select", map);
      TRUE(rslt.isSuccess());
      EQUAL(rslt.getResultCount(), 1);
      POSITIVE(rslt.getSpentTimeMillis());
      NULL(rslt.getReasonedException());
      EQUAL(rslt.getQueryId(), "select");
      EQUAL(rslt.getConnectionId(), "c0");
      EQUAL(rslt.getContent().get(XmlFileQueryResource.Key.NodePath),
        "AAA.BBB(ID=b2)");
      EQUAL(rslt.getContent().listParams().size(), 0);
      table = rslt.getResultTable();
      recIt = table.records();
      TRUE(recIt.hasNext());
      rec = recIt.next();
      EQUAL(rec.get("CCC"), "たちつてと");
      EQUAL(rec.get("DDD"), "８８８");
      EQUAL(rec.get("EEE"), "123456789");
      FALSE(recIt.hasNext());

      tran.commit();
    }
    catch (ReasonedException e) {
      tran.rollback();
      NG(e);
    }
    finally {
      tran.end();
      rollbackDataFile();
    }
  }

  public void executeAll_inputMap_Null()
  {
    MSG("引数がヌルの場合。");

    DefaultQueryTransaction tran = new DefaultQueryTransaction();
    tran.addConnectionFactory("c0", new XmlFileQueryConnectionFactory());

    Map<String,Object> map = new LinkedHashMap<String,Object>();

    try {
      tran.begin();

      QueryConnection conn = tran.getConnection("c0");
      Query query = conn.getQuery(new XmlFileQueryResource());

      try {
        query.executeAll(null);
        NG();
      } catch (AssertionError e) {
        OK(e);
      }
    } 
    catch (ReasonedException e) {
      tran.rollback();
      NG();
    }
    finally {
      tran.end();
    }
  }

  public void executeAll_inputMap_IllegalTransactionState()
  {
    MSG("トランザクションの状態が不正な場合。");

    QueryTransaction tran =
      QueryTransactionManager.createThreadSafeTransaction("t0");
    tran.addConnectionFactory("c0", new XmlFileQueryConnectionFactory());

    Map<String,Object> map = new LinkedHashMap<String,Object>();
    map.put("ID", "b2");
    map.put("CCC", "たちつてと");
    map.put("DDD", "８８８");
    map.put("EEE", "123456789");

    QueryResultList rsltLst;
    QueryResult rslt;
    List<QueryResult> rL;
    Table<String,Object> table;
    MapIterator<String,Object> recIt;
    Map<String,Object> rec;
    List<QueryParam> paramL;

    try {
      tran.begin();

      Thread th = new Thread() {
        public void run() {
          try {
            Thread.sleep(1000L);
          } catch (InterruptedException e) {}
          QueryTransactionManager.getThreadSafeTransaction("t0").end();
        }
      };
      th.start();

      QueryConnection conn = tran.getConnection("c0");
      Query query = conn.getQuery(new XmlFileQueryResource());

      rsltLst = query.executeAll(map);
      try {
        th.join();
      } catch (InterruptedException e) {}
      tran.commit();
      NG();
    }
    catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), QueryTransaction.Error.IllegalState);
      EQUAL(e.getMessage(), "Ended");
    }
    catch (ReasonedException e) {
      tran.rollback();
      NG();
    }
    finally {
      tran.end();
    }

    rsltLst = tran.getQueryResultList();
    TRUE(rsltLst.isSuccess());
    POSITIVE(rsltLst.getSpentTimeMillis());
    EQUAL(rsltLst.countResults(), 2);

    rL = rsltLst.getAllResults();
    EQUAL(rL.size(), 2);

    rslt = rL.get(0);
    TRUE(rslt.isSuccess());
    EQUAL(rslt.getResultCount(), 1);
    POSITIVE(rslt.getSpentTimeMillis());
    NULL(rslt.getReasonedException());
    EQUAL(rslt.getQueryId(), "select");
    EQUAL(rslt.getConnectionId(), "c0");
    EQUAL(rslt.getContent().get(XmlFileQueryResource.Key.NodePath),
      "AAA.BBB(ID=b2)");
    EQUAL(rslt.getContent().listParams().size(), 0);
    table = rslt.getResultTable();
    recIt = table.records();
    TRUE(recIt.hasNext());

    rec = recIt.next();
    EQUAL(rec.get("CCC"), "かきくけこ");
    EQUAL(rec.get("DDD"), "４５６");
    EQUAL(rec.get("EEE"), "888");
    FALSE(recIt.hasNext());

    rslt = rL.get(1);
    TRUE(rslt.isSuccess());
    EQUAL(rslt.getResultCount(), 1);
    POSITIVE(rslt.getSpentTimeMillis());
    NULL(rslt.getReasonedException());
    EQUAL(rslt.getQueryId(), "update");
    EQUAL(rslt.getConnectionId(), "c0");
    NULL(rslt.getResultTable());

    tran = new DefaultQueryTransaction();
    tran.addConnectionFactory("c0", new XmlFileQueryConnectionFactory());

    map = new LinkedHashMap<String,Object>();
    map.put("ID", "b2");

    try {
      tran.begin();

      QueryConnection conn = tran.getConnection("c0");
      Query query = conn.getQuery(new XmlFileQueryResource());

      rslt = query.execute("select", map);
      TRUE(rslt.isSuccess());
      EQUAL(rslt.getResultCount(), 1);
      POSITIVE(rslt.getSpentTimeMillis());
      NULL(rslt.getReasonedException());
      EQUAL(rslt.getQueryId(), "select");
      EQUAL(rslt.getConnectionId(), "c0");
      EQUAL(rslt.getContent().get(XmlFileQueryResource.Key.NodePath),
        "AAA.BBB(ID=b2)");
      EQUAL(rslt.getContent().listParams().size(), 0);
      table = rslt.getResultTable();
      recIt = table.records();
      TRUE(recIt.hasNext());
      rec = recIt.next();
      EQUAL(rec.get("CCC"), "かきくけこ");
      EQUAL(rec.get("DDD"), "４５６");
      EQUAL(rec.get("EEE"), "888");
      FALSE(recIt.hasNext());

      tran.commit();
    }
    catch (ReasonedException e) {
      tran.rollback();
      NG(e);
    }
    finally {
      tran.end();
      rollbackDataFile();
    }
  }

  public void executeAll_inputMap_FailToExecute()
  {
    MSG("クエリの実行中にエラーが発生した場合。");

    QueryTransaction tran =
      QueryTransactionManager.createThreadSafeTransaction("t0");
    tran.addConnectionFactory("c0", new XmlFileQueryConnectionFactory());

    Map<String,Object> map = new LinkedHashMap<String,Object>();
    map.put("ID", "b2");
    map.put("CCC", "たちつてと");
    map.put("DDD", "８８８");
    map.put("EEE", "123456789");

    QueryResultList rsltLst;
    QueryResult rslt;
    List<QueryResult> rL;
    Table<String,Object> table;
    MapIterator<String,Object> recIt;
    Map<String,Object> rec;
    List<QueryParam> paramL;

    try {
      tran.begin();

      QueryConnection conn = tran.getConnection("c0");
      Query query = conn.getQuery(new XmlFileQueryResource());

      FAIL = true;
      rsltLst = query.executeAll(map);
      NG();
    }
    catch (ReasonedException e) {
      tran.rollback();
      EQUAL(e.getReason(), Query.Error.FailToExecute);
    }
    finally {
      tran.end();
      FAIL = false;
    }

    rsltLst = tran.getQueryResultList();
    FALSE(rsltLst.isSuccess());
    POSITIVE(rsltLst.getSpentTimeMillis());
    EQUAL(rsltLst.countResults(), 2);

    rL = rsltLst.getAllResults();
    EQUAL(rL.size(), 2);

    rslt = rL.get(0);
    TRUE(rslt.isSuccess());
    EQUAL(rslt.getResultCount(), 1);
    POSITIVE(rslt.getSpentTimeMillis());
    NULL(rslt.getReasonedException());
    EQUAL(rslt.getQueryId(), "select");
    EQUAL(rslt.getConnectionId(), "c0");
    EQUAL(rslt.getContent().get(XmlFileQueryResource.Key.NodePath),
      "AAA.BBB(ID=b2)");
    EQUAL(rslt.getContent().listParams().size(), 0);
    table = rslt.getResultTable();
    recIt = table.records();
    TRUE(recIt.hasNext());

    rec = recIt.next();
    EQUAL(rec.get("CCC"), "かきくけこ");
    EQUAL(rec.get("DDD"), "４５６");
    EQUAL(rec.get("EEE"), "888");
    FALSE(recIt.hasNext());

    rslt = rL.get(1);
    FALSE(rslt.isSuccess());
    EQUAL(rslt.getResultCount(), 0);
    POSITIVE(rslt.getSpentTimeMillis());
    EQUAL(rslt.getReasonedException().getReason(), Query.Error.FailToExecute);
    EQUAL(rslt.getQueryId(), "update");
    EQUAL(rslt.getConnectionId(), "c0");
    NULL(rslt.getResultTable());

    rollbackDataFile();
  }


  /* -- Timeout --*/

  public void execute_qid_inputMap_Timeout()
  {
    MSG("タイムアウトが発生した場合。");

    DefaultQueryTransaction tran = new DefaultQueryTransaction();
    tran.addConnectionFactory("c0", new XmlFileQueryConnectionFactory());

    Map<String,Object> map = new LinkedHashMap<String,Object>();

    try {
      tran.begin(1000L);

      QueryConnection conn = tran.getConnection("c0");
      try {
        Thread.sleep(1000L);
      } catch (InterruptedException e) {
        NG(e);
      }
      Query query = conn.getQuery(new XmlFileQueryResource());
      query.execute("select", map);
      NG();
    }
    catch (ReasonedException e) {
      tran.rollback();
      EQUAL(e.getReason(), Query.Error.Timeout);
    }
    finally {
      tran.end();
    }
  }

  public void executeSet_qidArr_inputMap_Timeout()
  {
    MSG("タイムアウトが発生した場合。");

    DefaultQueryTransaction tran = new DefaultQueryTransaction();
    tran.addConnectionFactory("c0", new XmlFileQueryConnectionFactory());

    Map<String,Object> map = new LinkedHashMap<String,Object>();
    map.put("ID", "b2");
    map.put("CCC", "たちつてと");
    map.put("DDD", "８８８");
    map.put("EEE", "123456789");

    try {
      tran.begin(200L);

      QueryConnection conn = tran.getConnection("c0");
      Query query = conn.getQuery(new XmlFileQueryResource());

      QueryResultList rsltLst = query.executeSet(
        new String[]{"select","update","select"}, map);
      NG();
    }
    catch (ReasonedException e) {
      tran.rollback();
      EQUAL(e.getReason(), Query.Error.Timeout);
    }
    finally {
      tran.end();
    }

    rollbackDataFile();
  }

  public void executeSet_qidLst_inputMap_Timeout()
  {
    MSG("タイムアウトが発生した場合。");

    DefaultQueryTransaction tran = new DefaultQueryTransaction();
    tran.addConnectionFactory("c0", new XmlFileQueryConnectionFactory());

    Map<String,Object> map = new LinkedHashMap<String,Object>();
    map.put("ID", "b2");
    map.put("CCC", "たちつてと");
    map.put("DDD", "８８８");
    map.put("EEE", "123456789");

    List<String> qidLst = new ArrayList<String>(3);
    qidLst.add("select");
    qidLst.add("update");
    qidLst.add("select");

    try {
      tran.begin(200L);

      QueryConnection conn = tran.getConnection("c0");
      Query query = conn.getQuery(new XmlFileQueryResource());

      QueryResultList rsltLst = query.executeSet(qidLst, map);
      NG();
    }
    catch (ReasonedException e) {
      tran.rollback();
      EQUAL(e.getReason(), Query.Error.Timeout);
    }
    finally {
      tran.end();
    }

    rollbackDataFile();
  }

  public void executeAll_Timeout()
  {
    MSG("タイムアウトが発生した場合。");

    DefaultQueryTransaction tran = new DefaultQueryTransaction();
    tran.addConnectionFactory("c0", new XmlFileQueryConnectionFactory());

    Map<String,Object> map = new LinkedHashMap<String,Object>();
    map.put("ID", "b2");
    map.put("CCC", "たちつてと");
    map.put("DDD", "８８８");
    map.put("EEE", "123456789");

    try {
      tran.begin(200L);

      QueryConnection conn = tran.getConnection("c0");
      Query query = conn.getQuery(new XmlFileQueryResource());

      QueryResultList rsltLst = query.executeAll(map);
      NG();
    }
    catch (ReasonedException e) {
      tran.rollback();
      EQUAL(e.getReason(), Query.Error.Timeout);
    }
    finally {
      tran.end();
    }
  }
}
