package ts.query;

import ts.tester.UnitTest;
import ts.util.*;
import ts.util.table.*;
import ts.util.resource.*;
import java.util.*;

public class QueryTransactionManagerTest extends UnitTest
{
  public static void main(String[] args)
  {
    run(QueryTransactionManagerTest.class, args);
  }

  static class BadTransaction extends DefaultQueryTransaction {
    public BadTransaction() {
      throw new RuntimeException("!!!");
    }
  }

  static class MyConnectionFactory implements QueryConnectionFactory {
    public MyConnectionFactory() {}
    @Override
    public QueryConnection create(final String cid, final QueryTransaction tr) {
      return new QueryConnection() {
        private boolean isClosed = true;
        @Override
        public String getId() { return cid; }
        @Override
        public long getLimitTimeMillis() { return tr.getLimitTimeMillis(); }
        @Override
        public Query getQuery(QueryResource res) { return null; }
        @Override
        public int executeContent(QueryContent cont) throws ReasonedException {
          return 0;
        }
        @Override
        public int executeContent(QueryContent cont, Table<String,Object> tbl)
        throws ReasonedException {
          return 0;
        }
        @Override
        public QueryResultList getTransactionQueryResultList() {
          return tr.getQueryResultList();
        }
        @Override
        public boolean isClosed() { return this.isClosed; }
        @Override
        public void open() {
          LOG("[Open begin]");SLEEP();this.isClosed = false;LOG("[Open end]");
        }
        @Override
        public void commit() {
          LOG("[Commit begin]"); SLEEP(); LOG("[Commit end]");
        }
        @Override
        public void rollback() {
          LOG("[Rollback begin]"); SLEEP(); LOG("[Rollback end]");
        }
        @Override
        public void close() {
          LOG("[Close begin]");SLEEP();this.isClosed = true;LOG("[Close end]");
        }
      };
    }
  }

  static StringBuilder BUF = new StringBuilder();
  static void CLEAR_LOG() {
    BUF = new StringBuilder();
  }
  static void LOG(String str) {
    BUF.append(str);
  }
  static String OUTPUT_LOG() {
    return BUF.toString();
  }

  @Override
  protected void preInvocation(String method)
  {
    CLEAR_LOG();
  }

  static long SLEEP_TIME = 0L;
  static void SLEEP() {
    try {
      Thread.sleep(SLEEP_TIME);
    } catch (InterruptedException e) {}
  }


  public void createAndGetThreadLocalTransaction_InLocalThread()
  {
    MSG("スレッド・ローカルなトランザクションをローカルなスレッド内で操作。");

    QueryTransaction tr;

    tr = QueryTransactionManager.createThreadLocalTransaction();
    EQUAL(tr.getState(), QueryTransaction.State.Created);
    tr.begin();
    EQUAL(tr.getState(), QueryTransaction.State.Begined);
    tr.commit();
    EQUAL(tr.getState(), QueryTransaction.State.Committed);
    tr.end();
    EQUAL(tr.getState(), QueryTransaction.State.Ended);
    EQUAL(tr.getQueryResultList().countResults(), 0);
    EQUAL(OUTPUT_LOG(), "");

    tr = QueryTransactionManager.createThreadLocalTransaction();
    EQUAL(tr.getState(), QueryTransaction.State.Created);
    tr.addConnectionFactory("c0", new MyConnectionFactory());
    EQUAL(tr.getState(), QueryTransaction.State.Created);
    tr.begin();
    EQUAL(tr.getState(), QueryTransaction.State.Begined);
    tr.commit();
    EQUAL(tr.getState(), QueryTransaction.State.Committed);
    tr.end();
    EQUAL(tr.getState(), QueryTransaction.State.Ended);
    EQUAL(tr.getQueryResultList().countResults(), 0);
    EQUAL(OUTPUT_LOG(), "");


    QueryTransaction tr0, tr1, tr2, tr3, tr4;

    tr0 = QueryTransactionManager.createThreadLocalTransaction();
    tr1 = QueryTransactionManager.getThreadLocalTransaction();
    EQUAL(tr0, tr1);
    EQUAL(tr0.hashCode(), tr1.hashCode());
    EQUAL(tr1.getState(), QueryTransaction.State.Created);
    tr1.addConnectionFactory("c0", new MyConnectionFactory());
    EQUAL(tr1.getState(), QueryTransaction.State.Created);
    tr1.begin();
    EQUAL(tr1.getState(), QueryTransaction.State.Begined);
    try {
      tr1.getConnection("c0");
      EQUAL(tr1.getState(), QueryTransaction.State.Begined);
      tr1.commit();
      EQUAL(tr1.getState(), QueryTransaction.State.Committed);
    } catch (ReasonedException e) {
      tr1.rollback();
      EQUAL(tr1.getState(), QueryTransaction.State.Rollbacked);
    } finally {
      tr1.end();
      EQUAL(tr1.getState(), QueryTransaction.State.Ended);
      EQUAL(tr.getQueryResultList().countResults(), 0);
    }

    EQUAL(OUTPUT_LOG(),
    "[Open begin][Open end][Commit begin][Commit end][Close begin][Close end]");
  }

  public void createAndGetThreadLocalTransaction_InOtherThread()
  {
    MSG("スレッド・ローカルなトランザクションを別スレッド内で操作。");

    final QueryTransaction tr;
    tr = QueryTransactionManager.createThreadLocalTransaction();
    EQUAL(tr.getState(), QueryTransaction.State.Created);
    tr.addConnectionFactory("c0", new MyConnectionFactory());

    Thread th0 = new Thread() {
      public void run() {
        try {
          tr.begin();
          NG();
        } catch (ReasonedRuntimeException e) {
          EQUAL(e.getReason(),
            QueryTransactionManager.Error.ThreadLocalNotExist);
        }
      }
    };
    try {
      th0.start();
      th0.join();
    } catch (InterruptedException e) {
      NG(e);
    }

    EQUAL(tr.getState(), QueryTransaction.State.Created);

    tr.begin(1000L);
    OK("... transaction begin");
    EQUAL(tr.getState(), QueryTransaction.State.Begined);

    EQUAL(tr.getLimitTimeMillis(), tr.getBeginTimeMillis() + 1000L);

    try {
      EQUAL(tr.getConnection("c0").getId(), "c0");
    } catch (ReasonedException e) {
      NG(e);
    }

    Thread th1 = new Thread() {
      public void run() {
        try {
          tr.getConnection("c0");
          NG();
        } catch (ReasonedException e) {
          NG();
        } catch (ReasonedRuntimeException e) {
          EQUAL(e.getReason(),
            QueryTransactionManager.Error.ThreadLocalNotExist);
        }
      }
    };
    try {
      th1.start();
      th1.join();
    } catch (InterruptedException e) {
      NG(e);
    }

    Thread th2 = new Thread() {
      public void run() {
        try {
          tr.rollback();
          NG();
        } catch (ReasonedRuntimeException e) {
          EQUAL(e.getReason(),
            QueryTransactionManager.Error.ThreadLocalNotExist);
        }
      }
    };
    try {
      th2.start();
      th2.join();
    } catch (InterruptedException e) {
      NG(e);
    }

    tr.rollback();
    OK("... transaction rollback");
    EQUAL(tr.getState(), QueryTransaction.State.Rollbacked);

    Thread th3 = new Thread() {
      public void run() {
        try {
          tr.end();
          NG();
        } catch (ReasonedRuntimeException e) {
          EQUAL(e.getReason(),
            QueryTransactionManager.Error.ThreadLocalNotExist);
        }
      }
    };
    try {
      th3.start();
      th3.join();
    } catch (InterruptedException e) {
      NG(e);
    }

    tr.end();
    OK("... transaction end");
    EQUAL(tr.getState(), QueryTransaction.State.Ended);
    EQUAL(tr.getQueryResultList().countResults(), 0);
  }

  public void createAndGetThreadLocalTransaction_Null()
  {
    MSG("引数がヌルの場合。");

    try {
      QueryTransactionManager.createThreadLocalTransaction(null);
      NG();
    } catch (AssertionError e) {
      OK(e);
    }
  }

  public void createAndGetThreadLocalTransaction_NotFound()
  {
    MSG("スレッド・ローカルなトランザクションを作成前に取得しようとした場合。");

    try {
      Thread th = new Thread() {
        public void run() {
          try {
            QueryTransactionManager.getThreadLocalTransaction();
            NG();
          } catch (ReasonedRuntimeException e) {
            EQUAL(e.getReason(),
              QueryTransactionManager.Error.ThreadLocalNotExist);
          }
        }
      };
      th.start();
      th.join();
    } catch (Exception e) {
      NG(e);
    }
  }

  public void createAndGetThreadLocalTransaction_FailToCreate()
  {
    MSG("スレッド・ローカルなトランザクションの作成に失敗した場合。");

    QueryTransaction tr;
    try {
      tr = QueryTransactionManager.createThreadLocalTransaction(
        BadTransaction.class);
      NG();
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(),
        QueryTransactionManager.Error.ThreadLocalFailToCreate);
      EQUAL(e.getMessage(),
        "ts.query.QueryTransactionManagerTest$BadTransaction");
      EQUAL(e.getCause().getClass(), RuntimeException.class);
      EQUAL(e.getCause().getMessage(), "!!!");
    }
  }

  public void createAndGetThreadLocalTransaction_AlreadExists()
  {
    MSG("既にスレッド・ローカルなトランザクションが作成済みの場合。");

    QueryTransaction tr =
      QueryTransactionManager.createThreadLocalTransaction();
    EQUAL(tr.getState(), QueryTransaction.State.Created);

    try {
      QueryTransactionManager.createThreadLocalTransaction();
      NG();
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(),
        QueryTransactionManager.Error.ThreadLocalAlreadyExists);
    }

    tr.end();
    EQUAL(tr.getState(), QueryTransaction.State.Ended);
    EQUAL(tr.getQueryResultList().countResults(), 0);
  }

  public void createAndGetThreadSafeTransaction_InOtherThreads()
  {
    MSG("スレッド・セーフなトランザクションを別のスレッド内で操作。");

    final QueryTransaction tr;
    tr = QueryTransactionManager.createThreadSafeTransaction("tr00");
    EQUAL(tr.getState(), QueryTransaction.State.Created);
    tr.addConnectionFactory("c0", new MyConnectionFactory());

    Thread th0 = new Thread() {
      public void run() {
        tr.begin();
        EQUAL(tr.getState(), QueryTransaction.State.Begined);
      }
    };
    try {
      th0.start();
      th0.join();
    } catch (InterruptedException e) {
      NG(e);
    }

    EQUAL(tr.getLimitTimeMillis(), -1L);

    try {
      EQUAL(tr.getConnection("c0").getId(), "c0");
    } catch (ReasonedException e) {
      NG(e);
    }

    Thread th1 = new Thread() {
      public void run() {
        try {
          QueryConnection c0 = tr.getConnection("c0");
          EQUAL(c0.getId(), "c0");
          FALSE(c0.isClosed());
        } catch (ReasonedException e) {
          NG(e);
        }
      }
    };
    try {
      th1.start();
      th1.join();
    } catch (InterruptedException e) {
      NG(e);
    }

    Thread th2 = new Thread() {
      public void run() {
        tr.rollback();
        EQUAL(tr.getState(), QueryTransaction.State.Rollbacked);
      }
    };
    try {
      th2.start();
      th2.join();
    } catch (InterruptedException e) {
      NG(e);
    }

    Thread th3 = new Thread() {
      public void run() {
        tr.end();
        EQUAL(tr.getState(), QueryTransaction.State.Ended);
        EQUAL(tr.getQueryResultList().countResults(), 0);
      }
    };
    try {
      th3.start();
      th3.join();
    } catch (InterruptedException e) {
      NG(e);
    }

    final QueryTransaction tr0, tr1;
    tr0 = QueryTransactionManager.createThreadSafeTransaction("tr00");
    tr1 = QueryTransactionManager.getThreadSafeTransaction("tr00");
    EQUAL(tr0, tr1);
    EQUAL(tr0.hashCode(), tr1.hashCode());
    EQUAL(tr1.getState(), QueryTransaction.State.Created);

    tr1.addConnectionFactory("c0", new MyConnectionFactory());

    Thread th10 = new Thread() {
      public void run() {
        QueryTransaction tr10;
        tr10 = QueryTransactionManager.getThreadSafeTransaction("tr00");
        EQUAL(tr0, tr10);
        EQUAL(tr0.hashCode(), tr10.hashCode());
        tr10.begin(1000L);
        EQUAL(tr10.getState(), QueryTransaction.State.Begined);
      }
    };
    try {
      th10.start();
      th10.join();
    } catch (InterruptedException e) {
      NG(e);
    }

    EQUAL(tr0.getLimitTimeMillis(), tr0.getBeginTimeMillis() + 1000L);

    try {
      QueryConnection c0 = tr0.getConnection("c0");
      EQUAL(c0.getId(), "c0");
      FALSE(c0.isClosed());
    } catch (ReasonedException e) {
      NG(e);
    }

    Thread th11 = new Thread() {
      public void run() {
        QueryTransaction tr11;
        tr11 = QueryTransactionManager.getThreadSafeTransaction("tr00");
        EQUAL(tr0, tr11);
        EQUAL(tr0.hashCode(), tr11.hashCode());
        try {
          tr11.begin(1000L);
          NG();
        } catch (ReasonedRuntimeException e) {
          EQUAL(e.getReason(), QueryTransaction.Error.IllegalState);
        }
        EQUAL(tr11.getState(), QueryTransaction.State.Begined);
      }
    };
    try {
      th11.start();
      th11.join();
    } catch (InterruptedException e) {
      NG(e);
    }

    Thread th12 = new Thread() {
      public void run() {
        QueryTransaction tr12;
        tr12 = QueryTransactionManager.getThreadSafeTransaction("tr00");
        EQUAL(tr0, tr12);
        EQUAL(tr0.hashCode(), tr12.hashCode());
        tr12.commit();
        EQUAL(tr12.getState(), QueryTransaction.State.Committed);
      }
    };
    try {
      th12.start();
      th12.join();
    } catch (InterruptedException e) {
      NG(e);
    }

    Thread th13 = new Thread() {
      public void run() {
        QueryTransaction tr13;
        tr13 = QueryTransactionManager.getThreadSafeTransaction("tr00");
        EQUAL(tr0, tr13);
        EQUAL(tr0.hashCode(), tr13.hashCode());
        tr13.end();
        EQUAL(tr13.getState(), QueryTransaction.State.Ended);
        EQUAL(tr.getQueryResultList().countResults(), 0);
      }
    };
    try {
      th13.start();
      th13.join();
    } catch (InterruptedException e) {
      NG(e);
    }

    EQUAL(tr1.getState(), QueryTransaction.State.Ended);
  }

  public void createAndGetThreadSafeTransaction_Null()
  {
    MSG("引数がヌルの場合。");

    final QueryTransaction tr;
    tr = QueryTransactionManager.createThreadSafeTransaction(null);
    EQUAL(tr.getState(), QueryTransaction.State.Created);
    tr.addConnectionFactory("c0", new MyConnectionFactory());

    Thread th0 = new Thread() {
      public void run() {
        tr.begin(1000L);
        EQUAL(tr.getState(), QueryTransaction.State.Begined);
      }
    };
    try {
      th0.start();
      th0.join();
    } catch (InterruptedException e) {
      NG(e);
    }

    EQUAL(tr.getLimitTimeMillis(), tr.getBeginTimeMillis() + 1000L);

    try {
      EQUAL(tr.getConnection("c0").getId(), "c0");
    } catch (ReasonedException e) {
      NG(e);
    }

    Thread th1 = new Thread() {
      public void run() {
        try {
          QueryConnection c0 = tr.getConnection("c0");
          EQUAL(c0.getId(), "c0");
          FALSE(c0.isClosed());
        } catch (ReasonedException e) {
          NG(e);
        }
      }
    };
    try {
      th1.start();
      th1.join();
    } catch (InterruptedException e) {
      NG(e);
    }

    Thread th2 = new Thread() {
      public void run() {
        tr.rollback();
        EQUAL(tr.getState(), QueryTransaction.State.Rollbacked);
      }
    };
    try {
      th2.start();
      th2.join();
    } catch (InterruptedException e) {
      NG(e);
    }

    Thread th3 = new Thread() {
      public void run() {
        tr.end();
        EQUAL(tr.getState(), QueryTransaction.State.Ended);
        EQUAL(tr.getQueryResultList().countResults(), 0);
      }
    };
    try {
      th3.start();
      th3.join();
    } catch (InterruptedException e) {
      NG(e);
    }


    final QueryTransaction tr0, tr1;
    tr0 = QueryTransactionManager.createThreadSafeTransaction(null);
    tr1 = QueryTransactionManager.getThreadSafeTransaction(null);
    EQUAL(tr0, tr1);
    EQUAL(tr0.hashCode(), tr1.hashCode());
    EQUAL(tr1.getState(), QueryTransaction.State.Created);

    tr1.addConnectionFactory("c0", new MyConnectionFactory());

    Thread th10 = new Thread() {
      public void run() {
        QueryTransaction tr10;
        tr10 = QueryTransactionManager.getThreadSafeTransaction(null);
        EQUAL(tr0, tr10);
        EQUAL(tr0.hashCode(), tr10.hashCode());
        tr10.begin(1000L);
        EQUAL(tr10.getState(), QueryTransaction.State.Begined);
      }
    };
    try {
      th10.start();
      th10.join();
    } catch (InterruptedException e) {
      NG(e);
    }

    EQUAL(tr0.getLimitTimeMillis(), tr0.getBeginTimeMillis() + 1000L);

    try {
      QueryConnection c0 = tr0.getConnection("c0");
      EQUAL(c0.getId(), "c0");
      FALSE(c0.isClosed());
    } catch (ReasonedException e) {
      NG(e);
    }
    Thread th11 = new Thread() {
      public void run() {
        QueryTransaction tr11;
        tr11 = QueryTransactionManager.getThreadSafeTransaction(null);
        EQUAL(tr0, tr11);
        EQUAL(tr0.hashCode(), tr11.hashCode());
        try {
          tr11.begin(1000L);
          NG();
        } catch (ReasonedRuntimeException e) {
          EQUAL(e.getReason(), QueryTransaction.Error.IllegalState);
        }
        EQUAL(tr11.getState(), QueryTransaction.State.Begined);
      }
    };
    try {
      th11.start();
      th11.join();
    } catch (InterruptedException e) {
      NG(e);
    }

    Thread th12 = new Thread() {
      public void run() {
        QueryTransaction tr12;
        tr12 = QueryTransactionManager.getThreadSafeTransaction(null);
        EQUAL(tr0, tr12);
        EQUAL(tr0.hashCode(), tr12.hashCode());
        tr12.commit();
        EQUAL(tr12.getState(), QueryTransaction.State.Committed);
      }
    };
    try {
      th12.start();
      th12.join();
    } catch (InterruptedException e) {
      NG(e);
    }

    Thread th13 = new Thread() {
      public void run() {
        QueryTransaction tr13;
        tr13 = QueryTransactionManager.getThreadSafeTransaction(null);
        EQUAL(tr0, tr13);
        EQUAL(tr0.hashCode(), tr13.hashCode());
        tr13.end();
        EQUAL(tr13.getState(), QueryTransaction.State.Ended);
        EQUAL(tr.getQueryResultList().countResults(), 0);
      }
    };
    try {
      th13.start();
      th13.join();
    } catch (InterruptedException e) {
      NG(e);
    }

    EQUAL(tr1.getState(), QueryTransaction.State.Ended);
  }

  public void createAndGetThreadSafeTransaction_AlreadExists()
  {
    MSG("既にスレッド・セーフなトランザクションが作成済みの場合。");

    QueryTransaction tr;
    tr = QueryTransactionManager.createThreadSafeTransaction("t0");
    EQUAL(tr.getState(), QueryTransaction.State.Created);

    try {
      QueryTransactionManager.createThreadSafeTransaction("t0");
      NG();
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(),
        QueryTransactionManager.Error.ThreadSafeAlreadyExists);
      EQUAL(e.getMessage(), "t0 - ts.query.DefaultQueryTransaction");
    }

    tr.end();
    EQUAL(tr.getState(), QueryTransaction.State.Ended);
    EQUAL(tr.getQueryResultList().countResults(), 0);
  }

  public void createAndGetThreadSafeTransaction_FailToCreate()
  {
    MSG("スレッド・セーフなトランザクションの作成に失敗した場合。");

    QueryTransaction tr;
    try {
      tr = QueryTransactionManager.createThreadSafeTransaction(
        "t1", BadTransaction.class);
      NG();
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(),
        QueryTransactionManager.Error.ThreadSafeFailToCreate);
      EQUAL(e.getMessage(), "t1 - ts.query.QueryTransactionManagerTest$BadTransaction");
      EQUAL(e.getCause().getClass(), RuntimeException.class);
      EQUAL(e.getCause().getMessage(), "!!!");
    }
  }

  public void createAndGetThreadSafeTransaction_NotFound()
  {
    MSG("スレッド・セーフなトランザクションを作成前に取得しようとした場合。");

    try {
      QueryTransactionManager.getThreadSafeTransaction("key");
      NG();
    } catch (ReasonedRuntimeException e) {
      OK(e);
      EQUAL(e.getReason(), QueryTransactionManager.Error.ThreadSafeNotExist);
    }
  }

  public void ThreadLocalTransaction_equals()
  {
    MSG("スレッド・ローカル・トランザクションの等値判定メソッドの確認。");

    QueryTransaction tr =QueryTransactionManager.createThreadLocalTransaction();
    QueryTransaction tr0 = QueryTransactionManager.getThreadLocalTransaction();
    FALSE(tr.equals(null));
    FALSE(tr.equals(new DefaultQueryTransaction()));
    TRUE(tr.equals(tr0));
    tr.end();
  }

  public void ThreadSafeTransaction_equals()
  {
    MSG("スレッド・セーフ・トランザクションの等値判定メソッドの確認。");

    QueryTransaction tr=QueryTransactionManager.createThreadSafeTransaction("");
    QueryTransaction tr0 = QueryTransactionManager.getThreadSafeTransaction("");
    FALSE(tr.equals(null));
    FALSE(tr.equals(new DefaultQueryTransaction()));
    TRUE(tr.equals(tr0));
    tr.end();
  }
}
