package ts.query;

import ts.tester.UnitTest;
import ts.tester.function.ObjectInspector;
import ts.util.*;
import ts.util.file.*;
import ts.util.resource.*;
import ts.util.table.*;
import java.io.*;
import java.util.*;

public class ControllableQueryTest extends UnitTest
{
  public static void main(String[] args)
  {
    run(ControllableQueryTest.class, args);
  }

  static StringBuilder LOG = null;
  static void CLEAR_LOG() { LOG = new StringBuilder(); }

  static class MyExecution extends QueryExecution {
    public MyExecution(QueryExecutionConfig config) throws ReasonedException
    { super(config); }
    public MyExecution(QueryExecutionConfig config, IQueryConnection conn)
    { super(config, conn); }
    @Override
    protected void prepareContent(QueryExecutionContent content,
    Map<String,Object> inputMap) throws ReasonedException {}
    @Override
    protected void executeContent(QueryExecutionContent content)
    throws ReasonedException {
      LOG.append("[" + getExecutionId() + "]");
    }
  }

  static class MyConnection extends QueryConnection {
    public MyConnection(QueryConnectionConfig config) { super(config); }
    public MyConnection(QueryConnectionConfig config, IQueryTransaction tran)
    { super(config, tran); }
    @Override
    protected long getOpenTimeMillis() { return 0L; }
    @Override
    public void open() throws ReasonedException {}
    @Override
    public void commit() throws ReasonedException {}
    @Override
    public void rollback() {}
    @Override
    public void close() {}
    @Override
    public boolean isOpened() { return true; }
    @Override
    public boolean isClosed() { return false; }
  }

  protected void preInvocation(String method)
  {
    CLEAR_LOG();
  }

  public void constructor_config_NotUseControlXmlFile()
  {
    MSG("クエリ設定オブジェクトを引数にとるコンストラクタの確認。");
    MSG("- クエリ設定の中に実行IDの制御が記述されている場合。");

    final String QID =
      "ControllableQueryTest_constructor_config_NotUseControlXmlFile";

    QueryConfig config = new QueryConfig(QID);
    ControllableQuery query = new ControllableQuery(config);
    TRUE (query instanceof Query);
    TRUE (query instanceof IQuery);
    EQUAL(query.getConfig().getResource().getFirstValue(
      "ts-query.query.execution.control.xmlfile"),
      "");
    EQUAL(query.getConfig().getResource(), query.getControlResource());
  }

  public void constructor_config_UseControlXmlFile()
  {
    MSG("クエリ設定オブジェクトを引数にとるコンストラクタの確認。");
    MSG("- 別XMLファイルに実行IDの制御が記述されている場合。");

    final String QID =
      "ControllableQueryTest_constructor_config_UseControlXmlFile";

    QueryConfig config = new QueryConfig(QID);
    ControllableQuery query = new ControllableQuery(config);
    TRUE (query instanceof Query);
    TRUE (query instanceof IQuery);
    EQUAL(query.getConfig().getResource().getFirstValue(
      "ts-query.query.execution.control.xmlfile"),
      "control/ControllableQueryTest.xml");
    NOTEQUAL(query.getConfig().getResource(), query.getControlResource());
  }

  public void constructor_config_UseControlXmlFile_FileNotFound()
  {
    MSG("制御ファイルが見つからなかった場合。");

    final String QID =
      "ControllableQueryTest_constructor_config_UseControlXmlFile_FileNotFound";

    QueryConfig config = new QueryConfig(QID);
    try {
      new ControllableQuery(config);
      NG();
    } catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(), ControllableQuery.Error.QueryControlFileNotFound);
    }
  }

  public void constructor_config_UseControlXmlFile_FailToLoad()
  {
    MSG("制御ファイルのロードに失敗した場合。");

    final String QID =
      "ControllableQueryTest_constructor_config_UseControlXmlFile_FailToLoad";

    QueryConfig config = new QueryConfig(QID);
    try {
      new ControllableQuery(config);
      NG();
    } catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(), ControllableQuery.Error.FailToLoadQueryControlFile);
    }
  }

  public void constructor_config_Null()
  {
    MSG("引数がヌルの場合。");

    try {
      new ControllableQuery(null);
      NG();
    } catch (AssertionError e) {
      OK(e);
    }
  }

  public void constructor_config_tran()
  {
    MSG("クエリ設定オブジェクトとトランザクションを引数にとるコンストラクタの確認。");

    final String QID =
      "ControllableQueryTest_constructor_config_NotUseControlXmlFile";

    IQueryTransaction tran = new QueryTransaction();

    QueryConfig config = new QueryConfig(QID);
    ControllableQuery query = new ControllableQuery(config, tran);
    TRUE (query instanceof Query);
    TRUE (query instanceof IQuery);
    EQUAL(query.getTransaction(), tran);
    EQUAL(query.getConfig().getResource().getFirstValue(
      "ts-query.query.execution.control.xmlfile"),
      "");
    EQUAL(query.getConfig().getResource(), query.getControlResource());
  }

  public void constructor_config_tran_Null()
  {
    MSG("引数がヌルの場合。");

    final String QID =
      "ControllableQueryTest_constructor_config_NotUseControlXmlFile";

    IQueryTransaction tran = new QueryTransaction();
    QueryConfig config = new QueryConfig(QID);

    try {
      new ControllableQuery(null, tran);
      NG();
    } catch (AssertionError e) {
      OK(e);
    }

    try {
      new ControllableQuery(config, null);
      NG();
    } catch (AssertionError e) {
      OK(e);
    }
  }

  public void getParamValue_param_from_inputMap_result_FromInputMap()
  {
    MSG("パラメータ値を取得するメソッドの確認。");
    MSG("- 入力パラメータ・マップから取得。");

    final String QID =
      "ControllableQueryTest_constructor_config_NotUseControlXmlFile";

    QueryConfig config = new QueryConfig(QID);
    ControllableQuery query = new ControllableQuery(config);

    Header<String> header0 = new ArrayListTable.Header<String>(4);
    header0.addColumn("A");
    header0.addColumn("B");
    header0.addColumn("C");
    header0.addColumn("D");
    header0.addColumn("E");
    Table<String,Serializable> table0 =
      new ArrayListTable<String,Serializable>(header0);
    Map<String,Serializable> rec = table0.appendNew();
    rec.put("A", "x");
    rec.put("B", "y");
    rec.put("C", "z");
    rec.put("D", "v");
    rec.put("E", "w");

    QueryResult rslt = new QueryResult("q0");
    rslt.getResultTableMap().put("f0", table0);

    Map<String,Object> inputMap = new HashMap<String,Object>();
    inputMap.put("A", "abcdefg");
    inputMap.put("B", 123L);
    inputMap.put("C", false);
    inputMap.put("D", new DateTime(2012,01,23));

    EQUAL(query.getParamValue("A", "", inputMap, rslt), "abcdefg");
    EQUAL(query.getParamValue("B", "", inputMap, rslt), "123");
    EQUAL(query.getParamValue("C", "", inputMap, rslt), "false");
    EQUAL(query.getParamValue("D", "", inputMap, rslt), "2012-01-23 00:00:00");
    EQUAL(query.getParamValue("E", "", inputMap, rslt), null);
  }

  public void getParamValue_param_from_inputMap_result_FromResult()
  {
    MSG("パラメータ値を取得するメソッドの確認。");
    MSG("- クエリ結果オブジェクトから取得。");

    final String QID =
      "ControllableQueryTest_constructor_config_NotUseControlXmlFile";

    QueryConfig config = new QueryConfig(QID);
    ControllableQuery query = new ControllableQuery(config);

    Header<String> header0 = new ArrayListTable.Header<String>(4);
    header0.addColumn("A");
    header0.addColumn("B");
    header0.addColumn("C");
    header0.addColumn("D");
    Table<String,Serializable> table0 =
      new ArrayListTable<String,Serializable>(header0);
    Map<String,Serializable> rec = table0.appendNew();
    rec.put("A", "hijklmn");
    rec.put("B", 987L);
    rec.put("C", true);
    rec.put("D", new DateTime(2012,02,04));

    QueryResult rslt = new QueryResult("q0");
    rslt.getResultTableMap().put("f0", table0);

    Map<String,Object> inputMap = new HashMap<String,Object>();
    inputMap.put("A", "x");
    inputMap.put("B", "y");
    inputMap.put("C", "z");
    inputMap.put("D", "v");
    inputMap.put("E", "w");

    EQUAL(query.getParamValue("A", "f0", inputMap, rslt), "hijklmn");
    EQUAL(query.getParamValue("B", "f0", inputMap, rslt), "987");
    EQUAL(query.getParamValue("C", "f0", inputMap, rslt), "true");
    EQUAL(query.getParamValue("D", "f0", inputMap, rslt),"2012-02-04 00:00:00");
    EQUAL(query.getParamValue("E", "f0", inputMap, rslt), null);
  }

  public void checkExistsCondition_value_condition()
  {
    MSG("パラメータの存在チェックを行うメソッドの確認。");

    final String QID =
      "ControllableQueryTest_constructor_config_NotUseControlXmlFile";

    QueryConfig config = new QueryConfig(QID);
    ControllableQuery query = new ControllableQuery(config);

    EQUAL(query.checkExistsCondition(null, ""), true);
    EQUAL(query.checkExistsCondition(null, "true"), false);
    EQUAL(query.checkExistsCondition(null, "false"), true);
    try {
      query.checkExistsCondition(null, "zzz");
      NG();
    } catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(), ControllableQuery.Error.IllegalAttributeValue);
    }

    EQUAL(query.checkExistsCondition("x", ""), true);
    EQUAL(query.checkExistsCondition("x", "true"), true);
    EQUAL(query.checkExistsCondition("x", "false"), false);
    try {
      query.checkExistsCondition("x", "zzz");
      NG();
    } catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(), ControllableQuery.Error.IllegalAttributeValue);
    }

    EQUAL(query.checkExistsCondition("", ""), true);
    EQUAL(query.checkExistsCondition("", "true"), true);
    EQUAL(query.checkExistsCondition("", "false"), false);
    try {
      query.checkExistsCondition("", "zzz");
      NG();
    } catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(), ControllableQuery.Error.IllegalAttributeValue);
    }
  }

  public void checkEqualsCondition_value_condition()
  {
    MSG("パラメータの等値チェックを行うメソッドの確認。");

    final String QID =
      "ControllableQueryTest_constructor_config_NotUseControlXmlFile";

    QueryConfig config = new QueryConfig(QID);
    ControllableQuery query = new ControllableQuery(config);

    EQUAL(query.checkEqualsCondition(null, ""), true);
    EQUAL(query.checkEqualsCondition(null, "abc"), false);
    EQUAL(query.checkEqualsCondition(null, "DEF"), false);

    EQUAL(query.checkEqualsCondition("abc", ""), true);
    EQUAL(query.checkEqualsCondition("abc", "abc"), true);
    EQUAL(query.checkEqualsCondition("abc", "DEF"), false);

    EQUAL(query.checkEqualsCondition("def", ""), true);
    EQUAL(query.checkEqualsCondition("def", "abc"), false);
    EQUAL(query.checkEqualsCondition("def", "DEF"), false);

    EQUAL(query.checkEqualsCondition("", ""), true);
    EQUAL(query.checkEqualsCondition("", "abc"), false);
    EQUAL(query.checkEqualsCondition("", "DEF"), false);
  }

  public void checkNotEqualCondition_value_condition()
  {
    MSG("パラメータの不等値チェックを行うメソッドの確認。");

    final String QID =
      "ControllableQueryTest_constructor_config_NotUseControlXmlFile";

    QueryConfig config = new QueryConfig(QID);
    ControllableQuery query = new ControllableQuery(config);

    EQUAL(query.checkNotEqualCondition(null, ""), true);
    EQUAL(query.checkNotEqualCondition(null, "abc"), true);
    EQUAL(query.checkNotEqualCondition(null, "DEF"), true);

    EQUAL(query.checkNotEqualCondition("abc", ""), true);
    EQUAL(query.checkNotEqualCondition("abc", "abc"), false);
    EQUAL(query.checkNotEqualCondition("abc", "DEF"), true);

    EQUAL(query.checkNotEqualCondition("def", ""), true);
    EQUAL(query.checkNotEqualCondition("def", "abc"), true);
    EQUAL(query.checkNotEqualCondition("def", "DEF"), true);

    EQUAL(query.checkNotEqualCondition("", ""), true);
    EQUAL(query.checkNotEqualCondition("", "abc"), true);
    EQUAL(query.checkNotEqualCondition("", "DEF"), true);
  }

  public void checkContainsCondition_value_condition()
  {
    MSG("パラメータの包含チェックを行うメソッドの確認。");

    final String QID =
      "ControllableQueryTest_constructor_config_NotUseControlXmlFile";

    QueryConfig config = new QueryConfig(QID);
    ControllableQuery query = new ControllableQuery(config);

    EQUAL(query.checkContainsCondition(null, "", ","), true);
    EQUAL(query.checkContainsCondition(null, "abc", ","), false);
    EQUAL(query.checkContainsCondition(null, "DEF", ","), false);
    EQUAL(query.checkContainsCondition(null, "abc,def,ghi", ","), false);
    EQUAL(query.checkContainsCondition(null, "ABC,DEF,GHI", ","), false);

    EQUAL(query.checkContainsCondition("abc", "", ","), true);
    EQUAL(query.checkContainsCondition("abc", "abc", ","), true);
    EQUAL(query.checkContainsCondition("abc", "DEF", ","), false);
    EQUAL(query.checkContainsCondition("abc", "abc,def,ghi", ","), true);
    EQUAL(query.checkContainsCondition("abc", "ABC,DEF,GHI", ","), false);

    EQUAL(query.checkContainsCondition("abc", "", ""), true);
    EQUAL(query.checkContainsCondition("abc", "abc", ""), true);
    EQUAL(query.checkContainsCondition("abc", "DEF", ""), false);
    EQUAL(query.checkContainsCondition("abc", "abc,def,ghi", ""), false);
    EQUAL(query.checkContainsCondition("abc", "ABC,DEF,GHI", ""), false);

    EQUAL(query.checkContainsCondition("def", "", ","), true);
    EQUAL(query.checkContainsCondition("def", "abc", ","), false);
    EQUAL(query.checkContainsCondition("def", "DEF", ","), false);
    EQUAL(query.checkContainsCondition("def", "abc,def,ghi", ","), true);
    EQUAL(query.checkContainsCondition("def", "ABC,DEF,GHI", ","), false);

    EQUAL(query.checkContainsCondition("", "", ","), true);
    EQUAL(query.checkContainsCondition("", "abc", ","), false);
    EQUAL(query.checkContainsCondition("", "DEF", ","), false);
    EQUAL(query.checkContainsCondition("", "abc,def,ghi", ","), false);
    EQUAL(query.checkContainsCondition("", "ABC,DEF,GHI", ","), false);
  }

  public void checkNotContainCondition_value_condition()
  {
    MSG("パラメータの不包含チェックを行うメソッドの確認。");

    final String QID =
      "ControllableQueryTest_constructor_config_NotUseControlXmlFile";

    QueryConfig config = new QueryConfig(QID);
    ControllableQuery query = new ControllableQuery(config);

    EQUAL(query.checkNotContainCondition(null, "", ","), true);
    EQUAL(query.checkNotContainCondition(null, "abc", ","), true);
    EQUAL(query.checkNotContainCondition(null, "DEF", ","), true);
    EQUAL(query.checkNotContainCondition(null, "abc,def,ghi", ","), true);
    EQUAL(query.checkNotContainCondition(null, "ABC,DEF,GHI", ","), true);

    EQUAL(query.checkNotContainCondition("abc", "", ","), true);
    EQUAL(query.checkNotContainCondition("abc", "abc", ","), false);
    EQUAL(query.checkNotContainCondition("abc", "DEF", ","), true);
    EQUAL(query.checkNotContainCondition("abc", "abc,def,ghi", ","), false);
    EQUAL(query.checkNotContainCondition("abc", "ABC,DEF,GHI", ","), true);

    EQUAL(query.checkNotContainCondition("def", "", ","), true);
    EQUAL(query.checkNotContainCondition("def", "abc", ","), true);
    EQUAL(query.checkNotContainCondition("def", "DEF", ","), true);
    EQUAL(query.checkNotContainCondition("def", "abc,def,ghi", ","), false);
    EQUAL(query.checkNotContainCondition("def", "ABC,DEF,GHI", ","), true);

    EQUAL(query.checkNotContainCondition("", "", ","), true);
    EQUAL(query.checkNotContainCondition("", "abc", ","), true);
    EQUAL(query.checkNotContainCondition("", "DEF", ","), true);
    EQUAL(query.checkNotContainCondition("", "abc,def,ghi", ","), true);
    EQUAL(query.checkNotContainCondition("", "ABC,DEF,GHI", ","), true);
  }

  public void executeQuery_inputMap_result_0()
  {
    MSG("処理する実行IDを制御するクエリを処理するメソッドの確認。");
    MSG("- 制御リソースが空の場合。");

    final String QID =
      "ControllableQueryTest_executeQuery_inputMap_result_0";

    QueryConfig config = new QueryConfig(QID);
    ControllableQuery query = new ControllableQuery(config);

    Map<String,Object> inputMap = new HashMap<String,Object>();
    try {
      IQueryResult rslt = query.execute(inputMap);
      TRUE(rslt.isSuccess());
    } catch (Exception e) {
      NG(e.toString());
    }
    EQUAL(LOG.toString(), "");
  }

  public void executeQuery_inputMap_result_1()
  {
    MSG("処理する実行IDを制御するクエリを処理するメソッドの確認。");
    MSG("- if要素がない場合。");

    final String QID =
      "ControllableQueryTest_executeQuery_inputMap_result_1";

    QueryConfig config = new QueryConfig(QID);
    ControllableQuery query = new ControllableQuery(config);

    Map<String,Object> inputMap = new HashMap<String,Object>();
    try {
      IQueryResult rslt = query.execute(inputMap);
      TRUE(rslt.isSuccess());
    } catch (Exception e) {
      NG(e.toString());
    }
    EQUAL(LOG.toString(),
      "[ControllableQueryTest_e0]" +
      "[ControllableQueryTest_e1]" +
      "[ControllableQueryTest_e2]" +
    "");
  }

  public void executeQuery_inputMap_result_2()
  {
    MSG("処理する実行IDを制御するクエリを処理するメソッドの確認。");
    MSG("- if要素がある場合。");

    final String QID =
      "ControllableQueryTest_executeQuery_inputMap_result_2";

    QueryConfig config = new QueryConfig(QID);
    ControllableQuery query = new ControllableQuery(config);

    Map<String,Object> inputMap = new HashMap<String,Object>();
    try {
      IQueryResult rslt = query.execute(inputMap);
      TRUE(rslt.isSuccess());
    } catch (Exception e) {
      NG(e.toString());
    }
    EQUAL(LOG.toString(),
      "[ControllableQueryTest_e0]" +
      "[ControllableQueryTest_e1]" +
      "[ControllableQueryTest_e4]" +
      "[ControllableQueryTest_e5]" +
      "[ControllableQueryTest_e6]" +
      "[ControllableQueryTest_e9]" +
      "[ControllableQueryTest_e10]" +
      "[ControllableQueryTest_e11]" +
      "[ControllableQueryTest_e12]" +
      "[ControllableQueryTest_e14]" +
      "[ControllableQueryTest_e15]" +
      "[ControllableQueryTest_e16]" +
    "");

    CLEAR_LOG();
    inputMap.put("AAA", "");
    try {
      IQueryResult rslt = query.execute(inputMap);
      TRUE(rslt.isSuccess());
    } catch (Exception e) {
      NG(e.toString());
    }
    EQUAL(LOG.toString(),
      "[ControllableQueryTest_e0]" +
      "[ControllableQueryTest_e1]" +
      "[ControllableQueryTest_e2]" +
      "[ControllableQueryTest_e3]" +
      "[ControllableQueryTest_e4]" +
      "[ControllableQueryTest_e6]" +
      "[ControllableQueryTest_e9]" +
      "[ControllableQueryTest_e10]" +
      "[ControllableQueryTest_e11]" +
      "[ControllableQueryTest_e12]" +
      "[ControllableQueryTest_e14]" +
      "[ControllableQueryTest_e15]" +
      "[ControllableQueryTest_e16]" +
    "");
  }

  public void executeQuery_inputMap_result_3()
  {
    MSG("処理する実行IDを制御するクエリを処理するメソッドの確認。");
    MSG("- if要素のネストがある場合。");

    final String QID =
      "ControllableQueryTest_executeQuery_inputMap_result_3";

    QueryConfig config = new QueryConfig(QID);
    ControllableQuery query = new ControllableQuery(config);

    Map<String,Object> inputMap = new HashMap<String,Object>();
    try {
      IQueryResult rslt = query.execute(inputMap);
      TRUE(rslt.isSuccess());
    } catch (Exception e) {
      NG(e.toString());
    }
    EQUAL(LOG.toString(),
      "[ControllableQueryTest_e0]" +
      "[ControllableQueryTest_e4]" +
    "");
  }
}
