/*
 * IQueryConnection class.
 *
 * Copyright (C) 2012 SATOH Takayuki All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package ts.query;

import ts.util.ReasonedException;

/**
 * クエリを実行する対象への接続を表すコネクション・クラスのインターフェイス。
 *
 * @author 佐藤隆之
 * @version $Id: IQueryConnection.java,v 1.3 2012-01-30 16:41:20 tayu Exp $
 */
public interface IQueryConnection
{
  /**
   * 接続先IDを取得する。
   *
   * @return 接続先ID。
   */
  String getConnectionId();

  /**
   * 接続を開始する。
   *
   * @throws ReasonedException 接続の開始処理に失敗した場合。
   */
  void open() throws ReasonedException;

  /**
   * 接続先への更新を確定する。
   *
   * @throws ReasonedException 接続の確定処理に失敗した場合。
   */
  void commit() throws ReasonedException;

  /**
   * 接続先への更新を取り消す。
   *
   * @throws ReasonedException 接続の取消処理に失敗した場合。
   */
  void rollback() throws ReasonedException;
  
  /**
   * 接続を終了する。
   *
   * @throws ReasonedException 接続の終了処理に失敗した場合。
   */
  void close() throws ReasonedException;

  /**
   * 接続がオープンされているかどうかを確認する。
   *
   * @return 接続がオープンされている場合は<tt>true</tt>を返す。
   */
  boolean isOpened();

  /**
   * 接続がクローズされているかどうかを確認する。
   *
   * @return 接続がクローズされている場合は<tt>true</tt>を返す。
   */
  boolean isClosed();

  /**
   * トランザクションのタイムアウト時刻を取得する。
   * <br>
   * 値がゼロ以下の場合はタイムアウト制限がないものとする。
   *
   * @return トランザクションのタイムアウト時刻 [msec]。
   */
  long getLimitTimeMillis();

  /**
   * このコネクションを使って実行されたクエリの実行履歴オブジェクトを取得する。
   *
   * @return {@link IQueryHistory}オブジェクト。
   */
  IQueryHistory getQueryHistory();
}
