package ts.query;

import ts.tester.UnitTest;
import ts.tester.function.ObjectInspector;
import ts.util.*;
import ts.util.file.*;
import ts.util.resource.*;
import java.io.*;
import java.nio.channels.*;
import java.util.*;

public class QueryConfigTest extends UnitTest
{
  public static void main(String[] args)
  {
    run(QueryConfigTest.class, args);
  }

  File DATA_DIR = new File("test/data/");
  File CFG_DIR = new File(DATA_DIR, "ts/query/configs/query/");
  File SAMPLE_DIR = new File(DATA_DIR, "ts/query/QueryConfig/");

  static class MyQuery implements IQuery {
    public MyQuery(QueryConfig cfg) {}
    public MyQuery(QueryConfig cfg, IQueryTransaction tran) {}
    @Override
    public IQueryResult execute(Map<String,Object> inputMap)
    throws ReasonedException, ReasonedRuntimeException { return null; }
    @Override
    public IQueryHistory getQueryHistory() { return null; }
  }

  static class MyQuery2 implements IQuery {
    @Override
    public IQueryResult execute(Map<String,Object> inputMap)
    throws ReasonedException, ReasonedRuntimeException { return null; }
    @Override
    public IQueryHistory getQueryHistory() { return null; }
  }

  static class MyQuery3 implements IQuery {
    public MyQuery3(QueryConfig cfg) { throw new RuntimeException("ERR"); }
    public MyQuery3(QueryConfig cfg, IQueryTransaction tran)
    { throw new RuntimeException("ERR"); }
    @Override
    public IQueryResult execute(Map<String,Object> inputMap)
    throws ReasonedException, ReasonedRuntimeException { return null; }
    @Override
    public IQueryHistory getQueryHistory() { return null; }
  }

  public void constructor()
  {
    MSG("デフォルト・コンストラクタの確認。");

    QueryConfig config = new QueryConfig();
    EQUAL(config.getQueryId(), "");
    NOTNULL(config.getResource());
    NULL(AbstractResource.class.cast(config.getResource()).getPath());
    NOTNULL(config.typedGetter());
    EQUAL(config.getQueryClass(), "");
    EQUAL(config.getExecutionIds().size(), 0);
  }

  public void constructor_queryId()
  {
    MSG("クエリIDを引数にとるコンストラクタ。");

    String QID = "QueryConfigTest_constructor_queryId";
    QueryConfig config = new QueryConfig(QID);
    EQUAL(config.getQueryId(), QID);
    NOTNULL(config.getResource());
    EQUAL(new File(AbstractResource.class.cast(config.getResource()).getPath()).getAbsolutePath(), new File(CFG_DIR, QID + ".xml").getAbsolutePath());
    NOTNULL(config.typedGetter());
    EQUAL(config.getQueryClass(), "ts.query.QueryConfigTest$MyQuery0");
    EQUAL(config.getExecutionIds(), Arrays.asList("exec00","exec01","exec02"));
  }

  public void constructor_queryId_NullOrEmpty()
  {
    MSG("引数がヌルや空文字列の場合。");

    try {
      new QueryConfig(null);
      NG();
    }
    catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), QueryConfig.Error.QueryIdIsNullOrEmpty);
      EQUAL(e.getMessage(), "[query Id=null]");
    }

    try {
      new QueryConfig("");
      NG();
    }
    catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(), QueryConfig.Error.QueryIdIsNullOrEmpty);
      EQUAL(e.getMessage(), "[query Id=]");
    }
  }

  public void loadResource_LoadXmlFile()
  {
    MSG("XML形式の設定ファイルのロードの確認。");

    String QID = "QueryConfigTest_loadResource_LoadXmlFile";
    QueryConfig config = new QueryConfig(QID);
    EQUAL(config.getQueryId(), QID);
    NOTNULL(config.getResource());
    EQUAL(new File(AbstractResource.class.cast(config.getResource()).getPath()).getAbsolutePath(), new File(CFG_DIR, QID + ".xml").getAbsolutePath());
    NOTNULL(config.typedGetter());
    EQUAL(config.getQueryClass(), "ts.query.QueryConfigTest$MyQuery3");
    EQUAL(config.getExecutionIds(), Arrays.asList("exec30"));

    EQUAL(config.getResource().getFirstValue("ts-query.query.class"),
      "ts.query.QueryConfigTest$MyQuery3");
    EQUAL(config.getResource().getValues("ts-query.query.execution.id"),
      Arrays.asList("exec30"));

    EQUAL(config.typedGetter().getString("ts-query.query.class"),
      "ts.query.QueryConfigTest$MyQuery3");
    EQUAL(config.typedGetter().getList("ts-query.query.execution.id"),
      Arrays.asList("exec30"));
  }

  public void loadResource_LoadPropertyFile()
  {
    MSG("Javaプロパティ形式の設定ファイルのロードの確認。");

    String QID = "QueryConfigTest_loadResource_LoadPropertyFile";
    QueryConfig config = new QueryConfig(QID);
    EQUAL(config.getQueryId(), QID);
    NOTNULL(config.getResource());
    EQUAL(new File(AbstractResource.class.cast(config.getResource()).getPath()).getAbsolutePath(), new File(CFG_DIR, QID + ".properties").getAbsolutePath());
    NOTNULL(config.typedGetter());
    EQUAL(config.getQueryClass(), "ts.query.QueryConfigTest$MyQuery2");
    EQUAL(config.getExecutionIds(), Arrays.asList("exec20","exec21","exec21",
      "exec21","exec22"));

    EQUAL(config.getResource().getFirstValue("ts-query.query.class"),
      "ts.query.QueryConfigTest$MyQuery2");
    EQUAL(config.getResource().getValues("ts-query.query.execution.id"),
      Arrays.asList("exec20","exec21","exec21","exec21","exec22"));

    EQUAL(config.typedGetter().getString("ts-query.query.class"),
      "ts.query.QueryConfigTest$MyQuery2");
    EQUAL(config.typedGetter().getList("ts-query.query.execution.id"),
      Arrays.asList("exec20","exec21","exec21","exec21","exec22"));
  }

  public void loadResource_FailToLoadConfig_Xml()
  {
    MSG("XML形式の設定ファイルのロードに失敗した場合。");

    try {
      new QueryConfig("QueryConfigTest_loadResource_FailToLoadConfig_Xml");
      NG();
    }
    catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(), QueryConfig.Error.FailToLoadQueryConfigFile);
    }
  }

  public void loadResource_FailToLoadConfig_Properties()
  {
    MSG("Javaプロパティ形式の設定ファイルのロードに失敗した場合。");

    String QID = "QueryConfigTest_loadResource_FailToLoadConfig_Properties";
    File file = new File(CFG_DIR, QID + ".properties");

    RandomAccessFile raf = null;
    FileLock lock = null;

    try {
      raf = new RandomAccessFile(file, "rw");
      FileChannel channel = raf.getChannel();
      lock = channel.lock(0, Long.MAX_VALUE, false);
    }
    catch (Exception e) {
      NG(e);
    }

    try {
      new QueryConfig(QID);
      NG();
    }
    catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(), QueryConfig.Error.FailToLoadQueryConfigFile);
    }
    catch (Throwable e) {
      NG(e);
    }
    finally {
      try {
        if (lock != null) try { lock.release(); } catch (Exception e) {}
        if (raf  != null) try { raf.close(); } catch (Exception e) {}
      }
      catch (Exception e) {
        NG(e);
      }
    }
  }

  public void loadResource_ConfigFileNotFound()
  {
    MSG("設定ファイルがXML形式でもJavaプロパティ形式でも存在しなかった場合。");

    try {
      new QueryConfig("xx");
      NG();
    }
    catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(), QueryConfig.Error.QueryConfigFileNotFound);
    }
  }

  public void getQueryClass()
  {
    MSG("クエリ・クラス名の設定と取得の確認。");

    QueryConfig config;

    config = new QueryConfig();
    EQUAL(config.getQueryClass(), "");

    config.getResource().setFirstValue("ts-query.query.class", "X");
    EQUAL(config.getQueryClass(), "X");

    config.getResource().setFirstValue("ts-query.query.class", "Y");
    EQUAL(config.getQueryClass(), "Y");

    config.getResource().setFirstValue("ts-query.query.class", "Z");
    EQUAL(config.getQueryClass(), "Z");

    config.getResource().removeChildren("ts-query.query.class");
    EQUAL(config.getQueryClass(), "");
  }

  public void getExecutionIds()
  {
    MSG("実行IDのリストの設定と取得の確認。");

    QueryConfig config;

    config = new QueryConfig();
    EQUAL(config.getExecutionIds().size(), 0);

    config.getResource().setFirstValue("ts-query.query.execution.id", "e0");
    EQUAL(config.getExecutionIds(), Arrays.asList("e0"));

    config.getResource().setFirstValue("ts-query.query.execution.id", "e1");
    EQUAL(config.getExecutionIds(), Arrays.asList("e1"));

    config.getResource().setFirstValue("ts-query.query.execution.id(n=2)","e2");
    EQUAL(config.getExecutionIds(), Arrays.asList("e1", "e2"));

    config.getResource().removeChildren("ts-query.query.execution.id(n=2)");
    EQUAL(config.getExecutionIds().size(), 0);
  }

  public void create()
  {
    MSG("IQueryオブジェクトを作成するメソッドの確認。");

    QueryConfig config = new QueryConfig();
    config.getResource().setFirstValue("ts-query.query.class",
      "ts.query.QueryConfigTest$MyQuery");

    try {
      IQuery query = config.create();
      NOTNULL(query);
    }
    catch (Exception e) {
      NG(e.toString());
    }

    try {
      MyQuery query = config.create();
      NOTNULL(query);
    }
    catch (Exception e) {
      NG(e.toString());
    }

    QueryConfig config1 = new QueryConfig("QueryConfigTest_create");
    try {
      MyQuery query = config1.create();
      NOTNULL(query);
    }
    catch (Exception e) {
      NG(e.toString());
    }

    try {
      IQuery query = config1.create();
      NOTNULL(query);
    }
    catch (Exception e) {
      NG(e.toString());
    }
  }

  public void create_ClassNameIsEmpty()
  {
    MSG("クラス名が空の場合。");

    QueryConfig config = new QueryConfig();

    try {
      config.create();
      NG();
    }
    catch (ReasonedException e) {
      NG(e);
    }
    catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(), QueryConfig.Error.QueryClassNotFound);
    }
  }

  public void create_ClassNotFound()
  {
    MSG("設定された名前のクラスが存在しない場合。");

    QueryConfig config = new QueryConfig();
    config.getResource().setFirstValue("ts-query.query.class",
      "ts.query.QueryConfigTest$MyQueryXXX");

    try {
      config.create();
      NG();
    }
    catch (ReasonedException e) {
      NG(e);
    }
    catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getCause().getClass(), ClassNotFoundException.class);
      EQUAL(e.getReason(), QueryConfig.Error.QueryClassNotFound);
    }
  }

  public void create_ConstructorNotFound()
  {
    MSG("指定されたクラスに要求されるコンストラクタが存在しない場合。");

    QueryConfig config = new QueryConfig();
    config.getResource().setFirstValue("ts-query.query.class",
      "ts.query.QueryConfigTest$MyQuery2");

    try {
      config.create();
      NG();
    }
    catch (ReasonedException e) {
      NG(e);
    }
    catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getCause().getClass(), NoSuchMethodException.class);
      EQUAL(e.getReason(), QueryConfig.Error.QueryConstructorNotFound);
    }
  }

  public void create_FailToCreate()
  {
    MSG("IQueryオブジェクトの作成に失敗した場合。");

    QueryConfig config = new QueryConfig();
    config.getResource().setFirstValue("ts-query.query.class",
      "ts.query.QueryConfigTest$MyQuery3");

    try {
      config.create();
      NG();
    }
    catch (ReasonedException e) {
      OK(e.toString());
      EQUAL(e.getReason(), QueryConfig.Error.FailToCreateQuery);
    }
    catch (Exception e) {
      NG(e.toString());
    }
  }

  public void create_ClassNameIsDifferent()
  {
    MSG("設定されたクラスと戻り値のクラスが異なる場合。");

    QueryConfig config = new QueryConfig();
    config.getResource().setFirstValue("ts-query.query.class",
      "ts.query.QueryConfigTest$MyQuery");

    try {
      MyQuery2 query = config.create();
      NG();
    }
    catch (ClassCastException e) {
      OK(e.toString());
    }
    catch (Exception e) {
      NG(e.toString());
    }
  }

  public void create_tran()
  {
    MSG("トランザクションを引数にとるIQueryオブジェクトを作成するメソッドの確認。");

    IQueryTransaction tran = new QueryTransaction();

    QueryConfig config = new QueryConfig();
    config.getResource().setFirstValue("ts-query.query.class",
      "ts.query.QueryConfigTest$MyQuery");

    try {
      IQuery query = config.create(tran);
      NOTNULL(query);
    }
    catch (Exception e) {
      NG(e.toString());
    }

    try {
      MyQuery query = config.create(tran);
      NOTNULL(query);
    }
    catch (Exception e) {
      NG(e.toString());
    }

    QueryConfig config1 = new QueryConfig("QueryConfigTest_create_tran");
    try {
      MyQuery query = config1.create(tran);
      NOTNULL(query);
    }
    catch (Exception e) {
      NG(e.toString());
    }

    try {
      IQuery query = config1.create(tran);
      NOTNULL(query);
    }
    catch (Exception e) {
      NG(e.toString());
    }
  }

  public void create_tran_ClassNameIsEmpty()
  {
    MSG("クラス名が空の場合。");

    IQueryTransaction tran = new QueryTransaction();

    QueryConfig config = new QueryConfig();

    try {
      config.create(tran);
      NG();
    }
    catch (ReasonedException e) {
      NG(e);
    }
    catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getReason(), QueryConfig.Error.QueryClassNotFound);
    }
  }

  public void create_tran_ClassNotFound()
  {
    MSG("設定された名前のクラスが存在しない場合。");

    IQueryTransaction tran = new QueryTransaction();

    QueryConfig config = new QueryConfig();
    config.getResource().setFirstValue("ts-query.query.class",
      "ts.query.QueryConfigTest$MyQueryXXX");

    try {
      config.create(tran);
      NG();
    }
    catch (ReasonedException e) {
      NG(e);
    }
    catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getCause().getClass(), ClassNotFoundException.class);
      EQUAL(e.getReason(), QueryConfig.Error.QueryClassNotFound);
    }
  }

  public void create_tran_ConstructorNotFound()
  {
    MSG("指定されたクラスに要求されるコンストラクタが存在しない場合。");

    IQueryTransaction tran = new QueryTransaction();

    QueryConfig config = new QueryConfig();
    config.getResource().setFirstValue("ts-query.query.class",
      "ts.query.QueryConfigTest$MyQuery2");

    try {
      config.create(tran);
      NG();
    }
    catch (ReasonedException e) {
      NG(e);
    }
    catch (ReasonedRuntimeException e) {
      OK(e.toString());
      EQUAL(e.getCause().getClass(), NoSuchMethodException.class);
      EQUAL(e.getReason(), QueryConfig.Error.QueryConstructorNotFound);
    }
  }

  public void create_tran_FailToCreate()
  {
    MSG("IQueryオブジェクトの作成に失敗した場合。");

    IQueryTransaction tran = new QueryTransaction();

    QueryConfig config = new QueryConfig();
    config.getResource().setFirstValue("ts-query.query.class",
      "ts.query.QueryConfigTest$MyQuery3");

    try {
      config.create(tran);
      NG();
    }
    catch (ReasonedException e) {
      OK(e.toString());
      EQUAL(e.getReason(), QueryConfig.Error.FailToCreateQuery);
    }
    catch (Exception e) {
      NG(e.toString());
    }
  }

  public void create_tran_ClassNameIsDifferent()
  {
    MSG("設定されたクラスと戻り値のクラスが異なる場合。");

    IQueryTransaction tran = new QueryTransaction(); 

    QueryConfig config = new QueryConfig();
    config.getResource().setFirstValue("ts-query.query.class",
      "ts.query.QueryConfigTest$MyQuery");

    try {
      MyQuery2 query = config.create(tran);
      NG();
    }
    catch (ClassCastException e) {
      OK(e.toString());
    }
    catch (Exception e) {
      NG(e.toString());
    }
  }

  public void outputSampleXml_pw()
  {
    MSG("XML形式のクエリ設定ファイルの出力メソッドの確認。");

    String QID = "QueryConfigTest_outputSampleXml_pw";

    File file = new File(SAMPLE_DIR, QID + ".xml");
    try {
      if (file.exists()) FileOperation.delete(file);
    }
    catch (Exception e) {
      NG(e);
    }

    PrintWriter pw = null;
    try {
      pw = new PrintWriter(new OutputStreamWriter(new FileOutputStream(
        file), "Windows-31J"));

      QueryConfig config = new QueryConfig();
      config.outputSampleXml(pw, "Windows-31J");
    }
    catch (Exception e) {
      NG(e);
    }
    finally {
      if (pw != null) try { pw.close(); } catch (Exception e) {}
    }

    try {
      File file1 = new File(CFG_DIR, QID + ".xml");
      if (file1.exists()) FileOperation.delete(file1);
      FileOperation.copy(file, file1);
    }
    catch (Exception e) {
      NG(e);
    }

    try {
      QueryConfig config = new QueryConfig(QID);
      EQUAL(config.getQueryClass(), "...");
      EQUAL(config.getExecutionIds(), Arrays.asList("...", "...", "..."));
    }
    catch (Exception e) {
      NG(e);
    }
  }

  public void outputSampleXml_pw_Null()
  {
    MSG("引数がヌルの場合。");

    PrintWriter pw = null;
    try {
      QueryConfig config = new QueryConfig();
      config.outputSampleXml(null, "Windows-31J");
      NG();
    }
    catch (NullPointerException e) {
      OK(e);
    }
    catch (Exception e) {
      NG(e);
    }
    finally {
      if (pw != null) try { pw.close(); } catch (Exception e) {}
    }

    try {
      pw = new PrintWriter(new StringWriter());
      QueryConfig config = new QueryConfig();
      config.outputSampleXml(pw, null);
      NG();
    }
    catch (NullPointerException e) {
      OK(e);
    }
    catch (Exception e) {
      NG(e);
    }
    finally {
      if (pw != null) try { pw.close(); } catch (Exception e) {}
    }
  }

  public void outputSampleXmlEntries_pw_Inheritance()
  {
    MSG("XMLエントリを出力するメソッドの継承の確認。");

    String QID = "QueryConfigTest_outputSampleXmlEntries_pw_Inheritance";

    File file = new File(SAMPLE_DIR, QID + ".xml");
    try {
      if (file.exists()) FileOperation.delete(file);
    }
    catch (Exception e) {
      NG(e);
    }

    PrintWriter pw = null;
    try {
      pw = new PrintWriter(new OutputStreamWriter(new FileOutputStream(
        file), "UTF-8"));

      QueryConfig config = new QueryConfig() {
        static final long serialVersionUID = -1L;
        @Override
        public void outputSampleXmlEntries(PrintWriter pw) throws IOException {
          pw.println("    <eeee>");
          pw.println("     <ffff>FFFF</ffff>");
          pw.println("    </eeee>");
        }
      };
      config.outputSampleXml(pw, "UTF-8");
    }
    catch (Exception e) {
      NG(e);
    }
    finally {
      if (pw != null) try { pw.close(); } catch (Exception e) {}
    }

    try {
      File file1 = new File(CFG_DIR, QID + ".xml");
      if (file1.exists()) FileOperation.delete(file1);
      FileOperation.copy(file, file1);
    }
    catch (Exception e) {
      NG(e);
    }

    try {
      QueryConfig config = new QueryConfig(QID);
      EQUAL(config.getQueryClass(), "...");
      EQUAL(config.getExecutionIds().size(), 0);
      EQUAL(config.typedGetter().getString("ts-query.query.eeee.ffff"), "FFFF");
    }
    catch (Exception e) {
      NG(e);
    }
  }

  public void outputSampleProp_pw()
  {
    MSG("プロパティ形式のクエリ設定ファイルの出力メソッドの確認。");

    String QID = "QueryConfigTest_outputSampleProp_pw";

    File file = new File(SAMPLE_DIR, QID + ".properties");
    try {
      if (file.exists()) FileOperation.delete(file);
    }
    catch (Exception e) {
      NG(e);
    }

    PrintWriter pw = null;
    try {
      pw = new PrintWriter(new OutputStreamWriter(new FileOutputStream(
        file), "Windows-31J"));

      QueryConfig config = new QueryConfig();
      config.outputSampleProp(pw);
    }
    catch (Exception e) {
      NG(e);
    }
    finally {
      if (pw != null) try { pw.close(); } catch (Exception e) {}
    }

    try {
      File file1 = new File(CFG_DIR, QID + ".properties");
      if (file1.exists()) FileOperation.delete(file1);
      FileOperation.copy(file, file1);
    }
    catch (Exception e) {
      NG(e);
    }

    try {
      QueryConfig config = new QueryConfig(QID);
      EQUAL(config.getQueryClass(), "...");
      EQUAL(config.getExecutionIds(), Arrays.asList("...", "...", "..."));
    }
    catch (Exception e) {
      NG(e);
    }
  }

  public void outputSampleProp_pw_Null()
  {
    MSG("引数がヌルの場合。");

    try {
      QueryConfig config = new QueryConfig();
      config.outputSampleProp(null);
      NG();
    }
    catch (NullPointerException e) {
      OK(e);
    }
    catch (Exception e) {
      NG(e);
    }
  }

  public void outputSamplePropEntries_pw_Inheritance()
  {
    MSG("プロパティ・エントリを出力するメソッドの継承の確認。");

    String QID = "QueryConfigTest_outputSamplePropEntries_pw_Inheritance";

    File file = new File(SAMPLE_DIR, QID + ".properties");
    try {
      if (file.exists()) FileOperation.delete(file);
    }
    catch (Exception e) {
      NG(e);
    }

    PrintWriter pw = null;
    try {
      pw = new PrintWriter(new OutputStreamWriter(new FileOutputStream(
        file), "UTF-8"));

      QueryConfig config = new QueryConfig() {
        static final long serialVersionUID = -1L;
        @Override
        public void outputSamplePropEntries(PrintWriter pw) throws IOException {
          pw.println("ts-query.query.eeee.ffff = FFFF");
        }
      };
      config.outputSampleProp(pw);
    }
    catch (Exception e) {
      NG(e);
    }
    finally {
      if (pw != null) try { pw.close(); } catch (Exception e) {}
    }

    try {
      File file1 = new File(CFG_DIR, QID + ".properties");
      if (file1.exists()) FileOperation.delete(file1);
      FileOperation.copy(file, file1);
    }
    catch (Exception e) {
      NG(e);
    }

    try {
      QueryConfig config = new QueryConfig(QID);
      EQUAL(config.getQueryClass(), "...");
      EQUAL(config.getExecutionIds().size(), 0);
      EQUAL(config.typedGetter().getString("ts-query.query.eeee.ffff"), "FFFF");
    }
    catch (Exception e) {
      NG(e);
    }
  }

  public void executeCommand_args_config_FileTypeIsXml()
  {
    MSG("クエリ設定ファイルのサンプルを出力するメソッドの確認(XML)。");

    QueryConfig cfg = new QueryConfig();

    String QID = "QueryConfitTest_executeCommand_args_config_FileTypeIsXml";

    EQUAL(QueryConfig.executeCommand(new String[]{ "pgm", 
      new File(SAMPLE_DIR, QID + ".xml").getPath(), "XmL", "UTF-8" }, cfg), 0);
  }

  public void executeCommand_args_config_FileTypeIsProp()
  {
    MSG("クエリ設定ファイルのサンプルを出力するメソッドの確認(Prop)。");

    QueryConfig cfg = new QueryConfig();

    String QID = "QueryConfitTest_executeCommand_args_config_FileTypeIsProp";

    EQUAL(QueryConfig.executeCommand(new String[]{ "pgm", 
      new File(SAMPLE_DIR, QID + ".properties").getPath(), "ProP", "UTF-8"},
      cfg), 0);
  }

  public void executeCommand_args_config_Null()
  {
    MSG("引数がヌルの場合。");

    QueryConfig cfg = new QueryConfig();
    EQUAL(QueryConfig.executeCommand(null, cfg), 1);
  }

  public void executeCommand_args_config_ZeroArgs()
  {
    MSG("引数の配列の要素数がゼロの場合。");

    QueryConfig cfg = new QueryConfig();
    EQUAL(QueryConfig.executeCommand(new String[0], cfg), 1);
  }

  public void executeCommand_args_config_IllegalArgsCount()
  {
    MSG("引数の配列の要素数が不正な場合。");

    QueryConfig cfg = new QueryConfig();

    EQUAL(QueryConfig.executeCommand(new String[]{"p"}, cfg), 1);
    EQUAL(QueryConfig.executeCommand(new String[]{"p","q"}, cfg), 1);
    EQUAL(QueryConfig.executeCommand(new String[]{"p","q","xml"}, cfg), 1);
    EQUAL(QueryConfig.executeCommand(new String[]{"p","q","xml","UTF8","t"},
      cfg), 1);
    EQUAL(QueryConfig.executeCommand(new String[]{"p","q","xml","UTF8","t","u"},
      cfg), 1);
  }

  public void executeCommand_args_config_IllegalOutputFileType()
  {
    MSG("出力ファイルの形式が不正な場合。");

    QueryConfig cfg = new QueryConfig();

    EQUAL(QueryConfig.executeCommand(new String[]{ "pgm", 
      new File(SAMPLE_DIR, "xxx.xml").getPath(), "XXX", "UTF-8"}, cfg), 1);
  }

  public void executeCommand_args_config_IllegalEncoding()
  {
    MSG("出力ファイルの文字エンコーディングが不正な場合。");

    QueryConfig cfg = new QueryConfig();

    EQUAL(QueryConfig.executeCommand(new String[]{ "pgm",
      new File(SAMPLE_DIR, "xxx.xml").getPath(), "XML", "UUUU"}, cfg), 2);
  }

  public void executeCommand_args_config_OutputFilePathIsBad()
  {
    MSG("出力ファイルのパスが不正な場合。");
    MSG("- 指定したパスが既存のディレクトリだった場合。");

    QueryConfig cfg = new QueryConfig();

    EQUAL(QueryConfig.executeCommand(new String[]{ "pgm", SAMPLE_DIR.getPath(),
      "XML", "UTF-8"}, cfg), 3);
  }
}
