/*
 * QueryConnectionConfig class.
 *
 * Copyright (C) 2012 SATOH Takayuki All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package ts.query;

import ts.util.TypedGetter;
import ts.util.AbstractTypedGetter;
import ts.util.resource.Resource;
import ts.util.resource.PropertyResource;
import ts.util.resource.XmlResource;
import ts.util.ReasonedException;
import ts.util.ReasonedRuntimeException;
import java.io.File;
import java.io.FileOutputStream;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.io.Serializable;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.lang.reflect.Constructor;
import java.util.List;

/**
 * クエリ接続設定を保持するクラス。
 * <br>
 * 接続先に対応する{@link IQueryConnection}オブジェクトを作成するために必要な
 * 設定情報を格納する。
 * <br>
 * 引数なしのコンストラクタを使った場合は、空の設定オブジェクトが作成され、
 * {@link #getResource()}メソッドにより得られるリソース・オブジェクトに設定情報
 * を登録して使用する。
 * <br>
 * 引数に接続先IDを指定するコンストラクタを使った場合は、自動的に{@link
 * QueryEnvironmentConfig}オブジェクトから取得したディレクトリから接続先IDを
 * ファイル・タイトルにもつXMLファイル(.xml)又はJavaプロパティ・ファイル
 * (.properties)をロードするので、その設定情報を使用する。
 * <br>
 * 設定情報は各接続先によって異なるが、それに依らない共通の設定情報として、
 * <ul>
 *  <li>ts-query.connection.class - 使用される{@link IQueryConnection}の派生
 *   クラス名（必須）。</li>
 *  <li>ts-query.connection.limit.spenttime - 接続開始からの制限時間 [msec]</li>
 * </ul>
 * がある。
 *
 * @author 佐藤隆之
 * @version $Id: QueryConnectionConfig.java,v 1.19 2012-03-14 07:49:20 tayu Exp $
 */
public class QueryConnectionConfig implements Serializable
{
  /** このクラスで発生しうるエラーを定義する列挙型。 */
  public enum Error {
    /** 接続先IDがヌル又は空文字列の場合。 */
    ConnectionIdIsNullOrEmpty,

    /** 接続設定ファイルが見つからない場合。 */
    ConnectionConfigFileNotFound,

    /** 接続設定ファイルのロードに失敗した場合。 */
    FailToLoadConnectionConfigFile,

    /** 設定情報に指定されたクラスが見つからない場合。 */
    ConnectionClassNotFound,

    /** 設定情報のクラスに要求されるコンストラクタが存在しない場合。 */
    ConnectionConstructorNotFound,

    /** コネクション・オブジェクトの作成に失敗した場合。 */
    FailToCreateConnection,

    /** 設定情報の処理時間の制限値が不正な場合。 */
    IllegalLimitSpentTime,
  }

  /** シリアル・バージョン番号。 */
  static final long serialVersionUID = -6887474425180424994L;

  /*y 接続yID。 */
  private final String connectionId;

  /** 設定情報を保持するリソース・オブジェクト。 */
  private final Resource resource;

  /** 設定情報を型変換して取得するための{@link TypedGetter}オブジェクト。 */
  private final AbstractTypedGetter<String,String> typedGetter;

  /**
   * デフォルト・コンストラクタ。
   */
  public QueryConnectionConfig()
  {
    this.connectionId = "";
    this.resource = new PropertyResource();
    this.typedGetter = newTypedGetter();
  }

  /**
   * 接続先IDを引数にとるコンストラクタ。
   *
   * @param connId 接続先ID。
   * @throws AssertionError 引数がヌルの場合（デバッグ・モードのみ）。
   */
  public QueryConnectionConfig(String connId)
  {
    if (connId == null) {
      throw new ReasonedRuntimeException(Error.ConnectionIdIsNullOrEmpty,
        "[connection Id=" + connId + "]");
    }

    connId = connId.trim();
    if (connId.isEmpty()) {
      throw new ReasonedRuntimeException(Error.ConnectionIdIsNullOrEmpty,
        "[connection Id=" + connId + "]");
    }

    this.connectionId = connId;
    this.resource = loadResource();
    this.typedGetter = newTypedGetter();
  }

  /**
   * 接続先IDを取得する。
   *
   * @return 接続先ID。
   */
  public String getConnectionId()
  {
    return this.connectionId;
  }

  /**
   * 接続設定ファイルをロードしたリソース・オブジェクトを作成する。
   *
   * @return 接続設定ファイルをロードしたリソース・オブジェクト。
   * @throws ReasonedRuntimeException 接続設定ファイルのロードに失敗した場合。
   */
  protected Resource loadResource() throws ReasonedRuntimeException
  {
    QueryEnvironment env = QueryEnvironment.getInstance();
    File dir = env.getConnectionConfigDirectory();

    File xmlFile = new File(dir, getConnectionId() + ".xml");
    if (xmlFile.exists()) {
      try {
        XmlResource res = new XmlResource();
        res.setValidating(false);
        res.load(xmlFile.getCanonicalPath());
        return res;
      }
      catch (Exception e) {
        throw new ReasonedRuntimeException(Error.FailToLoadConnectionConfigFile,
          "[path=" + xmlFile.getAbsolutePath() + "]", e);
      }
    }

    File propFile = new File(dir, getConnectionId() + ".properties");
    if (propFile.exists()) {
      try {
        return new PropertyResource(propFile.getCanonicalPath());
      }
      catch (Exception e) {
        throw new ReasonedRuntimeException(Error.FailToLoadConnectionConfigFile,
          "[path=" + propFile.getAbsolutePath() + "]", e);
      }
    }

    throw new ReasonedRuntimeException(Error.ConnectionConfigFileNotFound,
      "[path=" + new File(dir, getConnectionId()).getAbsolutePath() + 
      ".{xml|properties}]");
  }

  /**
   * 設定情報を型変換して取得するための{@link TypedGetter}オブジェクトを作成
   * する。
   *
   * @return {@link TypedGetter}オブジェクト。
   */
  protected AbstractTypedGetter<String,String> newTypedGetter()
  {
    return new AbstractTypedGetter<String,String>() {
      static final long serialVersionUID =
        QueryConnectionConfig.serialVersionUID + 1L;
      @Override
      public String get(String key) {
        return QueryConnectionConfig.this.resource.getFirstValue(key);
      }
      @Override
      public List<String> getList(String key) {
        return QueryConnectionConfig.this.resource.getValues(key);
      }
    };
  }

  /**
   * 接続設定を保持するリソース・オブジェクトを取得する。
   *
   * @return 接続設定を保持するリソース・オブジェクト。
   */
  protected Resource getResource()
  {
    return this.resource;
  }

  /**
   * 設定情報を型変換して取得するための{@link TypedGetter}オブジェクトを取得
   * する。
   *
   * @return {@link TypedGetter}オブジェクト。
   */
  protected AbstractTypedGetter<String,String> typedGetter()
  {
    return this.typedGetter;
  }

  /**
   * {@link IQueryConnection}オブジェクトを作成する。
   * <br>
   * このオブジェクトが保持する接続設定を使って、{@link IQueryConnection}の派生
   * クラスのインスタンスを作成する。
   * 派生クラスの名前は、接続設定 <tt>ts-query.connection.class</tt>に指定された
   * クラス名が使用される。
   *
   * @return {@link IQueryConnection}オブジェクト。
   * @throws ReasonedException {@link IQueryConnection}オブジェクトの作成に失敗
   *   した場合。
   * @throws ReasonedRuntimeException 接続設定が不正だった場合。
   */
  public <T extends IQueryConnection> T create()
    throws ReasonedException, ReasonedRuntimeException
  {
    String cls = getConnectionClass();
    Class<T> connCls = null;
    try {
      @SuppressWarnings("unchecked")
      Class<T> c = (Class<T>) Class.forName(cls);
      connCls = c;
    }
    catch (Exception e) {
      throw new ReasonedRuntimeException(Error.ConnectionClassNotFound,
        "[class=" + cls + "]", e);
    }

    Constructor<T> cons = null;
    try {
      cons = connCls.getConstructor(QueryConnectionConfig.class);
    }
    catch (Exception e) {
      throw new ReasonedRuntimeException(Error.ConnectionConstructorNotFound,
        "[class=" + cls + "]", e);
    }

    try {
      return cons.newInstance(this);
    }
    catch (Exception e) {
      throw new ReasonedException(Error.FailToCreateConnection,
        "[class=" + cls + "]", e);
    }
  }

  /**
   * {@link IQueryConnection}オブジェクトを作成する。
   * <br>
   * このオブジェクトが保持する接続設定と引数のトランザクションを使って、{@link
   * IQueryConnection}の派生クラスのインスタンスを作成する。
   * 派生クラスの名前は、接続設定 <tt>ts-query.connection.class</tt>に指定された
   * クラス名が使用される。
   * <br>
   * 引数のトランザクション・オブジェクトからは
   *
   * @param tran トランザクション・オブジェクト。
   * @return {@link IQueryConnection}オブジェクト。
   * @throws ReasonedException {@link IQueryConnection}オブジェクトの作成に失敗
   *   した場合。
   * @throws ReasonedRuntimeException 接続設定が不正だった場合。
   */
  public <T extends IQueryConnection> T create(IQueryTransaction tran)
    throws ReasonedException, ReasonedRuntimeException
  {
    assert (tran != null) : "@param:tran is null.";

    String cls = getConnectionClass();
    Class<T> connCls = null;
    try {
      @SuppressWarnings("unchecked")
      Class<T> c = (Class<T>) Class.forName(cls);
      connCls = c;
    }
    catch (Exception e) {
      throw new ReasonedRuntimeException(Error.ConnectionClassNotFound,
        "[class=" + cls + "]", e);
    }

    Constructor<T> cons = null;
    try {
      cons = connCls.getConstructor(QueryConnectionConfig.class,
        IQueryTransaction.class);
    }
    catch (Exception e) {
      throw new ReasonedRuntimeException(Error.ConnectionConstructorNotFound,
        "[class=" + cls + "]", e);
    }

    try {
      return cons.newInstance(this, tran);
    }
    catch (Exception e) {
      throw new ReasonedException(Error.FailToCreateConnection,
        "[class=" + cls + "]", e);
    }
  }

  /**
   * このオブジェクトから作成される{@link IQueryConnection}オブジェクトのクラス
   * 名を取得する。
   *
   * @return {@link IQueryConnection}クラスの名前。
   */
  protected String getConnectionClass()
  {
    return typedGetter().getString("ts-query.connection.class");
  }

  /**
   * このオブジェクトから作成される{@link IQueryConnection}オブジェクトの接続
   * 開始からの制限時間を取得する。
   * <br>
   * 取得された制限時間がゼロ以下の場合は、制限がないものとする。
   * <br>
   * このオブジェクトに制限時間の設定がない場合は、制限なしとして<tt>0</tt>を
   * 返す。
   *
   * @return コネクションの接続開始からの制限時間 [msec]。
   */
  protected long getLimitSpentTime()
  {
    final String KEY = "ts-query.connection.limit.spenttime";

    try {
      return typedGetter().getLong(KEY);
    }
    catch (Exception e) {
      throw new ReasonedRuntimeException(Error.IllegalLimitSpentTime,
        "[property value=" + getResource().getFirstValue(KEY) + "]", e);
    }
  }

  /**
   * 接続設定ファイルのXML形式のサンプルを{@link PrintWriter}オブジェクトに
   * 出力する。
   *
   * @param pw {@link PrintWriter}オブジェクト。
   * @throws IOException 出力に失敗した場合。
   */
  protected void outputSampleXml(PrintWriter pw, String encoding)
    throws IOException
  {
    pw.println("<?xml version=\"1.0\" encoding=\"" + encoding.toString() +
               "\"?>");
    pw.println("");
    pw.println("<!--");
    pw.println("  || Query Connection Configuration XML File.");
    pw.println("  ||");
    pw.println("  || This file is an XML file which specifys " +
               "some configuration for an ");
    pw.println("  || IQueryConnection object.");
    pw.println("  ||");
    pw.println("  -->");
    pw.println("");
    pw.println("<ts-query>");
    pw.println(" <connection>");
    pw.println("  <!-- The query connection class -->");
    pw.println("  <class>...</class>");
    pw.println("  <limit>");
    pw.println("    <!-- Limit spent time [msec] -->");
    pw.println("    <spenttime>0</spenttime>");
    pw.println("  </limit>");
    pw.println("");

    outputSampleXmlEntries(pw);

    pw.println(" </connection>");
    pw.println("</ts-query>");
    pw.println();
  }

  /**
   * 接続設定ファイルのXMLエントリのサンプルを{@link PrintWriter}オブジェクトに
   * 出力する。
   *
   * @param pw {@link PrintWriter}オブジェクト。
   * @throws IOException 出力に失敗した場合。
   */
  protected void outputSampleXmlEntries(PrintWriter pw) throws IOException
  {}

  /**
   * 接続設定ファイルのJavaプロパティ形式のサンプルを{@link PrintWriter}
   * オブジェクトに出力する。
   *
   * @param pw {@link PrintWriter}オブジェクト。
   * @throws IOException 出力に失敗した場合。
   */
  protected void outputSampleProp(PrintWriter pw) throws IOException
  {
    pw.println("#");
    pw.println("# Query Connection Property File.");
    pw.println("#");
    pw.println("# This file is a Java property file which specifys " +
               "some configurations for an ");
    pw.println("# IQueryConnection object.");
    pw.println("#");
    pw.println("");
    pw.println("# The query connection class.");
    pw.println("ts-query.connection.class = ...");
    pw.println("");
    pw.println("# Limit spent time [msec]");
    pw.println("ts-query.execution.limit.spenttime = 0");
    pw.println("");

    outputSamplePropEntries(pw);

    pw.println("");
    pw.println("#.");
  }

  /***
   * 接続設定ファイルのプロパティ・エントリのサンプルを{@link PrintWriter}
   * オブジェクトに出力する。
   *
   * @param pw {@link PrintWriter}オブジェクト。
   * @throws IOException 出力に失敗した場合。
   */
  protected void outputSamplePropEntries(PrintWriter pw) throws IOException
  {}

  /**
   * 接続設定ファイルのサンプルを出力するコマンドを実行する。
   * <br>
   * コマンドライン引数の配列の構成は以下の通りである：
   * <ol>
   *  <li>args[0] - コマンド名</li>
   *  <li>args[1] - 出力ファイル・パス</li>
   *  <li>args[2] - 出力ファイル形式 (<tt>"xml"</tt>:XML形式、<tt>"prop"</tt>
   *   :Javaプロパティ形式、但し大文字小文字は区別しない)</li>
   *  <li>args[3] - 出力ファイルの文字エンコーディング</li>
   * </ol>
   * また、終了コードは以下の値をとりうる：
   * <ul>
   * <li><tt>0</tt> - 正常終了の場合</li>
   * <li><tt>1</tt> - コマンドライン引数が不正な場合</li>
   * <li><tt>2</tt> - 非対応の文字エンコーディングが指定された場合</li>
   * <li><tt>3</tt> - 上記以外のエラーの場合</li>
   * </ul>
   *
   * @param args コマンドライン引数の配列。
   * @param config 接続設定オブジェクト。
   * @return コマンドの終了コード。
   */
  protected static int executeCommand(
    String[] args, QueryConnectionConfig config)
  {
    String pgmName = "null", encoding = "";
    File file;
    boolean isXml;
    try {
      if (args != null && args.length > 0) {
        pgmName = args[0];
      }

      file = new File(args[1]);

      if ("xml".equalsIgnoreCase(args[2])) {
        isXml = true;
      }
      else if ("prop".equalsIgnoreCase(args[2])) {
        isXml = false;
      }
      else {
        throw new IllegalArgumentException();
      }

      encoding = args[3];

      if (args.length > 4) {
        throw new IllegalArgumentException();
      }
    }
    catch (Exception e) {
      System.err.println("HELP:");
      System.err.println(
        pgmName + " <output-file-path> {xml|prop} <output-file-encoding>");
      System.err.println();
      return 1;
    }

    try {
      new String(new byte[0], encoding);
    }
    catch (UnsupportedEncodingException e) {
      System.err.println("ERROR:");
      System.err.println("Specified encoding is unsupported: " + encoding);
      System.err.println();
      return 2;
    }

    try {
      PrintWriter pw = null;
      try {
        pw = new PrintWriter(new OutputStreamWriter(new FileOutputStream(file),
          encoding));

        if (isXml) {
          config.outputSampleXml(pw, encoding);
        }
        else {
          config.outputSampleProp(pw);
        }
        pw.flush();
        return 0;
      }
      finally {
        if (pw != null) try { pw.close(); } catch (Exception e) {}
      }
    }
    catch (Exception e) {
      System.err.println("ERROR:");
      System.err.println(e.toString());
      System.err.println();
      return 3;
    }
  }
}
